(ns bilus.pocketbase.sync.collections
  "PocketBase Collections API wrapper for collection management.

  Note: All operations require superuser authentication."
  (:require [bilus.pocketbase.sync.core :as core]
            [bilus.pocketbase.schema :as schema]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Public
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn paginated
  "A paginated collections list as a map with the following keys:
   :page :per-page :total-items :total-pages :items

   Requires superuser authentication.

   Supported options:
    :filter - Filter expression (e.g., \"type = 'auth'\")
    :sort - Sort expression (e.g., \"-created,name\")
    :fields - Fields to return
    :skip-total - Skip total count query"
  ([client]
   (paginated client 1 30 nil))
  ([client page]
   (paginated client page 30 nil))
  ([client page per-page]
   (paginated client page per-page nil))
  ([^bilus.pocketbase.sync.core.Client client page per-page opts]
   (let [path "/api/collections"
         query-params (schema/merge-opts {:page page :perPage per-page}
                                         [:filter :sort :fields :skipTotal] opts)]
     (core/request* client :get path {:query-params query-params}))))

(defn all
  "Returns all collections.

   Requires superuser authentication.

   Supported options:
    :batch - Collections per batch request (default 500)
    :filter - Filter expression
    :sort - Sort expression
    :fields - Fields to return"
  ([client]
   (all client nil))
  ([^bilus.pocketbase.sync.core.Client client opts]
   (let [batch-size (or (:batch opts) 500)]
     (loop [page 1
            all-items []]
       (let [path "/api/collections"
             query-params (schema/merge-opts {:page page :perPage batch-size :skipTotal false}
                                             [:filter :sort :fields] opts)
             response (core/request* client :get path {:query-params query-params})
             items (:items response)
             new-items (into all-items items)]
         (if (< (count items) batch-size)
           new-items
           (recur (inc page) new-items)))))))

(defn find-first
  "Returns the first collection matching the specified filter or throws if not found.

   Requires superuser authentication.

   Supported options:
    :fields - Fields to return"
  ([client filter]
   (find-first client filter nil))
  ([^bilus.pocketbase.sync.core.Client client filter opts]
   (let [path "/api/collections"
         query-params (schema/merge-opts {:page 1
                                          :perPage 1
                                          :skipTotal true
                                          :filter filter}
                                         [:fields] opts)

         response (core/request* client :get path {:query-params query-params})
         items (:items response)]
     (if (seq items)
       (first items)
       (throw (ex-info "No collection found matching filter"
                       {:filter filter}))))))

(defn one
  "Returns a single collection by its ID or name.

   Requires superuser authentication.

   Supported options:
    :fields - Fields to return"
  ([client id-or-name]
   (one client id-or-name nil))
  ([^bilus.pocketbase.sync.core.Client client id-or-name opts]
   (let [path (str "/api/collections/" id-or-name)
         query-params (schema/merge-opts {}
                                         [:fields] opts)]
     (core/request* client :get path {:query-params query-params}))))

(defn create!
  "Creates a new collection.

   Requires superuser authentication.

   Body configuration:
    :name - (required) Unique collection name
    :type - :base (default), :view, or :auth
    :fields - Array of field definitions
    :indexes - Array of index definitions
    :system - Mark as system collection
    :list-rule, :view-rule, :create-rule, :update-rule, :delete-rule - API rules

   For view collections:
    :view-query - (required) SQL SELECT query

   For auth collections:
    :auth-rule - Auth constraint rule
    :manage-rule - Admin management rule
    :password-auth - Password auth config {:enabled :identity-fields}
    :oauth2 - OAuth2 config
    :mfa - MFA config
    :otp - OTP config

   Supported options:
    :fields - Fields for response filtering"
  ([client body]
   (create! client body nil))
  ([^bilus.pocketbase.sync.core.Client client body opts]
   (let [path "/api/collections"
         query-params (schema/merge-opts {}
                                         [:fields] opts)]
     (core/request* client :post path
                    {:body (schema/map-keys->camel-case body)
                     :query-params query-params}))))

(defn update!
  "Updates an existing collection referenced by its id or name. Requires superuser authentication.

   Supported options:
   - :fields - Fields for response filtering (e.g. \"id,title,author\") TODO: Isn't this to specify fields to update?

   Example:
   ```clojure
   (collections/update! client \"posts\"
     {:list-rule \"@request.auth.id != ''\"
      :fields [(text-field :title {:required true})
               (editor-field :content)
               (text-field :category))})
   ```"
  ([client id-or-name body]
   (update! client id-or-name body nil))
  ([^bilus.pocketbase.sync.core.Client client id-or-name body opts]
   (let [path (str "/api/collections/" id-or-name)
         query-params (schema/merge-opts {}
                                         [:fields] opts)]
     (core/request* client :patch path
                    {:body body
                     :query-params query-params}))))

(defn delete!
  "Deletes a collection by its ID or name.

   Requires superuser authentication.

   Note: Will fail if the collection is referenced by other collections."
  [^bilus.pocketbase.sync.core.Client client id-or-name]
  (let [path (str "/api/collections/" id-or-name)]
    (core/request* client :delete path)
    nil))

(defn truncate!
  "Deletes all records in a collection (including files and cascade relations).

   Requires superuser authentication."
  [^bilus.pocketbase.sync.core.Client client id-or-name]
  (let [path (str "/api/collections/" id-or-name "/truncate")]
    (core/request* client :delete path)
    nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Field Type Helpers
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn text-field
  "Creates a text field configuration.

   Supported options:
    :required :min :max :pattern :hidden :presentable"
  ([name]
   (text-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :text}
                      [:required :min :max :pattern :hidden :presentable] opts)))

(defn number-field
  "Creates a number field configuration.

   Supported options:
    :required :min :max :only-int :hidden :presentable"
  ([name]
   (number-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :number}
                      [:required :min :max :onlyInt :hidden :presentable] opts)))

(defn bool-field
  "Creates a boolean field configuration.

   Supported options:
    :required :hidden :presentable"
  ([name]
   (bool-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :bool}
                      [:required :hidden :presentable] opts)))

(defn email-field
  "Creates an email field configuration.

   Supported options:
    :required :except-domains :only-domains :hidden :presentable"
  ([name]
   (email-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :email}
                      [:required :exceptDomains :onlyDomains :hidden :presentable] opts)))

(defn url-field
  "Creates a URL field configuration.

   Supported options:
    :required :except-domains :only-domains :hidden :presentable"
  ([name]
   (url-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :url}
                      [:required :exceptDomains :onlyDomains :hidden :presentable] opts)))

(defn date-field
  "Creates a date field configuration.

   Supported options:
    :required :min :max :hidden :presentable"
  ([name]
   (date-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :date}
                      [:required :min :max :hidden :presentable] opts)))

(defn select-field
  "Creates a select field configuration.

   Supported options:
    :required :max-select :hidden :presentable"
  ([name values]
   (select-field name values {}))
  ([name values opts]
   (schema/merge-opts {:name name :type :select :values values}
                      [:required :maxSelect :hidden :presentable] opts)))

(defn json-field
  "Creates a JSON field configuration.

   Supported options:
    :required :max-size :hidden :presentable"
  ([name]
   (json-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :json}
                      [:required :maxSize :hidden :presentable] opts)))

(defn file-field
  "Creates a file field configuration.

   Supported options:
    :required :max-select :max-size :mime-types :thumbs :protected :hidden :presentable"
  ([name]
   (file-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :file}
                      [:required :maxSelect :maxSize :mimeTypes :thumbs :protected :hidden :presentable] opts)))

(defn relation-field
  "Creates a relation field configuration.

   Supported options:
    :required :max-select :cascade-delete :min-select :hidden :presentable"
  ([name collection-id]
   (relation-field name collection-id {}))
  ([name collection-id opts]
   (schema/merge-opts {:name name :type :relation :collection-id collection-id}
                      [:required :maxSelect :cascadeDelete :minSelect :hidden :presentable] opts)))

(defn editor-field
  "Creates an editor (rich text) field configuration.

   Supported options:
    :required :max-size :convert-urls :hidden :presentable"
  ([name]
   (editor-field name {}))
  ([name opts]
   (schema/merge-opts {:name name :type :editor}
                      [:required :maxSize :convertUrls :hidden :presentable] opts)))
