(ns bilus.pocketbase.schema
  "Schema transformation utilities for converting between Clojure and PocketBase conventions. "
  (:require
   [camel-snake-kebab.core :as csk]
   #?(:clj [clojure.test :refer [is]]
      :cljs [cljs.test :refer [is]])))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Implementation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn transform-keys*
  "Recursively transforms all map and sequence keys using the given function."
  [f m]
  (cond
    (map? m)
    (into {} (map (fn [[k v]] [(f k) (transform-keys* f v)]) m))

    (sequential? m)
    (mapv #(transform-keys* f %) m)

    :else m))

(defn map-keys
  "Maps a function over the keys of a map."
  [f m]
  (into {} (map (fn [[k v]] [(f k) v]) m)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Public
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn normalize-response
  "Normalizes a response map by converting all keys to keywords.
   Used when receiving data from PocketBase API."
  {:test #(is (= {:userName "John" :createdAt "2024-01-01"}
                 (normalize-response {"userName" "John" "createdAt" "2024-01-01"})))}
  [response]
  (->> response
       (transform-keys* keyword)))

(defn map-keys->camel-case
  [opts]
  (map-keys csk/->camelCase opts))

(defn transform-keys->camel-case
  [m]
  (transform-keys* csk/->camelCase m))

(defn merge-opts
  "Merges default options with user-provided options, converting keys to camelCase.

  Parameters:
  - default: Default map (e.g., {:name name :type :text})
  - keys: Vector of allowed option keys (e.g., [:required :min :max])
  - opts: User-provided options map

  Returns: Merged map with camelCase keys for allowed options"
  [default keys opts]
  (let [cc-opts (map-keys->camel-case opts)]
    (merge default (select-keys cc-opts keys))))
