(ns bilus.bb-oneshot
  "Utility for creating self-removing tasks in bb.edn"
  (:require [rewrite-clj.zip :as z]))

(defn- remove-map-entry
  "Removes a key-value pair from a map at the current zipper location.
   zloc should be positioned at the key to remove.
   Returns zipper positioned at root."
  [zloc]
  (-> zloc
      z/remove        ; remove key
      z/next
      z/remove        ; remove value
      z/root
      z/of-node))

(defn- find-and-remove-task
  "Removes task with given name from :tasks map."
  [zloc task-name]
  (if-let [task-loc (-> zloc
                        (z/find-value z/next :tasks)
                        z/right
                        (z/find-value z/next task-name))]
    (remove-map-entry task-loc)
    zloc))

(defn- find-and-remove-dep
  "Removes bb-oneshot from :deps map."
  [zloc]
  (if-let [dep-loc (-> zloc
                       (z/find-value z/next :deps)
                       z/right
                       (z/find-value z/next 'dev.bilus/bb-oneshot))]
    (remove-map-entry dep-loc)
    zloc))

(defn- find-and-remove-require
  "Removes bb-oneshot require from :tasks :requires vector."
  [zloc]
  (if-let [requires-loc (-> zloc
                            (z/find-value z/next :tasks)
                            z/right
                            (z/find-value z/next :requires)
                            z/right)]
    ;; Find the require vector containing bilus.bb-oneshot
    (loop [loc (z/down requires-loc)]
      (if (nil? loc)
        zloc
        (let [node-str (z/string loc)]
          (if (and node-str (.contains node-str "bilus.bb-oneshot"))
            (-> loc z/remove z/root z/of-node)
            (recur (z/right loc))))))
    zloc))

(defn remove-task!
  "Removes a task from bb.edn and cleans up bb-oneshot dependency.

   Removes:
   1. The named task from :tasks
   2. dev.bilus/bb-oneshot from :deps
   3. bilus.bb-oneshot require from :tasks :requires

   Usage:
     (require '[bilus.bb-oneshot :as oneshot])
     (oneshot/remove-task! 'my-task)"
  [task-name]
  (let [bb-edn-path "bb.edn"]
    (-> (z/of-file bb-edn-path)
        (find-and-remove-task task-name)
        (find-and-remove-dep)
        (find-and-remove-require)
        z/root-string
        (->> (spit bb-edn-path)))
    (println (str "Task '" task-name "' removed from bb.edn"))))
