(ns kehaar
  (:require [clojure.core.async :as async]
            [clojure.edn :as edn]
            [langohr.core :as rmq]
            [langohr.basic :as lb]
            [langohr.channel :as lch]
            [langohr.consumers :as lc]
            [langohr.queue :as lq]))

(defn read-payload [^bytes payload]
  (-> payload
      (String. "UTF-8")
      edn/read-string))

(defn rabbit->async-handler-fn
  "Returns a RabbitMQ message handler function which forwards all
  message payloads to `channel`. Assumes that all payloads are UTF-8
  edn strings."
  [channel]
  (fn [ch meta ^bytes payload]
    (let [message (read-payload payload)]
      (async/>!! channel message))))

(defn rabbit->async
  "Subscribes to the RabbitMQ queue, taking each payload, decoding as
  edn, and putting the result onto the async channel."
  ([rabbit-channel queue channel]
   (rabbit->async rabbit-channel queue channel {:auto-ack true}))
  ([rabbit-channel queue channel options]
   (lc/subscribe rabbit-channel
                 queue
                 (rabbit->async-handler-fn channel)
                 options)))

(defn async->rabbit
  "Forward all messages on channel to the RabbitMQ queue."
  ([channel rabbit-channel queue]
   (async->rabbit channel rabbit-channel "" queue))
  ([channel rabbit-channel exchange queue]
   (async/go-loop []
     (let [message (async/<! channel)]
       (when-not (nil? message)
         (lb/publish rabbit-channel exchange queue (pr-str message))
         (recur))))))

(defn fn->handler-fn
  "Returns a RabbitMQ message handler function which calls f for each
  incoming message and replies on the reply-to queue with the
  response."
  ([f] (fn->handler-fn f ""))
  ([f exchange]
   (fn [ch {:keys [reply-to correlation-id]} ^bytes payload]
     (let [message (read-payload payload)
           response (f message)]
       (lb/publish ch exchange reply-to (pr-str response)
                   {:correlation-id correlation-id})))))

(defn responder
  "Given a RabbitMQ queue and a function, subscribes to that queue,
  calling the function on each edn-decoded message, and replies to the
  reply-to queue with the result."
  ([rabbit-channel queue f]
   (responder rabbit-channel queue f {:auto-ack true}))
  ([rabbit-channel queue f opts]
   (let [handler-fn (fn->handler-fn f)]
     (lc/subscribe rabbit-channel
                   queue
                   handler-fn
                   opts))))

(defn ch->response-fn
  "Returns a fn that takes a message, creates a promise for the
  response for that message, and puts [response-promise, message] on
  the channel given. Returns the response-promise."
  [channel]
  (fn [message]
    (let [response-promise (promise)]
      (async/go
        (async/>! channel [response-promise message]))
      response-promise)))

(defn wire-up-service
  "Wires up a core.async channel (managed through ch->response-fn) to
  a RabbitMQ queue that provides responses."
  ([rabbit-channel queue channel]
   (wire-up-service rabbit-channel ""
                    queue {:exclusive false :auto-delete true}
                    (* 5 60 1000) channel))
  ([rabbit-channel exchange queue queue-options timeout channel]
   (let [response-queue (lq/declare-server-named rabbit-channel {:exclusive true :auto-delete true})
         pending-calls (atom {})]
     (lc/subscribe rabbit-channel
                   response-queue
                   (fn [ch {:keys [correlation-id]} ^bytes payload]
                     (when-let [response-promise (@pending-calls correlation-id)]
                       (deliver response-promise (read-payload payload))
                       (swap! pending-calls dissoc correlation-id)))
                   {:auto-ack true})
     (async/go-loop []
       (let [ch-message (async/<! channel)]
         (when-not (nil? ch-message)
           (let [[response-promise message] ch-message
                 correlation-id (str (java.util.UUID/randomUUID))]
             (swap! pending-calls assoc correlation-id response-promise)
             (lb/publish rabbit-channel
                         exchange
                         queue
                         (pr-str message)
                         {:reply-to response-queue
                          :correlation-id correlation-id})
             (async/go
               (async/<! (async/timeout timeout))
               (swap! pending-calls dissoc correlation-id))
             (recur))))))))

(defn connect-with-retries
  "Creates and returns a new connection to RabbitMQ, retrying
  `max-retries` times (default 5), backing off an additional second
  for each attempt."
  ([]
   (connect-with-retries {}))
  ([settings]
   (connect-with-retries settings 5))
  ([settings max-retries]
   (let [connection-promise (promise)]
     (loop [attempt 1]
       (try
         (deliver connection-promise (rmq/connect settings))
         (catch Throwable t
           (when (>= attempt max-retries)
             (throw t))))
       (if (realized? connection-promise)
         @connection-promise
         (do (Thread/sleep (* attempt 1000))
             (recur (inc attempt))))))))

(defn setup-queue
  "Sets up a queue based on its settings. Returns the Rabbit channel
  created during setup.

  :queue    - Name of the queue
  :settings - Rabbit settings for the queue

  One of the following:
  :push-to   - A core.async channel to forward all messages to
  :pull-from - A core.async channel to take messages from and put on the queue
  :handler   - A function to apply to all incoming messages, replying with the return value
  :service   - A core.async channel (managed by ch->response-fn) to make calls to a handler on the other side of the queue

  The above are symbols which will be resolved with `find-var`."
  [connection {:keys [queue settings push-to pull-from handler service]}]
  (let [ch (lch/open connection)]
    (lq/declare ch queue settings)
    (cond push-to (rabbit->async ch queue (find-var push-to))
          pull-from (async->rabbit (find-var pull-from) ch queue)
          handler (responder ch queue (find-var handler))
          service (wire-up-service ch queue (find-var service)))
    ch))

(defn setup-queues
  "Sets up a collection of queues, returning the Rabbit channels
  created to set up each one."
  [connection queue-configurations]
  (doall (map (partial setup-queue connection) queue-configurations)))

(defn safe-close [conn-or-ch]
  (when-not (rmq/closed? conn-or-ch) (rmq/close conn-or-ch)))

(defn setup
  "Connect to RabbitMQ and set up queues. Returns a map containing all
  connections and channels created for later teardown."
  [{:keys [connection queues]}]
  (let [conn (connect-with-retries connection)
        channels (setup-queues conn queues)]
    {:connections [conn]
     :channels channels}))

(defn teardown
  "Close all connections and channels from `setup`."
  [{:keys [connections channels]}]
  (doseq [conn connections]
    (safe-close conn))
  (doseq [ch channels]
    (safe-close ch)))
