(ns election-reminders-specs.common
  "Election Reminders common specifications that are not specific to any
  particular data model."
  (:require
   [clojure.spec.alpha :as s])
  (:import
   (com.google.i18n.phonenumbers NumberParseException
                                 PhoneNumberUtil)
   (java.time DateTimeException
              LocalDate
              Period
              ZoneId)))

(defn iso-8601-date?
  "Test if `s` is a valid representation of an ISO 8601 date."
  [^CharSequence s]
  (try
    (instance? LocalDate (LocalDate/parse s))
    (catch DateTimeException _
      false)))

(s/def ::iso-8601-date
  (s/and string? iso-8601-date?))

(defn iso-8601-period?
  "Is `s` a string representation of an ISO-8601 period?

  Example: \"P2Y\""
  [^CharSequence s]
  (try
    (boolean (Period/parse s))
    (catch DateTimeException _
      false)))

(s/def ::iso-8601-period
  (s/and string? iso-8601-period?))

(defn phone-number?
  "Is `s` a string representation of a valid US phone number?"
  [^CharSequence s]
  (let [pnu (PhoneNumberUtil/getInstance)]
    (try
      (.isValidNumber pnu (.parse pnu s "US"))
      (catch NumberParseException _
        false))))

(s/def ::phone-number
  (s/and string? phone-number?))

(defn time-zone-id?
  "Is `s` a valid time zone identifier?"
  [^String s]
  (try
    (boolean (ZoneId/of s))
    (catch DateTimeException _
      false)))

(s/def ::time-zone-id
  (s/and string? time-zone-id?))
