# Custom Passport Strategies

If you need to support a custom passport authentication strategy or our existing authentication adapters don't support your strategy it's easy to implement your own without an adapter. In fact, it's not really any different than the way you would with Express as outlined in your passport strategy.

## An example using Steam OpenID

Instead of calling it like this:

```js
passport.use(new SteamStrategy({
    returnURL: 'http://localhost:3000/auth/steam/return',
    realm: 'http://localhost:3000/',
    apiKey: 'your steam API key'
  },
  function(identifier, profile, done) {
    User.findByOpenID({ openId: identifier }, function (err, user) {
      return done(err, user);
    });
  }
));

app.get('/auth/steam',
  passport.authenticate('steam'),
  function(req, res) {
    // The request will be redirected to Steam for authentication, so
    // this function will not be called.
  });

app.get('/auth/steam/return',
  passport.authenticate('steam', { failureRedirect: '/login' }),
  function(req, res) {
    // Successful authentication, redirect home.
    res.redirect('/');
  });
```

You set it up and call it like this:

```js
const feathers = require('feathers');
const auth = require('feathers-authentication');
const app = feathers();

app.configure(auth({ secret: 'supersecret' });

app.passport.use(new SteamStrategy({
    returnURL: 'http://localhost:3000/auth/steam/return',
    realm: 'http://localhost:3000/',
    apiKey: 'your steam API key'
  },
  function(identifier, profile, done) {
    const query = { openId: identifier };
    app.service.find({ query }).then(response) {
      const user = response[0];
      return done(null, user);
    }).catch(done);
  }
));

app.get('/auth/steam', auth.express.authenticate('steam'));

app.get('/auth/steam/return',
  auth.express.authenticate('steam', { failureRedirect: '/login' }),
  function(req, res) {
    // Successful authentication, redirect home.
    res.redirect('/');
  });
```