(ns degree9.service
  "A microservice proxy via websockets."
  (:require [degree9.socket-io :as io]
            ["@feathersjs/feathers" :as feathers]
            ["@feathersjs/socketio-client" :as socketio]
            [goog.object :as obj]
            [meta.server :as server]
            [degree9.debug :as dbg]))

(dbg/defdebug debug "degree9:enterprise:service")

(defn- log-setup [service]
  (debug "Setup proxy to remote service %s" service))

(defn- log-proxy [method service & [params]]
  (debug "Proxy %s to remote service %s with params %s" method service params))


(defn with-websockets [client socket]
  (doto client
    (.configure
      (socketio socket))))

(defn socket [uri & [opts]]
  (io/io uri (merge {:transports ["websockets"]} opts)))

(defn proxy [socket]
  (with-websockets (feathers) socket))

(defn µservice [client socket service & [opts]]
  (let [service (.service client service)]
    (debug "Initializing remote microservice %s" uri)
    (reify
      Object
      (setup [this app path]
        (log-setup service)
        (obj/set this "io" service))
      (find [this params]
        (debug "Proxy find to remote service %s with %s" service params)
        (.find service params))
      (get [this id params]
        (debug "Proxy get to remote service %s/%s with %s" service id params)
        (.get service id params))
      (create [this data params]
        (debug "Proxy create to remote service %s with %s" service params)
        (.create service data params))
      (update [this id data params]
        (debug "Proxy update to remote service %s with %s" service params)
        (.update service id data params))
      (patch [this id data params]
        (debug "Proxy patch to remote service %s with %s" service params)
        (.patch service id data params))
      (remove [this id params]
        (debug "Proxy remove to remote service %s with %s" service params)
        (.remove service id params)))))

(defn api
  "Mount a remote Microservice to a local endpoint."
  ([app path uri service hooks]
   (api app path uri service nil hooks))
  ([app path uri service opts hooks]
   (debug "Mount remote microservice at %s" path)
   (server/api app path (µservice uri service opts) hooks)))

;; Multi-Service ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- api-service [app [prefix url services] hooks]
  (reduce (fn [app service] (api app (str prefix service) url service hooks)) app services))

(defn- reduce-apis [app services hooks]
  (reduce (fn [app service] (api-service app service hooks)) app services))

(defn multi-service
  "Mount multiple remote services from different servers."
  [app specs & [hooks]]
  (reduce-apis app specs hooks))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
