(ns degree9.routing
  (:require [javelin.core :as j]
            [degree9.object :as obj]
            [degree9.string :as str]
            [degree9.browser :as bom]
            [degree9.events :as events]
            [degree9.url :as url])
  (:require-macros degree9.routing))

;; HTML5 History State ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- state-cell []
  (let [history (j/cell (:state (bom/get-history)))]
    (j/with-let [history= (j/cell= history (partial reset! history))]
      (events/listen (bom/get-window) :popstate
        (fn [{:keys [state]}] (reset! history= state))))))

(def state (state-cell))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; URL ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- url-cell []
  (let [url (j/cell (url/create-url))]
    (j/with-let [url= (j/cell= url (partial reset! url))]
      (events/listen (bom/get-window) :popstate
        (fn [event] (reset! url= (url/create-url)))))))

(def url (url-cell))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; URLSearchParams ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- search->clj [search]
  (let [entries (.entries search)]
    (js->clj (obj/from-entries entries))))

(defn- clj->search [data]
  (js/URLSearchParams. (clj->js data)))

(defn- search-cell []
  (let [params (j/cell (url/create-search-params))]
    (j/with-let [params= (j/cell= params (partial reset! params))]
      (events/listen (bom/get-window) :popstate
        (fn [event] (reset! params= (url/create-search-params)))))))

(def search (search-cell))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; URL Pathname ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- path-cell []
  (let [path (j/cell (:pathname (bom/get-location)))]
    (j/with-let [path= (j/cell= path (partial reset! path))]
      (events/listen (bom/get-window) :popstate
        (fn [event] (reset! path= (:pathname (bom/get-location))))))))

(def path (path-cell))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; HTML5 History Helpers ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Routing Public API ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn route!
  "Provides client (SPA) routing without reloading the page. "
  ([path] (route! path nil))
  ([path query] (route! path query @state))
  ([path query state]
   (let [title  (bom/get-title)
         state  (clj->js state)
         search (clj->search query)
         token  (if query (str/join "?" [path (str search)]) path)]
     (j/dosync
       (push! state title token)
       (popstate! state)))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
