(ns degree9.pathway
  (:require [clojure.spec.alpha :as spec]
            [degree9.string :as str]
            [degree9.debug :as dbg]
            [degree9.pathway.spec :as pspec]))

(dbg/defdebug debug "degree9:enterprise:pathway")

(defprotocol IPathway
  "A protocol for pathway matching."
  (-match-pathway [router pattern] "Returns a handler if pattern matches router."))

(extend-protocol IPathway
  nil
  (-match-pathway [router pattern]
    (debug "nil" router pattern)
    (if (= router pattern) true
      (reduced nil)))
  string
  (-match-pathway [router pattern]
    (debug "string" router pattern)
    (prn (type router) (type pattern))
    (= router pattern))
  Keyword
  (-match-pathway [router pattern]
    (debug "Keyword" (clj->js router) (clj->js pattern))
    (= router (keyword pattern)))
  js/RegExp
  (-match-pathway [router pattern]
    (debug "RegExp" (clj->js router) (clj->js pattern))
    (re-matches router pattern))
  PersistentHashSet
  (-match-pathway [router pattern]
    (debug "PersistentHashSet" (clj->js router) (clj->js pattern))
    (reduce (fn [_ r] (-match-pathway r pattern)) nil router))
  PersistentVector
  (-match-pathway [router pattern]
    (debug "PersistentVector" (clj->js router) (clj->js pattern))
    (reduce (fn [i [r h]] (if (-match-pathway r pattern) h i)) nil router))
  PersistentHashMap
  (-match-pathway [router pattern]
    (debug "PersistentHashMap" (clj->js router) (clj->js pattern))
    (reduce-kv (fn [i r h] (if (-match-pathway r pattern) h i)) nil router))
  PersistentArrayMap
  (-match-pathway [router pattern]
    (debug "PersistentArrayMap" (clj->js router) (clj->js pattern))
    (reduce-kv (fn [i r h] (prn i r h) (if (-match-pathway r pattern) h i)) nil router)))

(defn match-route
  ([router pattern] (match-route router pattern nil))
  ([router pattern default] (match-route router pattern default #"/"))
  ([router pattern default separator]
   {:pre [(spec/valid? ::pspec/pathway router)]}
   (let [patterns (remove empty? (str/split pattern separator))]
     (debug "MATCH-ROUTE" (clj->js router) (clj->js patterns))
     (if (empty? patterns) default
       (let [handler (reduce match-route router patterns)]
         (prn handler)
         (if (or (string? handler) (keyword? handler)) handler
           (when-let [default (get handler nil)]
             default)))))))
