(ns degree9.es6
  (:refer-clojure :exclude [class])
  (:require [cljs.analyzer :as analyzer]
            [cljs.compiler :as compiler]))

;; Extends Analyzer Special Forms ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(alter-var-root #'analyzer/specials #(conj % 'js-class* 'js-method* 'js-super*))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Native ES6 Class Super compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod analyzer/parse 'js-super*
  [op env [_  & params :as form] _ _]
  {:env env
   :op :js-super
   :form form
   :params (analyzer/disallowing-recur
            (mapv #(analyzer/analyze env %) params))})

(defmethod compiler/emit* :class-super
  [{:keys [params env]}]
  (compiler/emits "super(" (interpose "," params) ")"))

(defmacro super
  "Call the parent class from a new extended class method. (es6+)"
  [& params]
  `(~'super* ~@params))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Native ES6 Class compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- assoc-locals [env local]
  (assoc-in env [:locals local] {:name local}))

(defn- analyse-with-locals [env locals exp]
  (analyzer/analyze (reduce assoc-locals env locals) exp))

(defmethod analyzer/parse 'js-method*
  [op env [_ method params & exprs :as form] _ _]
  (analyzer/disallowing-recur
    {:env env
     :op :js-method
     :children [:exprs]
     :form form
     :method method
     :params params
     :exprs (mapv (partial analyse-with-locals env params) exprs)}))

(defmethod compiler/emit* :js-method
  [{:keys [method params exprs]}]
  (compiler/emitln method "(" (interpose "," params) "){")
  (doseq [e exprs]
    (compiler/emitln e))
  (compiler/emitln "}"))

(defn- methodize [[method params & body]]
  `(~'method* ~method ~params ~@body))

(defmethod analyzer/parse 'js-class*
  [op env [_ class extends & methods :as form] _ _]
  (analyzer/disallowing-recur
    {:env env
     :op :js-class
     :children [:ctor]
     :form form
     :class class
     :extends (when extends (analyzer/analyze-symbol env extends))
     :methods methods}))

(defmethod compiler/emit* :js-class
  [{:keys [class extends methods]}]
  (if class
    (compiler/emits "class " class)
    (compiler/emits "class"))
  (when extends
    (compiler/emits " extends " extends))
  (compiler/emitln " {")
  (doseq [m methods]
    (compiler/emitln m))
  (compiler/emits "}"))

(defmacro ^:private class
  "Create a named or unnamed javascript class. (es6+)"
  ([ctor] (class nil ctor))
  ([name ctor] (class name nil ctor))
  ([name extends & methods]
   `(~'class* ~name ~extends ~@(map methodize methods))))

(defmacro defclass
  "Define a named javascript class. (es6+)"
  ([name ctor] (defclass name nil ctor))
  ([name extends ctor]
   `(def ~name (class ~name ~extends ~ctor))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
