(ns degree9.es6
  (:refer-clojure :exclude [class])
  (:require [cljs.analyzer :as analyzer]
            [cljs.compiler :as compiler]))

;; Extends Analyzer Special Forms ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(alter-var-root #'analyzer/specials #(conj % 'class* 'super*))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Native ES6 Class Super compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod analyzer/parse 'super*
  [op env [_  & params :as form] _ _]
  {:env env
   :op :super
   :form form
   :params (analyzer/disallowing-recur
            (mapv #(analyzer/analyze env %) params))})

(defmethod compiler/emit* :super
  [{:keys [params env]}]
  (compiler/emits "super(" (interpose "," params) ")"))

(defmacro super
  "Call the parent class from a new extended class method. (es6+)"
  [& params]
  `(~'super* ~@params))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Native ES6 Class compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod analyzer/parse 'class*
  [op env [_ class extends [_ params & ctor] :as form] _ _]
  {:env env
   :op :class
   :children [:ctor]
   :form form
   :class class
   :extends (when extends
              (analyzer/disallowing-recur (analyzer/analyze-symbol env extends)))
   :params params
   :ctor (analyzer/disallowing-recur
           (mapv (fn [ctor] (analyzer/analyze (reduce (fn [i p] (assoc-in i [:locals p] {:name p})) env params) ctor)) ctor))})

(defmethod compiler/emit* :class
  [{:keys [class extends params ctor]}]
  (compiler/emits "class " class)
  (when extends (compiler/emits " extends " extends))
  (compiler/emitln " {")
  (compiler/emitln "constructor(" (interpose "," params) "){")
  (doseq [c ctor]
    (compiler/emitln c))
  (compiler/emitln "}")
  (compiler/emits "}"))

(defmacro ^:private class
  "Create a named or unnamed javascript class. (es6+)"
  ([ctor] (class nil ctor))
  ([name ctor] (class name nil ctor))
  ([name extends ctor]
   `(~'class* ~name ~extends ~ctor)))

(defmacro defclass
  "Define a named javascript class. (es6+)"
  ([name ctor] (defclass name nil ctor))
  ([name extends ctor]
   `(def ~name (class ~name ~extends ~ctor))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
