(ns degree9.es6
  (:refer-clojure :exclude [class])
  (:require [cljs.analyzer :as ana]
            [cljs.compiler :as compiler]))

(def ^:private ecmascript-unsupported
  #{:ecmascript3 :ecmascript5 :ecmascript5-strict})

(defn- ecmascript-in []
  (get-in @cljs.env/*compiler* [:options :language-in]))

;; Native ES6 Class Method compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(alter-var-root #'ana/specials #(conj % 'class* 'method* 'super*))

(defmethod ana/parse 'method*
  [op env [_ method [params & exprs] :as form] _ _]
  (prn method params exprs)
  {:env env
   :op :method
   :children [:exprs]
   :form form
   :method method
   :params params
   :exprs (ana/disallowing-recur
            (mapv #(ana/analyze (assoc env :context :statement) %) exprs))})

(defmethod compiler/emit* :method
  [{:keys [exprs method params env]}]
  (compiler/emits method "(")
  (compiler/emit-fn-params params)
  (compiler/emitln "){")
  (doseq [expr exprs]
    (compiler/emits expr))
  (compiler/emits "}"))

(defmacro ^:private method
  "Create a javascript class method. (es6+)"
  [name & body]
  `(~'method* ~name ~@body))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Native ES6 Class compiler extension ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod ana/parse 'class*
  [op env [_ class extends & methods :as form] _ _]
  (when (ecmascript-unsupported (ecmascript-in))
    (throw (ana/error env "Can't use defclass when compiler :language-in lower than :ecmascript-2015")))
  {:env env
   :op :class
   :children [:methods]
   :form form
   :class class
   :extends extends
   :methods (ana/disallowing-recur
              (mapv #(ana/analyze (assoc env :context :method) %) methods))})

(defmethod compiler/emit* :class
  [{:keys [class extends methods]}]
  (let [extends (when extends (str " extends " extends))]
    (compiler/emitln "class " class extends " {")
    (doseq [m methods]
      (compiler/emitln m))
    (compiler/emits "}")))

(defn- methodize [[fname & body]]
  (let [name (symbol (name fname))]
    `(method ~name ~@body)))

(defmacro ^:private class
  "Create a named or unnamed javascript class. (es6+)"
  ([method-map] (class nil method-map))
  ([name method-map] (class name nil method-map))
  ([name extends method-map]
   `(~'class* ~name ~extends ~@(map methodize method-map))))

(defmacro defclass
  "Define a named javascript class. (es6+)"
  ([name & [extends & methods :as body]]
   (let [methods (->> (if (symbol? extends) methods body)
                      (map (fn [[m & b]] [(keyword m) b]))
                      (into {}))]
     (prn extends methods)
     `(def ~name (class ~name ~extends ~methods)))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
