(ns degree9.env
  "Load configuration variables from process.env or .env file."
  (:refer-clojure :exclude [get keys])
  (:require [cuerdas.core :as str]
            [clojure.string :as cstr]
            [goog.object :as obj]
            [degree9.debug :as dbg]
            ["dotenv" :as dotenv]
            ["fs" :as fs]
            ["path" :as path]))

(dbg/defdebug debug "degree9:enterprise:env")

;; Env Public API ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn env-vars [])
  ;; return final map of env vars

(defn get
  "Return the environment value for `key` or `default`."
  ([key] (get key nil))
  ([key default] (get js/process.env key default))
  ([env key default] (obj/get env (-> key name str/snake str/upper) default)))

(defn keys
  "Return all keys from the environment."
  []
  (->> js/process.env
       (js-keys)
       (js->clj)))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- read-file [path]
  (.readFileSync fs path #js{:encoding "utf8"}))

(defn- env-file [dir]
  (.resolve path (or dir (.cwd js/process)) ".env"))

(defn- split-kv [kvstr]
  (cstr/split kvstr #"=" 2))

(defn- split-config [config]
  (->> (cstr/split-lines config)
       (map split-kv)
       (into {})))

(defn- dot-env [path]
  (-> (env-file path)
      (read-file)
      (split-config)))

(defn- node-env []
  (let [env js/process.env]
    (->> (js-keys env)
         (map (fn [key] [key (obj/get env key)]))
         (into {}))))

(defn- populate-env! [env]
  (map (fn [[k v]] (obj/set js/process.env k v)) env))

(defn init! [& [path]]
  (let [env (merge (dot-env path) (node-env))]
    (prn env)))
  ;   (populate-env! env)
  ;   (reset! *env* env)))
