(ns degree9.env
  "Load configuration variables from process.env or .env file."
  (:refer-clojure :exclude [get keys])
  (:require [cuerdas.core :as str]
            [clojure.string :as cstr]
            [goog.object :as obj]
            [degree9.debug :as dbg]
            ["dotenv" :as dotenv]
            ["fs" :as fs]
            ["path" :as path]))

(dbg/defdebug debug "degree9:enterprise:env")

;; Env State ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(def ^:dynamic *env* (atom nil))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- read-file [path encoding]
  (.readFileSync fs path #js{:encoding encoding}))

(defn- split-kv [kvstr]
  (cstr/split kvstr #"=" 1))

(defn- split-config [config]
  (->> (cstr/split-lines config)
       (map split-kv)
       (into {})))

(defn- dot-env! [path encoding]
  (split-config (read-file path encoding)))

(defn- node-env! []
  (js->clj js/process.env))

(defn init!
  ([] (init! (.resolve path (.cwd js/process) ".env")))
  ([path] (init! path "utf8"))
  ([path encoding]
   (reset! *env* (merge (dot-env!) (node-env!)))))

(defn env-obj
  "Initialize dotenv and return the process.env object."
  [& conf]
  (debug "Initializing env object with config %s" conf)
  (when-not @*env*
   (reset! *env* (.config dotenv conf)))
  (obj/get js/process "env"))

(defn keys
  "Return all keys from the process.env object."
  []
  (->> (env-obj)
    (js-keys)
    (js->clj)
    (map str/kebab)
    (map keyword)))

(defn get
  "Return the process.env object value for `key` or `default`."
  ([key] (get key nil))
  ([key default] (obj/get (env-obj) (-> key name str/snake str/upper) default)))
