(ns degree9.paysafe.core
  (:refer-clojure :exclude [get])
  (:require ["base64" :as base64]
            [degree9.env :as env]
            [degree9.request :as req]))

(defn- json->clj [res]
  (js->clj (.json res) :keywordize-keys true))

(defn- check-status [res]
  (if (.ok status) res
    (throw (js/Error. (.statusText res)))))

(defn- paysafe-url [path]
  (str (env/get "PAYSAFE_API_URL" "http://api.test.paysafe.com") path))

(defn- paysafe-headers [headers]
  (let [username (env/get "PAYSAFE_API_USERNAME")
        password (env/get "PAYSAFE_API_PASSWORD")
        auth     (.encode base64 (str username ":" password))]
    (merge {:authorization (str "Basic " auth)} headers)))

(defn- paysafe-request [{:keys [method path data query headers] :as opts}]
  (let [method (upper-case (name method))
        headers (paysafe-headers headers)]
    (-> (req/fetch path (clj->js {:method method :body data :headers headers}))
        (.then check-status)
        (.then json->clj))))

(defn post [path data & [{:keys [headers] :as opts}]]
  (paysafe-request
    {:method :post
     :path path
     :data data
     :headers headers}))

(defn get [path & [{:keys [headers] :as opts}]]
  (paysafe-request
    {:method :get
     :path path
     :data data
     :headers headers}))

(defn delete [path & [{:keys [headers] :as opts}]]
  (paysafe-request
    {:method :delete
     :path path
     :data data
     :headers headers}))

(defn put [path data & [{:keys [headers] :as opts}]]
  (paysafe-request
    {:method :put
     :path path
     :data data
     :headers headers}))

(defn patch [path data & [{:keys [headers] :as opts}]]
  (paysafe-request
    {:method :patch
     :path path
     :data data
     :headers headers}))
