(ns huaihua.core
  (:require [clojure.core.match :refer [match]]
            [instaparse.core :as insta])
  (:refer-clojure :exclude [compile]))

(defn- error! [msg & [map]]
  (let [map (or map {})]
    (throw (ex-info msg map))))

(def ^:private gramma
  "TEXT = ( TAG | NORMAL_STR )*
  TAG = OPEN_TOKEN, TEXT, CLOSE_TOKEN
  NORMAL_STR = #\"[^{{}}]*\"
  OPEN_TOKEN = <'{{'>,SYMBOL, <'}}'>
  CLOSE_TOKEN = <'{{/'> SYMBOL <'}}'>
  SYMBOL = #\"[a-zA-Z][a-zA-Z0-9-]+[a-zA-Z0-9]\" ")

(def ^:private parse (insta/parser gramma :start :TEXT))

(def ^:private root-name "root")

(defn- ast [ep]
  (letfn [(get-ast [ep]
            (match [ep]
              [[:TEXT & rest-text]]
              (mapv get-ast rest-text)

              [[:TAG
                [:OPEN_TOKEN [:SYMBOL open-symbol]]
                [:TEXT & tag-text]
                [:CLOSE_TOKEN [:SYMBOL close-symbol]]]]
              {:name (keyword open-symbol)
               :children (mapv get-ast tag-text)}

              [([:NORMAL_STR str] :seq)]
              str

              :else
              (throw (RuntimeException. "Unexpected error when parse"))))]
    {:name root-name
     :children (get-ast ep)}))

(defn get-snippet
  ([s]
   (-> (parse s) (ast)))
  ([s node-key]
   (letfn [(find-node [struct]
             (cond
               (map? struct)
               (if (= (:name struct) node-key)
                 struct
                 (some find-node (:children struct)))

               (vector? struct)
               (some find-node struct)

               (string? struct)
               nil

               :else (error! "Unknown Struct: " {:struct struct})))]
     (if-let [node (find-node (get-snippet s))]
       node
       (error! "Cannot find this node.")))))

(defn transform [struct & {:as key-transforms}]
  (letfn [(do-transform [struct]
            (cond
              (map? struct)
              (let [{:keys [name children]} struct
                    s (->> (map do-transform children)
                           (apply str))
                    f-or-s (get key-transforms name)]
                (if f-or-s
                  (cond
                    (fn? f-or-s) (f-or-s s)
                    (string? f-or-s) f-or-s
                    :else (error! "Should pass fn or string."))
                  s))

              (vector? struct)
              (->> (map do-transform struct)
                   (apply str))

              (string? struct)
              struct))]
    (do-transform struct)))