(ns deercreeklabs.lancaster
  (:require
   [clojure.string :as str]
   [deercreeklabs.baracus :as ba]
   [deercreeklabs.lancaster.impl :as impl]
   [deercreeklabs.lancaster.schemas :as schemas]
   [deercreeklabs.lancaster.utils :as u]
   [deercreeklabs.log-utils :as lu :refer [debugs]]
   #?(:cljs [goog.math :as gm])
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre :refer [debugf errorf infof]])
  #?(:cljs
     (:require-macros deercreeklabs.lancaster)))

(declare make-name*)

#?(:cljs (def Long gm/Long))

(def int-schema (schemas/make-primitive-schema :int))
(def null-schema (schemas/make-primitive-schema :null))
(def boolean-schema (schemas/make-primitive-schema :boolean))
(def long-schema (schemas/make-primitive-schema :long))
(def float-schema (schemas/make-primitive-schema :float))
(def double-schema (schemas/make-primitive-schema :double))
(def bytes-schema (schemas/make-primitive-schema :bytes))
(def string-schema (schemas/make-primitive-schema :string))

(s/defn make-record-schema :- (s/protocol u/IAvroSchema)
  [name-kw :- s/Keyword
   fields :- [schemas/RecordFieldDef]]
  (schemas/make-schema :record name-kw fields))

(s/defn make-enum-schema :- (s/protocol u/IAvroSchema)
  [name-kw :- s/Keyword
   symbols :- [s/Keyword]]
  (schemas/make-schema :enum name-kw symbols))

(s/defn make-fixed-schema :- (s/protocol u/IAvroSchema)
  [name-kw :- s/Keyword
   size :- s/Int]
  (schemas/make-schema :fixed name-kw size))

(s/defn make-array-schema :- (s/protocol u/IAvroSchema)
  [items-schema :- (s/protocol u/IAvroSchema)]
  (schemas/make-schema :array nil items-schema))

(s/defn make-map-schema :- (s/protocol u/IAvroSchema)
  [values-schema :- (s/protocol u/IAvroSchema)]
  (schemas/make-schema :map nil values-schema))

(s/defn make-union-schema :- (s/protocol u/IAvroSchema)
  [members :- [(s/protocol u/IAvroSchema)]]
  (schemas/make-schema :union nil members))

(s/defn make-maybe-schema :- (s/protocol u/IAvroSchema)
  [schema :- (s/protocol u/IAvroSchema)]
  (make-union-schema [null-schema schema]))

(s/defn serialize :- ba/ByteArray
  [schema-obj :- (s/protocol u/IAvroSchema)
   data :- s/Any]
  ;; TODO: Figure out how to set initial size better
  (let [os (impl/make-output-stream 100)]
    (u/serialize schema-obj os data)
    (u/to-byte-array os)))

(s/defn deserialize :- s/Any
  [reader-schema-obj :- (s/protocol u/IAvroSchema)
   writer-pcf :- s/Str
   ba :- ba/ByteArray]
  (let [is (impl/make-input-stream ba)]
    (u/deserialize reader-schema-obj writer-pcf is)))

(s/defn wrap :- schemas/WrappedData
  [schema :- (s/protocol u/IAvroSchema)
   data :- s/Any]
  (u/wrap schema data))

(s/defn get-edn-schema :- s/Any
  [schema :- (s/protocol u/IAvroSchema)]
  (u/get-edn-schema schema))

(s/defn get-json-schema :- s/Str
  [schema :- (s/protocol u/IAvroSchema)]
  (u/get-json-schema schema))

(s/defn get-plumatic-schema :- s/Any
  [schema :- (s/protocol u/IAvroSchema)]
  (u/get-plumatic-schema schema))

(s/defn get-parsing-canonical-form :- s/Str
  [schema :- (s/protocol u/IAvroSchema)]
  (u/get-parsing-canonical-form schema))

(s/defn get-fingerprint64 :- Long
  [schema :- (s/protocol u/IAvroSchema)]
  (u/get-fingerprint64 schema))

;;;;;;;;;; Named Schema Helper Macros ;;;;;;;;;;;;;;;;

(defmacro def-record-schema
  [clj-name & fields]
  (let [ns-name (str (or
                      (:name (:ns &env)) ;; cljs
                      *ns*))             ;; clj
        schema-name (u/make-schema-name clj-name)]
    `(def ~clj-name
       (schemas/make-schema :record ~ns-name ~schema-name (vector ~@fields)))))

(defmacro def-enum-schema
  [clj-name & symbols]
  (let [ns-name (str (or
                      (:name (:ns &env)) ;; cljs
                      *ns*))             ;; clj
        schema-name (u/make-schema-name clj-name)]
    `(def ~clj-name
       (schemas/make-schema :enum ~ns-name ~schema-name (vector ~@symbols)))))

(defmacro def-fixed-schema
  [clj-name size]
  (let [ns-name (str (or
                      (:name (:ns &env)) ;; cljs
                      *ns*))             ;; clj
        schema-name (u/make-schema-name clj-name)]
    `(def ~clj-name
       (schemas/make-schema :fixed ~ns-name ~schema-name ~size))))
