(ns declatests.extras
  (:require [base64-clj.core :as base64]
            [datomic.api :as d]
            [clojure.test :as t]
            [clojure.set :as set]
            [clojure.string :refer [ends-with? join split]]
            [medley.core :refer [map-keys map-vals]]))

(defn- date->iso8601 [date]
  (str (java.time.OffsetDateTime/ofInstant (.toInstant date) (java.time.ZoneId/systemDefault))))

(defn- keys-name-only [e]
  (map-keys #(-> % name keyword) e))

(defn- emap->hmap
  [entity]
  (clojure.walk/prewalk
    (fn [x]
      (if (instance? datomic.query.EntityMap x)
        (into {} x)
        x))
    entity))

(defn- touch [conn e]
  (->> e
       (d/entity (d/db conn))
       d/touch
       emap->hmap))

(defn- decode-relay-id [id]
  (-> id base64/decode (split #":") second))

(defn- question->is_name [string]
  (if (ends-with? string "?")
    (->> string drop-last (join "") (str "is_"))
    string))

(defn- dash-case->snake_case [x]
  (let [convert (fn [s] (-> s
                           (clojure.string/replace #"-" "_")
                           question->is_name
                           ))
        ns (when (keyword? x) (namespace x))
        n (name x)]
    (if (nil? ns)
      (keyword (convert n))
      (keyword
        (str
          (convert ns)
          "/"
          (convert n))))))

(defn snake_case_all [query]
  (clojure.walk/prewalk (fn [x]
                          (if (keyword? x)
                            (dash-case->snake_case x)
                            x))
                        query))

(defn back-refs->forward-refs [back-refs data]
  (clojure.walk/prewalk
    (fn [x]
      (if (map? x)
        (set/rename-keys x back-refs)
        x))
    data))


(defn rm-namespaces [data]
  (clojure.walk/prewalk (fn [x]
                          (if (keyword? x)
                            (-> x name keyword)
                            x))
                        data))

(defn decode-ids [m]
  (clojure.walk/prewalk
    (fn [x] (if (and (map? x) (:id x))
             (update x :id decode-relay-id)
             x))
    m))

(defn update-enums [conn enums m]
  (let [update-enum (fn [e] (->> e :db/id (touch conn) :db/ident))
        f (fn [x]
            (if (map? x)
              (loop [x x
                     [e & es] (seq enums)]
                (cond
                  (and e (get x e)) (recur (update x e update-enum) es)
                  e (recur x es)
                  :else x))
              x))]
    (clojure.walk/postwalk f m)))

(defn rm-nils
  "TODO: rename this to: rm-nils-and-empty

  Remove Key value pairs where the value is either: nil or empty. "
  [m]
  (let [f (fn [x]
            (if (map? x)
              (let [kvs (filter (fn [[_ v]]
                                  (and (-> v nil? not)
                                       (if (sequential? v)
                                         (-> v empty? not)
                                         true)))
                                x)]
                (if (empty? kvs) nil (into {} kvs)))
              x))]
    (clojure.walk/postwalk f m)))

(defn sort-seqs
  "sort maps of sequentials by their id property."
  [m]
  (let [f (fn [x]
            (if (and (sequential? x) (-> x first map?) (-> x first keys-name-only :id))
              (sort-by #(-> % keys-name-only :id) x)
              x))]
    (clojure.walk/postwalk f m)))

(defn sort-maps
  "this is simply a util fn used to debug failing tests. If your tests
  are failing, wrap the expected and actual in this fn, to easily compare
  the difference between two maps. Use like so:
  (is (= (sort-maps expected) (sort-maps actual)))"
  [m]
  (let [f (fn [x]
            (if (map? x)
              (into (sorted-map) x)
              x))]
    (clojure.walk/postwalk f m)))

(defn vals->strings [data]
  (clojure.walk/prewalk
    (fn [x]
      (if (map? x)
        (map-vals (fn [val]
                    (if (keyword? val)
                      (name val)
                      val))
                  x)
        x))
    data))

(defn dates->strings [data]
  (clojure.walk/prewalk
    (fn [x]
      (if (instance? java.util.Date x)
        (date->iso8601 x)
        x))
    data))

(defn rm-datomic-junk [data]
  (clojure.walk/prewalk
    (fn [x]
      (if (map? x)
        (cond-> x
          (:db/id x) (dissoc :db/id)
          (:db/ident x) (dissoc :db/ident))
        x))
    data))

(defn rm-edges-and-nodes-from-data [data]
  (clojure.walk/prewalk (fn [x]
                          (if (and (map? x) (:edges x))
                            (->> x :edges (map :node))
                            x))
                        data))
