(ns declatests.core
  (:require [clojure.set :as set]
            [declatests.extras :refer [snake_case_all]]
            [venia.core :as venia]
            [clojure.string :refer [replace-first]]))

(defn- snake_case->dash-case [x]
  (let [convert (fn [s] (clojure.string/replace s #"_" "-"))
        ns (when (keyword? x) (namespace x))
        n (name x)]
    (if (nil? ns)
      (keyword (convert n))
      (keyword
        (str
          (convert ns)
          "/"
          (convert n))))))

(defn- replace-fragments [fragments query]
  (clojure.walk/prewalk (fn [x]
                          (let [s (into #{} (keys fragments))]
                            (cond
                              (set? x) (let [intersection (set/intersection x s)]
                                         (if (not (empty? intersection))
                                           (set/union
                                             (set/difference x intersection)
                                             (get fragments (first intersection)))
                                           x))
                              :else x)))
                        query))

(defn- forward-refs->back-refs [back-refs query]
  (let [forward-refs (set/map-invert back-refs)]
    (clojure.walk/prewalk
      (fn [x]
        (if (map? x)
          (set/rename-keys x forward-refs)
          x))
      query)))

(defn- dash-case-all [query]
  (clojure.walk/prewalk (fn [x]
                          (if (keyword? x)
                            (snake_case->dash-case x)
                            x))
                        query))

(defn- vector-pairs->hash-map [query]
  (clojure.walk/prewalk (fn [x]
                          (if (set? x)
                            (let [groups (group-by vector? x)
                                  vectors (get groups true)
                                  scalars (get groups false)
                                  maps (for [v vectors]
                                         (cond
                                           (= 2 (count v)) (apply hash-map v)
                                           (= 3 (count v)) (hash-map (first v) (nth v 2))
                                           :else (throw (ex-info "wrong count of vector" {:vector v}))))]
                              (apply hash-set (concat scalars maps)))
                            x))
                        query))

(defn- rm-first-layer [query]
  (assert (and (set? query) (= 1 (count query)) (-> query first map?)))
  (-> query first vals first))

(defn- rm-edges-and-nodes-from-query [query]
  (clojure.walk/prewalk (fn [x]
                          (if (and (set? x) (= 1 (count x)) (map? (first x)) (:edges (first x)))
                            (-> x first vals first first vals first)
                            x))
                        query))

(defn- hash-sets->vectors [query]
  (clojure.walk/prewalk (fn [x]
                          (if (set? x)
                            (into [] x)
                            x))
                        query))

(defn- ->venia-query [fragments query]
  (->> query
       (replace-fragments fragments)
       hash-sets->vectors
       snake_case_all
       venia/graphql-query))

(defn ->->graphql-query
  [fragments]
  (fn [query]
    (let [q {:venia/operation {:operation/type :query
                               :operation/name "venia"}
             :venia/queries query}]
      (->venia-query fragments q))))

(defn ->->graphql-mutation
  [fragments]
  (fn [query]
    (let [q {:venia/operation {:operation/type :query
                               :operation/name "venia"}
             :venia/queries query}]
      (-> (->venia-query fragments q)
          (replace-first #"query" "mutation")))))

(defn ->->pull-query
  [fragments back-refs]
  (fn [query]
    (->> query
         (replace-fragments fragments)
         vector-pairs->hash-map
         rm-first-layer
         rm-edges-and-nodes-from-query
         dash-case-all
         (forward-refs->back-refs back-refs)
         hash-sets->vectors)))
