(ns de.uni-koblenz.ist.macro-utils.core
  (:require [clojure.tools.macro :as tm]))

;;* Functions

(defn symb
  "Creates or returns a symbol named by concatenating the string
  representations of args.  The string representations of all args must not
  contain whitespaces."
  [& args]
  {:pre [(not (some #(re-matches #".*\s.*" (str %)) args))]}
  (symbol (apply str args)))

(defn bang-symbol?
  "Returns true, if sym is a symbol with name ending in a exclamation
  mark (bang)."
  [sym]
  (and (symbol? sym)
       (= (last (name sym)) \!)))

;;* Macros

(defmacro defmacro!
  "Defines a macro name with the given docstring, args, and body.
  All args ending in an exclamation mark (!, bang) will be evaluated only once
  in the expansion, even if they are unquoted at several places in body.  This
  is especially important for args whose evaluation has side-effecs or who are
  expensive to evaluate."
  {:arglists '([name doc-string? attr-map? [params*] body])}
  [name & args]
  (let [[name attrs] (tm/name-with-attributes name args)
        meta-map     (meta name)
        args         (first attrs)
        body         (next attrs)
        bang-syms    (filter bang-symbol? (flatten args))
        rep-map      (apply hash-map
                            (mapcat (fn [s] [s `(quote ~(gensym))])
                                    bang-syms))]
    `(defmacro ~name
       ~@(when (seq meta-map) [meta-map])
       ~args
       `(let ~~(vec (mapcat (fn [[s t]] [t s]) rep-map))
          ;; Note: We must use prewalk instead of postwalk so that we replace
          ;; from root to leaves, e.g., we favor replacement of larger s-exps
          ;; over replacements of only parts.
          ~(clojure.walk/prewalk-replace ~rep-map ~@body)))))

(defmacro! nif
  "Numeric if: evals test! (only once) and executes either pos, zero, or neg
  depending on the result."
  [test! pos zero neg]
  `(cond
    (pos? ~test!)  ~pos
    (zero? ~test!) ~zero
    :else          ~neg))

(defmacro dfn
  "Destructuring fn (aka dlambda): returns a function that dispatches on its
  first argument (a keyword) which other nested function to call.  The last
  nested fn may be without keyword.  Then calling the returned function calls
  that.

  Example: A simple stateful counter closure:

    => (let [c (atom 0)]
         (dfn
           :inc (fn ([] (swap! c inc))
                    ([i] (swap! c #(+ % i))))
           (fn [] @c)))
    => (*1 :inc)
    1
    => (*2 :inc)
    2
    => (*3)
    2" [& ds]
  (let [args (gensym "args")]
    `(fn [& ~args]
       (case (first ~args)
         ~@(mapcat (fn [[kw f]]
                     (if (and kw f)
                       ;; A normal keyword-fn case
                       `(~kw (apply ~f (next ~args)))
                       ;; default case: here kw is actually the function
                       `((apply ~kw (next ~args)))))
                   (partition 2 2 [nil] ds))))))

(defmacro aif
  "Anaphoric if: binds the test result to `it'.
  This is similar to (if-let [it test] then else), except that `it' is visible
  also in the else part, so that one can check if test resulted in nil or
  false."
  ([test then]
     `(let [~'it ~test]
        (if ~'it ~then)))
  ([test then else]
     `(let [~'it ~test]
        (if ~'it ~then ~else))))

(defmacro alet
  "Anaphoric let: evals the last (returned) expression first and binds it to an
  atom `this', and then evals the other expressions (which may then use the
  `this' binding)."
  [bindings & body]
  `(let [~@bindings
         ~'this (atom ~(last body))]
     ~@(butlast body)
     ~'@this))

