package org.riediger.plist;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.TimeZone;
import java.util.Vector;
import java.util.Map.Entry;

import org.xml.sax.Attributes;
import org.xml.sax.helpers.DefaultHandler;

public class PList extends DefaultHandler {

	static class StackEntry {
		String tag;

		boolean ignoreCharacters;

		String content;

		String lastKey;

		PListDict dict;

		Vector<Object> array;

		Object value;
	}

	static DateFormat dateFormat;

	static {
		dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
		dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
	}

	private Stack<StackEntry> objectStack;

	private PListDict dict;

	private PListException plistException;

	private String uri;

	public PList() {
		dict = new PListDict();
	}

	public PList(String uri) throws PListException {
		this.uri = uri;
		try {
			loadFrom(uri);
		} catch (PListException e) {
			if (e.getCause() instanceof FileNotFoundException) {
				dict = new PListDict();
			} else {
				throw e;
			}
		}
	}

	public void store() throws PListException {
		storeTo(uri);
	}

	public void store(boolean makeBackup) throws PListException {
		storeTo(uri, makeBackup);
	}

	public void loadFrom(String uri) throws PListException {
		try {
			// SAXParserFactory spf = SAXParserFactory.newInstance();
			// spf.setValidating(false);
			// spf.setFeature(
			// "http://apache.org/xml/features/nonvalidating/load-external-dtd",
			// false);
			// spf.setFeature(
			// "http://xml.org/sax/features/external-parameter-entities",
			// false);
			// spf.setFeature(
			// "http://xml.org/sax/features/lexical-handler/parameter-entities",
			// false);
			// SAXParser parser spf.newSAXParser();
			MinimalSAXParser parser = new MinimalSAXParser();
			plistException = null;
			parser.parse(uri, this);
		} catch (Exception e) {
			throw new PListException("Error reading property list '" + uri
					+ "'", e);
		}
		if (plistException != null) {
			throw plistException;
		}
	}

	public void storeTo(String uri) throws PListException {
		storeTo(uri, false);
	}

	public void storeTo(String uri, boolean makeBackup) throws PListException {
		PrintStream configFile;
		try {
			if (makeBackup) {
				File oldFile = new File(uri);
				if (oldFile.exists()) {
					File backupFile = new File(uri + "~");
					if (backupFile.exists()) {
						if (!backupFile.delete()) {
							throw new PListException("Can't delete old backup "
									+ backupFile.getCanonicalPath());
						}
					}
					if (!oldFile.renameTo(backupFile)) {
						throw new PListException("Can't rename "
								+ oldFile.getCanonicalPath() + " to "
								+ backupFile.getCanonicalPath());
					}
				}
			}
			configFile = new PrintStream(new FileOutputStream(uri), false,
					"UTF-8");
			print(configFile);
			configFile.close();
		} catch (Exception e) {
			throw new PListException("Error writing property list '" + uri
					+ "'", e);
		}
	}

	@Override
	public void startDocument() {
		objectStack = new Stack<StackEntry>();
	}

	@Override
	public void endDocument() {
		objectStack = null;
	}

	@Override
	public void characters(char[] ch, int start, int length) {
		if (objectStack != null && objectStack.size() > 0) {
			StackEntry e = objectStack.peek();
			if (!e.ignoreCharacters) {
				String s = new String(ch, start, length);
				e.content += s;
			}
		}
	}

	@Override
	public void startElement(String uri, String localName, String qName,
			Attributes attributes) {
		StackEntry e = new StackEntry();
		e.tag = qName;
		e.ignoreCharacters = qName.equals("plist") || qName.equals("dict")
				|| qName.equals("array") || qName.equals("true")
				|| qName.equals("false");
		if (!e.ignoreCharacters) {
			e.content = "";
		} else if (e.tag.equals("dict")) {
			e.dict = new PListDict();
		} else if (e.tag.equals("array")) {
			e.array = new Vector<Object>();
		}
		objectStack.push(e);
	}

	@Override
	public void endElement(String uri, String localName, String qName) {
		StackEntry e = objectStack.pop();
		StackEntry top = null;
		if (objectStack.size() > 0) {
			top = (objectStack.peek());
		}
		if (e.tag.equals("string")) {
			e.value = e.content;
		} else if (e.tag.equals("key")) {
			top.lastKey = e.content;
		} else if (e.tag.equals("integer")) {
			e.value = Integer.parseInt(e.content);
		} else if (e.tag.equals("real")) {
			e.value = Double.parseDouble(e.content);
		} else if (e.tag.equals("date")) {
			try {
				synchronized (dateFormat) {
					e.value = dateFormat.parse(e.content);
				}
			} catch (ParseException e1) {
				e1.printStackTrace();
			}
		} else if (e.tag.equals("data")) {
			plistException = new PListException(
					"property list element 'data' not yet implemented");
		} else if (e.tag.equals("true")) {
			e.value = true;
		} else if (e.tag.equals("false")) {
			e.value = false;
		} else if (e.tag.equals("dict")) {
			e.value = e.dict;
		} else if (e.tag.equals("array")) {
			e.value = e.array;
		}
		if (top != null) {
			if (top.dict != null && top.lastKey != null && !e.tag.equals("key")) {
				top.dict.put(top.lastKey, e.value);
				top.lastKey = null;
			} else if (top.array != null) {
				top.array.add(e.value);
			} else if (top.tag.equals("plist")) {
				dict = (PListDict) (e.value);
			}
		}
	}

	public void print(PrintStream out) {
		out.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		out
				.println("<!DOCTYPE plist PUBLIC \"-//Apple Computer//DTD PLIST 1.0//EN\" \"http://www.apple.com/DTDs/PropertyList-1.0.dtd\">");
		out.println("<plist version=\"1.0\">");
		print(out, dict, 1);
		out.println("</plist>");
	}

	@SuppressWarnings("unchecked")
	private void print(PrintStream out, Object o, int indent) {
		printIndent(out, indent);
		if (o instanceof PListDict) {
			Set<Map.Entry<String, Object>> entries = ((PListDict) (o))
					.entrySet();
			if (entries.size() == 0) {
				out.println("<dict/>");
			} else {
				out.println("<dict>");
				for (Iterator<Entry<String, Object>> iter = entries.iterator(); iter
						.hasNext();) {
					Map.Entry<String, Object> element = iter.next();
					for (int i = 0; i <= indent; ++i) {
						out.print('\t');
					}
					out.println("<key>" + xmlQuote(element.getKey().toString())
							+ "</key>");
					print(out, element.getValue(), indent + 1);
				}
				printIndent(out, indent);
				out.println("</dict>");
			}
		} else if (o instanceof List) {
			List<Object> l = (List<Object>) o;
			if (l.size() == 0) {
				out.println("<array/>");
			} else {
				out.println("<array>");
				for (Iterator<Object> iter = l.iterator(); iter.hasNext();) {
					Object element = iter.next();
					print(out, element, indent + 1);
				}
				printIndent(out, indent);
				out.println("</array>");
			}
		} else if (o instanceof Integer) {
			out.println("<integer>" + xmlQuote(o.toString()) + "</integer>");
		} else if (o instanceof String) {
			out.println("<string>" + xmlQuote(o.toString()) + "</string>");
		} else if (o instanceof Double) {
			out.println("<real>" + xmlQuote(o.toString()) + "</real>");
		} else if (o instanceof Boolean) {
			out.println("<" + o.toString() + "/>");
		} else if (o instanceof Date) {
			String s;
			synchronized (dateFormat) {
				s = dateFormat.format((Date) o);
			}
			out.println("<date>" + xmlQuote(s) + "</date>");
		}
	}

	private void printIndent(PrintStream out, int indent) {
		for (int i = 0; i < indent; ++i) {
			out.print('\t');
		}
	}

	private String xmlQuote(String string) {
		StringBuffer result = new StringBuffer();
		for (int i = 0; i < string.length(); ++i) {
			char c = string.charAt(i);
			if (c == '<') {
				result.append("&lt;");
			} else if (c == '>') {
				result.append("&gt;");
			} else if (c == '"') {
				result.append("&quot;");
			} else if (c == '\'') {
				result.append("&apos;");
			} else if (c == '&') {
				result.append("&amp;");
			} else {
				result.append(c);
			}
		}
		return result.toString();
	}

	public PListDict getDict() {
		return dict;
	}
}
