(ns de.uni-koblenz.funtg.greql-eval
  (:use de.uni-koblenz.funtg.core)
  (:use de.uni-koblenz.funtg.funql)
  (:use ordered.set)
  (:use ordered.map)
  (:import
   (de.uni_koblenz.jgralab.greql2.funlib FunLib)
   (de.uni_koblenz.jgralab.greql2.parser GreqlParser)
   (de.uni_koblenz.jgralab.greql2.evaluator.fa DFA NFA Transition)
   (de.uni_koblenz.jgralab.greql2.schema
    AggregationPathDescription AlternativePathDescription BackwardVertexSet
    BoolLiteral ConditionalExpression Declaration Definition Direction
    DoubleLiteral EdgePathDescription EdgeRestriction EdgeSetExpression
    EdgeSubgraphExpression ExponentiatedPathDescription ForwardVertexSet
    FunctionApplication FunctionId Greql2Expression Identifier IntLiteral
    IntermediateVertexPathDescription IteratedPathDescription LetExpression
    ListComprehension ListConstruction ListRangeConstruction Literal LongLiteral
    MapComprehension MapConstruction OptionalPathDescription
    PathExistence QuantifiedExpression Quantifier RecordConstruction RecordElement
    RecordId RoleId SequentialPathDescription SetComprehension SetConstruction
    SimpleDeclaration SimplePathDescription StringLiteral TableComprehension
    ThisEdge ThisVertex TransposedPathDescription TupleConstruction TypeId
    Variable VertexSetExpression VertexSubgraphExpression WhereExpression)))

(def ^:dynamic *vars* {})

(def ^:dynamic *graph*)

(def +funs+
  { "add"     +'
    "sub"     -'
    "mul"     *'
    "div"     /
    "mod"     mod
    "leThan"  <
    "grThan"  >
    "leEqual" <=
    "grEqual" >=
    "equals"  =
    "nequals"   not=
    "reMatch"   #(re-matches (re-matcher %1) %2)
    "or"        #(or %1 %2)
    "and"       #(and %1 %2)
    "getValue"  value
    "getVertex" #(vertex *graph* %)
    "getEdge"   #(edge *graph* %)
    "startVertex" alpha
    "endVertex"   omega})

(defn cartesian-product
  "All the ways to take one item from each sequence."
  [& seqs]
  (let [v-original-seqs (vec seqs)
	step
	(fn step [v-seqs]
	  (let [increment
		(fn [v-seqs]
		  (loop [i (dec (count v-seqs)), v-seqs v-seqs]
		    (if (== i -1) nil
			(if-let [rst (next (v-seqs i))]
			  (assoc v-seqs i rst)
			  (recur (dec i) (assoc v-seqs i (v-original-seqs i)))))))]
	    (when v-seqs
              (cons (map first v-seqs)
                    (lazy-seq (step (increment v-seqs)))))))]
    (when (every? first seqs)
      (lazy-seq (step v-original-seqs)))))

(defn parse-greql-query
  [q]
  (GreqlParser/parse q))

(defn greql2exp
  [g]
  (the (vseq g 'Greql2Expression)))

(defn funlibcall
  [fun args]
  ;; TODO: Implement me as soon as jvalue has been removed and Volker's new
  ;; funlib is in place...
  )

(defprotocol GReQLEvaluation
  "A protocol for evaluating greql syntax graphs."
  (evaluate [this]
    "Evals this GReQL vertex on the graph g."))

(defn eval-list-comprehension
  [c]
  (let [bindings (evaluate (the (adjs c :compDecl)))]
    (doall
     (for [b bindings]
       (binding [*vars* (merge *vars* b)]
         (evaluate (the (adjs c :compResultDef))))))))

(defn eval-map-comprehension
  [c]
  (let [bindings (evaluate (the (adjs c :compDecl)))]
    (apply ordered-map
           (flatten
            (doall
             (for [b bindings]
               (binding [*vars* (merge *vars* b)]
                 [(evaluate (the (adjs c :keyExpr)))
                  (evaluate (the (adjs c :valueExpr)))])))))))

(extend-protocol GReQLEvaluation
  AggregationPathDescription
  AlternativePathDescription
  BackwardVertexSet
  BoolLiteral
  (evaluate [this]
    (value this :boolValue))
  ConditionalExpression
  Declaration
  (evaluate [this]
    ;; TODO: Currently, this implements only Z-style evaluations, where all
    ;; bindings are created in one go and a latter variable must not refer to
    ;; previous variables.
    (let [vars2ranges (apply hash-map
                             (mapcat (fn [sd]
                                       (let [vars (adjs sd :declaredVar)
                                             val (evaluate (the (adjs sd :typeExpr)))]
                                         (vec (mapcat (fn [v] [v val]) vars))))
                                     (adjs this :simpleDecl)))
          vars (keys vars2ranges)
          ranges (vals vars2ranges)
          bindmaps (map #(zipmap vars %)
                        (apply cartesian-product ranges))]
      (let [constr (adjs this :constraint)]
        (doall (if (seq constr)
                 ;; TODO: Currently, this expects that there's only one constraint
                 (filter #(binding [*vars* (merge *vars* %)]
                            (evaluate (the constr)))
                         bindmaps)
                 bindmaps)))))
  Definition
  Direction
  DoubleLiteral
  (evaluate [this]
    (value this :doubleValue))
  EdgePathDescription
  EdgeRestriction
  EdgeSetExpression
  (evaluate [this]
    (let [restrs (map evaluate (adjs this :typeRestr))]
      (eseq *graph* restrs)))
  EdgeSubgraphExpression
  ExponentiatedPathDescription
  ForwardVertexSet
  FunctionApplication
  (evaluate [this]
    (let [args (map evaluate (adjs this :argument))
          fname (evaluate (the (adjs this :functionId)))]
      ;; TODO: Comment in, as soon as it works
      ;; (funlibcall fname args)
      (let [fun (+funs+ fname)]
        (if fun
          (let [r (apply fun args)]
            ;;(println fname args "==>" r)
            r)
          (throw (RuntimeException. (format "No such function: %s" fname)))))))
  FunctionId
  (evaluate [this]
    (value this :name))
  Greql2Expression
  (evaluate [this]
    (-> (adjs this :queryExpr) the evaluate))
  Identifier
  (evaluate [this]
    (value this :name))
  IntLiteral
  (evaluate [this]
    (value this :intValue))
  IntermediateVertexPathDescription
  IteratedPathDescription
  LetExpression
  ListComprehension
  (evaluate [this]
    (eval-list-comprehension this))
  ListConstruction
  ListRangeConstruction
  (evaluate [this]
    (range (-> (adjs this :firstValue) the evaluate)
           ;; list(1..10) includes 10 in contrast to range
           (inc (-> (adjs this :lastValue) the evaluate))))
  LongLiteral
  (evaluate [this]
    (value this :longValue))
  MapComprehension
  (evaluate [this]
    (eval-map-comprehension this))
  MapConstruction
  OptionalPathDescription
  PathExistence
  QuantifiedExpression
  (evaluate [this]
    (let [quantifier (str (evaluate (the (adjs this :quantifier))))
          bindings (evaluate (the (adjs this :quantifiedDecl)))
          default (if (= quantifier "EXISTS") false true)]
      ;; TODO: Implement EXISTS! (exactly one)
      (loop [b bindings]
        (if (seq b)
          (let [v (binding [*vars* (merge *vars* (first b))]
                    (evaluate (the (adjs this :boundExprOfQuantifier))))]
            (cond
             (and v (= quantifier "EXISTS"))       true
             (and (not v) (= quantifier "FORALL")) false
             :else (recur (rest b))))
          default))))
  Quantifier
  (evaluate [this]
    (value this :type))
  RecordConstruction
  RecordElement
  RecordId
  RoleId
  SequentialPathDescription
  SetComprehension
  (evaluate [this]
    (into-oset (eval-list-comprehension this)))
  SetConstruction
  (evaluate [this]
    (into-oset (map evaluate (adjs this :part))))
  SimpleDeclaration ;; not evaled
  SimplePathDescription
  StringLiteral
  (evaluate [this]
    (value this :stringValue))
  TableComprehension
  ThisEdge
  ThisVertex
  TransposedPathDescription
  TupleConstruction
  (evaluate [this]
    (doall (map evaluate (adjs this :part))))
  TypeId
  (evaluate [this]
    (str (when (value this :excluded) "!")
         (value this :name)
         (when (value this :type) "!")))
  Variable
  (evaluate [this]
    (*vars* this))
  VertexSetExpression
  (evaluate [this]
    (let [restrs (map evaluate (adjs this :typeRestr))]
      (vseq *graph* restrs)))
  VertexSubgraphExpression
  WhereExpression)

(defn greql-eval
  [dg q]
  (let [sg (parse-greql-query q)]
    ;;(show-graph sg true)
    (binding [*graph* dg]
      (let [r (evaluate (greql2exp sg))]
        (if (seq? r)
          (doall r)
          r)))))

;; (greql-eval nil "from x, y: list(1..3) with x > y reportMap x -> y end")
;; (greql-eval nil "set(5, 7 ,1 , 4, 7)")
;; (greql-eval nil "V{Foo, Bar}")
;; (greql-eval nil "from x: list(1..5), y: list(1..10) with x * x < y report x, y, x*y end")

