(ns ^{:long-doc
      "Getting Started
===============

For a basic understanding of logic programming using miniKanren, have a look
here:

  http://tinyurl.com/czn2nxz

The first thing you want to do is to generate relations (derived from the
schema) and populate them with facts (from the graph).  Here's how to do this.

First, we require the FunTC core namespace with prefix core, so that we can use
its general graph loading function.

  user> (require '[de.uni-koblenz.ist.funtg.core :as core])

Then, we load the GReQL test graph and bind it to a var `g'.

  user> (def g (core/load-graph \"/home/horn/uni/repos/funtg/test/greqltestgraph.tg\"))

We import the FunRL namespace.

  user> (use 'de.uni-koblenz.ist.funtg.funrl)

Now, we can generate relations and facts from our graph and its schema.

  user> (graph->facts g)

This procedure created 2 relations per attributed element class of the schema
of our graph.  For any vertex class Foo, there's a relation (+Foo! x) which
succeeds if `x' can be unified with a vertex of exact type Foo.  Additionally,
there's a relation (+Foo x) which succeeds if `x' can be unified with a vertex
of exact type Foo or a subtype thereof.

For an edge class Bar, there is the relation (+Bar! e a o) which succeeds, if
`e' can be unified with an edge of exact type Bar, and `a' and `o' can be
unified with the start and end vertex of `e'.  Similarly to vertex classes, a
subtype-accepting relation +Bar is also created with the same signature.

For any attribute baz, a relation (+baz el val) is generated, which succeeds,
if the attributed element `el' has the value `val' for its baz attribute.

The relations are automatically generated into a namespace corresponding to the
schema qualified name.  So we import them into our user namespace (using
`refer' instead of `use', because that namespace is not in a class or clj file
on the CLASSPATH).

  user> (refer 'de.uni_koblenz.jgralabtest.schemas.greqltestschema.RouteSchema)

Finally, we import the standard logic library.

  user> (use 'clojure.core.logic)

Now, we are ready to go.

Asking Questions
================

After we've populated our user namespace with relations and facts about the
route graph, we can ask questions.  That's done with the `run' and `run*'
macros provided by clojure.core.logic.  For example

  (run 3 [q] <goals>)

returns at most 3 answers unifying `q' with the given goals, or () if there's
no answer at all, and

  (run* [q] <goals>)

returns all possible answers.

Now, lets ask what's the capital of the Village Kammerforst.

  user> (run* [q]
          (fresh [kammerforst county e1 e2]
            (+Village kammerforst)
            (+name kammerforst \"Kammerforst\")
            (+ContainsLocality e1 county kammerforst)
            (+HasCapital e2 county q)))

  (#<CityImpl v6: localities.City>)

We use `run*' because we want to get all answers.  `fresh' introduces new logic
vars that should be unified.  In its body, we declare that `kammerforst' has to
be unified with a Village vertex, whose name is \"Kammerforst\".  Furthermore,
there has to be a ContainsLocality edge `e1' starting at some `county' and
leading to `kammerforst'.  `county' has to be the capital of `q', which is
exactly what we wanted to ask.  Because `kammerforst' and `county' occur
multiple times, they are subject to unification.  Likewise, our question `q' is
unified with the end vertex of the edge `e2'.

Now let's try to pose a question about what capitals reign which localities,
e.g., what are the localities contained in the county of some capital, for all
capitals.  We want to get all pairs of the form [capital-name locality-name] as
answer.

Here, we use the `with-fresh' macro for convenience.  It creates one fresh
logic variable for any symbol in its body starting with a question mark (?).
Additionally, it creates one anonymous fresh logic variable per occurence of
`_'.  In the former example, we had the logic vars `e1' and `e2' explicit,
although we never unified them with some other var.  So `_' is a shortcut for
'I don't care for anything except existence'.

  user> (run* [q]
          (with-fresh
            (+Locality ?loc)
            (+ContainsLocality _ ?county ?loc)
            (+HasCapital _ ?county ?capital)
            (!= ?capital ?loc)
            (+name ?capital ?cname) (+name ?loc ?lname)
            (== q [?cname ?lname])))

  ([\"Main\" \"Lautzenhausen\"] [\"Main\" \"Montabaur\"]
   [\"Main\" \"Flughafen Frankfurt-Hahn\"] [\"Main\" \"Winningen\"]
   [\"Main\" \"Koblenz\"] [\"Main\" \"Kammerforst\"]
   [\"Frankfurt am Main\" \"Frankfurt-Flughafen\"]
   [\"Main\" \"Flugplatz Koblenz-Winningen\"] [\"Main\" \"Höhr-Grenzhausen\"])

Looks like in the graph we've misspelled Mainz as Main, but anyway.  We say
that `?loc' has to be a Locality that is contained in `?county' that in turn
has a `?capital'.  We don't want to get the captial as ruled by itself, so we
declare that `?capital' and `?loc' must not be unified with each other.

To define our result, we declare `?cname' and `?lname' to be the names of
`?capital' and `?loc', respectively.  Finally, we declare that our question `q'
should be unified with a vector containing `?cname' and `?lname'.

Custom Relations
================

We think that being able to query for the capital of a location or the
locations of some capital is a thing we're going to do frequently.  So we can
factor that out into a custom relation `(capital c l)' that succeeds if `c' is
the captial of `l' simply by defining a function.

  user> (defn capital
          \"Succeeds, if c is the capital of Locality l.\"
          [c l]
          (with-fresh
            (+Locality l)
            (+ContainsLocality _ ?county l)
            (+HasCapital _ ?county c)
            (!= c l)))

We can pose our question now using this new relation and still get the same
answer.

  user> (run* [q]
          (with-fresh
            (capital ?capital ?loc)
            (+name ?capital ?cname)
            (+name ?loc ?lname)
            (== q [?cname ?lname])))

Now, let's do something a bit more interesting.  When are two junctions `j1'
and `j2' connected?  Either they are directly connected (j1 -> j2, or j2 ->
j1), or they are connected with something in between.  So we need some
disjunction here, and that's exactly what `core.logic/conde' is for.  `conde'
succeeds if any of its clauses succeeds, each a conjunction given as a list.

  (defn connected
    [j1 j2]
    (with-fresh
      (conde
       ;; A direct connection, either from j1->j2
       ((+Connection _ j1 j2))
       ;; or the other way round...
       ((+Connection _ j2 j1))
       ;; or an indirect connection, i.e, there's another crossroad in the
       ;; middle.
       ((connected j1 ?middle)
        (connected ?middle j2)))))"}

  de.uni-koblenz.ist.funtg.funrl
  "FunRL: Querying graphs using relational programming."
  (:refer-clojure :exclude [==])
  (:use [clojure.core.logic])
  (:require [de.uni-koblenz.ist.funtg.core :as c])
  (:require [de.uni-koblenz.ist.funtg.funql :as q])
  (:import
   (de.uni_koblenz.jgralab Vertex Edge AttributedElement)
   (de.uni_koblenz.jgralab.schema AggregationKind Schema Domain RecordDomain
                                  AttributedElementClass NamedElement
                                  GraphClass VertexClass EdgeClass Attribute
                                  GraphElementClass)))

(defn- elem-class->rel-symbols
  [^AttributedElementClass c]
  (let [n (str "+" (.getSimpleName c))]
    [(symbol n) (symbol (str n "!"))]))

(defn- attr-or-role-name->rel-symbol
  [n]
  (symbol (str "+" n)))

(defn subtype-relation [^GraphElementClass gec sym sym!]
  (let [subs (seq (.getDirectSubClasses gec))
        args (if (instance? VertexClass gec) `[~'x] `[~'x ~'a ~'o])]
    (if (seq subs)
      `(defn ~sym ~args
         (conde
          ((~sym! ~@args))
          ~@(map (fn [sub]
                   (let [[s _] (elem-class->rel-symbols sub)]
                     `((~s ~@args))))
                 subs)))
      `(def ~sym ~sym!))))

(defn schema->relations
  "Populates the namespace corresponding to schema `s' qname with relations
  expressing the schema."
  [^Schema s]
  (let [atts (atom #{})
        schema-ns (symbol (.getQualifiedName s))
        old-ns *ns*
        code `(do
                (ns ~schema-ns
                  (:refer-clojure :exclude [~'==])
                  (:use [clojure.core.logic]))
                ~@(doall
                   (mapcat (fn [^VertexClass vc]
                             (let [[sym sym!] (elem-class->rel-symbols vc)]
                               (swap! atts clojure.set/union
                                      (set (map #(.getName ^Attribute %)
                                                (.getOwnAttributeList vc))))
                               `((defrel ~sym! ~'v)
                                 ~(subtype-relation vc sym sym!))))
                           (reverse
                            (filter #(not (.isInternal %))
                                    (seq (-> s .getVertexClassesInTopologicalOrder))))))
                ~@(doall
                   (mapcat (fn [^EdgeClass ec]
                             (let [[sym sym!] (elem-class->rel-symbols ec)]
                               (swap! atts clojure.set/union
                                      (set (map #(.getName ^Attribute %)
                                                (.getOwnAttributeList ec))))
                               `((defrel ~sym! ~'e ~'a ~'o)
                                 ~(subtype-relation ec sym sym!))))
                           (reverse
                            (filter #(not (.isInternal %))
                                    (seq (-> s .getEdgeClassesInTopologicalOrder))))))
                ~@(doall
                   (for [^Attribute a @atts]
                     `(defrel ~(attr-or-role-name->rel-symbol a)
                        ~'e ~'v))))]
    (clojure.pprint/pprint code)
    (eval code)
    (in-ns (ns-name old-ns))
    schema-ns))

(defn graph->facts
  "Populates the relations with facts about graph `g'.
  If there are no relations, then create them now (using `schema->relations')."
  [g]
  (let [schema-ns (or (find-ns (symbol (.getQualifiedName ^Schema (c/schema g))))
                      ;; If there's no schema ns, then create one.
                      (find-ns (schema->relations (c/schema  g))))
        resolve (fn [sym] (ns-resolve schema-ns sym))]
    ;; VERTICES
    (doseq [^Vertex v (q/vseq g)]
      (let [aec (.getAttributedElementClass v)
            [sym sym!] (elem-class->rel-symbols aec)]
        (fact @(resolve sym!) v))
      ;; ATTRS
      (doseq [[a val] (:slots (q/describe v))]
        (fact @(resolve (symbol (str "+" (name a)))) v val)))
    ;; EDGES
    (doseq [^Edge e (q/eseq g)]
      (let [aec (.getAttributedElementClass e)
            [sym sym!] (elem-class->rel-symbols aec)
            a (c/alpha e)
            o (c/omega e)]
        (fact @(resolve sym!) e a o))
      ;; ATTRS
      (doseq [a (map #(.getName ^Attribute %)
                     (-> e .getAttributedElementClass .getAttributeList))]
        (fact @(resolve (symbol (str "+" a))) e (c/value e a))))))

(defn- qmark-symbol?
  "Returns true, if sym is a symbol with name starting with a question mark."
  [sym]
  (and (symbol? sym)
       (= (first (name sym)) \?)))

(defmacro with-fresh
  "Replace all symbols with a leading question mark with fresh lvars.
  In addition, all occurences of `_' are replaced with fresh lvars, one per
  occurence.  That means, that in `forms' all occurences of ?foo will be
  unified, but all occurences of `_' are not."
  [& forms]
  (let [fs (clojure.walk/postwalk #(if (= '_ %) (gensym "?") %) forms)
        qsyms (vec (distinct (filter qmark-symbol? (flatten fs))))]
    `(fresh ~qsyms
       ~@fs)))


(comment
  ;; SETUP
  (require '[de.uni-koblenz.ist.funtg.core :as core])
  (def g (core/load-graph "/home/horn/uni/repos/funtg/test/greqltestgraph.tg"))
  (use 'de.uni-koblenz.ist.funtg.funrl)
  (graph->facts g)  ;; Will create the relations as well
  (refer 'de.uni_koblenz.jgralabtest.schemas.greqltestschema.RouteSchema)
  (use 'clojure.core.logic) ;; import the standard logic stuff

  ;; All Capitals of the County in which the Village Kammerforst is located in.
  (run* [q]
    (with-fresh
      (+Village ?kammerforst)
      (+name ?kammerforst "Kammerforst")
      (+ContainsLocality _ ?county ?kammerforst)
      (+HasCapital _ ?county q)))

  ;; All Capitals with their reigned Localities (names as result)
  (run* [q]
    (with-fresh
      (+Locality ?loc)
      (+ContainsLocality _ ?county ?loc)
      (+HasCapital _ ?county ?capital)
      (!= ?capital ?loc)
      (+name ?capital ?cname)
      (+name ?loc ?lname)
      (== q [?cname ?lname])))

  ;; We can factor out the is-capital-of relation as a function
  (defn capital
    "Succeeds, if c is the capital of l."
    [c l]
    (with-fresh
      (+Locality l)
      (+ContainsLocality _ ?county l)
      (+HasCapital _ ?county c)
      (!= c l)))

  ;; Now we can refactor our query
  (run* [q]
    (with-fresh
      (capital ?capital ?loc)
      (+name ?capital ?cname) (+name ?loc ?lname)
      (== q [?cname ?lname])))

  (defn connected
    "Succeeds, if the junctions j1 and j2 are connected by connections (Ways,
    Streets, AirRoutes)."
    [j1 j2]
    (with-fresh
      (conde
       ;; A direct connection, either from j1->j2
       ((+Connection _ j1 j2))
       ;; or the other way round...
       ((+Connection _ j2 j1))
       ;; or an indirect connection, i.e, there's another crossroad in the
       ;; middle.
       ((connected j1 ?middle)
        (connected ?middle j2)))))

  (defn connected-locs
    "Succeeds, if the localities l1 and l2 are connected.
    Localities are connected, if they contain crossroads that are connected by
    connections (Ways, Streets, AirRoutes)."
    [l1 l2]
    (with-fresh
      (conde
       ;; Airports can be connected by AirRoutes directly
       ((+Airport! l1)
        (+Airport! l2)
        (connected l1 l2))
       ;; Everything else is connected by streets connecting crossroads that
       ;; are contained in localities.
       ((+ContainsCrossroad _ l1 ?c1)
        (connected ?c1 ?c2)
        (+ContainsCrossroad _ l2 ?c2)))))

  ;; What locality is connected with what other locality?
  (run 3 [q]
    (with-fresh
      (+Locality ?l1)
      (+Locality ?l2)
      (!= ?l1 ?l2)
      (connected-locs ?l1 ?l2)
      (== q [?l1 ?l2]))))
