(ns ^{:long-doc
      "Here's an example that evaluates a binary tree by replacing all binary
operations whose arguments are constants with a new constant with calculated
value.

  ;; An evaluation protocol
  (defprotocol BinTreeEval (eval-exp [this]))

  ;; Extend the protocol to the classes of the schema
  (let [g (bin-tree) ;; returns an example binary tree
        eval-args #(map eval-exp (--> % 'HasArg))]
    (extend-type (m1class g 'Const) BinTreeEval
      (eval-exp [c] (value c :value)))
    (extend-type (m1class g 'Add)   BinTreeEval
      (eval-exp [b] (reduce + (eval-args b))))
    (extend-type (m1class g 'Sub)   BinTreeEval
      (eval-exp [b] (reduce - (eval-args b))))
    (extend-type (m1class g 'Mul)   BinTreeEval
      (eval-exp [b] (reduce * (eval-args b))))
    (extend-type (m1class g 'Div)   BinTreeEval
      (eval-exp [b] (reduce / (eval-args b)))))

  ;; Here's the single transformation rule
  (defrule replace-binaryop
    \"Replaces a binary operation with constant args with
    a constant of the result.\"
    [g] [b     (vseq g 'BinaryOp)
         :let [[a1 a2] (vec (--> b 'HasArg))]
         :when (has-type? a1 'Const)
         :when (has-type? a2 'Const)]
    (let [c (create-vertex! g 'Const)]
      (set-value! c :value (eval-exp b))
      (relink! b c nil :in))
    (delete! b a1 a2))

  ;; Transform the graph
  (replace-binaryop (bin-tree))"}
    de.uni-koblenz.ist.funtg.funml
  "FunML: The Functional Matching Language for in-place graph transformations."
  (:use de.uni-koblenz.ist.funtg.core)
  (:use [de.uni-koblenz.ist.funtg.funql :only [member?]]))

;;** Matching

(defn- bindings-to-arglist [bindings]
  (loop [p bindings l []]
    (if (seq p)
      (cond
       ;; Handle :let [x y, z a]
       (= :let (first p)) (recur (rest (rest p))
                                 (concat l
                                         (loop [ls (first (rest p)) bs []]
                                           (if (seq ls)
                                             (recur (rest (rest ls))
                                                    (conj bs (first ls)))
                                             bs))))
       ;; Ignore :when (exp ...)
       (keyword? (first p)) (recur (rest (rest p)) l)
       ;; Don't allow destructuring
       (coll? (first p)) (throw (RuntimeException.
                                 "Destructuring not allowed outside :let"))
       ;; That's a normal binding
       :default (recur (rest (rest p)) (conj l (first p))))
      (vec l))))

(defmacro with-match
  "Establish bindings as specified in `bindings', and execute `body'.
  `bindings' is a vector of bindings with the syntax of `for'.

  If a match could be found, `body' is executed with the established bindings and
  `body's value is the return value.  If no match is found, nil is returned."
  ;; Nicer arglist in doc
  {:arglists '([[bindings*] & body])}
  [bindings & body]
  (when (not= 0 (mod (count bindings) 2))
    (throw (RuntimeException. "bindings has to be var-exp pairs")))
  (let [arglist (bindings-to-arglist bindings)
        r `r#]
    `(when-let [~r (first (for ~bindings ~arglist))]
       (let ~(loop [a arglist, i 0, res []]
               (if (seq a)
                 (recur (rest a) (inc i) (concat res [(first a) `(~r ~i)]))
                 (vec res)))
         ~@body))))

(defn- mangle-let-vector
  [init args]
  (loop [m init, r []]
    (if (seq m)
      (let [v (first m)]
        (if (member? v (flatten args))
          (recur (rest (rest m))
                 (concat r [v `(or ~v ~(first (rest m)))]))
          (recur (rest (rest m))
                 (concat r [v (first (rest m))]))))
      (vec r))))

(defn- mangle-match-vector
  "In init (a for bindings vector) replace the expressions of vars contained in
  args with `(if var [var] <original-exp>)' so that parameters to a rule
  override the matches if provided.

  :let [v (foo)] will be transformed to :let [v (or v (foo))] if v is in args."
  [init args]
  (loop [m init, r []]
    (if (seq m)
      (let [v (first m)]
        (cond
         ;; v is a :let ==> mangle the :let vector
         (= v :let)
         (recur (rest (rest m))
                (concat r [v (mangle-let-vector (first (rest m)) args)]))

         ;; v is a member of args
         (member? v (flatten args))
         (recur (rest (rest m))
                (concat r [v `(if ~v [~v] ~(first (rest m)))]))

         ;; Anything else stays as it is
         :default (recur (rest (rest m))
                         (concat r [v (first (rest m))]))))
      (vec r))))

(defmacro defrule
  "Defines a rule function with `name', `doc', and `args'.
  `match' describes what it matches (a vector with the syntax of `for') and
  `body' is applied on the match.  If the rule could be applied, then it
  returns the value of the last form in body or true if that returned false.
  Thus one can rely on a rule to always return logical true if it was applied.

  If the `args' contain vars that occur also in `match', then passed `args'
  override the matched ones.  So if one rule creates a structure that the next
  rule matches, you can pass that directly.  Any :when clauses in `match' will
  be checked anyway, and if the passed `args' don't satisfy those constraints,
  the match fails and the rule won't be applied."
  [name doc args match & body]
  (let [matchvec (mangle-match-vector match args)]
    `(defn ~name
       ~(str "Rule:\n  " doc)
       ~args
       (with-match ~matchvec
         (or (do ~@body) true)))))

(defn iteratively
  "Applies the function `f' with `args' as long as it returns logical true.
  Returns the number of successful applications or nil if it couldn't be
  applied at least once."
  [f & args]
  (loop [val (apply f args), i 0]
    (if val
      (recur (apply f args) (inc i))
      (if (zero? i) nil i))))

(defn iteratively*
  "Applies the function `f' as long as it returns logical true.
  On the first application, `f' receives `args'.  The second till last
  application receive the value of the previous successful application.
  Returns the number of successful applications or nil if it couldn't be
  applied at least once."

  [f & args]
  (loop [val (apply f args), i 0]
    (if val
      (recur (apply f val) (inc i))
      (if (zero? i) nil i))))

(defn choose
  "Randomly chooses one of the given `fns' and applies it.
  Returns that fun's return value or nil, if no fun was applicable."
  [& fns]
  (let [f (rand-nth fns)
        v (f)]
    (or v (recur (remove #(= f %) fns)))))
