(ns de.uni-koblenz.ist.funtg.funql
  (:use de.uni-koblenz.ist.funtg.core)
  (:use ordered.set)
  (:require clojure.set)
  (:import
   (de.uni_koblenz.jgralab.algolib.algorithms.search IterativeDepthFirstSearch)
   (de.uni_koblenz.jgralab.algolib.functions.entries PermutationEntry)
   (de.uni_koblenz.jgralab.algolib.problems TopologicalOrderSolver AcyclicitySolver)
   (de.uni_koblenz.jgralab.algolib.algorithms.topological_order KahnKnuthAlgorithm TopologicalOrderWithDFS)
   (de.uni_koblenz.jgralab Graph Vertex Edge AttributedElement TraversalContext)
   (de.uni_koblenz.jgralab.schema Attribute RecordDomain GraphClass
                                  GraphElementClass AttributedElementClass VertexClass EdgeClass
                                  AggregationKind)))


;;* Utils

(defn into-oset
  "Returns to converted into an ordered-set containing all elements of all froms."
  [to & froms]
  (loop [t (if (instance? ordered.set.OrderedSet to)
             to
             (into (ordered-set) (if (coll? to) to [to])))
         fs froms]
    (if (seq fs)
      (recur (into t (first fs)) (rest fs))
      t)))

;;* Lazy Vertex, Edge, Incidence Seqs

(defprotocol VSeq
  "Protocol for types supporting vseq."
  (vseq-internal [this tm]
	"Returns a lazy seq of the graphs vertices restricted by type matcher tm."))

(extend-protocol VSeq
  Graph
  (vseq-internal
    [g tm]
    (lazy-seq
     (let [f (first-vertex g tm)]
       (and f (cons f (vseq-internal f tm))))))
  Vertex
  (vseq-internal
    [v tm]
    (lazy-seq
     (let [n (next-vertex v tm)]
       (and n (cons n (vseq-internal n tm)))))))

(alter-meta! (var vseq-internal) assoc :private true)

(defn vseq
  "Returns the lazy seq of vertices of g restricted by cls.
  g may be a graph or a vertex.  In the latter case, returns all vertices
  following g in the vertex sequence."
  ([g]
     (vseq-internal g identity))
  ([g cls]
     (vseq-internal g (type-matcher g cls))))

(defprotocol ESeq
  "Protocol for types supporting eseq."
  (eseq-internal [this tm]
	"Returns a lazy seq of the graph's edges restricted by tm."))

(extend-protocol ESeq
  Graph
  (eseq-internal
    [g tm]
    (lazy-seq
     (let [f (first-edge g tm)]
       (and f (cons f (eseq-internal f tm))))))

  Edge
  (eseq-internal
    [e tm]
    (lazy-seq
     (let [n (next-edge e tm)]
       (and n (cons n (eseq-internal n tm)))))))

(alter-meta! (var eseq-internal) assoc :private true)

(defn eseq
  "Returns the lazy seq of edges of e restricted by cls.
  g may be a graph or an edge.  In the latter case, returns all edges following
  g in the edge sequence."
  ([g]
     (eseq-internal g identity))
  ([g cls]
     (eseq-internal g (type-matcher g cls))))

(defprotocol ISeq
  "Protocol for types supporting iseq."
  (iseq-internal [this tm dm]
    "Returns a lazy seq of incident edges restricted by tm and dm."))

(extend-protocol ISeq
  Vertex
  (iseq-internal
    [v tm dm]
    (lazy-seq
     (let [f (first-inc v tm dm)]
       (and f (cons f (iseq-internal f tm dm))))))

  Edge
  (iseq-internal
    [e tm dm]
    (lazy-seq
     (let [n (next-inc e tm dm)]
       (and n (cons n (iseq-internal n tm dm)))))))

(alter-meta! (var iseq-internal) assoc :private true)

(defn iseq
    "Returns the lazy seq of incidences of v restricted by cls and dir.
  v may be a vertex or an edge.  In the latter case, returns all incidences
  following v in the current vertex's incidence sequence."
  ([v]
     (iseq-internal v identity identity))
  ([v cls]
     (iseq-internal v (type-matcher v cls) identity))
  ([v cls dir]
     (iseq-internal v (type-matcher v cls) (direction-matcher dir))))

;;* Vertex, edge counts

(defprotocol GraphCounts
  "Protocol for getting vertex and edge counts, possibly restricted by a type
  spec."
  (vcount [g] [g cls]
    "Returns the vertex count oy g.")
  (ecount [g] [g cls]
    "Returns the edge count of g."))

(extend-protocol GraphCounts
  Graph
  (vcount
    ([g]     (.getVCount g))
    ([g cls] (count (vseq g cls))))
  (ecount
    ([g]     (.getECount g))
    ([g cls] (count (eseq g cls)))))


;;* Traversal Context

(defmacro with-traversal-context
  "Sets the TraversalContext of g to tc and then executes body.  Guaranteed to
  restore the old TraversalContext after body finished (even if it errored).

  Also see: vsubgraph, esubgraph"
  [[g tc] & body]
  `(let [^Graph g# ~g
         ^TraversalContext old-tc# (.getTraversalContext g#)]
     (try
       (.setTraversalContext g# ~tc)
       ~@body
       (finally (.setTraversalContext g# old-tc#)))))

(defn merge-traversal-contexts
  "Returns a TraversalContext that accepts only elements that are accepted by
  both tc1 and tc2."
  [^TraversalContext tc1 ^TraversalContext tc2]
  (cond
   (nil? tc1) tc2
   (nil? tc2) tc1
   :else (reify TraversalContext
           (containsVertex [_ v]
             (and (.containsVertex tc1 v)
                  (.containsVertex tc2 v)))
           (containsEdge [_ e]
             (and (.containsEdge tc1 e)
                  (.containsEdge tc2 e))))))

(defmacro with-merged-traversal-context
  "Sets the TraversalContext of g to a new TraversalContext that accepts only
  elements which both tc and g's current TraversalContext accept and then
  executes body.  Guaranteed to restore the old TraversalContext.

  Also see: vsubgraph, esubgraph"
  [[g tc] & body]
  `(let [^Graph g# ~g
         ^TraversalContext old-tc# (.getTraversalContext g#)]
     (try
       (.setTraversalContext g# (merge-traversal-contexts old-tc# ~tc))
       ~@body
       (finally (.setTraversalContext g# old-tc#)))))

(defn- vsubgraph-tc
  "Returns a TraversalContext of a vertex induced subgraph restricted by pred
  on the vertices.  All vertices satisfying pred are accepted plus all edges
  between accepted vertices."
  [pred]
  (reify TraversalContext
    (containsVertex [_ v]
      (boolean (pred v)))
    (containsEdge [_ e]
      (boolean (and (pred (alpha e))
                    (pred (omega e)))))))

(defn vsubgraph
  "Returns an induced subgraph of g restricted by pred in terms of a TraversalContext.
  pred may be a predicate that is used to filter the vertices of g or a type
  specification (see type-spec) or a collection of vertices.  The subgraph
  contains all vertices matching the predicate, and all edges/incidences that
  are connected to vertices that are both in the subgraph."
  [g pred]
  (cond
   (fn? pred)        (vsubgraph-tc pred)
   (type-spec? pred) (vsubgraph-tc (type-matcher g pred))
   (coll? pred)      (vsubgraph-tc #(member? % pred))
   :default          (throw (RuntimeException.
                             (str "Don't know how to handle predicate " pred)))))

(defn- esubgraph-tc
  "Returns a TraversalContext of an edge induced subgraph restricted by pred on
  the edges.  All edges satisfying pred are accepted plus all vertices that are
  connected to accepted edges."
  [pred]
  (reify TraversalContext
    (containsVertex [this v]
      (boolean (some #(.containsEdge ^TraversalContext this ^Edge (normal-edge %))
                     (iseq v))))
    (containsEdge [_ e]
      (boolean (pred (normal-edge e))))))

(defn esubgraph
  "Returns an induced subgraph of g restricted by pred in terms of a TraversalContext.
  pred may be a predicate that is used to filter the edges of g or a type
  specification (see type-spec) or a collection of edges.  The subgraph
  contains all vertices matching the predicate, and all edges/incidences that
  are connected to vertices that are both in the subgraph."
  [g pred]
  (cond
   (fn? pred)        (esubgraph-tc pred)
   (type-spec? pred) (esubgraph-tc (type-matcher g pred))
   (coll? pred)      (esubgraph-tc #(member? % pred))
   :default          (throw (RuntimeException.
                             (str "Don't know how to handle predicate " pred)))))

;;* Adjacences

(defn adjs
  "Get vertices adjacent to v via role, or vertices reachable by traversing role
  and more roles.  The role may be given as symbol, keyword, or string."
  ([^Vertex v role]
     (try
       (.adjacences v (name role))
       (catch Exception e
	 (throw (RuntimeException.
		 ^String (format "No role %s at %s." (name role) v))))))
  ([^Vertex v role & more]
     (let [ads (adjs v role)]
       (if (seq more)
         (mapcat #(apply adjs % (first more) (rest more)) ads)
	 ads))))

;;* Path Functions

(defn p-apply
  "Applies the path description p on v and returns the result as ordered set."
  [v p]
  (cond
   ;; funs: -->
   (fn? p) (p v)
   ;; funs with params: [--> 'Foo], [p-alt --> <>--]
   (coll? p) (apply (first p) v (rest p))
   ;; adjacences / that-role names
   (qname? p) (into-oset (mapcat #(adjs % p) v))
   :else (throw (RuntimeException.
		 ^String (format "Don't know how to apply %s." p)))))

(defn- ---
  "Returns the vertices reachable from v via incidences with direction dir and
  aggregation kinds, restricted by cls, and pred (on the edges)."
  [v dir this-aks that-aks cls pred]
  (if (or (nil? v)
	  (and (coll? v) (empty? v)))
    (ordered-set)
    (let [pred (or pred identity)
          this-a (if (seq this-aks)
                   #(member? (.getThisAggregationKind ^Edge %) this-aks)
                   identity)
          that-a (if (seq that-aks)
                   #(member? (.getThatAggregationKind ^Edge %) that-aks)
                   identity)
          complete-pred (every-pred pred this-a that-a)]
      (into-oset
       (mapcat (fn [sv]
                 (map that
                      (filter complete-pred
                              (iseq sv cls dir))))
               (into-oset v))))))

(defn -->
  "Returns the vertices reachable from v via outgoing incidences,
  optionally restricted by cls and pred (on the edges)."
  ([v]
     (--> v nil nil))
  ([v cls]
     (--> v cls nil))
  ([v cls pred]
     (--- v :out nil nil cls pred)))

(defn <--
  "Returns the vertices reachable from v via incoming incidences,
  optionally restricted by cls and pred (on the edges)."
  ([v]
     (<-- v nil nil))
  ([v cls]
     (<-- v cls nil))
  ([v cls pred]
     (--- v :in nil nil cls pred)))

(defn <->
  "Returns the vertices reachable from v via all incidences,
  optionally restricted by cls and pred (on the edges)."
  ([v]
     (<-> v nil nil))
  ([v cls]
     (<-> v cls nil))
  ([v cls pred]
     (--- v :inout nil nil cls pred)))

(defn --->
  "Returns the vertices reachable from v via outgoing incidences,
  optionally restricted by cls and pred (on the edges).  In contrast to -->,
  travesal of edges with aggregation semantics is forbidden."
  ([v]
     (---> v nil nil))
  ([v cls]
     (---> v cls nil))
  ([v cls pred]
     (--- v :out [AggregationKind/NONE] [AggregationKind/NONE] cls pred)))

(defn <---
  "Returns the vertices reachable from v via incoming incidences,
  optionally restricted by cls and pred (on the edges).  In contrast to <--,
  travesal of edges with aggregation semantics is forbidden."
  ([v]
     (<--- v nil nil))
  ([v cls]
     (<--- v cls nil))
  ([v cls pred]
     (--- v :in [AggregationKind/NONE] [AggregationKind/NONE] cls pred)))

(defn <-->
  "Returns the vertices reachable from v via all incidences,
  optionally restricted by cls and pred (on the edges).  In contrast to <->,
  travesal of edges with aggregation semantics is forbidden."
  ([v]
     (<--> v nil nil))
  ([v cls]
     (<--> v cls nil))
  ([v cls pred]
     (--- v :inout [AggregationKind/NONE] [AggregationKind/NONE] cls pred)))

(defn <>--
  "Aggregation path expression starting at whole v.  May be restricted by cls
  and pred (on the edges)."
  ([v]
     (<>-- v nil nil))
  ([v cls]
     (<>-- v cls nil))
  ([v cls pred]
     (--- v :inout
	  nil [AggregationKind/SHARED AggregationKind/COMPOSITE]
	  cls pred)))

(defn --<>
  "Aggregation path expression starting at part v.  May be restricted by cls
  and pred (on the edges)."
  ([v]
     (--<> v nil nil))
  ([v cls]
     (--<> v cls nil))
  ([v cls pred]
     (--- v :inout
	  [AggregationKind/SHARED AggregationKind/COMPOSITE] nil
	  cls pred)))

(defn <_>--
  "Aggregation-only path expression starting at whole v.  May be restricted by
  cls and pred (on the edges)."
  ([v]
     (<_>-- v nil nil))
  ([v cls]
     (<_>-- v cls nil))
  ([v cls pred]
     (--- v :inout
	  nil [AggregationKind/SHARED]
	  cls pred)))

(defn --<_>
  "Aggregation-only path expression starting at part v.  May be restricted by
  cls and pred (on the edges)."
  ([v]
     (--<_> v nil nil))
  ([v cls]
     (--<_> v cls nil))
  ([v cls pred]
     (--- v :inout
	  [AggregationKind/SHARED] nil
	  cls pred)))

(defn <*>--
  "Composition path expression starting at whole v.  May be restricted by cls
  and pred (on the edges)."
  ([v]
     (<*>-- v nil nil))
  ([v cls]
     (<*>-- v cls nil))
  ([v cls pred]
     (--- v :inout
	  nil [AggregationKind/COMPOSITE]
	  cls pred)))

(defn --<*>
  "Composition path expression starting at part v.  May be restricted by cls
  and pred (on the edges)."
  ([v]
     (--<*> v nil nil))
  ([v cls]
     (--<*> v cls nil))
  ([v cls pred]
     (--- v :inout
	  [AggregationKind/COMPOSITE] nil
	  cls pred)))

(defn p-seq
  "Path sequence starting at v and traversing p.
  v may be a vertex or a seq of vertices.  p is a varargs seq of path
  descriptions."
  [v & p]
  (if (seq p)
    (recur (p-apply v (first p)) (rest p))
    (into-oset v)))

(defn p-opt
  "Path option starting at v and maybe traversing p.
  v may be a vertex or a seq of vertices.
  p is a path description."
  [v p]
  (into-oset v (p-apply v p)))

(defn p-alt
  "Path alternative starting at v and traversing one of p.
  v may be a vertex or a seq of vertices.
  p is a varags seq of alternative path descriptions."
  [v & p]
  (into-oset (mapcat #(p-apply v %) p)))

(defn p-+
  "Path iteration starting at v and traversing p one or many times.
  v may be a vertex or a seq of vertices.
  p is a path description."
  ([v p]
     (p-+ v p false true))
  ([v p d skip-v]
     (let [v  (into-oset v)
	   n  (p-apply (if (false? d) v d) p)
	   vn (into-oset v n)
	   df (clojure.set/difference n v)
	   sv (if skip-v n vn)]
       (if (== (count df) 0)
	 sv
	 (recur sv p df false)))))

(defn p-*
  "Path iteration starting at v and traversing p zero or many times.
  v may be a vertex or a seq of vertices.
  p is a path description."
  [v p]
  (into-oset v (p-+ v p false false)))

(defn p-exp
  "Path exponent starting at v and traversing p n times, or at least l and at most p times.
  v may be a vertex or a seq of vertices.
  n or l and v are integers with l <= b.
  p is a path description."
  ([v l u p]
     {:pre [(<= l u) (>= l 0) (>= u 0)]}
     (loop [i (- u l), s (p-exp v l p)]
       (if (pos? i)
         (let [ns (into s (p-apply s p))]
           (if (= (count s) (count ns))
             s
             (recur (dec i) ns)))
         s)))
  ([v n p]
     {:pre [(>= n 0)]}
     (if (zero? n)
       (into-oset v)
       (recur (p-apply v p) (dec n) p))))

(defn p-restr
  "Vertex restriction concerning cls and pred on vs or each vertex in vs."
  ([vs cls]
     (p-restr vs cls identity))
  ([vs cls pred]
     (let [vs (into-oset vs)]
       (into-oset
	(if (seq vs)
	  (let [tm (type-matcher (first vs) cls)]
	    (filter (every-pred tm pred)
		    vs))
	  vs)))))

;;* Describe Schema and Graph Elements

(defprotocol Describable
  "A protocol for elements supporting describe."
  (describe [this]
    "Describes this attributed element or attributed element class."))

(declare describe)
(defn- attr-desc
  "Returns a map of aec's own attributes as name-domain pairs."
  [^AttributedElementClass aec]
  (into (sorted-map)
        (for [^Attribute attr (.getOwnAttributeList aec)]
          [(keyword (.getName attr)) (describe (.getDomain attr))])))

(defn- slot-desc
  [^AttributedElement e]
  (let [aec (.getAttributedElementClass e)]
    (into (sorted-map)
          (for [^Attribute attr (.getAttributeList aec)]
            (let [n (.getName attr)]
              [(keyword n) (value e n)])))))

(defn- super-classes
  [^GraphElementClass gec]
  (set (map #(symbol (.getQualifiedName ^GraphElementClass %))
            (filter #(not (.isInternal ^GraphElementClass %))
                    (.getDirectSuperClasses gec)))))

(defn- sub-classes
  [^GraphElementClass gec]
  (set (map #(symbol (.getQualifiedName ^GraphElementClass %))
            (.getDirectSubClasses gec))))

(extend-protocol Describable
  Graph
  (describe [this]
    {:type 'Graph
     :qname (symbol (qname this))
     :slots (slot-desc this)})
  Vertex
  (describe [this]
    {:type 'Vertex
     :qname (symbol (qname this))
     :slots (slot-desc this)})
  Edge
  (describe [this]
    {:type 'Edge
     :qname (symbol (qname this))
     :slots (slot-desc this)
     :alpha (.getAlpha this)
     :omega (.getOmega this)})
  GraphClass
  (describe [this]
    {:type 'GraphClass
     :qname (symbol (.getQualifiedName this))
     :attributes (attr-desc this)})
  VertexClass
  (describe [this]
    {:type 'VertexClass
     :qname (symbol (.getQualifiedName this))
     :attributes (attr-desc this)
     :super-classes (super-classes this)
     :sub-classes (sub-classes this)})
  EdgeClass
  (describe [this]
    {:type 'EdgeClass
     :qname (symbol (.getQualifiedName this))
     :attributes (attr-desc this)
     :from-vc (-> this .getFrom .getVertexClass .getQualifiedName symbol)
     :to-vc (-> this .getTo .getVertexClass .getQualifiedName symbol)
     :super-classes (super-classes this)
     :sub-classes (sub-classes this)})
  de.uni_koblenz.jgralab.schema.BasicDomain
  (describe [this]
    (-> this .getQualifiedName symbol))
  RecordDomain
  (describe [this]
    {:type 'Record
     :qname (symbol (.getQualifiedName this))
     :components (into (sorted-map)
                       (for [^de.uni_koblenz.jgralab.schema.RecordDomain$RecordComponent
                             c (.getComponents this)]
                         [(keyword (.getName c)) (describe (.getDomain c))]))})
  de.uni_koblenz.jgralab.schema.EnumDomain
  (describe [this]
    {:type 'Enum
     :qname (symbol (.getQualifiedName this))
     :constants (vec (.getConsts this))})
  de.uni_koblenz.jgralab.schema.CollectionDomain
  (describe [this]
    (symbol (.getQualifiedName this)))
  de.uni_koblenz.jgralab.schema.MapDomain
  (describe [this]
    (symbol (-> this
                .getQualifiedName
                (clojure.string/replace #"\s" "")
                (clojure.string/replace "," "=>")))))

;;* Funlib

(defn reduce-values
  "Reduces f thru the a-attribute values of the elements es.
  f must be a function of 2 args (see reduce).  s may be additional accessors,
  if a is a composite attribute.  Each additinal accessor may be either a
  function (which is simply applied) or another keyword, string, or symbol
  denoting a record component."
  [f es a & s]
  (reduce f (map (if s
                   #(loop [v (value % a), r s]
                      (if (seq r)
                        (let [acc (first r)]
                          (recur (if (fn? acc)
                                   (acc v)
                                   (value v acc))
                                 (rest r)))
                        v))
                   #(value % a))
                 es)))

(defn the
  "Returns the only element of seq s and errors if s contains more or less
  elements."
  [s]
  (if-let [f (first s)]
    (if (next s)
      (throw (RuntimeException.
	    ^String (format "seq contains more than one element!")))
      f)
    (throw (RuntimeException.
	    ^String (format "seq contains zero elements!")))))

(defn degree
  "Returns the degree of vertex v, possibly restricted by type and direction."
  ([^Vertex v]         (.getDegree v))
  ([^Vertex v cls]     (count (iseq v cls)))
  ([^Vertex v cls dir] (count (iseq v cls dir))))

(defn- topological-sort-clj
  "Returns a vector of g's vertices in topological order.
  Returns false if the graph is cyclic."
  [g]
  (loop [rem (vseq g), es  #{}, sorted []]
    (if (seq rem)
      (let [gs (group-by (fn [v]
                           (if (seq (remove es (map normal-edge (iseq v nil :in))))
                             false
                             true))
                         rem)
            good (gs true)
            bad (gs false)]
        ;;(println (count rem) ": good" (count good) "bad" (count bad))
        (if (seq good)
          (recur bad
                 (into es (mapcat #(iseq % nil :out) good))
                 (into sorted good))
          false))
      sorted)))

(defn topological-sort
  "Returns a seq of g's vertices in topological order.
  Returns false if the graph is cyclic.  The actual algorithm may be chosen
  between :dfs (a depth-first variant, the default), :kahn-knuth, and :plain (a
  purely functional clojure implementation, which is nice but slow)."
  ([g]
     (topological-sort g :dfs))
  ([g alg]
     (if (= alg :plain)
       (topological-sort-clj g)
       (let [^TopologicalOrderSolver a
             (.execute
              (case alg
                :kahn-knuth (KahnKnuthAlgorithm. g)
                :dfs        (TopologicalOrderWithDFS. g (IterativeDepthFirstSearch. g))
                (throw (RuntimeException. (str "Unknown topo-sort algorithm" alg)))))]
         (if (.isAcyclic ^AcyclicitySolver a)
           (map #(.getSecond ^PermutationEntry %)
                (seq (.getTopologicalOrder a)))
           false)))))

