;; gorilla-repl.fileformat = 1

;; **
;;; #Marmoset
;;; Interactive cards for gorilla repl.
;; **

;; @@
(ns marmoset.core
  (:require clojure.repl
            [gorilla-plot.core :as plot]
            [gorilla-repl.core :as gorilla]
            [gorilla-renderable.core :as render]
            [cljs.closure :as cljs]
            [clojure.core.async :as async]
            [compojure.core :refer :all]
            [compojure.route :as route]
            [hiccup.core :as hic]
            [sock.server :as ws]
            [me.raynes.fs :as fs]
            [base64-clj.core :as base64]
            [cheshire.core :as json]))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; #Clojurescript environment values and live updates.
;;; 
;;; Even though the value based rendering model is gorilla repls greatest strength,
;;; it also makes some things straight impossible.
;;; 
;;; Therefore there are two options of communicating values to a clojurescript view:
;;; 
;;; 1. As part of the `:env` which is a clojure map that will get compiled into the clojurescript
;;;     code and will be available even in a saved journal so it is the preferred option. Note that it can therefore only contain only data that can be serialized as EDN.
;;; 
;;; 2. Through a websocket which will allow for live updates but which will cease function once the journal is exported.
;; **

;; **
;;; ##Server side communication.
;; **

;; @@
(def connections (atom {}))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/connections</span>","value":"#'marmoset.core/connections"}
;; <=

;; @@
(defn marmoset-ws [id]
  (let [ch (async/chan 1024)
        rch (@connections id)]
    (when rch
      (async/go (async/>! rch (async/<! ch)))
      (ws/ws-handler ch))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/marmoset-ws</span>","value":"#'marmoset.core/marmoset-ws"}
;; <=

;; @@
(defroutes marmoset-routes
  (GET "/marmoset-ws/:id" [id] (marmoset-ws id)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/marmoset-routes</span>","value":"#'marmoset.core/marmoset-routes"}
;; <=

;; @@
(alter-var-root #'gorilla/app-routes
                routes
                marmoset-routes)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>#&lt;core$routes$fn__488 compojure.core$routes$fn__488@616256f6&gt;</span>","value":"#<core$routes$fn__488 compojure.core$routes$fn__488@616256f6>"}
;; <=

;; @@
(defn register-chan! [id chan]
  (swap! connections #(merge {(str id) chan} %)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/register-chan!</span>","value":"#'marmoset.core/register-chan!"}
;; <=

;; **
;;; ##Cljs compilation.
;; **

;; @@
(defn encode-sourcemap [basedir srcmp]
    (as-> srcmp %
          (json/parse-string %)
          (assoc % "sourcesContent"
            (mapv #(slurp (fs/file basedir %)) (% "sources")))
          (json/generate-string %)
          (base64/encode %)
          (str "\n//# sourceMappingURL=data:application/json;base64," %)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/encode-sourcemap</span>","value":"#'marmoset.core/encode-sourcemap"}
;; <=

;; @@
(defn build-cljs [opts code]
  (let [dir (fs/temp-dir "marmoset")
        dirp (.getAbsolutePath dir)
        outp (.getAbsolutePath (fs/file dir "main.js"))
        mapp (.getAbsolutePath (fs/file dir "main.js.map"))]
    (cljs/build
      (vec code)
      (merge
        {:optimizations :whitespace
         :pretty-print true
         :output-dir dirp
         :output-to outp
         :source-map mapp}
        opts))
    (str (slurp outp)
         (encode-sourcemap dir (slurp mapp)))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/build-cljs</span>","value":"#'marmoset.core/build-cljs"}
;; <=

;; **
;;; We break the compilation steps into two parts to get rid of the compile time.
;;; This way one can pass the environment 
;; **

;; @@
(defn segment [opts code]
  (let [compiled (build-cljs opts code)]
    (fn [id env]
      (let [payload (as-> {:id id
                           :env env} %
                          (pr-str %)
                          (clojure.string/escape % {\" "\\\""})
                          (str "var marmoset_payload = \"" % "\";\n"))
            src (hic/html
                  [:html {}
                   [:head {}
                    [:script {:type "text/javascript"} payload]
                    [:script {:type "text/javascript"} compiled]]
                   [:body {}]])]
        (hic/html [:iframe {:id id
                            :seamless "seamless"
                            :style "width: 100%; height: 10;"
                            :scrolling "no"
                            :marginwidth "0"
                            :marginheight "0"
                            :frameborder "0"
                            :vspace "0"
                            :hspace "0"
                            :srcdoc src}])))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/segment</span>","value":"#'marmoset.core/segment"}
;; <=

;; @@
(defmacro defsegment [name opts & forms]
  `(def ~name (segment ~opts (quote [~@forms]))))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/defsegment</span>","value":"#'marmoset.core/defsegment"}
;; <=
