;; gorilla-repl.fileformat = 1

;; **
;;; #Marmoset
;;; Interactive cards for gorilla repl.
;; **

;; @@
(ns marmoset.core
  (:require clojure.repl
            [gorilla-plot.core :as plot]
            [gorilla-repl.core :as gorilla]
            [gorilla-renderable.core :as render]
            [cljs.closure :as cljs]
            [clojure.core.async :as async]
            [compojure.core :refer :all]
            [compojure.route :as route]
            [hiccup.core :as hic]
            [sock.server :as ws]))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; #Clojurescript environment values and live updates.
;;; 
;;; Even though the value based rendering model is gorilla repls greatest strength,
;;; it also makes some things straight impossible.
;;; 
;;; Therefore there are two options of communicating values to a clojurescript view:
;;; 
;;; 1. As part of the `:env` which is a clojure map that will get compiled into the clojurescript
;;;     code and will be available even in a saved journal so it is the preferred option. Note that it can therefore only contain only data that can be serialized as EDN.
;;; 
;;; 2. Through a websocket which will allow for live updates but which will cease function once the journal is exported.
;; **

;; @@
(def connections (atom {}))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/connections</span>","value":"#'marmoset.core/connections"}
;; <=

;; @@
(defn marmoset-ws [id]
  (let [ch (async/chan 1024)]
    (async/go (async/>! (@connections id)
                        (async/<! ch)))
    (ws/ws-handler ch)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/marmoset-ws</span>","value":"#'marmoset.core/marmoset-ws"}
;; <=

;; @@
(defroutes marmoset-routes
  (GET "/marmoset-ws/:id" [id] (marmoset-ws id)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/marmoset-routes</span>","value":"#'marmoset.core/marmoset-routes"}
;; <=

;; @@
(alter-var-root #'gorilla/app-routes
                routes
                marmoset-routes)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>#&lt;core$routes$fn__486 compojure.core$routes$fn__486@34d0665a&gt;</span>","value":"#<core$routes$fn__486 compojure.core$routes$fn__486@34d0665a>"}
;; <=

;; @@
(def marmoset-ns
  '[(ns marmoset.client
      [:require cljs.reader
                sock.client])
    (let [payload (cljs.reader/read-string js/marmoset-payload)]
      (def id (:id payload))
      (def env (:env payload))
      (def ws (sock.client/ws (str "ws://"
                                   (aget js/top "location" "host")
                                   "/marmoset-ws/"
                                   (str id)))))
    (ns cljs.user)])

(defn cljs-html [id env code & [opts]]
  (hic/html [:div {:style
"position: relative; width: 100%; height: 0; padding-bottom: 56.25%;"}
                               [:iframe {:seamless "seamless"
                                         :style
"position: absolute; width: 100%; height: 100%; left: 0; top: 0;"
                                         :scrolling "no"
                                         :marginwidth "0"
                                         :marginheight "0"
                                         :frameborder "0"
                                         :vspace "0"
                                         :hspace "0"
                                         :srcdoc
  (hic/html
    [:html {}
     [:head {}
      [:script {:type "text/javascript"}
       (str "var marmoset_payload = \""
            (clojure.string/escape
              (pr-str {:id id :env env})
              {\" "\\\""})
            "\";")]
      [:script {:type "text/javascript"}
       (cljs/build
         (vec (concat marmoset-ns code))
         (merge
           {:optimizations :whitespace
            :pretty-print true}
           opts))]]
     [:body {}
      [:div {:id "app"}]]])}]]))

(defn register-chan! [id chan]
  (swap! connections #(merge {(str id) chan} %)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/register-chan!</span>","value":"#'marmoset.core/register-chan!"}
;; <=

;; @@
(defrecord CljsView [id env opts code]
  render/Renderable
  (render [self]
          {:type :html
           :value (pr-str self)
           :content (cljs-html id env code opts)}))

(defmacro cljs-view [opts & code]
  `(let [opts# ~opts
         env# (:env opts#)
         chan# (:chan opts#)
         opts# (:opts opts#)
         uuid# (java.util.UUID/randomUUID)]
     (register-chan! uuid# chan#)
     (CljsView. uuid#
                env#
                opts#
                '[~@code])))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/cljs-view</span>","value":"#'marmoset.core/cljs-view"}
;; <=
