;; gorilla-repl.fileformat = 1

;; **
;;; #Marmoset
;;; Interactive cards for gorilla repl.
;; **

;; @@
(ns marmoset.api
  (:require clojure.repl
            [gorilla-plot.core :as plot]
            [gorilla-repl.core :as gorilla]
            [gorilla-renderable.core :as render]
            [cljs.closure :as cljs]
            [clojure.core.async :as async]
            [compojure.core :refer :all]
            [compojure.route :as route]
            [hiccup.core :as hic]
            [sock.server :as ws]))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}
;; <=

;; **
;;; #Clojurescript environment values and live updates.
;;; 
;;; Even though the value based rendering model is gorilla repls greatest strength,
;;; it also makes some things straight impossible.
;;; 
;;; Therefore there are two options of communicating values to a clojurescript view:
;;; 
;;; 1. As part of the `:env` which is a clojure map that will get compiled into the clojurescript
;;;     code and will be available even in a saved journal so it is the preferred option. Note that it can therefore only contain only data that can be serialized as EDN.
;;;     
;;; 2. Through a websocket which will allow for live updates but which will cease function once the journal is exported.
;; **

;; @@
(def connections (atom {}))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/connections</span>","value":"#'marmoset.core/connections"}
;; <=

;; @@
(defn marmoset-ws [id]
  (let [ch (async/chan 1024)]
    (async/go (async/>! (@connections id)
                        (async/<! ch)))
    (ws/ws-handler ch)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/marmoset-ws</span>","value":"#'marmoset.core/marmoset-ws"}
;; <=

;; @@
(defroutes marmoset-routes
  (GET "/marmoset-ws/:id" [id] (marmoset-ws id)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/marmoset-routes</span>","value":"#'marmoset.core/marmoset-routes"}
;; <=

;; @@
(alter-var-root #'gorilla/app-routes
                routes
                marmoset-routes)
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-unkown'>#&lt;core$routes$fn__486 compojure.core$routes$fn__486@34d0665a&gt;</span>","value":"#<core$routes$fn__486 compojure.core$routes$fn__486@34d0665a>"}
;; <=

;; @@
(def marmoset-ns
  '[(ns marmoset.client
      [:require cljs.reader
                sock.client])
    (let [payload (cljs.reader/read-string js/marmoset-payload)]
      (def id (:id payload))
      (def env (:env payload))
      (def ws (sock.client/ws (str "ws://"
                                   (aget js/top "location" "host")
                                   "/marmoset-ws/"
                                   (str id)))))
    (ns cljs.user)])

(defn cljs-html [id env code & [opts]]
  (hic/html [:div {:style
"position: relative; width: 100%; height: 0; padding-bottom: 56.25%;"}
                               [:iframe {:seamless "seamless"
                                         :style
"position: absolute; width: 100%; height: 100%; left: 0; top: 0;"
                                         :scrolling "no"
                                         :marginwidth "0"
                                         :marginheight "0"
                                         :frameborder "0"
                                         :vspace "0"
                                         :hspace "0"
                                         :srcdoc
  (hic/html
    [:html {}
     [:head {}
      [:script {:type "text/javascript"}
       (str "var marmoset_payload = \""
            (clojure.string/escape
              (pr-str {:id id :env env})
              {\" "\\\""})
            "\";")]
      [:script {:type "text/javascript"}
       (cljs/build
         (vec (concat marmoset-ns code))
         (merge
           {:optimizations :whitespace
            :pretty-print true}
           opts))]]
     [:body {}]])}]]))

(defn register-chan! [id chan]
  (swap! connections #(merge {(str id) chan} %)))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/register-chan!</span>","value":"#'marmoset.core/register-chan!"}
;; <=

;; @@
(defrecord CljsView [id env opts code]
  render/Renderable 
  (render [self]
          {:type :html
           :value (pr-str self)
           :content (cljs-html id env code opts)}))

(defmacro cljs-view [opts & code]
  `(let [opts# ~opts
         env# (:env opts#)
         chan# (:chan opts#)
         opts# (:opts opts#)
         uuid# (java.util.UUID/randomUUID)]
     (register-chan! uuid# chan#)
     (CljsView. uuid#
                env#
                opts#
                '[~@code])))
;; @@
;; =>
;;; {"type":"html","content":"<span class='clj-var'>#&#x27;marmoset.core/cljs-view</span>","value":"#'marmoset.core/cljs-view"}
;; <=

;; @@
(def ch (async/chan))
(cljs-view {:chan ch
            :env {:x "ho"}
            :opts {}}
      (ns foo
        [:require marmoset.client
                  [cljs.core.async :refer [<!]]]
        [:require-macros
         [cljs.core.async.macros :refer [go]]])
      (enable-console-print!)
      (go (loop []
            (js/alert (str (<! (:in marmoset.client/ws))))
            (recur))))
;; @@
;; =>
;;; {"type":"html","value":"#marmoset.core.CljsView{:id #uuid \"74ab960c-0b13-49a1-a9dc-30bd72c76cf6\", :env {:x \"ho\"}, :opts {}, :code [(ns foo [:require marmoset.client [cljs.core.async :refer [<!]]] [:require-macros [cljs.core.async.macros :refer [go]]]) (enable-console-print!) (go (loop [] (js/alert (str (<! (:in marmoset.client/ws)))) (recur)))]}","content":"<div style=\"position: relative; width: 100%; height: 0; padding-bottom: 56.25%;\"><iframe frameborder=\"0\" hspace=\"0\" marginheight=\"0\" marginwidth=\"0\" scrolling=\"no\" seamless=\"seamless\" srcdoc=\"&lt;html&gt;&lt;head&gt;&lt;script type=&quot;text/javascript&quot;&gt;var marmoset_payload = &quot;{:id #uuid \\&quot;74ab960c-0b13-49a1-a9dc-30bd72c76cf6\\&quot;, :env {:x \\&quot;ho\\&quot;}}&quot;;&lt;/script&gt;&lt;script type=&quot;text/javascript&quot;&gt;if(typeof Math.imul == &quot;undefined&quot; || (Math.imul(0xffffffff,5) == 0)) {\n    Math.imul = function (a, b) {\n        var ah  = (a &gt;&gt;&gt; 16) &amp; 0xffff;\n        var al = a &amp; 0xffff;\n        var bh  = (b &gt;&gt;&gt; 16) &amp; 0xffff;\n        var bl = b &amp; 0xffff;\n        // the shift by 0 fixes the sign on the high part\n        // the final |0 converts the unsigned value into a signed value\n        return ((al * bl) + (((ah * bl + al * bh) &lt;&lt; 16) &gt;&gt;&gt; 0)|0);\n    }\n}\n\nvar CLOSURE_NO_DEPS = true;\nvar COMPILED = false;\nvar goog = goog || {};\ngoog.global = this;\ngoog.global.CLOSURE_UNCOMPILED_DEFINES;\ngoog.global.CLOSURE_DEFINES;\ngoog.isDef = function(val) {\n  return val !== void 0;\n};\ngoog.exportPath_ = function(name, opt_object, opt_objectToExportTo) {\n  var parts = name.split(&quot;.&quot;);\n  var cur = opt_objectToExportTo || goog.global;\n  if (!(parts[0] in cur) &amp;&amp; cur.execScript) {\n    cur.execScript(&quot;var &quot; + parts[0]);\n  }\n  for (var part;parts.length &amp;&amp; (part = parts.shift());) {\n    if (!parts.length &amp;&amp; goog.isDef(opt_object)) {\n      cur[part] = opt_object;\n    } else {\n      if (cur[part]) {\n        cur = cur[part];\n      } else {\n        cur = cur[part] = {};\n      }\n    }\n  }\n};\ngoog.define = function(name, defaultValue) {\n  var value = defaultValue;\n  if (!COMPILED) {\n    if (goog.global.CLOSURE_UNCOMPILED_DEFINES &amp;&amp; Object.prototype.hasOwnProperty.call(goog.global.CLOSURE_UNCOMPILED_DEFINES, name)) {\n      value = goog.global.CLOSURE_UNCOMPILED_DEFINES[name];\n    } else {\n      if (goog.global.CLOSURE_DEFINES &amp;&amp; Object.prototype.hasOwnProperty.call(goog.global.CLOSURE_DEFINES, name)) {\n        value = goog.global.CLOSURE_DEFINES[name];\n      }\n    }\n  }\n  goog.exportPath_(name, value);\n};\ngoog.DEBUG = true;\ngoog.define(&quot;goog.LOCALE&quot;, &quot;en&quot;);\ngoog.define(&quot;goog.TRUSTED_SITE&quot;, true);\ngoog.define(&quot;goog.STRICT_MODE_COMPATIBLE&quot;, false);\ngoog.provide = function(name) {\n  if (!COMPILED) {\n    if (goog.isProvided_(name)) {\n      throw Error(&apos;Namespace &quot;&apos; + name + &apos;&quot; already declared.&apos;);\n    }\n    delete goog.implicitNamespaces_[name];\n    var namespace = name;\n    while (namespace = namespace.substring(0, namespace.lastIndexOf(&quot;.&quot;))) {\n      if (goog.getObjectByName(namespace)) {\n        break;\n      }\n      goog.implicitNamespaces_[namespace] = true;\n    }\n  }\n  goog.exportPath_(name);\n};\ngoog.setTestOnly = function(opt_message) {\n  if (COMPILED &amp;&amp; !goog.DEBUG) {\n    opt_message = opt_message || &quot;&quot;;\n    throw Error(&quot;Importing test-only code into non-debug environment&quot; + (opt_message ? &quot;: &quot; + opt_message : &quot;.&quot;));\n  }\n};\ngoog.forwardDeclare = function(name) {\n};\nif (!COMPILED) {\n  goog.isProvided_ = function(name) {\n    return!goog.implicitNamespaces_[name] &amp;&amp; goog.isDefAndNotNull(goog.getObjectByName(name));\n  };\n  goog.implicitNamespaces_ = {};\n}\ngoog.getObjectByName = function(name, opt_obj) {\n  var parts = name.split(&quot;.&quot;);\n  var cur = opt_obj || goog.global;\n  for (var part;part = parts.shift();) {\n    if (goog.isDefAndNotNull(cur[part])) {\n      cur = cur[part];\n    } else {\n      return null;\n    }\n  }\n  return cur;\n};\ngoog.globalize = function(obj, opt_global) {\n  var global = opt_global || goog.global;\n  for (var x in obj) {\n    global[x] = obj[x];\n  }\n};\ngoog.addDependency = function(relPath, provides, requires) {\n  if (goog.DEPENDENCIES_ENABLED) {\n    var provide, require;\n    var path = relPath.replace(/\\\\/g, &quot;/&quot;);\n    var deps = goog.dependencies_;\n    for (var i = 0;provide = provides[i];i++) {\n      deps.nameToPath[provide] = path;\n      if (!(path in deps.pathToNames)) {\n        deps.pathToNames[path] = {};\n      }\n      deps.pathToNames[path][provide] = true;\n    }\n    for (var j = 0;require = requires[j];j++) {\n      if (!(path in deps.requires)) {\n        deps.requires[path] = {};\n      }\n      deps.requires[path][require] = true;\n    }\n  }\n};\ngoog.define(&quot;goog.ENABLE_DEBUG_LOADER&quot;, true);\ngoog.require = function(name) {\n  if (!COMPILED) {\n    if (goog.isProvided_(name)) {\n      return;\n    }\n    if (goog.ENABLE_DEBUG_LOADER) {\n      var path = goog.getPathFromDeps_(name);\n      if (path) {\n        goog.included_[path] = true;\n        goog.writeScripts_();\n        return;\n      }\n    }\n    var errorMessage = &quot;goog.require could not find: &quot; + name;\n    if (goog.global.console) {\n      goog.global.console[&quot;error&quot;](errorMessage);\n    }\n    throw Error(errorMessage);\n  }\n};\ngoog.basePath = &quot;&quot;;\ngoog.global.CLOSURE_BASE_PATH;\ngoog.global.CLOSURE_NO_DEPS;\ngoog.global.CLOSURE_IMPORT_SCRIPT;\ngoog.nullFunction = function() {\n};\ngoog.identityFunction = function(opt_returnValue, var_args) {\n  return opt_returnValue;\n};\ngoog.abstractMethod = function() {\n  throw Error(&quot;unimplemented abstract method&quot;);\n};\ngoog.addSingletonGetter = function(ctor) {\n  ctor.getInstance = function() {\n    if (ctor.instance_) {\n      return ctor.instance_;\n    }\n    if (goog.DEBUG) {\n      goog.instantiatedSingletons_[goog.instantiatedSingletons_.length] = ctor;\n    }\n    return ctor.instance_ = new ctor;\n  };\n};\ngoog.instantiatedSingletons_ = [];\ngoog.DEPENDENCIES_ENABLED = !COMPILED &amp;&amp; goog.ENABLE_DEBUG_LOADER;\nif (goog.DEPENDENCIES_ENABLED) {\n  goog.included_ = {};\n  goog.dependencies_ = {pathToNames:{}, nameToPath:{}, requires:{}, visited:{}, written:{}};\n  goog.inHtmlDocument_ = function() {\n    var doc = goog.global.document;\n    return typeof doc != &quot;undefined&quot; &amp;&amp; &quot;write&quot; in doc;\n  };\n  goog.findBasePath_ = function() {\n    if (goog.global.CLOSURE_BASE_PATH) {\n      goog.basePath = goog.global.CLOSURE_BASE_PATH;\n      return;\n    } else {\n      if (!goog.inHtmlDocument_()) {\n        return;\n      }\n    }\n    var doc = goog.global.document;\n    var scripts = doc.getElementsByTagName(&quot;script&quot;);\n    for (var i = scripts.length - 1;i &gt;= 0;--i) {\n      var src = scripts[i].src;\n      var qmark = src.lastIndexOf(&quot;?&quot;);\n      var l = qmark == -1 ? src.length : qmark;\n      if (src.substr(l - 7, 7) == &quot;base.js&quot;) {\n        goog.basePath = src.substr(0, l - 7);\n        return;\n      }\n    }\n  };\n  goog.importScript_ = function(src) {\n    var importScript = goog.global.CLOSURE_IMPORT_SCRIPT || goog.writeScriptTag_;\n    if (!goog.dependencies_.written[src] &amp;&amp; importScript(src)) {\n      goog.dependencies_.written[src] = true;\n    }\n  };\n  goog.writeScriptTag_ = function(src) {\n    if (goog.inHtmlDocument_()) {\n      var doc = goog.global.document;\n      if (doc.readyState == &quot;complete&quot;) {\n        var isDeps = /\\bdeps.js$/.test(src);\n        if (isDeps) {\n          return false;\n        } else {\n          throw Error(&apos;Cannot write &quot;&apos; + src + &apos;&quot; after document load&apos;);\n        }\n      }\n      doc.write(&apos;\\x3cscript type\\x3d&quot;text/javascript&quot; src\\x3d&quot;&apos; + src + &apos;&quot;\\x3e\\x3c/&apos; + &quot;script\\x3e&quot;);\n      return true;\n    } else {\n      return false;\n    }\n  };\n  goog.writeScripts_ = function() {\n    var scripts = [];\n    var seenScript = {};\n    var deps = goog.dependencies_;\n    function visitNode(path) {\n      if (path in deps.written) {\n        return;\n      }\n      if (path in deps.visited) {\n        if (!(path in seenScript)) {\n          seenScript[path] = true;\n          scripts.push(path);\n        }\n        return;\n      }\n      deps.visited[path] = true;\n      if (path in deps.requires) {\n        for (var requireName in deps.requires[path]) {\n          if (!goog.isProvided_(requireName)) {\n            if (requireName in deps.nameToPath) {\n              visitNode(deps.nameToPath[requireName]);\n            } else {\n              throw Error(&quot;Undefined nameToPath for &quot; + requireName);\n            }\n          }\n        }\n      }\n      if (!(path in seenScript)) {\n        seenScript[path] = true;\n        scripts.push(path);\n      }\n    }\n    for (var path in goog.included_) {\n      if (!deps.written[path]) {\n        visitNode(path);\n      }\n    }\n    for (var i = 0;i &lt; scripts.length;i++) {\n      if (scripts[i]) {\n        goog.importScript_(goog.basePath + scripts[i]);\n      } else {\n        throw Error(&quot;Undefined script input&quot;);\n      }\n    }\n  };\n  goog.getPathFromDeps_ = function(rule) {\n    if (rule in goog.dependencies_.nameToPath) {\n      return goog.dependencies_.nameToPath[rule];\n    } else {\n      return null;\n    }\n  };\n  goog.findBasePath_();\n  if (!goog.global.CLOSURE_NO_DEPS) {\n    goog.importScript_(goog.basePath + &quot;deps.js&quot;);\n  }\n}\ngoog.typeOf = function(value) {\n  var s = typeof value;\n  if (s == &quot;object&quot;) {\n    if (value) {\n      if (value instanceof Array) {\n        return &quot;array&quot;;\n      } else {\n        if (value instanceof Object) {\n          return s;\n        }\n      }\n      var className = Object.prototype.toString.call((value));\n      if (className == &quot;[object Window]&quot;) {\n        return &quot;object&quot;;\n      }\n      if (className == &quot;[object Array]&quot; || typeof value.length == &quot;number&quot; &amp;&amp; typeof value.splice != &quot;undefined&quot; &amp;&amp; typeof value.propertyIsEnumerable != &quot;undefined&quot; &amp;&amp; !value.propertyIsEnumerable(&quot;splice&quot;)) {\n        return &quot;array&quot;;\n      }\n      if (className == &quot;[object Function]&quot; || typeof value.call != &quot;undefined&quot; &amp;&amp; typeof value.propertyIsEnumerable != &quot;undefined&quot; &amp;&amp; !value.propertyIsEnumerable(&quot;call&quot;)) {\n        return &quot;function&quot;;\n      }\n    } else {\n      return &quot;null&quot;;\n    }\n  } else {\n    if (s == &quot;function&quot; &amp;&amp; typeof value.call == &quot;undefined&quot;) {\n      return &quot;object&quot;;\n    }\n  }\n  return s;\n};\ngoog.isNull = function(val) {\n  return val === null;\n};\ngoog.isDefAndNotNull = function(val) {\n  return val != null;\n};\ngoog.isArray = function(val) {\n  return goog.typeOf(val) == &quot;array&quot;;\n};\ngoog.isArrayLike = function(val) {\n  var type = goog.typeOf(val);\n  return type == &quot;array&quot; || type == &quot;object&quot; &amp;&amp; typeof val.length == &quot;number&quot;;\n};\ngoog.isDateLike = function(val) {\n  return goog.isObject(val) &amp;&amp; typeof val.getFullYear == &quot;function&quot;;\n};\ngoog.isString = function(val) {\n  return typeof val == &quot;string&quot;;\n};\ngoog.isBoolean = function(val) {\n  return typeof val == &quot;boolean&quot;;\n};\ngoog.isNumber = function(val) {\n  return typeof val == &quot;number&quot;;\n};\ngoog.isFunction = function(val) {\n  return goog.typeOf(val) == &quot;function&quot;;\n};\ngoog.isObject = function(val) {\n  var type = typeof val;\n  return type == &quot;object&quot; &amp;&amp; val != null || type == &quot;function&quot;;\n};\ngoog.getUid = function(obj) {\n  return obj[goog.UID_PROPERTY_] || (obj[goog.UID_PROPERTY_] = ++goog.uidCounter_);\n};\ngoog.hasUid = function(obj) {\n  return!!obj[goog.UID_PROPERTY_];\n};\ngoog.removeUid = function(obj) {\n  if (&quot;removeAttribute&quot; in obj) {\n    obj.removeAttribute(goog.UID_PROPERTY_);\n  }\n  try {\n    delete obj[goog.UID_PROPERTY_];\n  } catch (ex) {\n  }\n};\ngoog.UID_PROPERTY_ = &quot;closure_uid_&quot; + (Math.random() * 1E9 &gt;&gt;&gt; 0);\ngoog.uidCounter_ = 0;\ngoog.getHashCode = goog.getUid;\ngoog.removeHashCode = goog.removeUid;\ngoog.cloneObject = function(obj) {\n  var type = goog.typeOf(obj);\n  if (type == &quot;object&quot; || type == &quot;array&quot;) {\n    if (obj.clone) {\n      return obj.clone();\n    }\n    var clone = type == &quot;array&quot; ? [] : {};\n    for (var key in obj) {\n      clone[key] = goog.cloneObject(obj[key]);\n    }\n    return clone;\n  }\n  return obj;\n};\ngoog.bindNative_ = function(fn, selfObj, var_args) {\n  return(fn.call.apply(fn.bind, arguments));\n};\ngoog.bindJs_ = function(fn, selfObj, var_args) {\n  if (!fn) {\n    throw new Error;\n  }\n  if (arguments.length &gt; 2) {\n    var boundArgs = Array.prototype.slice.call(arguments, 2);\n    return function() {\n      var newArgs = Array.prototype.slice.call(arguments);\n      Array.prototype.unshift.apply(newArgs, boundArgs);\n      return fn.apply(selfObj, newArgs);\n    };\n  } else {\n    return function() {\n      return fn.apply(selfObj, arguments);\n    };\n  }\n};\ngoog.bind = function(fn, selfObj, var_args) {\n  if (Function.prototype.bind &amp;&amp; Function.prototype.bind.toString().indexOf(&quot;native code&quot;) != -1) {\n    goog.bind = goog.bindNative_;\n  } else {\n    goog.bind = goog.bindJs_;\n  }\n  return goog.bind.apply(null, arguments);\n};\ngoog.partial = function(fn, var_args) {\n  var args = Array.prototype.slice.call(arguments, 1);\n  return function() {\n    var newArgs = args.slice();\n    newArgs.push.apply(newArgs, arguments);\n    return fn.apply(this, newArgs);\n  };\n};\ngoog.mixin = function(target, source) {\n  for (var x in source) {\n    target[x] = source[x];\n  }\n};\ngoog.now = goog.TRUSTED_SITE &amp;&amp; Date.now || function() {\n  return+new Date;\n};\ngoog.globalEval = function(script) {\n  if (goog.global.execScript) {\n    goog.global.execScript(script, &quot;JavaScript&quot;);\n  } else {\n    if (goog.global.eval) {\n      if (goog.evalWorksForGlobals_ == null) {\n        goog.global.eval(&quot;var _et_ \\x3d 1;&quot;);\n        if (typeof goog.global[&quot;_et_&quot;] != &quot;undefined&quot;) {\n          delete goog.global[&quot;_et_&quot;];\n          goog.evalWorksForGlobals_ = true;\n        } else {\n          goog.evalWorksForGlobals_ = false;\n        }\n      }\n      if (goog.evalWorksForGlobals_) {\n        goog.global.eval(script);\n      } else {\n        var doc = goog.global.document;\n        var scriptElt = doc.createElement(&quot;script&quot;);\n        scriptElt.type = &quot;text/javascript&quot;;\n        scriptElt.defer = false;\n        scriptElt.appendChild(doc.createTextNode(script));\n        doc.body.appendChild(scriptElt);\n        doc.body.removeChild(scriptElt);\n      }\n    } else {\n      throw Error(&quot;goog.globalEval not available&quot;);\n    }\n  }\n};\ngoog.evalWorksForGlobals_ = null;\ngoog.cssNameMapping_;\ngoog.cssNameMappingStyle_;\ngoog.getCssName = function(className, opt_modifier) {\n  var getMapping = function(cssName) {\n    return goog.cssNameMapping_[cssName] || cssName;\n  };\n  var renameByParts = function(cssName) {\n    var parts = cssName.split(&quot;-&quot;);\n    var mapped = [];\n    for (var i = 0;i &lt; parts.length;i++) {\n      mapped.push(getMapping(parts[i]));\n    }\n    return mapped.join(&quot;-&quot;);\n  };\n  var rename;\n  if (goog.cssNameMapping_) {\n    rename = goog.cssNameMappingStyle_ == &quot;BY_WHOLE&quot; ? getMapping : renameByParts;\n  } else {\n    rename = function(a) {\n      return a;\n    };\n  }\n  if (opt_modifier) {\n    return className + &quot;-&quot; + rename(opt_modifier);\n  } else {\n    return rename(className);\n  }\n};\ngoog.setCssNameMapping = function(mapping, opt_style) {\n  goog.cssNameMapping_ = mapping;\n  goog.cssNameMappingStyle_ = opt_style;\n};\ngoog.global.CLOSURE_CSS_NAME_MAPPING;\nif (!COMPILED &amp;&amp; goog.global.CLOSURE_CSS_NAME_MAPPING) {\n  goog.cssNameMapping_ = goog.global.CLOSURE_CSS_NAME_MAPPING;\n}\ngoog.getMsg = function(str, opt_values) {\n  if (opt_values) {\n    str = str.replace(/\\{\\$([^}]+)}/g, function(match, key) {\n      return key in opt_values ? opt_values[key] : match;\n    });\n  }\n  return str;\n};\ngoog.getMsgWithFallback = function(a, b) {\n  return a;\n};\ngoog.exportSymbol = function(publicPath, object, opt_objectToExportTo) {\n  goog.exportPath_(publicPath, object, opt_objectToExportTo);\n};\ngoog.exportProperty = function(object, publicName, symbol) {\n  object[publicName] = symbol;\n};\ngoog.inherits = function(childCtor, parentCtor) {\n  function tempCtor() {\n  }\n  tempCtor.prototype = parentCtor.prototype;\n  childCtor.superClass_ = parentCtor.prototype;\n  childCtor.prototype = new tempCtor;\n  childCtor.prototype.constructor = childCtor;\n  childCtor.base = function(me, methodName, var_args) {\n    var args = Array.prototype.slice.call(arguments, 2);\n    return parentCtor.prototype[methodName].apply(me, args);\n  };\n};\ngoog.base = function(me, opt_methodName, var_args) {\n  var caller = arguments.callee.caller;\n  if (goog.STRICT_MODE_COMPATIBLE || goog.DEBUG &amp;&amp; !caller) {\n    throw Error(&quot;arguments.caller not defined.  goog.base() cannot be used &quot; + &quot;with strict mode code. See &quot; + &quot;http://www.ecma-international.org/ecma-262/5.1/#sec-C&quot;);\n  }\n  if (caller.superClass_) {\n    return caller.superClass_.constructor.apply(me, Array.prototype.slice.call(arguments, 1));\n  }\n  var args = Array.prototype.slice.call(arguments, 2);\n  var foundCaller = false;\n  for (var ctor = me.constructor;ctor;ctor = ctor.superClass_ &amp;&amp; ctor.superClass_.constructor) {\n    if (ctor.prototype[opt_methodName] === caller) {\n      foundCaller = true;\n    } else {\n      if (foundCaller) {\n        return ctor.prototype[opt_methodName].apply(me, args);\n      }\n    }\n  }\n  if (me[opt_methodName] === caller) {\n    return me.constructor.prototype[opt_methodName].apply(me, args);\n  } else {\n    throw Error(&quot;goog.base called from a method of one name &quot; + &quot;to a method of a different name&quot;);\n  }\n};\ngoog.scope = function(fn) {\n  fn.call(goog.global);\n};\nif (!COMPILED) {\n  goog.global[&quot;COMPILED&quot;] = COMPILED;\n}\ngoog.defineClass = function(superClass, def) {\n  var constructor = def.constructor;\n  var statics = def.statics;\n  if (!constructor || constructor == Object.prototype.constructor) {\n    constructor = function() {\n      throw Error(&quot;cannot instantiate an interface (no constructor defined).&quot;);\n    };\n  }\n  var cls = goog.defineClass.createSealingConstructor_(constructor, superClass);\n  if (superClass) {\n    goog.inherits(cls, superClass);\n  }\n  delete def.constructor;\n  delete def.statics;\n  goog.defineClass.applyProperties_(cls.prototype, def);\n  if (statics != null) {\n    if (statics instanceof Function) {\n      statics(cls);\n    } else {\n      goog.defineClass.applyProperties_(cls, statics);\n    }\n  }\n  return cls;\n};\ngoog.defineClass.ClassDescriptor;\ngoog.define(&quot;goog.defineClass.SEAL_CLASS_INSTANCES&quot;, goog.DEBUG);\ngoog.defineClass.createSealingConstructor_ = function(ctr, superClass) {\n  if (goog.defineClass.SEAL_CLASS_INSTANCES &amp;&amp; Object.seal instanceof Function) {\n    if (superClass &amp;&amp; superClass.prototype &amp;&amp; superClass.prototype[goog.UNSEALABLE_CONSTRUCTOR_PROPERTY_]) {\n      return ctr;\n    }\n    var wrappedCtr = function() {\n      var instance = ctr.apply(this, arguments) || this;\n      if (this.constructor === wrappedCtr) {\n        Object.seal(instance);\n      }\n      return instance;\n    };\n    return wrappedCtr;\n  }\n  return ctr;\n};\ngoog.defineClass.OBJECT_PROTOTYPE_FIELDS_ = [&quot;constructor&quot;, &quot;hasOwnProperty&quot;, &quot;isPrototypeOf&quot;, &quot;propertyIsEnumerable&quot;, &quot;toLocaleString&quot;, &quot;toString&quot;, &quot;valueOf&quot;];\ngoog.defineClass.applyProperties_ = function(target, source) {\n  var key;\n  for (key in source) {\n    if (Object.prototype.hasOwnProperty.call(source, key)) {\n      target[key] = source[key];\n    }\n  }\n  for (var i = 0;i &lt; goog.defineClass.OBJECT_PROTOTYPE_FIELDS_.length;i++) {\n    key = goog.defineClass.OBJECT_PROTOTYPE_FIELDS_[i];\n    if (Object.prototype.hasOwnProperty.call(source, key)) {\n      target[key] = source[key];\n    }\n  }\n};\ngoog.tagUnsealableClass = function(ctr) {\n  if (!COMPILED &amp;&amp; goog.defineClass.SEAL_CLASS_INSTANCES) {\n    ctr.prototype[goog.UNSEALABLE_CONSTRUCTOR_PROPERTY_] = true;\n  }\n};\ngoog.UNSEALABLE_CONSTRUCTOR_PROPERTY_ = &quot;goog_defineClass_legacy_unsealable&quot;;\ngoog.provide(&quot;goog.string&quot;);\ngoog.provide(&quot;goog.string.Unicode&quot;);\ngoog.define(&quot;goog.string.DETECT_DOUBLE_ESCAPING&quot;, false);\ngoog.string.Unicode = {NBSP:&quot;\\u00a0&quot;};\ngoog.string.startsWith = function(str, prefix) {\n  return str.lastIndexOf(prefix, 0) == 0;\n};\ngoog.string.endsWith = function(str, suffix) {\n  var l = str.length - suffix.length;\n  return l &gt;= 0 &amp;&amp; str.indexOf(suffix, l) == l;\n};\ngoog.string.caseInsensitiveStartsWith = function(str, prefix) {\n  return goog.string.caseInsensitiveCompare(prefix, str.substr(0, prefix.length)) == 0;\n};\ngoog.string.caseInsensitiveEndsWith = function(str, suffix) {\n  return goog.string.caseInsensitiveCompare(suffix, str.substr(str.length - suffix.length, suffix.length)) == 0;\n};\ngoog.string.caseInsensitiveEquals = function(str1, str2) {\n  return str1.toLowerCase() == str2.toLowerCase();\n};\ngoog.string.subs = function(str, var_args) {\n  var splitParts = str.split(&quot;%s&quot;);\n  var returnString = &quot;&quot;;\n  var subsArguments = Array.prototype.slice.call(arguments, 1);\n  while (subsArguments.length &amp;&amp; splitParts.length &gt; 1) {\n    returnString += splitParts.shift() + subsArguments.shift();\n  }\n  return returnString + splitParts.join(&quot;%s&quot;);\n};\ngoog.string.collapseWhitespace = function(str) {\n  return str.replace(/[\\s\\xa0]+/g, &quot; &quot;).replace(/^\\s+|\\s+$/g, &quot;&quot;);\n};\ngoog.string.isEmpty = function(str) {\n  return/^[\\s\\xa0]*$/.test(str);\n};\ngoog.string.isEmptySafe = function(str) {\n  return goog.string.isEmpty(goog.string.makeSafe(str));\n};\ngoog.string.isBreakingWhitespace = function(str) {\n  return!/[^\\t\\n\\r ]/.test(str);\n};\ngoog.string.isAlpha = function(str) {\n  return!/[^a-zA-Z]/.test(str);\n};\ngoog.string.isNumeric = function(str) {\n  return!/[^0-9]/.test(str);\n};\ngoog.string.isAlphaNumeric = function(str) {\n  return!/[^a-zA-Z0-9]/.test(str);\n};\ngoog.string.isSpace = function(ch) {\n  return ch == &quot; &quot;;\n};\ngoog.string.isUnicodeChar = function(ch) {\n  return ch.length == 1 &amp;&amp; ch &gt;= &quot; &quot; &amp;&amp; ch &lt;= &quot;~&quot; || ch &gt;= &quot;\\u0080&quot; &amp;&amp; ch &lt;= &quot;\\ufffd&quot;;\n};\ngoog.string.stripNewlines = function(str) {\n  return str.replace(/(\\r\\n|\\r|\\n)+/g, &quot; &quot;);\n};\ngoog.string.canonicalizeNewlines = function(str) {\n  return str.replace(/(\\r\\n|\\r|\\n)/g, &quot;\\n&quot;);\n};\ngoog.string.normalizeWhitespace = function(str) {\n  return str.replace(/\\xa0|\\s/g, &quot; &quot;);\n};\ngoog.string.normalizeSpaces = function(str) {\n  return str.replace(/\\xa0|[ \\t]+/g, &quot; &quot;);\n};\ngoog.string.collapseBreakingSpaces = function(str) {\n  return str.replace(/[\\t\\r\\n ]+/g, &quot; &quot;).replace(/^[\\t\\r\\n ]+|[\\t\\r\\n ]+$/g, &quot;&quot;);\n};\ngoog.string.trim = function(str) {\n  return str.replace(/^[\\s\\xa0]+|[\\s\\xa0]+$/g, &quot;&quot;);\n};\ngoog.string.trimLeft = function(str) {\n  return str.replace(/^[\\s\\xa0]+/, &quot;&quot;);\n};\ngoog.string.trimRight = function(str) {\n  return str.replace(/[\\s\\xa0]+$/, &quot;&quot;);\n};\ngoog.string.caseInsensitiveCompare = function(str1, str2) {\n  var test1 = String(str1).toLowerCase();\n  var test2 = String(str2).toLowerCase();\n  if (test1 &lt; test2) {\n    return-1;\n  } else {\n    if (test1 == test2) {\n      return 0;\n    } else {\n      return 1;\n    }\n  }\n};\ngoog.string.numerateCompareRegExp_ = /(\\.\\d+)|(\\d+)|(\\D+)/g;\ngoog.string.numerateCompare = function(str1, str2) {\n  if (str1 == str2) {\n    return 0;\n  }\n  if (!str1) {\n    return-1;\n  }\n  if (!str2) {\n    return 1;\n  }\n  var tokens1 = str1.toLowerCase().match(goog.string.numerateCompareRegExp_);\n  var tokens2 = str2.toLowerCase().match(goog.string.numerateCompareRegExp_);\n  var count = Math.min(tokens1.length, tokens2.length);\n  for (var i = 0;i &lt; count;i++) {\n    var a = tokens1[i];\n    var b = tokens2[i];\n    if (a != b) {\n      var num1 = parseInt(a, 10);\n      if (!isNaN(num1)) {\n        var num2 = parseInt(b, 10);\n        if (!isNaN(num2) &amp;&amp; num1 - num2) {\n          return num1 - num2;\n        }\n      }\n      return a &lt; b ? -1 : 1;\n    }\n  }\n  if (tokens1.length != tokens2.length) {\n    return tokens1.length - tokens2.length;\n  }\n  return str1 &lt; str2 ? -1 : 1;\n};\ngoog.string.urlEncode = function(str) {\n  return encodeURIComponent(String(str));\n};\ngoog.string.urlDecode = function(str) {\n  return decodeURIComponent(str.replace(/\\+/g, &quot; &quot;));\n};\ngoog.string.newLineToBr = function(str, opt_xml) {\n  return str.replace(/(\\r\\n|\\r|\\n)/g, opt_xml ? &quot;\\x3cbr /\\x3e&quot; : &quot;\\x3cbr\\x3e&quot;);\n};\ngoog.string.htmlEscape = function(str, opt_isLikelyToContainHtmlChars) {\n  if (opt_isLikelyToContainHtmlChars) {\n    str = str.replace(goog.string.AMP_RE_, &quot;\\x26amp;&quot;).replace(goog.string.LT_RE_, &quot;\\x26lt;&quot;).replace(goog.string.GT_RE_, &quot;\\x26gt;&quot;).replace(goog.string.QUOT_RE_, &quot;\\x26quot;&quot;).replace(goog.string.SINGLE_QUOTE_RE_, &quot;\\x26#39;&quot;).replace(goog.string.NULL_RE_, &quot;\\x26#0;&quot;);\n    if (goog.string.DETECT_DOUBLE_ESCAPING) {\n      str = str.replace(goog.string.E_RE_, &quot;\\x26#101;&quot;);\n    }\n    return str;\n  } else {\n    if (!goog.string.ALL_RE_.test(str)) {\n      return str;\n    }\n    if (str.indexOf(&quot;\\x26&quot;) != -1) {\n      str = str.replace(goog.string.AMP_RE_, &quot;\\x26amp;&quot;);\n    }\n    if (str.indexOf(&quot;\\x3c&quot;) != -1) {\n      str = str.replace(goog.string.LT_RE_, &quot;\\x26lt;&quot;);\n    }\n    if (str.indexOf(&quot;\\x3e&quot;) != -1) {\n      str = str.replace(goog.string.GT_RE_, &quot;\\x26gt;&quot;);\n    }\n    if (str.indexOf(&apos;&quot;&apos;) != -1) {\n      str = str.replace(goog.string.QUOT_RE_, &quot;\\x26quot;&quot;);\n    }\n    if (str.indexOf(&quot;&apos;&quot;) != -1) {\n      str = str.replace(goog.string.SINGLE_QUOTE_RE_, &quot;\\x26#39;&quot;);\n    }\n    if (str.indexOf(&quot;\\x00&quot;) != -1) {\n      str = str.replace(goog.string.NULL_RE_, &quot;\\x26#0;&quot;);\n    }\n    if (goog.string.DETECT_DOUBLE_ESCAPING &amp;&amp; str.indexOf(&quot;e&quot;) != -1) {\n      str = str.replace(goog.string.E_RE_, &quot;\\x26#101;&quot;);\n    }\n    return str;\n  }\n};\ngoog.string.AMP_RE_ = /&amp;/g;\ngoog.string.LT_RE_ = /&lt;/g;\ngoog.string.GT_RE_ = /&gt;/g;\ngoog.string.QUOT_RE_ = /&quot;/g;\ngoog.string.SINGLE_QUOTE_RE_ = /&apos;/g;\ngoog.string.NULL_RE_ = /\\x00/g;\ngoog.string.E_RE_ = /e/g;\ngoog.string.ALL_RE_ = goog.string.DETECT_DOUBLE_ESCAPING ? /[\\x00&amp;&lt;&gt;&quot;&apos;e]/ : /[\\x00&amp;&lt;&gt;&quot;&apos;]/;\ngoog.string.unescapeEntities = function(str) {\n  if (goog.string.contains(str, &quot;\\x26&quot;)) {\n    if (&quot;document&quot; in goog.global) {\n      return goog.string.unescapeEntitiesUsingDom_(str);\n    } else {\n      return goog.string.unescapePureXmlEntities_(str);\n    }\n  }\n  return str;\n};\ngoog.string.unescapeEntitiesWithDocument = function(str, document) {\n  if (goog.string.contains(str, &quot;\\x26&quot;)) {\n    return goog.string.unescapeEntitiesUsingDom_(str, document);\n  }\n  return str;\n};\ngoog.string.unescapeEntitiesUsingDom_ = function(str, opt_document) {\n  var seen = {&quot;\\x26amp;&quot;:&quot;\\x26&quot;, &quot;\\x26lt;&quot;:&quot;\\x3c&quot;, &quot;\\x26gt;&quot;:&quot;\\x3e&quot;, &quot;\\x26quot;&quot;:&apos;&quot;&apos;};\n  var div;\n  if (opt_document) {\n    div = opt_document.createElement(&quot;div&quot;);\n  } else {\n    div = goog.global.document.createElement(&quot;div&quot;);\n  }\n  return str.replace(goog.string.HTML_ENTITY_PATTERN_, function(s, entity) {\n    var value = seen[s];\n    if (value) {\n      return value;\n    }\n    if (entity.charAt(0) == &quot;#&quot;) {\n      var n = Number(&quot;0&quot; + entity.substr(1));\n      if (!isNaN(n)) {\n        value = String.fromCharCode(n);\n      }\n    }\n    if (!value) {\n      div.innerHTML = s + &quot; &quot;;\n      value = div.firstChild.nodeValue.slice(0, -1);\n    }\n    return seen[s] = value;\n  });\n};\ngoog.string.unescapePureXmlEntities_ = function(str) {\n  return str.replace(/&amp;([^;]+);/g, function(s, entity) {\n    switch(entity) {\n      case &quot;amp&quot;:\n        return &quot;\\x26&quot;;\n      case &quot;lt&quot;:\n        return &quot;\\x3c&quot;;\n      case &quot;gt&quot;:\n        return &quot;\\x3e&quot;;\n      case &quot;quot&quot;:\n        return&apos;&quot;&apos;;\n      default:\n        if (entity.charAt(0) == &quot;#&quot;) {\n          var n = Number(&quot;0&quot; + entity.substr(1));\n          if (!isNaN(n)) {\n            return String.fromCharCode(n);\n          }\n        }\n        return s;\n    }\n  });\n};\ngoog.string.HTML_ENTITY_PATTERN_ = /&amp;([^;\\s&lt;&amp;]+);?/g;\ngoog.string.whitespaceEscape = function(str, opt_xml) {\n  return goog.string.newLineToBr(str.replace(/  /g, &quot; \\x26#160;&quot;), opt_xml);\n};\ngoog.string.preserveSpaces = function(str) {\n  return str.replace(/(^|[\\n ]) /g, &quot;$1&quot; + goog.string.Unicode.NBSP);\n};\ngoog.string.stripQuotes = function(str, quoteChars) {\n  var length = quoteChars.length;\n  for (var i = 0;i &lt; length;i++) {\n    var quoteChar = length == 1 ? quoteChars : quoteChars.charAt(i);\n    if (str.charAt(0) == quoteChar &amp;&amp; str.charAt(str.length - 1) == quoteChar) {\n      return str.substring(1, str.length - 1);\n    }\n  }\n  return str;\n};\ngoog.string.truncate = function(str, chars, opt_protectEscapedCharacters) {\n  if (opt_protectEscapedCharacters) {\n    str = goog.string.unescapeEntities(str);\n  }\n  if (str.length &gt; chars) {\n    str = str.substring(0, chars - 3) + &quot;...&quot;;\n  }\n  if (opt_protectEscapedCharacters) {\n    str = goog.string.htmlEscape(str);\n  }\n  return str;\n};\ngoog.string.truncateMiddle = function(str, chars, opt_protectEscapedCharacters, opt_trailingChars) {\n  if (opt_protectEscapedCharacters) {\n    str = goog.string.unescapeEntities(str);\n  }\n  if (opt_trailingChars &amp;&amp; str.length &gt; chars) {\n    if (opt_trailingChars &gt; chars) {\n      opt_trailingChars = chars;\n    }\n    var endPoint = str.length - opt_trailingChars;\n    var startPoint = chars - opt_trailingChars;\n    str = str.substring(0, startPoint) + &quot;...&quot; + str.substring(endPoint);\n  } else {\n    if (str.length &gt; chars) {\n      var half = Math.floor(chars / 2);\n      var endPos = str.length - half;\n      half += chars % 2;\n      str = str.substring(0, half) + &quot;...&quot; + str.substring(endPos);\n    }\n  }\n  if (opt_protectEscapedCharacters) {\n    str = goog.string.htmlEscape(str);\n  }\n  return str;\n};\ngoog.string.specialEscapeChars_ = {&quot;\\x00&quot;:&quot;\\\\0&quot;, &quot;\\b&quot;:&quot;\\\\b&quot;, &quot;\\f&quot;:&quot;\\\\f&quot;, &quot;\\n&quot;:&quot;\\\\n&quot;, &quot;\\r&quot;:&quot;\\\\r&quot;, &quot;\\t&quot;:&quot;\\\\t&quot;, &quot;\\x0B&quot;:&quot;\\\\x0B&quot;, &apos;&quot;&apos;:&apos;\\\\&quot;&apos;, &quot;\\\\&quot;:&quot;\\\\\\\\&quot;};\ngoog.string.jsEscapeCache_ = {&quot;&apos;&quot;:&quot;\\\\&apos;&quot;};\ngoog.string.quote = function(s) {\n  s = String(s);\n  if (s.quote) {\n    return s.quote();\n  } else {\n    var sb = [&apos;&quot;&apos;];\n    for (var i = 0;i &lt; s.length;i++) {\n      var ch = s.charAt(i);\n      var cc = ch.charCodeAt(0);\n      sb[i + 1] = goog.string.specialEscapeChars_[ch] || (cc &gt; 31 &amp;&amp; cc &lt; 127 ? ch : goog.string.escapeChar(ch));\n    }\n    sb.push(&apos;&quot;&apos;);\n    return sb.join(&quot;&quot;);\n  }\n};\ngoog.string.escapeString = function(str) {\n  var sb = [];\n  for (var i = 0;i &lt; str.length;i++) {\n    sb[i] = goog.string.escapeChar(str.charAt(i));\n  }\n  return sb.join(&quot;&quot;);\n};\ngoog.string.escapeChar = function(c) {\n  if (c in goog.string.jsEscapeCache_) {\n    return goog.string.jsEscapeCache_[c];\n  }\n  if (c in goog.string.specialEscapeChars_) {\n    return goog.string.jsEscapeCache_[c] = goog.string.specialEscapeChars_[c];\n  }\n  var rv = c;\n  var cc = c.charCodeAt(0);\n  if (cc &gt; 31 &amp;&amp; cc &lt; 127) {\n    rv = c;\n  } else {\n    if (cc &lt; 256) {\n      rv = &quot;\\\\x&quot;;\n      if (cc &lt; 16 || cc &gt; 256) {\n        rv += &quot;0&quot;;\n      }\n    } else {\n      rv = &quot;\\\\u&quot;;\n      if (cc &lt; 4096) {\n        rv += &quot;0&quot;;\n      }\n    }\n    rv += cc.toString(16).toUpperCase();\n  }\n  return goog.string.jsEscapeCache_[c] = rv;\n};\ngoog.string.toMap = function(s) {\n  var rv = {};\n  for (var i = 0;i &lt; s.length;i++) {\n    rv[s.charAt(i)] = true;\n  }\n  return rv;\n};\ngoog.string.contains = function(str, subString) {\n  return str.indexOf(subString) != -1;\n};\ngoog.string.caseInsensitiveContains = function(str, subString) {\n  return goog.string.contains(str.toLowerCase(), subString.toLowerCase());\n};\ngoog.string.countOf = function(s, ss) {\n  return s &amp;&amp; ss ? s.split(ss).length - 1 : 0;\n};\ngoog.string.removeAt = function(s, index, stringLength) {\n  var resultStr = s;\n  if (index &gt;= 0 &amp;&amp; index &lt; s.length &amp;&amp; stringLength &gt; 0) {\n    resultStr = s.substr(0, index) + s.substr(index + stringLength, s.length - index - stringLength);\n  }\n  return resultStr;\n};\ngoog.string.remove = function(s, ss) {\n  var re = new RegExp(goog.string.regExpEscape(ss), &quot;&quot;);\n  return s.replace(re, &quot;&quot;);\n};\ngoog.string.removeAll = function(s, ss) {\n  var re = new RegExp(goog.string.regExpEscape(ss), &quot;g&quot;);\n  return s.replace(re, &quot;&quot;);\n};\ngoog.string.regExpEscape = function(s) {\n  return String(s).replace(/([-()\\[\\]{}+?*.$\\^|,:#&lt;!\\\\])/g, &quot;\\\\$1&quot;).replace(/\\x08/g, &quot;\\\\x08&quot;);\n};\ngoog.string.repeat = function(string, length) {\n  return(new Array(length + 1)).join(string);\n};\ngoog.string.padNumber = function(num, length, opt_precision) {\n  var s = goog.isDef(opt_precision) ? num.toFixed(opt_precision) : String(num);\n  var index = s.indexOf(&quot;.&quot;);\n  if (index == -1) {\n    index = s.length;\n  }\n  return goog.string.repeat(&quot;0&quot;, Math.max(0, length - index)) + s;\n};\ngoog.string.makeSafe = function(obj) {\n  return obj == null ? &quot;&quot; : String(obj);\n};\ngoog.string.buildString = function(var_args) {\n  return Array.prototype.join.call(arguments, &quot;&quot;);\n};\ngoog.string.getRandomString = function() {\n  var x = 2147483648;\n  return Math.floor(Math.random() * x).toString(36) + Math.abs(Math.floor(Math.random() * x) ^ goog.now()).toString(36);\n};\ngoog.string.compareVersions = function(version1, version2) {\n  var order = 0;\n  var v1Subs = goog.string.trim(String(version1)).split(&quot;.&quot;);\n  var v2Subs = goog.string.trim(String(version2)).split(&quot;.&quot;);\n  var subCount = Math.max(v1Subs.length, v2Subs.length);\n  for (var subIdx = 0;order == 0 &amp;&amp; subIdx &lt; subCount;subIdx++) {\n    var v1Sub = v1Subs[subIdx] || &quot;&quot;;\n    var v2Sub = v2Subs[subIdx] || &quot;&quot;;\n    var v1CompParser = new RegExp(&quot;(\\\\d*)(\\\\D*)&quot;, &quot;g&quot;);\n    var v2CompParser = new RegExp(&quot;(\\\\d*)(\\\\D*)&quot;, &quot;g&quot;);\n    do {\n      var v1Comp = v1CompParser.exec(v1Sub) || [&quot;&quot;, &quot;&quot;, &quot;&quot;];\n      var v2Comp = v2CompParser.exec(v2Sub) || [&quot;&quot;, &quot;&quot;, &quot;&quot;];\n      if (v1Comp[0].length == 0 &amp;&amp; v2Comp[0].length == 0) {\n        break;\n      }\n      var v1CompNum = v1Comp[1].length == 0 ? 0 : parseInt(v1Comp[1], 10);\n      var v2CompNum = v2Comp[1].length == 0 ? 0 : parseInt(v2Comp[1], 10);\n      order = goog.string.compareElements_(v1CompNum, v2CompNum) || goog.string.compareElements_(v1Comp[2].length == 0, v2Comp[2].length == 0) || goog.string.compareElements_(v1Comp[2], v2Comp[2]);\n    } while (order == 0);\n  }\n  return order;\n};\ngoog.string.compareElements_ = function(left, right) {\n  if (left &lt; right) {\n    return-1;\n  } else {\n    if (left &gt; right) {\n      return 1;\n    }\n  }\n  return 0;\n};\ngoog.string.HASHCODE_MAX_ = 4294967296;\ngoog.string.hashCode = function(str) {\n  var result = 0;\n  for (var i = 0;i &lt; str.length;++i) {\n    result = 31 * result + str.charCodeAt(i);\n    result %= goog.string.HASHCODE_MAX_;\n  }\n  return result;\n};\ngoog.string.uniqueStringCounter_ = Math.random() * 2147483648 | 0;\ngoog.string.createUniqueString = function() {\n  return &quot;goog_&quot; + goog.string.uniqueStringCounter_++;\n};\ngoog.string.toNumber = function(str) {\n  var num = Number(str);\n  if (num == 0 &amp;&amp; goog.string.isEmpty(str)) {\n    return NaN;\n  }\n  return num;\n};\ngoog.string.isLowerCamelCase = function(str) {\n  return/^[a-z]+([A-Z][a-z]*)*$/.test(str);\n};\ngoog.string.isUpperCamelCase = function(str) {\n  return/^([A-Z][a-z]*)+$/.test(str);\n};\ngoog.string.toCamelCase = function(str) {\n  return String(str).replace(/\\-([a-z])/g, function(all, match) {\n    return match.toUpperCase();\n  });\n};\ngoog.string.toSelectorCase = function(str) {\n  return String(str).replace(/([A-Z])/g, &quot;-$1&quot;).toLowerCase();\n};\ngoog.string.toTitleCase = function(str, opt_delimiters) {\n  var delimiters = goog.isString(opt_delimiters) ? goog.string.regExpEscape(opt_delimiters) : &quot;\\\\s&quot;;\n  delimiters = delimiters ? &quot;|[&quot; + delimiters + &quot;]+&quot; : &quot;&quot;;\n  var regexp = new RegExp(&quot;(^&quot; + delimiters + &quot;)([a-z])&quot;, &quot;g&quot;);\n  return str.replace(regexp, function(all, p1, p2) {\n    return p1 + p2.toUpperCase();\n  });\n};\ngoog.string.parseInt = function(value) {\n  if (isFinite(value)) {\n    value = String(value);\n  }\n  if (goog.isString(value)) {\n    return/^\\s*-?0x/i.test(value) ? parseInt(value, 16) : parseInt(value, 10);\n  }\n  return NaN;\n};\ngoog.string.splitLimit = function(str, separator, limit) {\n  var parts = str.split(separator);\n  var returnVal = [];\n  while (limit &gt; 0 &amp;&amp; parts.length) {\n    returnVal.push(parts.shift());\n    limit--;\n  }\n  if (parts.length) {\n    returnVal.push(parts.join(separator));\n  }\n  return returnVal;\n};\ngoog.provide(&quot;goog.object&quot;);\ngoog.object.forEach = function(obj, f, opt_obj) {\n  for (var key in obj) {\n    f.call(opt_obj, obj[key], key, obj);\n  }\n};\ngoog.object.filter = function(obj, f, opt_obj) {\n  var res = {};\n  for (var key in obj) {\n    if (f.call(opt_obj, obj[key], key, obj)) {\n      res[key] = obj[key];\n    }\n  }\n  return res;\n};\ngoog.object.map = function(obj, f, opt_obj) {\n  var res = {};\n  for (var key in obj) {\n    res[key] = f.call(opt_obj, obj[key], key, obj);\n  }\n  return res;\n};\ngoog.object.some = function(obj, f, opt_obj) {\n  for (var key in obj) {\n    if (f.call(opt_obj, obj[key], key, obj)) {\n      return true;\n    }\n  }\n  return false;\n};\ngoog.object.every = function(obj, f, opt_obj) {\n  for (var key in obj) {\n    if (!f.call(opt_obj, obj[key], key, obj)) {\n      return false;\n    }\n  }\n  return true;\n};\ngoog.object.getCount = function(obj) {\n  var rv = 0;\n  for (var key in obj) {\n    rv++;\n  }\n  return rv;\n};\ngoog.object.getAnyKey = function(obj) {\n  for (var key in obj) {\n    return key;\n  }\n};\ngoog.object.getAnyValue = function(obj) {\n  for (var key in obj) {\n    return obj[key];\n  }\n};\ngoog.object.contains = function(obj, val) {\n  return goog.object.containsValue(obj, val);\n};\ngoog.object.getValues = function(obj) {\n  var res = [];\n  var i = 0;\n  for (var key in obj) {\n    res[i++] = obj[key];\n  }\n  return res;\n};\ngoog.object.getKeys = function(obj) {\n  var res = [];\n  var i = 0;\n  for (var key in obj) {\n    res[i++] = key;\n  }\n  return res;\n};\ngoog.object.getValueByKeys = function(obj, var_args) {\n  var isArrayLike = goog.isArrayLike(var_args);\n  var keys = isArrayLike ? var_args : arguments;\n  for (var i = isArrayLike ? 0 : 1;i &lt; keys.length;i++) {\n    obj = obj[keys[i]];\n    if (!goog.isDef(obj)) {\n      break;\n    }\n  }\n  return obj;\n};\ngoog.object.containsKey = function(obj, key) {\n  return key in obj;\n};\ngoog.object.containsValue = function(obj, val) {\n  for (var key in obj) {\n    if (obj[key] == val) {\n      return true;\n    }\n  }\n  return false;\n};\ngoog.object.findKey = function(obj, f, opt_this) {\n  for (var key in obj) {\n    if (f.call(opt_this, obj[key], key, obj)) {\n      return key;\n    }\n  }\n  return undefined;\n};\ngoog.object.findValue = function(obj, f, opt_this) {\n  var key = goog.object.findKey(obj, f, opt_this);\n  return key &amp;&amp; obj[key];\n};\ngoog.object.isEmpty = function(obj) {\n  for (var key in obj) {\n    return false;\n  }\n  return true;\n};\ngoog.object.clear = function(obj) {\n  for (var i in obj) {\n    delete obj[i];\n  }\n};\ngoog.object.remove = function(obj, key) {\n  var rv;\n  if (rv = key in obj) {\n    delete obj[key];\n  }\n  return rv;\n};\ngoog.object.add = function(obj, key, val) {\n  if (key in obj) {\n    throw Error(&apos;The object already contains the key &quot;&apos; + key + &apos;&quot;&apos;);\n  }\n  goog.object.set(obj, key, val);\n};\ngoog.object.get = function(obj, key, opt_val) {\n  if (key in obj) {\n    return obj[key];\n  }\n  return opt_val;\n};\ngoog.object.set = function(obj, key, value) {\n  obj[key] = value;\n};\ngoog.object.setIfUndefined = function(obj, key, value) {\n  return key in obj ? obj[key] : obj[key] = value;\n};\ngoog.object.clone = function(obj) {\n  var res = {};\n  for (var key in obj) {\n    res[key] = obj[key];\n  }\n  return res;\n};\ngoog.object.unsafeClone = function(obj) {\n  var type = goog.typeOf(obj);\n  if (type == &quot;object&quot; || type == &quot;array&quot;) {\n    if (obj.clone) {\n      return obj.clone();\n    }\n    var clone = type == &quot;array&quot; ? [] : {};\n    for (var key in obj) {\n      clone[key] = goog.object.unsafeClone(obj[key]);\n    }\n    return clone;\n  }\n  return obj;\n};\ngoog.object.transpose = function(obj) {\n  var transposed = {};\n  for (var key in obj) {\n    transposed[obj[key]] = key;\n  }\n  return transposed;\n};\ngoog.object.PROTOTYPE_FIELDS_ = [&quot;constructor&quot;, &quot;hasOwnProperty&quot;, &quot;isPrototypeOf&quot;, &quot;propertyIsEnumerable&quot;, &quot;toLocaleString&quot;, &quot;toString&quot;, &quot;valueOf&quot;];\ngoog.object.extend = function(target, var_args) {\n  var key, source;\n  for (var i = 1;i &lt; arguments.length;i++) {\n    source = arguments[i];\n    for (key in source) {\n      target[key] = source[key];\n    }\n    for (var j = 0;j &lt; goog.object.PROTOTYPE_FIELDS_.length;j++) {\n      key = goog.object.PROTOTYPE_FIELDS_[j];\n      if (Object.prototype.hasOwnProperty.call(source, key)) {\n        target[key] = source[key];\n      }\n    }\n  }\n};\ngoog.object.create = function(var_args) {\n  var argLength = arguments.length;\n  if (argLength == 1 &amp;&amp; goog.isArray(arguments[0])) {\n    return goog.object.create.apply(null, arguments[0]);\n  }\n  if (argLength % 2) {\n    throw Error(&quot;Uneven number of arguments&quot;);\n  }\n  var rv = {};\n  for (var i = 0;i &lt; argLength;i += 2) {\n    rv[arguments[i]] = arguments[i + 1];\n  }\n  return rv;\n};\ngoog.object.createSet = function(var_args) {\n  var argLength = arguments.length;\n  if (argLength == 1 &amp;&amp; goog.isArray(arguments[0])) {\n    return goog.object.createSet.apply(null, arguments[0]);\n  }\n  var rv = {};\n  for (var i = 0;i &lt; argLength;i++) {\n    rv[arguments[i]] = true;\n  }\n  return rv;\n};\ngoog.object.createImmutableView = function(obj) {\n  var result = obj;\n  if (Object.isFrozen &amp;&amp; !Object.isFrozen(obj)) {\n    result = Object.create(obj);\n    Object.freeze(result);\n  }\n  return result;\n};\ngoog.object.isImmutableView = function(obj) {\n  return!!Object.isFrozen &amp;&amp; Object.isFrozen(obj);\n};\ngoog.provide(&quot;goog.string.StringBuffer&quot;);\ngoog.string.StringBuffer = function(opt_a1, var_args) {\n  if (opt_a1 != null) {\n    this.append.apply(this, arguments);\n  }\n};\ngoog.string.StringBuffer.prototype.buffer_ = &quot;&quot;;\ngoog.string.StringBuffer.prototype.set = function(s) {\n  this.buffer_ = &quot;&quot; + s;\n};\ngoog.string.StringBuffer.prototype.append = function(a1, opt_a2, var_args) {\n  this.buffer_ += a1;\n  if (opt_a2 != null) {\n    for (var i = 1;i &lt; arguments.length;i++) {\n      this.buffer_ += arguments[i];\n    }\n  }\n  return this;\n};\ngoog.string.StringBuffer.prototype.clear = function() {\n  this.buffer_ = &quot;&quot;;\n};\ngoog.string.StringBuffer.prototype.getLength = function() {\n  return this.buffer_.length;\n};\ngoog.string.StringBuffer.prototype.toString = function() {\n  return this.buffer_;\n};\ngoog.provide(&quot;goog.debug.Error&quot;);\ngoog.debug.Error = function(opt_msg) {\n  if (Error.captureStackTrace) {\n    Error.captureStackTrace(this, goog.debug.Error);\n  } else {\n    var stack = (new Error).stack;\n    if (stack) {\n      this.stack = stack;\n    }\n  }\n  if (opt_msg) {\n    this.message = String(opt_msg);\n  }\n};\ngoog.inherits(goog.debug.Error, Error);\ngoog.debug.Error.prototype.name = &quot;CustomError&quot;;\ngoog.provide(&quot;goog.dom.NodeType&quot;);\ngoog.dom.NodeType = {ELEMENT:1, ATTRIBUTE:2, TEXT:3, CDATA_SECTION:4, ENTITY_REFERENCE:5, ENTITY:6, PROCESSING_INSTRUCTION:7, COMMENT:8, DOCUMENT:9, DOCUMENT_TYPE:10, DOCUMENT_FRAGMENT:11, NOTATION:12};\ngoog.provide(&quot;goog.asserts&quot;);\ngoog.provide(&quot;goog.asserts.AssertionError&quot;);\ngoog.require(&quot;goog.debug.Error&quot;);\ngoog.require(&quot;goog.dom.NodeType&quot;);\ngoog.require(&quot;goog.string&quot;);\ngoog.define(&quot;goog.asserts.ENABLE_ASSERTS&quot;, goog.DEBUG);\ngoog.asserts.AssertionError = function(messagePattern, messageArgs) {\n  messageArgs.unshift(messagePattern);\n  goog.debug.Error.call(this, goog.string.subs.apply(null, messageArgs));\n  messageArgs.shift();\n  this.messagePattern = messagePattern;\n};\ngoog.inherits(goog.asserts.AssertionError, goog.debug.Error);\ngoog.asserts.AssertionError.prototype.name = &quot;AssertionError&quot;;\ngoog.asserts.DEFAULT_ERROR_HANDLER = function(e) {\n  throw e;\n};\ngoog.asserts.errorHandler_ = goog.asserts.DEFAULT_ERROR_HANDLER;\ngoog.asserts.doAssertFailure_ = function(defaultMessage, defaultArgs, givenMessage, givenArgs) {\n  var message = &quot;Assertion failed&quot;;\n  if (givenMessage) {\n    message += &quot;: &quot; + givenMessage;\n    var args = givenArgs;\n  } else {\n    if (defaultMessage) {\n      message += &quot;: &quot; + defaultMessage;\n      args = defaultArgs;\n    }\n  }\n  var e = new goog.asserts.AssertionError(&quot;&quot; + message, args || []);\n  goog.asserts.errorHandler_(e);\n};\ngoog.asserts.setErrorHandler = function(errorHandler) {\n  if (goog.asserts.ENABLE_ASSERTS) {\n    goog.asserts.errorHandler_ = errorHandler;\n  }\n};\ngoog.asserts.assert = function(condition, opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS &amp;&amp; !condition) {\n    goog.asserts.doAssertFailure_(&quot;&quot;, null, opt_message, Array.prototype.slice.call(arguments, 2));\n  }\n  return condition;\n};\ngoog.asserts.fail = function(opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS) {\n    goog.asserts.errorHandler_(new goog.asserts.AssertionError(&quot;Failure&quot; + (opt_message ? &quot;: &quot; + opt_message : &quot;&quot;), Array.prototype.slice.call(arguments, 1)));\n  }\n};\ngoog.asserts.assertNumber = function(value, opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS &amp;&amp; !goog.isNumber(value)) {\n    goog.asserts.doAssertFailure_(&quot;Expected number but got %s: %s.&quot;, [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));\n  }\n  return(value);\n};\ngoog.asserts.assertString = function(value, opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS &amp;&amp; !goog.isString(value)) {\n    goog.asserts.doAssertFailure_(&quot;Expected string but got %s: %s.&quot;, [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));\n  }\n  return(value);\n};\ngoog.asserts.assertFunction = function(value, opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS &amp;&amp; !goog.isFunction(value)) {\n    goog.asserts.doAssertFailure_(&quot;Expected function but got %s: %s.&quot;, [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));\n  }\n  return(value);\n};\ngoog.asserts.assertObject = function(value, opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS &amp;&amp; !goog.isObject(value)) {\n    goog.asserts.doAssertFailure_(&quot;Expected object but got %s: %s.&quot;, [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));\n  }\n  return(value);\n};\ngoog.asserts.assertArray = function(value, opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS &amp;&amp; !goog.isArray(value)) {\n    goog.asserts.doAssertFailure_(&quot;Expected array but got %s: %s.&quot;, [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));\n  }\n  return(value);\n};\ngoog.asserts.assertBoolean = function(value, opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS &amp;&amp; !goog.isBoolean(value)) {\n    goog.asserts.doAssertFailure_(&quot;Expected boolean but got %s: %s.&quot;, [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));\n  }\n  return(value);\n};\ngoog.asserts.assertElement = function(value, opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS &amp;&amp; (!goog.isObject(value) || value.nodeType != goog.dom.NodeType.ELEMENT)) {\n    goog.asserts.doAssertFailure_(&quot;Expected Element but got %s: %s.&quot;, [goog.typeOf(value), value], opt_message, Array.prototype.slice.call(arguments, 2));\n  }\n  return(value);\n};\ngoog.asserts.assertInstanceof = function(value, type, opt_message, var_args) {\n  if (goog.asserts.ENABLE_ASSERTS &amp;&amp; !(value instanceof type)) {\n    goog.asserts.doAssertFailure_(&quot;instanceof check failed.&quot;, null, opt_message, Array.prototype.slice.call(arguments, 3));\n  }\n  return value;\n};\ngoog.asserts.assertObjectPrototypeIsIntact = function() {\n  for (var key in Object.prototype) {\n    goog.asserts.fail(key + &quot; should not be enumerable in Object.prototype.&quot;);\n  }\n};\ngoog.provide(&quot;goog.array&quot;);\ngoog.provide(&quot;goog.array.ArrayLike&quot;);\ngoog.require(&quot;goog.asserts&quot;);\ngoog.define(&quot;goog.NATIVE_ARRAY_PROTOTYPES&quot;, goog.TRUSTED_SITE);\ngoog.define(&quot;goog.array.ASSUME_NATIVE_FUNCTIONS&quot;, false);\ngoog.array.ArrayLike;\ngoog.array.peek = function(array) {\n  return array[array.length - 1];\n};\ngoog.array.last = goog.array.peek;\ngoog.array.ARRAY_PROTOTYPE_ = Array.prototype;\ngoog.array.indexOf = goog.NATIVE_ARRAY_PROTOTYPES &amp;&amp; (goog.array.ASSUME_NATIVE_FUNCTIONS || goog.array.ARRAY_PROTOTYPE_.indexOf) ? function(arr, obj, opt_fromIndex) {\n  goog.asserts.assert(arr.length != null);\n  return goog.array.ARRAY_PROTOTYPE_.indexOf.call(arr, obj, opt_fromIndex);\n} : function(arr, obj, opt_fromIndex) {\n  var fromIndex = opt_fromIndex == null ? 0 : opt_fromIndex &lt; 0 ? Math.max(0, arr.length + opt_fromIndex) : opt_fromIndex;\n  if (goog.isString(arr)) {\n    if (!goog.isString(obj) || obj.length != 1) {\n      return-1;\n    }\n    return arr.indexOf(obj, fromIndex);\n  }\n  for (var i = fromIndex;i &lt; arr.length;i++) {\n    if (i in arr &amp;&amp; arr[i] === obj) {\n      return i;\n    }\n  }\n  return-1;\n};\ngoog.array.lastIndexOf = goog.NATIVE_ARRAY_PROTOTYPES &amp;&amp; (goog.array.ASSUME_NATIVE_FUNCTIONS || goog.array.ARRAY_PROTOTYPE_.lastIndexOf) ? function(arr, obj, opt_fromIndex) {\n  goog.asserts.assert(arr.length != null);\n  var fromIndex = opt_fromIndex == null ? arr.length - 1 : opt_fromIndex;\n  return goog.array.ARRAY_PROTOTYPE_.lastIndexOf.call(arr, obj, fromIndex);\n} : function(arr, obj, opt_fromIndex) {\n  var fromIndex = opt_fromIndex == null ? arr.length - 1 : opt_fromIndex;\n  if (fromIndex &lt; 0) {\n    fromIndex = Math.max(0, arr.length + fromIndex);\n  }\n  if (goog.isString(arr)) {\n    if (!goog.isString(obj) || obj.length != 1) {\n      return-1;\n    }\n    return arr.lastIndexOf(obj, fromIndex);\n  }\n  for (var i = fromIndex;i &gt;= 0;i--) {\n    if (i in arr &amp;&amp; arr[i] === obj) {\n      return i;\n    }\n  }\n  return-1;\n};\ngoog.array.forEach = goog.NATIVE_ARRAY_PROTOTYPES &amp;&amp; (goog.array.ASSUME_NATIVE_FUNCTIONS || goog.array.ARRAY_PROTOTYPE_.forEach) ? function(arr, f, opt_obj) {\n  goog.asserts.assert(arr.length != null);\n  goog.array.ARRAY_PROTOTYPE_.forEach.call(arr, f, opt_obj);\n} : function(arr, f, opt_obj) {\n  var l = arr.length;\n  var arr2 = goog.isString(arr) ? arr.split(&quot;&quot;) : arr;\n  for (var i = 0;i &lt; l;i++) {\n    if (i in arr2) {\n      f.call(opt_obj, arr2[i], i, arr);\n    }\n  }\n};\ngoog.array.forEachRight = function(arr, f, opt_obj) {\n  var l = arr.length;\n  var arr2 = goog.isString(arr) ? arr.split(&quot;&quot;) : arr;\n  for (var i = l - 1;i &gt;= 0;--i) {\n    if (i in arr2) {\n      f.call(opt_obj, arr2[i], i, arr);\n    }\n  }\n};\ngoog.array.filter = goog.NATIVE_ARRAY_PROTOTYPES &amp;&amp; (goog.array.ASSUME_NATIVE_FUNCTIONS || goog.array.ARRAY_PROTOTYPE_.filter) ? function(arr, f, opt_obj) {\n  goog.asserts.assert(arr.length != null);\n  return goog.array.ARRAY_PROTOTYPE_.filter.call(arr, f, opt_obj);\n} : function(arr, f, opt_obj) {\n  var l = arr.length;\n  var res = [];\n  var resLength = 0;\n  var arr2 = goog.isString(arr) ? arr.split(&quot;&quot;) : arr;\n  for (var i = 0;i &lt; l;i++) {\n    if (i in arr2) {\n      var val = arr2[i];\n      if (f.call(opt_obj, val, i, arr)) {\n        res[resLength++] = val;\n      }\n    }\n  }\n  return res;\n};\ngoog.array.map = goog.NATIVE_ARRAY_PROTOTYPES &amp;&amp; (goog.array.ASSUME_NATIVE_FUNCTIONS || goog.array.ARRAY_PROTOTYPE_.map) ? function(arr, f, opt_obj) {\n  goog.asserts.assert(arr.length != null);\n  return goog.array.ARRAY_PROTOTYPE_.map.call(arr, f, opt_obj);\n} : function(arr, f, opt_obj) {\n  var l = arr.length;\n  var res = new Array(l);\n  var arr2 = goog.isString(arr) ? arr.split(&quot;&quot;) : arr;\n  for (var i = 0;i &lt; l;i++) {\n    if (i in arr2) {\n      res[i] = f.call(opt_obj, arr2[i], i, arr);\n    }\n  }\n  return res;\n};\ngoog.array.reduce = goog.NATIVE_ARRAY_PROTOTYPES &amp;&amp; (goog.array.ASSUME_NATIVE_FUNCTIONS || goog.array.ARRAY_PROTOTYPE_.reduce) ? function(arr, f, val, opt_obj) {\n  goog.asserts.assert(arr.length != null);\n  if (opt_obj) {\n    f = goog.bind(f, opt_obj);\n  }\n  return goog.array.ARRAY_PROTOTYPE_.reduce.call(arr, f, val);\n} : function(arr, f, val, opt_obj) {\n  var rval = val;\n  goog.array.forEach(arr, function(val, index) {\n    rval = f.call(opt_obj, rval, val, index, arr);\n  });\n  return rval;\n};\ngoog.array.reduceRight = goog.NATIVE_ARRAY_PROTOTYPES &amp;&amp; (goog.array.ASSUME_NATIVE_FUNCTIONS || goog.array.ARRAY_PROTOTYPE_.reduceRight) ? function(arr, f, val, opt_obj) {\n  goog.asserts.assert(arr.length != null);\n  if (opt_obj) {\n    f = goog.bind(f, opt_obj);\n  }\n  return goog.array.ARRAY_PROTOTYPE_.reduceRight.call(arr, f, val);\n} : function(arr, f, val, opt_obj) {\n  var rval = val;\n  goog.array.forEachRight(arr, function(val, index) {\n    rval = f.call(opt_obj, rval, val, index, arr);\n  });\n  return rval;\n};\ngoog.array.some = goog.NATIVE_ARRAY_PROTOTYPES &amp;&amp; (goog.array.ASSUME_NATIVE_FUNCTIONS || goog.array.ARRAY_PROTOTYPE_.some) ? function(arr, f, opt_obj) {\n  goog.asserts.assert(arr.length != null);\n  return goog.array.ARRAY_PROTOTYPE_.some.call(arr, f, opt_obj);\n} : function(arr, f, opt_obj) {\n  var l = arr.length;\n  var arr2 = goog.isString(arr) ? arr.split(&quot;&quot;) : arr;\n  for (var i = 0;i &lt; l;i++) {\n    if (i in arr2 &amp;&amp; f.call(opt_obj, arr2[i], i, arr)) {\n      return true;\n    }\n  }\n  return false;\n};\ngoog.array.every = goog.NATIVE_ARRAY_PROTOTYPES &amp;&amp; (goog.array.ASSUME_NATIVE_FUNCTIONS || goog.array.ARRAY_PROTOTYPE_.every) ? function(arr, f, opt_obj) {\n  goog.asserts.assert(arr.length != null);\n  return goog.array.ARRAY_PROTOTYPE_.every.call(arr, f, opt_obj);\n} : function(arr, f, opt_obj) {\n  var l = arr.length;\n  var arr2 = goog.isString(arr) ? arr.split(&quot;&quot;) : arr;\n  for (var i = 0;i &lt; l;i++) {\n    if (i in arr2 &amp;&amp; !f.call(opt_obj, arr2[i], i, arr)) {\n      return false;\n    }\n  }\n  return true;\n};\ngoog.array.count = function(arr, f, opt_obj) {\n  var count = 0;\n  goog.array.forEach(arr, function(element, index, arr) {\n    if (f.call(opt_obj, element, index, arr)) {\n      ++count;\n    }\n  }, opt_obj);\n  return count;\n};\ngoog.array.find = function(arr, f, opt_obj) {\n  var i = goog.array.findIndex(arr, f, opt_obj);\n  return i &lt; 0 ? null : goog.isString(arr) ? arr.charAt(i) : arr[i];\n};\ngoog.array.findIndex = function(arr, f, opt_obj) {\n  var l = arr.length;\n  var arr2 = goog.isString(arr) ? arr.split(&quot;&quot;) : arr;\n  for (var i = 0;i &lt; l;i++) {\n    if (i in arr2 &amp;&amp; f.call(opt_obj, arr2[i], i, arr)) {\n      return i;\n    }\n  }\n  return-1;\n};\ngoog.array.findRight = function(arr, f, opt_obj) {\n  var i = goog.array.findIndexRight(arr, f, opt_obj);\n  return i &lt; 0 ? null : goog.isString(arr) ? arr.charAt(i) : arr[i];\n};\ngoog.array.findIndexRight = function(arr, f, opt_obj) {\n  var l = arr.length;\n  var arr2 = goog.isString(arr) ? arr.split(&quot;&quot;) : arr;\n  for (var i = l - 1;i &gt;= 0;i--) {\n    if (i in arr2 &amp;&amp; f.call(opt_obj, arr2[i], i, arr)) {\n      return i;\n    }\n  }\n  return-1;\n};\ngoog.array.contains = function(arr, obj) {\n  return goog.array.indexOf(arr, obj) &gt;= 0;\n};\ngoog.array.isEmpty = function(arr) {\n  return arr.length == 0;\n};\ngoog.array.clear = function(arr) {\n  if (!goog.isArray(arr)) {\n    for (var i = arr.length - 1;i &gt;= 0;i--) {\n      delete arr[i];\n    }\n  }\n  arr.length = 0;\n};\ngoog.array.insert = function(arr, obj) {\n  if (!goog.array.contains(arr, obj)) {\n    arr.push(obj);\n  }\n};\ngoog.array.insertAt = function(arr, obj, opt_i) {\n  goog.array.splice(arr, opt_i, 0, obj);\n};\ngoog.array.insertArrayAt = function(arr, elementsToAdd, opt_i) {\n  goog.partial(goog.array.splice, arr, opt_i, 0).apply(null, elementsToAdd);\n};\ngoog.array.insertBefore = function(arr, obj, opt_obj2) {\n  var i;\n  if (arguments.length == 2 || (i = goog.array.indexOf(arr, opt_obj2)) &lt; 0) {\n    arr.push(obj);\n  } else {\n    goog.array.insertAt(arr, obj, i);\n  }\n};\ngoog.array.remove = function(arr, obj) {\n  var i = goog.array.indexOf(arr, obj);\n  var rv;\n  if (rv = i &gt;= 0) {\n    goog.array.removeAt(arr, i);\n  }\n  return rv;\n};\ngoog.array.removeAt = function(arr, i) {\n  goog.asserts.assert(arr.length != null);\n  return goog.array.ARRAY_PROTOTYPE_.splice.call(arr, i, 1).length == 1;\n};\ngoog.array.removeIf = function(arr, f, opt_obj) {\n  var i = goog.array.findIndex(arr, f, opt_obj);\n  if (i &gt;= 0) {\n    goog.array.removeAt(arr, i);\n    return true;\n  }\n  return false;\n};\ngoog.array.concat = function(var_args) {\n  return goog.array.ARRAY_PROTOTYPE_.concat.apply(goog.array.ARRAY_PROTOTYPE_, arguments);\n};\ngoog.array.join = function(var_args) {\n  return goog.array.ARRAY_PROTOTYPE_.concat.apply(goog.array.ARRAY_PROTOTYPE_, arguments);\n};\ngoog.array.toArray = function(object) {\n  var length = object.length;\n  if (length &gt; 0) {\n    var rv = new Array(length);\n    for (var i = 0;i &lt; length;i++) {\n      rv[i] = object[i];\n    }\n    return rv;\n  }\n  return[];\n};\ngoog.array.clone = goog.array.toArray;\ngoog.array.extend = function(arr1, var_args) {\n  for (var i = 1;i &lt; arguments.length;i++) {\n    var arr2 = arguments[i];\n    var isArrayLike;\n    if (goog.isArray(arr2) || (isArrayLike = goog.isArrayLike(arr2)) &amp;&amp; Object.prototype.hasOwnProperty.call(arr2, &quot;callee&quot;)) {\n      arr1.push.apply(arr1, arr2);\n    } else {\n      if (isArrayLike) {\n        var len1 = arr1.length;\n        var len2 = arr2.length;\n        for (var j = 0;j &lt; len2;j++) {\n          arr1[len1 + j] = arr2[j];\n        }\n      } else {\n        arr1.push(arr2);\n      }\n    }\n  }\n};\ngoog.array.splice = function(arr, index, howMany, var_args) {\n  goog.asserts.assert(arr.length != null);\n  return goog.array.ARRAY_PROTOTYPE_.splice.apply(arr, goog.array.slice(arguments, 1));\n};\ngoog.array.slice = function(arr, start, opt_end) {\n  goog.asserts.assert(arr.length != null);\n  if (arguments.length &lt;= 2) {\n    return goog.array.ARRAY_PROTOTYPE_.slice.call(arr, start);\n  } else {\n    return goog.array.ARRAY_PROTOTYPE_.slice.call(arr, start, opt_end);\n  }\n};\ngoog.array.removeDuplicates = function(arr, opt_rv, opt_hashFn) {\n  var returnArray = opt_rv || arr;\n  var defaultHashFn = function(item) {\n    return goog.isObject(current) ? &quot;o&quot; + goog.getUid(current) : (typeof current).charAt(0) + current;\n  };\n  var hashFn = opt_hashFn || defaultHashFn;\n  var seen = {}, cursorInsert = 0, cursorRead = 0;\n  while (cursorRead &lt; arr.length) {\n    var current = arr[cursorRead++];\n    var key = hashFn(current);\n    if (!Object.prototype.hasOwnProperty.call(seen, key)) {\n      seen[key] = true;\n      returnArray[cursorInsert++] = current;\n    }\n  }\n  returnArray.length = cursorInsert;\n};\ngoog.array.binarySearch = function(arr, target, opt_compareFn) {\n  return goog.array.binarySearch_(arr, opt_compareFn || goog.array.defaultCompare, false, target);\n};\ngoog.array.binarySelect = function(arr, evaluator, opt_obj) {\n  return goog.array.binarySearch_(arr, evaluator, true, undefined, opt_obj);\n};\ngoog.array.binarySearch_ = function(arr, compareFn, isEvaluator, opt_target, opt_selfObj) {\n  var left = 0;\n  var right = arr.length;\n  var found;\n  while (left &lt; right) {\n    var middle = left + right &gt;&gt; 1;\n    var compareResult;\n    if (isEvaluator) {\n      compareResult = compareFn.call(opt_selfObj, arr[middle], middle, arr);\n    } else {\n      compareResult = compareFn(opt_target, arr[middle]);\n    }\n    if (compareResult &gt; 0) {\n      left = middle + 1;\n    } else {\n      right = middle;\n      found = !compareResult;\n    }\n  }\n  return found ? left : ~left;\n};\ngoog.array.sort = function(arr, opt_compareFn) {\n  arr.sort(opt_compareFn || goog.array.defaultCompare);\n};\ngoog.array.stableSort = function(arr, opt_compareFn) {\n  for (var i = 0;i &lt; arr.length;i++) {\n    arr[i] = {index:i, value:arr[i]};\n  }\n  var valueCompareFn = opt_compareFn || goog.array.defaultCompare;\n  function stableCompareFn(obj1, obj2) {\n    return valueCompareFn(obj1.value, obj2.value) || obj1.index - obj2.index;\n  }\n  goog.array.sort(arr, stableCompareFn);\n  for (var i = 0;i &lt; arr.length;i++) {\n    arr[i] = arr[i].value;\n  }\n};\ngoog.array.sortObjectsByKey = function(arr, key, opt_compareFn) {\n  var compare = opt_compareFn || goog.array.defaultCompare;\n  goog.array.sort(arr, function(a, b) {\n    return compare(a[key], b[key]);\n  });\n};\ngoog.array.isSorted = function(arr, opt_compareFn, opt_strict) {\n  var compare = opt_compareFn || goog.array.defaultCompare;\n  for (var i = 1;i &lt; arr.length;i++) {\n    var compareResult = compare(arr[i - 1], arr[i]);\n    if (compareResult &gt; 0 || compareResult == 0 &amp;&amp; opt_strict) {\n      return false;\n    }\n  }\n  return true;\n};\ngoog.array.equals = function(arr1, arr2, opt_equalsFn) {\n  if (!goog.isArrayLike(arr1) || !goog.isArrayLike(arr2) || arr1.length != arr2.length) {\n    return false;\n  }\n  var l = arr1.length;\n  var equalsFn = opt_equalsFn || goog.array.defaultCompareEquality;\n  for (var i = 0;i &lt; l;i++) {\n    if (!equalsFn(arr1[i], arr2[i])) {\n      return false;\n    }\n  }\n  return true;\n};\ngoog.array.compare3 = function(arr1, arr2, opt_compareFn) {\n  var compare = opt_compareFn || goog.array.defaultCompare;\n  var l = Math.min(arr1.length, arr2.length);\n  for (var i = 0;i &lt; l;i++) {\n    var result = compare(arr1[i], arr2[i]);\n    if (result != 0) {\n      return result;\n    }\n  }\n  return goog.array.defaultCompare(arr1.length, arr2.length);\n};\ngoog.array.defaultCompare = function(a, b) {\n  return a &gt; b ? 1 : a &lt; b ? -1 : 0;\n};\ngoog.array.defaultCompareEquality = function(a, b) {\n  return a === b;\n};\ngoog.array.binaryInsert = function(array, value, opt_compareFn) {\n  var index = goog.array.binarySearch(array, value, opt_compareFn);\n  if (index &lt; 0) {\n    goog.array.insertAt(array, value, -(index + 1));\n    return true;\n  }\n  return false;\n};\ngoog.array.binaryRemove = function(array, value, opt_compareFn) {\n  var index = goog.array.binarySearch(array, value, opt_compareFn);\n  return index &gt;= 0 ? goog.array.removeAt(array, index) : false;\n};\ngoog.array.bucket = function(array, sorter, opt_obj) {\n  var buckets = {};\n  for (var i = 0;i &lt; array.length;i++) {\n    var value = array[i];\n    var key = sorter.call(opt_obj, value, i, array);\n    if (goog.isDef(key)) {\n      var bucket = buckets[key] || (buckets[key] = []);\n      bucket.push(value);\n    }\n  }\n  return buckets;\n};\ngoog.array.toObject = function(arr, keyFunc, opt_obj) {\n  var ret = {};\n  goog.array.forEach(arr, function(element, index) {\n    ret[keyFunc.call(opt_obj, element, index, arr)] = element;\n  });\n  return ret;\n};\ngoog.array.range = function(startOrEnd, opt_end, opt_step) {\n  var array = [];\n  var start = 0;\n  var end = startOrEnd;\n  var step = opt_step || 1;\n  if (opt_end !== undefined) {\n    start = startOrEnd;\n    end = opt_end;\n  }\n  if (step * (end - start) &lt; 0) {\n    return[];\n  }\n  if (step &gt; 0) {\n    for (var i = start;i &lt; end;i += step) {\n      array.push(i);\n    }\n  } else {\n    for (var i = start;i &gt; end;i += step) {\n      array.push(i);\n    }\n  }\n  return array;\n};\ngoog.array.repeat = function(value, n) {\n  var array = [];\n  for (var i = 0;i &lt; n;i++) {\n    array[i] = value;\n  }\n  return array;\n};\ngoog.array.flatten = function(var_args) {\n  var result = [];\n  for (var i = 0;i &lt; arguments.length;i++) {\n    var element = arguments[i];\n    if (goog.isArray(element)) {\n      result.push.apply(result, goog.array.flatten.apply(null, element));\n    } else {\n      result.push(element);\n    }\n  }\n  return result;\n};\ngoog.array.rotate = function(array, n) {\n  goog.asserts.assert(array.length != null);\n  if (array.length) {\n    n %= array.length;\n    if (n &gt; 0) {\n      goog.array.ARRAY_PROTOTYPE_.unshift.apply(array, array.splice(-n, n));\n    } else {\n      if (n &lt; 0) {\n        goog.array.ARRAY_PROTOTYPE_.push.apply(array, array.splice(0, -n));\n      }\n    }\n  }\n  return array;\n};\ngoog.array.moveItem = function(arr, fromIndex, toIndex) {\n  goog.asserts.assert(fromIndex &gt;= 0 &amp;&amp; fromIndex &lt; arr.length);\n  goog.asserts.assert(toIndex &gt;= 0 &amp;&amp; toIndex &lt; arr.length);\n  var removedItems = goog.array.ARRAY_PROTOTYPE_.splice.call(arr, fromIndex, 1);\n  goog.array.ARRAY_PROTOTYPE_.splice.call(arr, toIndex, 0, removedItems[0]);\n};\ngoog.array.zip = function(var_args) {\n  if (!arguments.length) {\n    return[];\n  }\n  var result = [];\n  for (var i = 0;true;i++) {\n    var value = [];\n    for (var j = 0;j &lt; arguments.length;j++) {\n      var arr = arguments[j];\n      if (i &gt;= arr.length) {\n        return result;\n      }\n      value.push(arr[i]);\n    }\n    result.push(value);\n  }\n};\ngoog.array.shuffle = function(arr, opt_randFn) {\n  var randFn = opt_randFn || Math.random;\n  for (var i = arr.length - 1;i &gt; 0;i--) {\n    var j = Math.floor(randFn() * (i + 1));\n    var tmp = arr[i];\n    arr[i] = arr[j];\n    arr[j] = tmp;\n  }\n};\ngoog.provide(&quot;cljs.core&quot;);\ngoog.require(&quot;goog.string.StringBuffer&quot;);\ngoog.require(&quot;goog.array&quot;);\ngoog.require(&quot;goog.array&quot;);\ngoog.require(&quot;goog.object&quot;);\ngoog.require(&quot;goog.object&quot;);\ngoog.require(&quot;goog.string&quot;);\ngoog.require(&quot;goog.string&quot;);\ncljs.core._STAR_clojurescript_version_STAR_ = &quot;0.0-2322&quot;;\ncljs.core._STAR_unchecked_if_STAR_ = false;\ncljs.core._STAR_print_fn_STAR_ = function _STAR_print_fn_STAR_(_) {\n  throw new Error(&quot;No *print-fn* fn set for evaluation environment&quot;);\n};\ncljs.core.set_print_fn_BANG_ = function set_print_fn_BANG_(f) {\n  return cljs.core._STAR_print_fn_STAR_ = f;\n};\ncljs.core._STAR_flush_on_newline_STAR_ = true;\ncljs.core._STAR_print_newline_STAR_ = true;\ncljs.core._STAR_print_readably_STAR_ = true;\ncljs.core._STAR_print_meta_STAR_ = false;\ncljs.core._STAR_print_dup_STAR_ = false;\ncljs.core._STAR_print_length_STAR_ = null;\ncljs.core._STAR_print_level_STAR_ = null;\ncljs.core.pr_opts = function pr_opts() {\n  return new cljs.core.PersistentArrayMap(null, 5, [new cljs.core.Keyword(null, &quot;flush-on-newline&quot;, &quot;flush-on-newline&quot;, -151457939), cljs.core._STAR_flush_on_newline_STAR_, new cljs.core.Keyword(null, &quot;readably&quot;, &quot;readably&quot;, 1129599760), cljs.core._STAR_print_readably_STAR_, new cljs.core.Keyword(null, &quot;meta&quot;, &quot;meta&quot;, 1499536964), cljs.core._STAR_print_meta_STAR_, new cljs.core.Keyword(null, &quot;dup&quot;, &quot;dup&quot;, 556298533), cljs.core._STAR_print_dup_STAR_, new cljs.core.Keyword(null, &quot;print-length&quot;, &quot;print-length&quot;, \n  1931866356), cljs.core._STAR_print_length_STAR_], null);\n};\ncljs.core.enable_console_print_BANG_ = function enable_console_print_BANG_() {\n  cljs.core._STAR_print_newline_STAR_ = false;\n  return cljs.core._STAR_print_fn_STAR_ = function() {\n    var G__22053__delegate = function(args) {\n      return console.log.apply(console, cljs.core.into_array.call(null, args));\n    };\n    var G__22053 = function(var_args) {\n      var args = null;\n      if (arguments.length &gt; 0) {\n        args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n      }\n      return G__22053__delegate.call(this, args);\n    };\n    G__22053.cljs$lang$maxFixedArity = 0;\n    G__22053.cljs$lang$applyTo = function(arglist__22054) {\n      var args = cljs.core.seq(arglist__22054);\n      return G__22053__delegate(args);\n    };\n    G__22053.cljs$core$IFn$_invoke$arity$variadic = G__22053__delegate;\n    return G__22053;\n  }();\n};\ncljs.core.truth_ = function truth_(x) {\n  return x != null &amp;&amp; x !== false;\n};\ncljs.core.not_native = null;\ncljs.core.identical_QMARK_ = function identical_QMARK_(x, y) {\n  return x === y;\n};\ncljs.core.nil_QMARK_ = function nil_QMARK_(x) {\n  return x == null;\n};\ncljs.core.array_QMARK_ = function array_QMARK_(x) {\n  return x instanceof Array;\n};\ncljs.core.number_QMARK_ = function number_QMARK_(n) {\n  return typeof n === &quot;number&quot;;\n};\ncljs.core.not = function not(x) {\n  if (cljs.core.truth_(x)) {\n    return false;\n  } else {\n    return true;\n  }\n};\ncljs.core.some_QMARK_ = function some_QMARK_(x) {\n  return!(x == null);\n};\ncljs.core.object_QMARK_ = function object_QMARK_(x) {\n  if (!(x == null)) {\n    return x.constructor === Object;\n  } else {\n    return false;\n  }\n};\ncljs.core.string_QMARK_ = function string_QMARK_(x) {\n  return goog.isString(x);\n};\ncljs.core.native_satisfies_QMARK_ = function native_satisfies_QMARK_(p, x) {\n  var x__$1 = x == null ? null : x;\n  if (p[goog.typeOf(x__$1)]) {\n    return true;\n  } else {\n    if (p[&quot;_&quot;]) {\n      return true;\n    } else {\n      return false;\n    }\n  }\n};\ncljs.core.is_proto_ = function is_proto_(x) {\n  return x.constructor.prototype === x;\n};\ncljs.core._STAR_main_cli_fn_STAR_ = null;\ncljs.core.type = function type(x) {\n  if (x == null) {\n    return null;\n  } else {\n    return x.constructor;\n  }\n};\ncljs.core.missing_protocol = function missing_protocol(proto, obj) {\n  var ty = cljs.core.type.call(null, obj);\n  var ty__$1 = cljs.core.truth_(function() {\n    var and__17133__auto__ = ty;\n    if (cljs.core.truth_(and__17133__auto__)) {\n      return ty.cljs$lang$type;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) ? ty.cljs$lang$ctorStr : goog.typeOf(obj);\n  return new Error([&quot;No protocol method &quot;, proto, &quot; defined for type &quot;, ty__$1, &quot;: &quot;, obj].join(&quot;&quot;));\n};\ncljs.core.type__GT_str = function type__GT_str(ty) {\n  var temp__4124__auto__ = ty.cljs$lang$ctorStr;\n  if (cljs.core.truth_(temp__4124__auto__)) {\n    var s = temp__4124__auto__;\n    return s;\n  } else {\n    return &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(ty);\n  }\n};\ncljs.core.make_array = function() {\n  var make_array = null;\n  var make_array__1 = function(size) {\n    return new Array(size);\n  };\n  var make_array__2 = function(type, size) {\n    return make_array.call(null, size);\n  };\n  make_array = function(type, size) {\n    switch(arguments.length) {\n      case 1:\n        return make_array__1.call(this, type);\n      case 2:\n        return make_array__2.call(this, type, size);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  make_array.cljs$core$IFn$_invoke$arity$1 = make_array__1;\n  make_array.cljs$core$IFn$_invoke$arity$2 = make_array__2;\n  return make_array;\n}();\ncljs.core.aclone = function aclone(arr) {\n  var len = arr.length;\n  var new_arr = new Array(len);\n  var n__18001__auto___22055 = len;\n  var i_22056 = 0;\n  while (true) {\n    if (i_22056 &lt; n__18001__auto___22055) {\n      new_arr[i_22056] = arr[i_22056];\n      var G__22057 = i_22056 + 1;\n      i_22056 = G__22057;\n      continue;\n    } else {\n    }\n    break;\n  }\n  return new_arr;\n};\ncljs.core.array = function array(var_args) {\n  return Array.prototype.slice.call(arguments);\n};\ncljs.core.aget = function() {\n  var aget = null;\n  var aget__2 = function(array, i) {\n    return array[i];\n  };\n  var aget__3 = function() {\n    var G__22058__delegate = function(array, i, idxs) {\n      return cljs.core.apply.call(null, aget, aget.call(null, array, i), idxs);\n    };\n    var G__22058 = function(array, i, var_args) {\n      var idxs = null;\n      if (arguments.length &gt; 2) {\n        idxs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22058__delegate.call(this, array, i, idxs);\n    };\n    G__22058.cljs$lang$maxFixedArity = 2;\n    G__22058.cljs$lang$applyTo = function(arglist__22059) {\n      var array = cljs.core.first(arglist__22059);\n      arglist__22059 = cljs.core.next(arglist__22059);\n      var i = cljs.core.first(arglist__22059);\n      var idxs = cljs.core.rest(arglist__22059);\n      return G__22058__delegate(array, i, idxs);\n    };\n    G__22058.cljs$core$IFn$_invoke$arity$variadic = G__22058__delegate;\n    return G__22058;\n  }();\n  aget = function(array, i, var_args) {\n    var idxs = var_args;\n    switch(arguments.length) {\n      case 2:\n        return aget__2.call(this, array, i);\n      default:\n        return aget__3.cljs$core$IFn$_invoke$arity$variadic(array, i, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  aget.cljs$lang$maxFixedArity = 2;\n  aget.cljs$lang$applyTo = aget__3.cljs$lang$applyTo;\n  aget.cljs$core$IFn$_invoke$arity$2 = aget__2;\n  aget.cljs$core$IFn$_invoke$arity$variadic = aget__3.cljs$core$IFn$_invoke$arity$variadic;\n  return aget;\n}();\ncljs.core.aset = function() {\n  var aset = null;\n  var aset__3 = function(array, i, val) {\n    return array[i] = val;\n  };\n  var aset__4 = function() {\n    var G__22060__delegate = function(array, idx, idx2, idxv) {\n      return cljs.core.apply.call(null, aset, array[idx], idx2, idxv);\n    };\n    var G__22060 = function(array, idx, idx2, var_args) {\n      var idxv = null;\n      if (arguments.length &gt; 3) {\n        idxv = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n      }\n      return G__22060__delegate.call(this, array, idx, idx2, idxv);\n    };\n    G__22060.cljs$lang$maxFixedArity = 3;\n    G__22060.cljs$lang$applyTo = function(arglist__22061) {\n      var array = cljs.core.first(arglist__22061);\n      arglist__22061 = cljs.core.next(arglist__22061);\n      var idx = cljs.core.first(arglist__22061);\n      arglist__22061 = cljs.core.next(arglist__22061);\n      var idx2 = cljs.core.first(arglist__22061);\n      var idxv = cljs.core.rest(arglist__22061);\n      return G__22060__delegate(array, idx, idx2, idxv);\n    };\n    G__22060.cljs$core$IFn$_invoke$arity$variadic = G__22060__delegate;\n    return G__22060;\n  }();\n  aset = function(array, idx, idx2, var_args) {\n    var idxv = var_args;\n    switch(arguments.length) {\n      case 3:\n        return aset__3.call(this, array, idx, idx2);\n      default:\n        return aset__4.cljs$core$IFn$_invoke$arity$variadic(array, idx, idx2, cljs.core.array_seq(arguments, 3));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  aset.cljs$lang$maxFixedArity = 3;\n  aset.cljs$lang$applyTo = aset__4.cljs$lang$applyTo;\n  aset.cljs$core$IFn$_invoke$arity$3 = aset__3;\n  aset.cljs$core$IFn$_invoke$arity$variadic = aset__4.cljs$core$IFn$_invoke$arity$variadic;\n  return aset;\n}();\ncljs.core.alength = function alength(array) {\n  return array.length;\n};\ncljs.core.into_array = function() {\n  var into_array = null;\n  var into_array__1 = function(aseq) {\n    return into_array.call(null, null, aseq);\n  };\n  var into_array__2 = function(type, aseq) {\n    return cljs.core.reduce.call(null, function(a, x) {\n      a.push(x);\n      return a;\n    }, [], aseq);\n  };\n  into_array = function(type, aseq) {\n    switch(arguments.length) {\n      case 1:\n        return into_array__1.call(this, type);\n      case 2:\n        return into_array__2.call(this, type, aseq);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  into_array.cljs$core$IFn$_invoke$arity$1 = into_array__1;\n  into_array.cljs$core$IFn$_invoke$arity$2 = into_array__2;\n  return into_array;\n}();\ncljs.core.Fn = function() {\n  var obj22063 = {};\n  return obj22063;\n}();\ncljs.core.IFn = function() {\n  var obj22065 = {};\n  return obj22065;\n}();\ncljs.core._invoke = function() {\n  var _invoke = null;\n  var _invoke__1 = function(this$) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$1;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$1(this$);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$);\n    }\n  };\n  var _invoke__2 = function(this$, a) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$2;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$2(this$, a);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a);\n    }\n  };\n  var _invoke__3 = function(this$, a, b) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$3;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$3(this$, a, b);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b);\n    }\n  };\n  var _invoke__4 = function(this$, a, b, c) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$4;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$4(this$, a, b, c);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c);\n    }\n  };\n  var _invoke__5 = function(this$, a, b, c, d) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$5;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$5(this$, a, b, c, d);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d);\n    }\n  };\n  var _invoke__6 = function(this$, a, b, c, d, e) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$6;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$6(this$, a, b, c, d, e);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e);\n    }\n  };\n  var _invoke__7 = function(this$, a, b, c, d, e, f) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$7;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$7(this$, a, b, c, d, e, f);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f);\n    }\n  };\n  var _invoke__8 = function(this$, a, b, c, d, e, f, g) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$8;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$8(this$, a, b, c, d, e, f, g);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g);\n    }\n  };\n  var _invoke__9 = function(this$, a, b, c, d, e, f, g, h) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$9;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$9(this$, a, b, c, d, e, f, g, h);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h);\n    }\n  };\n  var _invoke__10 = function(this$, a, b, c, d, e, f, g, h, i) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$10;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$10(this$, a, b, c, d, e, f, g, h, i);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i);\n    }\n  };\n  var _invoke__11 = function(this$, a, b, c, d, e, f, g, h, i, j) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$11;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$11(this$, a, b, c, d, e, f, g, h, i, j);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j);\n    }\n  };\n  var _invoke__12 = function(this$, a, b, c, d, e, f, g, h, i, j, k) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$12;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$12(this$, a, b, c, d, e, f, g, h, i, j, k);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k);\n    }\n  };\n  var _invoke__13 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$13;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$13(this$, a, b, c, d, e, f, g, h, i, j, k, l);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l);\n    }\n  };\n  var _invoke__14 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$14;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$14(this$, a, b, c, d, e, f, g, h, i, j, k, l, m);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m);\n    }\n  };\n  var _invoke__15 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$15;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$15(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n    }\n  };\n  var _invoke__16 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$16;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$16(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);\n    }\n  };\n  var _invoke__17 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$17;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$17(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);\n    }\n  };\n  var _invoke__18 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$18;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$18(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);\n    }\n  };\n  var _invoke__19 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$19;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$19(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s);\n    }\n  };\n  var _invoke__20 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s, t) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$20;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$20(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s, t);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s, t);\n    }\n  };\n  var _invoke__21 = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s, t, rest) {\n    if (function() {\n      var and__17133__auto__ = this$;\n      if (and__17133__auto__) {\n        return this$.cljs$core$IFn$_invoke$arity$21;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return this$.cljs$core$IFn$_invoke$arity$21(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s, t, rest);\n    } else {\n      var x__17772__auto__ = this$ == null ? null : this$;\n      return function() {\n        var or__17145__auto__ = cljs.core._invoke[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._invoke[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IFn.-invoke&quot;, this$);\n          }\n        }\n      }().call(null, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s, t, rest);\n    }\n  };\n  _invoke = function(this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s, t, rest) {\n    switch(arguments.length) {\n      case 1:\n        return _invoke__1.call(this, this$);\n      case 2:\n        return _invoke__2.call(this, this$, a);\n      case 3:\n        return _invoke__3.call(this, this$, a, b);\n      case 4:\n        return _invoke__4.call(this, this$, a, b, c);\n      case 5:\n        return _invoke__5.call(this, this$, a, b, c, d);\n      case 6:\n        return _invoke__6.call(this, this$, a, b, c, d, e);\n      case 7:\n        return _invoke__7.call(this, this$, a, b, c, d, e, f);\n      case 8:\n        return _invoke__8.call(this, this$, a, b, c, d, e, f, g);\n      case 9:\n        return _invoke__9.call(this, this$, a, b, c, d, e, f, g, h);\n      case 10:\n        return _invoke__10.call(this, this$, a, b, c, d, e, f, g, h, i);\n      case 11:\n        return _invoke__11.call(this, this$, a, b, c, d, e, f, g, h, i, j);\n      case 12:\n        return _invoke__12.call(this, this$, a, b, c, d, e, f, g, h, i, j, k);\n      case 13:\n        return _invoke__13.call(this, this$, a, b, c, d, e, f, g, h, i, j, k, l);\n      case 14:\n        return _invoke__14.call(this, this$, a, b, c, d, e, f, g, h, i, j, k, l, m);\n      case 15:\n        return _invoke__15.call(this, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n      case 16:\n        return _invoke__16.call(this, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);\n      case 17:\n        return _invoke__17.call(this, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);\n      case 18:\n        return _invoke__18.call(this, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);\n      case 19:\n        return _invoke__19.call(this, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s);\n      case 20:\n        return _invoke__20.call(this, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s, t);\n      case 21:\n        return _invoke__21.call(this, this$, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, s, t, rest);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _invoke.cljs$core$IFn$_invoke$arity$1 = _invoke__1;\n  _invoke.cljs$core$IFn$_invoke$arity$2 = _invoke__2;\n  _invoke.cljs$core$IFn$_invoke$arity$3 = _invoke__3;\n  _invoke.cljs$core$IFn$_invoke$arity$4 = _invoke__4;\n  _invoke.cljs$core$IFn$_invoke$arity$5 = _invoke__5;\n  _invoke.cljs$core$IFn$_invoke$arity$6 = _invoke__6;\n  _invoke.cljs$core$IFn$_invoke$arity$7 = _invoke__7;\n  _invoke.cljs$core$IFn$_invoke$arity$8 = _invoke__8;\n  _invoke.cljs$core$IFn$_invoke$arity$9 = _invoke__9;\n  _invoke.cljs$core$IFn$_invoke$arity$10 = _invoke__10;\n  _invoke.cljs$core$IFn$_invoke$arity$11 = _invoke__11;\n  _invoke.cljs$core$IFn$_invoke$arity$12 = _invoke__12;\n  _invoke.cljs$core$IFn$_invoke$arity$13 = _invoke__13;\n  _invoke.cljs$core$IFn$_invoke$arity$14 = _invoke__14;\n  _invoke.cljs$core$IFn$_invoke$arity$15 = _invoke__15;\n  _invoke.cljs$core$IFn$_invoke$arity$16 = _invoke__16;\n  _invoke.cljs$core$IFn$_invoke$arity$17 = _invoke__17;\n  _invoke.cljs$core$IFn$_invoke$arity$18 = _invoke__18;\n  _invoke.cljs$core$IFn$_invoke$arity$19 = _invoke__19;\n  _invoke.cljs$core$IFn$_invoke$arity$20 = _invoke__20;\n  _invoke.cljs$core$IFn$_invoke$arity$21 = _invoke__21;\n  return _invoke;\n}();\ncljs.core.ICloneable = function() {\n  var obj22067 = {};\n  return obj22067;\n}();\ncljs.core._clone = function _clone(value) {\n  if (function() {\n    var and__17133__auto__ = value;\n    if (and__17133__auto__) {\n      return value.cljs$core$ICloneable$_clone$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return value.cljs$core$ICloneable$_clone$arity$1(value);\n  } else {\n    var x__17772__auto__ = value == null ? null : value;\n    return function() {\n      var or__17145__auto__ = cljs.core._clone[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._clone[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ICloneable.-clone&quot;, value);\n        }\n      }\n    }().call(null, value);\n  }\n};\ncljs.core.ICounted = function() {\n  var obj22069 = {};\n  return obj22069;\n}();\ncljs.core._count = function _count(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$ICounted$_count$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$ICounted$_count$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._count[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._count[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ICounted.-count&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.IEmptyableCollection = function() {\n  var obj22071 = {};\n  return obj22071;\n}();\ncljs.core._empty = function _empty(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IEmptyableCollection$_empty$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IEmptyableCollection$_empty$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._empty[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._empty[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IEmptyableCollection.-empty&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.ICollection = function() {\n  var obj22073 = {};\n  return obj22073;\n}();\ncljs.core._conj = function _conj(coll, o) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$ICollection$_conj$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$ICollection$_conj$arity$2(coll, o);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._conj[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._conj[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ICollection.-conj&quot;, coll);\n        }\n      }\n    }().call(null, coll, o);\n  }\n};\ncljs.core.IIndexed = function() {\n  var obj22075 = {};\n  return obj22075;\n}();\ncljs.core._nth = function() {\n  var _nth = null;\n  var _nth__2 = function(coll, n) {\n    if (function() {\n      var and__17133__auto__ = coll;\n      if (and__17133__auto__) {\n        return coll.cljs$core$IIndexed$_nth$arity$2;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return coll.cljs$core$IIndexed$_nth$arity$2(coll, n);\n    } else {\n      var x__17772__auto__ = coll == null ? null : coll;\n      return function() {\n        var or__17145__auto__ = cljs.core._nth[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._nth[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IIndexed.-nth&quot;, coll);\n          }\n        }\n      }().call(null, coll, n);\n    }\n  };\n  var _nth__3 = function(coll, n, not_found) {\n    if (function() {\n      var and__17133__auto__ = coll;\n      if (and__17133__auto__) {\n        return coll.cljs$core$IIndexed$_nth$arity$3;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return coll.cljs$core$IIndexed$_nth$arity$3(coll, n, not_found);\n    } else {\n      var x__17772__auto__ = coll == null ? null : coll;\n      return function() {\n        var or__17145__auto__ = cljs.core._nth[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._nth[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IIndexed.-nth&quot;, coll);\n          }\n        }\n      }().call(null, coll, n, not_found);\n    }\n  };\n  _nth = function(coll, n, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return _nth__2.call(this, coll, n);\n      case 3:\n        return _nth__3.call(this, coll, n, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _nth.cljs$core$IFn$_invoke$arity$2 = _nth__2;\n  _nth.cljs$core$IFn$_invoke$arity$3 = _nth__3;\n  return _nth;\n}();\ncljs.core.ASeq = function() {\n  var obj22077 = {};\n  return obj22077;\n}();\ncljs.core.ISeq = function() {\n  var obj22079 = {};\n  return obj22079;\n}();\ncljs.core._first = function _first(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$ISeq$_first$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$ISeq$_first$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._first[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._first[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ISeq.-first&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core._rest = function _rest(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$ISeq$_rest$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$ISeq$_rest$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._rest[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._rest[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ISeq.-rest&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.INext = function() {\n  var obj22081 = {};\n  return obj22081;\n}();\ncljs.core._next = function _next(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$INext$_next$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$INext$_next$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._next[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._next[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;INext.-next&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.ILookup = function() {\n  var obj22083 = {};\n  return obj22083;\n}();\ncljs.core._lookup = function() {\n  var _lookup = null;\n  var _lookup__2 = function(o, k) {\n    if (function() {\n      var and__17133__auto__ = o;\n      if (and__17133__auto__) {\n        return o.cljs$core$ILookup$_lookup$arity$2;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return o.cljs$core$ILookup$_lookup$arity$2(o, k);\n    } else {\n      var x__17772__auto__ = o == null ? null : o;\n      return function() {\n        var or__17145__auto__ = cljs.core._lookup[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._lookup[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;ILookup.-lookup&quot;, o);\n          }\n        }\n      }().call(null, o, k);\n    }\n  };\n  var _lookup__3 = function(o, k, not_found) {\n    if (function() {\n      var and__17133__auto__ = o;\n      if (and__17133__auto__) {\n        return o.cljs$core$ILookup$_lookup$arity$3;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return o.cljs$core$ILookup$_lookup$arity$3(o, k, not_found);\n    } else {\n      var x__17772__auto__ = o == null ? null : o;\n      return function() {\n        var or__17145__auto__ = cljs.core._lookup[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._lookup[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;ILookup.-lookup&quot;, o);\n          }\n        }\n      }().call(null, o, k, not_found);\n    }\n  };\n  _lookup = function(o, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return _lookup__2.call(this, o, k);\n      case 3:\n        return _lookup__3.call(this, o, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _lookup.cljs$core$IFn$_invoke$arity$2 = _lookup__2;\n  _lookup.cljs$core$IFn$_invoke$arity$3 = _lookup__3;\n  return _lookup;\n}();\ncljs.core.IAssociative = function() {\n  var obj22085 = {};\n  return obj22085;\n}();\ncljs.core._contains_key_QMARK_ = function _contains_key_QMARK_(coll, k) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IAssociative$_contains_key_QMARK_$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IAssociative$_contains_key_QMARK_$arity$2(coll, k);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._contains_key_QMARK_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._contains_key_QMARK_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IAssociative.-contains-key?&quot;, coll);\n        }\n      }\n    }().call(null, coll, k);\n  }\n};\ncljs.core._assoc = function _assoc(coll, k, v) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IAssociative$_assoc$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IAssociative$_assoc$arity$3(coll, k, v);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._assoc[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._assoc[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IAssociative.-assoc&quot;, coll);\n        }\n      }\n    }().call(null, coll, k, v);\n  }\n};\ncljs.core.IMap = function() {\n  var obj22087 = {};\n  return obj22087;\n}();\ncljs.core._dissoc = function _dissoc(coll, k) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IMap$_dissoc$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IMap$_dissoc$arity$2(coll, k);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._dissoc[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._dissoc[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMap.-dissoc&quot;, coll);\n        }\n      }\n    }().call(null, coll, k);\n  }\n};\ncljs.core.IMapEntry = function() {\n  var obj22089 = {};\n  return obj22089;\n}();\ncljs.core._key = function _key(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IMapEntry$_key$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IMapEntry$_key$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._key[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._key[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMapEntry.-key&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core._val = function _val(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IMapEntry$_val$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IMapEntry$_val$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._val[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._val[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMapEntry.-val&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.ISet = function() {\n  var obj22091 = {};\n  return obj22091;\n}();\ncljs.core._disjoin = function _disjoin(coll, v) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$ISet$_disjoin$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$ISet$_disjoin$arity$2(coll, v);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._disjoin[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._disjoin[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ISet.-disjoin&quot;, coll);\n        }\n      }\n    }().call(null, coll, v);\n  }\n};\ncljs.core.IStack = function() {\n  var obj22093 = {};\n  return obj22093;\n}();\ncljs.core._peek = function _peek(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IStack$_peek$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IStack$_peek$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._peek[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._peek[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IStack.-peek&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core._pop = function _pop(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IStack$_pop$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IStack$_pop$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._pop[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._pop[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IStack.-pop&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.IVector = function() {\n  var obj22095 = {};\n  return obj22095;\n}();\ncljs.core._assoc_n = function _assoc_n(coll, n, val) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IVector$_assoc_n$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IVector$_assoc_n$arity$3(coll, n, val);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._assoc_n[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._assoc_n[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IVector.-assoc-n&quot;, coll);\n        }\n      }\n    }().call(null, coll, n, val);\n  }\n};\ncljs.core.IDeref = function() {\n  var obj22097 = {};\n  return obj22097;\n}();\ncljs.core._deref = function _deref(o) {\n  if (function() {\n    var and__17133__auto__ = o;\n    if (and__17133__auto__) {\n      return o.cljs$core$IDeref$_deref$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return o.cljs$core$IDeref$_deref$arity$1(o);\n  } else {\n    var x__17772__auto__ = o == null ? null : o;\n    return function() {\n      var or__17145__auto__ = cljs.core._deref[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._deref[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IDeref.-deref&quot;, o);\n        }\n      }\n    }().call(null, o);\n  }\n};\ncljs.core.IDerefWithTimeout = function() {\n  var obj22099 = {};\n  return obj22099;\n}();\ncljs.core._deref_with_timeout = function _deref_with_timeout(o, msec, timeout_val) {\n  if (function() {\n    var and__17133__auto__ = o;\n    if (and__17133__auto__) {\n      return o.cljs$core$IDerefWithTimeout$_deref_with_timeout$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return o.cljs$core$IDerefWithTimeout$_deref_with_timeout$arity$3(o, msec, timeout_val);\n  } else {\n    var x__17772__auto__ = o == null ? null : o;\n    return function() {\n      var or__17145__auto__ = cljs.core._deref_with_timeout[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._deref_with_timeout[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IDerefWithTimeout.-deref-with-timeout&quot;, o);\n        }\n      }\n    }().call(null, o, msec, timeout_val);\n  }\n};\ncljs.core.IMeta = function() {\n  var obj22101 = {};\n  return obj22101;\n}();\ncljs.core._meta = function _meta(o) {\n  if (function() {\n    var and__17133__auto__ = o;\n    if (and__17133__auto__) {\n      return o.cljs$core$IMeta$_meta$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return o.cljs$core$IMeta$_meta$arity$1(o);\n  } else {\n    var x__17772__auto__ = o == null ? null : o;\n    return function() {\n      var or__17145__auto__ = cljs.core._meta[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._meta[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMeta.-meta&quot;, o);\n        }\n      }\n    }().call(null, o);\n  }\n};\ncljs.core.IWithMeta = function() {\n  var obj22103 = {};\n  return obj22103;\n}();\ncljs.core._with_meta = function _with_meta(o, meta) {\n  if (function() {\n    var and__17133__auto__ = o;\n    if (and__17133__auto__) {\n      return o.cljs$core$IWithMeta$_with_meta$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return o.cljs$core$IWithMeta$_with_meta$arity$2(o, meta);\n  } else {\n    var x__17772__auto__ = o == null ? null : o;\n    return function() {\n      var or__17145__auto__ = cljs.core._with_meta[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._with_meta[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IWithMeta.-with-meta&quot;, o);\n        }\n      }\n    }().call(null, o, meta);\n  }\n};\ncljs.core.IReduce = function() {\n  var obj22105 = {};\n  return obj22105;\n}();\ncljs.core._reduce = function() {\n  var _reduce = null;\n  var _reduce__2 = function(coll, f) {\n    if (function() {\n      var and__17133__auto__ = coll;\n      if (and__17133__auto__) {\n        return coll.cljs$core$IReduce$_reduce$arity$2;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return coll.cljs$core$IReduce$_reduce$arity$2(coll, f);\n    } else {\n      var x__17772__auto__ = coll == null ? null : coll;\n      return function() {\n        var or__17145__auto__ = cljs.core._reduce[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._reduce[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IReduce.-reduce&quot;, coll);\n          }\n        }\n      }().call(null, coll, f);\n    }\n  };\n  var _reduce__3 = function(coll, f, start) {\n    if (function() {\n      var and__17133__auto__ = coll;\n      if (and__17133__auto__) {\n        return coll.cljs$core$IReduce$_reduce$arity$3;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return coll.cljs$core$IReduce$_reduce$arity$3(coll, f, start);\n    } else {\n      var x__17772__auto__ = coll == null ? null : coll;\n      return function() {\n        var or__17145__auto__ = cljs.core._reduce[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._reduce[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;IReduce.-reduce&quot;, coll);\n          }\n        }\n      }().call(null, coll, f, start);\n    }\n  };\n  _reduce = function(coll, f, start) {\n    switch(arguments.length) {\n      case 2:\n        return _reduce__2.call(this, coll, f);\n      case 3:\n        return _reduce__3.call(this, coll, f, start);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _reduce.cljs$core$IFn$_invoke$arity$2 = _reduce__2;\n  _reduce.cljs$core$IFn$_invoke$arity$3 = _reduce__3;\n  return _reduce;\n}();\ncljs.core.IKVReduce = function() {\n  var obj22107 = {};\n  return obj22107;\n}();\ncljs.core._kv_reduce = function _kv_reduce(coll, f, init) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IKVReduce$_kv_reduce$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IKVReduce$_kv_reduce$arity$3(coll, f, init);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._kv_reduce[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._kv_reduce[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IKVReduce.-kv-reduce&quot;, coll);\n        }\n      }\n    }().call(null, coll, f, init);\n  }\n};\ncljs.core.IEquiv = function() {\n  var obj22109 = {};\n  return obj22109;\n}();\ncljs.core._equiv = function _equiv(o, other) {\n  if (function() {\n    var and__17133__auto__ = o;\n    if (and__17133__auto__) {\n      return o.cljs$core$IEquiv$_equiv$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return o.cljs$core$IEquiv$_equiv$arity$2(o, other);\n  } else {\n    var x__17772__auto__ = o == null ? null : o;\n    return function() {\n      var or__17145__auto__ = cljs.core._equiv[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._equiv[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IEquiv.-equiv&quot;, o);\n        }\n      }\n    }().call(null, o, other);\n  }\n};\ncljs.core.IHash = function() {\n  var obj22111 = {};\n  return obj22111;\n}();\ncljs.core._hash = function _hash(o) {\n  if (function() {\n    var and__17133__auto__ = o;\n    if (and__17133__auto__) {\n      return o.cljs$core$IHash$_hash$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return o.cljs$core$IHash$_hash$arity$1(o);\n  } else {\n    var x__17772__auto__ = o == null ? null : o;\n    return function() {\n      var or__17145__auto__ = cljs.core._hash[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._hash[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IHash.-hash&quot;, o);\n        }\n      }\n    }().call(null, o);\n  }\n};\ncljs.core.ISeqable = function() {\n  var obj22113 = {};\n  return obj22113;\n}();\ncljs.core._seq = function _seq(o) {\n  if (function() {\n    var and__17133__auto__ = o;\n    if (and__17133__auto__) {\n      return o.cljs$core$ISeqable$_seq$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return o.cljs$core$ISeqable$_seq$arity$1(o);\n  } else {\n    var x__17772__auto__ = o == null ? null : o;\n    return function() {\n      var or__17145__auto__ = cljs.core._seq[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._seq[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ISeqable.-seq&quot;, o);\n        }\n      }\n    }().call(null, o);\n  }\n};\ncljs.core.ISequential = function() {\n  var obj22115 = {};\n  return obj22115;\n}();\ncljs.core.IList = function() {\n  var obj22117 = {};\n  return obj22117;\n}();\ncljs.core.IRecord = function() {\n  var obj22119 = {};\n  return obj22119;\n}();\ncljs.core.IReversible = function() {\n  var obj22121 = {};\n  return obj22121;\n}();\ncljs.core._rseq = function _rseq(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IReversible$_rseq$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IReversible$_rseq$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._rseq[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._rseq[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IReversible.-rseq&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.ISorted = function() {\n  var obj22123 = {};\n  return obj22123;\n}();\ncljs.core._sorted_seq = function _sorted_seq(coll, ascending_QMARK_) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$ISorted$_sorted_seq$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$ISorted$_sorted_seq$arity$2(coll, ascending_QMARK_);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._sorted_seq[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._sorted_seq[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ISorted.-sorted-seq&quot;, coll);\n        }\n      }\n    }().call(null, coll, ascending_QMARK_);\n  }\n};\ncljs.core._sorted_seq_from = function _sorted_seq_from(coll, k, ascending_QMARK_) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$ISorted$_sorted_seq_from$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$ISorted$_sorted_seq_from$arity$3(coll, k, ascending_QMARK_);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._sorted_seq_from[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._sorted_seq_from[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ISorted.-sorted-seq-from&quot;, coll);\n        }\n      }\n    }().call(null, coll, k, ascending_QMARK_);\n  }\n};\ncljs.core._entry_key = function _entry_key(coll, entry) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$ISorted$_entry_key$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$ISorted$_entry_key$arity$2(coll, entry);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._entry_key[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._entry_key[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ISorted.-entry-key&quot;, coll);\n        }\n      }\n    }().call(null, coll, entry);\n  }\n};\ncljs.core._comparator = function _comparator(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$ISorted$_comparator$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$ISorted$_comparator$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._comparator[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._comparator[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ISorted.-comparator&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.IWriter = function() {\n  var obj22125 = {};\n  return obj22125;\n}();\ncljs.core._write = function _write(writer, s) {\n  if (function() {\n    var and__17133__auto__ = writer;\n    if (and__17133__auto__) {\n      return writer.cljs$core$IWriter$_write$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return writer.cljs$core$IWriter$_write$arity$2(writer, s);\n  } else {\n    var x__17772__auto__ = writer == null ? null : writer;\n    return function() {\n      var or__17145__auto__ = cljs.core._write[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._write[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IWriter.-write&quot;, writer);\n        }\n      }\n    }().call(null, writer, s);\n  }\n};\ncljs.core._flush = function _flush(writer) {\n  if (function() {\n    var and__17133__auto__ = writer;\n    if (and__17133__auto__) {\n      return writer.cljs$core$IWriter$_flush$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return writer.cljs$core$IWriter$_flush$arity$1(writer);\n  } else {\n    var x__17772__auto__ = writer == null ? null : writer;\n    return function() {\n      var or__17145__auto__ = cljs.core._flush[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._flush[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IWriter.-flush&quot;, writer);\n        }\n      }\n    }().call(null, writer);\n  }\n};\ncljs.core.IPrintWithWriter = function() {\n  var obj22127 = {};\n  return obj22127;\n}();\ncljs.core._pr_writer = function _pr_writer(o, writer, opts) {\n  if (function() {\n    var and__17133__auto__ = o;\n    if (and__17133__auto__) {\n      return o.cljs$core$IPrintWithWriter$_pr_writer$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return o.cljs$core$IPrintWithWriter$_pr_writer$arity$3(o, writer, opts);\n  } else {\n    var x__17772__auto__ = o == null ? null : o;\n    return function() {\n      var or__17145__auto__ = cljs.core._pr_writer[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._pr_writer[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IPrintWithWriter.-pr-writer&quot;, o);\n        }\n      }\n    }().call(null, o, writer, opts);\n  }\n};\ncljs.core.IPending = function() {\n  var obj22129 = {};\n  return obj22129;\n}();\ncljs.core._realized_QMARK_ = function _realized_QMARK_(d) {\n  if (function() {\n    var and__17133__auto__ = d;\n    if (and__17133__auto__) {\n      return d.cljs$core$IPending$_realized_QMARK_$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return d.cljs$core$IPending$_realized_QMARK_$arity$1(d);\n  } else {\n    var x__17772__auto__ = d == null ? null : d;\n    return function() {\n      var or__17145__auto__ = cljs.core._realized_QMARK_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._realized_QMARK_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IPending.-realized?&quot;, d);\n        }\n      }\n    }().call(null, d);\n  }\n};\ncljs.core.IWatchable = function() {\n  var obj22131 = {};\n  return obj22131;\n}();\ncljs.core._notify_watches = function _notify_watches(this$, oldval, newval) {\n  if (function() {\n    var and__17133__auto__ = this$;\n    if (and__17133__auto__) {\n      return this$.cljs$core$IWatchable$_notify_watches$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return this$.cljs$core$IWatchable$_notify_watches$arity$3(this$, oldval, newval);\n  } else {\n    var x__17772__auto__ = this$ == null ? null : this$;\n    return function() {\n      var or__17145__auto__ = cljs.core._notify_watches[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._notify_watches[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IWatchable.-notify-watches&quot;, this$);\n        }\n      }\n    }().call(null, this$, oldval, newval);\n  }\n};\ncljs.core._add_watch = function _add_watch(this$, key, f) {\n  if (function() {\n    var and__17133__auto__ = this$;\n    if (and__17133__auto__) {\n      return this$.cljs$core$IWatchable$_add_watch$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return this$.cljs$core$IWatchable$_add_watch$arity$3(this$, key, f);\n  } else {\n    var x__17772__auto__ = this$ == null ? null : this$;\n    return function() {\n      var or__17145__auto__ = cljs.core._add_watch[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._add_watch[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IWatchable.-add-watch&quot;, this$);\n        }\n      }\n    }().call(null, this$, key, f);\n  }\n};\ncljs.core._remove_watch = function _remove_watch(this$, key) {\n  if (function() {\n    var and__17133__auto__ = this$;\n    if (and__17133__auto__) {\n      return this$.cljs$core$IWatchable$_remove_watch$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return this$.cljs$core$IWatchable$_remove_watch$arity$2(this$, key);\n  } else {\n    var x__17772__auto__ = this$ == null ? null : this$;\n    return function() {\n      var or__17145__auto__ = cljs.core._remove_watch[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._remove_watch[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IWatchable.-remove-watch&quot;, this$);\n        }\n      }\n    }().call(null, this$, key);\n  }\n};\ncljs.core.IEditableCollection = function() {\n  var obj22133 = {};\n  return obj22133;\n}();\ncljs.core._as_transient = function _as_transient(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IEditableCollection$_as_transient$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IEditableCollection$_as_transient$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._as_transient[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._as_transient[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IEditableCollection.-as-transient&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.ITransientCollection = function() {\n  var obj22135 = {};\n  return obj22135;\n}();\ncljs.core._conj_BANG_ = function _conj_BANG_(tcoll, val) {\n  if (function() {\n    var and__17133__auto__ = tcoll;\n    if (and__17133__auto__) {\n      return tcoll.cljs$core$ITransientCollection$_conj_BANG_$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return tcoll.cljs$core$ITransientCollection$_conj_BANG_$arity$2(tcoll, val);\n  } else {\n    var x__17772__auto__ = tcoll == null ? null : tcoll;\n    return function() {\n      var or__17145__auto__ = cljs.core._conj_BANG_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._conj_BANG_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ITransientCollection.-conj!&quot;, tcoll);\n        }\n      }\n    }().call(null, tcoll, val);\n  }\n};\ncljs.core._persistent_BANG_ = function _persistent_BANG_(tcoll) {\n  if (function() {\n    var and__17133__auto__ = tcoll;\n    if (and__17133__auto__) {\n      return tcoll.cljs$core$ITransientCollection$_persistent_BANG_$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return tcoll.cljs$core$ITransientCollection$_persistent_BANG_$arity$1(tcoll);\n  } else {\n    var x__17772__auto__ = tcoll == null ? null : tcoll;\n    return function() {\n      var or__17145__auto__ = cljs.core._persistent_BANG_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._persistent_BANG_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ITransientCollection.-persistent!&quot;, tcoll);\n        }\n      }\n    }().call(null, tcoll);\n  }\n};\ncljs.core.ITransientAssociative = function() {\n  var obj22137 = {};\n  return obj22137;\n}();\ncljs.core._assoc_BANG_ = function _assoc_BANG_(tcoll, key, val) {\n  if (function() {\n    var and__17133__auto__ = tcoll;\n    if (and__17133__auto__) {\n      return tcoll.cljs$core$ITransientAssociative$_assoc_BANG_$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return tcoll.cljs$core$ITransientAssociative$_assoc_BANG_$arity$3(tcoll, key, val);\n  } else {\n    var x__17772__auto__ = tcoll == null ? null : tcoll;\n    return function() {\n      var or__17145__auto__ = cljs.core._assoc_BANG_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._assoc_BANG_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ITransientAssociative.-assoc!&quot;, tcoll);\n        }\n      }\n    }().call(null, tcoll, key, val);\n  }\n};\ncljs.core.ITransientMap = function() {\n  var obj22139 = {};\n  return obj22139;\n}();\ncljs.core._dissoc_BANG_ = function _dissoc_BANG_(tcoll, key) {\n  if (function() {\n    var and__17133__auto__ = tcoll;\n    if (and__17133__auto__) {\n      return tcoll.cljs$core$ITransientMap$_dissoc_BANG_$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return tcoll.cljs$core$ITransientMap$_dissoc_BANG_$arity$2(tcoll, key);\n  } else {\n    var x__17772__auto__ = tcoll == null ? null : tcoll;\n    return function() {\n      var or__17145__auto__ = cljs.core._dissoc_BANG_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._dissoc_BANG_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ITransientMap.-dissoc!&quot;, tcoll);\n        }\n      }\n    }().call(null, tcoll, key);\n  }\n};\ncljs.core.ITransientVector = function() {\n  var obj22141 = {};\n  return obj22141;\n}();\ncljs.core._assoc_n_BANG_ = function _assoc_n_BANG_(tcoll, n, val) {\n  if (function() {\n    var and__17133__auto__ = tcoll;\n    if (and__17133__auto__) {\n      return tcoll.cljs$core$ITransientVector$_assoc_n_BANG_$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return tcoll.cljs$core$ITransientVector$_assoc_n_BANG_$arity$3(tcoll, n, val);\n  } else {\n    var x__17772__auto__ = tcoll == null ? null : tcoll;\n    return function() {\n      var or__17145__auto__ = cljs.core._assoc_n_BANG_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._assoc_n_BANG_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ITransientVector.-assoc-n!&quot;, tcoll);\n        }\n      }\n    }().call(null, tcoll, n, val);\n  }\n};\ncljs.core._pop_BANG_ = function _pop_BANG_(tcoll) {\n  if (function() {\n    var and__17133__auto__ = tcoll;\n    if (and__17133__auto__) {\n      return tcoll.cljs$core$ITransientVector$_pop_BANG_$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return tcoll.cljs$core$ITransientVector$_pop_BANG_$arity$1(tcoll);\n  } else {\n    var x__17772__auto__ = tcoll == null ? null : tcoll;\n    return function() {\n      var or__17145__auto__ = cljs.core._pop_BANG_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._pop_BANG_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ITransientVector.-pop!&quot;, tcoll);\n        }\n      }\n    }().call(null, tcoll);\n  }\n};\ncljs.core.ITransientSet = function() {\n  var obj22143 = {};\n  return obj22143;\n}();\ncljs.core._disjoin_BANG_ = function _disjoin_BANG_(tcoll, v) {\n  if (function() {\n    var and__17133__auto__ = tcoll;\n    if (and__17133__auto__) {\n      return tcoll.cljs$core$ITransientSet$_disjoin_BANG_$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return tcoll.cljs$core$ITransientSet$_disjoin_BANG_$arity$2(tcoll, v);\n  } else {\n    var x__17772__auto__ = tcoll == null ? null : tcoll;\n    return function() {\n      var or__17145__auto__ = cljs.core._disjoin_BANG_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._disjoin_BANG_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;ITransientSet.-disjoin!&quot;, tcoll);\n        }\n      }\n    }().call(null, tcoll, v);\n  }\n};\ncljs.core.IComparable = function() {\n  var obj22145 = {};\n  return obj22145;\n}();\ncljs.core._compare = function _compare(x, y) {\n  if (function() {\n    var and__17133__auto__ = x;\n    if (and__17133__auto__) {\n      return x.cljs$core$IComparable$_compare$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return x.cljs$core$IComparable$_compare$arity$2(x, y);\n  } else {\n    var x__17772__auto__ = x == null ? null : x;\n    return function() {\n      var or__17145__auto__ = cljs.core._compare[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._compare[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IComparable.-compare&quot;, x);\n        }\n      }\n    }().call(null, x, y);\n  }\n};\ncljs.core.IChunk = function() {\n  var obj22147 = {};\n  return obj22147;\n}();\ncljs.core._drop_first = function _drop_first(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IChunk$_drop_first$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IChunk$_drop_first$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._drop_first[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._drop_first[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IChunk.-drop-first&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.IChunkedSeq = function() {\n  var obj22149 = {};\n  return obj22149;\n}();\ncljs.core._chunked_first = function _chunked_first(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IChunkedSeq$_chunked_first$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IChunkedSeq$_chunked_first$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._chunked_first[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._chunked_first[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IChunkedSeq.-chunked-first&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core._chunked_rest = function _chunked_rest(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IChunkedSeq$_chunked_rest$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IChunkedSeq$_chunked_rest$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._chunked_rest[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._chunked_rest[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IChunkedSeq.-chunked-rest&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.IChunkedNext = function() {\n  var obj22151 = {};\n  return obj22151;\n}();\ncljs.core._chunked_next = function _chunked_next(coll) {\n  if (function() {\n    var and__17133__auto__ = coll;\n    if (and__17133__auto__) {\n      return coll.cljs$core$IChunkedNext$_chunked_next$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return coll.cljs$core$IChunkedNext$_chunked_next$arity$1(coll);\n  } else {\n    var x__17772__auto__ = coll == null ? null : coll;\n    return function() {\n      var or__17145__auto__ = cljs.core._chunked_next[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._chunked_next[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IChunkedNext.-chunked-next&quot;, coll);\n        }\n      }\n    }().call(null, coll);\n  }\n};\ncljs.core.INamed = function() {\n  var obj22153 = {};\n  return obj22153;\n}();\ncljs.core._name = function _name(x) {\n  if (function() {\n    var and__17133__auto__ = x;\n    if (and__17133__auto__) {\n      return x.cljs$core$INamed$_name$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return x.cljs$core$INamed$_name$arity$1(x);\n  } else {\n    var x__17772__auto__ = x == null ? null : x;\n    return function() {\n      var or__17145__auto__ = cljs.core._name[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._name[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;INamed.-name&quot;, x);\n        }\n      }\n    }().call(null, x);\n  }\n};\ncljs.core._namespace = function _namespace(x) {\n  if (function() {\n    var and__17133__auto__ = x;\n    if (and__17133__auto__) {\n      return x.cljs$core$INamed$_namespace$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return x.cljs$core$INamed$_namespace$arity$1(x);\n  } else {\n    var x__17772__auto__ = x == null ? null : x;\n    return function() {\n      var or__17145__auto__ = cljs.core._namespace[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._namespace[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;INamed.-namespace&quot;, x);\n        }\n      }\n    }().call(null, x);\n  }\n};\ncljs.core.IAtom = function() {\n  var obj22155 = {};\n  return obj22155;\n}();\ncljs.core.IReset = function() {\n  var obj22157 = {};\n  return obj22157;\n}();\ncljs.core._reset_BANG_ = function _reset_BANG_(o, new_value) {\n  if (function() {\n    var and__17133__auto__ = o;\n    if (and__17133__auto__) {\n      return o.cljs$core$IReset$_reset_BANG_$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return o.cljs$core$IReset$_reset_BANG_$arity$2(o, new_value);\n  } else {\n    var x__17772__auto__ = o == null ? null : o;\n    return function() {\n      var or__17145__auto__ = cljs.core._reset_BANG_[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._reset_BANG_[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IReset.-reset!&quot;, o);\n        }\n      }\n    }().call(null, o, new_value);\n  }\n};\ncljs.core.ISwap = function() {\n  var obj22159 = {};\n  return obj22159;\n}();\ncljs.core._swap_BANG_ = function() {\n  var _swap_BANG_ = null;\n  var _swap_BANG___2 = function(o, f) {\n    if (function() {\n      var and__17133__auto__ = o;\n      if (and__17133__auto__) {\n        return o.cljs$core$ISwap$_swap_BANG_$arity$2;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return o.cljs$core$ISwap$_swap_BANG_$arity$2(o, f);\n    } else {\n      var x__17772__auto__ = o == null ? null : o;\n      return function() {\n        var or__17145__auto__ = cljs.core._swap_BANG_[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._swap_BANG_[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;ISwap.-swap!&quot;, o);\n          }\n        }\n      }().call(null, o, f);\n    }\n  };\n  var _swap_BANG___3 = function(o, f, a) {\n    if (function() {\n      var and__17133__auto__ = o;\n      if (and__17133__auto__) {\n        return o.cljs$core$ISwap$_swap_BANG_$arity$3;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return o.cljs$core$ISwap$_swap_BANG_$arity$3(o, f, a);\n    } else {\n      var x__17772__auto__ = o == null ? null : o;\n      return function() {\n        var or__17145__auto__ = cljs.core._swap_BANG_[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._swap_BANG_[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;ISwap.-swap!&quot;, o);\n          }\n        }\n      }().call(null, o, f, a);\n    }\n  };\n  var _swap_BANG___4 = function(o, f, a, b) {\n    if (function() {\n      var and__17133__auto__ = o;\n      if (and__17133__auto__) {\n        return o.cljs$core$ISwap$_swap_BANG_$arity$4;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return o.cljs$core$ISwap$_swap_BANG_$arity$4(o, f, a, b);\n    } else {\n      var x__17772__auto__ = o == null ? null : o;\n      return function() {\n        var or__17145__auto__ = cljs.core._swap_BANG_[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._swap_BANG_[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;ISwap.-swap!&quot;, o);\n          }\n        }\n      }().call(null, o, f, a, b);\n    }\n  };\n  var _swap_BANG___5 = function(o, f, a, b, xs) {\n    if (function() {\n      var and__17133__auto__ = o;\n      if (and__17133__auto__) {\n        return o.cljs$core$ISwap$_swap_BANG_$arity$5;\n      } else {\n        return and__17133__auto__;\n      }\n    }()) {\n      return o.cljs$core$ISwap$_swap_BANG_$arity$5(o, f, a, b, xs);\n    } else {\n      var x__17772__auto__ = o == null ? null : o;\n      return function() {\n        var or__17145__auto__ = cljs.core._swap_BANG_[goog.typeOf(x__17772__auto__)];\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = cljs.core._swap_BANG_[&quot;_&quot;];\n          if (or__17145__auto____$1) {\n            return or__17145__auto____$1;\n          } else {\n            throw cljs.core.missing_protocol.call(null, &quot;ISwap.-swap!&quot;, o);\n          }\n        }\n      }().call(null, o, f, a, b, xs);\n    }\n  };\n  _swap_BANG_ = function(o, f, a, b, xs) {\n    switch(arguments.length) {\n      case 2:\n        return _swap_BANG___2.call(this, o, f);\n      case 3:\n        return _swap_BANG___3.call(this, o, f, a);\n      case 4:\n        return _swap_BANG___4.call(this, o, f, a, b);\n      case 5:\n        return _swap_BANG___5.call(this, o, f, a, b, xs);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _swap_BANG_.cljs$core$IFn$_invoke$arity$2 = _swap_BANG___2;\n  _swap_BANG_.cljs$core$IFn$_invoke$arity$3 = _swap_BANG___3;\n  _swap_BANG_.cljs$core$IFn$_invoke$arity$4 = _swap_BANG___4;\n  _swap_BANG_.cljs$core$IFn$_invoke$arity$5 = _swap_BANG___5;\n  return _swap_BANG_;\n}();\ncljs.core.StringBufferWriter = function(sb) {\n  this.sb = sb;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 1073741824;\n};\ncljs.core.StringBufferWriter.cljs$lang$type = true;\ncljs.core.StringBufferWriter.cljs$lang$ctorStr = &quot;cljs.core/StringBufferWriter&quot;;\ncljs.core.StringBufferWriter.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/StringBufferWriter&quot;);\n};\ncljs.core.StringBufferWriter.prototype.cljs$core$IWriter$_write$arity$2 = function(_, s) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.sb.append(s);\n};\ncljs.core.StringBufferWriter.prototype.cljs$core$IWriter$_flush$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return null;\n};\ncljs.core.__GT_StringBufferWriter = function __GT_StringBufferWriter(sb) {\n  return new cljs.core.StringBufferWriter(sb);\n};\ncljs.core.pr_str_STAR_ = function pr_str_STAR_(obj) {\n  var sb = new goog.string.StringBuffer;\n  var writer = new cljs.core.StringBufferWriter(sb);\n  cljs.core._pr_writer.call(null, obj, writer, cljs.core.pr_opts.call(null));\n  cljs.core._flush.call(null, writer);\n  return &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(sb);\n};\ncljs.core.int_rotate_left = function int_rotate_left(x, n) {\n  return x &lt;&lt; n | x &gt;&gt;&gt; -n;\n};\nif (typeof Math.imul !== &quot;undefined&quot; &amp;&amp; !(Math.imul.call(null, 4294967295, 5) === 0)) {\n  cljs.core.imul = function imul(a, b) {\n    return Math.imul.call(null, a, b);\n  };\n} else {\n  cljs.core.imul = function imul(a, b) {\n    var ah = a &gt;&gt;&gt; 16 &amp; 65535;\n    var al = a &amp; 65535;\n    var bh = b &gt;&gt;&gt; 16 &amp; 65535;\n    var bl = b &amp; 65535;\n    return al * bl + (ah * bl + al * bh &lt;&lt; 16 &gt;&gt;&gt; 0) | 0;\n  };\n}\ncljs.core.m3_seed = 0;\ncljs.core.m3_C1 = 3432918353;\ncljs.core.m3_C2 = 461845907;\ncljs.core.m3_mix_K1 = function m3_mix_K1(k1) {\n  return cljs.core.imul.call(null, cljs.core.int_rotate_left.call(null, cljs.core.imul.call(null, k1, cljs.core.m3_C1), 15), cljs.core.m3_C2);\n};\ncljs.core.m3_mix_H1 = function m3_mix_H1(h1, k1) {\n  return cljs.core.imul.call(null, cljs.core.int_rotate_left.call(null, h1 ^ k1, 13), 5) + 3864292196;\n};\ncljs.core.m3_fmix = function m3_fmix(h1, len) {\n  var h1__$1 = h1;\n  var h1__$2 = h1__$1 ^ len;\n  var h1__$3 = h1__$2 ^ h1__$2 &gt;&gt;&gt; 16;\n  var h1__$4 = cljs.core.imul.call(null, h1__$3, 2246822507);\n  var h1__$5 = h1__$4 ^ h1__$4 &gt;&gt;&gt; 13;\n  var h1__$6 = cljs.core.imul.call(null, h1__$5, 3266489909);\n  var h1__$7 = h1__$6 ^ h1__$6 &gt;&gt;&gt; 16;\n  return h1__$7;\n};\ncljs.core.m3_hash_int = function m3_hash_int(in$) {\n  if (in$ === 0) {\n    return in$;\n  } else {\n    var k1 = cljs.core.m3_mix_K1.call(null, in$);\n    var h1 = cljs.core.m3_mix_H1.call(null, cljs.core.m3_seed, k1);\n    return cljs.core.m3_fmix.call(null, h1, 4);\n  }\n};\ncljs.core.m3_hash_unencoded_chars = function m3_hash_unencoded_chars(in$) {\n  var h1 = function() {\n    var i = 1;\n    var h1 = cljs.core.m3_seed;\n    while (true) {\n      if (i &lt; in$.length) {\n        var G__22160 = i + 2;\n        var G__22161 = cljs.core.m3_mix_H1.call(null, h1, cljs.core.m3_mix_K1.call(null, in$.charCodeAt(i - 1) | in$.charCodeAt(i) &lt;&lt; 16));\n        i = G__22160;\n        h1 = G__22161;\n        continue;\n      } else {\n        return h1;\n      }\n      break;\n    }\n  }();\n  var h1__$1 = (in$.length &amp; 1) === 1 ? h1 ^ cljs.core.m3_mix_K1.call(null, in$.charCodeAt(in$.length - 1)) : h1;\n  return cljs.core.m3_fmix.call(null, h1__$1, cljs.core.imul.call(null, 2, in$.length));\n};\ncljs.core.string_hash_cache = function() {\n  var obj22163 = {};\n  return obj22163;\n}();\ncljs.core.string_hash_cache_count = 0;\ncljs.core.hash_string_STAR_ = function hash_string_STAR_(s) {\n  if (!(s == null)) {\n    var len = s.length;\n    if (len &gt; 0) {\n      var i = 0;\n      var hash = 0;\n      while (true) {\n        if (i &lt; len) {\n          var G__22164 = i + 1;\n          var G__22165 = cljs.core.imul.call(null, 31, hash) + s.charCodeAt(i);\n          i = G__22164;\n          hash = G__22165;\n          continue;\n        } else {\n          return hash;\n        }\n        break;\n      }\n    } else {\n      return 0;\n    }\n  } else {\n    return 0;\n  }\n};\ncljs.core.add_to_string_hash_cache = function add_to_string_hash_cache(k) {\n  var h = cljs.core.hash_string_STAR_.call(null, k);\n  cljs.core.string_hash_cache[k] = h;\n  cljs.core.string_hash_cache_count = cljs.core.string_hash_cache_count + 1;\n  return h;\n};\ncljs.core.hash_string = function hash_string(k) {\n  if (cljs.core.string_hash_cache_count &gt; 255) {\n    cljs.core.string_hash_cache = function() {\n      var obj22169 = {};\n      return obj22169;\n    }();\n    cljs.core.string_hash_cache_count = 0;\n  } else {\n  }\n  var h = cljs.core.string_hash_cache[k];\n  if (typeof h === &quot;number&quot;) {\n    return h;\n  } else {\n    return cljs.core.add_to_string_hash_cache.call(null, k);\n  }\n};\ncljs.core.hash = function hash(o) {\n  if (function() {\n    var G__22171 = o;\n    if (G__22171) {\n      var bit__17788__auto__ = G__22171.cljs$lang$protocol_mask$partition0$ &amp; 4194304;\n      if (bit__17788__auto__ || G__22171.cljs$core$IHash$) {\n        return true;\n      } else {\n        return false;\n      }\n    } else {\n      return false;\n    }\n  }()) {\n    return cljs.core._hash.call(null, o);\n  } else {\n    if (typeof o === &quot;number&quot;) {\n      return Math.floor.call(null, o) % 2147483647;\n    } else {\n      if (o === true) {\n        return 1;\n      } else {\n        if (o === false) {\n          return 0;\n        } else {\n          if (typeof o === &quot;string&quot;) {\n            return cljs.core.m3_hash_int.call(null, cljs.core.hash_string.call(null, o));\n          } else {\n            if (o == null) {\n              return 0;\n            } else {\n              return cljs.core._hash.call(null, o);\n            }\n          }\n        }\n      }\n    }\n  }\n};\ncljs.core.hash_combine = function hash_combine(seed, hash) {\n  return seed ^ hash + 2654435769 + (seed &lt;&lt; 6) + (seed &gt;&gt; 2);\n};\ncljs.core.instance_QMARK_ = function instance_QMARK_(t, o) {\n  return o instanceof t;\n};\ncljs.core.symbol_QMARK_ = function symbol_QMARK_(x) {\n  return x instanceof cljs.core.Symbol;\n};\ncljs.core.hash_symbol = function hash_symbol(sym) {\n  return cljs.core.hash_combine.call(null, cljs.core.m3_hash_unencoded_chars.call(null, sym.name), cljs.core.hash_string.call(null, sym.ns));\n};\ncljs.core.compare_symbols = function compare_symbols(a, b) {\n  if (cljs.core.truth_(cljs.core._EQ_.call(null, a, b))) {\n    return 0;\n  } else {\n    if (cljs.core.truth_(function() {\n      var and__17133__auto__ = cljs.core.not.call(null, a.ns);\n      if (and__17133__auto__) {\n        return b.ns;\n      } else {\n        return and__17133__auto__;\n      }\n    }())) {\n      return-1;\n    } else {\n      if (cljs.core.truth_(a.ns)) {\n        if (cljs.core.not.call(null, b.ns)) {\n          return 1;\n        } else {\n          var nsc = cljs.core.compare.call(null, a.ns, b.ns);\n          if (nsc === 0) {\n            return cljs.core.compare.call(null, a.name, b.name);\n          } else {\n            return nsc;\n          }\n        }\n      } else {\n        return cljs.core.compare.call(null, a.name, b.name);\n      }\n    }\n  }\n};\ncljs.core.Symbol = function(ns, name, str, _hash, _meta) {\n  this.ns = ns;\n  this.name = name;\n  this.str = str;\n  this._hash = _hash;\n  this._meta = _meta;\n  this.cljs$lang$protocol_mask$partition0$ = 2154168321;\n  this.cljs$lang$protocol_mask$partition1$ = 4096;\n};\ncljs.core.Symbol.cljs$lang$type = true;\ncljs.core.Symbol.cljs$lang$ctorStr = &quot;cljs.core/Symbol&quot;;\ncljs.core.Symbol.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Symbol&quot;);\n};\ncljs.core.Symbol.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(o, writer, _) {\n  var self__ = this;\n  var o__$1 = this;\n  return cljs.core._write.call(null, writer, self__.str);\n};\ncljs.core.Symbol.prototype.cljs$core$INamed$_name$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.name;\n};\ncljs.core.Symbol.prototype.cljs$core$INamed$_namespace$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.ns;\n};\ncljs.core.Symbol.prototype.cljs$core$IHash$_hash$arity$1 = function(sym) {\n  var self__ = this;\n  var sym__$1 = this;\n  var h__17556__auto__ = self__._hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_symbol.call(null, sym__$1);\n    self__._hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.Symbol.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(_, new_meta) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.Symbol(self__.ns, self__.name, self__.str, self__._hash, new_meta);\n};\ncljs.core.Symbol.prototype.cljs$core$IMeta$_meta$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__._meta;\n};\ncljs.core.Symbol.prototype.call = function() {\n  var G__22173 = null;\n  var G__22173__2 = function(self__, coll) {\n    var self__ = this;\n    var self____$1 = this;\n    var sym = self____$1;\n    return cljs.core._lookup.call(null, coll, sym, null);\n  };\n  var G__22173__3 = function(self__, coll, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var sym = self____$1;\n    return cljs.core._lookup.call(null, coll, sym, not_found);\n  };\n  G__22173 = function(self__, coll, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22173__2.call(this, self__, coll);\n      case 3:\n        return G__22173__3.call(this, self__, coll, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22173.cljs$core$IFn$_invoke$arity$2 = G__22173__2;\n  G__22173.cljs$core$IFn$_invoke$arity$3 = G__22173__3;\n  return G__22173;\n}();\ncljs.core.Symbol.prototype.apply = function(self__, args22172) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22172)));\n};\ncljs.core.Symbol.prototype.cljs$core$IFn$_invoke$arity$1 = function(coll) {\n  var self__ = this;\n  var sym = this;\n  return cljs.core._lookup.call(null, coll, sym, null);\n};\ncljs.core.Symbol.prototype.cljs$core$IFn$_invoke$arity$2 = function(coll, not_found) {\n  var self__ = this;\n  var sym = this;\n  return cljs.core._lookup.call(null, coll, sym, not_found);\n};\ncljs.core.Symbol.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(_, other) {\n  var self__ = this;\n  var ___$1 = this;\n  if (other instanceof cljs.core.Symbol) {\n    return self__.str === other.str;\n  } else {\n    return false;\n  }\n};\ncljs.core.Symbol.prototype.toString = function() {\n  var self__ = this;\n  var _ = this;\n  return self__.str;\n};\ncljs.core.Symbol.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.__GT_Symbol = function __GT_Symbol(ns, name, str, _hash, _meta) {\n  return new cljs.core.Symbol(ns, name, str, _hash, _meta);\n};\ncljs.core.symbol = function() {\n  var symbol = null;\n  var symbol__1 = function(name) {\n    if (name instanceof cljs.core.Symbol) {\n      return name;\n    } else {\n      return symbol.call(null, null, name);\n    }\n  };\n  var symbol__2 = function(ns, name) {\n    var sym_str = !(ns == null) ? &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(ns) + &quot;/&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(name) : name;\n    return new cljs.core.Symbol(ns, name, sym_str, null, null);\n  };\n  symbol = function(ns, name) {\n    switch(arguments.length) {\n      case 1:\n        return symbol__1.call(this, ns);\n      case 2:\n        return symbol__2.call(this, ns, name);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  symbol.cljs$core$IFn$_invoke$arity$1 = symbol__1;\n  symbol.cljs$core$IFn$_invoke$arity$2 = symbol__2;\n  return symbol;\n}();\ncljs.core.clone = function clone(value) {\n  return cljs.core._clone.call(null, value);\n};\ncljs.core.cloneable_QMARK_ = function cloneable_QMARK_(value) {\n  var G__22175 = value;\n  if (G__22175) {\n    var bit__17795__auto__ = G__22175.cljs$lang$protocol_mask$partition1$ &amp; 8192;\n    if (bit__17795__auto__ || G__22175.cljs$core$ICloneable$) {\n      return true;\n    } else {\n      if (!G__22175.cljs$lang$protocol_mask$partition1$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ICloneable, G__22175);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ICloneable, G__22175);\n  }\n};\ncljs.core.seq = function seq(coll) {\n  if (coll == null) {\n    return null;\n  } else {\n    if (function() {\n      var G__22177 = coll;\n      if (G__22177) {\n        var bit__17788__auto__ = G__22177.cljs$lang$protocol_mask$partition0$ &amp; 8388608;\n        if (bit__17788__auto__ || G__22177.cljs$core$ISeqable$) {\n          return true;\n        } else {\n          return false;\n        }\n      } else {\n        return false;\n      }\n    }()) {\n      return cljs.core._seq.call(null, coll);\n    } else {\n      if (coll instanceof Array) {\n        if (coll.length === 0) {\n          return null;\n        } else {\n          return new cljs.core.IndexedSeq(coll, 0);\n        }\n      } else {\n        if (typeof coll === &quot;string&quot;) {\n          if (coll.length === 0) {\n            return null;\n          } else {\n            return new cljs.core.IndexedSeq(coll, 0);\n          }\n        } else {\n          if (cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISeqable, coll)) {\n            return cljs.core._seq.call(null, coll);\n          } else {\n            throw new Error(&quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(coll) + &quot; is not ISeqable&quot;);\n          }\n        }\n      }\n    }\n  }\n};\ncljs.core.first = function first(coll) {\n  if (coll == null) {\n    return null;\n  } else {\n    if (function() {\n      var G__22179 = coll;\n      if (G__22179) {\n        var bit__17788__auto__ = G__22179.cljs$lang$protocol_mask$partition0$ &amp; 64;\n        if (bit__17788__auto__ || G__22179.cljs$core$ISeq$) {\n          return true;\n        } else {\n          return false;\n        }\n      } else {\n        return false;\n      }\n    }()) {\n      return cljs.core._first.call(null, coll);\n    } else {\n      var s = cljs.core.seq.call(null, coll);\n      if (s == null) {\n        return null;\n      } else {\n        return cljs.core._first.call(null, s);\n      }\n    }\n  }\n};\ncljs.core.rest = function rest(coll) {\n  if (!(coll == null)) {\n    if (function() {\n      var G__22181 = coll;\n      if (G__22181) {\n        var bit__17788__auto__ = G__22181.cljs$lang$protocol_mask$partition0$ &amp; 64;\n        if (bit__17788__auto__ || G__22181.cljs$core$ISeq$) {\n          return true;\n        } else {\n          return false;\n        }\n      } else {\n        return false;\n      }\n    }()) {\n      return cljs.core._rest.call(null, coll);\n    } else {\n      var s = cljs.core.seq.call(null, coll);\n      if (s) {\n        return cljs.core._rest.call(null, s);\n      } else {\n        return cljs.core.List.EMPTY;\n      }\n    }\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.next = function next(coll) {\n  if (coll == null) {\n    return null;\n  } else {\n    if (function() {\n      var G__22183 = coll;\n      if (G__22183) {\n        var bit__17788__auto__ = G__22183.cljs$lang$protocol_mask$partition0$ &amp; 128;\n        if (bit__17788__auto__ || G__22183.cljs$core$INext$) {\n          return true;\n        } else {\n          return false;\n        }\n      } else {\n        return false;\n      }\n    }()) {\n      return cljs.core._next.call(null, coll);\n    } else {\n      return cljs.core.seq.call(null, cljs.core.rest.call(null, coll));\n    }\n  }\n};\ncljs.core._EQ_ = function() {\n  var _EQ_ = null;\n  var _EQ___1 = function(x) {\n    return true;\n  };\n  var _EQ___2 = function(x, y) {\n    if (x == null) {\n      return y == null;\n    } else {\n      return x === y || cljs.core._equiv.call(null, x, y);\n    }\n  };\n  var _EQ___3 = function() {\n    var G__22184__delegate = function(x, y, more) {\n      while (true) {\n        if (_EQ_.call(null, x, y)) {\n          if (cljs.core.next.call(null, more)) {\n            var G__22185 = y;\n            var G__22186 = cljs.core.first.call(null, more);\n            var G__22187 = cljs.core.next.call(null, more);\n            x = G__22185;\n            y = G__22186;\n            more = G__22187;\n            continue;\n          } else {\n            return _EQ_.call(null, y, cljs.core.first.call(null, more));\n          }\n        } else {\n          return false;\n        }\n        break;\n      }\n    };\n    var G__22184 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22184__delegate.call(this, x, y, more);\n    };\n    G__22184.cljs$lang$maxFixedArity = 2;\n    G__22184.cljs$lang$applyTo = function(arglist__22188) {\n      var x = cljs.core.first(arglist__22188);\n      arglist__22188 = cljs.core.next(arglist__22188);\n      var y = cljs.core.first(arglist__22188);\n      var more = cljs.core.rest(arglist__22188);\n      return G__22184__delegate(x, y, more);\n    };\n    G__22184.cljs$core$IFn$_invoke$arity$variadic = G__22184__delegate;\n    return G__22184;\n  }();\n  _EQ_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return _EQ___1.call(this, x);\n      case 2:\n        return _EQ___2.call(this, x, y);\n      default:\n        return _EQ___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _EQ_.cljs$lang$maxFixedArity = 2;\n  _EQ_.cljs$lang$applyTo = _EQ___3.cljs$lang$applyTo;\n  _EQ_.cljs$core$IFn$_invoke$arity$1 = _EQ___1;\n  _EQ_.cljs$core$IFn$_invoke$arity$2 = _EQ___2;\n  _EQ_.cljs$core$IFn$_invoke$arity$variadic = _EQ___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _EQ_;\n}();\ncljs.core.mix_collection_hash = function mix_collection_hash(hash_basis, count) {\n  var h1 = cljs.core.m3_seed;\n  var k1 = cljs.core.m3_mix_K1.call(null, hash_basis);\n  var h1__$1 = cljs.core.m3_mix_H1.call(null, h1, k1);\n  return cljs.core.m3_fmix.call(null, h1__$1, count);\n};\ncljs.core.hash_ordered_coll = function hash_ordered_coll(coll) {\n  var n = 0;\n  var hash_code = 1;\n  var coll__$1 = cljs.core.seq.call(null, coll);\n  while (true) {\n    if (!(coll__$1 == null)) {\n      var G__22189 = n + 1;\n      var G__22190 = cljs.core.imul.call(null, 31, hash_code) + cljs.core.hash.call(null, cljs.core.first.call(null, coll__$1)) | 0;\n      var G__22191 = cljs.core.next.call(null, coll__$1);\n      n = G__22189;\n      hash_code = G__22190;\n      coll__$1 = G__22191;\n      continue;\n    } else {\n      return cljs.core.mix_collection_hash.call(null, hash_code, n);\n    }\n    break;\n  }\n};\ncljs.core.hash_unordered_coll = function hash_unordered_coll(coll) {\n  var n = 0;\n  var hash_code = 0;\n  var coll__$1 = cljs.core.seq.call(null, coll);\n  while (true) {\n    if (!(coll__$1 == null)) {\n      var G__22192 = n + 1;\n      var G__22193 = hash_code + cljs.core.hash.call(null, cljs.core.first.call(null, coll__$1)) | 0;\n      var G__22194 = cljs.core.next.call(null, coll__$1);\n      n = G__22192;\n      hash_code = G__22193;\n      coll__$1 = G__22194;\n      continue;\n    } else {\n      return cljs.core.mix_collection_hash.call(null, hash_code, n);\n    }\n    break;\n  }\n};\ncljs.core.ICounted[&quot;null&quot;] = true;\ncljs.core._count[&quot;null&quot;] = function(_) {\n  return 0;\n};\nDate.prototype.cljs$core$IEquiv$ = true;\nDate.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(o, other) {\n  var o__$1 = this;\n  return other instanceof Date &amp;&amp; o__$1.toString() === other.toString();\n};\ncljs.core.IEquiv[&quot;number&quot;] = true;\ncljs.core._equiv[&quot;number&quot;] = function(x, o) {\n  return x === o;\n};\ncljs.core.IMeta[&quot;function&quot;] = true;\ncljs.core._meta[&quot;function&quot;] = function(_) {\n  return null;\n};\ncljs.core.Fn[&quot;function&quot;] = true;\ncljs.core.IHash[&quot;_&quot;] = true;\ncljs.core._hash[&quot;_&quot;] = function(o) {\n  return goog.getUid(o);\n};\ncljs.core.inc = function inc(x) {\n  return x + 1;\n};\ncljs.core.Reduced = function(val) {\n  this.val = val;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32768;\n};\ncljs.core.Reduced.cljs$lang$type = true;\ncljs.core.Reduced.cljs$lang$ctorStr = &quot;cljs.core/Reduced&quot;;\ncljs.core.Reduced.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Reduced&quot;);\n};\ncljs.core.Reduced.prototype.cljs$core$IDeref$_deref$arity$1 = function(o) {\n  var self__ = this;\n  var o__$1 = this;\n  return self__.val;\n};\ncljs.core.__GT_Reduced = function __GT_Reduced(val) {\n  return new cljs.core.Reduced(val);\n};\ncljs.core.reduced = function reduced(x) {\n  return new cljs.core.Reduced(x);\n};\ncljs.core.reduced_QMARK_ = function reduced_QMARK_(r) {\n  return r instanceof cljs.core.Reduced;\n};\ncljs.core.deref = function deref(o) {\n  return cljs.core._deref.call(null, o);\n};\ncljs.core.ci_reduce = function() {\n  var ci_reduce = null;\n  var ci_reduce__2 = function(cicoll, f) {\n    var cnt = cljs.core._count.call(null, cicoll);\n    if (cnt === 0) {\n      return f.call(null);\n    } else {\n      var val = cljs.core._nth.call(null, cicoll, 0);\n      var n = 1;\n      while (true) {\n        if (n &lt; cnt) {\n          var nval = f.call(null, val, cljs.core._nth.call(null, cicoll, n));\n          if (cljs.core.reduced_QMARK_.call(null, nval)) {\n            return cljs.core.deref.call(null, nval);\n          } else {\n            var G__22195 = nval;\n            var G__22196 = n + 1;\n            val = G__22195;\n            n = G__22196;\n            continue;\n          }\n        } else {\n          return val;\n        }\n        break;\n      }\n    }\n  };\n  var ci_reduce__3 = function(cicoll, f, val) {\n    var cnt = cljs.core._count.call(null, cicoll);\n    var val__$1 = val;\n    var n = 0;\n    while (true) {\n      if (n &lt; cnt) {\n        var nval = f.call(null, val__$1, cljs.core._nth.call(null, cicoll, n));\n        if (cljs.core.reduced_QMARK_.call(null, nval)) {\n          return cljs.core.deref.call(null, nval);\n        } else {\n          var G__22197 = nval;\n          var G__22198 = n + 1;\n          val__$1 = G__22197;\n          n = G__22198;\n          continue;\n        }\n      } else {\n        return val__$1;\n      }\n      break;\n    }\n  };\n  var ci_reduce__4 = function(cicoll, f, val, idx) {\n    var cnt = cljs.core._count.call(null, cicoll);\n    var val__$1 = val;\n    var n = idx;\n    while (true) {\n      if (n &lt; cnt) {\n        var nval = f.call(null, val__$1, cljs.core._nth.call(null, cicoll, n));\n        if (cljs.core.reduced_QMARK_.call(null, nval)) {\n          return cljs.core.deref.call(null, nval);\n        } else {\n          var G__22199 = nval;\n          var G__22200 = n + 1;\n          val__$1 = G__22199;\n          n = G__22200;\n          continue;\n        }\n      } else {\n        return val__$1;\n      }\n      break;\n    }\n  };\n  ci_reduce = function(cicoll, f, val, idx) {\n    switch(arguments.length) {\n      case 2:\n        return ci_reduce__2.call(this, cicoll, f);\n      case 3:\n        return ci_reduce__3.call(this, cicoll, f, val);\n      case 4:\n        return ci_reduce__4.call(this, cicoll, f, val, idx);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  ci_reduce.cljs$core$IFn$_invoke$arity$2 = ci_reduce__2;\n  ci_reduce.cljs$core$IFn$_invoke$arity$3 = ci_reduce__3;\n  ci_reduce.cljs$core$IFn$_invoke$arity$4 = ci_reduce__4;\n  return ci_reduce;\n}();\ncljs.core.array_reduce = function() {\n  var array_reduce = null;\n  var array_reduce__2 = function(arr, f) {\n    var cnt = arr.length;\n    if (arr.length === 0) {\n      return f.call(null);\n    } else {\n      var val = arr[0];\n      var n = 1;\n      while (true) {\n        if (n &lt; cnt) {\n          var nval = f.call(null, val, arr[n]);\n          if (cljs.core.reduced_QMARK_.call(null, nval)) {\n            return cljs.core.deref.call(null, nval);\n          } else {\n            var G__22201 = nval;\n            var G__22202 = n + 1;\n            val = G__22201;\n            n = G__22202;\n            continue;\n          }\n        } else {\n          return val;\n        }\n        break;\n      }\n    }\n  };\n  var array_reduce__3 = function(arr, f, val) {\n    var cnt = arr.length;\n    var val__$1 = val;\n    var n = 0;\n    while (true) {\n      if (n &lt; cnt) {\n        var nval = f.call(null, val__$1, arr[n]);\n        if (cljs.core.reduced_QMARK_.call(null, nval)) {\n          return cljs.core.deref.call(null, nval);\n        } else {\n          var G__22203 = nval;\n          var G__22204 = n + 1;\n          val__$1 = G__22203;\n          n = G__22204;\n          continue;\n        }\n      } else {\n        return val__$1;\n      }\n      break;\n    }\n  };\n  var array_reduce__4 = function(arr, f, val, idx) {\n    var cnt = arr.length;\n    var val__$1 = val;\n    var n = idx;\n    while (true) {\n      if (n &lt; cnt) {\n        var nval = f.call(null, val__$1, arr[n]);\n        if (cljs.core.reduced_QMARK_.call(null, nval)) {\n          return cljs.core.deref.call(null, nval);\n        } else {\n          var G__22205 = nval;\n          var G__22206 = n + 1;\n          val__$1 = G__22205;\n          n = G__22206;\n          continue;\n        }\n      } else {\n        return val__$1;\n      }\n      break;\n    }\n  };\n  array_reduce = function(arr, f, val, idx) {\n    switch(arguments.length) {\n      case 2:\n        return array_reduce__2.call(this, arr, f);\n      case 3:\n        return array_reduce__3.call(this, arr, f, val);\n      case 4:\n        return array_reduce__4.call(this, arr, f, val, idx);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  array_reduce.cljs$core$IFn$_invoke$arity$2 = array_reduce__2;\n  array_reduce.cljs$core$IFn$_invoke$arity$3 = array_reduce__3;\n  array_reduce.cljs$core$IFn$_invoke$arity$4 = array_reduce__4;\n  return array_reduce;\n}();\ncljs.core.counted_QMARK_ = function counted_QMARK_(x) {\n  var G__22208 = x;\n  if (G__22208) {\n    var bit__17795__auto__ = G__22208.cljs$lang$protocol_mask$partition0$ &amp; 2;\n    if (bit__17795__auto__ || G__22208.cljs$core$ICounted$) {\n      return true;\n    } else {\n      if (!G__22208.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ICounted, G__22208);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ICounted, G__22208);\n  }\n};\ncljs.core.indexed_QMARK_ = function indexed_QMARK_(x) {\n  var G__22210 = x;\n  if (G__22210) {\n    var bit__17795__auto__ = G__22210.cljs$lang$protocol_mask$partition0$ &amp; 16;\n    if (bit__17795__auto__ || G__22210.cljs$core$IIndexed$) {\n      return true;\n    } else {\n      if (!G__22210.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IIndexed, G__22210);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IIndexed, G__22210);\n  }\n};\ncljs.core.IndexedSeq = function(arr, i) {\n  this.arr = arr;\n  this.i = i;\n  this.cljs$lang$protocol_mask$partition0$ = 166199550;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.IndexedSeq.cljs$lang$type = true;\ncljs.core.IndexedSeq.cljs$lang$ctorStr = &quot;cljs.core/IndexedSeq&quot;;\ncljs.core.IndexedSeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/IndexedSeq&quot;);\n};\ncljs.core.IndexedSeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.IndexedSeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.IndexedSeq.prototype.cljs$core$IIndexed$_nth$arity$2 = function(coll, n) {\n  var self__ = this;\n  var coll__$1 = this;\n  var i__$1 = n + self__.i;\n  if (i__$1 &lt; self__.arr.length) {\n    return self__.arr[i__$1];\n  } else {\n    return null;\n  }\n};\ncljs.core.IndexedSeq.prototype.cljs$core$IIndexed$_nth$arity$3 = function(coll, n, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  var i__$1 = n + self__.i;\n  if (i__$1 &lt; self__.arr.length) {\n    return self__.arr[i__$1];\n  } else {\n    return not_found;\n  }\n};\ncljs.core.IndexedSeq.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.IndexedSeq(self__.arr, self__.i);\n};\ncljs.core.IndexedSeq.prototype.cljs$core$INext$_next$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  if (self__.i + 1 &lt; self__.arr.length) {\n    return new cljs.core.IndexedSeq(self__.arr, self__.i + 1);\n  } else {\n    return null;\n  }\n};\ncljs.core.IndexedSeq.prototype.cljs$core$ICounted$_count$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.arr.length - self__.i;\n};\ncljs.core.IndexedSeq.prototype.cljs$core$IReversible$_rseq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var c = cljs.core._count.call(null, coll__$1);\n  if (c &gt; 0) {\n    return new cljs.core.RSeq(coll__$1, c - 1, null);\n  } else {\n    return null;\n  }\n};\ncljs.core.IndexedSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.hash_ordered_coll.call(null, coll__$1);\n};\ncljs.core.IndexedSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.IndexedSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.List.EMPTY;\n};\ncljs.core.IndexedSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.array_reduce.call(null, self__.arr, f, self__.arr[self__.i], self__.i + 1);\n};\ncljs.core.IndexedSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.array_reduce.call(null, self__.arr, f, start, self__.i);\n};\ncljs.core.IndexedSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.arr[self__.i];\n};\ncljs.core.IndexedSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  if (self__.i + 1 &lt; self__.arr.length) {\n    return new cljs.core.IndexedSeq(self__.arr, self__.i + 1);\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.IndexedSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  return this$__$1;\n};\ncljs.core.IndexedSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_IndexedSeq = function __GT_IndexedSeq(arr, i) {\n  return new cljs.core.IndexedSeq(arr, i);\n};\ncljs.core.prim_seq = function() {\n  var prim_seq = null;\n  var prim_seq__1 = function(prim) {\n    return prim_seq.call(null, prim, 0);\n  };\n  var prim_seq__2 = function(prim, i) {\n    if (i &lt; prim.length) {\n      return new cljs.core.IndexedSeq(prim, i);\n    } else {\n      return null;\n    }\n  };\n  prim_seq = function(prim, i) {\n    switch(arguments.length) {\n      case 1:\n        return prim_seq__1.call(this, prim);\n      case 2:\n        return prim_seq__2.call(this, prim, i);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  prim_seq.cljs$core$IFn$_invoke$arity$1 = prim_seq__1;\n  prim_seq.cljs$core$IFn$_invoke$arity$2 = prim_seq__2;\n  return prim_seq;\n}();\ncljs.core.array_seq = function() {\n  var array_seq = null;\n  var array_seq__1 = function(array) {\n    return cljs.core.prim_seq.call(null, array, 0);\n  };\n  var array_seq__2 = function(array, i) {\n    return cljs.core.prim_seq.call(null, array, i);\n  };\n  array_seq = function(array, i) {\n    switch(arguments.length) {\n      case 1:\n        return array_seq__1.call(this, array);\n      case 2:\n        return array_seq__2.call(this, array, i);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  array_seq.cljs$core$IFn$_invoke$arity$1 = array_seq__1;\n  array_seq.cljs$core$IFn$_invoke$arity$2 = array_seq__2;\n  return array_seq;\n}();\ncljs.core.RSeq = function(ci, i, meta) {\n  this.ci = ci;\n  this.i = i;\n  this.meta = meta;\n  this.cljs$lang$protocol_mask$partition0$ = 32374990;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.RSeq.cljs$lang$type = true;\ncljs.core.RSeq.cljs$lang$ctorStr = &quot;cljs.core/RSeq&quot;;\ncljs.core.RSeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/RSeq&quot;);\n};\ncljs.core.RSeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.RSeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.RSeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.RSeq.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.RSeq(self__.ci, self__.i, self__.meta);\n};\ncljs.core.RSeq.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.i &gt; 0) {\n    return new cljs.core.RSeq(self__.ci, self__.i - 1, null);\n  } else {\n    return null;\n  }\n};\ncljs.core.RSeq.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.i + 1;\n};\ncljs.core.RSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.hash_ordered_coll.call(null, coll__$1);\n};\ncljs.core.RSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.RSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__.meta);\n};\ncljs.core.RSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(col, f) {\n  var self__ = this;\n  var col__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, col__$1);\n};\ncljs.core.RSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(col, f, start) {\n  var self__ = this;\n  var col__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, col__$1);\n};\ncljs.core.RSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._nth.call(null, self__.ci, self__.i);\n};\ncljs.core.RSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.i &gt; 0) {\n    return new cljs.core.RSeq(self__.ci, self__.i - 1, null);\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.RSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.RSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, new_meta) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.RSeq(self__.ci, self__.i, new_meta);\n};\ncljs.core.RSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_RSeq = function __GT_RSeq(ci, i, meta) {\n  return new cljs.core.RSeq(ci, i, meta);\n};\ncljs.core.second = function second(coll) {\n  return cljs.core.first.call(null, cljs.core.next.call(null, coll));\n};\ncljs.core.ffirst = function ffirst(coll) {\n  return cljs.core.first.call(null, cljs.core.first.call(null, coll));\n};\ncljs.core.nfirst = function nfirst(coll) {\n  return cljs.core.next.call(null, cljs.core.first.call(null, coll));\n};\ncljs.core.fnext = function fnext(coll) {\n  return cljs.core.first.call(null, cljs.core.next.call(null, coll));\n};\ncljs.core.nnext = function nnext(coll) {\n  return cljs.core.next.call(null, cljs.core.next.call(null, coll));\n};\ncljs.core.last = function last(s) {\n  while (true) {\n    var sn = cljs.core.next.call(null, s);\n    if (!(sn == null)) {\n      var G__22211 = sn;\n      s = G__22211;\n      continue;\n    } else {\n      return cljs.core.first.call(null, s);\n    }\n    break;\n  }\n};\ncljs.core.IEquiv[&quot;_&quot;] = true;\ncljs.core._equiv[&quot;_&quot;] = function(x, o) {\n  return x === o;\n};\ncljs.core.conj = function() {\n  var conj = null;\n  var conj__0 = function() {\n    return cljs.core.PersistentVector.EMPTY;\n  };\n  var conj__1 = function(coll) {\n    return coll;\n  };\n  var conj__2 = function(coll, x) {\n    if (!(coll == null)) {\n      return cljs.core._conj.call(null, coll, x);\n    } else {\n      return cljs.core._conj.call(null, cljs.core.List.EMPTY, x);\n    }\n  };\n  var conj__3 = function() {\n    var G__22212__delegate = function(coll, x, xs) {\n      while (true) {\n        if (cljs.core.truth_(xs)) {\n          var G__22213 = conj.call(null, coll, x);\n          var G__22214 = cljs.core.first.call(null, xs);\n          var G__22215 = cljs.core.next.call(null, xs);\n          coll = G__22213;\n          x = G__22214;\n          xs = G__22215;\n          continue;\n        } else {\n          return conj.call(null, coll, x);\n        }\n        break;\n      }\n    };\n    var G__22212 = function(coll, x, var_args) {\n      var xs = null;\n      if (arguments.length &gt; 2) {\n        xs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22212__delegate.call(this, coll, x, xs);\n    };\n    G__22212.cljs$lang$maxFixedArity = 2;\n    G__22212.cljs$lang$applyTo = function(arglist__22216) {\n      var coll = cljs.core.first(arglist__22216);\n      arglist__22216 = cljs.core.next(arglist__22216);\n      var x = cljs.core.first(arglist__22216);\n      var xs = cljs.core.rest(arglist__22216);\n      return G__22212__delegate(coll, x, xs);\n    };\n    G__22212.cljs$core$IFn$_invoke$arity$variadic = G__22212__delegate;\n    return G__22212;\n  }();\n  conj = function(coll, x, var_args) {\n    var xs = var_args;\n    switch(arguments.length) {\n      case 0:\n        return conj__0.call(this);\n      case 1:\n        return conj__1.call(this, coll);\n      case 2:\n        return conj__2.call(this, coll, x);\n      default:\n        return conj__3.cljs$core$IFn$_invoke$arity$variadic(coll, x, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  conj.cljs$lang$maxFixedArity = 2;\n  conj.cljs$lang$applyTo = conj__3.cljs$lang$applyTo;\n  conj.cljs$core$IFn$_invoke$arity$0 = conj__0;\n  conj.cljs$core$IFn$_invoke$arity$1 = conj__1;\n  conj.cljs$core$IFn$_invoke$arity$2 = conj__2;\n  conj.cljs$core$IFn$_invoke$arity$variadic = conj__3.cljs$core$IFn$_invoke$arity$variadic;\n  return conj;\n}();\ncljs.core.empty = function empty(coll) {\n  if (coll == null) {\n    return null;\n  } else {\n    return cljs.core._empty.call(null, coll);\n  }\n};\ncljs.core.accumulating_seq_count = function accumulating_seq_count(coll) {\n  var s = cljs.core.seq.call(null, coll);\n  var acc = 0;\n  while (true) {\n    if (cljs.core.counted_QMARK_.call(null, s)) {\n      return acc + cljs.core._count.call(null, s);\n    } else {\n      var G__22217 = cljs.core.next.call(null, s);\n      var G__22218 = acc + 1;\n      s = G__22217;\n      acc = G__22218;\n      continue;\n    }\n    break;\n  }\n};\ncljs.core.count = function count(coll) {\n  if (!(coll == null)) {\n    if (function() {\n      var G__22220 = coll;\n      if (G__22220) {\n        var bit__17788__auto__ = G__22220.cljs$lang$protocol_mask$partition0$ &amp; 2;\n        if (bit__17788__auto__ || G__22220.cljs$core$ICounted$) {\n          return true;\n        } else {\n          return false;\n        }\n      } else {\n        return false;\n      }\n    }()) {\n      return cljs.core._count.call(null, coll);\n    } else {\n      if (coll instanceof Array) {\n        return coll.length;\n      } else {\n        if (typeof coll === &quot;string&quot;) {\n          return coll.length;\n        } else {\n          if (cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ICounted, coll)) {\n            return cljs.core._count.call(null, coll);\n          } else {\n            return cljs.core.accumulating_seq_count.call(null, coll);\n          }\n        }\n      }\n    }\n  } else {\n    return 0;\n  }\n};\ncljs.core.linear_traversal_nth = function() {\n  var linear_traversal_nth = null;\n  var linear_traversal_nth__2 = function(coll, n) {\n    while (true) {\n      if (coll == null) {\n        throw new Error(&quot;Index out of bounds&quot;);\n      } else {\n        if (n === 0) {\n          if (cljs.core.seq.call(null, coll)) {\n            return cljs.core.first.call(null, coll);\n          } else {\n            throw new Error(&quot;Index out of bounds&quot;);\n          }\n        } else {\n          if (cljs.core.indexed_QMARK_.call(null, coll)) {\n            return cljs.core._nth.call(null, coll, n);\n          } else {\n            if (cljs.core.seq.call(null, coll)) {\n              var G__22221 = cljs.core.next.call(null, coll);\n              var G__22222 = n - 1;\n              coll = G__22221;\n              n = G__22222;\n              continue;\n            } else {\n              throw new Error(&quot;Index out of bounds&quot;);\n            }\n          }\n        }\n      }\n      break;\n    }\n  };\n  var linear_traversal_nth__3 = function(coll, n, not_found) {\n    while (true) {\n      if (coll == null) {\n        return not_found;\n      } else {\n        if (n === 0) {\n          if (cljs.core.seq.call(null, coll)) {\n            return cljs.core.first.call(null, coll);\n          } else {\n            return not_found;\n          }\n        } else {\n          if (cljs.core.indexed_QMARK_.call(null, coll)) {\n            return cljs.core._nth.call(null, coll, n, not_found);\n          } else {\n            if (cljs.core.seq.call(null, coll)) {\n              var G__22223 = cljs.core.next.call(null, coll);\n              var G__22224 = n - 1;\n              var G__22225 = not_found;\n              coll = G__22223;\n              n = G__22224;\n              not_found = G__22225;\n              continue;\n            } else {\n              return not_found;\n            }\n          }\n        }\n      }\n      break;\n    }\n  };\n  linear_traversal_nth = function(coll, n, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return linear_traversal_nth__2.call(this, coll, n);\n      case 3:\n        return linear_traversal_nth__3.call(this, coll, n, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  linear_traversal_nth.cljs$core$IFn$_invoke$arity$2 = linear_traversal_nth__2;\n  linear_traversal_nth.cljs$core$IFn$_invoke$arity$3 = linear_traversal_nth__3;\n  return linear_traversal_nth;\n}();\ncljs.core.nth = function() {\n  var nth = null;\n  var nth__2 = function(coll, n) {\n    if (!(typeof n === &quot;number&quot;)) {\n      throw new Error(&quot;index argument to nth must be a number&quot;);\n    } else {\n      if (coll == null) {\n        return coll;\n      } else {\n        if (function() {\n          var G__22230 = coll;\n          if (G__22230) {\n            var bit__17788__auto__ = G__22230.cljs$lang$protocol_mask$partition0$ &amp; 16;\n            if (bit__17788__auto__ || G__22230.cljs$core$IIndexed$) {\n              return true;\n            } else {\n              return false;\n            }\n          } else {\n            return false;\n          }\n        }()) {\n          return cljs.core._nth.call(null, coll, n);\n        } else {\n          if (coll instanceof Array) {\n            if (n &lt; coll.length) {\n              return coll[n];\n            } else {\n              return null;\n            }\n          } else {\n            if (typeof coll === &quot;string&quot;) {\n              if (n &lt; coll.length) {\n                return coll[n];\n              } else {\n                return null;\n              }\n            } else {\n              if (cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IIndexed, coll)) {\n                return cljs.core._nth.call(null, coll, n);\n              } else {\n                if (function() {\n                  var G__22231 = coll;\n                  if (G__22231) {\n                    var bit__17795__auto__ = G__22231.cljs$lang$protocol_mask$partition0$ &amp; 64;\n                    if (bit__17795__auto__ || G__22231.cljs$core$ISeq$) {\n                      return true;\n                    } else {\n                      if (!G__22231.cljs$lang$protocol_mask$partition0$) {\n                        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISeq, G__22231);\n                      } else {\n                        return false;\n                      }\n                    }\n                  } else {\n                    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISeq, G__22231);\n                  }\n                }()) {\n                  return cljs.core.linear_traversal_nth.call(null, coll, n);\n                } else {\n                  throw new Error(&quot;nth not supported on this type &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.type__GT_str.call(null, cljs.core.type.call(null, coll))));\n                }\n              }\n            }\n          }\n        }\n      }\n    }\n  };\n  var nth__3 = function(coll, n, not_found) {\n    if (!(typeof n === &quot;number&quot;)) {\n      throw new Error(&quot;index argument to nth must be a number.&quot;);\n    } else {\n      if (coll == null) {\n        return not_found;\n      } else {\n        if (function() {\n          var G__22232 = coll;\n          if (G__22232) {\n            var bit__17788__auto__ = G__22232.cljs$lang$protocol_mask$partition0$ &amp; 16;\n            if (bit__17788__auto__ || G__22232.cljs$core$IIndexed$) {\n              return true;\n            } else {\n              return false;\n            }\n          } else {\n            return false;\n          }\n        }()) {\n          return cljs.core._nth.call(null, coll, n, not_found);\n        } else {\n          if (coll instanceof Array) {\n            if (n &lt; coll.length) {\n              return coll[n];\n            } else {\n              return not_found;\n            }\n          } else {\n            if (typeof coll === &quot;string&quot;) {\n              if (n &lt; coll.length) {\n                return coll[n];\n              } else {\n                return not_found;\n              }\n            } else {\n              if (cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IIndexed, coll)) {\n                return cljs.core._nth.call(null, coll, n);\n              } else {\n                if (function() {\n                  var G__22233 = coll;\n                  if (G__22233) {\n                    var bit__17795__auto__ = G__22233.cljs$lang$protocol_mask$partition0$ &amp; 64;\n                    if (bit__17795__auto__ || G__22233.cljs$core$ISeq$) {\n                      return true;\n                    } else {\n                      if (!G__22233.cljs$lang$protocol_mask$partition0$) {\n                        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISeq, G__22233);\n                      } else {\n                        return false;\n                      }\n                    }\n                  } else {\n                    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISeq, G__22233);\n                  }\n                }()) {\n                  return cljs.core.linear_traversal_nth.call(null, coll, n, not_found);\n                } else {\n                  throw new Error(&quot;nth not supported on this type &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.type__GT_str.call(null, cljs.core.type.call(null, coll))));\n                }\n              }\n            }\n          }\n        }\n      }\n    }\n  };\n  nth = function(coll, n, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return nth__2.call(this, coll, n);\n      case 3:\n        return nth__3.call(this, coll, n, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  nth.cljs$core$IFn$_invoke$arity$2 = nth__2;\n  nth.cljs$core$IFn$_invoke$arity$3 = nth__3;\n  return nth;\n}();\ncljs.core.get = function() {\n  var get = null;\n  var get__2 = function(o, k) {\n    if (o == null) {\n      return null;\n    } else {\n      if (function() {\n        var G__22236 = o;\n        if (G__22236) {\n          var bit__17788__auto__ = G__22236.cljs$lang$protocol_mask$partition0$ &amp; 256;\n          if (bit__17788__auto__ || G__22236.cljs$core$ILookup$) {\n            return true;\n          } else {\n            return false;\n          }\n        } else {\n          return false;\n        }\n      }()) {\n        return cljs.core._lookup.call(null, o, k);\n      } else {\n        if (o instanceof Array) {\n          if (k &lt; o.length) {\n            return o[k];\n          } else {\n            return null;\n          }\n        } else {\n          if (typeof o === &quot;string&quot;) {\n            if (k &lt; o.length) {\n              return o[k];\n            } else {\n              return null;\n            }\n          } else {\n            if (cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ILookup, o)) {\n              return cljs.core._lookup.call(null, o, k);\n            } else {\n              return null;\n            }\n          }\n        }\n      }\n    }\n  };\n  var get__3 = function(o, k, not_found) {\n    if (!(o == null)) {\n      if (function() {\n        var G__22237 = o;\n        if (G__22237) {\n          var bit__17788__auto__ = G__22237.cljs$lang$protocol_mask$partition0$ &amp; 256;\n          if (bit__17788__auto__ || G__22237.cljs$core$ILookup$) {\n            return true;\n          } else {\n            return false;\n          }\n        } else {\n          return false;\n        }\n      }()) {\n        return cljs.core._lookup.call(null, o, k, not_found);\n      } else {\n        if (o instanceof Array) {\n          if (k &lt; o.length) {\n            return o[k];\n          } else {\n            return not_found;\n          }\n        } else {\n          if (typeof o === &quot;string&quot;) {\n            if (k &lt; o.length) {\n              return o[k];\n            } else {\n              return not_found;\n            }\n          } else {\n            if (cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ILookup, o)) {\n              return cljs.core._lookup.call(null, o, k, not_found);\n            } else {\n              return not_found;\n            }\n          }\n        }\n      }\n    } else {\n      return not_found;\n    }\n  };\n  get = function(o, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return get__2.call(this, o, k);\n      case 3:\n        return get__3.call(this, o, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  get.cljs$core$IFn$_invoke$arity$2 = get__2;\n  get.cljs$core$IFn$_invoke$arity$3 = get__3;\n  return get;\n}();\ncljs.core.assoc = function() {\n  var assoc = null;\n  var assoc__3 = function(coll, k, v) {\n    if (!(coll == null)) {\n      return cljs.core._assoc.call(null, coll, k, v);\n    } else {\n      return cljs.core.PersistentHashMap.fromArrays([k], [v]);\n    }\n  };\n  var assoc__4 = function() {\n    var G__22238__delegate = function(coll, k, v, kvs) {\n      while (true) {\n        var ret = assoc.call(null, coll, k, v);\n        if (cljs.core.truth_(kvs)) {\n          var G__22239 = ret;\n          var G__22240 = cljs.core.first.call(null, kvs);\n          var G__22241 = cljs.core.second.call(null, kvs);\n          var G__22242 = cljs.core.nnext.call(null, kvs);\n          coll = G__22239;\n          k = G__22240;\n          v = G__22241;\n          kvs = G__22242;\n          continue;\n        } else {\n          return ret;\n        }\n        break;\n      }\n    };\n    var G__22238 = function(coll, k, v, var_args) {\n      var kvs = null;\n      if (arguments.length &gt; 3) {\n        kvs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n      }\n      return G__22238__delegate.call(this, coll, k, v, kvs);\n    };\n    G__22238.cljs$lang$maxFixedArity = 3;\n    G__22238.cljs$lang$applyTo = function(arglist__22243) {\n      var coll = cljs.core.first(arglist__22243);\n      arglist__22243 = cljs.core.next(arglist__22243);\n      var k = cljs.core.first(arglist__22243);\n      arglist__22243 = cljs.core.next(arglist__22243);\n      var v = cljs.core.first(arglist__22243);\n      var kvs = cljs.core.rest(arglist__22243);\n      return G__22238__delegate(coll, k, v, kvs);\n    };\n    G__22238.cljs$core$IFn$_invoke$arity$variadic = G__22238__delegate;\n    return G__22238;\n  }();\n  assoc = function(coll, k, v, var_args) {\n    var kvs = var_args;\n    switch(arguments.length) {\n      case 3:\n        return assoc__3.call(this, coll, k, v);\n      default:\n        return assoc__4.cljs$core$IFn$_invoke$arity$variadic(coll, k, v, cljs.core.array_seq(arguments, 3));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  assoc.cljs$lang$maxFixedArity = 3;\n  assoc.cljs$lang$applyTo = assoc__4.cljs$lang$applyTo;\n  assoc.cljs$core$IFn$_invoke$arity$3 = assoc__3;\n  assoc.cljs$core$IFn$_invoke$arity$variadic = assoc__4.cljs$core$IFn$_invoke$arity$variadic;\n  return assoc;\n}();\ncljs.core.dissoc = function() {\n  var dissoc = null;\n  var dissoc__1 = function(coll) {\n    return coll;\n  };\n  var dissoc__2 = function(coll, k) {\n    if (coll == null) {\n      return null;\n    } else {\n      return cljs.core._dissoc.call(null, coll, k);\n    }\n  };\n  var dissoc__3 = function() {\n    var G__22244__delegate = function(coll, k, ks) {\n      while (true) {\n        if (coll == null) {\n          return null;\n        } else {\n          var ret = dissoc.call(null, coll, k);\n          if (cljs.core.truth_(ks)) {\n            var G__22245 = ret;\n            var G__22246 = cljs.core.first.call(null, ks);\n            var G__22247 = cljs.core.next.call(null, ks);\n            coll = G__22245;\n            k = G__22246;\n            ks = G__22247;\n            continue;\n          } else {\n            return ret;\n          }\n        }\n        break;\n      }\n    };\n    var G__22244 = function(coll, k, var_args) {\n      var ks = null;\n      if (arguments.length &gt; 2) {\n        ks = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22244__delegate.call(this, coll, k, ks);\n    };\n    G__22244.cljs$lang$maxFixedArity = 2;\n    G__22244.cljs$lang$applyTo = function(arglist__22248) {\n      var coll = cljs.core.first(arglist__22248);\n      arglist__22248 = cljs.core.next(arglist__22248);\n      var k = cljs.core.first(arglist__22248);\n      var ks = cljs.core.rest(arglist__22248);\n      return G__22244__delegate(coll, k, ks);\n    };\n    G__22244.cljs$core$IFn$_invoke$arity$variadic = G__22244__delegate;\n    return G__22244;\n  }();\n  dissoc = function(coll, k, var_args) {\n    var ks = var_args;\n    switch(arguments.length) {\n      case 1:\n        return dissoc__1.call(this, coll);\n      case 2:\n        return dissoc__2.call(this, coll, k);\n      default:\n        return dissoc__3.cljs$core$IFn$_invoke$arity$variadic(coll, k, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  dissoc.cljs$lang$maxFixedArity = 2;\n  dissoc.cljs$lang$applyTo = dissoc__3.cljs$lang$applyTo;\n  dissoc.cljs$core$IFn$_invoke$arity$1 = dissoc__1;\n  dissoc.cljs$core$IFn$_invoke$arity$2 = dissoc__2;\n  dissoc.cljs$core$IFn$_invoke$arity$variadic = dissoc__3.cljs$core$IFn$_invoke$arity$variadic;\n  return dissoc;\n}();\ncljs.core.fn_QMARK_ = function fn_QMARK_(f) {\n  var or__17145__auto__ = goog.isFunction(f);\n  if (or__17145__auto__) {\n    return or__17145__auto__;\n  } else {\n    var G__22252 = f;\n    if (G__22252) {\n      var bit__17795__auto__ = null;\n      if (cljs.core.truth_(function() {\n        var or__17145__auto____$1 = bit__17795__auto__;\n        if (cljs.core.truth_(or__17145__auto____$1)) {\n          return or__17145__auto____$1;\n        } else {\n          return G__22252.cljs$core$Fn$;\n        }\n      }())) {\n        return true;\n      } else {\n        if (!G__22252.cljs$lang$protocol_mask$partition$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.Fn, G__22252);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.Fn, G__22252);\n    }\n  }\n};\ncljs.core.MetaFn = function(afn, meta) {\n  this.afn = afn;\n  this.meta = meta;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 393217;\n};\ncljs.core.MetaFn.cljs$lang$type = true;\ncljs.core.MetaFn.cljs$lang$ctorStr = &quot;cljs.core/MetaFn&quot;;\ncljs.core.MetaFn.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/MetaFn&quot;);\n};\ncljs.core.MetaFn.prototype.call = function() {\n  var G__22254 = null;\n  var G__22254__1 = function(self__) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null);\n  };\n  var G__22254__2 = function(self__, a) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a);\n  };\n  var G__22254__3 = function(self__, a, b) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b);\n  };\n  var G__22254__4 = function(self__, a, b, c) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c);\n  };\n  var G__22254__5 = function(self__, a, b, c, d) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d);\n  };\n  var G__22254__6 = function(self__, a, b, c, d, e) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e);\n  };\n  var G__22254__7 = function(self__, a, b, c, d, e, f) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f);\n  };\n  var G__22254__8 = function(self__, a, b, c, d, e, f, g) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g);\n  };\n  var G__22254__9 = function(self__, a, b, c, d, e, f, g, h) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h);\n  };\n  var G__22254__10 = function(self__, a, b, c, d, e, f, g, h, i) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i);\n  };\n  var G__22254__11 = function(self__, a, b, c, d, e, f, g, h, i, j) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j);\n  };\n  var G__22254__12 = function(self__, a, b, c, d, e, f, g, h, i, j, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k);\n  };\n  var G__22254__13 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l);\n  };\n  var G__22254__14 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);\n  };\n  var G__22254__15 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n  };\n  var G__22254__16 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);\n  };\n  var G__22254__17 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);\n  };\n  var G__22254__18 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);\n  };\n  var G__22254__19 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);\n  };\n  var G__22254__20 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);\n  };\n  var G__22254__21 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);\n  };\n  var G__22254__22 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {\n    var self__ = this;\n    var self____$1 = this;\n    var _ = self____$1;\n    return cljs.core.apply.call(null, self__.afn, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);\n  };\n  G__22254 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {\n    switch(arguments.length) {\n      case 1:\n        return G__22254__1.call(this, self__);\n      case 2:\n        return G__22254__2.call(this, self__, a);\n      case 3:\n        return G__22254__3.call(this, self__, a, b);\n      case 4:\n        return G__22254__4.call(this, self__, a, b, c);\n      case 5:\n        return G__22254__5.call(this, self__, a, b, c, d);\n      case 6:\n        return G__22254__6.call(this, self__, a, b, c, d, e);\n      case 7:\n        return G__22254__7.call(this, self__, a, b, c, d, e, f);\n      case 8:\n        return G__22254__8.call(this, self__, a, b, c, d, e, f, g);\n      case 9:\n        return G__22254__9.call(this, self__, a, b, c, d, e, f, g, h);\n      case 10:\n        return G__22254__10.call(this, self__, a, b, c, d, e, f, g, h, i);\n      case 11:\n        return G__22254__11.call(this, self__, a, b, c, d, e, f, g, h, i, j);\n      case 12:\n        return G__22254__12.call(this, self__, a, b, c, d, e, f, g, h, i, j, k);\n      case 13:\n        return G__22254__13.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l);\n      case 14:\n        return G__22254__14.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m);\n      case 15:\n        return G__22254__15.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n      case 16:\n        return G__22254__16.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);\n      case 17:\n        return G__22254__17.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);\n      case 18:\n        return G__22254__18.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);\n      case 19:\n        return G__22254__19.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);\n      case 20:\n        return G__22254__20.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);\n      case 21:\n        return G__22254__21.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);\n      case 22:\n        return G__22254__22.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22254.cljs$core$IFn$_invoke$arity$1 = G__22254__1;\n  G__22254.cljs$core$IFn$_invoke$arity$2 = G__22254__2;\n  G__22254.cljs$core$IFn$_invoke$arity$3 = G__22254__3;\n  G__22254.cljs$core$IFn$_invoke$arity$4 = G__22254__4;\n  G__22254.cljs$core$IFn$_invoke$arity$5 = G__22254__5;\n  G__22254.cljs$core$IFn$_invoke$arity$6 = G__22254__6;\n  G__22254.cljs$core$IFn$_invoke$arity$7 = G__22254__7;\n  G__22254.cljs$core$IFn$_invoke$arity$8 = G__22254__8;\n  G__22254.cljs$core$IFn$_invoke$arity$9 = G__22254__9;\n  G__22254.cljs$core$IFn$_invoke$arity$10 = G__22254__10;\n  G__22254.cljs$core$IFn$_invoke$arity$11 = G__22254__11;\n  G__22254.cljs$core$IFn$_invoke$arity$12 = G__22254__12;\n  G__22254.cljs$core$IFn$_invoke$arity$13 = G__22254__13;\n  G__22254.cljs$core$IFn$_invoke$arity$14 = G__22254__14;\n  G__22254.cljs$core$IFn$_invoke$arity$15 = G__22254__15;\n  G__22254.cljs$core$IFn$_invoke$arity$16 = G__22254__16;\n  G__22254.cljs$core$IFn$_invoke$arity$17 = G__22254__17;\n  G__22254.cljs$core$IFn$_invoke$arity$18 = G__22254__18;\n  G__22254.cljs$core$IFn$_invoke$arity$19 = G__22254__19;\n  G__22254.cljs$core$IFn$_invoke$arity$20 = G__22254__20;\n  G__22254.cljs$core$IFn$_invoke$arity$21 = G__22254__21;\n  G__22254.cljs$core$IFn$_invoke$arity$22 = G__22254__22;\n  return G__22254;\n}();\ncljs.core.MetaFn.prototype.apply = function(self__, args22253) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22253)));\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$0 = function() {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$1 = function(a) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$2 = function(a, b) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$3 = function(a, b, c) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$4 = function(a, b, c, d) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$5 = function(a, b, c, d, e) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$6 = function(a, b, c, d, e, f) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$7 = function(a, b, c, d, e, f, g) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$8 = function(a, b, c, d, e, f, g, h) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$9 = function(a, b, c, d, e, f, g, h, i) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$10 = function(a, b, c, d, e, f, g, h, i, j) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$11 = function(a, b, c, d, e, f, g, h, i, j, k) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$12 = function(a, b, c, d, e, f, g, h, i, j, k, l) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$13 = function(a, b, c, d, e, f, g, h, i, j, k, l, m) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$14 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$15 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$16 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$17 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$18 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$19 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$20 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) {\n  var self__ = this;\n  var _ = this;\n  return self__.afn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);\n};\ncljs.core.MetaFn.prototype.cljs$core$IFn$_invoke$arity$21 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {\n  var self__ = this;\n  var _ = this;\n  return cljs.core.apply.call(null, self__.afn, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);\n};\ncljs.core.MetaFn.prototype.cljs$core$Fn$ = true;\ncljs.core.MetaFn.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(_, new_meta) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.MetaFn(self__.afn, new_meta);\n};\ncljs.core.MetaFn.prototype.cljs$core$IMeta$_meta$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.meta;\n};\ncljs.core.__GT_MetaFn = function __GT_MetaFn(afn, meta) {\n  return new cljs.core.MetaFn(afn, meta);\n};\ncljs.core.with_meta = function with_meta(o, meta) {\n  if (cljs.core.fn_QMARK_.call(null, o) &amp;&amp; !function() {\n    var G__22258 = o;\n    if (G__22258) {\n      var bit__17795__auto__ = G__22258.cljs$lang$protocol_mask$partition0$ &amp; 262144;\n      if (bit__17795__auto__ || G__22258.cljs$core$IWithMeta$) {\n        return true;\n      } else {\n        if (!G__22258.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IWithMeta, G__22258);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IWithMeta, G__22258);\n    }\n  }()) {\n    return new cljs.core.MetaFn(o, meta);\n  } else {\n    if (o == null) {\n      return null;\n    } else {\n      return cljs.core._with_meta.call(null, o, meta);\n    }\n  }\n};\ncljs.core.meta = function meta(o) {\n  if (function() {\n    var and__17133__auto__ = !(o == null);\n    if (and__17133__auto__) {\n      var G__22262 = o;\n      if (G__22262) {\n        var bit__17795__auto__ = G__22262.cljs$lang$protocol_mask$partition0$ &amp; 131072;\n        if (bit__17795__auto__ || G__22262.cljs$core$IMeta$) {\n          return true;\n        } else {\n          if (!G__22262.cljs$lang$protocol_mask$partition0$) {\n            return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMeta, G__22262);\n          } else {\n            return false;\n          }\n        }\n      } else {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMeta, G__22262);\n      }\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return cljs.core._meta.call(null, o);\n  } else {\n    return null;\n  }\n};\ncljs.core.peek = function peek(coll) {\n  if (coll == null) {\n    return null;\n  } else {\n    return cljs.core._peek.call(null, coll);\n  }\n};\ncljs.core.pop = function pop(coll) {\n  if (coll == null) {\n    return null;\n  } else {\n    return cljs.core._pop.call(null, coll);\n  }\n};\ncljs.core.disj = function() {\n  var disj = null;\n  var disj__1 = function(coll) {\n    return coll;\n  };\n  var disj__2 = function(coll, k) {\n    if (coll == null) {\n      return null;\n    } else {\n      return cljs.core._disjoin.call(null, coll, k);\n    }\n  };\n  var disj__3 = function() {\n    var G__22263__delegate = function(coll, k, ks) {\n      while (true) {\n        if (coll == null) {\n          return null;\n        } else {\n          var ret = disj.call(null, coll, k);\n          if (cljs.core.truth_(ks)) {\n            var G__22264 = ret;\n            var G__22265 = cljs.core.first.call(null, ks);\n            var G__22266 = cljs.core.next.call(null, ks);\n            coll = G__22264;\n            k = G__22265;\n            ks = G__22266;\n            continue;\n          } else {\n            return ret;\n          }\n        }\n        break;\n      }\n    };\n    var G__22263 = function(coll, k, var_args) {\n      var ks = null;\n      if (arguments.length &gt; 2) {\n        ks = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22263__delegate.call(this, coll, k, ks);\n    };\n    G__22263.cljs$lang$maxFixedArity = 2;\n    G__22263.cljs$lang$applyTo = function(arglist__22267) {\n      var coll = cljs.core.first(arglist__22267);\n      arglist__22267 = cljs.core.next(arglist__22267);\n      var k = cljs.core.first(arglist__22267);\n      var ks = cljs.core.rest(arglist__22267);\n      return G__22263__delegate(coll, k, ks);\n    };\n    G__22263.cljs$core$IFn$_invoke$arity$variadic = G__22263__delegate;\n    return G__22263;\n  }();\n  disj = function(coll, k, var_args) {\n    var ks = var_args;\n    switch(arguments.length) {\n      case 1:\n        return disj__1.call(this, coll);\n      case 2:\n        return disj__2.call(this, coll, k);\n      default:\n        return disj__3.cljs$core$IFn$_invoke$arity$variadic(coll, k, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  disj.cljs$lang$maxFixedArity = 2;\n  disj.cljs$lang$applyTo = disj__3.cljs$lang$applyTo;\n  disj.cljs$core$IFn$_invoke$arity$1 = disj__1;\n  disj.cljs$core$IFn$_invoke$arity$2 = disj__2;\n  disj.cljs$core$IFn$_invoke$arity$variadic = disj__3.cljs$core$IFn$_invoke$arity$variadic;\n  return disj;\n}();\ncljs.core.empty_QMARK_ = function empty_QMARK_(coll) {\n  return coll == null || cljs.core.not.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.coll_QMARK_ = function coll_QMARK_(x) {\n  if (x == null) {\n    return false;\n  } else {\n    var G__22269 = x;\n    if (G__22269) {\n      var bit__17795__auto__ = G__22269.cljs$lang$protocol_mask$partition0$ &amp; 8;\n      if (bit__17795__auto__ || G__22269.cljs$core$ICollection$) {\n        return true;\n      } else {\n        if (!G__22269.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ICollection, G__22269);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ICollection, G__22269);\n    }\n  }\n};\ncljs.core.set_QMARK_ = function set_QMARK_(x) {\n  if (x == null) {\n    return false;\n  } else {\n    var G__22271 = x;\n    if (G__22271) {\n      var bit__17795__auto__ = G__22271.cljs$lang$protocol_mask$partition0$ &amp; 4096;\n      if (bit__17795__auto__ || G__22271.cljs$core$ISet$) {\n        return true;\n      } else {\n        if (!G__22271.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISet, G__22271);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISet, G__22271);\n    }\n  }\n};\ncljs.core.associative_QMARK_ = function associative_QMARK_(x) {\n  var G__22273 = x;\n  if (G__22273) {\n    var bit__17795__auto__ = G__22273.cljs$lang$protocol_mask$partition0$ &amp; 512;\n    if (bit__17795__auto__ || G__22273.cljs$core$IAssociative$) {\n      return true;\n    } else {\n      if (!G__22273.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IAssociative, G__22273);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IAssociative, G__22273);\n  }\n};\ncljs.core.sequential_QMARK_ = function sequential_QMARK_(x) {\n  var G__22275 = x;\n  if (G__22275) {\n    var bit__17795__auto__ = G__22275.cljs$lang$protocol_mask$partition0$ &amp; 16777216;\n    if (bit__17795__auto__ || G__22275.cljs$core$ISequential$) {\n      return true;\n    } else {\n      if (!G__22275.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISequential, G__22275);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISequential, G__22275);\n  }\n};\ncljs.core.sorted_QMARK_ = function sorted_QMARK_(x) {\n  var G__22277 = x;\n  if (G__22277) {\n    var bit__17795__auto__ = G__22277.cljs$lang$protocol_mask$partition0$ &amp; 268435456;\n    if (bit__17795__auto__ || G__22277.cljs$core$ISorted$) {\n      return true;\n    } else {\n      if (!G__22277.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISorted, G__22277);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISorted, G__22277);\n  }\n};\ncljs.core.reduceable_QMARK_ = function reduceable_QMARK_(x) {\n  var G__22279 = x;\n  if (G__22279) {\n    var bit__17795__auto__ = G__22279.cljs$lang$protocol_mask$partition0$ &amp; 524288;\n    if (bit__17795__auto__ || G__22279.cljs$core$IReduce$) {\n      return true;\n    } else {\n      if (!G__22279.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IReduce, G__22279);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IReduce, G__22279);\n  }\n};\ncljs.core.map_QMARK_ = function map_QMARK_(x) {\n  if (x == null) {\n    return false;\n  } else {\n    var G__22281 = x;\n    if (G__22281) {\n      var bit__17795__auto__ = G__22281.cljs$lang$protocol_mask$partition0$ &amp; 1024;\n      if (bit__17795__auto__ || G__22281.cljs$core$IMap$) {\n        return true;\n      } else {\n        if (!G__22281.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMap, G__22281);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMap, G__22281);\n    }\n  }\n};\ncljs.core.vector_QMARK_ = function vector_QMARK_(x) {\n  var G__22283 = x;\n  if (G__22283) {\n    var bit__17795__auto__ = G__22283.cljs$lang$protocol_mask$partition0$ &amp; 16384;\n    if (bit__17795__auto__ || G__22283.cljs$core$IVector$) {\n      return true;\n    } else {\n      if (!G__22283.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IVector, G__22283);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IVector, G__22283);\n  }\n};\ncljs.core.chunked_seq_QMARK_ = function chunked_seq_QMARK_(x) {\n  var G__22285 = x;\n  if (G__22285) {\n    var bit__17788__auto__ = G__22285.cljs$lang$protocol_mask$partition1$ &amp; 512;\n    if (bit__17788__auto__ || G__22285.cljs$core$IChunkedSeq$) {\n      return true;\n    } else {\n      return false;\n    }\n  } else {\n    return false;\n  }\n};\ncljs.core.js_obj = function() {\n  var js_obj = null;\n  var js_obj__0 = function() {\n    var obj22289 = {};\n    return obj22289;\n  };\n  var js_obj__1 = function() {\n    var G__22290__delegate = function(keyvals) {\n      return cljs.core.apply.call(null, goog.object.create, keyvals);\n    };\n    var G__22290 = function(var_args) {\n      var keyvals = null;\n      if (arguments.length &gt; 0) {\n        keyvals = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n      }\n      return G__22290__delegate.call(this, keyvals);\n    };\n    G__22290.cljs$lang$maxFixedArity = 0;\n    G__22290.cljs$lang$applyTo = function(arglist__22291) {\n      var keyvals = cljs.core.seq(arglist__22291);\n      return G__22290__delegate(keyvals);\n    };\n    G__22290.cljs$core$IFn$_invoke$arity$variadic = G__22290__delegate;\n    return G__22290;\n  }();\n  js_obj = function(var_args) {\n    var keyvals = var_args;\n    switch(arguments.length) {\n      case 0:\n        return js_obj__0.call(this);\n      default:\n        return js_obj__1.cljs$core$IFn$_invoke$arity$variadic(cljs.core.array_seq(arguments, 0));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  js_obj.cljs$lang$maxFixedArity = 0;\n  js_obj.cljs$lang$applyTo = js_obj__1.cljs$lang$applyTo;\n  js_obj.cljs$core$IFn$_invoke$arity$0 = js_obj__0;\n  js_obj.cljs$core$IFn$_invoke$arity$variadic = js_obj__1.cljs$core$IFn$_invoke$arity$variadic;\n  return js_obj;\n}();\ncljs.core.js_keys = function js_keys(obj) {\n  var keys = [];\n  goog.object.forEach(obj, function(keys) {\n    return function(val, key, obj__$1) {\n      return keys.push(key);\n    };\n  }(keys));\n  return keys;\n};\ncljs.core.js_delete = function js_delete(obj, key) {\n  return delete obj[key];\n};\ncljs.core.array_copy = function array_copy(from, i, to, j, len) {\n  var i__$1 = i;\n  var j__$1 = j;\n  var len__$1 = len;\n  while (true) {\n    if (len__$1 === 0) {\n      return to;\n    } else {\n      to[j__$1] = from[i__$1];\n      var G__22292 = i__$1 + 1;\n      var G__22293 = j__$1 + 1;\n      var G__22294 = len__$1 - 1;\n      i__$1 = G__22292;\n      j__$1 = G__22293;\n      len__$1 = G__22294;\n      continue;\n    }\n    break;\n  }\n};\ncljs.core.array_copy_downward = function array_copy_downward(from, i, to, j, len) {\n  var i__$1 = i + (len - 1);\n  var j__$1 = j + (len - 1);\n  var len__$1 = len;\n  while (true) {\n    if (len__$1 === 0) {\n      return to;\n    } else {\n      to[j__$1] = from[i__$1];\n      var G__22295 = i__$1 - 1;\n      var G__22296 = j__$1 - 1;\n      var G__22297 = len__$1 - 1;\n      i__$1 = G__22295;\n      j__$1 = G__22296;\n      len__$1 = G__22297;\n      continue;\n    }\n    break;\n  }\n};\ncljs.core.lookup_sentinel = function() {\n  var obj22299 = {};\n  return obj22299;\n}();\ncljs.core.false_QMARK_ = function false_QMARK_(x) {\n  return x === false;\n};\ncljs.core.true_QMARK_ = function true_QMARK_(x) {\n  return x === true;\n};\ncljs.core.undefined_QMARK_ = function undefined_QMARK_(x) {\n  return void 0 === x;\n};\ncljs.core.seq_QMARK_ = function seq_QMARK_(s) {\n  if (s == null) {\n    return false;\n  } else {\n    var G__22301 = s;\n    if (G__22301) {\n      var bit__17795__auto__ = G__22301.cljs$lang$protocol_mask$partition0$ &amp; 64;\n      if (bit__17795__auto__ || G__22301.cljs$core$ISeq$) {\n        return true;\n      } else {\n        if (!G__22301.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISeq, G__22301);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISeq, G__22301);\n    }\n  }\n};\ncljs.core.seqable_QMARK_ = function seqable_QMARK_(s) {\n  var G__22303 = s;\n  if (G__22303) {\n    var bit__17795__auto__ = G__22303.cljs$lang$protocol_mask$partition0$ &amp; 8388608;\n    if (bit__17795__auto__ || G__22303.cljs$core$ISeqable$) {\n      return true;\n    } else {\n      if (!G__22303.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISeqable, G__22303);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ISeqable, G__22303);\n  }\n};\ncljs.core.boolean$ = function boolean$(x) {\n  if (cljs.core.truth_(x)) {\n    return true;\n  } else {\n    return false;\n  }\n};\ncljs.core.ifn_QMARK_ = function ifn_QMARK_(f) {\n  var or__17145__auto__ = cljs.core.fn_QMARK_.call(null, f);\n  if (or__17145__auto__) {\n    return or__17145__auto__;\n  } else {\n    var G__22307 = f;\n    if (G__22307) {\n      var bit__17795__auto__ = G__22307.cljs$lang$protocol_mask$partition0$ &amp; 1;\n      if (bit__17795__auto__ || G__22307.cljs$core$IFn$) {\n        return true;\n      } else {\n        if (!G__22307.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IFn, G__22307);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IFn, G__22307);\n    }\n  }\n};\ncljs.core.integer_QMARK_ = function integer_QMARK_(n) {\n  return typeof n === &quot;number&quot; &amp;&amp; !isNaN(n) &amp;&amp; !(n === Infinity) &amp;&amp; parseFloat(n) === parseInt(n, 10);\n};\ncljs.core.contains_QMARK_ = function contains_QMARK_(coll, v) {\n  if (cljs.core.get.call(null, coll, v, cljs.core.lookup_sentinel) === cljs.core.lookup_sentinel) {\n    return false;\n  } else {\n    return true;\n  }\n};\ncljs.core.find = function find(coll, k) {\n  if (!(coll == null) &amp;&amp; cljs.core.associative_QMARK_.call(null, coll) &amp;&amp; cljs.core.contains_QMARK_.call(null, coll, k)) {\n    return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [k, cljs.core.get.call(null, coll, k)], null);\n  } else {\n    return null;\n  }\n};\ncljs.core.distinct_QMARK_ = function() {\n  var distinct_QMARK_ = null;\n  var distinct_QMARK___1 = function(x) {\n    return true;\n  };\n  var distinct_QMARK___2 = function(x, y) {\n    return!cljs.core._EQ_.call(null, x, y);\n  };\n  var distinct_QMARK___3 = function() {\n    var G__22308__delegate = function(x, y, more) {\n      if (!cljs.core._EQ_.call(null, x, y)) {\n        var s = cljs.core.PersistentHashSet.fromArray([x, y], true);\n        var xs = more;\n        while (true) {\n          var x__$1 = cljs.core.first.call(null, xs);\n          var etc = cljs.core.next.call(null, xs);\n          if (cljs.core.truth_(xs)) {\n            if (cljs.core.contains_QMARK_.call(null, s, x__$1)) {\n              return false;\n            } else {\n              var G__22309 = cljs.core.conj.call(null, s, x__$1);\n              var G__22310 = etc;\n              s = G__22309;\n              xs = G__22310;\n              continue;\n            }\n          } else {\n            return true;\n          }\n          break;\n        }\n      } else {\n        return false;\n      }\n    };\n    var G__22308 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22308__delegate.call(this, x, y, more);\n    };\n    G__22308.cljs$lang$maxFixedArity = 2;\n    G__22308.cljs$lang$applyTo = function(arglist__22311) {\n      var x = cljs.core.first(arglist__22311);\n      arglist__22311 = cljs.core.next(arglist__22311);\n      var y = cljs.core.first(arglist__22311);\n      var more = cljs.core.rest(arglist__22311);\n      return G__22308__delegate(x, y, more);\n    };\n    G__22308.cljs$core$IFn$_invoke$arity$variadic = G__22308__delegate;\n    return G__22308;\n  }();\n  distinct_QMARK_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return distinct_QMARK___1.call(this, x);\n      case 2:\n        return distinct_QMARK___2.call(this, x, y);\n      default:\n        return distinct_QMARK___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  distinct_QMARK_.cljs$lang$maxFixedArity = 2;\n  distinct_QMARK_.cljs$lang$applyTo = distinct_QMARK___3.cljs$lang$applyTo;\n  distinct_QMARK_.cljs$core$IFn$_invoke$arity$1 = distinct_QMARK___1;\n  distinct_QMARK_.cljs$core$IFn$_invoke$arity$2 = distinct_QMARK___2;\n  distinct_QMARK_.cljs$core$IFn$_invoke$arity$variadic = distinct_QMARK___3.cljs$core$IFn$_invoke$arity$variadic;\n  return distinct_QMARK_;\n}();\ncljs.core.sequence = function sequence(coll) {\n  if (cljs.core.seq_QMARK_.call(null, coll)) {\n    return coll;\n  } else {\n    var or__17145__auto__ = cljs.core.seq.call(null, coll);\n    if (or__17145__auto__) {\n      return or__17145__auto__;\n    } else {\n      return cljs.core.List.EMPTY;\n    }\n  }\n};\ncljs.core.compare = function compare(x, y) {\n  if (x === y) {\n    return 0;\n  } else {\n    if (x == null) {\n      return-1;\n    } else {\n      if (y == null) {\n        return 1;\n      } else {\n        if (cljs.core.type.call(null, x) === cljs.core.type.call(null, y)) {\n          if (function() {\n            var G__22313 = x;\n            if (G__22313) {\n              var bit__17788__auto__ = G__22313.cljs$lang$protocol_mask$partition1$ &amp; 2048;\n              if (bit__17788__auto__ || G__22313.cljs$core$IComparable$) {\n                return true;\n              } else {\n                return false;\n              }\n            } else {\n              return false;\n            }\n          }()) {\n            return cljs.core._compare.call(null, x, y);\n          } else {\n            return goog.array.defaultCompare(x, y);\n          }\n        } else {\n          throw new Error(&quot;compare on non-nil objects of different types&quot;);\n        }\n      }\n    }\n  }\n};\ncljs.core.compare_indexed = function() {\n  var compare_indexed = null;\n  var compare_indexed__2 = function(xs, ys) {\n    var xl = cljs.core.count.call(null, xs);\n    var yl = cljs.core.count.call(null, ys);\n    if (xl &lt; yl) {\n      return-1;\n    } else {\n      if (xl &gt; yl) {\n        return 1;\n      } else {\n        return compare_indexed.call(null, xs, ys, xl, 0);\n      }\n    }\n  };\n  var compare_indexed__4 = function(xs, ys, len, n) {\n    while (true) {\n      var d = cljs.core.compare.call(null, cljs.core.nth.call(null, xs, n), cljs.core.nth.call(null, ys, n));\n      if (d === 0 &amp;&amp; n + 1 &lt; len) {\n        var G__22314 = xs;\n        var G__22315 = ys;\n        var G__22316 = len;\n        var G__22317 = n + 1;\n        xs = G__22314;\n        ys = G__22315;\n        len = G__22316;\n        n = G__22317;\n        continue;\n      } else {\n        return d;\n      }\n      break;\n    }\n  };\n  compare_indexed = function(xs, ys, len, n) {\n    switch(arguments.length) {\n      case 2:\n        return compare_indexed__2.call(this, xs, ys);\n      case 4:\n        return compare_indexed__4.call(this, xs, ys, len, n);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  compare_indexed.cljs$core$IFn$_invoke$arity$2 = compare_indexed__2;\n  compare_indexed.cljs$core$IFn$_invoke$arity$4 = compare_indexed__4;\n  return compare_indexed;\n}();\ncljs.core.fn__GT_comparator = function fn__GT_comparator(f) {\n  if (cljs.core._EQ_.call(null, f, cljs.core.compare)) {\n    return cljs.core.compare;\n  } else {\n    return function(x, y) {\n      var r = f.call(null, x, y);\n      if (typeof r === &quot;number&quot;) {\n        return r;\n      } else {\n        if (cljs.core.truth_(r)) {\n          return-1;\n        } else {\n          if (cljs.core.truth_(f.call(null, y, x))) {\n            return 1;\n          } else {\n            return 0;\n          }\n        }\n      }\n    };\n  }\n};\ncljs.core.sort = function() {\n  var sort = null;\n  var sort__1 = function(coll) {\n    return sort.call(null, cljs.core.compare, coll);\n  };\n  var sort__2 = function(comp, coll) {\n    if (cljs.core.seq.call(null, coll)) {\n      var a = cljs.core.to_array.call(null, coll);\n      goog.array.stableSort(a, cljs.core.fn__GT_comparator.call(null, comp));\n      return cljs.core.seq.call(null, a);\n    } else {\n      return cljs.core.List.EMPTY;\n    }\n  };\n  sort = function(comp, coll) {\n    switch(arguments.length) {\n      case 1:\n        return sort__1.call(this, comp);\n      case 2:\n        return sort__2.call(this, comp, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  sort.cljs$core$IFn$_invoke$arity$1 = sort__1;\n  sort.cljs$core$IFn$_invoke$arity$2 = sort__2;\n  return sort;\n}();\ncljs.core.sort_by = function() {\n  var sort_by = null;\n  var sort_by__2 = function(keyfn, coll) {\n    return sort_by.call(null, keyfn, cljs.core.compare, coll);\n  };\n  var sort_by__3 = function(keyfn, comp, coll) {\n    return cljs.core.sort.call(null, function(x, y) {\n      return cljs.core.fn__GT_comparator.call(null, comp).call(null, keyfn.call(null, x), keyfn.call(null, y));\n    }, coll);\n  };\n  sort_by = function(keyfn, comp, coll) {\n    switch(arguments.length) {\n      case 2:\n        return sort_by__2.call(this, keyfn, comp);\n      case 3:\n        return sort_by__3.call(this, keyfn, comp, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  sort_by.cljs$core$IFn$_invoke$arity$2 = sort_by__2;\n  sort_by.cljs$core$IFn$_invoke$arity$3 = sort_by__3;\n  return sort_by;\n}();\ncljs.core.seq_reduce = function() {\n  var seq_reduce = null;\n  var seq_reduce__2 = function(f, coll) {\n    var temp__4124__auto__ = cljs.core.seq.call(null, coll);\n    if (temp__4124__auto__) {\n      var s = temp__4124__auto__;\n      return cljs.core.reduce.call(null, f, cljs.core.first.call(null, s), cljs.core.next.call(null, s));\n    } else {\n      return f.call(null);\n    }\n  };\n  var seq_reduce__3 = function(f, val, coll) {\n    var val__$1 = val;\n    var coll__$1 = cljs.core.seq.call(null, coll);\n    while (true) {\n      if (coll__$1) {\n        var nval = f.call(null, val__$1, cljs.core.first.call(null, coll__$1));\n        if (cljs.core.reduced_QMARK_.call(null, nval)) {\n          return cljs.core.deref.call(null, nval);\n        } else {\n          var G__22318 = nval;\n          var G__22319 = cljs.core.next.call(null, coll__$1);\n          val__$1 = G__22318;\n          coll__$1 = G__22319;\n          continue;\n        }\n      } else {\n        return val__$1;\n      }\n      break;\n    }\n  };\n  seq_reduce = function(f, val, coll) {\n    switch(arguments.length) {\n      case 2:\n        return seq_reduce__2.call(this, f, val);\n      case 3:\n        return seq_reduce__3.call(this, f, val, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  seq_reduce.cljs$core$IFn$_invoke$arity$2 = seq_reduce__2;\n  seq_reduce.cljs$core$IFn$_invoke$arity$3 = seq_reduce__3;\n  return seq_reduce;\n}();\ncljs.core.shuffle = function shuffle(coll) {\n  var a = cljs.core.to_array.call(null, coll);\n  goog.array.shuffle(a);\n  return cljs.core.vec.call(null, a);\n};\ncljs.core.reduce = function() {\n  var reduce = null;\n  var reduce__2 = function(f, coll) {\n    if (function() {\n      var G__22322 = coll;\n      if (G__22322) {\n        var bit__17788__auto__ = G__22322.cljs$lang$protocol_mask$partition0$ &amp; 524288;\n        if (bit__17788__auto__ || G__22322.cljs$core$IReduce$) {\n          return true;\n        } else {\n          return false;\n        }\n      } else {\n        return false;\n      }\n    }()) {\n      return cljs.core._reduce.call(null, coll, f);\n    } else {\n      if (coll instanceof Array) {\n        return cljs.core.array_reduce.call(null, coll, f);\n      } else {\n        if (typeof coll === &quot;string&quot;) {\n          return cljs.core.array_reduce.call(null, coll, f);\n        } else {\n          if (cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IReduce, coll)) {\n            return cljs.core._reduce.call(null, coll, f);\n          } else {\n            return cljs.core.seq_reduce.call(null, f, coll);\n          }\n        }\n      }\n    }\n  };\n  var reduce__3 = function(f, val, coll) {\n    if (function() {\n      var G__22323 = coll;\n      if (G__22323) {\n        var bit__17788__auto__ = G__22323.cljs$lang$protocol_mask$partition0$ &amp; 524288;\n        if (bit__17788__auto__ || G__22323.cljs$core$IReduce$) {\n          return true;\n        } else {\n          return false;\n        }\n      } else {\n        return false;\n      }\n    }()) {\n      return cljs.core._reduce.call(null, coll, f, val);\n    } else {\n      if (coll instanceof Array) {\n        return cljs.core.array_reduce.call(null, coll, f, val);\n      } else {\n        if (typeof coll === &quot;string&quot;) {\n          return cljs.core.array_reduce.call(null, coll, f, val);\n        } else {\n          if (cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IReduce, coll)) {\n            return cljs.core._reduce.call(null, coll, f, val);\n          } else {\n            return cljs.core.seq_reduce.call(null, f, val, coll);\n          }\n        }\n      }\n    }\n  };\n  reduce = function(f, val, coll) {\n    switch(arguments.length) {\n      case 2:\n        return reduce__2.call(this, f, val);\n      case 3:\n        return reduce__3.call(this, f, val, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  reduce.cljs$core$IFn$_invoke$arity$2 = reduce__2;\n  reduce.cljs$core$IFn$_invoke$arity$3 = reduce__3;\n  return reduce;\n}();\ncljs.core.reduce_kv = function reduce_kv(f, init, coll) {\n  if (!(coll == null)) {\n    return cljs.core._kv_reduce.call(null, coll, f, init);\n  } else {\n    return init;\n  }\n};\ncljs.core.completing = function completing(f) {\n  return function() {\n    var G__22324 = null;\n    var G__22324__0 = function() {\n      return f.call(null);\n    };\n    var G__22324__1 = function(x) {\n      return x;\n    };\n    var G__22324__2 = function(x, y) {\n      return f.call(null, x, y);\n    };\n    G__22324 = function(x, y) {\n      switch(arguments.length) {\n        case 0:\n          return G__22324__0.call(this);\n        case 1:\n          return G__22324__1.call(this, x);\n        case 2:\n          return G__22324__2.call(this, x, y);\n      }\n      throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n    };\n    G__22324.cljs$core$IFn$_invoke$arity$0 = G__22324__0;\n    G__22324.cljs$core$IFn$_invoke$arity$1 = G__22324__1;\n    G__22324.cljs$core$IFn$_invoke$arity$2 = G__22324__2;\n    return G__22324;\n  }();\n};\ncljs.core.transduce = function() {\n  var transduce = null;\n  var transduce__3 = function(xform, f, coll) {\n    return transduce.call(null, xform, f, f.call(null), coll);\n  };\n  var transduce__4 = function(xform, f, init, coll) {\n    var f__$1 = xform.call(null, cljs.core.completing.call(null, f));\n    var ret = cljs.core.reduce.call(null, f__$1, init, coll);\n    var ret__$1 = f__$1.call(null, cljs.core.reduced_QMARK_.call(null, ret) ? cljs.core.deref.call(null, ret) : ret);\n    if (cljs.core.reduced_QMARK_.call(null, ret__$1)) {\n      return cljs.core.deref.call(null, ret__$1);\n    } else {\n      return ret__$1;\n    }\n  };\n  transduce = function(xform, f, init, coll) {\n    switch(arguments.length) {\n      case 3:\n        return transduce__3.call(this, xform, f, init);\n      case 4:\n        return transduce__4.call(this, xform, f, init, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  transduce.cljs$core$IFn$_invoke$arity$3 = transduce__3;\n  transduce.cljs$core$IFn$_invoke$arity$4 = transduce__4;\n  return transduce;\n}();\ncljs.core._PLUS_ = function() {\n  var _PLUS_ = null;\n  var _PLUS___0 = function() {\n    return 0;\n  };\n  var _PLUS___1 = function(x) {\n    return x;\n  };\n  var _PLUS___2 = function(x, y) {\n    return x + y;\n  };\n  var _PLUS___3 = function() {\n    var G__22325__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, _PLUS_, x + y, more);\n    };\n    var G__22325 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22325__delegate.call(this, x, y, more);\n    };\n    G__22325.cljs$lang$maxFixedArity = 2;\n    G__22325.cljs$lang$applyTo = function(arglist__22326) {\n      var x = cljs.core.first(arglist__22326);\n      arglist__22326 = cljs.core.next(arglist__22326);\n      var y = cljs.core.first(arglist__22326);\n      var more = cljs.core.rest(arglist__22326);\n      return G__22325__delegate(x, y, more);\n    };\n    G__22325.cljs$core$IFn$_invoke$arity$variadic = G__22325__delegate;\n    return G__22325;\n  }();\n  _PLUS_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 0:\n        return _PLUS___0.call(this);\n      case 1:\n        return _PLUS___1.call(this, x);\n      case 2:\n        return _PLUS___2.call(this, x, y);\n      default:\n        return _PLUS___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _PLUS_.cljs$lang$maxFixedArity = 2;\n  _PLUS_.cljs$lang$applyTo = _PLUS___3.cljs$lang$applyTo;\n  _PLUS_.cljs$core$IFn$_invoke$arity$0 = _PLUS___0;\n  _PLUS_.cljs$core$IFn$_invoke$arity$1 = _PLUS___1;\n  _PLUS_.cljs$core$IFn$_invoke$arity$2 = _PLUS___2;\n  _PLUS_.cljs$core$IFn$_invoke$arity$variadic = _PLUS___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _PLUS_;\n}();\ncljs.core._ = function() {\n  var _ = null;\n  var ___1 = function(x) {\n    return-x;\n  };\n  var ___2 = function(x, y) {\n    return x - y;\n  };\n  var ___3 = function() {\n    var G__22327__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, _, x - y, more);\n    };\n    var G__22327 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22327__delegate.call(this, x, y, more);\n    };\n    G__22327.cljs$lang$maxFixedArity = 2;\n    G__22327.cljs$lang$applyTo = function(arglist__22328) {\n      var x = cljs.core.first(arglist__22328);\n      arglist__22328 = cljs.core.next(arglist__22328);\n      var y = cljs.core.first(arglist__22328);\n      var more = cljs.core.rest(arglist__22328);\n      return G__22327__delegate(x, y, more);\n    };\n    G__22327.cljs$core$IFn$_invoke$arity$variadic = G__22327__delegate;\n    return G__22327;\n  }();\n  _ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return ___1.call(this, x);\n      case 2:\n        return ___2.call(this, x, y);\n      default:\n        return ___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _.cljs$lang$maxFixedArity = 2;\n  _.cljs$lang$applyTo = ___3.cljs$lang$applyTo;\n  _.cljs$core$IFn$_invoke$arity$1 = ___1;\n  _.cljs$core$IFn$_invoke$arity$2 = ___2;\n  _.cljs$core$IFn$_invoke$arity$variadic = ___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _;\n}();\ncljs.core._STAR_ = function() {\n  var _STAR_ = null;\n  var _STAR___0 = function() {\n    return 1;\n  };\n  var _STAR___1 = function(x) {\n    return x;\n  };\n  var _STAR___2 = function(x, y) {\n    return x * y;\n  };\n  var _STAR___3 = function() {\n    var G__22329__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, _STAR_, x * y, more);\n    };\n    var G__22329 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22329__delegate.call(this, x, y, more);\n    };\n    G__22329.cljs$lang$maxFixedArity = 2;\n    G__22329.cljs$lang$applyTo = function(arglist__22330) {\n      var x = cljs.core.first(arglist__22330);\n      arglist__22330 = cljs.core.next(arglist__22330);\n      var y = cljs.core.first(arglist__22330);\n      var more = cljs.core.rest(arglist__22330);\n      return G__22329__delegate(x, y, more);\n    };\n    G__22329.cljs$core$IFn$_invoke$arity$variadic = G__22329__delegate;\n    return G__22329;\n  }();\n  _STAR_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 0:\n        return _STAR___0.call(this);\n      case 1:\n        return _STAR___1.call(this, x);\n      case 2:\n        return _STAR___2.call(this, x, y);\n      default:\n        return _STAR___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _STAR_.cljs$lang$maxFixedArity = 2;\n  _STAR_.cljs$lang$applyTo = _STAR___3.cljs$lang$applyTo;\n  _STAR_.cljs$core$IFn$_invoke$arity$0 = _STAR___0;\n  _STAR_.cljs$core$IFn$_invoke$arity$1 = _STAR___1;\n  _STAR_.cljs$core$IFn$_invoke$arity$2 = _STAR___2;\n  _STAR_.cljs$core$IFn$_invoke$arity$variadic = _STAR___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _STAR_;\n}();\ncljs.core._SLASH_ = function() {\n  var _SLASH_ = null;\n  var _SLASH___1 = function(x) {\n    return _SLASH_.call(null, 1, x);\n  };\n  var _SLASH___2 = function(x, y) {\n    return x / y;\n  };\n  var _SLASH___3 = function() {\n    var G__22331__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, _SLASH_, _SLASH_.call(null, x, y), more);\n    };\n    var G__22331 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22331__delegate.call(this, x, y, more);\n    };\n    G__22331.cljs$lang$maxFixedArity = 2;\n    G__22331.cljs$lang$applyTo = function(arglist__22332) {\n      var x = cljs.core.first(arglist__22332);\n      arglist__22332 = cljs.core.next(arglist__22332);\n      var y = cljs.core.first(arglist__22332);\n      var more = cljs.core.rest(arglist__22332);\n      return G__22331__delegate(x, y, more);\n    };\n    G__22331.cljs$core$IFn$_invoke$arity$variadic = G__22331__delegate;\n    return G__22331;\n  }();\n  _SLASH_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return _SLASH___1.call(this, x);\n      case 2:\n        return _SLASH___2.call(this, x, y);\n      default:\n        return _SLASH___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _SLASH_.cljs$lang$maxFixedArity = 2;\n  _SLASH_.cljs$lang$applyTo = _SLASH___3.cljs$lang$applyTo;\n  _SLASH_.cljs$core$IFn$_invoke$arity$1 = _SLASH___1;\n  _SLASH_.cljs$core$IFn$_invoke$arity$2 = _SLASH___2;\n  _SLASH_.cljs$core$IFn$_invoke$arity$variadic = _SLASH___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _SLASH_;\n}();\ncljs.core._LT_ = function() {\n  var _LT_ = null;\n  var _LT___1 = function(x) {\n    return true;\n  };\n  var _LT___2 = function(x, y) {\n    return x &lt; y;\n  };\n  var _LT___3 = function() {\n    var G__22333__delegate = function(x, y, more) {\n      while (true) {\n        if (x &lt; y) {\n          if (cljs.core.next.call(null, more)) {\n            var G__22334 = y;\n            var G__22335 = cljs.core.first.call(null, more);\n            var G__22336 = cljs.core.next.call(null, more);\n            x = G__22334;\n            y = G__22335;\n            more = G__22336;\n            continue;\n          } else {\n            return y &lt; cljs.core.first.call(null, more);\n          }\n        } else {\n          return false;\n        }\n        break;\n      }\n    };\n    var G__22333 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22333__delegate.call(this, x, y, more);\n    };\n    G__22333.cljs$lang$maxFixedArity = 2;\n    G__22333.cljs$lang$applyTo = function(arglist__22337) {\n      var x = cljs.core.first(arglist__22337);\n      arglist__22337 = cljs.core.next(arglist__22337);\n      var y = cljs.core.first(arglist__22337);\n      var more = cljs.core.rest(arglist__22337);\n      return G__22333__delegate(x, y, more);\n    };\n    G__22333.cljs$core$IFn$_invoke$arity$variadic = G__22333__delegate;\n    return G__22333;\n  }();\n  _LT_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return _LT___1.call(this, x);\n      case 2:\n        return _LT___2.call(this, x, y);\n      default:\n        return _LT___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _LT_.cljs$lang$maxFixedArity = 2;\n  _LT_.cljs$lang$applyTo = _LT___3.cljs$lang$applyTo;\n  _LT_.cljs$core$IFn$_invoke$arity$1 = _LT___1;\n  _LT_.cljs$core$IFn$_invoke$arity$2 = _LT___2;\n  _LT_.cljs$core$IFn$_invoke$arity$variadic = _LT___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _LT_;\n}();\ncljs.core._LT__EQ_ = function() {\n  var _LT__EQ_ = null;\n  var _LT__EQ___1 = function(x) {\n    return true;\n  };\n  var _LT__EQ___2 = function(x, y) {\n    return x &lt;= y;\n  };\n  var _LT__EQ___3 = function() {\n    var G__22338__delegate = function(x, y, more) {\n      while (true) {\n        if (x &lt;= y) {\n          if (cljs.core.next.call(null, more)) {\n            var G__22339 = y;\n            var G__22340 = cljs.core.first.call(null, more);\n            var G__22341 = cljs.core.next.call(null, more);\n            x = G__22339;\n            y = G__22340;\n            more = G__22341;\n            continue;\n          } else {\n            return y &lt;= cljs.core.first.call(null, more);\n          }\n        } else {\n          return false;\n        }\n        break;\n      }\n    };\n    var G__22338 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22338__delegate.call(this, x, y, more);\n    };\n    G__22338.cljs$lang$maxFixedArity = 2;\n    G__22338.cljs$lang$applyTo = function(arglist__22342) {\n      var x = cljs.core.first(arglist__22342);\n      arglist__22342 = cljs.core.next(arglist__22342);\n      var y = cljs.core.first(arglist__22342);\n      var more = cljs.core.rest(arglist__22342);\n      return G__22338__delegate(x, y, more);\n    };\n    G__22338.cljs$core$IFn$_invoke$arity$variadic = G__22338__delegate;\n    return G__22338;\n  }();\n  _LT__EQ_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return _LT__EQ___1.call(this, x);\n      case 2:\n        return _LT__EQ___2.call(this, x, y);\n      default:\n        return _LT__EQ___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _LT__EQ_.cljs$lang$maxFixedArity = 2;\n  _LT__EQ_.cljs$lang$applyTo = _LT__EQ___3.cljs$lang$applyTo;\n  _LT__EQ_.cljs$core$IFn$_invoke$arity$1 = _LT__EQ___1;\n  _LT__EQ_.cljs$core$IFn$_invoke$arity$2 = _LT__EQ___2;\n  _LT__EQ_.cljs$core$IFn$_invoke$arity$variadic = _LT__EQ___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _LT__EQ_;\n}();\ncljs.core._GT_ = function() {\n  var _GT_ = null;\n  var _GT___1 = function(x) {\n    return true;\n  };\n  var _GT___2 = function(x, y) {\n    return x &gt; y;\n  };\n  var _GT___3 = function() {\n    var G__22343__delegate = function(x, y, more) {\n      while (true) {\n        if (x &gt; y) {\n          if (cljs.core.next.call(null, more)) {\n            var G__22344 = y;\n            var G__22345 = cljs.core.first.call(null, more);\n            var G__22346 = cljs.core.next.call(null, more);\n            x = G__22344;\n            y = G__22345;\n            more = G__22346;\n            continue;\n          } else {\n            return y &gt; cljs.core.first.call(null, more);\n          }\n        } else {\n          return false;\n        }\n        break;\n      }\n    };\n    var G__22343 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22343__delegate.call(this, x, y, more);\n    };\n    G__22343.cljs$lang$maxFixedArity = 2;\n    G__22343.cljs$lang$applyTo = function(arglist__22347) {\n      var x = cljs.core.first(arglist__22347);\n      arglist__22347 = cljs.core.next(arglist__22347);\n      var y = cljs.core.first(arglist__22347);\n      var more = cljs.core.rest(arglist__22347);\n      return G__22343__delegate(x, y, more);\n    };\n    G__22343.cljs$core$IFn$_invoke$arity$variadic = G__22343__delegate;\n    return G__22343;\n  }();\n  _GT_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return _GT___1.call(this, x);\n      case 2:\n        return _GT___2.call(this, x, y);\n      default:\n        return _GT___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _GT_.cljs$lang$maxFixedArity = 2;\n  _GT_.cljs$lang$applyTo = _GT___3.cljs$lang$applyTo;\n  _GT_.cljs$core$IFn$_invoke$arity$1 = _GT___1;\n  _GT_.cljs$core$IFn$_invoke$arity$2 = _GT___2;\n  _GT_.cljs$core$IFn$_invoke$arity$variadic = _GT___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _GT_;\n}();\ncljs.core._GT__EQ_ = function() {\n  var _GT__EQ_ = null;\n  var _GT__EQ___1 = function(x) {\n    return true;\n  };\n  var _GT__EQ___2 = function(x, y) {\n    return x &gt;= y;\n  };\n  var _GT__EQ___3 = function() {\n    var G__22348__delegate = function(x, y, more) {\n      while (true) {\n        if (x &gt;= y) {\n          if (cljs.core.next.call(null, more)) {\n            var G__22349 = y;\n            var G__22350 = cljs.core.first.call(null, more);\n            var G__22351 = cljs.core.next.call(null, more);\n            x = G__22349;\n            y = G__22350;\n            more = G__22351;\n            continue;\n          } else {\n            return y &gt;= cljs.core.first.call(null, more);\n          }\n        } else {\n          return false;\n        }\n        break;\n      }\n    };\n    var G__22348 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22348__delegate.call(this, x, y, more);\n    };\n    G__22348.cljs$lang$maxFixedArity = 2;\n    G__22348.cljs$lang$applyTo = function(arglist__22352) {\n      var x = cljs.core.first(arglist__22352);\n      arglist__22352 = cljs.core.next(arglist__22352);\n      var y = cljs.core.first(arglist__22352);\n      var more = cljs.core.rest(arglist__22352);\n      return G__22348__delegate(x, y, more);\n    };\n    G__22348.cljs$core$IFn$_invoke$arity$variadic = G__22348__delegate;\n    return G__22348;\n  }();\n  _GT__EQ_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return _GT__EQ___1.call(this, x);\n      case 2:\n        return _GT__EQ___2.call(this, x, y);\n      default:\n        return _GT__EQ___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _GT__EQ_.cljs$lang$maxFixedArity = 2;\n  _GT__EQ_.cljs$lang$applyTo = _GT__EQ___3.cljs$lang$applyTo;\n  _GT__EQ_.cljs$core$IFn$_invoke$arity$1 = _GT__EQ___1;\n  _GT__EQ_.cljs$core$IFn$_invoke$arity$2 = _GT__EQ___2;\n  _GT__EQ_.cljs$core$IFn$_invoke$arity$variadic = _GT__EQ___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _GT__EQ_;\n}();\ncljs.core.dec = function dec(x) {\n  return x - 1;\n};\ncljs.core.max = function() {\n  var max = null;\n  var max__1 = function(x) {\n    return x;\n  };\n  var max__2 = function(x, y) {\n    var x__17452__auto__ = x;\n    var y__17453__auto__ = y;\n    return x__17452__auto__ &gt; y__17453__auto__ ? x__17452__auto__ : y__17453__auto__;\n  };\n  var max__3 = function() {\n    var G__22353__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, max, function() {\n        var x__17452__auto__ = x;\n        var y__17453__auto__ = y;\n        return x__17452__auto__ &gt; y__17453__auto__ ? x__17452__auto__ : y__17453__auto__;\n      }(), more);\n    };\n    var G__22353 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22353__delegate.call(this, x, y, more);\n    };\n    G__22353.cljs$lang$maxFixedArity = 2;\n    G__22353.cljs$lang$applyTo = function(arglist__22354) {\n      var x = cljs.core.first(arglist__22354);\n      arglist__22354 = cljs.core.next(arglist__22354);\n      var y = cljs.core.first(arglist__22354);\n      var more = cljs.core.rest(arglist__22354);\n      return G__22353__delegate(x, y, more);\n    };\n    G__22353.cljs$core$IFn$_invoke$arity$variadic = G__22353__delegate;\n    return G__22353;\n  }();\n  max = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return max__1.call(this, x);\n      case 2:\n        return max__2.call(this, x, y);\n      default:\n        return max__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  max.cljs$lang$maxFixedArity = 2;\n  max.cljs$lang$applyTo = max__3.cljs$lang$applyTo;\n  max.cljs$core$IFn$_invoke$arity$1 = max__1;\n  max.cljs$core$IFn$_invoke$arity$2 = max__2;\n  max.cljs$core$IFn$_invoke$arity$variadic = max__3.cljs$core$IFn$_invoke$arity$variadic;\n  return max;\n}();\ncljs.core.min = function() {\n  var min = null;\n  var min__1 = function(x) {\n    return x;\n  };\n  var min__2 = function(x, y) {\n    var x__17459__auto__ = x;\n    var y__17460__auto__ = y;\n    return x__17459__auto__ &lt; y__17460__auto__ ? x__17459__auto__ : y__17460__auto__;\n  };\n  var min__3 = function() {\n    var G__22355__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, min, function() {\n        var x__17459__auto__ = x;\n        var y__17460__auto__ = y;\n        return x__17459__auto__ &lt; y__17460__auto__ ? x__17459__auto__ : y__17460__auto__;\n      }(), more);\n    };\n    var G__22355 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22355__delegate.call(this, x, y, more);\n    };\n    G__22355.cljs$lang$maxFixedArity = 2;\n    G__22355.cljs$lang$applyTo = function(arglist__22356) {\n      var x = cljs.core.first(arglist__22356);\n      arglist__22356 = cljs.core.next(arglist__22356);\n      var y = cljs.core.first(arglist__22356);\n      var more = cljs.core.rest(arglist__22356);\n      return G__22355__delegate(x, y, more);\n    };\n    G__22355.cljs$core$IFn$_invoke$arity$variadic = G__22355__delegate;\n    return G__22355;\n  }();\n  min = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return min__1.call(this, x);\n      case 2:\n        return min__2.call(this, x, y);\n      default:\n        return min__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  min.cljs$lang$maxFixedArity = 2;\n  min.cljs$lang$applyTo = min__3.cljs$lang$applyTo;\n  min.cljs$core$IFn$_invoke$arity$1 = min__1;\n  min.cljs$core$IFn$_invoke$arity$2 = min__2;\n  min.cljs$core$IFn$_invoke$arity$variadic = min__3.cljs$core$IFn$_invoke$arity$variadic;\n  return min;\n}();\ncljs.core.byte$ = function byte$(x) {\n  return x;\n};\ncljs.core.char$ = function char$(x) {\n  if (typeof x === &quot;number&quot;) {\n    return String.fromCharCode(x);\n  } else {\n    if (typeof x === &quot;string&quot; &amp;&amp; x.length === 1) {\n      return x;\n    } else {\n      throw new Error(&quot;Argument to char must be a character or number&quot;);\n    }\n  }\n};\ncljs.core.short$ = function short$(x) {\n  return x;\n};\ncljs.core.float$ = function float$(x) {\n  return x;\n};\ncljs.core.double$ = function double$(x) {\n  return x;\n};\ncljs.core.unchecked_byte = function unchecked_byte(x) {\n  return x;\n};\ncljs.core.unchecked_char = function unchecked_char(x) {\n  return x;\n};\ncljs.core.unchecked_short = function unchecked_short(x) {\n  return x;\n};\ncljs.core.unchecked_float = function unchecked_float(x) {\n  return x;\n};\ncljs.core.unchecked_double = function unchecked_double(x) {\n  return x;\n};\ncljs.core.unchecked_add = function() {\n  var unchecked_add = null;\n  var unchecked_add__0 = function() {\n    return 0;\n  };\n  var unchecked_add__1 = function(x) {\n    return x;\n  };\n  var unchecked_add__2 = function(x, y) {\n    return x + y;\n  };\n  var unchecked_add__3 = function() {\n    var G__22357__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, unchecked_add, x + y, more);\n    };\n    var G__22357 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22357__delegate.call(this, x, y, more);\n    };\n    G__22357.cljs$lang$maxFixedArity = 2;\n    G__22357.cljs$lang$applyTo = function(arglist__22358) {\n      var x = cljs.core.first(arglist__22358);\n      arglist__22358 = cljs.core.next(arglist__22358);\n      var y = cljs.core.first(arglist__22358);\n      var more = cljs.core.rest(arglist__22358);\n      return G__22357__delegate(x, y, more);\n    };\n    G__22357.cljs$core$IFn$_invoke$arity$variadic = G__22357__delegate;\n    return G__22357;\n  }();\n  unchecked_add = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 0:\n        return unchecked_add__0.call(this);\n      case 1:\n        return unchecked_add__1.call(this, x);\n      case 2:\n        return unchecked_add__2.call(this, x, y);\n      default:\n        return unchecked_add__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  unchecked_add.cljs$lang$maxFixedArity = 2;\n  unchecked_add.cljs$lang$applyTo = unchecked_add__3.cljs$lang$applyTo;\n  unchecked_add.cljs$core$IFn$_invoke$arity$0 = unchecked_add__0;\n  unchecked_add.cljs$core$IFn$_invoke$arity$1 = unchecked_add__1;\n  unchecked_add.cljs$core$IFn$_invoke$arity$2 = unchecked_add__2;\n  unchecked_add.cljs$core$IFn$_invoke$arity$variadic = unchecked_add__3.cljs$core$IFn$_invoke$arity$variadic;\n  return unchecked_add;\n}();\ncljs.core.unchecked_add_int = function() {\n  var unchecked_add_int = null;\n  var unchecked_add_int__0 = function() {\n    return 0;\n  };\n  var unchecked_add_int__1 = function(x) {\n    return x;\n  };\n  var unchecked_add_int__2 = function(x, y) {\n    return x + y;\n  };\n  var unchecked_add_int__3 = function() {\n    var G__22359__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, unchecked_add_int, x + y, more);\n    };\n    var G__22359 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22359__delegate.call(this, x, y, more);\n    };\n    G__22359.cljs$lang$maxFixedArity = 2;\n    G__22359.cljs$lang$applyTo = function(arglist__22360) {\n      var x = cljs.core.first(arglist__22360);\n      arglist__22360 = cljs.core.next(arglist__22360);\n      var y = cljs.core.first(arglist__22360);\n      var more = cljs.core.rest(arglist__22360);\n      return G__22359__delegate(x, y, more);\n    };\n    G__22359.cljs$core$IFn$_invoke$arity$variadic = G__22359__delegate;\n    return G__22359;\n  }();\n  unchecked_add_int = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 0:\n        return unchecked_add_int__0.call(this);\n      case 1:\n        return unchecked_add_int__1.call(this, x);\n      case 2:\n        return unchecked_add_int__2.call(this, x, y);\n      default:\n        return unchecked_add_int__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  unchecked_add_int.cljs$lang$maxFixedArity = 2;\n  unchecked_add_int.cljs$lang$applyTo = unchecked_add_int__3.cljs$lang$applyTo;\n  unchecked_add_int.cljs$core$IFn$_invoke$arity$0 = unchecked_add_int__0;\n  unchecked_add_int.cljs$core$IFn$_invoke$arity$1 = unchecked_add_int__1;\n  unchecked_add_int.cljs$core$IFn$_invoke$arity$2 = unchecked_add_int__2;\n  unchecked_add_int.cljs$core$IFn$_invoke$arity$variadic = unchecked_add_int__3.cljs$core$IFn$_invoke$arity$variadic;\n  return unchecked_add_int;\n}();\ncljs.core.unchecked_dec = function unchecked_dec(x) {\n  return x - 1;\n};\ncljs.core.unchecked_dec_int = function unchecked_dec_int(x) {\n  return x - 1;\n};\ncljs.core.unchecked_divide_int = function() {\n  var unchecked_divide_int = null;\n  var unchecked_divide_int__1 = function(x) {\n    return unchecked_divide_int.call(null, 1, x);\n  };\n  var unchecked_divide_int__2 = function(x, y) {\n    return x / y;\n  };\n  var unchecked_divide_int__3 = function() {\n    var G__22361__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, unchecked_divide_int, unchecked_divide_int.call(null, x, y), more);\n    };\n    var G__22361 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22361__delegate.call(this, x, y, more);\n    };\n    G__22361.cljs$lang$maxFixedArity = 2;\n    G__22361.cljs$lang$applyTo = function(arglist__22362) {\n      var x = cljs.core.first(arglist__22362);\n      arglist__22362 = cljs.core.next(arglist__22362);\n      var y = cljs.core.first(arglist__22362);\n      var more = cljs.core.rest(arglist__22362);\n      return G__22361__delegate(x, y, more);\n    };\n    G__22361.cljs$core$IFn$_invoke$arity$variadic = G__22361__delegate;\n    return G__22361;\n  }();\n  unchecked_divide_int = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return unchecked_divide_int__1.call(this, x);\n      case 2:\n        return unchecked_divide_int__2.call(this, x, y);\n      default:\n        return unchecked_divide_int__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  unchecked_divide_int.cljs$lang$maxFixedArity = 2;\n  unchecked_divide_int.cljs$lang$applyTo = unchecked_divide_int__3.cljs$lang$applyTo;\n  unchecked_divide_int.cljs$core$IFn$_invoke$arity$1 = unchecked_divide_int__1;\n  unchecked_divide_int.cljs$core$IFn$_invoke$arity$2 = unchecked_divide_int__2;\n  unchecked_divide_int.cljs$core$IFn$_invoke$arity$variadic = unchecked_divide_int__3.cljs$core$IFn$_invoke$arity$variadic;\n  return unchecked_divide_int;\n}();\ncljs.core.unchecked_inc = function unchecked_inc(x) {\n  return x + 1;\n};\ncljs.core.unchecked_inc_int = function unchecked_inc_int(x) {\n  return x + 1;\n};\ncljs.core.unchecked_multiply = function() {\n  var unchecked_multiply = null;\n  var unchecked_multiply__0 = function() {\n    return 1;\n  };\n  var unchecked_multiply__1 = function(x) {\n    return x;\n  };\n  var unchecked_multiply__2 = function(x, y) {\n    return x * y;\n  };\n  var unchecked_multiply__3 = function() {\n    var G__22363__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, unchecked_multiply, x * y, more);\n    };\n    var G__22363 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22363__delegate.call(this, x, y, more);\n    };\n    G__22363.cljs$lang$maxFixedArity = 2;\n    G__22363.cljs$lang$applyTo = function(arglist__22364) {\n      var x = cljs.core.first(arglist__22364);\n      arglist__22364 = cljs.core.next(arglist__22364);\n      var y = cljs.core.first(arglist__22364);\n      var more = cljs.core.rest(arglist__22364);\n      return G__22363__delegate(x, y, more);\n    };\n    G__22363.cljs$core$IFn$_invoke$arity$variadic = G__22363__delegate;\n    return G__22363;\n  }();\n  unchecked_multiply = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 0:\n        return unchecked_multiply__0.call(this);\n      case 1:\n        return unchecked_multiply__1.call(this, x);\n      case 2:\n        return unchecked_multiply__2.call(this, x, y);\n      default:\n        return unchecked_multiply__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  unchecked_multiply.cljs$lang$maxFixedArity = 2;\n  unchecked_multiply.cljs$lang$applyTo = unchecked_multiply__3.cljs$lang$applyTo;\n  unchecked_multiply.cljs$core$IFn$_invoke$arity$0 = unchecked_multiply__0;\n  unchecked_multiply.cljs$core$IFn$_invoke$arity$1 = unchecked_multiply__1;\n  unchecked_multiply.cljs$core$IFn$_invoke$arity$2 = unchecked_multiply__2;\n  unchecked_multiply.cljs$core$IFn$_invoke$arity$variadic = unchecked_multiply__3.cljs$core$IFn$_invoke$arity$variadic;\n  return unchecked_multiply;\n}();\ncljs.core.unchecked_multiply_int = function() {\n  var unchecked_multiply_int = null;\n  var unchecked_multiply_int__0 = function() {\n    return 1;\n  };\n  var unchecked_multiply_int__1 = function(x) {\n    return x;\n  };\n  var unchecked_multiply_int__2 = function(x, y) {\n    return x * y;\n  };\n  var unchecked_multiply_int__3 = function() {\n    var G__22365__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, unchecked_multiply_int, x * y, more);\n    };\n    var G__22365 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22365__delegate.call(this, x, y, more);\n    };\n    G__22365.cljs$lang$maxFixedArity = 2;\n    G__22365.cljs$lang$applyTo = function(arglist__22366) {\n      var x = cljs.core.first(arglist__22366);\n      arglist__22366 = cljs.core.next(arglist__22366);\n      var y = cljs.core.first(arglist__22366);\n      var more = cljs.core.rest(arglist__22366);\n      return G__22365__delegate(x, y, more);\n    };\n    G__22365.cljs$core$IFn$_invoke$arity$variadic = G__22365__delegate;\n    return G__22365;\n  }();\n  unchecked_multiply_int = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 0:\n        return unchecked_multiply_int__0.call(this);\n      case 1:\n        return unchecked_multiply_int__1.call(this, x);\n      case 2:\n        return unchecked_multiply_int__2.call(this, x, y);\n      default:\n        return unchecked_multiply_int__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  unchecked_multiply_int.cljs$lang$maxFixedArity = 2;\n  unchecked_multiply_int.cljs$lang$applyTo = unchecked_multiply_int__3.cljs$lang$applyTo;\n  unchecked_multiply_int.cljs$core$IFn$_invoke$arity$0 = unchecked_multiply_int__0;\n  unchecked_multiply_int.cljs$core$IFn$_invoke$arity$1 = unchecked_multiply_int__1;\n  unchecked_multiply_int.cljs$core$IFn$_invoke$arity$2 = unchecked_multiply_int__2;\n  unchecked_multiply_int.cljs$core$IFn$_invoke$arity$variadic = unchecked_multiply_int__3.cljs$core$IFn$_invoke$arity$variadic;\n  return unchecked_multiply_int;\n}();\ncljs.core.unchecked_negate = function unchecked_negate(x) {\n  return-x;\n};\ncljs.core.unchecked_negate_int = function unchecked_negate_int(x) {\n  return-x;\n};\ncljs.core.unchecked_remainder_int = function unchecked_remainder_int(x, n) {\n  return cljs.core.mod.call(null, x, n);\n};\ncljs.core.unchecked_subtract = function() {\n  var unchecked_subtract = null;\n  var unchecked_subtract__1 = function(x) {\n    return-x;\n  };\n  var unchecked_subtract__2 = function(x, y) {\n    return x - y;\n  };\n  var unchecked_subtract__3 = function() {\n    var G__22367__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, unchecked_subtract, x - y, more);\n    };\n    var G__22367 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22367__delegate.call(this, x, y, more);\n    };\n    G__22367.cljs$lang$maxFixedArity = 2;\n    G__22367.cljs$lang$applyTo = function(arglist__22368) {\n      var x = cljs.core.first(arglist__22368);\n      arglist__22368 = cljs.core.next(arglist__22368);\n      var y = cljs.core.first(arglist__22368);\n      var more = cljs.core.rest(arglist__22368);\n      return G__22367__delegate(x, y, more);\n    };\n    G__22367.cljs$core$IFn$_invoke$arity$variadic = G__22367__delegate;\n    return G__22367;\n  }();\n  unchecked_subtract = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return unchecked_subtract__1.call(this, x);\n      case 2:\n        return unchecked_subtract__2.call(this, x, y);\n      default:\n        return unchecked_subtract__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  unchecked_subtract.cljs$lang$maxFixedArity = 2;\n  unchecked_subtract.cljs$lang$applyTo = unchecked_subtract__3.cljs$lang$applyTo;\n  unchecked_subtract.cljs$core$IFn$_invoke$arity$1 = unchecked_subtract__1;\n  unchecked_subtract.cljs$core$IFn$_invoke$arity$2 = unchecked_subtract__2;\n  unchecked_subtract.cljs$core$IFn$_invoke$arity$variadic = unchecked_subtract__3.cljs$core$IFn$_invoke$arity$variadic;\n  return unchecked_subtract;\n}();\ncljs.core.unchecked_subtract_int = function() {\n  var unchecked_subtract_int = null;\n  var unchecked_subtract_int__1 = function(x) {\n    return-x;\n  };\n  var unchecked_subtract_int__2 = function(x, y) {\n    return x - y;\n  };\n  var unchecked_subtract_int__3 = function() {\n    var G__22369__delegate = function(x, y, more) {\n      return cljs.core.reduce.call(null, unchecked_subtract_int, x - y, more);\n    };\n    var G__22369 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22369__delegate.call(this, x, y, more);\n    };\n    G__22369.cljs$lang$maxFixedArity = 2;\n    G__22369.cljs$lang$applyTo = function(arglist__22370) {\n      var x = cljs.core.first(arglist__22370);\n      arglist__22370 = cljs.core.next(arglist__22370);\n      var y = cljs.core.first(arglist__22370);\n      var more = cljs.core.rest(arglist__22370);\n      return G__22369__delegate(x, y, more);\n    };\n    G__22369.cljs$core$IFn$_invoke$arity$variadic = G__22369__delegate;\n    return G__22369;\n  }();\n  unchecked_subtract_int = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return unchecked_subtract_int__1.call(this, x);\n      case 2:\n        return unchecked_subtract_int__2.call(this, x, y);\n      default:\n        return unchecked_subtract_int__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  unchecked_subtract_int.cljs$lang$maxFixedArity = 2;\n  unchecked_subtract_int.cljs$lang$applyTo = unchecked_subtract_int__3.cljs$lang$applyTo;\n  unchecked_subtract_int.cljs$core$IFn$_invoke$arity$1 = unchecked_subtract_int__1;\n  unchecked_subtract_int.cljs$core$IFn$_invoke$arity$2 = unchecked_subtract_int__2;\n  unchecked_subtract_int.cljs$core$IFn$_invoke$arity$variadic = unchecked_subtract_int__3.cljs$core$IFn$_invoke$arity$variadic;\n  return unchecked_subtract_int;\n}();\ncljs.core.fix = function fix(q) {\n  if (q &gt;= 0) {\n    return Math.floor.call(null, q);\n  } else {\n    return Math.ceil.call(null, q);\n  }\n};\ncljs.core.int$ = function int$(x) {\n  return x | 0;\n};\ncljs.core.unchecked_int = function unchecked_int(x) {\n  return cljs.core.fix.call(null, x);\n};\ncljs.core.long$ = function long$(x) {\n  return cljs.core.fix.call(null, x);\n};\ncljs.core.unchecked_long = function unchecked_long(x) {\n  return cljs.core.fix.call(null, x);\n};\ncljs.core.booleans = function booleans(x) {\n  return x;\n};\ncljs.core.bytes = function bytes(x) {\n  return x;\n};\ncljs.core.chars = function chars(x) {\n  return x;\n};\ncljs.core.shorts = function shorts(x) {\n  return x;\n};\ncljs.core.ints = function ints(x) {\n  return x;\n};\ncljs.core.floats = function floats(x) {\n  return x;\n};\ncljs.core.doubles = function doubles(x) {\n  return x;\n};\ncljs.core.longs = function longs(x) {\n  return x;\n};\ncljs.core.js_mod = function js_mod(n, d) {\n  return n % d;\n};\ncljs.core.mod = function mod(n, d) {\n  return(n % d + d) % d;\n};\ncljs.core.quot = function quot(n, d) {\n  var rem = n % d;\n  return cljs.core.fix.call(null, (n - rem) / d);\n};\ncljs.core.rem = function rem(n, d) {\n  var q = cljs.core.quot.call(null, n, d);\n  return n - d * q;\n};\ncljs.core.rand = function() {\n  var rand = null;\n  var rand__0 = function() {\n    return Math.random.call(null);\n  };\n  var rand__1 = function(n) {\n    return n * rand.call(null);\n  };\n  rand = function(n) {\n    switch(arguments.length) {\n      case 0:\n        return rand__0.call(this);\n      case 1:\n        return rand__1.call(this, n);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  rand.cljs$core$IFn$_invoke$arity$0 = rand__0;\n  rand.cljs$core$IFn$_invoke$arity$1 = rand__1;\n  return rand;\n}();\ncljs.core.rand_int = function rand_int(n) {\n  return cljs.core.fix.call(null, cljs.core.rand.call(null, n));\n};\ncljs.core.bit_xor = function bit_xor(x, y) {\n  return x ^ y;\n};\ncljs.core.bit_and = function bit_and(x, y) {\n  return x &amp; y;\n};\ncljs.core.bit_or = function bit_or(x, y) {\n  return x | y;\n};\ncljs.core.bit_and_not = function bit_and_not(x, y) {\n  return x &amp; ~y;\n};\ncljs.core.bit_clear = function bit_clear(x, n) {\n  return x &amp; ~(1 &lt;&lt; n);\n};\ncljs.core.bit_flip = function bit_flip(x, n) {\n  return x ^ 1 &lt;&lt; n;\n};\ncljs.core.bit_not = function bit_not(x) {\n  return~x;\n};\ncljs.core.bit_set = function bit_set(x, n) {\n  return x | 1 &lt;&lt; n;\n};\ncljs.core.bit_test = function bit_test(x, n) {\n  return(x &amp; 1 &lt;&lt; n) != 0;\n};\ncljs.core.bit_shift_left = function bit_shift_left(x, n) {\n  return x &lt;&lt; n;\n};\ncljs.core.bit_shift_right = function bit_shift_right(x, n) {\n  return x &gt;&gt; n;\n};\ncljs.core.bit_shift_right_zero_fill = function bit_shift_right_zero_fill(x, n) {\n  return x &gt;&gt;&gt; n;\n};\ncljs.core.unsigned_bit_shift_right = function unsigned_bit_shift_right(x, n) {\n  return x &gt;&gt;&gt; n;\n};\ncljs.core.bit_count = function bit_count(v) {\n  var v__$1 = v - (v &gt;&gt; 1 &amp; 1431655765);\n  var v__$2 = (v__$1 &amp; 858993459) + (v__$1 &gt;&gt; 2 &amp; 858993459);\n  return(v__$2 + (v__$2 &gt;&gt; 4) &amp; 252645135) * 16843009 &gt;&gt; 24;\n};\ncljs.core._EQ__EQ_ = function() {\n  var _EQ__EQ_ = null;\n  var _EQ__EQ___1 = function(x) {\n    return true;\n  };\n  var _EQ__EQ___2 = function(x, y) {\n    return cljs.core._equiv.call(null, x, y);\n  };\n  var _EQ__EQ___3 = function() {\n    var G__22371__delegate = function(x, y, more) {\n      while (true) {\n        if (_EQ__EQ_.call(null, x, y)) {\n          if (cljs.core.next.call(null, more)) {\n            var G__22372 = y;\n            var G__22373 = cljs.core.first.call(null, more);\n            var G__22374 = cljs.core.next.call(null, more);\n            x = G__22372;\n            y = G__22373;\n            more = G__22374;\n            continue;\n          } else {\n            return _EQ__EQ_.call(null, y, cljs.core.first.call(null, more));\n          }\n        } else {\n          return false;\n        }\n        break;\n      }\n    };\n    var G__22371 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22371__delegate.call(this, x, y, more);\n    };\n    G__22371.cljs$lang$maxFixedArity = 2;\n    G__22371.cljs$lang$applyTo = function(arglist__22375) {\n      var x = cljs.core.first(arglist__22375);\n      arglist__22375 = cljs.core.next(arglist__22375);\n      var y = cljs.core.first(arglist__22375);\n      var more = cljs.core.rest(arglist__22375);\n      return G__22371__delegate(x, y, more);\n    };\n    G__22371.cljs$core$IFn$_invoke$arity$variadic = G__22371__delegate;\n    return G__22371;\n  }();\n  _EQ__EQ_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return _EQ__EQ___1.call(this, x);\n      case 2:\n        return _EQ__EQ___2.call(this, x, y);\n      default:\n        return _EQ__EQ___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  _EQ__EQ_.cljs$lang$maxFixedArity = 2;\n  _EQ__EQ_.cljs$lang$applyTo = _EQ__EQ___3.cljs$lang$applyTo;\n  _EQ__EQ_.cljs$core$IFn$_invoke$arity$1 = _EQ__EQ___1;\n  _EQ__EQ_.cljs$core$IFn$_invoke$arity$2 = _EQ__EQ___2;\n  _EQ__EQ_.cljs$core$IFn$_invoke$arity$variadic = _EQ__EQ___3.cljs$core$IFn$_invoke$arity$variadic;\n  return _EQ__EQ_;\n}();\ncljs.core.pos_QMARK_ = function pos_QMARK_(n) {\n  return n &gt; 0;\n};\ncljs.core.zero_QMARK_ = function zero_QMARK_(n) {\n  return n === 0;\n};\ncljs.core.neg_QMARK_ = function neg_QMARK_(x) {\n  return x &lt; 0;\n};\ncljs.core.nthnext = function nthnext(coll, n) {\n  var n__$1 = n;\n  var xs = cljs.core.seq.call(null, coll);\n  while (true) {\n    if (xs &amp;&amp; n__$1 &gt; 0) {\n      var G__22376 = n__$1 - 1;\n      var G__22377 = cljs.core.next.call(null, xs);\n      n__$1 = G__22376;\n      xs = G__22377;\n      continue;\n    } else {\n      return xs;\n    }\n    break;\n  }\n};\ncljs.core.str = function() {\n  var str = null;\n  var str__0 = function() {\n    return &quot;&quot;;\n  };\n  var str__1 = function(x) {\n    if (x == null) {\n      return &quot;&quot;;\n    } else {\n      return x.toString();\n    }\n  };\n  var str__2 = function() {\n    var G__22378__delegate = function(x, ys) {\n      var sb = new goog.string.StringBuffer(str.call(null, x));\n      var more = ys;\n      while (true) {\n        if (cljs.core.truth_(more)) {\n          var G__22379 = sb.append(str.call(null, cljs.core.first.call(null, more)));\n          var G__22380 = cljs.core.next.call(null, more);\n          sb = G__22379;\n          more = G__22380;\n          continue;\n        } else {\n          return sb.toString();\n        }\n        break;\n      }\n    };\n    var G__22378 = function(x, var_args) {\n      var ys = null;\n      if (arguments.length &gt; 1) {\n        ys = cljs.core.array_seq(Array.prototype.slice.call(arguments, 1), 0);\n      }\n      return G__22378__delegate.call(this, x, ys);\n    };\n    G__22378.cljs$lang$maxFixedArity = 1;\n    G__22378.cljs$lang$applyTo = function(arglist__22381) {\n      var x = cljs.core.first(arglist__22381);\n      var ys = cljs.core.rest(arglist__22381);\n      return G__22378__delegate(x, ys);\n    };\n    G__22378.cljs$core$IFn$_invoke$arity$variadic = G__22378__delegate;\n    return G__22378;\n  }();\n  str = function(x, var_args) {\n    var ys = var_args;\n    switch(arguments.length) {\n      case 0:\n        return str__0.call(this);\n      case 1:\n        return str__1.call(this, x);\n      default:\n        return str__2.cljs$core$IFn$_invoke$arity$variadic(x, cljs.core.array_seq(arguments, 1));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  str.cljs$lang$maxFixedArity = 1;\n  str.cljs$lang$applyTo = str__2.cljs$lang$applyTo;\n  str.cljs$core$IFn$_invoke$arity$0 = str__0;\n  str.cljs$core$IFn$_invoke$arity$1 = str__1;\n  str.cljs$core$IFn$_invoke$arity$variadic = str__2.cljs$core$IFn$_invoke$arity$variadic;\n  return str;\n}();\ncljs.core.subs = function() {\n  var subs = null;\n  var subs__2 = function(s, start) {\n    return s.substring(start);\n  };\n  var subs__3 = function(s, start, end) {\n    return s.substring(start, end);\n  };\n  subs = function(s, start, end) {\n    switch(arguments.length) {\n      case 2:\n        return subs__2.call(this, s, start);\n      case 3:\n        return subs__3.call(this, s, start, end);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  subs.cljs$core$IFn$_invoke$arity$2 = subs__2;\n  subs.cljs$core$IFn$_invoke$arity$3 = subs__3;\n  return subs;\n}();\ncljs.core.equiv_sequential = function equiv_sequential(x, y) {\n  return cljs.core.boolean$.call(null, cljs.core.sequential_QMARK_.call(null, y) ? cljs.core.counted_QMARK_.call(null, x) &amp;&amp; cljs.core.counted_QMARK_.call(null, y) &amp;&amp; !(cljs.core.count.call(null, x) === cljs.core.count.call(null, y)) ? false : function() {\n    var xs = cljs.core.seq.call(null, x);\n    var ys = cljs.core.seq.call(null, y);\n    while (true) {\n      if (xs == null) {\n        return ys == null;\n      } else {\n        if (ys == null) {\n          return false;\n        } else {\n          if (cljs.core._EQ_.call(null, cljs.core.first.call(null, xs), cljs.core.first.call(null, ys))) {\n            var G__22382 = cljs.core.next.call(null, xs);\n            var G__22383 = cljs.core.next.call(null, ys);\n            xs = G__22382;\n            ys = G__22383;\n            continue;\n          } else {\n            return false;\n          }\n        }\n      }\n      break;\n    }\n  }() : null);\n};\ncljs.core.hash_coll = function hash_coll(coll) {\n  if (cljs.core.seq.call(null, coll)) {\n    var res = cljs.core.hash.call(null, cljs.core.first.call(null, coll));\n    var s = cljs.core.next.call(null, coll);\n    while (true) {\n      if (s == null) {\n        return res;\n      } else {\n        var G__22384 = cljs.core.hash_combine.call(null, res, cljs.core.hash.call(null, cljs.core.first.call(null, s)));\n        var G__22385 = cljs.core.next.call(null, s);\n        res = G__22384;\n        s = G__22385;\n        continue;\n      }\n      break;\n    }\n  } else {\n    return 0;\n  }\n};\ncljs.core.hash_imap = function hash_imap(m) {\n  var h = 0;\n  var s = cljs.core.seq.call(null, m);\n  while (true) {\n    if (s) {\n      var e = cljs.core.first.call(null, s);\n      var G__22386 = (h + (cljs.core.hash.call(null, cljs.core.key.call(null, e)) ^ cljs.core.hash.call(null, cljs.core.val.call(null, e)))) % 4503599627370496;\n      var G__22387 = cljs.core.next.call(null, s);\n      h = G__22386;\n      s = G__22387;\n      continue;\n    } else {\n      return h;\n    }\n    break;\n  }\n};\ncljs.core.hash_iset = function hash_iset(s) {\n  var h = 0;\n  var s__$1 = cljs.core.seq.call(null, s);\n  while (true) {\n    if (s__$1) {\n      var e = cljs.core.first.call(null, s__$1);\n      var G__22388 = (h + cljs.core.hash.call(null, e)) % 4503599627370496;\n      var G__22389 = cljs.core.next.call(null, s__$1);\n      h = G__22388;\n      s__$1 = G__22389;\n      continue;\n    } else {\n      return h;\n    }\n    break;\n  }\n};\ncljs.core.extend_object_BANG_ = function extend_object_BANG_(obj, fn_map) {\n  var seq__22396_22402 = cljs.core.seq.call(null, fn_map);\n  var chunk__22397_22403 = null;\n  var count__22398_22404 = 0;\n  var i__22399_22405 = 0;\n  while (true) {\n    if (i__22399_22405 &lt; count__22398_22404) {\n      var vec__22400_22406 = cljs.core._nth.call(null, chunk__22397_22403, i__22399_22405);\n      var key_name_22407 = cljs.core.nth.call(null, vec__22400_22406, 0, null);\n      var f_22408 = cljs.core.nth.call(null, vec__22400_22406, 1, null);\n      var str_name_22409 = cljs.core.name.call(null, key_name_22407);\n      obj[str_name_22409] = f_22408;\n      var G__22410 = seq__22396_22402;\n      var G__22411 = chunk__22397_22403;\n      var G__22412 = count__22398_22404;\n      var G__22413 = i__22399_22405 + 1;\n      seq__22396_22402 = G__22410;\n      chunk__22397_22403 = G__22411;\n      count__22398_22404 = G__22412;\n      i__22399_22405 = G__22413;\n      continue;\n    } else {\n      var temp__4126__auto___22414 = cljs.core.seq.call(null, seq__22396_22402);\n      if (temp__4126__auto___22414) {\n        var seq__22396_22415__$1 = temp__4126__auto___22414;\n        if (cljs.core.chunked_seq_QMARK_.call(null, seq__22396_22415__$1)) {\n          var c__17901__auto___22416 = cljs.core.chunk_first.call(null, seq__22396_22415__$1);\n          var G__22417 = cljs.core.chunk_rest.call(null, seq__22396_22415__$1);\n          var G__22418 = c__17901__auto___22416;\n          var G__22419 = cljs.core.count.call(null, c__17901__auto___22416);\n          var G__22420 = 0;\n          seq__22396_22402 = G__22417;\n          chunk__22397_22403 = G__22418;\n          count__22398_22404 = G__22419;\n          i__22399_22405 = G__22420;\n          continue;\n        } else {\n          var vec__22401_22421 = cljs.core.first.call(null, seq__22396_22415__$1);\n          var key_name_22422 = cljs.core.nth.call(null, vec__22401_22421, 0, null);\n          var f_22423 = cljs.core.nth.call(null, vec__22401_22421, 1, null);\n          var str_name_22424 = cljs.core.name.call(null, key_name_22422);\n          obj[str_name_22424] = f_22423;\n          var G__22425 = cljs.core.next.call(null, seq__22396_22415__$1);\n          var G__22426 = null;\n          var G__22427 = 0;\n          var G__22428 = 0;\n          seq__22396_22402 = G__22425;\n          chunk__22397_22403 = G__22426;\n          count__22398_22404 = G__22427;\n          i__22399_22405 = G__22428;\n          continue;\n        }\n      } else {\n      }\n    }\n    break;\n  }\n  return obj;\n};\ncljs.core.List = function(meta, first, rest, count, __hash) {\n  this.meta = meta;\n  this.first = first;\n  this.rest = rest;\n  this.count = count;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 65937646;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.List.cljs$lang$type = true;\ncljs.core.List.cljs$lang$ctorStr = &quot;cljs.core/List&quot;;\ncljs.core.List.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/List&quot;);\n};\ncljs.core.List.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.List.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.List.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.List.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.List(self__.meta, self__.first, self__.rest, self__.count, self__.__hash);\n};\ncljs.core.List.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.count === 1) {\n    return null;\n  } else {\n    return self__.rest;\n  }\n};\ncljs.core.List.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.count;\n};\ncljs.core.List.prototype.cljs$core$IStack$_peek$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.first;\n};\ncljs.core.List.prototype.cljs$core$IStack$_pop$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._rest.call(null, coll__$1);\n};\ncljs.core.List.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.List.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.List.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.List.EMPTY;\n};\ncljs.core.List.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.List.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.List.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.first;\n};\ncljs.core.List.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.count === 1) {\n    return cljs.core.List.EMPTY;\n  } else {\n    return self__.rest;\n  }\n};\ncljs.core.List.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.List.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.List(meta__$1, self__.first, self__.rest, self__.count, self__.__hash);\n};\ncljs.core.List.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.List(self__.meta, o, coll__$1, self__.count + 1, null);\n};\ncljs.core.__GT_List = function __GT_List(meta, first, rest, count, __hash) {\n  return new cljs.core.List(meta, first, rest, count, __hash);\n};\ncljs.core.EmptyList = function(meta) {\n  this.meta = meta;\n  this.cljs$lang$protocol_mask$partition0$ = 65937614;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.EmptyList.cljs$lang$type = true;\ncljs.core.EmptyList.cljs$lang$ctorStr = &quot;cljs.core/EmptyList&quot;;\ncljs.core.EmptyList.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/EmptyList&quot;);\n};\ncljs.core.EmptyList.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.EmptyList.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.EmptyList.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.EmptyList.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.EmptyList(self__.meta);\n};\ncljs.core.EmptyList.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return null;\n};\ncljs.core.EmptyList.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return 0;\n};\ncljs.core.EmptyList.prototype.cljs$core$IStack$_peek$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return null;\n};\ncljs.core.EmptyList.prototype.cljs$core$IStack$_pop$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  throw new Error(&quot;Can&apos;t pop empty list&quot;);\n};\ncljs.core.EmptyList.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return 0;\n};\ncljs.core.EmptyList.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.EmptyList.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.EmptyList.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.EmptyList.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.EmptyList.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return null;\n};\ncljs.core.EmptyList.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.List.EMPTY;\n};\ncljs.core.EmptyList.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return null;\n};\ncljs.core.EmptyList.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.EmptyList(meta__$1);\n};\ncljs.core.EmptyList.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.List(self__.meta, o, null, 1, null);\n};\ncljs.core.__GT_EmptyList = function __GT_EmptyList(meta) {\n  return new cljs.core.EmptyList(meta);\n};\ncljs.core.List.EMPTY = new cljs.core.EmptyList(null);\ncljs.core.reversible_QMARK_ = function reversible_QMARK_(coll) {\n  var G__22430 = coll;\n  if (G__22430) {\n    var bit__17795__auto__ = G__22430.cljs$lang$protocol_mask$partition0$ &amp; 134217728;\n    if (bit__17795__auto__ || G__22430.cljs$core$IReversible$) {\n      return true;\n    } else {\n      if (!G__22430.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IReversible, G__22430);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IReversible, G__22430);\n  }\n};\ncljs.core.rseq = function rseq(coll) {\n  return cljs.core._rseq.call(null, coll);\n};\ncljs.core.reverse = function reverse(coll) {\n  if (cljs.core.reversible_QMARK_.call(null, coll)) {\n    return cljs.core.rseq.call(null, coll);\n  } else {\n    return cljs.core.reduce.call(null, cljs.core.conj, cljs.core.List.EMPTY, coll);\n  }\n};\ncljs.core.list = function() {\n  var list__delegate = function(xs) {\n    var arr = xs instanceof cljs.core.IndexedSeq &amp;&amp; xs.i === 0 ? xs.arr : function() {\n      var arr = [];\n      var xs__$1 = xs;\n      while (true) {\n        if (!(xs__$1 == null)) {\n          arr.push(cljs.core._first.call(null, xs__$1));\n          var G__22431 = cljs.core._next.call(null, xs__$1);\n          xs__$1 = G__22431;\n          continue;\n        } else {\n          return arr;\n        }\n        break;\n      }\n    }();\n    var i = arr.length;\n    var r = cljs.core.List.EMPTY;\n    while (true) {\n      if (i &gt; 0) {\n        var G__22432 = i - 1;\n        var G__22433 = cljs.core._conj.call(null, r, arr[i - 1]);\n        i = G__22432;\n        r = G__22433;\n        continue;\n      } else {\n        return r;\n      }\n      break;\n    }\n  };\n  var list = function(var_args) {\n    var xs = null;\n    if (arguments.length &gt; 0) {\n      xs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return list__delegate.call(this, xs);\n  };\n  list.cljs$lang$maxFixedArity = 0;\n  list.cljs$lang$applyTo = function(arglist__22434) {\n    var xs = cljs.core.seq(arglist__22434);\n    return list__delegate(xs);\n  };\n  list.cljs$core$IFn$_invoke$arity$variadic = list__delegate;\n  return list;\n}();\ncljs.core.Cons = function(meta, first, rest, __hash) {\n  this.meta = meta;\n  this.first = first;\n  this.rest = rest;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 65929452;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.Cons.cljs$lang$type = true;\ncljs.core.Cons.cljs$lang$ctorStr = &quot;cljs.core/Cons&quot;;\ncljs.core.Cons.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Cons&quot;);\n};\ncljs.core.Cons.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.Cons.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.Cons.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.Cons.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.Cons(self__.meta, self__.first, self__.rest, self__.__hash);\n};\ncljs.core.Cons.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.rest == null) {\n    return null;\n  } else {\n    return cljs.core.seq.call(null, self__.rest);\n  }\n};\ncljs.core.Cons.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.Cons.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.Cons.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__.meta);\n};\ncljs.core.Cons.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.Cons.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.Cons.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.first;\n};\ncljs.core.Cons.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.rest == null) {\n    return cljs.core.List.EMPTY;\n  } else {\n    return self__.rest;\n  }\n};\ncljs.core.Cons.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.Cons.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.Cons(meta__$1, self__.first, self__.rest, self__.__hash);\n};\ncljs.core.Cons.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.Cons(null, o, coll__$1, self__.__hash);\n};\ncljs.core.__GT_Cons = function __GT_Cons(meta, first, rest, __hash) {\n  return new cljs.core.Cons(meta, first, rest, __hash);\n};\ncljs.core.cons = function cons(x, coll) {\n  if (function() {\n    var or__17145__auto__ = coll == null;\n    if (or__17145__auto__) {\n      return or__17145__auto__;\n    } else {\n      var G__22438 = coll;\n      if (G__22438) {\n        var bit__17788__auto__ = G__22438.cljs$lang$protocol_mask$partition0$ &amp; 64;\n        if (bit__17788__auto__ || G__22438.cljs$core$ISeq$) {\n          return true;\n        } else {\n          return false;\n        }\n      } else {\n        return false;\n      }\n    }\n  }()) {\n    return new cljs.core.Cons(null, x, coll, null);\n  } else {\n    return new cljs.core.Cons(null, x, cljs.core.seq.call(null, coll), null);\n  }\n};\ncljs.core.list_QMARK_ = function list_QMARK_(x) {\n  var G__22440 = x;\n  if (G__22440) {\n    var bit__17795__auto__ = G__22440.cljs$lang$protocol_mask$partition0$ &amp; 33554432;\n    if (bit__17795__auto__ || G__22440.cljs$core$IList$) {\n      return true;\n    } else {\n      if (!G__22440.cljs$lang$protocol_mask$partition0$) {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IList, G__22440);\n      } else {\n        return false;\n      }\n    }\n  } else {\n    return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IList, G__22440);\n  }\n};\ncljs.core.hash_keyword = function hash_keyword(k) {\n  return cljs.core.hash_symbol.call(null, k) + 2654435769 | 0;\n};\ncljs.core.Keyword = function(ns, name, fqn, _hash) {\n  this.ns = ns;\n  this.name = name;\n  this.fqn = fqn;\n  this._hash = _hash;\n  this.cljs$lang$protocol_mask$partition0$ = 2153775105;\n  this.cljs$lang$protocol_mask$partition1$ = 4096;\n};\ncljs.core.Keyword.cljs$lang$type = true;\ncljs.core.Keyword.cljs$lang$ctorStr = &quot;cljs.core/Keyword&quot;;\ncljs.core.Keyword.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Keyword&quot;);\n};\ncljs.core.Keyword.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(o, writer, _) {\n  var self__ = this;\n  var o__$1 = this;\n  return cljs.core._write.call(null, writer, &quot;:&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(self__.fqn));\n};\ncljs.core.Keyword.prototype.cljs$core$INamed$_name$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.name;\n};\ncljs.core.Keyword.prototype.cljs$core$INamed$_namespace$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.ns;\n};\ncljs.core.Keyword.prototype.cljs$core$IHash$_hash$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  var h__17556__auto__ = self__._hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_keyword.call(null, this$__$1);\n    self__._hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.Keyword.prototype.call = function() {\n  var G__22442 = null;\n  var G__22442__2 = function(self__, coll) {\n    var self__ = this;\n    var self____$1 = this;\n    var kw = self____$1;\n    return cljs.core.get.call(null, coll, kw);\n  };\n  var G__22442__3 = function(self__, coll, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var kw = self____$1;\n    return cljs.core.get.call(null, coll, kw, not_found);\n  };\n  G__22442 = function(self__, coll, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22442__2.call(this, self__, coll);\n      case 3:\n        return G__22442__3.call(this, self__, coll, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22442.cljs$core$IFn$_invoke$arity$2 = G__22442__2;\n  G__22442.cljs$core$IFn$_invoke$arity$3 = G__22442__3;\n  return G__22442;\n}();\ncljs.core.Keyword.prototype.apply = function(self__, args22441) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22441)));\n};\ncljs.core.Keyword.prototype.cljs$core$IFn$_invoke$arity$1 = function(coll) {\n  var self__ = this;\n  var kw = this;\n  return cljs.core.get.call(null, coll, kw);\n};\ncljs.core.Keyword.prototype.cljs$core$IFn$_invoke$arity$2 = function(coll, not_found) {\n  var self__ = this;\n  var kw = this;\n  return cljs.core.get.call(null, coll, kw, not_found);\n};\ncljs.core.Keyword.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(_, other) {\n  var self__ = this;\n  var ___$1 = this;\n  if (other instanceof cljs.core.Keyword) {\n    return self__.fqn === other.fqn;\n  } else {\n    return false;\n  }\n};\ncljs.core.Keyword.prototype.toString = function() {\n  var self__ = this;\n  var _ = this;\n  return &quot;:&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(self__.fqn);\n};\ncljs.core.Keyword.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.__GT_Keyword = function __GT_Keyword(ns, name, fqn, _hash) {\n  return new cljs.core.Keyword(ns, name, fqn, _hash);\n};\ncljs.core.keyword_QMARK_ = function keyword_QMARK_(x) {\n  return x instanceof cljs.core.Keyword;\n};\ncljs.core.keyword_identical_QMARK_ = function keyword_identical_QMARK_(x, y) {\n  if (x === y) {\n    return true;\n  } else {\n    if (x instanceof cljs.core.Keyword &amp;&amp; y instanceof cljs.core.Keyword) {\n      return x.fqn === y.fqn;\n    } else {\n      return false;\n    }\n  }\n};\ncljs.core.namespace = function namespace(x) {\n  if (function() {\n    var G__22444 = x;\n    if (G__22444) {\n      var bit__17788__auto__ = G__22444.cljs$lang$protocol_mask$partition1$ &amp; 4096;\n      if (bit__17788__auto__ || G__22444.cljs$core$INamed$) {\n        return true;\n      } else {\n        return false;\n      }\n    } else {\n      return false;\n    }\n  }()) {\n    return cljs.core._namespace.call(null, x);\n  } else {\n    throw new Error(&quot;Doesn&apos;t support namespace: &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(x));\n  }\n};\ncljs.core.keyword = function() {\n  var keyword = null;\n  var keyword__1 = function(name) {\n    if (name instanceof cljs.core.Keyword) {\n      return name;\n    } else {\n      if (name instanceof cljs.core.Symbol) {\n        return new cljs.core.Keyword(cljs.core.namespace.call(null, name), cljs.core.name.call(null, name), name.str, null);\n      } else {\n        if (typeof name === &quot;string&quot;) {\n          var parts = name.split(&quot;/&quot;);\n          if (parts.length === 2) {\n            return new cljs.core.Keyword(parts[0], parts[1], name, null);\n          } else {\n            return new cljs.core.Keyword(null, parts[0], name, null);\n          }\n        } else {\n          return null;\n        }\n      }\n    }\n  };\n  var keyword__2 = function(ns, name) {\n    return new cljs.core.Keyword(ns, name, &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.truth_(ns) ? &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(ns) + &quot;/&quot; : null) + cljs.core.str.cljs$core$IFn$_invoke$arity$1(name), null);\n  };\n  keyword = function(ns, name) {\n    switch(arguments.length) {\n      case 1:\n        return keyword__1.call(this, ns);\n      case 2:\n        return keyword__2.call(this, ns, name);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  keyword.cljs$core$IFn$_invoke$arity$1 = keyword__1;\n  keyword.cljs$core$IFn$_invoke$arity$2 = keyword__2;\n  return keyword;\n}();\ncljs.core.LazySeq = function(meta, fn, s, __hash) {\n  this.meta = meta;\n  this.fn = fn;\n  this.s = s;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32374988;\n};\ncljs.core.LazySeq.cljs$lang$type = true;\ncljs.core.LazySeq.cljs$lang$ctorStr = &quot;cljs.core/LazySeq&quot;;\ncljs.core.LazySeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/LazySeq&quot;);\n};\ncljs.core.LazySeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.LazySeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.LazySeq.prototype.sval = function() {\n  var self__ = this;\n  var coll = this;\n  if (self__.fn == null) {\n    return self__.s;\n  } else {\n    self__.s = self__.fn.call(null);\n    self__.fn = null;\n    return self__.s;\n  }\n};\ncljs.core.LazySeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.LazySeq.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  cljs.core._seq.call(null, coll__$1);\n  if (self__.s == null) {\n    return null;\n  } else {\n    return cljs.core.next.call(null, self__.s);\n  }\n};\ncljs.core.LazySeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.LazySeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.LazySeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__.meta);\n};\ncljs.core.LazySeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.LazySeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.LazySeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  cljs.core._seq.call(null, coll__$1);\n  if (self__.s == null) {\n    return null;\n  } else {\n    return cljs.core.first.call(null, self__.s);\n  }\n};\ncljs.core.LazySeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  cljs.core._seq.call(null, coll__$1);\n  if (!(self__.s == null)) {\n    return cljs.core.rest.call(null, self__.s);\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.LazySeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  coll__$1.sval();\n  if (self__.s == null) {\n    return null;\n  } else {\n    var ls = self__.s;\n    while (true) {\n      if (ls instanceof cljs.core.LazySeq) {\n        var G__22445 = ls.sval();\n        ls = G__22445;\n        continue;\n      } else {\n        self__.s = ls;\n        return cljs.core.seq.call(null, self__.s);\n      }\n      break;\n    }\n  }\n};\ncljs.core.LazySeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.LazySeq(meta__$1, self__.fn, self__.s, self__.__hash);\n};\ncljs.core.LazySeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_LazySeq = function __GT_LazySeq(meta, fn, s, __hash) {\n  return new cljs.core.LazySeq(meta, fn, s, __hash);\n};\ncljs.core.ChunkBuffer = function(buf, end) {\n  this.buf = buf;\n  this.end = end;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 2;\n};\ncljs.core.ChunkBuffer.cljs$lang$type = true;\ncljs.core.ChunkBuffer.cljs$lang$ctorStr = &quot;cljs.core/ChunkBuffer&quot;;\ncljs.core.ChunkBuffer.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ChunkBuffer&quot;);\n};\ncljs.core.ChunkBuffer.prototype.cljs$core$ICounted$_count$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.end;\n};\ncljs.core.ChunkBuffer.prototype.add = function(o) {\n  var self__ = this;\n  var _ = this;\n  self__.buf[self__.end] = o;\n  return self__.end = self__.end + 1;\n};\ncljs.core.ChunkBuffer.prototype.chunk = function(o) {\n  var self__ = this;\n  var _ = this;\n  var ret = new cljs.core.ArrayChunk(self__.buf, 0, self__.end);\n  self__.buf = null;\n  return ret;\n};\ncljs.core.__GT_ChunkBuffer = function __GT_ChunkBuffer(buf, end) {\n  return new cljs.core.ChunkBuffer(buf, end);\n};\ncljs.core.chunk_buffer = function chunk_buffer(capacity) {\n  return new cljs.core.ChunkBuffer(new Array(capacity), 0);\n};\ncljs.core.ArrayChunk = function(arr, off, end) {\n  this.arr = arr;\n  this.off = off;\n  this.end = end;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 524306;\n};\ncljs.core.ArrayChunk.cljs$lang$type = true;\ncljs.core.ArrayChunk.cljs$lang$ctorStr = &quot;cljs.core/ArrayChunk&quot;;\ncljs.core.ArrayChunk.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ArrayChunk&quot;);\n};\ncljs.core.ArrayChunk.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.array_reduce.call(null, self__.arr, f, self__.arr[self__.off], self__.off + 1);\n};\ncljs.core.ArrayChunk.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.array_reduce.call(null, self__.arr, f, start, self__.off);\n};\ncljs.core.ArrayChunk.prototype.cljs$core$IChunk$ = true;\ncljs.core.ArrayChunk.prototype.cljs$core$IChunk$_drop_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.off === self__.end) {\n    throw new Error(&quot;-drop-first of empty chunk&quot;);\n  } else {\n    return new cljs.core.ArrayChunk(self__.arr, self__.off + 1, self__.end);\n  }\n};\ncljs.core.ArrayChunk.prototype.cljs$core$IIndexed$_nth$arity$2 = function(coll, i) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.arr[self__.off + i];\n};\ncljs.core.ArrayChunk.prototype.cljs$core$IIndexed$_nth$arity$3 = function(coll, i, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (i &gt;= 0 &amp;&amp; i &lt; self__.end - self__.off) {\n    return self__.arr[self__.off + i];\n  } else {\n    return not_found;\n  }\n};\ncljs.core.ArrayChunk.prototype.cljs$core$ICounted$_count$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.end - self__.off;\n};\ncljs.core.__GT_ArrayChunk = function __GT_ArrayChunk(arr, off, end) {\n  return new cljs.core.ArrayChunk(arr, off, end);\n};\ncljs.core.array_chunk = function() {\n  var array_chunk = null;\n  var array_chunk__1 = function(arr) {\n    return new cljs.core.ArrayChunk(arr, 0, arr.length);\n  };\n  var array_chunk__2 = function(arr, off) {\n    return new cljs.core.ArrayChunk(arr, off, arr.length);\n  };\n  var array_chunk__3 = function(arr, off, end) {\n    return new cljs.core.ArrayChunk(arr, off, end);\n  };\n  array_chunk = function(arr, off, end) {\n    switch(arguments.length) {\n      case 1:\n        return array_chunk__1.call(this, arr);\n      case 2:\n        return array_chunk__2.call(this, arr, off);\n      case 3:\n        return array_chunk__3.call(this, arr, off, end);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  array_chunk.cljs$core$IFn$_invoke$arity$1 = array_chunk__1;\n  array_chunk.cljs$core$IFn$_invoke$arity$2 = array_chunk__2;\n  array_chunk.cljs$core$IFn$_invoke$arity$3 = array_chunk__3;\n  return array_chunk;\n}();\ncljs.core.ChunkedCons = function(chunk, more, meta, __hash) {\n  this.chunk = chunk;\n  this.more = more;\n  this.meta = meta;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 31850732;\n  this.cljs$lang$protocol_mask$partition1$ = 1536;\n};\ncljs.core.ChunkedCons.cljs$lang$type = true;\ncljs.core.ChunkedCons.cljs$lang$ctorStr = &quot;cljs.core/ChunkedCons&quot;;\ncljs.core.ChunkedCons.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ChunkedCons&quot;);\n};\ncljs.core.ChunkedCons.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.ChunkedCons.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.ChunkedCons.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.ChunkedCons.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core._count.call(null, self__.chunk) &gt; 1) {\n    return new cljs.core.ChunkedCons(cljs.core._drop_first.call(null, self__.chunk), self__.more, self__.meta, null);\n  } else {\n    var more__$1 = cljs.core._seq.call(null, self__.more);\n    if (more__$1 == null) {\n      return null;\n    } else {\n      return more__$1;\n    }\n  }\n};\ncljs.core.ChunkedCons.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.ChunkedCons.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.ChunkedCons.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__.meta);\n};\ncljs.core.ChunkedCons.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._nth.call(null, self__.chunk, 0);\n};\ncljs.core.ChunkedCons.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core._count.call(null, self__.chunk) &gt; 1) {\n    return new cljs.core.ChunkedCons(cljs.core._drop_first.call(null, self__.chunk), self__.more, self__.meta, null);\n  } else {\n    if (self__.more == null) {\n      return cljs.core.List.EMPTY;\n    } else {\n      return self__.more;\n    }\n  }\n};\ncljs.core.ChunkedCons.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.ChunkedCons.prototype.cljs$core$IChunkedSeq$_chunked_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.chunk;\n};\ncljs.core.ChunkedCons.prototype.cljs$core$IChunkedSeq$_chunked_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.more == null) {\n    return cljs.core.List.EMPTY;\n  } else {\n    return self__.more;\n  }\n};\ncljs.core.ChunkedCons.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, m) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.ChunkedCons(self__.chunk, self__.more, m, self__.__hash);\n};\ncljs.core.ChunkedCons.prototype.cljs$core$ICollection$_conj$arity$2 = function(this$, o) {\n  var self__ = this;\n  var this$__$1 = this;\n  return cljs.core.cons.call(null, o, this$__$1);\n};\ncljs.core.ChunkedCons.prototype.cljs$core$IChunkedNext$_chunked_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.more == null) {\n    return null;\n  } else {\n    return self__.more;\n  }\n};\ncljs.core.__GT_ChunkedCons = function __GT_ChunkedCons(chunk, more, meta, __hash) {\n  return new cljs.core.ChunkedCons(chunk, more, meta, __hash);\n};\ncljs.core.chunk_cons = function chunk_cons(chunk, rest) {\n  if (cljs.core._count.call(null, chunk) === 0) {\n    return rest;\n  } else {\n    return new cljs.core.ChunkedCons(chunk, rest, null, null);\n  }\n};\ncljs.core.chunk_append = function chunk_append(b, x) {\n  return b.add(x);\n};\ncljs.core.chunk = function chunk(b) {\n  return b.chunk();\n};\ncljs.core.chunk_first = function chunk_first(s) {\n  return cljs.core._chunked_first.call(null, s);\n};\ncljs.core.chunk_rest = function chunk_rest(s) {\n  return cljs.core._chunked_rest.call(null, s);\n};\ncljs.core.chunk_next = function chunk_next(s) {\n  if (function() {\n    var G__22447 = s;\n    if (G__22447) {\n      var bit__17788__auto__ = G__22447.cljs$lang$protocol_mask$partition1$ &amp; 1024;\n      if (bit__17788__auto__ || G__22447.cljs$core$IChunkedNext$) {\n        return true;\n      } else {\n        return false;\n      }\n    } else {\n      return false;\n    }\n  }()) {\n    return cljs.core._chunked_next.call(null, s);\n  } else {\n    return cljs.core.seq.call(null, cljs.core._chunked_rest.call(null, s));\n  }\n};\ncljs.core.to_array = function to_array(s) {\n  var ary = [];\n  var s__$1 = s;\n  while (true) {\n    if (cljs.core.seq.call(null, s__$1)) {\n      ary.push(cljs.core.first.call(null, s__$1));\n      var G__22448 = cljs.core.next.call(null, s__$1);\n      s__$1 = G__22448;\n      continue;\n    } else {\n      return ary;\n    }\n    break;\n  }\n};\ncljs.core.to_array_2d = function to_array_2d(coll) {\n  var ret = new Array(cljs.core.count.call(null, coll));\n  var i_22449 = 0;\n  var xs_22450 = cljs.core.seq.call(null, coll);\n  while (true) {\n    if (xs_22450) {\n      ret[i_22449] = cljs.core.to_array.call(null, cljs.core.first.call(null, xs_22450));\n      var G__22451 = i_22449 + 1;\n      var G__22452 = cljs.core.next.call(null, xs_22450);\n      i_22449 = G__22451;\n      xs_22450 = G__22452;\n      continue;\n    } else {\n    }\n    break;\n  }\n  return ret;\n};\ncljs.core.int_array = function() {\n  var int_array = null;\n  var int_array__1 = function(size_or_seq) {\n    if (typeof size_or_seq === &quot;number&quot;) {\n      return int_array.call(null, size_or_seq, null);\n    } else {\n      return cljs.core.into_array.call(null, size_or_seq);\n    }\n  };\n  var int_array__2 = function(size, init_val_or_seq) {\n    var a = new Array(size);\n    if (cljs.core.seq_QMARK_.call(null, init_val_or_seq)) {\n      var s = cljs.core.seq.call(null, init_val_or_seq);\n      var i = 0;\n      var s__$1 = s;\n      while (true) {\n        if (s__$1 &amp;&amp; i &lt; size) {\n          a[i] = cljs.core.first.call(null, s__$1);\n          var G__22453 = i + 1;\n          var G__22454 = cljs.core.next.call(null, s__$1);\n          i = G__22453;\n          s__$1 = G__22454;\n          continue;\n        } else {\n          return a;\n        }\n        break;\n      }\n    } else {\n      var n__18001__auto___22455 = size;\n      var i_22456 = 0;\n      while (true) {\n        if (i_22456 &lt; n__18001__auto___22455) {\n          a[i_22456] = init_val_or_seq;\n          var G__22457 = i_22456 + 1;\n          i_22456 = G__22457;\n          continue;\n        } else {\n        }\n        break;\n      }\n      return a;\n    }\n  };\n  int_array = function(size, init_val_or_seq) {\n    switch(arguments.length) {\n      case 1:\n        return int_array__1.call(this, size);\n      case 2:\n        return int_array__2.call(this, size, init_val_or_seq);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  int_array.cljs$core$IFn$_invoke$arity$1 = int_array__1;\n  int_array.cljs$core$IFn$_invoke$arity$2 = int_array__2;\n  return int_array;\n}();\ncljs.core.long_array = function() {\n  var long_array = null;\n  var long_array__1 = function(size_or_seq) {\n    if (typeof size_or_seq === &quot;number&quot;) {\n      return long_array.call(null, size_or_seq, null);\n    } else {\n      return cljs.core.into_array.call(null, size_or_seq);\n    }\n  };\n  var long_array__2 = function(size, init_val_or_seq) {\n    var a = new Array(size);\n    if (cljs.core.seq_QMARK_.call(null, init_val_or_seq)) {\n      var s = cljs.core.seq.call(null, init_val_or_seq);\n      var i = 0;\n      var s__$1 = s;\n      while (true) {\n        if (s__$1 &amp;&amp; i &lt; size) {\n          a[i] = cljs.core.first.call(null, s__$1);\n          var G__22458 = i + 1;\n          var G__22459 = cljs.core.next.call(null, s__$1);\n          i = G__22458;\n          s__$1 = G__22459;\n          continue;\n        } else {\n          return a;\n        }\n        break;\n      }\n    } else {\n      var n__18001__auto___22460 = size;\n      var i_22461 = 0;\n      while (true) {\n        if (i_22461 &lt; n__18001__auto___22460) {\n          a[i_22461] = init_val_or_seq;\n          var G__22462 = i_22461 + 1;\n          i_22461 = G__22462;\n          continue;\n        } else {\n        }\n        break;\n      }\n      return a;\n    }\n  };\n  long_array = function(size, init_val_or_seq) {\n    switch(arguments.length) {\n      case 1:\n        return long_array__1.call(this, size);\n      case 2:\n        return long_array__2.call(this, size, init_val_or_seq);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  long_array.cljs$core$IFn$_invoke$arity$1 = long_array__1;\n  long_array.cljs$core$IFn$_invoke$arity$2 = long_array__2;\n  return long_array;\n}();\ncljs.core.double_array = function() {\n  var double_array = null;\n  var double_array__1 = function(size_or_seq) {\n    if (typeof size_or_seq === &quot;number&quot;) {\n      return double_array.call(null, size_or_seq, null);\n    } else {\n      return cljs.core.into_array.call(null, size_or_seq);\n    }\n  };\n  var double_array__2 = function(size, init_val_or_seq) {\n    var a = new Array(size);\n    if (cljs.core.seq_QMARK_.call(null, init_val_or_seq)) {\n      var s = cljs.core.seq.call(null, init_val_or_seq);\n      var i = 0;\n      var s__$1 = s;\n      while (true) {\n        if (s__$1 &amp;&amp; i &lt; size) {\n          a[i] = cljs.core.first.call(null, s__$1);\n          var G__22463 = i + 1;\n          var G__22464 = cljs.core.next.call(null, s__$1);\n          i = G__22463;\n          s__$1 = G__22464;\n          continue;\n        } else {\n          return a;\n        }\n        break;\n      }\n    } else {\n      var n__18001__auto___22465 = size;\n      var i_22466 = 0;\n      while (true) {\n        if (i_22466 &lt; n__18001__auto___22465) {\n          a[i_22466] = init_val_or_seq;\n          var G__22467 = i_22466 + 1;\n          i_22466 = G__22467;\n          continue;\n        } else {\n        }\n        break;\n      }\n      return a;\n    }\n  };\n  double_array = function(size, init_val_or_seq) {\n    switch(arguments.length) {\n      case 1:\n        return double_array__1.call(this, size);\n      case 2:\n        return double_array__2.call(this, size, init_val_or_seq);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  double_array.cljs$core$IFn$_invoke$arity$1 = double_array__1;\n  double_array.cljs$core$IFn$_invoke$arity$2 = double_array__2;\n  return double_array;\n}();\ncljs.core.object_array = function() {\n  var object_array = null;\n  var object_array__1 = function(size_or_seq) {\n    if (typeof size_or_seq === &quot;number&quot;) {\n      return object_array.call(null, size_or_seq, null);\n    } else {\n      return cljs.core.into_array.call(null, size_or_seq);\n    }\n  };\n  var object_array__2 = function(size, init_val_or_seq) {\n    var a = new Array(size);\n    if (cljs.core.seq_QMARK_.call(null, init_val_or_seq)) {\n      var s = cljs.core.seq.call(null, init_val_or_seq);\n      var i = 0;\n      var s__$1 = s;\n      while (true) {\n        if (s__$1 &amp;&amp; i &lt; size) {\n          a[i] = cljs.core.first.call(null, s__$1);\n          var G__22468 = i + 1;\n          var G__22469 = cljs.core.next.call(null, s__$1);\n          i = G__22468;\n          s__$1 = G__22469;\n          continue;\n        } else {\n          return a;\n        }\n        break;\n      }\n    } else {\n      var n__18001__auto___22470 = size;\n      var i_22471 = 0;\n      while (true) {\n        if (i_22471 &lt; n__18001__auto___22470) {\n          a[i_22471] = init_val_or_seq;\n          var G__22472 = i_22471 + 1;\n          i_22471 = G__22472;\n          continue;\n        } else {\n        }\n        break;\n      }\n      return a;\n    }\n  };\n  object_array = function(size, init_val_or_seq) {\n    switch(arguments.length) {\n      case 1:\n        return object_array__1.call(this, size);\n      case 2:\n        return object_array__2.call(this, size, init_val_or_seq);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  object_array.cljs$core$IFn$_invoke$arity$1 = object_array__1;\n  object_array.cljs$core$IFn$_invoke$arity$2 = object_array__2;\n  return object_array;\n}();\ncljs.core.bounded_count = function bounded_count(s, n) {\n  if (cljs.core.counted_QMARK_.call(null, s)) {\n    return cljs.core.count.call(null, s);\n  } else {\n    var s__$1 = s;\n    var i = n;\n    var sum = 0;\n    while (true) {\n      if (i &gt; 0 &amp;&amp; cljs.core.seq.call(null, s__$1)) {\n        var G__22473 = cljs.core.next.call(null, s__$1);\n        var G__22474 = i - 1;\n        var G__22475 = sum + 1;\n        s__$1 = G__22473;\n        i = G__22474;\n        sum = G__22475;\n        continue;\n      } else {\n        return sum;\n      }\n      break;\n    }\n  }\n};\ncljs.core.spread = function spread(arglist) {\n  if (arglist == null) {\n    return null;\n  } else {\n    if (cljs.core.next.call(null, arglist) == null) {\n      return cljs.core.seq.call(null, cljs.core.first.call(null, arglist));\n    } else {\n      return cljs.core.cons.call(null, cljs.core.first.call(null, arglist), spread.call(null, cljs.core.next.call(null, arglist)));\n    }\n  }\n};\ncljs.core.concat = function() {\n  var concat = null;\n  var concat__0 = function() {\n    return new cljs.core.LazySeq(null, function() {\n      return null;\n    }, null, null);\n  };\n  var concat__1 = function(x) {\n    return new cljs.core.LazySeq(null, function() {\n      return x;\n    }, null, null);\n  };\n  var concat__2 = function(x, y) {\n    return new cljs.core.LazySeq(null, function() {\n      var s = cljs.core.seq.call(null, x);\n      if (s) {\n        if (cljs.core.chunked_seq_QMARK_.call(null, s)) {\n          return cljs.core.chunk_cons.call(null, cljs.core.chunk_first.call(null, s), concat.call(null, cljs.core.chunk_rest.call(null, s), y));\n        } else {\n          return cljs.core.cons.call(null, cljs.core.first.call(null, s), concat.call(null, cljs.core.rest.call(null, s), y));\n        }\n      } else {\n        return y;\n      }\n    }, null, null);\n  };\n  var concat__3 = function() {\n    var G__22476__delegate = function(x, y, zs) {\n      var cat = function cat(xys, zs__$1) {\n        return new cljs.core.LazySeq(null, function() {\n          var xys__$1 = cljs.core.seq.call(null, xys);\n          if (xys__$1) {\n            if (cljs.core.chunked_seq_QMARK_.call(null, xys__$1)) {\n              return cljs.core.chunk_cons.call(null, cljs.core.chunk_first.call(null, xys__$1), cat.call(null, cljs.core.chunk_rest.call(null, xys__$1), zs__$1));\n            } else {\n              return cljs.core.cons.call(null, cljs.core.first.call(null, xys__$1), cat.call(null, cljs.core.rest.call(null, xys__$1), zs__$1));\n            }\n          } else {\n            if (cljs.core.truth_(zs__$1)) {\n              return cat.call(null, cljs.core.first.call(null, zs__$1), cljs.core.next.call(null, zs__$1));\n            } else {\n              return null;\n            }\n          }\n        }, null, null);\n      };\n      return cat.call(null, concat.call(null, x, y), zs);\n    };\n    var G__22476 = function(x, y, var_args) {\n      var zs = null;\n      if (arguments.length &gt; 2) {\n        zs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22476__delegate.call(this, x, y, zs);\n    };\n    G__22476.cljs$lang$maxFixedArity = 2;\n    G__22476.cljs$lang$applyTo = function(arglist__22477) {\n      var x = cljs.core.first(arglist__22477);\n      arglist__22477 = cljs.core.next(arglist__22477);\n      var y = cljs.core.first(arglist__22477);\n      var zs = cljs.core.rest(arglist__22477);\n      return G__22476__delegate(x, y, zs);\n    };\n    G__22476.cljs$core$IFn$_invoke$arity$variadic = G__22476__delegate;\n    return G__22476;\n  }();\n  concat = function(x, y, var_args) {\n    var zs = var_args;\n    switch(arguments.length) {\n      case 0:\n        return concat__0.call(this);\n      case 1:\n        return concat__1.call(this, x);\n      case 2:\n        return concat__2.call(this, x, y);\n      default:\n        return concat__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  concat.cljs$lang$maxFixedArity = 2;\n  concat.cljs$lang$applyTo = concat__3.cljs$lang$applyTo;\n  concat.cljs$core$IFn$_invoke$arity$0 = concat__0;\n  concat.cljs$core$IFn$_invoke$arity$1 = concat__1;\n  concat.cljs$core$IFn$_invoke$arity$2 = concat__2;\n  concat.cljs$core$IFn$_invoke$arity$variadic = concat__3.cljs$core$IFn$_invoke$arity$variadic;\n  return concat;\n}();\ncljs.core.list_STAR_ = function() {\n  var list_STAR_ = null;\n  var list_STAR___1 = function(args) {\n    return cljs.core.seq.call(null, args);\n  };\n  var list_STAR___2 = function(a, args) {\n    return cljs.core.cons.call(null, a, args);\n  };\n  var list_STAR___3 = function(a, b, args) {\n    return cljs.core.cons.call(null, a, cljs.core.cons.call(null, b, args));\n  };\n  var list_STAR___4 = function(a, b, c, args) {\n    return cljs.core.cons.call(null, a, cljs.core.cons.call(null, b, cljs.core.cons.call(null, c, args)));\n  };\n  var list_STAR___5 = function() {\n    var G__22478__delegate = function(a, b, c, d, more) {\n      return cljs.core.cons.call(null, a, cljs.core.cons.call(null, b, cljs.core.cons.call(null, c, cljs.core.cons.call(null, d, cljs.core.spread.call(null, more)))));\n    };\n    var G__22478 = function(a, b, c, d, var_args) {\n      var more = null;\n      if (arguments.length &gt; 4) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 4), 0);\n      }\n      return G__22478__delegate.call(this, a, b, c, d, more);\n    };\n    G__22478.cljs$lang$maxFixedArity = 4;\n    G__22478.cljs$lang$applyTo = function(arglist__22479) {\n      var a = cljs.core.first(arglist__22479);\n      arglist__22479 = cljs.core.next(arglist__22479);\n      var b = cljs.core.first(arglist__22479);\n      arglist__22479 = cljs.core.next(arglist__22479);\n      var c = cljs.core.first(arglist__22479);\n      arglist__22479 = cljs.core.next(arglist__22479);\n      var d = cljs.core.first(arglist__22479);\n      var more = cljs.core.rest(arglist__22479);\n      return G__22478__delegate(a, b, c, d, more);\n    };\n    G__22478.cljs$core$IFn$_invoke$arity$variadic = G__22478__delegate;\n    return G__22478;\n  }();\n  list_STAR_ = function(a, b, c, d, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return list_STAR___1.call(this, a);\n      case 2:\n        return list_STAR___2.call(this, a, b);\n      case 3:\n        return list_STAR___3.call(this, a, b, c);\n      case 4:\n        return list_STAR___4.call(this, a, b, c, d);\n      default:\n        return list_STAR___5.cljs$core$IFn$_invoke$arity$variadic(a, b, c, d, cljs.core.array_seq(arguments, 4));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  list_STAR_.cljs$lang$maxFixedArity = 4;\n  list_STAR_.cljs$lang$applyTo = list_STAR___5.cljs$lang$applyTo;\n  list_STAR_.cljs$core$IFn$_invoke$arity$1 = list_STAR___1;\n  list_STAR_.cljs$core$IFn$_invoke$arity$2 = list_STAR___2;\n  list_STAR_.cljs$core$IFn$_invoke$arity$3 = list_STAR___3;\n  list_STAR_.cljs$core$IFn$_invoke$arity$4 = list_STAR___4;\n  list_STAR_.cljs$core$IFn$_invoke$arity$variadic = list_STAR___5.cljs$core$IFn$_invoke$arity$variadic;\n  return list_STAR_;\n}();\ncljs.core.transient$ = function transient$(coll) {\n  return cljs.core._as_transient.call(null, coll);\n};\ncljs.core.persistent_BANG_ = function persistent_BANG_(tcoll) {\n  return cljs.core._persistent_BANG_.call(null, tcoll);\n};\ncljs.core.conj_BANG_ = function() {\n  var conj_BANG_ = null;\n  var conj_BANG___0 = function() {\n    return cljs.core.transient$.call(null, cljs.core.PersistentVector.EMPTY);\n  };\n  var conj_BANG___1 = function(coll) {\n    return coll;\n  };\n  var conj_BANG___2 = function(tcoll, val) {\n    return cljs.core._conj_BANG_.call(null, tcoll, val);\n  };\n  var conj_BANG___3 = function() {\n    var G__22480__delegate = function(tcoll, val, vals) {\n      while (true) {\n        var ntcoll = cljs.core._conj_BANG_.call(null, tcoll, val);\n        if (cljs.core.truth_(vals)) {\n          var G__22481 = ntcoll;\n          var G__22482 = cljs.core.first.call(null, vals);\n          var G__22483 = cljs.core.next.call(null, vals);\n          tcoll = G__22481;\n          val = G__22482;\n          vals = G__22483;\n          continue;\n        } else {\n          return ntcoll;\n        }\n        break;\n      }\n    };\n    var G__22480 = function(tcoll, val, var_args) {\n      var vals = null;\n      if (arguments.length &gt; 2) {\n        vals = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22480__delegate.call(this, tcoll, val, vals);\n    };\n    G__22480.cljs$lang$maxFixedArity = 2;\n    G__22480.cljs$lang$applyTo = function(arglist__22484) {\n      var tcoll = cljs.core.first(arglist__22484);\n      arglist__22484 = cljs.core.next(arglist__22484);\n      var val = cljs.core.first(arglist__22484);\n      var vals = cljs.core.rest(arglist__22484);\n      return G__22480__delegate(tcoll, val, vals);\n    };\n    G__22480.cljs$core$IFn$_invoke$arity$variadic = G__22480__delegate;\n    return G__22480;\n  }();\n  conj_BANG_ = function(tcoll, val, var_args) {\n    var vals = var_args;\n    switch(arguments.length) {\n      case 0:\n        return conj_BANG___0.call(this);\n      case 1:\n        return conj_BANG___1.call(this, tcoll);\n      case 2:\n        return conj_BANG___2.call(this, tcoll, val);\n      default:\n        return conj_BANG___3.cljs$core$IFn$_invoke$arity$variadic(tcoll, val, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  conj_BANG_.cljs$lang$maxFixedArity = 2;\n  conj_BANG_.cljs$lang$applyTo = conj_BANG___3.cljs$lang$applyTo;\n  conj_BANG_.cljs$core$IFn$_invoke$arity$0 = conj_BANG___0;\n  conj_BANG_.cljs$core$IFn$_invoke$arity$1 = conj_BANG___1;\n  conj_BANG_.cljs$core$IFn$_invoke$arity$2 = conj_BANG___2;\n  conj_BANG_.cljs$core$IFn$_invoke$arity$variadic = conj_BANG___3.cljs$core$IFn$_invoke$arity$variadic;\n  return conj_BANG_;\n}();\ncljs.core.assoc_BANG_ = function() {\n  var assoc_BANG_ = null;\n  var assoc_BANG___3 = function(tcoll, key, val) {\n    return cljs.core._assoc_BANG_.call(null, tcoll, key, val);\n  };\n  var assoc_BANG___4 = function() {\n    var G__22485__delegate = function(tcoll, key, val, kvs) {\n      while (true) {\n        var ntcoll = cljs.core._assoc_BANG_.call(null, tcoll, key, val);\n        if (cljs.core.truth_(kvs)) {\n          var G__22486 = ntcoll;\n          var G__22487 = cljs.core.first.call(null, kvs);\n          var G__22488 = cljs.core.second.call(null, kvs);\n          var G__22489 = cljs.core.nnext.call(null, kvs);\n          tcoll = G__22486;\n          key = G__22487;\n          val = G__22488;\n          kvs = G__22489;\n          continue;\n        } else {\n          return ntcoll;\n        }\n        break;\n      }\n    };\n    var G__22485 = function(tcoll, key, val, var_args) {\n      var kvs = null;\n      if (arguments.length &gt; 3) {\n        kvs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n      }\n      return G__22485__delegate.call(this, tcoll, key, val, kvs);\n    };\n    G__22485.cljs$lang$maxFixedArity = 3;\n    G__22485.cljs$lang$applyTo = function(arglist__22490) {\n      var tcoll = cljs.core.first(arglist__22490);\n      arglist__22490 = cljs.core.next(arglist__22490);\n      var key = cljs.core.first(arglist__22490);\n      arglist__22490 = cljs.core.next(arglist__22490);\n      var val = cljs.core.first(arglist__22490);\n      var kvs = cljs.core.rest(arglist__22490);\n      return G__22485__delegate(tcoll, key, val, kvs);\n    };\n    G__22485.cljs$core$IFn$_invoke$arity$variadic = G__22485__delegate;\n    return G__22485;\n  }();\n  assoc_BANG_ = function(tcoll, key, val, var_args) {\n    var kvs = var_args;\n    switch(arguments.length) {\n      case 3:\n        return assoc_BANG___3.call(this, tcoll, key, val);\n      default:\n        return assoc_BANG___4.cljs$core$IFn$_invoke$arity$variadic(tcoll, key, val, cljs.core.array_seq(arguments, 3));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  assoc_BANG_.cljs$lang$maxFixedArity = 3;\n  assoc_BANG_.cljs$lang$applyTo = assoc_BANG___4.cljs$lang$applyTo;\n  assoc_BANG_.cljs$core$IFn$_invoke$arity$3 = assoc_BANG___3;\n  assoc_BANG_.cljs$core$IFn$_invoke$arity$variadic = assoc_BANG___4.cljs$core$IFn$_invoke$arity$variadic;\n  return assoc_BANG_;\n}();\ncljs.core.dissoc_BANG_ = function() {\n  var dissoc_BANG_ = null;\n  var dissoc_BANG___2 = function(tcoll, key) {\n    return cljs.core._dissoc_BANG_.call(null, tcoll, key);\n  };\n  var dissoc_BANG___3 = function() {\n    var G__22491__delegate = function(tcoll, key, ks) {\n      while (true) {\n        var ntcoll = cljs.core._dissoc_BANG_.call(null, tcoll, key);\n        if (cljs.core.truth_(ks)) {\n          var G__22492 = ntcoll;\n          var G__22493 = cljs.core.first.call(null, ks);\n          var G__22494 = cljs.core.next.call(null, ks);\n          tcoll = G__22492;\n          key = G__22493;\n          ks = G__22494;\n          continue;\n        } else {\n          return ntcoll;\n        }\n        break;\n      }\n    };\n    var G__22491 = function(tcoll, key, var_args) {\n      var ks = null;\n      if (arguments.length &gt; 2) {\n        ks = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22491__delegate.call(this, tcoll, key, ks);\n    };\n    G__22491.cljs$lang$maxFixedArity = 2;\n    G__22491.cljs$lang$applyTo = function(arglist__22495) {\n      var tcoll = cljs.core.first(arglist__22495);\n      arglist__22495 = cljs.core.next(arglist__22495);\n      var key = cljs.core.first(arglist__22495);\n      var ks = cljs.core.rest(arglist__22495);\n      return G__22491__delegate(tcoll, key, ks);\n    };\n    G__22491.cljs$core$IFn$_invoke$arity$variadic = G__22491__delegate;\n    return G__22491;\n  }();\n  dissoc_BANG_ = function(tcoll, key, var_args) {\n    var ks = var_args;\n    switch(arguments.length) {\n      case 2:\n        return dissoc_BANG___2.call(this, tcoll, key);\n      default:\n        return dissoc_BANG___3.cljs$core$IFn$_invoke$arity$variadic(tcoll, key, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  dissoc_BANG_.cljs$lang$maxFixedArity = 2;\n  dissoc_BANG_.cljs$lang$applyTo = dissoc_BANG___3.cljs$lang$applyTo;\n  dissoc_BANG_.cljs$core$IFn$_invoke$arity$2 = dissoc_BANG___2;\n  dissoc_BANG_.cljs$core$IFn$_invoke$arity$variadic = dissoc_BANG___3.cljs$core$IFn$_invoke$arity$variadic;\n  return dissoc_BANG_;\n}();\ncljs.core.pop_BANG_ = function pop_BANG_(tcoll) {\n  return cljs.core._pop_BANG_.call(null, tcoll);\n};\ncljs.core.disj_BANG_ = function() {\n  var disj_BANG_ = null;\n  var disj_BANG___2 = function(tcoll, val) {\n    return cljs.core._disjoin_BANG_.call(null, tcoll, val);\n  };\n  var disj_BANG___3 = function() {\n    var G__22496__delegate = function(tcoll, val, vals) {\n      while (true) {\n        var ntcoll = cljs.core._disjoin_BANG_.call(null, tcoll, val);\n        if (cljs.core.truth_(vals)) {\n          var G__22497 = ntcoll;\n          var G__22498 = cljs.core.first.call(null, vals);\n          var G__22499 = cljs.core.next.call(null, vals);\n          tcoll = G__22497;\n          val = G__22498;\n          vals = G__22499;\n          continue;\n        } else {\n          return ntcoll;\n        }\n        break;\n      }\n    };\n    var G__22496 = function(tcoll, val, var_args) {\n      var vals = null;\n      if (arguments.length &gt; 2) {\n        vals = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22496__delegate.call(this, tcoll, val, vals);\n    };\n    G__22496.cljs$lang$maxFixedArity = 2;\n    G__22496.cljs$lang$applyTo = function(arglist__22500) {\n      var tcoll = cljs.core.first(arglist__22500);\n      arglist__22500 = cljs.core.next(arglist__22500);\n      var val = cljs.core.first(arglist__22500);\n      var vals = cljs.core.rest(arglist__22500);\n      return G__22496__delegate(tcoll, val, vals);\n    };\n    G__22496.cljs$core$IFn$_invoke$arity$variadic = G__22496__delegate;\n    return G__22496;\n  }();\n  disj_BANG_ = function(tcoll, val, var_args) {\n    var vals = var_args;\n    switch(arguments.length) {\n      case 2:\n        return disj_BANG___2.call(this, tcoll, val);\n      default:\n        return disj_BANG___3.cljs$core$IFn$_invoke$arity$variadic(tcoll, val, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  disj_BANG_.cljs$lang$maxFixedArity = 2;\n  disj_BANG_.cljs$lang$applyTo = disj_BANG___3.cljs$lang$applyTo;\n  disj_BANG_.cljs$core$IFn$_invoke$arity$2 = disj_BANG___2;\n  disj_BANG_.cljs$core$IFn$_invoke$arity$variadic = disj_BANG___3.cljs$core$IFn$_invoke$arity$variadic;\n  return disj_BANG_;\n}();\ncljs.core.apply_to = function apply_to(f, argc, args) {\n  var args__$1 = cljs.core.seq.call(null, args);\n  if (argc === 0) {\n    return f.call(null);\n  } else {\n    var a18045 = cljs.core._first.call(null, args__$1);\n    var args__$2 = cljs.core._rest.call(null, args__$1);\n    if (argc === 1) {\n      if (f.cljs$core$IFn$_invoke$arity$1) {\n        return f.cljs$core$IFn$_invoke$arity$1(a18045);\n      } else {\n        return f.call(null, a18045);\n      }\n    } else {\n      var b18046 = cljs.core._first.call(null, args__$2);\n      var args__$3 = cljs.core._rest.call(null, args__$2);\n      if (argc === 2) {\n        if (f.cljs$core$IFn$_invoke$arity$2) {\n          return f.cljs$core$IFn$_invoke$arity$2(a18045, b18046);\n        } else {\n          return f.call(null, a18045, b18046);\n        }\n      } else {\n        var c18047 = cljs.core._first.call(null, args__$3);\n        var args__$4 = cljs.core._rest.call(null, args__$3);\n        if (argc === 3) {\n          if (f.cljs$core$IFn$_invoke$arity$3) {\n            return f.cljs$core$IFn$_invoke$arity$3(a18045, b18046, c18047);\n          } else {\n            return f.call(null, a18045, b18046, c18047);\n          }\n        } else {\n          var d18048 = cljs.core._first.call(null, args__$4);\n          var args__$5 = cljs.core._rest.call(null, args__$4);\n          if (argc === 4) {\n            if (f.cljs$core$IFn$_invoke$arity$4) {\n              return f.cljs$core$IFn$_invoke$arity$4(a18045, b18046, c18047, d18048);\n            } else {\n              return f.call(null, a18045, b18046, c18047, d18048);\n            }\n          } else {\n            var e18049 = cljs.core._first.call(null, args__$5);\n            var args__$6 = cljs.core._rest.call(null, args__$5);\n            if (argc === 5) {\n              if (f.cljs$core$IFn$_invoke$arity$5) {\n                return f.cljs$core$IFn$_invoke$arity$5(a18045, b18046, c18047, d18048, e18049);\n              } else {\n                return f.call(null, a18045, b18046, c18047, d18048, e18049);\n              }\n            } else {\n              var f18050 = cljs.core._first.call(null, args__$6);\n              var args__$7 = cljs.core._rest.call(null, args__$6);\n              if (argc === 6) {\n                if (f.cljs$core$IFn$_invoke$arity$6) {\n                  return f.cljs$core$IFn$_invoke$arity$6(a18045, b18046, c18047, d18048, e18049, f18050);\n                } else {\n                  return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050);\n                }\n              } else {\n                var g18051 = cljs.core._first.call(null, args__$7);\n                var args__$8 = cljs.core._rest.call(null, args__$7);\n                if (argc === 7) {\n                  if (f.cljs$core$IFn$_invoke$arity$7) {\n                    return f.cljs$core$IFn$_invoke$arity$7(a18045, b18046, c18047, d18048, e18049, f18050, g18051);\n                  } else {\n                    return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051);\n                  }\n                } else {\n                  var h18052 = cljs.core._first.call(null, args__$8);\n                  var args__$9 = cljs.core._rest.call(null, args__$8);\n                  if (argc === 8) {\n                    if (f.cljs$core$IFn$_invoke$arity$8) {\n                      return f.cljs$core$IFn$_invoke$arity$8(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052);\n                    } else {\n                      return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052);\n                    }\n                  } else {\n                    var i18053 = cljs.core._first.call(null, args__$9);\n                    var args__$10 = cljs.core._rest.call(null, args__$9);\n                    if (argc === 9) {\n                      if (f.cljs$core$IFn$_invoke$arity$9) {\n                        return f.cljs$core$IFn$_invoke$arity$9(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053);\n                      } else {\n                        return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053);\n                      }\n                    } else {\n                      var j18054 = cljs.core._first.call(null, args__$10);\n                      var args__$11 = cljs.core._rest.call(null, args__$10);\n                      if (argc === 10) {\n                        if (f.cljs$core$IFn$_invoke$arity$10) {\n                          return f.cljs$core$IFn$_invoke$arity$10(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054);\n                        } else {\n                          return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054);\n                        }\n                      } else {\n                        var k18055 = cljs.core._first.call(null, args__$11);\n                        var args__$12 = cljs.core._rest.call(null, args__$11);\n                        if (argc === 11) {\n                          if (f.cljs$core$IFn$_invoke$arity$11) {\n                            return f.cljs$core$IFn$_invoke$arity$11(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055);\n                          } else {\n                            return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055);\n                          }\n                        } else {\n                          var l18056 = cljs.core._first.call(null, args__$12);\n                          var args__$13 = cljs.core._rest.call(null, args__$12);\n                          if (argc === 12) {\n                            if (f.cljs$core$IFn$_invoke$arity$12) {\n                              return f.cljs$core$IFn$_invoke$arity$12(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056);\n                            } else {\n                              return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056);\n                            }\n                          } else {\n                            var m18057 = cljs.core._first.call(null, args__$13);\n                            var args__$14 = cljs.core._rest.call(null, args__$13);\n                            if (argc === 13) {\n                              if (f.cljs$core$IFn$_invoke$arity$13) {\n                                return f.cljs$core$IFn$_invoke$arity$13(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057);\n                              } else {\n                                return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057);\n                              }\n                            } else {\n                              var n18058 = cljs.core._first.call(null, args__$14);\n                              var args__$15 = cljs.core._rest.call(null, args__$14);\n                              if (argc === 14) {\n                                if (f.cljs$core$IFn$_invoke$arity$14) {\n                                  return f.cljs$core$IFn$_invoke$arity$14(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058);\n                                } else {\n                                  return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058);\n                                }\n                              } else {\n                                var o18059 = cljs.core._first.call(null, args__$15);\n                                var args__$16 = cljs.core._rest.call(null, args__$15);\n                                if (argc === 15) {\n                                  if (f.cljs$core$IFn$_invoke$arity$15) {\n                                    return f.cljs$core$IFn$_invoke$arity$15(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059);\n                                  } else {\n                                    return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059);\n                                  }\n                                } else {\n                                  var p18060 = cljs.core._first.call(null, args__$16);\n                                  var args__$17 = cljs.core._rest.call(null, args__$16);\n                                  if (argc === 16) {\n                                    if (f.cljs$core$IFn$_invoke$arity$16) {\n                                      return f.cljs$core$IFn$_invoke$arity$16(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060);\n                                    } else {\n                                      return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060);\n                                    }\n                                  } else {\n                                    var q18061 = cljs.core._first.call(null, args__$17);\n                                    var args__$18 = cljs.core._rest.call(null, args__$17);\n                                    if (argc === 17) {\n                                      if (f.cljs$core$IFn$_invoke$arity$17) {\n                                        return f.cljs$core$IFn$_invoke$arity$17(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060, q18061);\n                                      } else {\n                                        return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060, q18061);\n                                      }\n                                    } else {\n                                      var r18062 = cljs.core._first.call(null, args__$18);\n                                      var args__$19 = cljs.core._rest.call(null, args__$18);\n                                      if (argc === 18) {\n                                        if (f.cljs$core$IFn$_invoke$arity$18) {\n                                          return f.cljs$core$IFn$_invoke$arity$18(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060, q18061, r18062);\n                                        } else {\n                                          return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060, q18061, r18062);\n                                        }\n                                      } else {\n                                        var s18063 = cljs.core._first.call(null, args__$19);\n                                        var args__$20 = cljs.core._rest.call(null, args__$19);\n                                        if (argc === 19) {\n                                          if (f.cljs$core$IFn$_invoke$arity$19) {\n                                            return f.cljs$core$IFn$_invoke$arity$19(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060, q18061, r18062, s18063);\n                                          } else {\n                                            return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060, q18061, r18062, s18063);\n                                          }\n                                        } else {\n                                          var t18064 = cljs.core._first.call(null, args__$20);\n                                          var args__$21 = cljs.core._rest.call(null, args__$20);\n                                          if (argc === 20) {\n                                            if (f.cljs$core$IFn$_invoke$arity$20) {\n                                              return f.cljs$core$IFn$_invoke$arity$20(a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060, q18061, r18062, s18063, t18064);\n                                            } else {\n                                              return f.call(null, a18045, b18046, c18047, d18048, e18049, f18050, g18051, h18052, i18053, j18054, k18055, l18056, m18057, n18058, o18059, p18060, q18061, r18062, s18063, t18064);\n                                            }\n                                          } else {\n                                            throw new Error(&quot;Only up to 20 arguments supported on functions&quot;);\n                                          }\n                                        }\n                                      }\n                                    }\n                                  }\n                                }\n                              }\n                            }\n                          }\n                        }\n                      }\n                    }\n                  }\n                }\n              }\n            }\n          }\n        }\n      }\n    }\n  }\n};\ncljs.core.apply = function() {\n  var apply = null;\n  var apply__2 = function(f, args) {\n    var fixed_arity = f.cljs$lang$maxFixedArity;\n    if (f.cljs$lang$applyTo) {\n      var bc = cljs.core.bounded_count.call(null, args, fixed_arity + 1);\n      if (bc &lt;= fixed_arity) {\n        return cljs.core.apply_to.call(null, f, bc, args);\n      } else {\n        return f.cljs$lang$applyTo(args);\n      }\n    } else {\n      return f.apply(f, cljs.core.to_array.call(null, args));\n    }\n  };\n  var apply__3 = function(f, x, args) {\n    var arglist = cljs.core.list_STAR_.call(null, x, args);\n    var fixed_arity = f.cljs$lang$maxFixedArity;\n    if (f.cljs$lang$applyTo) {\n      var bc = cljs.core.bounded_count.call(null, arglist, fixed_arity + 1);\n      if (bc &lt;= fixed_arity) {\n        return cljs.core.apply_to.call(null, f, bc, arglist);\n      } else {\n        return f.cljs$lang$applyTo(arglist);\n      }\n    } else {\n      return f.apply(f, cljs.core.to_array.call(null, arglist));\n    }\n  };\n  var apply__4 = function(f, x, y, args) {\n    var arglist = cljs.core.list_STAR_.call(null, x, y, args);\n    var fixed_arity = f.cljs$lang$maxFixedArity;\n    if (f.cljs$lang$applyTo) {\n      var bc = cljs.core.bounded_count.call(null, arglist, fixed_arity + 1);\n      if (bc &lt;= fixed_arity) {\n        return cljs.core.apply_to.call(null, f, bc, arglist);\n      } else {\n        return f.cljs$lang$applyTo(arglist);\n      }\n    } else {\n      return f.apply(f, cljs.core.to_array.call(null, arglist));\n    }\n  };\n  var apply__5 = function(f, x, y, z, args) {\n    var arglist = cljs.core.list_STAR_.call(null, x, y, z, args);\n    var fixed_arity = f.cljs$lang$maxFixedArity;\n    if (f.cljs$lang$applyTo) {\n      var bc = cljs.core.bounded_count.call(null, arglist, fixed_arity + 1);\n      if (bc &lt;= fixed_arity) {\n        return cljs.core.apply_to.call(null, f, bc, arglist);\n      } else {\n        return f.cljs$lang$applyTo(arglist);\n      }\n    } else {\n      return f.apply(f, cljs.core.to_array.call(null, arglist));\n    }\n  };\n  var apply__6 = function() {\n    var G__22501__delegate = function(f, a, b, c, d, args) {\n      var arglist = cljs.core.cons.call(null, a, cljs.core.cons.call(null, b, cljs.core.cons.call(null, c, cljs.core.cons.call(null, d, cljs.core.spread.call(null, args)))));\n      var fixed_arity = f.cljs$lang$maxFixedArity;\n      if (f.cljs$lang$applyTo) {\n        var bc = cljs.core.bounded_count.call(null, arglist, fixed_arity + 1);\n        if (bc &lt;= fixed_arity) {\n          return cljs.core.apply_to.call(null, f, bc, arglist);\n        } else {\n          return f.cljs$lang$applyTo(arglist);\n        }\n      } else {\n        return f.apply(f, cljs.core.to_array.call(null, arglist));\n      }\n    };\n    var G__22501 = function(f, a, b, c, d, var_args) {\n      var args = null;\n      if (arguments.length &gt; 5) {\n        args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 5), 0);\n      }\n      return G__22501__delegate.call(this, f, a, b, c, d, args);\n    };\n    G__22501.cljs$lang$maxFixedArity = 5;\n    G__22501.cljs$lang$applyTo = function(arglist__22502) {\n      var f = cljs.core.first(arglist__22502);\n      arglist__22502 = cljs.core.next(arglist__22502);\n      var a = cljs.core.first(arglist__22502);\n      arglist__22502 = cljs.core.next(arglist__22502);\n      var b = cljs.core.first(arglist__22502);\n      arglist__22502 = cljs.core.next(arglist__22502);\n      var c = cljs.core.first(arglist__22502);\n      arglist__22502 = cljs.core.next(arglist__22502);\n      var d = cljs.core.first(arglist__22502);\n      var args = cljs.core.rest(arglist__22502);\n      return G__22501__delegate(f, a, b, c, d, args);\n    };\n    G__22501.cljs$core$IFn$_invoke$arity$variadic = G__22501__delegate;\n    return G__22501;\n  }();\n  apply = function(f, a, b, c, d, var_args) {\n    var args = var_args;\n    switch(arguments.length) {\n      case 2:\n        return apply__2.call(this, f, a);\n      case 3:\n        return apply__3.call(this, f, a, b);\n      case 4:\n        return apply__4.call(this, f, a, b, c);\n      case 5:\n        return apply__5.call(this, f, a, b, c, d);\n      default:\n        return apply__6.cljs$core$IFn$_invoke$arity$variadic(f, a, b, c, d, cljs.core.array_seq(arguments, 5));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  apply.cljs$lang$maxFixedArity = 5;\n  apply.cljs$lang$applyTo = apply__6.cljs$lang$applyTo;\n  apply.cljs$core$IFn$_invoke$arity$2 = apply__2;\n  apply.cljs$core$IFn$_invoke$arity$3 = apply__3;\n  apply.cljs$core$IFn$_invoke$arity$4 = apply__4;\n  apply.cljs$core$IFn$_invoke$arity$5 = apply__5;\n  apply.cljs$core$IFn$_invoke$arity$variadic = apply__6.cljs$core$IFn$_invoke$arity$variadic;\n  return apply;\n}();\ncljs.core.vary_meta = function() {\n  var vary_meta = null;\n  var vary_meta__2 = function(obj, f) {\n    return cljs.core.with_meta.call(null, obj, f.call(null, cljs.core.meta.call(null, obj)));\n  };\n  var vary_meta__3 = function(obj, f, a) {\n    return cljs.core.with_meta.call(null, obj, f.call(null, cljs.core.meta.call(null, obj), a));\n  };\n  var vary_meta__4 = function(obj, f, a, b) {\n    return cljs.core.with_meta.call(null, obj, f.call(null, cljs.core.meta.call(null, obj), a, b));\n  };\n  var vary_meta__5 = function(obj, f, a, b, c) {\n    return cljs.core.with_meta.call(null, obj, f.call(null, cljs.core.meta.call(null, obj), a, b, c));\n  };\n  var vary_meta__6 = function(obj, f, a, b, c, d) {\n    return cljs.core.with_meta.call(null, obj, f.call(null, cljs.core.meta.call(null, obj), a, b, c, d));\n  };\n  var vary_meta__7 = function() {\n    var G__22503__delegate = function(obj, f, a, b, c, d, args) {\n      return cljs.core.with_meta.call(null, obj, cljs.core.apply.call(null, f, cljs.core.meta.call(null, obj), a, b, c, d, args));\n    };\n    var G__22503 = function(obj, f, a, b, c, d, var_args) {\n      var args = null;\n      if (arguments.length &gt; 6) {\n        args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 6), 0);\n      }\n      return G__22503__delegate.call(this, obj, f, a, b, c, d, args);\n    };\n    G__22503.cljs$lang$maxFixedArity = 6;\n    G__22503.cljs$lang$applyTo = function(arglist__22504) {\n      var obj = cljs.core.first(arglist__22504);\n      arglist__22504 = cljs.core.next(arglist__22504);\n      var f = cljs.core.first(arglist__22504);\n      arglist__22504 = cljs.core.next(arglist__22504);\n      var a = cljs.core.first(arglist__22504);\n      arglist__22504 = cljs.core.next(arglist__22504);\n      var b = cljs.core.first(arglist__22504);\n      arglist__22504 = cljs.core.next(arglist__22504);\n      var c = cljs.core.first(arglist__22504);\n      arglist__22504 = cljs.core.next(arglist__22504);\n      var d = cljs.core.first(arglist__22504);\n      var args = cljs.core.rest(arglist__22504);\n      return G__22503__delegate(obj, f, a, b, c, d, args);\n    };\n    G__22503.cljs$core$IFn$_invoke$arity$variadic = G__22503__delegate;\n    return G__22503;\n  }();\n  vary_meta = function(obj, f, a, b, c, d, var_args) {\n    var args = var_args;\n    switch(arguments.length) {\n      case 2:\n        return vary_meta__2.call(this, obj, f);\n      case 3:\n        return vary_meta__3.call(this, obj, f, a);\n      case 4:\n        return vary_meta__4.call(this, obj, f, a, b);\n      case 5:\n        return vary_meta__5.call(this, obj, f, a, b, c);\n      case 6:\n        return vary_meta__6.call(this, obj, f, a, b, c, d);\n      default:\n        return vary_meta__7.cljs$core$IFn$_invoke$arity$variadic(obj, f, a, b, c, d, cljs.core.array_seq(arguments, 6));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  vary_meta.cljs$lang$maxFixedArity = 6;\n  vary_meta.cljs$lang$applyTo = vary_meta__7.cljs$lang$applyTo;\n  vary_meta.cljs$core$IFn$_invoke$arity$2 = vary_meta__2;\n  vary_meta.cljs$core$IFn$_invoke$arity$3 = vary_meta__3;\n  vary_meta.cljs$core$IFn$_invoke$arity$4 = vary_meta__4;\n  vary_meta.cljs$core$IFn$_invoke$arity$5 = vary_meta__5;\n  vary_meta.cljs$core$IFn$_invoke$arity$6 = vary_meta__6;\n  vary_meta.cljs$core$IFn$_invoke$arity$variadic = vary_meta__7.cljs$core$IFn$_invoke$arity$variadic;\n  return vary_meta;\n}();\ncljs.core.not_EQ_ = function() {\n  var not_EQ_ = null;\n  var not_EQ___1 = function(x) {\n    return false;\n  };\n  var not_EQ___2 = function(x, y) {\n    return!cljs.core._EQ_.call(null, x, y);\n  };\n  var not_EQ___3 = function() {\n    var G__22505__delegate = function(x, y, more) {\n      return cljs.core.not.call(null, cljs.core.apply.call(null, cljs.core._EQ_, x, y, more));\n    };\n    var G__22505 = function(x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 2) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22505__delegate.call(this, x, y, more);\n    };\n    G__22505.cljs$lang$maxFixedArity = 2;\n    G__22505.cljs$lang$applyTo = function(arglist__22506) {\n      var x = cljs.core.first(arglist__22506);\n      arglist__22506 = cljs.core.next(arglist__22506);\n      var y = cljs.core.first(arglist__22506);\n      var more = cljs.core.rest(arglist__22506);\n      return G__22505__delegate(x, y, more);\n    };\n    G__22505.cljs$core$IFn$_invoke$arity$variadic = G__22505__delegate;\n    return G__22505;\n  }();\n  not_EQ_ = function(x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return not_EQ___1.call(this, x);\n      case 2:\n        return not_EQ___2.call(this, x, y);\n      default:\n        return not_EQ___3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  not_EQ_.cljs$lang$maxFixedArity = 2;\n  not_EQ_.cljs$lang$applyTo = not_EQ___3.cljs$lang$applyTo;\n  not_EQ_.cljs$core$IFn$_invoke$arity$1 = not_EQ___1;\n  not_EQ_.cljs$core$IFn$_invoke$arity$2 = not_EQ___2;\n  not_EQ_.cljs$core$IFn$_invoke$arity$variadic = not_EQ___3.cljs$core$IFn$_invoke$arity$variadic;\n  return not_EQ_;\n}();\ncljs.core.not_empty = function not_empty(coll) {\n  if (cljs.core.seq.call(null, coll)) {\n    return coll;\n  } else {\n    return null;\n  }\n};\ncljs.core.nil_iter = function nil_iter() {\n  if (typeof cljs.core.t22510 !== &quot;undefined&quot;) {\n  } else {\n    cljs.core.t22510 = function(nil_iter, meta22511) {\n      this.nil_iter = nil_iter;\n      this.meta22511 = meta22511;\n      this.cljs$lang$protocol_mask$partition1$ = 0;\n      this.cljs$lang$protocol_mask$partition0$ = 393216;\n    };\n    cljs.core.t22510.cljs$lang$type = true;\n    cljs.core.t22510.cljs$lang$ctorStr = &quot;cljs.core/t22510&quot;;\n    cljs.core.t22510.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n      return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/t22510&quot;);\n    };\n    cljs.core.t22510.prototype.hasNext = function() {\n      var self__ = this;\n      var _ = this;\n      return false;\n    };\n    cljs.core.t22510.prototype.next = function() {\n      var self__ = this;\n      var _ = this;\n      return new Error(&quot;No such element&quot;);\n    };\n    cljs.core.t22510.prototype.remove = function() {\n      var self__ = this;\n      var _ = this;\n      return new Error(&quot;Unsupported operation&quot;);\n    };\n    cljs.core.t22510.prototype.cljs$core$IMeta$_meta$arity$1 = function(_22512) {\n      var self__ = this;\n      var _22512__$1 = this;\n      return self__.meta22511;\n    };\n    cljs.core.t22510.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(_22512, meta22511__$1) {\n      var self__ = this;\n      var _22512__$1 = this;\n      return new cljs.core.t22510(self__.nil_iter, meta22511__$1);\n    };\n    cljs.core.__GT_t22510 = function __GT_t22510(nil_iter__$1, meta22511) {\n      return new cljs.core.t22510(nil_iter__$1, meta22511);\n    };\n  }\n  return new cljs.core.t22510(nil_iter, null);\n};\ncljs.core.StringIter = function(s, i) {\n  this.s = s;\n  this.i = i;\n};\ncljs.core.StringIter.cljs$lang$type = true;\ncljs.core.StringIter.cljs$lang$ctorStr = &quot;cljs.core/StringIter&quot;;\ncljs.core.StringIter.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/StringIter&quot;);\n};\ncljs.core.StringIter.prototype.hasNext = function() {\n  var self__ = this;\n  var _ = this;\n  return self__.i &lt; self__.s.length;\n};\ncljs.core.StringIter.prototype.next = function() {\n  var self__ = this;\n  var _ = this;\n  var ret = self__.s.charAt(self__.i);\n  self__.i = self__.i + 1;\n  return ret;\n};\ncljs.core.StringIter.prototype.remove = function() {\n  var self__ = this;\n  var _ = this;\n  return new Error(&quot;Unsupported operation&quot;);\n};\ncljs.core.__GT_StringIter = function __GT_StringIter(s, i) {\n  return new cljs.core.StringIter(s, i);\n};\ncljs.core.string_iter = function string_iter(x) {\n  return new cljs.core.StringIter(x, 0);\n};\ncljs.core.ArrayIter = function(arr, i) {\n  this.arr = arr;\n  this.i = i;\n};\ncljs.core.ArrayIter.cljs$lang$type = true;\ncljs.core.ArrayIter.cljs$lang$ctorStr = &quot;cljs.core/ArrayIter&quot;;\ncljs.core.ArrayIter.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ArrayIter&quot;);\n};\ncljs.core.ArrayIter.prototype.hasNext = function() {\n  var self__ = this;\n  var _ = this;\n  return self__.i &lt; self__.arr.length;\n};\ncljs.core.ArrayIter.prototype.next = function() {\n  var self__ = this;\n  var _ = this;\n  var ret = self__.arr[self__.i];\n  self__.i = self__.i + 1;\n  return ret;\n};\ncljs.core.ArrayIter.prototype.remove = function() {\n  var self__ = this;\n  var _ = this;\n  return new Error(&quot;Unsupported operation&quot;);\n};\ncljs.core.__GT_ArrayIter = function __GT_ArrayIter(arr, i) {\n  return new cljs.core.ArrayIter(arr, i);\n};\ncljs.core.array_iter = function array_iter(x) {\n  return new cljs.core.ArrayIter(x, 0);\n};\ncljs.core.INIT = {};\ncljs.core.START = {};\ncljs.core.SeqIter = function(_seq, _next) {\n  this._seq = _seq;\n  this._next = _next;\n};\ncljs.core.SeqIter.cljs$lang$type = true;\ncljs.core.SeqIter.cljs$lang$ctorStr = &quot;cljs.core/SeqIter&quot;;\ncljs.core.SeqIter.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/SeqIter&quot;);\n};\ncljs.core.SeqIter.prototype.hasNext = function() {\n  var self__ = this;\n  var _ = this;\n  if (self__._seq === cljs.core.INIT) {\n    self__._seq = cljs.core.START;\n    self__._next = cljs.core.seq.call(null, self__._next);\n  } else {\n    if (self__._seq === self__._next) {\n      self__._next = cljs.core.next.call(null, self__._seq);\n    } else {\n    }\n  }\n  return!(self__._next == null);\n};\ncljs.core.SeqIter.prototype.next = function() {\n  var self__ = this;\n  var this$ = this;\n  if (cljs.core.not.call(null, this$.hasNext())) {\n    throw new Error(&quot;No such element&quot;);\n  } else {\n    self__._seq = self__._next;\n    return cljs.core.first.call(null, self__._next);\n  }\n};\ncljs.core.SeqIter.prototype.remove = function() {\n  var self__ = this;\n  var _ = this;\n  return new Error(&quot;Unsupported operation&quot;);\n};\ncljs.core.__GT_SeqIter = function __GT_SeqIter(_seq, _next) {\n  return new cljs.core.SeqIter(_seq, _next);\n};\ncljs.core.seq_iter = function seq_iter(coll) {\n  return new cljs.core.SeqIter(cljs.core.INIT, coll);\n};\ncljs.core.iter = function iter(coll) {\n  if (coll == null) {\n    return cljs.core.nil_iter.call(null);\n  } else {\n    if (typeof coll === &quot;string&quot;) {\n      return cljs.core.string_iter.call(null, coll);\n    } else {\n      if (coll instanceof Array) {\n        return cljs.core.array_iter.call(null, coll);\n      } else {\n        return cljs.core.seq_iter.call(null, coll);\n      }\n    }\n  }\n};\ncljs.core.lazy_transformer = function lazy_transformer(stepper) {\n  return new cljs.core.LazyTransformer(stepper, null, null, null);\n};\ncljs.core.Stepper = function(xform, iter) {\n  this.xform = xform;\n  this.iter = iter;\n};\ncljs.core.Stepper.cljs$lang$type = true;\ncljs.core.Stepper.cljs$lang$ctorStr = &quot;cljs.core/Stepper&quot;;\ncljs.core.Stepper.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Stepper&quot;);\n};\ncljs.core.Stepper.prototype.step = function(lt) {\n  var self__ = this;\n  var this$ = this;\n  while (true) {\n    if (cljs.core.truth_(function() {\n      var and__17133__auto__ = !(lt.stepper == null);\n      if (and__17133__auto__) {\n        return self__.iter.hasNext();\n      } else {\n        return and__17133__auto__;\n      }\n    }())) {\n      if (cljs.core.reduced_QMARK_.call(null, self__.xform.call(null, lt, self__.iter.next()))) {\n        if (lt.rest == null) {\n        } else {\n          lt.rest.stepper = null;\n        }\n      } else {\n        continue;\n      }\n    } else {\n    }\n    break;\n  }\n  if (lt.stepper == null) {\n    return null;\n  } else {\n    return self__.xform.call(null, lt);\n  }\n};\ncljs.core.__GT_Stepper = function __GT_Stepper(xform, iter) {\n  return new cljs.core.Stepper(xform, iter);\n};\ncljs.core.stepper = function stepper(xform, iter) {\n  var stepfn = function() {\n    var stepfn = null;\n    var stepfn__1 = function(result) {\n      var lt = cljs.core.reduced_QMARK_.call(null, result) ? cljs.core.deref.call(null, result) : result;\n      lt.stepper = null;\n      return result;\n    };\n    var stepfn__2 = function(result, input) {\n      var lt = result;\n      lt.first = input;\n      lt.rest = cljs.core.lazy_transformer.call(null, lt.stepper);\n      lt.stepper = null;\n      return lt.rest;\n    };\n    stepfn = function(result, input) {\n      switch(arguments.length) {\n        case 1:\n          return stepfn__1.call(this, result);\n        case 2:\n          return stepfn__2.call(this, result, input);\n      }\n      throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n    };\n    stepfn.cljs$core$IFn$_invoke$arity$1 = stepfn__1;\n    stepfn.cljs$core$IFn$_invoke$arity$2 = stepfn__2;\n    return stepfn;\n  }();\n  return new cljs.core.Stepper(xform.call(null, stepfn), iter);\n};\ncljs.core.MultiStepper = function(xform, iters, nexts) {\n  this.xform = xform;\n  this.iters = iters;\n  this.nexts = nexts;\n};\ncljs.core.MultiStepper.cljs$lang$type = true;\ncljs.core.MultiStepper.cljs$lang$ctorStr = &quot;cljs.core/MultiStepper&quot;;\ncljs.core.MultiStepper.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/MultiStepper&quot;);\n};\ncljs.core.MultiStepper.prototype.hasNext = function() {\n  var self__ = this;\n  var _ = this;\n  var iters__$1 = cljs.core.seq.call(null, self__.iters);\n  while (true) {\n    if (!(iters__$1 == null)) {\n      var iter = cljs.core.first.call(null, iters__$1);\n      if (cljs.core.not.call(null, iter.hasNext())) {\n        return false;\n      } else {\n        var G__22513 = cljs.core.next.call(null, iters__$1);\n        iters__$1 = G__22513;\n        continue;\n      }\n    } else {\n      return true;\n    }\n    break;\n  }\n};\ncljs.core.MultiStepper.prototype.next = function() {\n  var self__ = this;\n  var _ = this;\n  var n__18001__auto___22514 = self__.iters.length;\n  var i_22515 = 0;\n  while (true) {\n    if (i_22515 &lt; n__18001__auto___22514) {\n      self__.nexts[i_22515] = self__.iters[i_22515].next();\n      var G__22516 = i_22515 + 1;\n      i_22515 = G__22516;\n      continue;\n    } else {\n    }\n    break;\n  }\n  return cljs.core.prim_seq.call(null, self__.nexts, 0);\n};\ncljs.core.MultiStepper.prototype.step = function(lt) {\n  var self__ = this;\n  var this$ = this;\n  while (true) {\n    if (cljs.core.truth_(function() {\n      var and__17133__auto__ = !(lt.stepper == null);\n      if (and__17133__auto__) {\n        return this$.hasNext();\n      } else {\n        return and__17133__auto__;\n      }\n    }())) {\n      if (cljs.core.reduced_QMARK_.call(null, cljs.core.apply.call(null, self__.xform, cljs.core.cons.call(null, lt, this$.next())))) {\n        if (lt.rest == null) {\n        } else {\n          lt.rest.stepper = null;\n        }\n      } else {\n        continue;\n      }\n    } else {\n    }\n    break;\n  }\n  if (lt.stepper == null) {\n    return null;\n  } else {\n    return self__.xform.call(null, lt);\n  }\n};\ncljs.core.__GT_MultiStepper = function __GT_MultiStepper(xform, iters, nexts) {\n  return new cljs.core.MultiStepper(xform, iters, nexts);\n};\ncljs.core.multi_stepper = function() {\n  var multi_stepper = null;\n  var multi_stepper__2 = function(xform, iters) {\n    return multi_stepper.call(null, xform, iters, new Array(iters.length));\n  };\n  var multi_stepper__3 = function(xform, iters, nexts) {\n    var stepfn = function() {\n      var stepfn = null;\n      var stepfn__1 = function(result) {\n        var lt = cljs.core.reduced_QMARK_.call(null, result) ? cljs.core.deref.call(null, result) : result;\n        lt.stepper = null;\n        return lt;\n      };\n      var stepfn__2 = function(result, input) {\n        var lt = result;\n        lt.first = input;\n        lt.rest = cljs.core.lazy_transformer.call(null, lt.stepper);\n        lt.stepper = null;\n        return lt.rest;\n      };\n      stepfn = function(result, input) {\n        switch(arguments.length) {\n          case 1:\n            return stepfn__1.call(this, result);\n          case 2:\n            return stepfn__2.call(this, result, input);\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      stepfn.cljs$core$IFn$_invoke$arity$1 = stepfn__1;\n      stepfn.cljs$core$IFn$_invoke$arity$2 = stepfn__2;\n      return stepfn;\n    }();\n    return new cljs.core.MultiStepper(xform.call(null, stepfn), iters, nexts);\n  };\n  multi_stepper = function(xform, iters, nexts) {\n    switch(arguments.length) {\n      case 2:\n        return multi_stepper__2.call(this, xform, iters);\n      case 3:\n        return multi_stepper__3.call(this, xform, iters, nexts);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  multi_stepper.cljs$core$IFn$_invoke$arity$2 = multi_stepper__2;\n  multi_stepper.cljs$core$IFn$_invoke$arity$3 = multi_stepper__3;\n  return multi_stepper;\n}();\ncljs.core.LazyTransformer = function(stepper, first, rest, meta) {\n  this.stepper = stepper;\n  this.first = first;\n  this.rest = rest;\n  this.meta = meta;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 31719628;\n};\ncljs.core.LazyTransformer.cljs$lang$type = true;\ncljs.core.LazyTransformer.cljs$lang$ctorStr = &quot;cljs.core/LazyTransformer&quot;;\ncljs.core.LazyTransformer.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/LazyTransformer&quot;);\n};\ncljs.core.LazyTransformer.prototype.cljs$core$INext$_next$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  if (self__.stepper == null) {\n  } else {\n    cljs.core._seq.call(null, this$__$1);\n  }\n  if (self__.rest == null) {\n    return null;\n  } else {\n    return cljs.core._seq.call(null, self__.rest);\n  }\n};\ncljs.core.LazyTransformer.prototype.cljs$core$ISeq$_first$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  if (self__.stepper == null) {\n  } else {\n    cljs.core._seq.call(null, this$__$1);\n  }\n  if (self__.rest == null) {\n    return null;\n  } else {\n    return self__.first;\n  }\n};\ncljs.core.LazyTransformer.prototype.cljs$core$ISeq$_rest$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  if (self__.stepper == null) {\n  } else {\n    cljs.core._seq.call(null, this$__$1);\n  }\n  if (self__.rest == null) {\n    return cljs.core.List.EMPTY;\n  } else {\n    return self__.rest;\n  }\n};\ncljs.core.LazyTransformer.prototype.cljs$core$ISeqable$_seq$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  if (self__.stepper == null) {\n  } else {\n    self__.stepper.step(this$__$1);\n  }\n  if (self__.rest == null) {\n    return null;\n  } else {\n    return this$__$1;\n  }\n};\ncljs.core.LazyTransformer.prototype.cljs$core$IHash$_hash$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  return cljs.core.hash_ordered_coll.call(null, this$__$1);\n};\ncljs.core.LazyTransformer.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(this$, other) {\n  var self__ = this;\n  var this$__$1 = this;\n  var s = cljs.core._seq.call(null, this$__$1);\n  if (!(s == null)) {\n    return cljs.core.equiv_sequential.call(null, this$__$1, other);\n  } else {\n    return cljs.core.sequential_QMARK_.call(null, other) &amp;&amp; cljs.core.seq.call(null, other) == null;\n  }\n};\ncljs.core.LazyTransformer.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  return cljs.core.List.EMPTY;\n};\ncljs.core.LazyTransformer.prototype.cljs$core$ICollection$_conj$arity$2 = function(this$, o) {\n  var self__ = this;\n  var this$__$1 = this;\n  return cljs.core.cons.call(null, o, cljs.core._seq.call(null, this$__$1));\n};\ncljs.core.LazyTransformer.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(this$, new_meta) {\n  var self__ = this;\n  var this$__$1 = this;\n  return new cljs.core.LazyTransformer(self__.stepper, self__.first, self__.rest, new_meta);\n};\ncljs.core.__GT_LazyTransformer = function __GT_LazyTransformer(stepper, first, rest, meta) {\n  return new cljs.core.LazyTransformer(stepper, first, rest, meta);\n};\ncljs.core.LazyTransformer.create = function(xform, coll) {\n  return new cljs.core.LazyTransformer(cljs.core.stepper.call(null, xform, cljs.core.iter.call(null, coll)), null, null, null);\n};\ncljs.core.LazyTransformer.createMulti = function(xform, colls) {\n  var iters = [];\n  var seq__22517_22521 = cljs.core.seq.call(null, colls);\n  var chunk__22518_22522 = null;\n  var count__22519_22523 = 0;\n  var i__22520_22524 = 0;\n  while (true) {\n    if (i__22520_22524 &lt; count__22519_22523) {\n      var coll_22525 = cljs.core._nth.call(null, chunk__22518_22522, i__22520_22524);\n      iters.push(cljs.core.iter.call(null, coll_22525));\n      var G__22526 = seq__22517_22521;\n      var G__22527 = chunk__22518_22522;\n      var G__22528 = count__22519_22523;\n      var G__22529 = i__22520_22524 + 1;\n      seq__22517_22521 = G__22526;\n      chunk__22518_22522 = G__22527;\n      count__22519_22523 = G__22528;\n      i__22520_22524 = G__22529;\n      continue;\n    } else {\n      var temp__4126__auto___22530 = cljs.core.seq.call(null, seq__22517_22521);\n      if (temp__4126__auto___22530) {\n        var seq__22517_22531__$1 = temp__4126__auto___22530;\n        if (cljs.core.chunked_seq_QMARK_.call(null, seq__22517_22531__$1)) {\n          var c__17901__auto___22532 = cljs.core.chunk_first.call(null, seq__22517_22531__$1);\n          var G__22533 = cljs.core.chunk_rest.call(null, seq__22517_22531__$1);\n          var G__22534 = c__17901__auto___22532;\n          var G__22535 = cljs.core.count.call(null, c__17901__auto___22532);\n          var G__22536 = 0;\n          seq__22517_22521 = G__22533;\n          chunk__22518_22522 = G__22534;\n          count__22519_22523 = G__22535;\n          i__22520_22524 = G__22536;\n          continue;\n        } else {\n          var coll_22537 = cljs.core.first.call(null, seq__22517_22531__$1);\n          iters.push(cljs.core.iter.call(null, coll_22537));\n          var G__22538 = cljs.core.next.call(null, seq__22517_22531__$1);\n          var G__22539 = null;\n          var G__22540 = 0;\n          var G__22541 = 0;\n          seq__22517_22521 = G__22538;\n          chunk__22518_22522 = G__22539;\n          count__22519_22523 = G__22540;\n          i__22520_22524 = G__22541;\n          continue;\n        }\n      } else {\n      }\n    }\n    break;\n  }\n  return new cljs.core.LazyTransformer(cljs.core.multi_stepper.call(null, xform, iters, new Array(iters.length)), null, null, null);\n};\ncljs.core.sequence = function() {\n  var sequence = null;\n  var sequence__1 = function(coll) {\n    if (cljs.core.seq_QMARK_.call(null, coll)) {\n      return coll;\n    } else {\n      var or__17145__auto__ = cljs.core.seq.call(null, coll);\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        return cljs.core.List.EMPTY;\n      }\n    }\n  };\n  var sequence__2 = function(xform, coll) {\n    return cljs.core.LazyTransformer.create(xform, coll);\n  };\n  var sequence__3 = function() {\n    var G__22542__delegate = function(xform, coll, colls) {\n      return cljs.core.LazyTransformer.createMulti(xform, cljs.core.to_array.call(null, cljs.core.cons.call(null, coll, colls)));\n    };\n    var G__22542 = function(xform, coll, var_args) {\n      var colls = null;\n      if (arguments.length &gt; 2) {\n        colls = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22542__delegate.call(this, xform, coll, colls);\n    };\n    G__22542.cljs$lang$maxFixedArity = 2;\n    G__22542.cljs$lang$applyTo = function(arglist__22543) {\n      var xform = cljs.core.first(arglist__22543);\n      arglist__22543 = cljs.core.next(arglist__22543);\n      var coll = cljs.core.first(arglist__22543);\n      var colls = cljs.core.rest(arglist__22543);\n      return G__22542__delegate(xform, coll, colls);\n    };\n    G__22542.cljs$core$IFn$_invoke$arity$variadic = G__22542__delegate;\n    return G__22542;\n  }();\n  sequence = function(xform, coll, var_args) {\n    var colls = var_args;\n    switch(arguments.length) {\n      case 1:\n        return sequence__1.call(this, xform);\n      case 2:\n        return sequence__2.call(this, xform, coll);\n      default:\n        return sequence__3.cljs$core$IFn$_invoke$arity$variadic(xform, coll, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  sequence.cljs$lang$maxFixedArity = 2;\n  sequence.cljs$lang$applyTo = sequence__3.cljs$lang$applyTo;\n  sequence.cljs$core$IFn$_invoke$arity$1 = sequence__1;\n  sequence.cljs$core$IFn$_invoke$arity$2 = sequence__2;\n  sequence.cljs$core$IFn$_invoke$arity$variadic = sequence__3.cljs$core$IFn$_invoke$arity$variadic;\n  return sequence;\n}();\ncljs.core.every_QMARK_ = function every_QMARK_(pred, coll) {\n  while (true) {\n    if (cljs.core.seq.call(null, coll) == null) {\n      return true;\n    } else {\n      if (cljs.core.truth_(pred.call(null, cljs.core.first.call(null, coll)))) {\n        var G__22544 = pred;\n        var G__22545 = cljs.core.next.call(null, coll);\n        pred = G__22544;\n        coll = G__22545;\n        continue;\n      } else {\n        return false;\n      }\n    }\n    break;\n  }\n};\ncljs.core.not_every_QMARK_ = function not_every_QMARK_(pred, coll) {\n  return!cljs.core.every_QMARK_.call(null, pred, coll);\n};\ncljs.core.some = function some(pred, coll) {\n  while (true) {\n    if (cljs.core.seq.call(null, coll)) {\n      var or__17145__auto__ = pred.call(null, cljs.core.first.call(null, coll));\n      if (cljs.core.truth_(or__17145__auto__)) {\n        return or__17145__auto__;\n      } else {\n        var G__22546 = pred;\n        var G__22547 = cljs.core.next.call(null, coll);\n        pred = G__22546;\n        coll = G__22547;\n        continue;\n      }\n    } else {\n      return null;\n    }\n    break;\n  }\n};\ncljs.core.not_any_QMARK_ = function not_any_QMARK_(pred, coll) {\n  return cljs.core.not.call(null, cljs.core.some.call(null, pred, coll));\n};\ncljs.core.even_QMARK_ = function even_QMARK_(n) {\n  if (cljs.core.integer_QMARK_.call(null, n)) {\n    return(n &amp; 1) === 0;\n  } else {\n    throw new Error(&quot;Argument must be an integer: &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(n));\n  }\n};\ncljs.core.odd_QMARK_ = function odd_QMARK_(n) {\n  return!cljs.core.even_QMARK_.call(null, n);\n};\ncljs.core.identity = function identity(x) {\n  return x;\n};\ncljs.core.complement = function complement(f) {\n  return function() {\n    var G__22548 = null;\n    var G__22548__0 = function() {\n      return cljs.core.not.call(null, f.call(null));\n    };\n    var G__22548__1 = function(x) {\n      return cljs.core.not.call(null, f.call(null, x));\n    };\n    var G__22548__2 = function(x, y) {\n      return cljs.core.not.call(null, f.call(null, x, y));\n    };\n    var G__22548__3 = function() {\n      var G__22549__delegate = function(x, y, zs) {\n        return cljs.core.not.call(null, cljs.core.apply.call(null, f, x, y, zs));\n      };\n      var G__22549 = function(x, y, var_args) {\n        var zs = null;\n        if (arguments.length &gt; 2) {\n          zs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n        }\n        return G__22549__delegate.call(this, x, y, zs);\n      };\n      G__22549.cljs$lang$maxFixedArity = 2;\n      G__22549.cljs$lang$applyTo = function(arglist__22550) {\n        var x = cljs.core.first(arglist__22550);\n        arglist__22550 = cljs.core.next(arglist__22550);\n        var y = cljs.core.first(arglist__22550);\n        var zs = cljs.core.rest(arglist__22550);\n        return G__22549__delegate(x, y, zs);\n      };\n      G__22549.cljs$core$IFn$_invoke$arity$variadic = G__22549__delegate;\n      return G__22549;\n    }();\n    G__22548 = function(x, y, var_args) {\n      var zs = var_args;\n      switch(arguments.length) {\n        case 0:\n          return G__22548__0.call(this);\n        case 1:\n          return G__22548__1.call(this, x);\n        case 2:\n          return G__22548__2.call(this, x, y);\n        default:\n          return G__22548__3.cljs$core$IFn$_invoke$arity$variadic(x, y, cljs.core.array_seq(arguments, 2));\n      }\n      throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n    };\n    G__22548.cljs$lang$maxFixedArity = 2;\n    G__22548.cljs$lang$applyTo = G__22548__3.cljs$lang$applyTo;\n    G__22548.cljs$core$IFn$_invoke$arity$0 = G__22548__0;\n    G__22548.cljs$core$IFn$_invoke$arity$1 = G__22548__1;\n    G__22548.cljs$core$IFn$_invoke$arity$2 = G__22548__2;\n    G__22548.cljs$core$IFn$_invoke$arity$variadic = G__22548__3.cljs$core$IFn$_invoke$arity$variadic;\n    return G__22548;\n  }();\n};\ncljs.core.constantly = function constantly(x) {\n  return function() {\n    var G__22551__delegate = function(args) {\n      return x;\n    };\n    var G__22551 = function(var_args) {\n      var args = null;\n      if (arguments.length &gt; 0) {\n        args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n      }\n      return G__22551__delegate.call(this, args);\n    };\n    G__22551.cljs$lang$maxFixedArity = 0;\n    G__22551.cljs$lang$applyTo = function(arglist__22552) {\n      var args = cljs.core.seq(arglist__22552);\n      return G__22551__delegate(args);\n    };\n    G__22551.cljs$core$IFn$_invoke$arity$variadic = G__22551__delegate;\n    return G__22551;\n  }();\n};\ncljs.core.comp = function() {\n  var comp = null;\n  var comp__0 = function() {\n    return cljs.core.identity;\n  };\n  var comp__1 = function(f) {\n    return f;\n  };\n  var comp__2 = function(f, g) {\n    return function() {\n      var G__22553 = null;\n      var G__22553__0 = function() {\n        return f.call(null, g.call(null));\n      };\n      var G__22553__1 = function(x) {\n        return f.call(null, g.call(null, x));\n      };\n      var G__22553__2 = function(x, y) {\n        return f.call(null, g.call(null, x, y));\n      };\n      var G__22553__3 = function(x, y, z) {\n        return f.call(null, g.call(null, x, y, z));\n      };\n      var G__22553__4 = function() {\n        var G__22554__delegate = function(x, y, z, args) {\n          return f.call(null, cljs.core.apply.call(null, g, x, y, z, args));\n        };\n        var G__22554 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22554__delegate.call(this, x, y, z, args);\n        };\n        G__22554.cljs$lang$maxFixedArity = 3;\n        G__22554.cljs$lang$applyTo = function(arglist__22555) {\n          var x = cljs.core.first(arglist__22555);\n          arglist__22555 = cljs.core.next(arglist__22555);\n          var y = cljs.core.first(arglist__22555);\n          arglist__22555 = cljs.core.next(arglist__22555);\n          var z = cljs.core.first(arglist__22555);\n          var args = cljs.core.rest(arglist__22555);\n          return G__22554__delegate(x, y, z, args);\n        };\n        G__22554.cljs$core$IFn$_invoke$arity$variadic = G__22554__delegate;\n        return G__22554;\n      }();\n      G__22553 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return G__22553__0.call(this);\n          case 1:\n            return G__22553__1.call(this, x);\n          case 2:\n            return G__22553__2.call(this, x, y);\n          case 3:\n            return G__22553__3.call(this, x, y, z);\n          default:\n            return G__22553__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      G__22553.cljs$lang$maxFixedArity = 3;\n      G__22553.cljs$lang$applyTo = G__22553__4.cljs$lang$applyTo;\n      G__22553.cljs$core$IFn$_invoke$arity$0 = G__22553__0;\n      G__22553.cljs$core$IFn$_invoke$arity$1 = G__22553__1;\n      G__22553.cljs$core$IFn$_invoke$arity$2 = G__22553__2;\n      G__22553.cljs$core$IFn$_invoke$arity$3 = G__22553__3;\n      G__22553.cljs$core$IFn$_invoke$arity$variadic = G__22553__4.cljs$core$IFn$_invoke$arity$variadic;\n      return G__22553;\n    }();\n  };\n  var comp__3 = function(f, g, h) {\n    return function() {\n      var G__22556 = null;\n      var G__22556__0 = function() {\n        return f.call(null, g.call(null, h.call(null)));\n      };\n      var G__22556__1 = function(x) {\n        return f.call(null, g.call(null, h.call(null, x)));\n      };\n      var G__22556__2 = function(x, y) {\n        return f.call(null, g.call(null, h.call(null, x, y)));\n      };\n      var G__22556__3 = function(x, y, z) {\n        return f.call(null, g.call(null, h.call(null, x, y, z)));\n      };\n      var G__22556__4 = function() {\n        var G__22557__delegate = function(x, y, z, args) {\n          return f.call(null, g.call(null, cljs.core.apply.call(null, h, x, y, z, args)));\n        };\n        var G__22557 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22557__delegate.call(this, x, y, z, args);\n        };\n        G__22557.cljs$lang$maxFixedArity = 3;\n        G__22557.cljs$lang$applyTo = function(arglist__22558) {\n          var x = cljs.core.first(arglist__22558);\n          arglist__22558 = cljs.core.next(arglist__22558);\n          var y = cljs.core.first(arglist__22558);\n          arglist__22558 = cljs.core.next(arglist__22558);\n          var z = cljs.core.first(arglist__22558);\n          var args = cljs.core.rest(arglist__22558);\n          return G__22557__delegate(x, y, z, args);\n        };\n        G__22557.cljs$core$IFn$_invoke$arity$variadic = G__22557__delegate;\n        return G__22557;\n      }();\n      G__22556 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return G__22556__0.call(this);\n          case 1:\n            return G__22556__1.call(this, x);\n          case 2:\n            return G__22556__2.call(this, x, y);\n          case 3:\n            return G__22556__3.call(this, x, y, z);\n          default:\n            return G__22556__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      G__22556.cljs$lang$maxFixedArity = 3;\n      G__22556.cljs$lang$applyTo = G__22556__4.cljs$lang$applyTo;\n      G__22556.cljs$core$IFn$_invoke$arity$0 = G__22556__0;\n      G__22556.cljs$core$IFn$_invoke$arity$1 = G__22556__1;\n      G__22556.cljs$core$IFn$_invoke$arity$2 = G__22556__2;\n      G__22556.cljs$core$IFn$_invoke$arity$3 = G__22556__3;\n      G__22556.cljs$core$IFn$_invoke$arity$variadic = G__22556__4.cljs$core$IFn$_invoke$arity$variadic;\n      return G__22556;\n    }();\n  };\n  var comp__4 = function() {\n    var G__22559__delegate = function(f1, f2, f3, fs) {\n      var fs__$1 = cljs.core.reverse.call(null, cljs.core.list_STAR_.call(null, f1, f2, f3, fs));\n      return function(fs__$1) {\n        return function() {\n          var G__22560__delegate = function(args) {\n            var ret = cljs.core.apply.call(null, cljs.core.first.call(null, fs__$1), args);\n            var fs__$2 = cljs.core.next.call(null, fs__$1);\n            while (true) {\n              if (fs__$2) {\n                var G__22561 = cljs.core.first.call(null, fs__$2).call(null, ret);\n                var G__22562 = cljs.core.next.call(null, fs__$2);\n                ret = G__22561;\n                fs__$2 = G__22562;\n                continue;\n              } else {\n                return ret;\n              }\n              break;\n            }\n          };\n          var G__22560 = function(var_args) {\n            var args = null;\n            if (arguments.length &gt; 0) {\n              args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n            }\n            return G__22560__delegate.call(this, args);\n          };\n          G__22560.cljs$lang$maxFixedArity = 0;\n          G__22560.cljs$lang$applyTo = function(arglist__22563) {\n            var args = cljs.core.seq(arglist__22563);\n            return G__22560__delegate(args);\n          };\n          G__22560.cljs$core$IFn$_invoke$arity$variadic = G__22560__delegate;\n          return G__22560;\n        }();\n      }(fs__$1);\n    };\n    var G__22559 = function(f1, f2, f3, var_args) {\n      var fs = null;\n      if (arguments.length &gt; 3) {\n        fs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n      }\n      return G__22559__delegate.call(this, f1, f2, f3, fs);\n    };\n    G__22559.cljs$lang$maxFixedArity = 3;\n    G__22559.cljs$lang$applyTo = function(arglist__22564) {\n      var f1 = cljs.core.first(arglist__22564);\n      arglist__22564 = cljs.core.next(arglist__22564);\n      var f2 = cljs.core.first(arglist__22564);\n      arglist__22564 = cljs.core.next(arglist__22564);\n      var f3 = cljs.core.first(arglist__22564);\n      var fs = cljs.core.rest(arglist__22564);\n      return G__22559__delegate(f1, f2, f3, fs);\n    };\n    G__22559.cljs$core$IFn$_invoke$arity$variadic = G__22559__delegate;\n    return G__22559;\n  }();\n  comp = function(f1, f2, f3, var_args) {\n    var fs = var_args;\n    switch(arguments.length) {\n      case 0:\n        return comp__0.call(this);\n      case 1:\n        return comp__1.call(this, f1);\n      case 2:\n        return comp__2.call(this, f1, f2);\n      case 3:\n        return comp__3.call(this, f1, f2, f3);\n      default:\n        return comp__4.cljs$core$IFn$_invoke$arity$variadic(f1, f2, f3, cljs.core.array_seq(arguments, 3));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  comp.cljs$lang$maxFixedArity = 3;\n  comp.cljs$lang$applyTo = comp__4.cljs$lang$applyTo;\n  comp.cljs$core$IFn$_invoke$arity$0 = comp__0;\n  comp.cljs$core$IFn$_invoke$arity$1 = comp__1;\n  comp.cljs$core$IFn$_invoke$arity$2 = comp__2;\n  comp.cljs$core$IFn$_invoke$arity$3 = comp__3;\n  comp.cljs$core$IFn$_invoke$arity$variadic = comp__4.cljs$core$IFn$_invoke$arity$variadic;\n  return comp;\n}();\ncljs.core.partial = function() {\n  var partial = null;\n  var partial__1 = function(f) {\n    return f;\n  };\n  var partial__2 = function(f, arg1) {\n    return function() {\n      var G__22565__delegate = function(args) {\n        return cljs.core.apply.call(null, f, arg1, args);\n      };\n      var G__22565 = function(var_args) {\n        var args = null;\n        if (arguments.length &gt; 0) {\n          args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n        }\n        return G__22565__delegate.call(this, args);\n      };\n      G__22565.cljs$lang$maxFixedArity = 0;\n      G__22565.cljs$lang$applyTo = function(arglist__22566) {\n        var args = cljs.core.seq(arglist__22566);\n        return G__22565__delegate(args);\n      };\n      G__22565.cljs$core$IFn$_invoke$arity$variadic = G__22565__delegate;\n      return G__22565;\n    }();\n  };\n  var partial__3 = function(f, arg1, arg2) {\n    return function() {\n      var G__22567__delegate = function(args) {\n        return cljs.core.apply.call(null, f, arg1, arg2, args);\n      };\n      var G__22567 = function(var_args) {\n        var args = null;\n        if (arguments.length &gt; 0) {\n          args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n        }\n        return G__22567__delegate.call(this, args);\n      };\n      G__22567.cljs$lang$maxFixedArity = 0;\n      G__22567.cljs$lang$applyTo = function(arglist__22568) {\n        var args = cljs.core.seq(arglist__22568);\n        return G__22567__delegate(args);\n      };\n      G__22567.cljs$core$IFn$_invoke$arity$variadic = G__22567__delegate;\n      return G__22567;\n    }();\n  };\n  var partial__4 = function(f, arg1, arg2, arg3) {\n    return function() {\n      var G__22569__delegate = function(args) {\n        return cljs.core.apply.call(null, f, arg1, arg2, arg3, args);\n      };\n      var G__22569 = function(var_args) {\n        var args = null;\n        if (arguments.length &gt; 0) {\n          args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n        }\n        return G__22569__delegate.call(this, args);\n      };\n      G__22569.cljs$lang$maxFixedArity = 0;\n      G__22569.cljs$lang$applyTo = function(arglist__22570) {\n        var args = cljs.core.seq(arglist__22570);\n        return G__22569__delegate(args);\n      };\n      G__22569.cljs$core$IFn$_invoke$arity$variadic = G__22569__delegate;\n      return G__22569;\n    }();\n  };\n  var partial__5 = function() {\n    var G__22571__delegate = function(f, arg1, arg2, arg3, more) {\n      return function() {\n        var G__22572__delegate = function(args) {\n          return cljs.core.apply.call(null, f, arg1, arg2, arg3, cljs.core.concat.call(null, more, args));\n        };\n        var G__22572 = function(var_args) {\n          var args = null;\n          if (arguments.length &gt; 0) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n          }\n          return G__22572__delegate.call(this, args);\n        };\n        G__22572.cljs$lang$maxFixedArity = 0;\n        G__22572.cljs$lang$applyTo = function(arglist__22573) {\n          var args = cljs.core.seq(arglist__22573);\n          return G__22572__delegate(args);\n        };\n        G__22572.cljs$core$IFn$_invoke$arity$variadic = G__22572__delegate;\n        return G__22572;\n      }();\n    };\n    var G__22571 = function(f, arg1, arg2, arg3, var_args) {\n      var more = null;\n      if (arguments.length &gt; 4) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 4), 0);\n      }\n      return G__22571__delegate.call(this, f, arg1, arg2, arg3, more);\n    };\n    G__22571.cljs$lang$maxFixedArity = 4;\n    G__22571.cljs$lang$applyTo = function(arglist__22574) {\n      var f = cljs.core.first(arglist__22574);\n      arglist__22574 = cljs.core.next(arglist__22574);\n      var arg1 = cljs.core.first(arglist__22574);\n      arglist__22574 = cljs.core.next(arglist__22574);\n      var arg2 = cljs.core.first(arglist__22574);\n      arglist__22574 = cljs.core.next(arglist__22574);\n      var arg3 = cljs.core.first(arglist__22574);\n      var more = cljs.core.rest(arglist__22574);\n      return G__22571__delegate(f, arg1, arg2, arg3, more);\n    };\n    G__22571.cljs$core$IFn$_invoke$arity$variadic = G__22571__delegate;\n    return G__22571;\n  }();\n  partial = function(f, arg1, arg2, arg3, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 1:\n        return partial__1.call(this, f);\n      case 2:\n        return partial__2.call(this, f, arg1);\n      case 3:\n        return partial__3.call(this, f, arg1, arg2);\n      case 4:\n        return partial__4.call(this, f, arg1, arg2, arg3);\n      default:\n        return partial__5.cljs$core$IFn$_invoke$arity$variadic(f, arg1, arg2, arg3, cljs.core.array_seq(arguments, 4));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  partial.cljs$lang$maxFixedArity = 4;\n  partial.cljs$lang$applyTo = partial__5.cljs$lang$applyTo;\n  partial.cljs$core$IFn$_invoke$arity$1 = partial__1;\n  partial.cljs$core$IFn$_invoke$arity$2 = partial__2;\n  partial.cljs$core$IFn$_invoke$arity$3 = partial__3;\n  partial.cljs$core$IFn$_invoke$arity$4 = partial__4;\n  partial.cljs$core$IFn$_invoke$arity$variadic = partial__5.cljs$core$IFn$_invoke$arity$variadic;\n  return partial;\n}();\ncljs.core.fnil = function() {\n  var fnil = null;\n  var fnil__2 = function(f, x) {\n    return function() {\n      var G__22575 = null;\n      var G__22575__1 = function(a) {\n        return f.call(null, a == null ? x : a);\n      };\n      var G__22575__2 = function(a, b) {\n        return f.call(null, a == null ? x : a, b);\n      };\n      var G__22575__3 = function(a, b, c) {\n        return f.call(null, a == null ? x : a, b, c);\n      };\n      var G__22575__4 = function() {\n        var G__22576__delegate = function(a, b, c, ds) {\n          return cljs.core.apply.call(null, f, a == null ? x : a, b, c, ds);\n        };\n        var G__22576 = function(a, b, c, var_args) {\n          var ds = null;\n          if (arguments.length &gt; 3) {\n            ds = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22576__delegate.call(this, a, b, c, ds);\n        };\n        G__22576.cljs$lang$maxFixedArity = 3;\n        G__22576.cljs$lang$applyTo = function(arglist__22577) {\n          var a = cljs.core.first(arglist__22577);\n          arglist__22577 = cljs.core.next(arglist__22577);\n          var b = cljs.core.first(arglist__22577);\n          arglist__22577 = cljs.core.next(arglist__22577);\n          var c = cljs.core.first(arglist__22577);\n          var ds = cljs.core.rest(arglist__22577);\n          return G__22576__delegate(a, b, c, ds);\n        };\n        G__22576.cljs$core$IFn$_invoke$arity$variadic = G__22576__delegate;\n        return G__22576;\n      }();\n      G__22575 = function(a, b, c, var_args) {\n        var ds = var_args;\n        switch(arguments.length) {\n          case 1:\n            return G__22575__1.call(this, a);\n          case 2:\n            return G__22575__2.call(this, a, b);\n          case 3:\n            return G__22575__3.call(this, a, b, c);\n          default:\n            return G__22575__4.cljs$core$IFn$_invoke$arity$variadic(a, b, c, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      G__22575.cljs$lang$maxFixedArity = 3;\n      G__22575.cljs$lang$applyTo = G__22575__4.cljs$lang$applyTo;\n      G__22575.cljs$core$IFn$_invoke$arity$1 = G__22575__1;\n      G__22575.cljs$core$IFn$_invoke$arity$2 = G__22575__2;\n      G__22575.cljs$core$IFn$_invoke$arity$3 = G__22575__3;\n      G__22575.cljs$core$IFn$_invoke$arity$variadic = G__22575__4.cljs$core$IFn$_invoke$arity$variadic;\n      return G__22575;\n    }();\n  };\n  var fnil__3 = function(f, x, y) {\n    return function() {\n      var G__22578 = null;\n      var G__22578__2 = function(a, b) {\n        return f.call(null, a == null ? x : a, b == null ? y : b);\n      };\n      var G__22578__3 = function(a, b, c) {\n        return f.call(null, a == null ? x : a, b == null ? y : b, c);\n      };\n      var G__22578__4 = function() {\n        var G__22579__delegate = function(a, b, c, ds) {\n          return cljs.core.apply.call(null, f, a == null ? x : a, b == null ? y : b, c, ds);\n        };\n        var G__22579 = function(a, b, c, var_args) {\n          var ds = null;\n          if (arguments.length &gt; 3) {\n            ds = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22579__delegate.call(this, a, b, c, ds);\n        };\n        G__22579.cljs$lang$maxFixedArity = 3;\n        G__22579.cljs$lang$applyTo = function(arglist__22580) {\n          var a = cljs.core.first(arglist__22580);\n          arglist__22580 = cljs.core.next(arglist__22580);\n          var b = cljs.core.first(arglist__22580);\n          arglist__22580 = cljs.core.next(arglist__22580);\n          var c = cljs.core.first(arglist__22580);\n          var ds = cljs.core.rest(arglist__22580);\n          return G__22579__delegate(a, b, c, ds);\n        };\n        G__22579.cljs$core$IFn$_invoke$arity$variadic = G__22579__delegate;\n        return G__22579;\n      }();\n      G__22578 = function(a, b, c, var_args) {\n        var ds = var_args;\n        switch(arguments.length) {\n          case 2:\n            return G__22578__2.call(this, a, b);\n          case 3:\n            return G__22578__3.call(this, a, b, c);\n          default:\n            return G__22578__4.cljs$core$IFn$_invoke$arity$variadic(a, b, c, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      G__22578.cljs$lang$maxFixedArity = 3;\n      G__22578.cljs$lang$applyTo = G__22578__4.cljs$lang$applyTo;\n      G__22578.cljs$core$IFn$_invoke$arity$2 = G__22578__2;\n      G__22578.cljs$core$IFn$_invoke$arity$3 = G__22578__3;\n      G__22578.cljs$core$IFn$_invoke$arity$variadic = G__22578__4.cljs$core$IFn$_invoke$arity$variadic;\n      return G__22578;\n    }();\n  };\n  var fnil__4 = function(f, x, y, z) {\n    return function() {\n      var G__22581 = null;\n      var G__22581__2 = function(a, b) {\n        return f.call(null, a == null ? x : a, b == null ? y : b);\n      };\n      var G__22581__3 = function(a, b, c) {\n        return f.call(null, a == null ? x : a, b == null ? y : b, c == null ? z : c);\n      };\n      var G__22581__4 = function() {\n        var G__22582__delegate = function(a, b, c, ds) {\n          return cljs.core.apply.call(null, f, a == null ? x : a, b == null ? y : b, c == null ? z : c, ds);\n        };\n        var G__22582 = function(a, b, c, var_args) {\n          var ds = null;\n          if (arguments.length &gt; 3) {\n            ds = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22582__delegate.call(this, a, b, c, ds);\n        };\n        G__22582.cljs$lang$maxFixedArity = 3;\n        G__22582.cljs$lang$applyTo = function(arglist__22583) {\n          var a = cljs.core.first(arglist__22583);\n          arglist__22583 = cljs.core.next(arglist__22583);\n          var b = cljs.core.first(arglist__22583);\n          arglist__22583 = cljs.core.next(arglist__22583);\n          var c = cljs.core.first(arglist__22583);\n          var ds = cljs.core.rest(arglist__22583);\n          return G__22582__delegate(a, b, c, ds);\n        };\n        G__22582.cljs$core$IFn$_invoke$arity$variadic = G__22582__delegate;\n        return G__22582;\n      }();\n      G__22581 = function(a, b, c, var_args) {\n        var ds = var_args;\n        switch(arguments.length) {\n          case 2:\n            return G__22581__2.call(this, a, b);\n          case 3:\n            return G__22581__3.call(this, a, b, c);\n          default:\n            return G__22581__4.cljs$core$IFn$_invoke$arity$variadic(a, b, c, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      G__22581.cljs$lang$maxFixedArity = 3;\n      G__22581.cljs$lang$applyTo = G__22581__4.cljs$lang$applyTo;\n      G__22581.cljs$core$IFn$_invoke$arity$2 = G__22581__2;\n      G__22581.cljs$core$IFn$_invoke$arity$3 = G__22581__3;\n      G__22581.cljs$core$IFn$_invoke$arity$variadic = G__22581__4.cljs$core$IFn$_invoke$arity$variadic;\n      return G__22581;\n    }();\n  };\n  fnil = function(f, x, y, z) {\n    switch(arguments.length) {\n      case 2:\n        return fnil__2.call(this, f, x);\n      case 3:\n        return fnil__3.call(this, f, x, y);\n      case 4:\n        return fnil__4.call(this, f, x, y, z);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  fnil.cljs$core$IFn$_invoke$arity$2 = fnil__2;\n  fnil.cljs$core$IFn$_invoke$arity$3 = fnil__3;\n  fnil.cljs$core$IFn$_invoke$arity$4 = fnil__4;\n  return fnil;\n}();\ncljs.core.map_indexed = function map_indexed(f, coll) {\n  var mapi = function mapi(idx, coll__$1) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll__$1);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, s)) {\n          var c = cljs.core.chunk_first.call(null, s);\n          var size = cljs.core.count.call(null, c);\n          var b = cljs.core.chunk_buffer.call(null, size);\n          var n__18001__auto___22584 = size;\n          var i_22585 = 0;\n          while (true) {\n            if (i_22585 &lt; n__18001__auto___22584) {\n              cljs.core.chunk_append.call(null, b, f.call(null, idx + i_22585, cljs.core._nth.call(null, c, i_22585)));\n              var G__22586 = i_22585 + 1;\n              i_22585 = G__22586;\n              continue;\n            } else {\n            }\n            break;\n          }\n          return cljs.core.chunk_cons.call(null, cljs.core.chunk.call(null, b), mapi.call(null, idx + size, cljs.core.chunk_rest.call(null, s)));\n        } else {\n          return cljs.core.cons.call(null, f.call(null, idx, cljs.core.first.call(null, s)), mapi.call(null, idx + 1, cljs.core.rest.call(null, s)));\n        }\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  return mapi.call(null, 0, coll);\n};\ncljs.core.keep = function() {\n  var keep = null;\n  var keep__1 = function(f) {\n    return function(f1) {\n      return function() {\n        var G__22587 = null;\n        var G__22587__0 = function() {\n          return f1.call(null);\n        };\n        var G__22587__1 = function(result) {\n          return f1.call(null, result);\n        };\n        var G__22587__2 = function(result, input) {\n          var v = f.call(null, input);\n          if (v == null) {\n            return result;\n          } else {\n            return f1.call(null, result, v);\n          }\n        };\n        G__22587 = function(result, input) {\n          switch(arguments.length) {\n            case 0:\n              return G__22587__0.call(this);\n            case 1:\n              return G__22587__1.call(this, result);\n            case 2:\n              return G__22587__2.call(this, result, input);\n          }\n          throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n        };\n        G__22587.cljs$core$IFn$_invoke$arity$0 = G__22587__0;\n        G__22587.cljs$core$IFn$_invoke$arity$1 = G__22587__1;\n        G__22587.cljs$core$IFn$_invoke$arity$2 = G__22587__2;\n        return G__22587;\n      }();\n    };\n  };\n  var keep__2 = function(f, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, s)) {\n          var c = cljs.core.chunk_first.call(null, s);\n          var size = cljs.core.count.call(null, c);\n          var b = cljs.core.chunk_buffer.call(null, size);\n          var n__18001__auto___22588 = size;\n          var i_22589 = 0;\n          while (true) {\n            if (i_22589 &lt; n__18001__auto___22588) {\n              var x_22590 = f.call(null, cljs.core._nth.call(null, c, i_22589));\n              if (x_22590 == null) {\n              } else {\n                cljs.core.chunk_append.call(null, b, x_22590);\n              }\n              var G__22591 = i_22589 + 1;\n              i_22589 = G__22591;\n              continue;\n            } else {\n            }\n            break;\n          }\n          return cljs.core.chunk_cons.call(null, cljs.core.chunk.call(null, b), keep.call(null, f, cljs.core.chunk_rest.call(null, s)));\n        } else {\n          var x = f.call(null, cljs.core.first.call(null, s));\n          if (x == null) {\n            return keep.call(null, f, cljs.core.rest.call(null, s));\n          } else {\n            return cljs.core.cons.call(null, x, keep.call(null, f, cljs.core.rest.call(null, s)));\n          }\n        }\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  keep = function(f, coll) {\n    switch(arguments.length) {\n      case 1:\n        return keep__1.call(this, f);\n      case 2:\n        return keep__2.call(this, f, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  keep.cljs$core$IFn$_invoke$arity$1 = keep__1;\n  keep.cljs$core$IFn$_invoke$arity$2 = keep__2;\n  return keep;\n}();\ncljs.core.Atom = function(state, meta, validator, watches) {\n  this.state = state;\n  this.meta = meta;\n  this.validator = validator;\n  this.watches = watches;\n  this.cljs$lang$protocol_mask$partition0$ = 6455296;\n  this.cljs$lang$protocol_mask$partition1$ = 16386;\n};\ncljs.core.Atom.cljs$lang$type = true;\ncljs.core.Atom.cljs$lang$ctorStr = &quot;cljs.core/Atom&quot;;\ncljs.core.Atom.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Atom&quot;);\n};\ncljs.core.Atom.prototype.cljs$core$IHash$_hash$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  return goog.getUid(this$__$1);\n};\ncljs.core.Atom.prototype.cljs$core$IWatchable$_notify_watches$arity$3 = function(this$, oldval, newval) {\n  var self__ = this;\n  var this$__$1 = this;\n  var seq__22592 = cljs.core.seq.call(null, self__.watches);\n  var chunk__22593 = null;\n  var count__22594 = 0;\n  var i__22595 = 0;\n  while (true) {\n    if (i__22595 &lt; count__22594) {\n      var vec__22596 = cljs.core._nth.call(null, chunk__22593, i__22595);\n      var key = cljs.core.nth.call(null, vec__22596, 0, null);\n      var f = cljs.core.nth.call(null, vec__22596, 1, null);\n      f.call(null, key, this$__$1, oldval, newval);\n      var G__22598 = seq__22592;\n      var G__22599 = chunk__22593;\n      var G__22600 = count__22594;\n      var G__22601 = i__22595 + 1;\n      seq__22592 = G__22598;\n      chunk__22593 = G__22599;\n      count__22594 = G__22600;\n      i__22595 = G__22601;\n      continue;\n    } else {\n      var temp__4126__auto__ = cljs.core.seq.call(null, seq__22592);\n      if (temp__4126__auto__) {\n        var seq__22592__$1 = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, seq__22592__$1)) {\n          var c__17901__auto__ = cljs.core.chunk_first.call(null, seq__22592__$1);\n          var G__22602 = cljs.core.chunk_rest.call(null, seq__22592__$1);\n          var G__22603 = c__17901__auto__;\n          var G__22604 = cljs.core.count.call(null, c__17901__auto__);\n          var G__22605 = 0;\n          seq__22592 = G__22602;\n          chunk__22593 = G__22603;\n          count__22594 = G__22604;\n          i__22595 = G__22605;\n          continue;\n        } else {\n          var vec__22597 = cljs.core.first.call(null, seq__22592__$1);\n          var key = cljs.core.nth.call(null, vec__22597, 0, null);\n          var f = cljs.core.nth.call(null, vec__22597, 1, null);\n          f.call(null, key, this$__$1, oldval, newval);\n          var G__22606 = cljs.core.next.call(null, seq__22592__$1);\n          var G__22607 = null;\n          var G__22608 = 0;\n          var G__22609 = 0;\n          seq__22592 = G__22606;\n          chunk__22593 = G__22607;\n          count__22594 = G__22608;\n          i__22595 = G__22609;\n          continue;\n        }\n      } else {\n        return null;\n      }\n    }\n    break;\n  }\n};\ncljs.core.Atom.prototype.cljs$core$IWatchable$_add_watch$arity$3 = function(this$, key, f) {\n  var self__ = this;\n  var this$__$1 = this;\n  this$__$1.watches = cljs.core.assoc.call(null, self__.watches, key, f);\n  return this$__$1;\n};\ncljs.core.Atom.prototype.cljs$core$IWatchable$_remove_watch$arity$2 = function(this$, key) {\n  var self__ = this;\n  var this$__$1 = this;\n  return this$__$1.watches = cljs.core.dissoc.call(null, self__.watches, key);\n};\ncljs.core.Atom.prototype.cljs$core$IMeta$_meta$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.meta;\n};\ncljs.core.Atom.prototype.cljs$core$IDeref$_deref$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return self__.state;\n};\ncljs.core.Atom.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(o, other) {\n  var self__ = this;\n  var o__$1 = this;\n  return o__$1 === other;\n};\ncljs.core.Atom.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.__GT_Atom = function __GT_Atom(state, meta, validator, watches) {\n  return new cljs.core.Atom(state, meta, validator, watches);\n};\ncljs.core.atom = function() {\n  var atom = null;\n  var atom__1 = function(x) {\n    return new cljs.core.Atom(x, null, null, null);\n  };\n  var atom__2 = function() {\n    var G__22613__delegate = function(x, p__22610) {\n      var map__22612 = p__22610;\n      var map__22612__$1 = cljs.core.seq_QMARK_.call(null, map__22612) ? cljs.core.apply.call(null, cljs.core.hash_map, map__22612) : map__22612;\n      var validator = cljs.core.get.call(null, map__22612__$1, new cljs.core.Keyword(null, &quot;validator&quot;, &quot;validator&quot;, -1966190681));\n      var meta = cljs.core.get.call(null, map__22612__$1, new cljs.core.Keyword(null, &quot;meta&quot;, &quot;meta&quot;, 1499536964));\n      return new cljs.core.Atom(x, meta, validator, null);\n    };\n    var G__22613 = function(x, var_args) {\n      var p__22610 = null;\n      if (arguments.length &gt; 1) {\n        p__22610 = cljs.core.array_seq(Array.prototype.slice.call(arguments, 1), 0);\n      }\n      return G__22613__delegate.call(this, x, p__22610);\n    };\n    G__22613.cljs$lang$maxFixedArity = 1;\n    G__22613.cljs$lang$applyTo = function(arglist__22614) {\n      var x = cljs.core.first(arglist__22614);\n      var p__22610 = cljs.core.rest(arglist__22614);\n      return G__22613__delegate(x, p__22610);\n    };\n    G__22613.cljs$core$IFn$_invoke$arity$variadic = G__22613__delegate;\n    return G__22613;\n  }();\n  atom = function(x, var_args) {\n    var p__22610 = var_args;\n    switch(arguments.length) {\n      case 1:\n        return atom__1.call(this, x);\n      default:\n        return atom__2.cljs$core$IFn$_invoke$arity$variadic(x, cljs.core.array_seq(arguments, 1));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  atom.cljs$lang$maxFixedArity = 1;\n  atom.cljs$lang$applyTo = atom__2.cljs$lang$applyTo;\n  atom.cljs$core$IFn$_invoke$arity$1 = atom__1;\n  atom.cljs$core$IFn$_invoke$arity$variadic = atom__2.cljs$core$IFn$_invoke$arity$variadic;\n  return atom;\n}();\ncljs.core.reset_BANG_ = function reset_BANG_(a, new_value) {\n  if (a instanceof cljs.core.Atom) {\n    var validate = a.validator;\n    if (validate == null) {\n    } else {\n      if (cljs.core.truth_(validate.call(null, new_value))) {\n      } else {\n        throw new Error(&quot;Assert failed: Validator rejected reference state\\n&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.pr_str.call(null, cljs.core.list(new cljs.core.Symbol(null, &quot;validate&quot;, &quot;validate&quot;, 1439230700, null), new cljs.core.Symbol(null, &quot;new-value&quot;, &quot;new-value&quot;, -1567397401, null)))));\n      }\n    }\n    var old_value = a.state;\n    a.state = new_value;\n    if (a.watches == null) {\n    } else {\n      cljs.core._notify_watches.call(null, a, old_value, new_value);\n    }\n    return new_value;\n  } else {\n    return cljs.core._reset_BANG_.call(null, a, new_value);\n  }\n};\ncljs.core.swap_BANG_ = function() {\n  var swap_BANG_ = null;\n  var swap_BANG___2 = function(a, f) {\n    if (a instanceof cljs.core.Atom) {\n      return cljs.core.reset_BANG_.call(null, a, f.call(null, a.state));\n    } else {\n      return cljs.core._swap_BANG_.call(null, a, f);\n    }\n  };\n  var swap_BANG___3 = function(a, f, x) {\n    if (a instanceof cljs.core.Atom) {\n      return cljs.core.reset_BANG_.call(null, a, f.call(null, a.state, x));\n    } else {\n      return cljs.core._swap_BANG_.call(null, a, f, x);\n    }\n  };\n  var swap_BANG___4 = function(a, f, x, y) {\n    if (a instanceof cljs.core.Atom) {\n      return cljs.core.reset_BANG_.call(null, a, f.call(null, a.state, x, y));\n    } else {\n      return cljs.core._swap_BANG_.call(null, a, f, x, y);\n    }\n  };\n  var swap_BANG___5 = function() {\n    var G__22615__delegate = function(a, f, x, y, more) {\n      if (a instanceof cljs.core.Atom) {\n        return cljs.core.reset_BANG_.call(null, a, cljs.core.apply.call(null, f, a.state, x, y, more));\n      } else {\n        return cljs.core._swap_BANG_.call(null, a, f, x, y, more);\n      }\n    };\n    var G__22615 = function(a, f, x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 4) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 4), 0);\n      }\n      return G__22615__delegate.call(this, a, f, x, y, more);\n    };\n    G__22615.cljs$lang$maxFixedArity = 4;\n    G__22615.cljs$lang$applyTo = function(arglist__22616) {\n      var a = cljs.core.first(arglist__22616);\n      arglist__22616 = cljs.core.next(arglist__22616);\n      var f = cljs.core.first(arglist__22616);\n      arglist__22616 = cljs.core.next(arglist__22616);\n      var x = cljs.core.first(arglist__22616);\n      arglist__22616 = cljs.core.next(arglist__22616);\n      var y = cljs.core.first(arglist__22616);\n      var more = cljs.core.rest(arglist__22616);\n      return G__22615__delegate(a, f, x, y, more);\n    };\n    G__22615.cljs$core$IFn$_invoke$arity$variadic = G__22615__delegate;\n    return G__22615;\n  }();\n  swap_BANG_ = function(a, f, x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 2:\n        return swap_BANG___2.call(this, a, f);\n      case 3:\n        return swap_BANG___3.call(this, a, f, x);\n      case 4:\n        return swap_BANG___4.call(this, a, f, x, y);\n      default:\n        return swap_BANG___5.cljs$core$IFn$_invoke$arity$variadic(a, f, x, y, cljs.core.array_seq(arguments, 4));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  swap_BANG_.cljs$lang$maxFixedArity = 4;\n  swap_BANG_.cljs$lang$applyTo = swap_BANG___5.cljs$lang$applyTo;\n  swap_BANG_.cljs$core$IFn$_invoke$arity$2 = swap_BANG___2;\n  swap_BANG_.cljs$core$IFn$_invoke$arity$3 = swap_BANG___3;\n  swap_BANG_.cljs$core$IFn$_invoke$arity$4 = swap_BANG___4;\n  swap_BANG_.cljs$core$IFn$_invoke$arity$variadic = swap_BANG___5.cljs$core$IFn$_invoke$arity$variadic;\n  return swap_BANG_;\n}();\ncljs.core.compare_and_set_BANG_ = function compare_and_set_BANG_(a, oldval, newval) {\n  if (cljs.core._EQ_.call(null, a.state, oldval)) {\n    cljs.core.reset_BANG_.call(null, a, newval);\n    return true;\n  } else {\n    return false;\n  }\n};\ncljs.core.set_validator_BANG_ = function set_validator_BANG_(iref, val) {\n  return iref.validator = val;\n};\ncljs.core.get_validator = function get_validator(iref) {\n  return iref.validator;\n};\ncljs.core.keep_indexed = function() {\n  var keep_indexed = null;\n  var keep_indexed__1 = function(f) {\n    return function(f1) {\n      var ia = cljs.core.atom.call(null, -1);\n      return function(ia) {\n        return function() {\n          var G__22617 = null;\n          var G__22617__0 = function() {\n            return f1.call(null);\n          };\n          var G__22617__1 = function(result) {\n            return f1.call(null, result);\n          };\n          var G__22617__2 = function(result, input) {\n            var i = cljs.core.swap_BANG_.call(null, ia, cljs.core.inc);\n            var v = f.call(null, i, input);\n            if (v == null) {\n              return result;\n            } else {\n              return f1.call(null, result, v);\n            }\n          };\n          G__22617 = function(result, input) {\n            switch(arguments.length) {\n              case 0:\n                return G__22617__0.call(this);\n              case 1:\n                return G__22617__1.call(this, result);\n              case 2:\n                return G__22617__2.call(this, result, input);\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          G__22617.cljs$core$IFn$_invoke$arity$0 = G__22617__0;\n          G__22617.cljs$core$IFn$_invoke$arity$1 = G__22617__1;\n          G__22617.cljs$core$IFn$_invoke$arity$2 = G__22617__2;\n          return G__22617;\n        }();\n      }(ia);\n    };\n  };\n  var keep_indexed__2 = function(f, coll) {\n    var keepi = function keepi(idx, coll__$1) {\n      return new cljs.core.LazySeq(null, function() {\n        var temp__4126__auto__ = cljs.core.seq.call(null, coll__$1);\n        if (temp__4126__auto__) {\n          var s = temp__4126__auto__;\n          if (cljs.core.chunked_seq_QMARK_.call(null, s)) {\n            var c = cljs.core.chunk_first.call(null, s);\n            var size = cljs.core.count.call(null, c);\n            var b = cljs.core.chunk_buffer.call(null, size);\n            var n__18001__auto___22618 = size;\n            var i_22619 = 0;\n            while (true) {\n              if (i_22619 &lt; n__18001__auto___22618) {\n                var x_22620 = f.call(null, idx + i_22619, cljs.core._nth.call(null, c, i_22619));\n                if (x_22620 == null) {\n                } else {\n                  cljs.core.chunk_append.call(null, b, x_22620);\n                }\n                var G__22621 = i_22619 + 1;\n                i_22619 = G__22621;\n                continue;\n              } else {\n              }\n              break;\n            }\n            return cljs.core.chunk_cons.call(null, cljs.core.chunk.call(null, b), keepi.call(null, idx + size, cljs.core.chunk_rest.call(null, s)));\n          } else {\n            var x = f.call(null, idx, cljs.core.first.call(null, s));\n            if (x == null) {\n              return keepi.call(null, idx + 1, cljs.core.rest.call(null, s));\n            } else {\n              return cljs.core.cons.call(null, x, keepi.call(null, idx + 1, cljs.core.rest.call(null, s)));\n            }\n          }\n        } else {\n          return null;\n        }\n      }, null, null);\n    };\n    return keepi.call(null, 0, coll);\n  };\n  keep_indexed = function(f, coll) {\n    switch(arguments.length) {\n      case 1:\n        return keep_indexed__1.call(this, f);\n      case 2:\n        return keep_indexed__2.call(this, f, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  keep_indexed.cljs$core$IFn$_invoke$arity$1 = keep_indexed__1;\n  keep_indexed.cljs$core$IFn$_invoke$arity$2 = keep_indexed__2;\n  return keep_indexed;\n}();\ncljs.core.every_pred = function() {\n  var every_pred = null;\n  var every_pred__1 = function(p) {\n    return function() {\n      var ep1 = null;\n      var ep1__0 = function() {\n        return true;\n      };\n      var ep1__1 = function(x) {\n        return cljs.core.boolean$.call(null, p.call(null, x));\n      };\n      var ep1__2 = function(x, y) {\n        return cljs.core.boolean$.call(null, function() {\n          var and__17133__auto__ = p.call(null, x);\n          if (cljs.core.truth_(and__17133__auto__)) {\n            return p.call(null, y);\n          } else {\n            return and__17133__auto__;\n          }\n        }());\n      };\n      var ep1__3 = function(x, y, z) {\n        return cljs.core.boolean$.call(null, function() {\n          var and__17133__auto__ = p.call(null, x);\n          if (cljs.core.truth_(and__17133__auto__)) {\n            var and__17133__auto____$1 = p.call(null, y);\n            if (cljs.core.truth_(and__17133__auto____$1)) {\n              return p.call(null, z);\n            } else {\n              return and__17133__auto____$1;\n            }\n          } else {\n            return and__17133__auto__;\n          }\n        }());\n      };\n      var ep1__4 = function() {\n        var G__22628__delegate = function(x, y, z, args) {\n          return cljs.core.boolean$.call(null, ep1.call(null, x, y, z) &amp;&amp; cljs.core.every_QMARK_.call(null, p, args));\n        };\n        var G__22628 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22628__delegate.call(this, x, y, z, args);\n        };\n        G__22628.cljs$lang$maxFixedArity = 3;\n        G__22628.cljs$lang$applyTo = function(arglist__22629) {\n          var x = cljs.core.first(arglist__22629);\n          arglist__22629 = cljs.core.next(arglist__22629);\n          var y = cljs.core.first(arglist__22629);\n          arglist__22629 = cljs.core.next(arglist__22629);\n          var z = cljs.core.first(arglist__22629);\n          var args = cljs.core.rest(arglist__22629);\n          return G__22628__delegate(x, y, z, args);\n        };\n        G__22628.cljs$core$IFn$_invoke$arity$variadic = G__22628__delegate;\n        return G__22628;\n      }();\n      ep1 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return ep1__0.call(this);\n          case 1:\n            return ep1__1.call(this, x);\n          case 2:\n            return ep1__2.call(this, x, y);\n          case 3:\n            return ep1__3.call(this, x, y, z);\n          default:\n            return ep1__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      ep1.cljs$lang$maxFixedArity = 3;\n      ep1.cljs$lang$applyTo = ep1__4.cljs$lang$applyTo;\n      ep1.cljs$core$IFn$_invoke$arity$0 = ep1__0;\n      ep1.cljs$core$IFn$_invoke$arity$1 = ep1__1;\n      ep1.cljs$core$IFn$_invoke$arity$2 = ep1__2;\n      ep1.cljs$core$IFn$_invoke$arity$3 = ep1__3;\n      ep1.cljs$core$IFn$_invoke$arity$variadic = ep1__4.cljs$core$IFn$_invoke$arity$variadic;\n      return ep1;\n    }();\n  };\n  var every_pred__2 = function(p1, p2) {\n    return function() {\n      var ep2 = null;\n      var ep2__0 = function() {\n        return true;\n      };\n      var ep2__1 = function(x) {\n        return cljs.core.boolean$.call(null, function() {\n          var and__17133__auto__ = p1.call(null, x);\n          if (cljs.core.truth_(and__17133__auto__)) {\n            return p2.call(null, x);\n          } else {\n            return and__17133__auto__;\n          }\n        }());\n      };\n      var ep2__2 = function(x, y) {\n        return cljs.core.boolean$.call(null, function() {\n          var and__17133__auto__ = p1.call(null, x);\n          if (cljs.core.truth_(and__17133__auto__)) {\n            var and__17133__auto____$1 = p1.call(null, y);\n            if (cljs.core.truth_(and__17133__auto____$1)) {\n              var and__17133__auto____$2 = p2.call(null, x);\n              if (cljs.core.truth_(and__17133__auto____$2)) {\n                return p2.call(null, y);\n              } else {\n                return and__17133__auto____$2;\n              }\n            } else {\n              return and__17133__auto____$1;\n            }\n          } else {\n            return and__17133__auto__;\n          }\n        }());\n      };\n      var ep2__3 = function(x, y, z) {\n        return cljs.core.boolean$.call(null, function() {\n          var and__17133__auto__ = p1.call(null, x);\n          if (cljs.core.truth_(and__17133__auto__)) {\n            var and__17133__auto____$1 = p1.call(null, y);\n            if (cljs.core.truth_(and__17133__auto____$1)) {\n              var and__17133__auto____$2 = p1.call(null, z);\n              if (cljs.core.truth_(and__17133__auto____$2)) {\n                var and__17133__auto____$3 = p2.call(null, x);\n                if (cljs.core.truth_(and__17133__auto____$3)) {\n                  var and__17133__auto____$4 = p2.call(null, y);\n                  if (cljs.core.truth_(and__17133__auto____$4)) {\n                    return p2.call(null, z);\n                  } else {\n                    return and__17133__auto____$4;\n                  }\n                } else {\n                  return and__17133__auto____$3;\n                }\n              } else {\n                return and__17133__auto____$2;\n              }\n            } else {\n              return and__17133__auto____$1;\n            }\n          } else {\n            return and__17133__auto__;\n          }\n        }());\n      };\n      var ep2__4 = function() {\n        var G__22630__delegate = function(x, y, z, args) {\n          return cljs.core.boolean$.call(null, ep2.call(null, x, y, z) &amp;&amp; cljs.core.every_QMARK_.call(null, function(p1__22622_SHARP_) {\n            var and__17133__auto__ = p1.call(null, p1__22622_SHARP_);\n            if (cljs.core.truth_(and__17133__auto__)) {\n              return p2.call(null, p1__22622_SHARP_);\n            } else {\n              return and__17133__auto__;\n            }\n          }, args));\n        };\n        var G__22630 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22630__delegate.call(this, x, y, z, args);\n        };\n        G__22630.cljs$lang$maxFixedArity = 3;\n        G__22630.cljs$lang$applyTo = function(arglist__22631) {\n          var x = cljs.core.first(arglist__22631);\n          arglist__22631 = cljs.core.next(arglist__22631);\n          var y = cljs.core.first(arglist__22631);\n          arglist__22631 = cljs.core.next(arglist__22631);\n          var z = cljs.core.first(arglist__22631);\n          var args = cljs.core.rest(arglist__22631);\n          return G__22630__delegate(x, y, z, args);\n        };\n        G__22630.cljs$core$IFn$_invoke$arity$variadic = G__22630__delegate;\n        return G__22630;\n      }();\n      ep2 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return ep2__0.call(this);\n          case 1:\n            return ep2__1.call(this, x);\n          case 2:\n            return ep2__2.call(this, x, y);\n          case 3:\n            return ep2__3.call(this, x, y, z);\n          default:\n            return ep2__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      ep2.cljs$lang$maxFixedArity = 3;\n      ep2.cljs$lang$applyTo = ep2__4.cljs$lang$applyTo;\n      ep2.cljs$core$IFn$_invoke$arity$0 = ep2__0;\n      ep2.cljs$core$IFn$_invoke$arity$1 = ep2__1;\n      ep2.cljs$core$IFn$_invoke$arity$2 = ep2__2;\n      ep2.cljs$core$IFn$_invoke$arity$3 = ep2__3;\n      ep2.cljs$core$IFn$_invoke$arity$variadic = ep2__4.cljs$core$IFn$_invoke$arity$variadic;\n      return ep2;\n    }();\n  };\n  var every_pred__3 = function(p1, p2, p3) {\n    return function() {\n      var ep3 = null;\n      var ep3__0 = function() {\n        return true;\n      };\n      var ep3__1 = function(x) {\n        return cljs.core.boolean$.call(null, function() {\n          var and__17133__auto__ = p1.call(null, x);\n          if (cljs.core.truth_(and__17133__auto__)) {\n            var and__17133__auto____$1 = p2.call(null, x);\n            if (cljs.core.truth_(and__17133__auto____$1)) {\n              return p3.call(null, x);\n            } else {\n              return and__17133__auto____$1;\n            }\n          } else {\n            return and__17133__auto__;\n          }\n        }());\n      };\n      var ep3__2 = function(x, y) {\n        return cljs.core.boolean$.call(null, function() {\n          var and__17133__auto__ = p1.call(null, x);\n          if (cljs.core.truth_(and__17133__auto__)) {\n            var and__17133__auto____$1 = p2.call(null, x);\n            if (cljs.core.truth_(and__17133__auto____$1)) {\n              var and__17133__auto____$2 = p3.call(null, x);\n              if (cljs.core.truth_(and__17133__auto____$2)) {\n                var and__17133__auto____$3 = p1.call(null, y);\n                if (cljs.core.truth_(and__17133__auto____$3)) {\n                  var and__17133__auto____$4 = p2.call(null, y);\n                  if (cljs.core.truth_(and__17133__auto____$4)) {\n                    return p3.call(null, y);\n                  } else {\n                    return and__17133__auto____$4;\n                  }\n                } else {\n                  return and__17133__auto____$3;\n                }\n              } else {\n                return and__17133__auto____$2;\n              }\n            } else {\n              return and__17133__auto____$1;\n            }\n          } else {\n            return and__17133__auto__;\n          }\n        }());\n      };\n      var ep3__3 = function(x, y, z) {\n        return cljs.core.boolean$.call(null, function() {\n          var and__17133__auto__ = p1.call(null, x);\n          if (cljs.core.truth_(and__17133__auto__)) {\n            var and__17133__auto____$1 = p2.call(null, x);\n            if (cljs.core.truth_(and__17133__auto____$1)) {\n              var and__17133__auto____$2 = p3.call(null, x);\n              if (cljs.core.truth_(and__17133__auto____$2)) {\n                var and__17133__auto____$3 = p1.call(null, y);\n                if (cljs.core.truth_(and__17133__auto____$3)) {\n                  var and__17133__auto____$4 = p2.call(null, y);\n                  if (cljs.core.truth_(and__17133__auto____$4)) {\n                    var and__17133__auto____$5 = p3.call(null, y);\n                    if (cljs.core.truth_(and__17133__auto____$5)) {\n                      var and__17133__auto____$6 = p1.call(null, z);\n                      if (cljs.core.truth_(and__17133__auto____$6)) {\n                        var and__17133__auto____$7 = p2.call(null, z);\n                        if (cljs.core.truth_(and__17133__auto____$7)) {\n                          return p3.call(null, z);\n                        } else {\n                          return and__17133__auto____$7;\n                        }\n                      } else {\n                        return and__17133__auto____$6;\n                      }\n                    } else {\n                      return and__17133__auto____$5;\n                    }\n                  } else {\n                    return and__17133__auto____$4;\n                  }\n                } else {\n                  return and__17133__auto____$3;\n                }\n              } else {\n                return and__17133__auto____$2;\n              }\n            } else {\n              return and__17133__auto____$1;\n            }\n          } else {\n            return and__17133__auto__;\n          }\n        }());\n      };\n      var ep3__4 = function() {\n        var G__22632__delegate = function(x, y, z, args) {\n          return cljs.core.boolean$.call(null, ep3.call(null, x, y, z) &amp;&amp; cljs.core.every_QMARK_.call(null, function(p1__22623_SHARP_) {\n            var and__17133__auto__ = p1.call(null, p1__22623_SHARP_);\n            if (cljs.core.truth_(and__17133__auto__)) {\n              var and__17133__auto____$1 = p2.call(null, p1__22623_SHARP_);\n              if (cljs.core.truth_(and__17133__auto____$1)) {\n                return p3.call(null, p1__22623_SHARP_);\n              } else {\n                return and__17133__auto____$1;\n              }\n            } else {\n              return and__17133__auto__;\n            }\n          }, args));\n        };\n        var G__22632 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22632__delegate.call(this, x, y, z, args);\n        };\n        G__22632.cljs$lang$maxFixedArity = 3;\n        G__22632.cljs$lang$applyTo = function(arglist__22633) {\n          var x = cljs.core.first(arglist__22633);\n          arglist__22633 = cljs.core.next(arglist__22633);\n          var y = cljs.core.first(arglist__22633);\n          arglist__22633 = cljs.core.next(arglist__22633);\n          var z = cljs.core.first(arglist__22633);\n          var args = cljs.core.rest(arglist__22633);\n          return G__22632__delegate(x, y, z, args);\n        };\n        G__22632.cljs$core$IFn$_invoke$arity$variadic = G__22632__delegate;\n        return G__22632;\n      }();\n      ep3 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return ep3__0.call(this);\n          case 1:\n            return ep3__1.call(this, x);\n          case 2:\n            return ep3__2.call(this, x, y);\n          case 3:\n            return ep3__3.call(this, x, y, z);\n          default:\n            return ep3__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      ep3.cljs$lang$maxFixedArity = 3;\n      ep3.cljs$lang$applyTo = ep3__4.cljs$lang$applyTo;\n      ep3.cljs$core$IFn$_invoke$arity$0 = ep3__0;\n      ep3.cljs$core$IFn$_invoke$arity$1 = ep3__1;\n      ep3.cljs$core$IFn$_invoke$arity$2 = ep3__2;\n      ep3.cljs$core$IFn$_invoke$arity$3 = ep3__3;\n      ep3.cljs$core$IFn$_invoke$arity$variadic = ep3__4.cljs$core$IFn$_invoke$arity$variadic;\n      return ep3;\n    }();\n  };\n  var every_pred__4 = function() {\n    var G__22634__delegate = function(p1, p2, p3, ps) {\n      var ps__$1 = cljs.core.list_STAR_.call(null, p1, p2, p3, ps);\n      return function(ps__$1) {\n        return function() {\n          var epn = null;\n          var epn__0 = function() {\n            return true;\n          };\n          var epn__1 = function(x) {\n            return cljs.core.every_QMARK_.call(null, function(ps__$1) {\n              return function(p1__22624_SHARP_) {\n                return p1__22624_SHARP_.call(null, x);\n              };\n            }(ps__$1), ps__$1);\n          };\n          var epn__2 = function(x, y) {\n            return cljs.core.every_QMARK_.call(null, function(ps__$1) {\n              return function(p1__22625_SHARP_) {\n                var and__17133__auto__ = p1__22625_SHARP_.call(null, x);\n                if (cljs.core.truth_(and__17133__auto__)) {\n                  return p1__22625_SHARP_.call(null, y);\n                } else {\n                  return and__17133__auto__;\n                }\n              };\n            }(ps__$1), ps__$1);\n          };\n          var epn__3 = function(x, y, z) {\n            return cljs.core.every_QMARK_.call(null, function(ps__$1) {\n              return function(p1__22626_SHARP_) {\n                var and__17133__auto__ = p1__22626_SHARP_.call(null, x);\n                if (cljs.core.truth_(and__17133__auto__)) {\n                  var and__17133__auto____$1 = p1__22626_SHARP_.call(null, y);\n                  if (cljs.core.truth_(and__17133__auto____$1)) {\n                    return p1__22626_SHARP_.call(null, z);\n                  } else {\n                    return and__17133__auto____$1;\n                  }\n                } else {\n                  return and__17133__auto__;\n                }\n              };\n            }(ps__$1), ps__$1);\n          };\n          var epn__4 = function() {\n            var G__22635__delegate = function(x, y, z, args) {\n              return cljs.core.boolean$.call(null, epn.call(null, x, y, z) &amp;&amp; cljs.core.every_QMARK_.call(null, function(ps__$1) {\n                return function(p1__22627_SHARP_) {\n                  return cljs.core.every_QMARK_.call(null, p1__22627_SHARP_, args);\n                };\n              }(ps__$1), ps__$1));\n            };\n            var G__22635 = function(x, y, z, var_args) {\n              var args = null;\n              if (arguments.length &gt; 3) {\n                args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n              }\n              return G__22635__delegate.call(this, x, y, z, args);\n            };\n            G__22635.cljs$lang$maxFixedArity = 3;\n            G__22635.cljs$lang$applyTo = function(arglist__22636) {\n              var x = cljs.core.first(arglist__22636);\n              arglist__22636 = cljs.core.next(arglist__22636);\n              var y = cljs.core.first(arglist__22636);\n              arglist__22636 = cljs.core.next(arglist__22636);\n              var z = cljs.core.first(arglist__22636);\n              var args = cljs.core.rest(arglist__22636);\n              return G__22635__delegate(x, y, z, args);\n            };\n            G__22635.cljs$core$IFn$_invoke$arity$variadic = G__22635__delegate;\n            return G__22635;\n          }();\n          epn = function(x, y, z, var_args) {\n            var args = var_args;\n            switch(arguments.length) {\n              case 0:\n                return epn__0.call(this);\n              case 1:\n                return epn__1.call(this, x);\n              case 2:\n                return epn__2.call(this, x, y);\n              case 3:\n                return epn__3.call(this, x, y, z);\n              default:\n                return epn__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          epn.cljs$lang$maxFixedArity = 3;\n          epn.cljs$lang$applyTo = epn__4.cljs$lang$applyTo;\n          epn.cljs$core$IFn$_invoke$arity$0 = epn__0;\n          epn.cljs$core$IFn$_invoke$arity$1 = epn__1;\n          epn.cljs$core$IFn$_invoke$arity$2 = epn__2;\n          epn.cljs$core$IFn$_invoke$arity$3 = epn__3;\n          epn.cljs$core$IFn$_invoke$arity$variadic = epn__4.cljs$core$IFn$_invoke$arity$variadic;\n          return epn;\n        }();\n      }(ps__$1);\n    };\n    var G__22634 = function(p1, p2, p3, var_args) {\n      var ps = null;\n      if (arguments.length &gt; 3) {\n        ps = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n      }\n      return G__22634__delegate.call(this, p1, p2, p3, ps);\n    };\n    G__22634.cljs$lang$maxFixedArity = 3;\n    G__22634.cljs$lang$applyTo = function(arglist__22637) {\n      var p1 = cljs.core.first(arglist__22637);\n      arglist__22637 = cljs.core.next(arglist__22637);\n      var p2 = cljs.core.first(arglist__22637);\n      arglist__22637 = cljs.core.next(arglist__22637);\n      var p3 = cljs.core.first(arglist__22637);\n      var ps = cljs.core.rest(arglist__22637);\n      return G__22634__delegate(p1, p2, p3, ps);\n    };\n    G__22634.cljs$core$IFn$_invoke$arity$variadic = G__22634__delegate;\n    return G__22634;\n  }();\n  every_pred = function(p1, p2, p3, var_args) {\n    var ps = var_args;\n    switch(arguments.length) {\n      case 1:\n        return every_pred__1.call(this, p1);\n      case 2:\n        return every_pred__2.call(this, p1, p2);\n      case 3:\n        return every_pred__3.call(this, p1, p2, p3);\n      default:\n        return every_pred__4.cljs$core$IFn$_invoke$arity$variadic(p1, p2, p3, cljs.core.array_seq(arguments, 3));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  every_pred.cljs$lang$maxFixedArity = 3;\n  every_pred.cljs$lang$applyTo = every_pred__4.cljs$lang$applyTo;\n  every_pred.cljs$core$IFn$_invoke$arity$1 = every_pred__1;\n  every_pred.cljs$core$IFn$_invoke$arity$2 = every_pred__2;\n  every_pred.cljs$core$IFn$_invoke$arity$3 = every_pred__3;\n  every_pred.cljs$core$IFn$_invoke$arity$variadic = every_pred__4.cljs$core$IFn$_invoke$arity$variadic;\n  return every_pred;\n}();\ncljs.core.some_fn = function() {\n  var some_fn = null;\n  var some_fn__1 = function(p) {\n    return function() {\n      var sp1 = null;\n      var sp1__0 = function() {\n        return null;\n      };\n      var sp1__1 = function(x) {\n        return p.call(null, x);\n      };\n      var sp1__2 = function(x, y) {\n        var or__17145__auto__ = p.call(null, x);\n        if (cljs.core.truth_(or__17145__auto__)) {\n          return or__17145__auto__;\n        } else {\n          return p.call(null, y);\n        }\n      };\n      var sp1__3 = function(x, y, z) {\n        var or__17145__auto__ = p.call(null, x);\n        if (cljs.core.truth_(or__17145__auto__)) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = p.call(null, y);\n          if (cljs.core.truth_(or__17145__auto____$1)) {\n            return or__17145__auto____$1;\n          } else {\n            return p.call(null, z);\n          }\n        }\n      };\n      var sp1__4 = function() {\n        var G__22644__delegate = function(x, y, z, args) {\n          var or__17145__auto__ = sp1.call(null, x, y, z);\n          if (cljs.core.truth_(or__17145__auto__)) {\n            return or__17145__auto__;\n          } else {\n            return cljs.core.some.call(null, p, args);\n          }\n        };\n        var G__22644 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22644__delegate.call(this, x, y, z, args);\n        };\n        G__22644.cljs$lang$maxFixedArity = 3;\n        G__22644.cljs$lang$applyTo = function(arglist__22645) {\n          var x = cljs.core.first(arglist__22645);\n          arglist__22645 = cljs.core.next(arglist__22645);\n          var y = cljs.core.first(arglist__22645);\n          arglist__22645 = cljs.core.next(arglist__22645);\n          var z = cljs.core.first(arglist__22645);\n          var args = cljs.core.rest(arglist__22645);\n          return G__22644__delegate(x, y, z, args);\n        };\n        G__22644.cljs$core$IFn$_invoke$arity$variadic = G__22644__delegate;\n        return G__22644;\n      }();\n      sp1 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return sp1__0.call(this);\n          case 1:\n            return sp1__1.call(this, x);\n          case 2:\n            return sp1__2.call(this, x, y);\n          case 3:\n            return sp1__3.call(this, x, y, z);\n          default:\n            return sp1__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      sp1.cljs$lang$maxFixedArity = 3;\n      sp1.cljs$lang$applyTo = sp1__4.cljs$lang$applyTo;\n      sp1.cljs$core$IFn$_invoke$arity$0 = sp1__0;\n      sp1.cljs$core$IFn$_invoke$arity$1 = sp1__1;\n      sp1.cljs$core$IFn$_invoke$arity$2 = sp1__2;\n      sp1.cljs$core$IFn$_invoke$arity$3 = sp1__3;\n      sp1.cljs$core$IFn$_invoke$arity$variadic = sp1__4.cljs$core$IFn$_invoke$arity$variadic;\n      return sp1;\n    }();\n  };\n  var some_fn__2 = function(p1, p2) {\n    return function() {\n      var sp2 = null;\n      var sp2__0 = function() {\n        return null;\n      };\n      var sp2__1 = function(x) {\n        var or__17145__auto__ = p1.call(null, x);\n        if (cljs.core.truth_(or__17145__auto__)) {\n          return or__17145__auto__;\n        } else {\n          return p2.call(null, x);\n        }\n      };\n      var sp2__2 = function(x, y) {\n        var or__17145__auto__ = p1.call(null, x);\n        if (cljs.core.truth_(or__17145__auto__)) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = p1.call(null, y);\n          if (cljs.core.truth_(or__17145__auto____$1)) {\n            return or__17145__auto____$1;\n          } else {\n            var or__17145__auto____$2 = p2.call(null, x);\n            if (cljs.core.truth_(or__17145__auto____$2)) {\n              return or__17145__auto____$2;\n            } else {\n              return p2.call(null, y);\n            }\n          }\n        }\n      };\n      var sp2__3 = function(x, y, z) {\n        var or__17145__auto__ = p1.call(null, x);\n        if (cljs.core.truth_(or__17145__auto__)) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = p1.call(null, y);\n          if (cljs.core.truth_(or__17145__auto____$1)) {\n            return or__17145__auto____$1;\n          } else {\n            var or__17145__auto____$2 = p1.call(null, z);\n            if (cljs.core.truth_(or__17145__auto____$2)) {\n              return or__17145__auto____$2;\n            } else {\n              var or__17145__auto____$3 = p2.call(null, x);\n              if (cljs.core.truth_(or__17145__auto____$3)) {\n                return or__17145__auto____$3;\n              } else {\n                var or__17145__auto____$4 = p2.call(null, y);\n                if (cljs.core.truth_(or__17145__auto____$4)) {\n                  return or__17145__auto____$4;\n                } else {\n                  return p2.call(null, z);\n                }\n              }\n            }\n          }\n        }\n      };\n      var sp2__4 = function() {\n        var G__22646__delegate = function(x, y, z, args) {\n          var or__17145__auto__ = sp2.call(null, x, y, z);\n          if (cljs.core.truth_(or__17145__auto__)) {\n            return or__17145__auto__;\n          } else {\n            return cljs.core.some.call(null, function(or__17145__auto__) {\n              return function(p1__22638_SHARP_) {\n                var or__17145__auto____$1 = p1.call(null, p1__22638_SHARP_);\n                if (cljs.core.truth_(or__17145__auto____$1)) {\n                  return or__17145__auto____$1;\n                } else {\n                  return p2.call(null, p1__22638_SHARP_);\n                }\n              };\n            }(or__17145__auto__), args);\n          }\n        };\n        var G__22646 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22646__delegate.call(this, x, y, z, args);\n        };\n        G__22646.cljs$lang$maxFixedArity = 3;\n        G__22646.cljs$lang$applyTo = function(arglist__22647) {\n          var x = cljs.core.first(arglist__22647);\n          arglist__22647 = cljs.core.next(arglist__22647);\n          var y = cljs.core.first(arglist__22647);\n          arglist__22647 = cljs.core.next(arglist__22647);\n          var z = cljs.core.first(arglist__22647);\n          var args = cljs.core.rest(arglist__22647);\n          return G__22646__delegate(x, y, z, args);\n        };\n        G__22646.cljs$core$IFn$_invoke$arity$variadic = G__22646__delegate;\n        return G__22646;\n      }();\n      sp2 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return sp2__0.call(this);\n          case 1:\n            return sp2__1.call(this, x);\n          case 2:\n            return sp2__2.call(this, x, y);\n          case 3:\n            return sp2__3.call(this, x, y, z);\n          default:\n            return sp2__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      sp2.cljs$lang$maxFixedArity = 3;\n      sp2.cljs$lang$applyTo = sp2__4.cljs$lang$applyTo;\n      sp2.cljs$core$IFn$_invoke$arity$0 = sp2__0;\n      sp2.cljs$core$IFn$_invoke$arity$1 = sp2__1;\n      sp2.cljs$core$IFn$_invoke$arity$2 = sp2__2;\n      sp2.cljs$core$IFn$_invoke$arity$3 = sp2__3;\n      sp2.cljs$core$IFn$_invoke$arity$variadic = sp2__4.cljs$core$IFn$_invoke$arity$variadic;\n      return sp2;\n    }();\n  };\n  var some_fn__3 = function(p1, p2, p3) {\n    return function() {\n      var sp3 = null;\n      var sp3__0 = function() {\n        return null;\n      };\n      var sp3__1 = function(x) {\n        var or__17145__auto__ = p1.call(null, x);\n        if (cljs.core.truth_(or__17145__auto__)) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = p2.call(null, x);\n          if (cljs.core.truth_(or__17145__auto____$1)) {\n            return or__17145__auto____$1;\n          } else {\n            return p3.call(null, x);\n          }\n        }\n      };\n      var sp3__2 = function(x, y) {\n        var or__17145__auto__ = p1.call(null, x);\n        if (cljs.core.truth_(or__17145__auto__)) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = p2.call(null, x);\n          if (cljs.core.truth_(or__17145__auto____$1)) {\n            return or__17145__auto____$1;\n          } else {\n            var or__17145__auto____$2 = p3.call(null, x);\n            if (cljs.core.truth_(or__17145__auto____$2)) {\n              return or__17145__auto____$2;\n            } else {\n              var or__17145__auto____$3 = p1.call(null, y);\n              if (cljs.core.truth_(or__17145__auto____$3)) {\n                return or__17145__auto____$3;\n              } else {\n                var or__17145__auto____$4 = p2.call(null, y);\n                if (cljs.core.truth_(or__17145__auto____$4)) {\n                  return or__17145__auto____$4;\n                } else {\n                  return p3.call(null, y);\n                }\n              }\n            }\n          }\n        }\n      };\n      var sp3__3 = function(x, y, z) {\n        var or__17145__auto__ = p1.call(null, x);\n        if (cljs.core.truth_(or__17145__auto__)) {\n          return or__17145__auto__;\n        } else {\n          var or__17145__auto____$1 = p2.call(null, x);\n          if (cljs.core.truth_(or__17145__auto____$1)) {\n            return or__17145__auto____$1;\n          } else {\n            var or__17145__auto____$2 = p3.call(null, x);\n            if (cljs.core.truth_(or__17145__auto____$2)) {\n              return or__17145__auto____$2;\n            } else {\n              var or__17145__auto____$3 = p1.call(null, y);\n              if (cljs.core.truth_(or__17145__auto____$3)) {\n                return or__17145__auto____$3;\n              } else {\n                var or__17145__auto____$4 = p2.call(null, y);\n                if (cljs.core.truth_(or__17145__auto____$4)) {\n                  return or__17145__auto____$4;\n                } else {\n                  var or__17145__auto____$5 = p3.call(null, y);\n                  if (cljs.core.truth_(or__17145__auto____$5)) {\n                    return or__17145__auto____$5;\n                  } else {\n                    var or__17145__auto____$6 = p1.call(null, z);\n                    if (cljs.core.truth_(or__17145__auto____$6)) {\n                      return or__17145__auto____$6;\n                    } else {\n                      var or__17145__auto____$7 = p2.call(null, z);\n                      if (cljs.core.truth_(or__17145__auto____$7)) {\n                        return or__17145__auto____$7;\n                      } else {\n                        return p3.call(null, z);\n                      }\n                    }\n                  }\n                }\n              }\n            }\n          }\n        }\n      };\n      var sp3__4 = function() {\n        var G__22648__delegate = function(x, y, z, args) {\n          var or__17145__auto__ = sp3.call(null, x, y, z);\n          if (cljs.core.truth_(or__17145__auto__)) {\n            return or__17145__auto__;\n          } else {\n            return cljs.core.some.call(null, function(or__17145__auto__) {\n              return function(p1__22639_SHARP_) {\n                var or__17145__auto____$1 = p1.call(null, p1__22639_SHARP_);\n                if (cljs.core.truth_(or__17145__auto____$1)) {\n                  return or__17145__auto____$1;\n                } else {\n                  var or__17145__auto____$2 = p2.call(null, p1__22639_SHARP_);\n                  if (cljs.core.truth_(or__17145__auto____$2)) {\n                    return or__17145__auto____$2;\n                  } else {\n                    return p3.call(null, p1__22639_SHARP_);\n                  }\n                }\n              };\n            }(or__17145__auto__), args);\n          }\n        };\n        var G__22648 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__22648__delegate.call(this, x, y, z, args);\n        };\n        G__22648.cljs$lang$maxFixedArity = 3;\n        G__22648.cljs$lang$applyTo = function(arglist__22649) {\n          var x = cljs.core.first(arglist__22649);\n          arglist__22649 = cljs.core.next(arglist__22649);\n          var y = cljs.core.first(arglist__22649);\n          arglist__22649 = cljs.core.next(arglist__22649);\n          var z = cljs.core.first(arglist__22649);\n          var args = cljs.core.rest(arglist__22649);\n          return G__22648__delegate(x, y, z, args);\n        };\n        G__22648.cljs$core$IFn$_invoke$arity$variadic = G__22648__delegate;\n        return G__22648;\n      }();\n      sp3 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return sp3__0.call(this);\n          case 1:\n            return sp3__1.call(this, x);\n          case 2:\n            return sp3__2.call(this, x, y);\n          case 3:\n            return sp3__3.call(this, x, y, z);\n          default:\n            return sp3__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      sp3.cljs$lang$maxFixedArity = 3;\n      sp3.cljs$lang$applyTo = sp3__4.cljs$lang$applyTo;\n      sp3.cljs$core$IFn$_invoke$arity$0 = sp3__0;\n      sp3.cljs$core$IFn$_invoke$arity$1 = sp3__1;\n      sp3.cljs$core$IFn$_invoke$arity$2 = sp3__2;\n      sp3.cljs$core$IFn$_invoke$arity$3 = sp3__3;\n      sp3.cljs$core$IFn$_invoke$arity$variadic = sp3__4.cljs$core$IFn$_invoke$arity$variadic;\n      return sp3;\n    }();\n  };\n  var some_fn__4 = function() {\n    var G__22650__delegate = function(p1, p2, p3, ps) {\n      var ps__$1 = cljs.core.list_STAR_.call(null, p1, p2, p3, ps);\n      return function(ps__$1) {\n        return function() {\n          var spn = null;\n          var spn__0 = function() {\n            return null;\n          };\n          var spn__1 = function(x) {\n            return cljs.core.some.call(null, function(ps__$1) {\n              return function(p1__22640_SHARP_) {\n                return p1__22640_SHARP_.call(null, x);\n              };\n            }(ps__$1), ps__$1);\n          };\n          var spn__2 = function(x, y) {\n            return cljs.core.some.call(null, function(ps__$1) {\n              return function(p1__22641_SHARP_) {\n                var or__17145__auto__ = p1__22641_SHARP_.call(null, x);\n                if (cljs.core.truth_(or__17145__auto__)) {\n                  return or__17145__auto__;\n                } else {\n                  return p1__22641_SHARP_.call(null, y);\n                }\n              };\n            }(ps__$1), ps__$1);\n          };\n          var spn__3 = function(x, y, z) {\n            return cljs.core.some.call(null, function(ps__$1) {\n              return function(p1__22642_SHARP_) {\n                var or__17145__auto__ = p1__22642_SHARP_.call(null, x);\n                if (cljs.core.truth_(or__17145__auto__)) {\n                  return or__17145__auto__;\n                } else {\n                  var or__17145__auto____$1 = p1__22642_SHARP_.call(null, y);\n                  if (cljs.core.truth_(or__17145__auto____$1)) {\n                    return or__17145__auto____$1;\n                  } else {\n                    return p1__22642_SHARP_.call(null, z);\n                  }\n                }\n              };\n            }(ps__$1), ps__$1);\n          };\n          var spn__4 = function() {\n            var G__22651__delegate = function(x, y, z, args) {\n              var or__17145__auto__ = spn.call(null, x, y, z);\n              if (cljs.core.truth_(or__17145__auto__)) {\n                return or__17145__auto__;\n              } else {\n                return cljs.core.some.call(null, function(or__17145__auto__, ps__$1) {\n                  return function(p1__22643_SHARP_) {\n                    return cljs.core.some.call(null, p1__22643_SHARP_, args);\n                  };\n                }(or__17145__auto__, ps__$1), ps__$1);\n              }\n            };\n            var G__22651 = function(x, y, z, var_args) {\n              var args = null;\n              if (arguments.length &gt; 3) {\n                args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n              }\n              return G__22651__delegate.call(this, x, y, z, args);\n            };\n            G__22651.cljs$lang$maxFixedArity = 3;\n            G__22651.cljs$lang$applyTo = function(arglist__22652) {\n              var x = cljs.core.first(arglist__22652);\n              arglist__22652 = cljs.core.next(arglist__22652);\n              var y = cljs.core.first(arglist__22652);\n              arglist__22652 = cljs.core.next(arglist__22652);\n              var z = cljs.core.first(arglist__22652);\n              var args = cljs.core.rest(arglist__22652);\n              return G__22651__delegate(x, y, z, args);\n            };\n            G__22651.cljs$core$IFn$_invoke$arity$variadic = G__22651__delegate;\n            return G__22651;\n          }();\n          spn = function(x, y, z, var_args) {\n            var args = var_args;\n            switch(arguments.length) {\n              case 0:\n                return spn__0.call(this);\n              case 1:\n                return spn__1.call(this, x);\n              case 2:\n                return spn__2.call(this, x, y);\n              case 3:\n                return spn__3.call(this, x, y, z);\n              default:\n                return spn__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          spn.cljs$lang$maxFixedArity = 3;\n          spn.cljs$lang$applyTo = spn__4.cljs$lang$applyTo;\n          spn.cljs$core$IFn$_invoke$arity$0 = spn__0;\n          spn.cljs$core$IFn$_invoke$arity$1 = spn__1;\n          spn.cljs$core$IFn$_invoke$arity$2 = spn__2;\n          spn.cljs$core$IFn$_invoke$arity$3 = spn__3;\n          spn.cljs$core$IFn$_invoke$arity$variadic = spn__4.cljs$core$IFn$_invoke$arity$variadic;\n          return spn;\n        }();\n      }(ps__$1);\n    };\n    var G__22650 = function(p1, p2, p3, var_args) {\n      var ps = null;\n      if (arguments.length &gt; 3) {\n        ps = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n      }\n      return G__22650__delegate.call(this, p1, p2, p3, ps);\n    };\n    G__22650.cljs$lang$maxFixedArity = 3;\n    G__22650.cljs$lang$applyTo = function(arglist__22653) {\n      var p1 = cljs.core.first(arglist__22653);\n      arglist__22653 = cljs.core.next(arglist__22653);\n      var p2 = cljs.core.first(arglist__22653);\n      arglist__22653 = cljs.core.next(arglist__22653);\n      var p3 = cljs.core.first(arglist__22653);\n      var ps = cljs.core.rest(arglist__22653);\n      return G__22650__delegate(p1, p2, p3, ps);\n    };\n    G__22650.cljs$core$IFn$_invoke$arity$variadic = G__22650__delegate;\n    return G__22650;\n  }();\n  some_fn = function(p1, p2, p3, var_args) {\n    var ps = var_args;\n    switch(arguments.length) {\n      case 1:\n        return some_fn__1.call(this, p1);\n      case 2:\n        return some_fn__2.call(this, p1, p2);\n      case 3:\n        return some_fn__3.call(this, p1, p2, p3);\n      default:\n        return some_fn__4.cljs$core$IFn$_invoke$arity$variadic(p1, p2, p3, cljs.core.array_seq(arguments, 3));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  some_fn.cljs$lang$maxFixedArity = 3;\n  some_fn.cljs$lang$applyTo = some_fn__4.cljs$lang$applyTo;\n  some_fn.cljs$core$IFn$_invoke$arity$1 = some_fn__1;\n  some_fn.cljs$core$IFn$_invoke$arity$2 = some_fn__2;\n  some_fn.cljs$core$IFn$_invoke$arity$3 = some_fn__3;\n  some_fn.cljs$core$IFn$_invoke$arity$variadic = some_fn__4.cljs$core$IFn$_invoke$arity$variadic;\n  return some_fn;\n}();\ncljs.core.map = function() {\n  var map = null;\n  var map__1 = function(f) {\n    return function(f1) {\n      return function() {\n        var G__22655 = null;\n        var G__22655__0 = function() {\n          return f1.call(null);\n        };\n        var G__22655__1 = function(result) {\n          return f1.call(null, result);\n        };\n        var G__22655__2 = function(result, input) {\n          return f1.call(null, result, f.call(null, input));\n        };\n        var G__22655__3 = function() {\n          var G__22656__delegate = function(result, input, inputs) {\n            return f1.call(null, result, cljs.core.apply.call(null, f, input, inputs));\n          };\n          var G__22656 = function(result, input, var_args) {\n            var inputs = null;\n            if (arguments.length &gt; 2) {\n              inputs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n            }\n            return G__22656__delegate.call(this, result, input, inputs);\n          };\n          G__22656.cljs$lang$maxFixedArity = 2;\n          G__22656.cljs$lang$applyTo = function(arglist__22657) {\n            var result = cljs.core.first(arglist__22657);\n            arglist__22657 = cljs.core.next(arglist__22657);\n            var input = cljs.core.first(arglist__22657);\n            var inputs = cljs.core.rest(arglist__22657);\n            return G__22656__delegate(result, input, inputs);\n          };\n          G__22656.cljs$core$IFn$_invoke$arity$variadic = G__22656__delegate;\n          return G__22656;\n        }();\n        G__22655 = function(result, input, var_args) {\n          var inputs = var_args;\n          switch(arguments.length) {\n            case 0:\n              return G__22655__0.call(this);\n            case 1:\n              return G__22655__1.call(this, result);\n            case 2:\n              return G__22655__2.call(this, result, input);\n            default:\n              return G__22655__3.cljs$core$IFn$_invoke$arity$variadic(result, input, cljs.core.array_seq(arguments, 2));\n          }\n          throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n        };\n        G__22655.cljs$lang$maxFixedArity = 2;\n        G__22655.cljs$lang$applyTo = G__22655__3.cljs$lang$applyTo;\n        G__22655.cljs$core$IFn$_invoke$arity$0 = G__22655__0;\n        G__22655.cljs$core$IFn$_invoke$arity$1 = G__22655__1;\n        G__22655.cljs$core$IFn$_invoke$arity$2 = G__22655__2;\n        G__22655.cljs$core$IFn$_invoke$arity$variadic = G__22655__3.cljs$core$IFn$_invoke$arity$variadic;\n        return G__22655;\n      }();\n    };\n  };\n  var map__2 = function(f, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, s)) {\n          var c = cljs.core.chunk_first.call(null, s);\n          var size = cljs.core.count.call(null, c);\n          var b = cljs.core.chunk_buffer.call(null, size);\n          var n__18001__auto___22658 = size;\n          var i_22659 = 0;\n          while (true) {\n            if (i_22659 &lt; n__18001__auto___22658) {\n              cljs.core.chunk_append.call(null, b, f.call(null, cljs.core._nth.call(null, c, i_22659)));\n              var G__22660 = i_22659 + 1;\n              i_22659 = G__22660;\n              continue;\n            } else {\n            }\n            break;\n          }\n          return cljs.core.chunk_cons.call(null, cljs.core.chunk.call(null, b), map.call(null, f, cljs.core.chunk_rest.call(null, s)));\n        } else {\n          return cljs.core.cons.call(null, f.call(null, cljs.core.first.call(null, s)), map.call(null, f, cljs.core.rest.call(null, s)));\n        }\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  var map__3 = function(f, c1, c2) {\n    return new cljs.core.LazySeq(null, function() {\n      var s1 = cljs.core.seq.call(null, c1);\n      var s2 = cljs.core.seq.call(null, c2);\n      if (s1 &amp;&amp; s2) {\n        return cljs.core.cons.call(null, f.call(null, cljs.core.first.call(null, s1), cljs.core.first.call(null, s2)), map.call(null, f, cljs.core.rest.call(null, s1), cljs.core.rest.call(null, s2)));\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  var map__4 = function(f, c1, c2, c3) {\n    return new cljs.core.LazySeq(null, function() {\n      var s1 = cljs.core.seq.call(null, c1);\n      var s2 = cljs.core.seq.call(null, c2);\n      var s3 = cljs.core.seq.call(null, c3);\n      if (s1 &amp;&amp; s2 &amp;&amp; s3) {\n        return cljs.core.cons.call(null, f.call(null, cljs.core.first.call(null, s1), cljs.core.first.call(null, s2), cljs.core.first.call(null, s3)), map.call(null, f, cljs.core.rest.call(null, s1), cljs.core.rest.call(null, s2), cljs.core.rest.call(null, s3)));\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  var map__5 = function() {\n    var G__22661__delegate = function(f, c1, c2, c3, colls) {\n      var step = function step(cs) {\n        return new cljs.core.LazySeq(null, function() {\n          var ss = map.call(null, cljs.core.seq, cs);\n          if (cljs.core.every_QMARK_.call(null, cljs.core.identity, ss)) {\n            return cljs.core.cons.call(null, map.call(null, cljs.core.first, ss), step.call(null, map.call(null, cljs.core.rest, ss)));\n          } else {\n            return null;\n          }\n        }, null, null);\n      };\n      return map.call(null, function(step) {\n        return function(p1__22654_SHARP_) {\n          return cljs.core.apply.call(null, f, p1__22654_SHARP_);\n        };\n      }(step), step.call(null, cljs.core.conj.call(null, colls, c3, c2, c1)));\n    };\n    var G__22661 = function(f, c1, c2, c3, var_args) {\n      var colls = null;\n      if (arguments.length &gt; 4) {\n        colls = cljs.core.array_seq(Array.prototype.slice.call(arguments, 4), 0);\n      }\n      return G__22661__delegate.call(this, f, c1, c2, c3, colls);\n    };\n    G__22661.cljs$lang$maxFixedArity = 4;\n    G__22661.cljs$lang$applyTo = function(arglist__22662) {\n      var f = cljs.core.first(arglist__22662);\n      arglist__22662 = cljs.core.next(arglist__22662);\n      var c1 = cljs.core.first(arglist__22662);\n      arglist__22662 = cljs.core.next(arglist__22662);\n      var c2 = cljs.core.first(arglist__22662);\n      arglist__22662 = cljs.core.next(arglist__22662);\n      var c3 = cljs.core.first(arglist__22662);\n      var colls = cljs.core.rest(arglist__22662);\n      return G__22661__delegate(f, c1, c2, c3, colls);\n    };\n    G__22661.cljs$core$IFn$_invoke$arity$variadic = G__22661__delegate;\n    return G__22661;\n  }();\n  map = function(f, c1, c2, c3, var_args) {\n    var colls = var_args;\n    switch(arguments.length) {\n      case 1:\n        return map__1.call(this, f);\n      case 2:\n        return map__2.call(this, f, c1);\n      case 3:\n        return map__3.call(this, f, c1, c2);\n      case 4:\n        return map__4.call(this, f, c1, c2, c3);\n      default:\n        return map__5.cljs$core$IFn$_invoke$arity$variadic(f, c1, c2, c3, cljs.core.array_seq(arguments, 4));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  map.cljs$lang$maxFixedArity = 4;\n  map.cljs$lang$applyTo = map__5.cljs$lang$applyTo;\n  map.cljs$core$IFn$_invoke$arity$1 = map__1;\n  map.cljs$core$IFn$_invoke$arity$2 = map__2;\n  map.cljs$core$IFn$_invoke$arity$3 = map__3;\n  map.cljs$core$IFn$_invoke$arity$4 = map__4;\n  map.cljs$core$IFn$_invoke$arity$variadic = map__5.cljs$core$IFn$_invoke$arity$variadic;\n  return map;\n}();\ncljs.core.take = function() {\n  var take = null;\n  var take__1 = function(n) {\n    return function(f1) {\n      var na = cljs.core.atom.call(null, n);\n      return function(na) {\n        return function() {\n          var G__22663 = null;\n          var G__22663__0 = function() {\n            return f1.call(null);\n          };\n          var G__22663__1 = function(result) {\n            return f1.call(null, result);\n          };\n          var G__22663__2 = function(result, input) {\n            var n__$1 = cljs.core.deref.call(null, na);\n            var nn = cljs.core.swap_BANG_.call(null, na, cljs.core.dec);\n            var result__$1 = n__$1 &gt; 0 ? f1.call(null, result, input) : result;\n            if (!(nn &gt; 0)) {\n              return cljs.core.reduced.call(null, result__$1);\n            } else {\n              return result__$1;\n            }\n          };\n          G__22663 = function(result, input) {\n            switch(arguments.length) {\n              case 0:\n                return G__22663__0.call(this);\n              case 1:\n                return G__22663__1.call(this, result);\n              case 2:\n                return G__22663__2.call(this, result, input);\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          G__22663.cljs$core$IFn$_invoke$arity$0 = G__22663__0;\n          G__22663.cljs$core$IFn$_invoke$arity$1 = G__22663__1;\n          G__22663.cljs$core$IFn$_invoke$arity$2 = G__22663__2;\n          return G__22663;\n        }();\n      }(na);\n    };\n  };\n  var take__2 = function(n, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      if (n &gt; 0) {\n        var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n        if (temp__4126__auto__) {\n          var s = temp__4126__auto__;\n          return cljs.core.cons.call(null, cljs.core.first.call(null, s), take.call(null, n - 1, cljs.core.rest.call(null, s)));\n        } else {\n          return null;\n        }\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  take = function(n, coll) {\n    switch(arguments.length) {\n      case 1:\n        return take__1.call(this, n);\n      case 2:\n        return take__2.call(this, n, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  take.cljs$core$IFn$_invoke$arity$1 = take__1;\n  take.cljs$core$IFn$_invoke$arity$2 = take__2;\n  return take;\n}();\ncljs.core.drop = function() {\n  var drop = null;\n  var drop__1 = function(n) {\n    return function(f1) {\n      var na = cljs.core.atom.call(null, n);\n      return function(na) {\n        return function() {\n          var G__22664 = null;\n          var G__22664__0 = function() {\n            return f1.call(null);\n          };\n          var G__22664__1 = function(result) {\n            return f1.call(null, result);\n          };\n          var G__22664__2 = function(result, input) {\n            var n__$1 = cljs.core.deref.call(null, na);\n            cljs.core.swap_BANG_.call(null, na, cljs.core.dec);\n            if (n__$1 &gt; 0) {\n              return result;\n            } else {\n              return f1.call(null, result, input);\n            }\n          };\n          G__22664 = function(result, input) {\n            switch(arguments.length) {\n              case 0:\n                return G__22664__0.call(this);\n              case 1:\n                return G__22664__1.call(this, result);\n              case 2:\n                return G__22664__2.call(this, result, input);\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          G__22664.cljs$core$IFn$_invoke$arity$0 = G__22664__0;\n          G__22664.cljs$core$IFn$_invoke$arity$1 = G__22664__1;\n          G__22664.cljs$core$IFn$_invoke$arity$2 = G__22664__2;\n          return G__22664;\n        }();\n      }(na);\n    };\n  };\n  var drop__2 = function(n, coll) {\n    var step = function(n__$1, coll__$1) {\n      while (true) {\n        var s = cljs.core.seq.call(null, coll__$1);\n        if (n__$1 &gt; 0 &amp;&amp; s) {\n          var G__22665 = n__$1 - 1;\n          var G__22666 = cljs.core.rest.call(null, s);\n          n__$1 = G__22665;\n          coll__$1 = G__22666;\n          continue;\n        } else {\n          return s;\n        }\n        break;\n      }\n    };\n    return new cljs.core.LazySeq(null, function(step) {\n      return function() {\n        return step.call(null, n, coll);\n      };\n    }(step), null, null);\n  };\n  drop = function(n, coll) {\n    switch(arguments.length) {\n      case 1:\n        return drop__1.call(this, n);\n      case 2:\n        return drop__2.call(this, n, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  drop.cljs$core$IFn$_invoke$arity$1 = drop__1;\n  drop.cljs$core$IFn$_invoke$arity$2 = drop__2;\n  return drop;\n}();\ncljs.core.drop_last = function() {\n  var drop_last = null;\n  var drop_last__1 = function(s) {\n    return drop_last.call(null, 1, s);\n  };\n  var drop_last__2 = function(n, s) {\n    return cljs.core.map.call(null, function(x, _) {\n      return x;\n    }, s, cljs.core.drop.call(null, n, s));\n  };\n  drop_last = function(n, s) {\n    switch(arguments.length) {\n      case 1:\n        return drop_last__1.call(this, n);\n      case 2:\n        return drop_last__2.call(this, n, s);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  drop_last.cljs$core$IFn$_invoke$arity$1 = drop_last__1;\n  drop_last.cljs$core$IFn$_invoke$arity$2 = drop_last__2;\n  return drop_last;\n}();\ncljs.core.take_last = function take_last(n, coll) {\n  var s = cljs.core.seq.call(null, coll);\n  var lead = cljs.core.seq.call(null, cljs.core.drop.call(null, n, coll));\n  while (true) {\n    if (lead) {\n      var G__22667 = cljs.core.next.call(null, s);\n      var G__22668 = cljs.core.next.call(null, lead);\n      s = G__22667;\n      lead = G__22668;\n      continue;\n    } else {\n      return s;\n    }\n    break;\n  }\n};\ncljs.core.drop_while = function() {\n  var drop_while = null;\n  var drop_while__1 = function(pred) {\n    return function(f1) {\n      var da = cljs.core.atom.call(null, true);\n      return function(da) {\n        return function() {\n          var G__22669 = null;\n          var G__22669__0 = function() {\n            return f1.call(null);\n          };\n          var G__22669__1 = function(result) {\n            return f1.call(null, result);\n          };\n          var G__22669__2 = function(result, input) {\n            var drop_QMARK_ = cljs.core.deref.call(null, da);\n            if (cljs.core.truth_(function() {\n              var and__17133__auto__ = drop_QMARK_;\n              if (cljs.core.truth_(and__17133__auto__)) {\n                return pred.call(null, input);\n              } else {\n                return and__17133__auto__;\n              }\n            }())) {\n              return result;\n            } else {\n              cljs.core.reset_BANG_.call(null, da, null);\n              return f1.call(null, result, input);\n            }\n          };\n          G__22669 = function(result, input) {\n            switch(arguments.length) {\n              case 0:\n                return G__22669__0.call(this);\n              case 1:\n                return G__22669__1.call(this, result);\n              case 2:\n                return G__22669__2.call(this, result, input);\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          G__22669.cljs$core$IFn$_invoke$arity$0 = G__22669__0;\n          G__22669.cljs$core$IFn$_invoke$arity$1 = G__22669__1;\n          G__22669.cljs$core$IFn$_invoke$arity$2 = G__22669__2;\n          return G__22669;\n        }();\n      }(da);\n    };\n  };\n  var drop_while__2 = function(pred, coll) {\n    var step = function(pred__$1, coll__$1) {\n      while (true) {\n        var s = cljs.core.seq.call(null, coll__$1);\n        if (cljs.core.truth_(function() {\n          var and__17133__auto__ = s;\n          if (and__17133__auto__) {\n            return pred__$1.call(null, cljs.core.first.call(null, s));\n          } else {\n            return and__17133__auto__;\n          }\n        }())) {\n          var G__22670 = pred__$1;\n          var G__22671 = cljs.core.rest.call(null, s);\n          pred__$1 = G__22670;\n          coll__$1 = G__22671;\n          continue;\n        } else {\n          return s;\n        }\n        break;\n      }\n    };\n    return new cljs.core.LazySeq(null, function(step) {\n      return function() {\n        return step.call(null, pred, coll);\n      };\n    }(step), null, null);\n  };\n  drop_while = function(pred, coll) {\n    switch(arguments.length) {\n      case 1:\n        return drop_while__1.call(this, pred);\n      case 2:\n        return drop_while__2.call(this, pred, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  drop_while.cljs$core$IFn$_invoke$arity$1 = drop_while__1;\n  drop_while.cljs$core$IFn$_invoke$arity$2 = drop_while__2;\n  return drop_while;\n}();\ncljs.core.cycle = function cycle(coll) {\n  return new cljs.core.LazySeq(null, function() {\n    var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n    if (temp__4126__auto__) {\n      var s = temp__4126__auto__;\n      return cljs.core.concat.call(null, s, cycle.call(null, s));\n    } else {\n      return null;\n    }\n  }, null, null);\n};\ncljs.core.split_at = function split_at(n, coll) {\n  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.take.call(null, n, coll), cljs.core.drop.call(null, n, coll)], null);\n};\ncljs.core.repeat = function() {\n  var repeat = null;\n  var repeat__1 = function(x) {\n    return new cljs.core.LazySeq(null, function() {\n      return cljs.core.cons.call(null, x, repeat.call(null, x));\n    }, null, null);\n  };\n  var repeat__2 = function(n, x) {\n    return cljs.core.take.call(null, n, repeat.call(null, x));\n  };\n  repeat = function(n, x) {\n    switch(arguments.length) {\n      case 1:\n        return repeat__1.call(this, n);\n      case 2:\n        return repeat__2.call(this, n, x);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  repeat.cljs$core$IFn$_invoke$arity$1 = repeat__1;\n  repeat.cljs$core$IFn$_invoke$arity$2 = repeat__2;\n  return repeat;\n}();\ncljs.core.replicate = function replicate(n, x) {\n  return cljs.core.take.call(null, n, cljs.core.repeat.call(null, x));\n};\ncljs.core.repeatedly = function() {\n  var repeatedly = null;\n  var repeatedly__1 = function(f) {\n    return new cljs.core.LazySeq(null, function() {\n      return cljs.core.cons.call(null, f.call(null), repeatedly.call(null, f));\n    }, null, null);\n  };\n  var repeatedly__2 = function(n, f) {\n    return cljs.core.take.call(null, n, repeatedly.call(null, f));\n  };\n  repeatedly = function(n, f) {\n    switch(arguments.length) {\n      case 1:\n        return repeatedly__1.call(this, n);\n      case 2:\n        return repeatedly__2.call(this, n, f);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  repeatedly.cljs$core$IFn$_invoke$arity$1 = repeatedly__1;\n  repeatedly.cljs$core$IFn$_invoke$arity$2 = repeatedly__2;\n  return repeatedly;\n}();\ncljs.core.iterate = function iterate(f, x) {\n  return cljs.core.cons.call(null, x, new cljs.core.LazySeq(null, function() {\n    return iterate.call(null, f, f.call(null, x));\n  }, null, null));\n};\ncljs.core.interleave = function() {\n  var interleave = null;\n  var interleave__2 = function(c1, c2) {\n    return new cljs.core.LazySeq(null, function() {\n      var s1 = cljs.core.seq.call(null, c1);\n      var s2 = cljs.core.seq.call(null, c2);\n      if (s1 &amp;&amp; s2) {\n        return cljs.core.cons.call(null, cljs.core.first.call(null, s1), cljs.core.cons.call(null, cljs.core.first.call(null, s2), interleave.call(null, cljs.core.rest.call(null, s1), cljs.core.rest.call(null, s2))));\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  var interleave__3 = function() {\n    var G__22672__delegate = function(c1, c2, colls) {\n      return new cljs.core.LazySeq(null, function() {\n        var ss = cljs.core.map.call(null, cljs.core.seq, cljs.core.conj.call(null, colls, c2, c1));\n        if (cljs.core.every_QMARK_.call(null, cljs.core.identity, ss)) {\n          return cljs.core.concat.call(null, cljs.core.map.call(null, cljs.core.first, ss), cljs.core.apply.call(null, interleave, cljs.core.map.call(null, cljs.core.rest, ss)));\n        } else {\n          return null;\n        }\n      }, null, null);\n    };\n    var G__22672 = function(c1, c2, var_args) {\n      var colls = null;\n      if (arguments.length &gt; 2) {\n        colls = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22672__delegate.call(this, c1, c2, colls);\n    };\n    G__22672.cljs$lang$maxFixedArity = 2;\n    G__22672.cljs$lang$applyTo = function(arglist__22673) {\n      var c1 = cljs.core.first(arglist__22673);\n      arglist__22673 = cljs.core.next(arglist__22673);\n      var c2 = cljs.core.first(arglist__22673);\n      var colls = cljs.core.rest(arglist__22673);\n      return G__22672__delegate(c1, c2, colls);\n    };\n    G__22672.cljs$core$IFn$_invoke$arity$variadic = G__22672__delegate;\n    return G__22672;\n  }();\n  interleave = function(c1, c2, var_args) {\n    var colls = var_args;\n    switch(arguments.length) {\n      case 2:\n        return interleave__2.call(this, c1, c2);\n      default:\n        return interleave__3.cljs$core$IFn$_invoke$arity$variadic(c1, c2, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  interleave.cljs$lang$maxFixedArity = 2;\n  interleave.cljs$lang$applyTo = interleave__3.cljs$lang$applyTo;\n  interleave.cljs$core$IFn$_invoke$arity$2 = interleave__2;\n  interleave.cljs$core$IFn$_invoke$arity$variadic = interleave__3.cljs$core$IFn$_invoke$arity$variadic;\n  return interleave;\n}();\ncljs.core.interpose = function interpose(sep, coll) {\n  return cljs.core.drop.call(null, 1, cljs.core.interleave.call(null, cljs.core.repeat.call(null, sep), coll));\n};\ncljs.core.flatten1 = function flatten1(colls) {\n  var cat = function cat(coll, colls__$1) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4124__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4124__auto__) {\n        var coll__$1 = temp__4124__auto__;\n        return cljs.core.cons.call(null, cljs.core.first.call(null, coll__$1), cat.call(null, cljs.core.rest.call(null, coll__$1), colls__$1));\n      } else {\n        if (cljs.core.seq.call(null, colls__$1)) {\n          return cat.call(null, cljs.core.first.call(null, colls__$1), cljs.core.rest.call(null, colls__$1));\n        } else {\n          return null;\n        }\n      }\n    }, null, null);\n  };\n  return cat.call(null, null, colls);\n};\ncljs.core.mapcat = function() {\n  var mapcat = null;\n  var mapcat__2 = function(f, coll) {\n    return cljs.core.flatten1.call(null, cljs.core.map.call(null, f, coll));\n  };\n  var mapcat__3 = function() {\n    var G__22674__delegate = function(f, coll, colls) {\n      return cljs.core.flatten1.call(null, cljs.core.apply.call(null, cljs.core.map, f, coll, colls));\n    };\n    var G__22674 = function(f, coll, var_args) {\n      var colls = null;\n      if (arguments.length &gt; 2) {\n        colls = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n      }\n      return G__22674__delegate.call(this, f, coll, colls);\n    };\n    G__22674.cljs$lang$maxFixedArity = 2;\n    G__22674.cljs$lang$applyTo = function(arglist__22675) {\n      var f = cljs.core.first(arglist__22675);\n      arglist__22675 = cljs.core.next(arglist__22675);\n      var coll = cljs.core.first(arglist__22675);\n      var colls = cljs.core.rest(arglist__22675);\n      return G__22674__delegate(f, coll, colls);\n    };\n    G__22674.cljs$core$IFn$_invoke$arity$variadic = G__22674__delegate;\n    return G__22674;\n  }();\n  mapcat = function(f, coll, var_args) {\n    var colls = var_args;\n    switch(arguments.length) {\n      case 2:\n        return mapcat__2.call(this, f, coll);\n      default:\n        return mapcat__3.cljs$core$IFn$_invoke$arity$variadic(f, coll, cljs.core.array_seq(arguments, 2));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  mapcat.cljs$lang$maxFixedArity = 2;\n  mapcat.cljs$lang$applyTo = mapcat__3.cljs$lang$applyTo;\n  mapcat.cljs$core$IFn$_invoke$arity$2 = mapcat__2;\n  mapcat.cljs$core$IFn$_invoke$arity$variadic = mapcat__3.cljs$core$IFn$_invoke$arity$variadic;\n  return mapcat;\n}();\ncljs.core.filter = function() {\n  var filter = null;\n  var filter__1 = function(pred) {\n    return function(f1) {\n      return function() {\n        var G__22676 = null;\n        var G__22676__0 = function() {\n          return f1.call(null);\n        };\n        var G__22676__1 = function(result) {\n          return f1.call(null, result);\n        };\n        var G__22676__2 = function(result, input) {\n          if (cljs.core.truth_(pred.call(null, input))) {\n            return f1.call(null, result, input);\n          } else {\n            return result;\n          }\n        };\n        G__22676 = function(result, input) {\n          switch(arguments.length) {\n            case 0:\n              return G__22676__0.call(this);\n            case 1:\n              return G__22676__1.call(this, result);\n            case 2:\n              return G__22676__2.call(this, result, input);\n          }\n          throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n        };\n        G__22676.cljs$core$IFn$_invoke$arity$0 = G__22676__0;\n        G__22676.cljs$core$IFn$_invoke$arity$1 = G__22676__1;\n        G__22676.cljs$core$IFn$_invoke$arity$2 = G__22676__2;\n        return G__22676;\n      }();\n    };\n  };\n  var filter__2 = function(pred, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, s)) {\n          var c = cljs.core.chunk_first.call(null, s);\n          var size = cljs.core.count.call(null, c);\n          var b = cljs.core.chunk_buffer.call(null, size);\n          var n__18001__auto___22677 = size;\n          var i_22678 = 0;\n          while (true) {\n            if (i_22678 &lt; n__18001__auto___22677) {\n              if (cljs.core.truth_(pred.call(null, cljs.core._nth.call(null, c, i_22678)))) {\n                cljs.core.chunk_append.call(null, b, cljs.core._nth.call(null, c, i_22678));\n              } else {\n              }\n              var G__22679 = i_22678 + 1;\n              i_22678 = G__22679;\n              continue;\n            } else {\n            }\n            break;\n          }\n          return cljs.core.chunk_cons.call(null, cljs.core.chunk.call(null, b), filter.call(null, pred, cljs.core.chunk_rest.call(null, s)));\n        } else {\n          var f = cljs.core.first.call(null, s);\n          var r = cljs.core.rest.call(null, s);\n          if (cljs.core.truth_(pred.call(null, f))) {\n            return cljs.core.cons.call(null, f, filter.call(null, pred, r));\n          } else {\n            return filter.call(null, pred, r);\n          }\n        }\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  filter = function(pred, coll) {\n    switch(arguments.length) {\n      case 1:\n        return filter__1.call(this, pred);\n      case 2:\n        return filter__2.call(this, pred, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  filter.cljs$core$IFn$_invoke$arity$1 = filter__1;\n  filter.cljs$core$IFn$_invoke$arity$2 = filter__2;\n  return filter;\n}();\ncljs.core.remove = function() {\n  var remove = null;\n  var remove__1 = function(pred) {\n    return cljs.core.filter.call(null, cljs.core.complement.call(null, pred));\n  };\n  var remove__2 = function(pred, coll) {\n    return cljs.core.filter.call(null, cljs.core.complement.call(null, pred), coll);\n  };\n  remove = function(pred, coll) {\n    switch(arguments.length) {\n      case 1:\n        return remove__1.call(this, pred);\n      case 2:\n        return remove__2.call(this, pred, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  remove.cljs$core$IFn$_invoke$arity$1 = remove__1;\n  remove.cljs$core$IFn$_invoke$arity$2 = remove__2;\n  return remove;\n}();\ncljs.core.tree_seq = function tree_seq(branch_QMARK_, children, root) {\n  var walk = function walk(node) {\n    return new cljs.core.LazySeq(null, function() {\n      return cljs.core.cons.call(null, node, cljs.core.truth_(branch_QMARK_.call(null, node)) ? cljs.core.mapcat.call(null, walk, children.call(null, node)) : null);\n    }, null, null);\n  };\n  return walk.call(null, root);\n};\ncljs.core.flatten = function flatten(x) {\n  return cljs.core.filter.call(null, function(p1__22680_SHARP_) {\n    return!cljs.core.sequential_QMARK_.call(null, p1__22680_SHARP_);\n  }, cljs.core.rest.call(null, cljs.core.tree_seq.call(null, cljs.core.sequential_QMARK_, cljs.core.seq, x)));\n};\ncljs.core.into = function() {\n  var into = null;\n  var into__2 = function(to, from) {\n    if (!(to == null)) {\n      if (function() {\n        var G__22683 = to;\n        if (G__22683) {\n          var bit__17788__auto__ = G__22683.cljs$lang$protocol_mask$partition1$ &amp; 4;\n          if (bit__17788__auto__ || G__22683.cljs$core$IEditableCollection$) {\n            return true;\n          } else {\n            return false;\n          }\n        } else {\n          return false;\n        }\n      }()) {\n        return cljs.core.with_meta.call(null, cljs.core.persistent_BANG_.call(null, cljs.core.reduce.call(null, cljs.core._conj_BANG_, cljs.core.transient$.call(null, to), from)), cljs.core.meta.call(null, to));\n      } else {\n        return cljs.core.reduce.call(null, cljs.core._conj, to, from);\n      }\n    } else {\n      return cljs.core.reduce.call(null, cljs.core.conj, cljs.core.List.EMPTY, from);\n    }\n  };\n  var into__3 = function(to, xform, from) {\n    if (function() {\n      var G__22684 = to;\n      if (G__22684) {\n        var bit__17788__auto__ = G__22684.cljs$lang$protocol_mask$partition1$ &amp; 4;\n        if (bit__17788__auto__ || G__22684.cljs$core$IEditableCollection$) {\n          return true;\n        } else {\n          return false;\n        }\n      } else {\n        return false;\n      }\n    }()) {\n      return cljs.core.with_meta.call(null, cljs.core.persistent_BANG_.call(null, cljs.core.transduce.call(null, xform, cljs.core._conj_BANG_, cljs.core.transient$.call(null, to), from)), cljs.core.meta.call(null, to));\n    } else {\n      return cljs.core.transduce.call(null, xform, cljs.core.conj, to, from);\n    }\n  };\n  into = function(to, xform, from) {\n    switch(arguments.length) {\n      case 2:\n        return into__2.call(this, to, xform);\n      case 3:\n        return into__3.call(this, to, xform, from);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  into.cljs$core$IFn$_invoke$arity$2 = into__2;\n  into.cljs$core$IFn$_invoke$arity$3 = into__3;\n  return into;\n}();\ncljs.core.mapv = function() {\n  var mapv = null;\n  var mapv__2 = function(f, coll) {\n    return cljs.core.persistent_BANG_.call(null, cljs.core.reduce.call(null, function(v, o) {\n      return cljs.core.conj_BANG_.call(null, v, f.call(null, o));\n    }, cljs.core.transient$.call(null, cljs.core.PersistentVector.EMPTY), coll));\n  };\n  var mapv__3 = function(f, c1, c2) {\n    return cljs.core.into.call(null, cljs.core.PersistentVector.EMPTY, cljs.core.map.call(null, f, c1, c2));\n  };\n  var mapv__4 = function(f, c1, c2, c3) {\n    return cljs.core.into.call(null, cljs.core.PersistentVector.EMPTY, cljs.core.map.call(null, f, c1, c2, c3));\n  };\n  var mapv__5 = function() {\n    var G__22685__delegate = function(f, c1, c2, c3, colls) {\n      return cljs.core.into.call(null, cljs.core.PersistentVector.EMPTY, cljs.core.apply.call(null, cljs.core.map, f, c1, c2, c3, colls));\n    };\n    var G__22685 = function(f, c1, c2, c3, var_args) {\n      var colls = null;\n      if (arguments.length &gt; 4) {\n        colls = cljs.core.array_seq(Array.prototype.slice.call(arguments, 4), 0);\n      }\n      return G__22685__delegate.call(this, f, c1, c2, c3, colls);\n    };\n    G__22685.cljs$lang$maxFixedArity = 4;\n    G__22685.cljs$lang$applyTo = function(arglist__22686) {\n      var f = cljs.core.first(arglist__22686);\n      arglist__22686 = cljs.core.next(arglist__22686);\n      var c1 = cljs.core.first(arglist__22686);\n      arglist__22686 = cljs.core.next(arglist__22686);\n      var c2 = cljs.core.first(arglist__22686);\n      arglist__22686 = cljs.core.next(arglist__22686);\n      var c3 = cljs.core.first(arglist__22686);\n      var colls = cljs.core.rest(arglist__22686);\n      return G__22685__delegate(f, c1, c2, c3, colls);\n    };\n    G__22685.cljs$core$IFn$_invoke$arity$variadic = G__22685__delegate;\n    return G__22685;\n  }();\n  mapv = function(f, c1, c2, c3, var_args) {\n    var colls = var_args;\n    switch(arguments.length) {\n      case 2:\n        return mapv__2.call(this, f, c1);\n      case 3:\n        return mapv__3.call(this, f, c1, c2);\n      case 4:\n        return mapv__4.call(this, f, c1, c2, c3);\n      default:\n        return mapv__5.cljs$core$IFn$_invoke$arity$variadic(f, c1, c2, c3, cljs.core.array_seq(arguments, 4));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  mapv.cljs$lang$maxFixedArity = 4;\n  mapv.cljs$lang$applyTo = mapv__5.cljs$lang$applyTo;\n  mapv.cljs$core$IFn$_invoke$arity$2 = mapv__2;\n  mapv.cljs$core$IFn$_invoke$arity$3 = mapv__3;\n  mapv.cljs$core$IFn$_invoke$arity$4 = mapv__4;\n  mapv.cljs$core$IFn$_invoke$arity$variadic = mapv__5.cljs$core$IFn$_invoke$arity$variadic;\n  return mapv;\n}();\ncljs.core.filterv = function filterv(pred, coll) {\n  return cljs.core.persistent_BANG_.call(null, cljs.core.reduce.call(null, function(v, o) {\n    if (cljs.core.truth_(pred.call(null, o))) {\n      return cljs.core.conj_BANG_.call(null, v, o);\n    } else {\n      return v;\n    }\n  }, cljs.core.transient$.call(null, cljs.core.PersistentVector.EMPTY), coll));\n};\ncljs.core.partition = function() {\n  var partition = null;\n  var partition__2 = function(n, coll) {\n    return partition.call(null, n, n, coll);\n  };\n  var partition__3 = function(n, step, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        var p = cljs.core.take.call(null, n, s);\n        if (n === cljs.core.count.call(null, p)) {\n          return cljs.core.cons.call(null, p, partition.call(null, n, step, cljs.core.drop.call(null, step, s)));\n        } else {\n          return null;\n        }\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  var partition__4 = function(n, step, pad, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        var p = cljs.core.take.call(null, n, s);\n        if (n === cljs.core.count.call(null, p)) {\n          return cljs.core.cons.call(null, p, partition.call(null, n, step, pad, cljs.core.drop.call(null, step, s)));\n        } else {\n          return cljs.core._conj.call(null, cljs.core.List.EMPTY, cljs.core.take.call(null, n, cljs.core.concat.call(null, p, pad)));\n        }\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  partition = function(n, step, pad, coll) {\n    switch(arguments.length) {\n      case 2:\n        return partition__2.call(this, n, step);\n      case 3:\n        return partition__3.call(this, n, step, pad);\n      case 4:\n        return partition__4.call(this, n, step, pad, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  partition.cljs$core$IFn$_invoke$arity$2 = partition__2;\n  partition.cljs$core$IFn$_invoke$arity$3 = partition__3;\n  partition.cljs$core$IFn$_invoke$arity$4 = partition__4;\n  return partition;\n}();\ncljs.core.get_in = function() {\n  var get_in = null;\n  var get_in__2 = function(m, ks) {\n    return get_in.call(null, m, ks, null);\n  };\n  var get_in__3 = function(m, ks, not_found) {\n    var sentinel = cljs.core.lookup_sentinel;\n    var m__$1 = m;\n    var ks__$1 = cljs.core.seq.call(null, ks);\n    while (true) {\n      if (ks__$1) {\n        if (!function() {\n          var G__22688 = m__$1;\n          if (G__22688) {\n            var bit__17795__auto__ = G__22688.cljs$lang$protocol_mask$partition0$ &amp; 256;\n            if (bit__17795__auto__ || G__22688.cljs$core$ILookup$) {\n              return true;\n            } else {\n              if (!G__22688.cljs$lang$protocol_mask$partition0$) {\n                return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ILookup, G__22688);\n              } else {\n                return false;\n              }\n            }\n          } else {\n            return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.ILookup, G__22688);\n          }\n        }()) {\n          return not_found;\n        } else {\n          var m__$2 = cljs.core.get.call(null, m__$1, cljs.core.first.call(null, ks__$1), sentinel);\n          if (sentinel === m__$2) {\n            return not_found;\n          } else {\n            var G__22689 = sentinel;\n            var G__22690 = m__$2;\n            var G__22691 = cljs.core.next.call(null, ks__$1);\n            sentinel = G__22689;\n            m__$1 = G__22690;\n            ks__$1 = G__22691;\n            continue;\n          }\n        }\n      } else {\n        return m__$1;\n      }\n      break;\n    }\n  };\n  get_in = function(m, ks, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return get_in__2.call(this, m, ks);\n      case 3:\n        return get_in__3.call(this, m, ks, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  get_in.cljs$core$IFn$_invoke$arity$2 = get_in__2;\n  get_in.cljs$core$IFn$_invoke$arity$3 = get_in__3;\n  return get_in;\n}();\ncljs.core.assoc_in = function assoc_in(m, p__22692, v) {\n  var vec__22694 = p__22692;\n  var k = cljs.core.nth.call(null, vec__22694, 0, null);\n  var ks = cljs.core.nthnext.call(null, vec__22694, 1);\n  if (ks) {\n    return cljs.core.assoc.call(null, m, k, assoc_in.call(null, cljs.core.get.call(null, m, k), ks, v));\n  } else {\n    return cljs.core.assoc.call(null, m, k, v);\n  }\n};\ncljs.core.update_in = function() {\n  var update_in = null;\n  var update_in__3 = function(m, p__22695, f) {\n    var vec__22705 = p__22695;\n    var k = cljs.core.nth.call(null, vec__22705, 0, null);\n    var ks = cljs.core.nthnext.call(null, vec__22705, 1);\n    if (ks) {\n      return cljs.core.assoc.call(null, m, k, update_in.call(null, cljs.core.get.call(null, m, k), ks, f));\n    } else {\n      return cljs.core.assoc.call(null, m, k, f.call(null, cljs.core.get.call(null, m, k)));\n    }\n  };\n  var update_in__4 = function(m, p__22696, f, a) {\n    var vec__22706 = p__22696;\n    var k = cljs.core.nth.call(null, vec__22706, 0, null);\n    var ks = cljs.core.nthnext.call(null, vec__22706, 1);\n    if (ks) {\n      return cljs.core.assoc.call(null, m, k, update_in.call(null, cljs.core.get.call(null, m, k), ks, f, a));\n    } else {\n      return cljs.core.assoc.call(null, m, k, f.call(null, cljs.core.get.call(null, m, k), a));\n    }\n  };\n  var update_in__5 = function(m, p__22697, f, a, b) {\n    var vec__22707 = p__22697;\n    var k = cljs.core.nth.call(null, vec__22707, 0, null);\n    var ks = cljs.core.nthnext.call(null, vec__22707, 1);\n    if (ks) {\n      return cljs.core.assoc.call(null, m, k, update_in.call(null, cljs.core.get.call(null, m, k), ks, f, a, b));\n    } else {\n      return cljs.core.assoc.call(null, m, k, f.call(null, cljs.core.get.call(null, m, k), a, b));\n    }\n  };\n  var update_in__6 = function(m, p__22698, f, a, b, c) {\n    var vec__22708 = p__22698;\n    var k = cljs.core.nth.call(null, vec__22708, 0, null);\n    var ks = cljs.core.nthnext.call(null, vec__22708, 1);\n    if (ks) {\n      return cljs.core.assoc.call(null, m, k, update_in.call(null, cljs.core.get.call(null, m, k), ks, f, a, b, c));\n    } else {\n      return cljs.core.assoc.call(null, m, k, f.call(null, cljs.core.get.call(null, m, k), a, b, c));\n    }\n  };\n  var update_in__7 = function() {\n    var G__22710__delegate = function(m, p__22699, f, a, b, c, args) {\n      var vec__22709 = p__22699;\n      var k = cljs.core.nth.call(null, vec__22709, 0, null);\n      var ks = cljs.core.nthnext.call(null, vec__22709, 1);\n      if (ks) {\n        return cljs.core.assoc.call(null, m, k, cljs.core.apply.call(null, update_in, cljs.core.get.call(null, m, k), ks, f, a, b, c, args));\n      } else {\n        return cljs.core.assoc.call(null, m, k, cljs.core.apply.call(null, f, cljs.core.get.call(null, m, k), a, b, c, args));\n      }\n    };\n    var G__22710 = function(m, p__22699, f, a, b, c, var_args) {\n      var args = null;\n      if (arguments.length &gt; 6) {\n        args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 6), 0);\n      }\n      return G__22710__delegate.call(this, m, p__22699, f, a, b, c, args);\n    };\n    G__22710.cljs$lang$maxFixedArity = 6;\n    G__22710.cljs$lang$applyTo = function(arglist__22711) {\n      var m = cljs.core.first(arglist__22711);\n      arglist__22711 = cljs.core.next(arglist__22711);\n      var p__22699 = cljs.core.first(arglist__22711);\n      arglist__22711 = cljs.core.next(arglist__22711);\n      var f = cljs.core.first(arglist__22711);\n      arglist__22711 = cljs.core.next(arglist__22711);\n      var a = cljs.core.first(arglist__22711);\n      arglist__22711 = cljs.core.next(arglist__22711);\n      var b = cljs.core.first(arglist__22711);\n      arglist__22711 = cljs.core.next(arglist__22711);\n      var c = cljs.core.first(arglist__22711);\n      var args = cljs.core.rest(arglist__22711);\n      return G__22710__delegate(m, p__22699, f, a, b, c, args);\n    };\n    G__22710.cljs$core$IFn$_invoke$arity$variadic = G__22710__delegate;\n    return G__22710;\n  }();\n  update_in = function(m, p__22699, f, a, b, c, var_args) {\n    var args = var_args;\n    switch(arguments.length) {\n      case 3:\n        return update_in__3.call(this, m, p__22699, f);\n      case 4:\n        return update_in__4.call(this, m, p__22699, f, a);\n      case 5:\n        return update_in__5.call(this, m, p__22699, f, a, b);\n      case 6:\n        return update_in__6.call(this, m, p__22699, f, a, b, c);\n      default:\n        return update_in__7.cljs$core$IFn$_invoke$arity$variadic(m, p__22699, f, a, b, c, cljs.core.array_seq(arguments, 6));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  update_in.cljs$lang$maxFixedArity = 6;\n  update_in.cljs$lang$applyTo = update_in__7.cljs$lang$applyTo;\n  update_in.cljs$core$IFn$_invoke$arity$3 = update_in__3;\n  update_in.cljs$core$IFn$_invoke$arity$4 = update_in__4;\n  update_in.cljs$core$IFn$_invoke$arity$5 = update_in__5;\n  update_in.cljs$core$IFn$_invoke$arity$6 = update_in__6;\n  update_in.cljs$core$IFn$_invoke$arity$variadic = update_in__7.cljs$core$IFn$_invoke$arity$variadic;\n  return update_in;\n}();\ncljs.core.VectorNode = function(edit, arr) {\n  this.edit = edit;\n  this.arr = arr;\n};\ncljs.core.VectorNode.cljs$lang$type = true;\ncljs.core.VectorNode.cljs$lang$ctorStr = &quot;cljs.core/VectorNode&quot;;\ncljs.core.VectorNode.cljs$lang$ctorPrWriter = function(this__17715__auto__, writer__17716__auto__, opts__17717__auto__) {\n  return cljs.core._write.call(null, writer__17716__auto__, &quot;cljs.core/VectorNode&quot;);\n};\ncljs.core.__GT_VectorNode = function __GT_VectorNode(edit, arr) {\n  return new cljs.core.VectorNode(edit, arr);\n};\ncljs.core.pv_fresh_node = function pv_fresh_node(edit) {\n  return new cljs.core.VectorNode(edit, [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null]);\n};\ncljs.core.pv_aget = function pv_aget(node, idx) {\n  return node.arr[idx];\n};\ncljs.core.pv_aset = function pv_aset(node, idx, val) {\n  return node.arr[idx] = val;\n};\ncljs.core.pv_clone_node = function pv_clone_node(node) {\n  return new cljs.core.VectorNode(node.edit, cljs.core.aclone.call(null, node.arr));\n};\ncljs.core.tail_off = function tail_off(pv) {\n  var cnt = pv.cnt;\n  if (cnt &lt; 32) {\n    return 0;\n  } else {\n    return cnt - 1 &gt;&gt;&gt; 5 &lt;&lt; 5;\n  }\n};\ncljs.core.new_path = function new_path(edit, level, node) {\n  var ll = level;\n  var ret = node;\n  while (true) {\n    if (ll === 0) {\n      return ret;\n    } else {\n      var embed = ret;\n      var r = cljs.core.pv_fresh_node.call(null, edit);\n      var _ = cljs.core.pv_aset.call(null, r, 0, embed);\n      var G__22712 = ll - 5;\n      var G__22713 = r;\n      ll = G__22712;\n      ret = G__22713;\n      continue;\n    }\n    break;\n  }\n};\ncljs.core.push_tail = function push_tail(pv, level, parent, tailnode) {\n  var ret = cljs.core.pv_clone_node.call(null, parent);\n  var subidx = pv.cnt - 1 &gt;&gt;&gt; level &amp; 31;\n  if (5 === level) {\n    cljs.core.pv_aset.call(null, ret, subidx, tailnode);\n    return ret;\n  } else {\n    var child = cljs.core.pv_aget.call(null, parent, subidx);\n    if (!(child == null)) {\n      var node_to_insert = push_tail.call(null, pv, level - 5, child, tailnode);\n      cljs.core.pv_aset.call(null, ret, subidx, node_to_insert);\n      return ret;\n    } else {\n      var node_to_insert = cljs.core.new_path.call(null, null, level - 5, tailnode);\n      cljs.core.pv_aset.call(null, ret, subidx, node_to_insert);\n      return ret;\n    }\n  }\n};\ncljs.core.vector_index_out_of_bounds = function vector_index_out_of_bounds(i, cnt) {\n  throw new Error(&quot;No item &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(i) + &quot; in vector of length &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cnt));\n};\ncljs.core.first_array_for_longvec = function first_array_for_longvec(pv) {\n  var node = pv.root;\n  var level = pv.shift;\n  while (true) {\n    if (level &gt; 0) {\n      var G__22714 = cljs.core.pv_aget.call(null, node, 0);\n      var G__22715 = level - 5;\n      node = G__22714;\n      level = G__22715;\n      continue;\n    } else {\n      return node.arr;\n    }\n    break;\n  }\n};\ncljs.core.unchecked_array_for = function unchecked_array_for(pv, i) {\n  if (i &gt;= cljs.core.tail_off.call(null, pv)) {\n    return pv.tail;\n  } else {\n    var node = pv.root;\n    var level = pv.shift;\n    while (true) {\n      if (level &gt; 0) {\n        var G__22716 = cljs.core.pv_aget.call(null, node, i &gt;&gt;&gt; level &amp; 31);\n        var G__22717 = level - 5;\n        node = G__22716;\n        level = G__22717;\n        continue;\n      } else {\n        return node.arr;\n      }\n      break;\n    }\n  }\n};\ncljs.core.array_for = function array_for(pv, i) {\n  if (0 &lt;= i &amp;&amp; i &lt; pv.cnt) {\n    return cljs.core.unchecked_array_for.call(null, pv, i);\n  } else {\n    return cljs.core.vector_index_out_of_bounds.call(null, i, pv.cnt);\n  }\n};\ncljs.core.do_assoc = function do_assoc(pv, level, node, i, val) {\n  var ret = cljs.core.pv_clone_node.call(null, node);\n  if (level === 0) {\n    cljs.core.pv_aset.call(null, ret, i &amp; 31, val);\n    return ret;\n  } else {\n    var subidx = i &gt;&gt;&gt; level &amp; 31;\n    cljs.core.pv_aset.call(null, ret, subidx, do_assoc.call(null, pv, level - 5, cljs.core.pv_aget.call(null, node, subidx), i, val));\n    return ret;\n  }\n};\ncljs.core.pop_tail = function pop_tail(pv, level, node) {\n  var subidx = pv.cnt - 2 &gt;&gt;&gt; level &amp; 31;\n  if (level &gt; 5) {\n    var new_child = pop_tail.call(null, pv, level - 5, cljs.core.pv_aget.call(null, node, subidx));\n    if (new_child == null &amp;&amp; subidx === 0) {\n      return null;\n    } else {\n      var ret = cljs.core.pv_clone_node.call(null, node);\n      cljs.core.pv_aset.call(null, ret, subidx, new_child);\n      return ret;\n    }\n  } else {\n    if (subidx === 0) {\n      return null;\n    } else {\n      var ret = cljs.core.pv_clone_node.call(null, node);\n      cljs.core.pv_aset.call(null, ret, subidx, null);\n      return ret;\n    }\n  }\n};\ncljs.core.PersistentVector = function(meta, cnt, shift, root, tail, __hash) {\n  this.meta = meta;\n  this.cnt = cnt;\n  this.shift = shift;\n  this.root = root;\n  this.tail = tail;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 167668511;\n  this.cljs$lang$protocol_mask$partition1$ = 8196;\n};\ncljs.core.PersistentVector.cljs$lang$type = true;\ncljs.core.PersistentVector.cljs$lang$ctorStr = &quot;cljs.core/PersistentVector&quot;;\ncljs.core.PersistentVector.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentVector&quot;);\n};\ncljs.core.PersistentVector.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentVector.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentVector.prototype.cljs$core$ILookup$_lookup$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._lookup.call(null, coll__$1, k, null);\n};\ncljs.core.PersistentVector.prototype.cljs$core$ILookup$_lookup$arity$3 = function(coll, k, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (typeof k === &quot;number&quot;) {\n    return cljs.core._nth.call(null, coll__$1, k, not_found);\n  } else {\n    return not_found;\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$IKVReduce$_kv_reduce$arity$3 = function(v, f, init) {\n  var self__ = this;\n  var v__$1 = this;\n  var step_init = [0, init];\n  var i = 0;\n  while (true) {\n    if (i &lt; self__.cnt) {\n      var arr = cljs.core.unchecked_array_for.call(null, v__$1, i);\n      var len = arr.length;\n      var init__$1 = function() {\n        var j = 0;\n        var init__$1 = step_init[1];\n        while (true) {\n          if (j &lt; len) {\n            var init__$2 = f.call(null, init__$1, j + i, arr[j]);\n            if (cljs.core.reduced_QMARK_.call(null, init__$2)) {\n              return init__$2;\n            } else {\n              var G__22719 = j + 1;\n              var G__22720 = init__$2;\n              j = G__22719;\n              init__$1 = G__22720;\n              continue;\n            }\n          } else {\n            step_init[0] = len;\n            step_init[1] = init__$1;\n            return init__$1;\n          }\n          break;\n        }\n      }();\n      if (cljs.core.reduced_QMARK_.call(null, init__$1)) {\n        return cljs.core.deref.call(null, init__$1);\n      } else {\n        var G__22721 = i + step_init[0];\n        i = G__22721;\n        continue;\n      }\n    } else {\n      return step_init[1];\n    }\n    break;\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$IIndexed$_nth$arity$2 = function(coll, n) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.array_for.call(null, coll__$1, n)[n &amp; 31];\n};\ncljs.core.PersistentVector.prototype.cljs$core$IIndexed$_nth$arity$3 = function(coll, n, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (0 &lt;= n &amp;&amp; n &lt; self__.cnt) {\n    return cljs.core.unchecked_array_for.call(null, coll__$1, n)[n &amp; 31];\n  } else {\n    return not_found;\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$IVector$_assoc_n$arity$3 = function(coll, n, val) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (0 &lt;= n &amp;&amp; n &lt; self__.cnt) {\n    if (cljs.core.tail_off.call(null, coll__$1) &lt;= n) {\n      var new_tail = cljs.core.aclone.call(null, self__.tail);\n      new_tail[n &amp; 31] = val;\n      return new cljs.core.PersistentVector(self__.meta, self__.cnt, self__.shift, self__.root, new_tail, null);\n    } else {\n      return new cljs.core.PersistentVector(self__.meta, self__.cnt, self__.shift, cljs.core.do_assoc.call(null, coll__$1, self__.shift, self__.root, n, val), self__.tail, null);\n    }\n  } else {\n    if (n === self__.cnt) {\n      return cljs.core._conj.call(null, coll__$1, val);\n    } else {\n      throw new Error(&quot;Index &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(n) + &quot; out of bounds  [0,&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(self__.cnt) + &quot;]&quot;);\n    }\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.PersistentVector.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.PersistentVector(self__.meta, self__.cnt, self__.shift, self__.root, self__.tail, self__.__hash);\n};\ncljs.core.PersistentVector.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.cnt;\n};\ncljs.core.PersistentVector.prototype.cljs$core$IMapEntry$_key$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._nth.call(null, coll__$1, 0);\n};\ncljs.core.PersistentVector.prototype.cljs$core$IMapEntry$_val$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._nth.call(null, coll__$1, 1);\n};\ncljs.core.PersistentVector.prototype.cljs$core$IStack$_peek$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt &gt; 0) {\n    return cljs.core._nth.call(null, coll__$1, self__.cnt - 1);\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$IStack$_pop$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt === 0) {\n    throw new Error(&quot;Can&apos;t pop empty vector&quot;);\n  } else {\n    if (1 === self__.cnt) {\n      return cljs.core._with_meta.call(null, cljs.core.PersistentVector.EMPTY, self__.meta);\n    } else {\n      if (1 &lt; self__.cnt - cljs.core.tail_off.call(null, coll__$1)) {\n        return new cljs.core.PersistentVector(self__.meta, self__.cnt - 1, self__.shift, self__.root, self__.tail.slice(0, -1), null);\n      } else {\n        var new_tail = cljs.core.unchecked_array_for.call(null, coll__$1, self__.cnt - 2);\n        var nr = cljs.core.pop_tail.call(null, coll__$1, self__.shift, self__.root);\n        var new_root = nr == null ? cljs.core.PersistentVector.EMPTY_NODE : nr;\n        var cnt_1 = self__.cnt - 1;\n        if (5 &lt; self__.shift &amp;&amp; cljs.core.pv_aget.call(null, new_root, 1) == null) {\n          return new cljs.core.PersistentVector(self__.meta, cnt_1, self__.shift - 5, cljs.core.pv_aget.call(null, new_root, 0), new_tail, null);\n        } else {\n          return new cljs.core.PersistentVector(self__.meta, cnt_1, self__.shift, new_root, new_tail, null);\n        }\n      }\n    }\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$IReversible$_rseq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt &gt; 0) {\n    return new cljs.core.RSeq(coll__$1, self__.cnt - 1, null);\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.PersistentVector.prototype.cljs$core$IEditableCollection$_as_transient$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.TransientVector(self__.cnt, self__.shift, cljs.core.tv_editable_root.call(null, self__.root), cljs.core.tv_editable_tail.call(null, self__.tail));\n};\ncljs.core.PersistentVector.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.PersistentVector.EMPTY, self__.meta);\n};\ncljs.core.PersistentVector.prototype.cljs$core$IReduce$_reduce$arity$2 = function(v, f) {\n  var self__ = this;\n  var v__$1 = this;\n  return cljs.core.ci_reduce.call(null, v__$1, f);\n};\ncljs.core.PersistentVector.prototype.cljs$core$IReduce$_reduce$arity$3 = function(v, f, start) {\n  var self__ = this;\n  var v__$1 = this;\n  return cljs.core.ci_reduce.call(null, v__$1, f, start);\n};\ncljs.core.PersistentVector.prototype.cljs$core$IAssociative$_assoc$arity$3 = function(coll, k, v) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (typeof k === &quot;number&quot;) {\n    return cljs.core._assoc_n.call(null, coll__$1, k, v);\n  } else {\n    throw new Error(&quot;Vector&apos;s key for assoc must be a number.&quot;);\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt === 0) {\n    return null;\n  } else {\n    if (self__.cnt &lt;= 32) {\n      return new cljs.core.IndexedSeq(self__.tail, 0);\n    } else {\n      return cljs.core.chunked_seq.call(null, coll__$1, cljs.core.first_array_for_longvec.call(null, coll__$1), 0, 0);\n    }\n  }\n};\ncljs.core.PersistentVector.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentVector(meta__$1, self__.cnt, self__.shift, self__.root, self__.tail, self__.__hash);\n};\ncljs.core.PersistentVector.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt - cljs.core.tail_off.call(null, coll__$1) &lt; 32) {\n    var len = self__.tail.length;\n    var new_tail = new Array(len + 1);\n    var n__18001__auto___22722 = len;\n    var i_22723 = 0;\n    while (true) {\n      if (i_22723 &lt; n__18001__auto___22722) {\n        new_tail[i_22723] = self__.tail[i_22723];\n        var G__22724 = i_22723 + 1;\n        i_22723 = G__22724;\n        continue;\n      } else {\n      }\n      break;\n    }\n    new_tail[len] = o;\n    return new cljs.core.PersistentVector(self__.meta, self__.cnt + 1, self__.shift, self__.root, new_tail, null);\n  } else {\n    var root_overflow_QMARK_ = self__.cnt &gt;&gt;&gt; 5 &gt; 1 &lt;&lt; self__.shift;\n    var new_shift = root_overflow_QMARK_ ? self__.shift + 5 : self__.shift;\n    var new_root = root_overflow_QMARK_ ? function() {\n      var n_r = cljs.core.pv_fresh_node.call(null, null);\n      cljs.core.pv_aset.call(null, n_r, 0, self__.root);\n      cljs.core.pv_aset.call(null, n_r, 1, cljs.core.new_path.call(null, null, self__.shift, new cljs.core.VectorNode(null, self__.tail)));\n      return n_r;\n    }() : cljs.core.push_tail.call(null, coll__$1, self__.shift, self__.root, new cljs.core.VectorNode(null, self__.tail));\n    return new cljs.core.PersistentVector(self__.meta, self__.cnt + 1, new_shift, new_root, [o], null);\n  }\n};\ncljs.core.PersistentVector.prototype.call = function() {\n  var G__22725 = null;\n  var G__22725__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$IIndexed$_nth$arity$2(null, k);\n  };\n  var G__22725__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$IIndexed$_nth$arity$3(null, k, not_found);\n  };\n  G__22725 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22725__2.call(this, self__, k);\n      case 3:\n        return G__22725__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22725.cljs$core$IFn$_invoke$arity$2 = G__22725__2;\n  G__22725.cljs$core$IFn$_invoke$arity$3 = G__22725__3;\n  return G__22725;\n}();\ncljs.core.PersistentVector.prototype.apply = function(self__, args22718) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22718)));\n};\ncljs.core.PersistentVector.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$IIndexed$_nth$arity$2(null, k);\n};\ncljs.core.PersistentVector.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$IIndexed$_nth$arity$3(null, k, not_found);\n};\ncljs.core.__GT_PersistentVector = function __GT_PersistentVector(meta, cnt, shift, root, tail, __hash) {\n  return new cljs.core.PersistentVector(meta, cnt, shift, root, tail, __hash);\n};\ncljs.core.PersistentVector.EMPTY_NODE = new cljs.core.VectorNode(null, [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null]);\ncljs.core.PersistentVector.EMPTY = new cljs.core.PersistentVector(null, 0, 5, cljs.core.PersistentVector.EMPTY_NODE, [], 0);\ncljs.core.PersistentVector.fromArray = function(xs, no_clone) {\n  var l = xs.length;\n  var xs__$1 = no_clone ? xs : cljs.core.aclone.call(null, xs);\n  if (l &lt; 32) {\n    return new cljs.core.PersistentVector(null, l, 5, cljs.core.PersistentVector.EMPTY_NODE, xs__$1, null);\n  } else {\n    var node = xs__$1.slice(0, 32);\n    var v = new cljs.core.PersistentVector(null, 32, 5, cljs.core.PersistentVector.EMPTY_NODE, node, null);\n    var i = 32;\n    var out = cljs.core._as_transient.call(null, v);\n    while (true) {\n      if (i &lt; l) {\n        var G__22726 = i + 1;\n        var G__22727 = cljs.core.conj_BANG_.call(null, out, xs__$1[i]);\n        i = G__22726;\n        out = G__22727;\n        continue;\n      } else {\n        return cljs.core.persistent_BANG_.call(null, out);\n      }\n      break;\n    }\n  }\n};\ncljs.core.vec = function vec(coll) {\n  return cljs.core._persistent_BANG_.call(null, cljs.core.reduce.call(null, cljs.core._conj_BANG_, cljs.core._as_transient.call(null, cljs.core.PersistentVector.EMPTY), coll));\n};\ncljs.core.vector = function() {\n  var vector__delegate = function(args) {\n    if (args instanceof cljs.core.IndexedSeq &amp;&amp; args.i === 0) {\n      return cljs.core.PersistentVector.fromArray(args.arr, true);\n    } else {\n      return cljs.core.vec.call(null, args);\n    }\n  };\n  var vector = function(var_args) {\n    var args = null;\n    if (arguments.length &gt; 0) {\n      args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return vector__delegate.call(this, args);\n  };\n  vector.cljs$lang$maxFixedArity = 0;\n  vector.cljs$lang$applyTo = function(arglist__22728) {\n    var args = cljs.core.seq(arglist__22728);\n    return vector__delegate(args);\n  };\n  vector.cljs$core$IFn$_invoke$arity$variadic = vector__delegate;\n  return vector;\n}();\ncljs.core.ChunkedSeq = function(vec, node, i, off, meta, __hash) {\n  this.vec = vec;\n  this.node = node;\n  this.i = i;\n  this.off = off;\n  this.meta = meta;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 32243948;\n  this.cljs$lang$protocol_mask$partition1$ = 1536;\n};\ncljs.core.ChunkedSeq.cljs$lang$type = true;\ncljs.core.ChunkedSeq.cljs$lang$ctorStr = &quot;cljs.core/ChunkedSeq&quot;;\ncljs.core.ChunkedSeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ChunkedSeq&quot;);\n};\ncljs.core.ChunkedSeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.ChunkedSeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.off + 1 &lt; self__.node.length) {\n    var s = cljs.core.chunked_seq.call(null, self__.vec, self__.node, self__.i, self__.off + 1);\n    if (s == null) {\n      return null;\n    } else {\n      return s;\n    }\n  } else {\n    return cljs.core._chunked_next.call(null, coll__$1);\n  }\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.PersistentVector.EMPTY, self__.meta);\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.ci_reduce.call(null, cljs.core.subvec.call(null, self__.vec, self__.i + self__.off, cljs.core.count.call(null, self__.vec)), f);\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.ci_reduce.call(null, cljs.core.subvec.call(null, self__.vec, self__.i + self__.off, cljs.core.count.call(null, self__.vec)), f, start);\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.node[self__.off];\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.off + 1 &lt; self__.node.length) {\n    var s = cljs.core.chunked_seq.call(null, self__.vec, self__.node, self__.i, self__.off + 1);\n    if (s == null) {\n      return cljs.core.List.EMPTY;\n    } else {\n      return s;\n    }\n  } else {\n    return cljs.core._chunked_rest.call(null, coll__$1);\n  }\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IChunkedSeq$_chunked_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.array_chunk.call(null, self__.node, self__.off);\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IChunkedSeq$_chunked_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var end = self__.i + self__.node.length;\n  if (end &lt; cljs.core._count.call(null, self__.vec)) {\n    return cljs.core.chunked_seq.call(null, self__.vec, cljs.core.unchecked_array_for.call(null, self__.vec, end), end, 0);\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, m) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.chunked_seq.call(null, self__.vec, self__.node, self__.i, self__.off, m);\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IWithMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IChunkedNext$_chunked_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var end = self__.i + self__.node.length;\n  if (end &lt; cljs.core._count.call(null, self__.vec)) {\n    return cljs.core.chunked_seq.call(null, self__.vec, cljs.core.unchecked_array_for.call(null, self__.vec, end), end, 0);\n  } else {\n    return null;\n  }\n};\ncljs.core.__GT_ChunkedSeq = function __GT_ChunkedSeq(vec, node, i, off, meta, __hash) {\n  return new cljs.core.ChunkedSeq(vec, node, i, off, meta, __hash);\n};\ncljs.core.chunked_seq = function() {\n  var chunked_seq = null;\n  var chunked_seq__3 = function(vec, i, off) {\n    return new cljs.core.ChunkedSeq(vec, cljs.core.array_for.call(null, vec, i), i, off, null, null);\n  };\n  var chunked_seq__4 = function(vec, node, i, off) {\n    return new cljs.core.ChunkedSeq(vec, node, i, off, null, null);\n  };\n  var chunked_seq__5 = function(vec, node, i, off, meta) {\n    return new cljs.core.ChunkedSeq(vec, node, i, off, meta, null);\n  };\n  chunked_seq = function(vec, node, i, off, meta) {\n    switch(arguments.length) {\n      case 3:\n        return chunked_seq__3.call(this, vec, node, i);\n      case 4:\n        return chunked_seq__4.call(this, vec, node, i, off);\n      case 5:\n        return chunked_seq__5.call(this, vec, node, i, off, meta);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  chunked_seq.cljs$core$IFn$_invoke$arity$3 = chunked_seq__3;\n  chunked_seq.cljs$core$IFn$_invoke$arity$4 = chunked_seq__4;\n  chunked_seq.cljs$core$IFn$_invoke$arity$5 = chunked_seq__5;\n  return chunked_seq;\n}();\ncljs.core.Subvec = function(meta, v, start, end, __hash) {\n  this.meta = meta;\n  this.v = v;\n  this.start = start;\n  this.end = end;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 166617887;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.Subvec.cljs$lang$type = true;\ncljs.core.Subvec.cljs$lang$ctorStr = &quot;cljs.core/Subvec&quot;;\ncljs.core.Subvec.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Subvec&quot;);\n};\ncljs.core.Subvec.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.Subvec.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.Subvec.prototype.cljs$core$ILookup$_lookup$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._lookup.call(null, coll__$1, k, null);\n};\ncljs.core.Subvec.prototype.cljs$core$ILookup$_lookup$arity$3 = function(coll, k, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (typeof k === &quot;number&quot;) {\n    return cljs.core._nth.call(null, coll__$1, k, not_found);\n  } else {\n    return not_found;\n  }\n};\ncljs.core.Subvec.prototype.cljs$core$IIndexed$_nth$arity$2 = function(coll, n) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (n &lt; 0 || self__.end &lt;= self__.start + n) {\n    return cljs.core.vector_index_out_of_bounds.call(null, n, self__.end - self__.start);\n  } else {\n    return cljs.core._nth.call(null, self__.v, self__.start + n);\n  }\n};\ncljs.core.Subvec.prototype.cljs$core$IIndexed$_nth$arity$3 = function(coll, n, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (n &lt; 0 || self__.end &lt;= self__.start + n) {\n    return not_found;\n  } else {\n    return cljs.core._nth.call(null, self__.v, self__.start + n, not_found);\n  }\n};\ncljs.core.Subvec.prototype.cljs$core$IVector$_assoc_n$arity$3 = function(coll, n, val) {\n  var self__ = this;\n  var coll__$1 = this;\n  var v_pos = self__.start + n;\n  return cljs.core.build_subvec.call(null, self__.meta, cljs.core.assoc.call(null, self__.v, v_pos, val), self__.start, function() {\n    var x__17452__auto__ = self__.end;\n    var y__17453__auto__ = v_pos + 1;\n    return x__17452__auto__ &gt; y__17453__auto__ ? x__17452__auto__ : y__17453__auto__;\n  }(), null);\n};\ncljs.core.Subvec.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.Subvec.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.Subvec(self__.meta, self__.v, self__.start, self__.end, self__.__hash);\n};\ncljs.core.Subvec.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.end - self__.start;\n};\ncljs.core.Subvec.prototype.cljs$core$IStack$_peek$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._nth.call(null, self__.v, self__.end - 1);\n};\ncljs.core.Subvec.prototype.cljs$core$IStack$_pop$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.start === self__.end) {\n    throw new Error(&quot;Can&apos;t pop empty vector&quot;);\n  } else {\n    return cljs.core.build_subvec.call(null, self__.meta, self__.v, self__.start, self__.end - 1, null);\n  }\n};\ncljs.core.Subvec.prototype.cljs$core$IReversible$_rseq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (!(self__.start === self__.end)) {\n    return new cljs.core.RSeq(coll__$1, self__.end - self__.start - 1, null);\n  } else {\n    return null;\n  }\n};\ncljs.core.Subvec.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.Subvec.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.Subvec.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.PersistentVector.EMPTY, self__.meta);\n};\ncljs.core.Subvec.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.ci_reduce.call(null, coll__$1, f);\n};\ncljs.core.Subvec.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.ci_reduce.call(null, coll__$1, f, start__$1);\n};\ncljs.core.Subvec.prototype.cljs$core$IAssociative$_assoc$arity$3 = function(coll, key, val) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (typeof key === &quot;number&quot;) {\n    return cljs.core._assoc_n.call(null, coll__$1, key, val);\n  } else {\n    throw new Error(&quot;Subvec&apos;s key for assoc must be a number.&quot;);\n  }\n};\ncljs.core.Subvec.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var subvec_seq = function(coll__$1) {\n    return function subvec_seq(i) {\n      if (i === self__.end) {\n        return null;\n      } else {\n        return cljs.core.cons.call(null, cljs.core._nth.call(null, self__.v, i), new cljs.core.LazySeq(null, function(coll__$1) {\n          return function() {\n            return subvec_seq.call(null, i + 1);\n          };\n        }(coll__$1), null, null));\n      }\n    };\n  }(coll__$1);\n  return subvec_seq.call(null, self__.start);\n};\ncljs.core.Subvec.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.build_subvec.call(null, meta__$1, self__.v, self__.start, self__.end, self__.__hash);\n};\ncljs.core.Subvec.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.build_subvec.call(null, self__.meta, cljs.core._assoc_n.call(null, self__.v, self__.end, o), self__.start, self__.end + 1, null);\n};\ncljs.core.Subvec.prototype.call = function() {\n  var G__22730 = null;\n  var G__22730__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$IIndexed$_nth$arity$2(null, k);\n  };\n  var G__22730__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$IIndexed$_nth$arity$3(null, k, not_found);\n  };\n  G__22730 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22730__2.call(this, self__, k);\n      case 3:\n        return G__22730__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22730.cljs$core$IFn$_invoke$arity$2 = G__22730__2;\n  G__22730.cljs$core$IFn$_invoke$arity$3 = G__22730__3;\n  return G__22730;\n}();\ncljs.core.Subvec.prototype.apply = function(self__, args22729) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22729)));\n};\ncljs.core.Subvec.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$IIndexed$_nth$arity$2(null, k);\n};\ncljs.core.Subvec.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$IIndexed$_nth$arity$3(null, k, not_found);\n};\ncljs.core.__GT_Subvec = function __GT_Subvec(meta, v, start, end, __hash) {\n  return new cljs.core.Subvec(meta, v, start, end, __hash);\n};\ncljs.core.build_subvec = function build_subvec(meta, v, start, end, __hash) {\n  while (true) {\n    if (v instanceof cljs.core.Subvec) {\n      var G__22731 = meta;\n      var G__22732 = v.v;\n      var G__22733 = v.start + start;\n      var G__22734 = v.start + end;\n      var G__22735 = __hash;\n      meta = G__22731;\n      v = G__22732;\n      start = G__22733;\n      end = G__22734;\n      __hash = G__22735;\n      continue;\n    } else {\n      var c = cljs.core.count.call(null, v);\n      if (start &lt; 0 || end &lt; 0 || start &gt; c || end &gt; c) {\n        throw new Error(&quot;Index out of bounds&quot;);\n      } else {\n      }\n      return new cljs.core.Subvec(meta, v, start, end, __hash);\n    }\n    break;\n  }\n};\ncljs.core.subvec = function() {\n  var subvec = null;\n  var subvec__2 = function(v, start) {\n    return subvec.call(null, v, start, cljs.core.count.call(null, v));\n  };\n  var subvec__3 = function(v, start, end) {\n    return cljs.core.build_subvec.call(null, null, v, start, end, null);\n  };\n  subvec = function(v, start, end) {\n    switch(arguments.length) {\n      case 2:\n        return subvec__2.call(this, v, start);\n      case 3:\n        return subvec__3.call(this, v, start, end);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  subvec.cljs$core$IFn$_invoke$arity$2 = subvec__2;\n  subvec.cljs$core$IFn$_invoke$arity$3 = subvec__3;\n  return subvec;\n}();\ncljs.core.tv_ensure_editable = function tv_ensure_editable(edit, node) {\n  if (edit === node.edit) {\n    return node;\n  } else {\n    return new cljs.core.VectorNode(edit, cljs.core.aclone.call(null, node.arr));\n  }\n};\ncljs.core.tv_editable_root = function tv_editable_root(node) {\n  return new cljs.core.VectorNode(function() {\n    var obj22739 = {};\n    return obj22739;\n  }(), cljs.core.aclone.call(null, node.arr));\n};\ncljs.core.tv_editable_tail = function tv_editable_tail(tl) {\n  var ret = [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null];\n  cljs.core.array_copy.call(null, tl, 0, ret, 0, tl.length);\n  return ret;\n};\ncljs.core.tv_push_tail = function tv_push_tail(tv, level, parent, tail_node) {\n  var ret = cljs.core.tv_ensure_editable.call(null, tv.root.edit, parent);\n  var subidx = tv.cnt - 1 &gt;&gt;&gt; level &amp; 31;\n  cljs.core.pv_aset.call(null, ret, subidx, level === 5 ? tail_node : function() {\n    var child = cljs.core.pv_aget.call(null, ret, subidx);\n    if (!(child == null)) {\n      return tv_push_tail.call(null, tv, level - 5, child, tail_node);\n    } else {\n      return cljs.core.new_path.call(null, tv.root.edit, level - 5, tail_node);\n    }\n  }());\n  return ret;\n};\ncljs.core.tv_pop_tail = function tv_pop_tail(tv, level, node) {\n  var node__$1 = cljs.core.tv_ensure_editable.call(null, tv.root.edit, node);\n  var subidx = tv.cnt - 2 &gt;&gt;&gt; level &amp; 31;\n  if (level &gt; 5) {\n    var new_child = tv_pop_tail.call(null, tv, level - 5, cljs.core.pv_aget.call(null, node__$1, subidx));\n    if (new_child == null &amp;&amp; subidx === 0) {\n      return null;\n    } else {\n      cljs.core.pv_aset.call(null, node__$1, subidx, new_child);\n      return node__$1;\n    }\n  } else {\n    if (subidx === 0) {\n      return null;\n    } else {\n      cljs.core.pv_aset.call(null, node__$1, subidx, null);\n      return node__$1;\n    }\n  }\n};\ncljs.core.unchecked_editable_array_for = function unchecked_editable_array_for(tv, i) {\n  if (i &gt;= cljs.core.tail_off.call(null, tv)) {\n    return tv.tail;\n  } else {\n    var root = tv.root;\n    var node = root;\n    var level = tv.shift;\n    while (true) {\n      if (level &gt; 0) {\n        var G__22740 = cljs.core.tv_ensure_editable.call(null, root.edit, cljs.core.pv_aget.call(null, node, i &gt;&gt;&gt; level &amp; 31));\n        var G__22741 = level - 5;\n        node = G__22740;\n        level = G__22741;\n        continue;\n      } else {\n        return node.arr;\n      }\n      break;\n    }\n  }\n};\ncljs.core.TransientVector = function(cnt, shift, root, tail) {\n  this.cnt = cnt;\n  this.shift = shift;\n  this.root = root;\n  this.tail = tail;\n  this.cljs$lang$protocol_mask$partition0$ = 275;\n  this.cljs$lang$protocol_mask$partition1$ = 88;\n};\ncljs.core.TransientVector.cljs$lang$type = true;\ncljs.core.TransientVector.cljs$lang$ctorStr = &quot;cljs.core/TransientVector&quot;;\ncljs.core.TransientVector.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/TransientVector&quot;);\n};\ncljs.core.TransientVector.prototype.call = function() {\n  var G__22743 = null;\n  var G__22743__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n  };\n  var G__22743__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n  };\n  G__22743 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22743__2.call(this, self__, k);\n      case 3:\n        return G__22743__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22743.cljs$core$IFn$_invoke$arity$2 = G__22743__2;\n  G__22743.cljs$core$IFn$_invoke$arity$3 = G__22743__3;\n  return G__22743;\n}();\ncljs.core.TransientVector.prototype.apply = function(self__, args22742) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22742)));\n};\ncljs.core.TransientVector.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.TransientVector.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n};\ncljs.core.TransientVector.prototype.cljs$core$ILookup$_lookup$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._lookup.call(null, coll__$1, k, null);\n};\ncljs.core.TransientVector.prototype.cljs$core$ILookup$_lookup$arity$3 = function(coll, k, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (typeof k === &quot;number&quot;) {\n    return cljs.core._nth.call(null, coll__$1, k, not_found);\n  } else {\n    return not_found;\n  }\n};\ncljs.core.TransientVector.prototype.cljs$core$IIndexed$_nth$arity$2 = function(coll, n) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.root.edit) {\n    return cljs.core.array_for.call(null, coll__$1, n)[n &amp; 31];\n  } else {\n    throw new Error(&quot;nth after persistent!&quot;);\n  }\n};\ncljs.core.TransientVector.prototype.cljs$core$IIndexed$_nth$arity$3 = function(coll, n, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (0 &lt;= n &amp;&amp; n &lt; self__.cnt) {\n    return cljs.core._nth.call(null, coll__$1, n);\n  } else {\n    return not_found;\n  }\n};\ncljs.core.TransientVector.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.root.edit) {\n    return self__.cnt;\n  } else {\n    throw new Error(&quot;count after persistent!&quot;);\n  }\n};\ncljs.core.TransientVector.prototype.cljs$core$ITransientVector$_assoc_n_BANG_$arity$3 = function(tcoll, n, val) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (self__.root.edit) {\n    if (0 &lt;= n &amp;&amp; n &lt; self__.cnt) {\n      if (cljs.core.tail_off.call(null, tcoll__$1) &lt;= n) {\n        self__.tail[n &amp; 31] = val;\n        return tcoll__$1;\n      } else {\n        var new_root = function(tcoll__$1) {\n          return function go(level, node) {\n            var node__$1 = cljs.core.tv_ensure_editable.call(null, self__.root.edit, node);\n            if (level === 0) {\n              cljs.core.pv_aset.call(null, node__$1, n &amp; 31, val);\n              return node__$1;\n            } else {\n              var subidx = n &gt;&gt;&gt; level &amp; 31;\n              cljs.core.pv_aset.call(null, node__$1, subidx, go.call(null, level - 5, cljs.core.pv_aget.call(null, node__$1, subidx)));\n              return node__$1;\n            }\n          };\n        }(tcoll__$1).call(null, self__.shift, self__.root);\n        self__.root = new_root;\n        return tcoll__$1;\n      }\n    } else {\n      if (n === self__.cnt) {\n        return cljs.core._conj_BANG_.call(null, tcoll__$1, val);\n      } else {\n        throw new Error(&quot;Index &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(n) + &quot; out of bounds for TransientVector of length&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(self__.cnt));\n      }\n    }\n  } else {\n    throw new Error(&quot;assoc! after persistent!&quot;);\n  }\n};\ncljs.core.TransientVector.prototype.cljs$core$ITransientVector$_pop_BANG_$arity$1 = function(tcoll) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (self__.root.edit) {\n    if (self__.cnt === 0) {\n      throw new Error(&quot;Can&apos;t pop empty vector&quot;);\n    } else {\n      if (1 === self__.cnt) {\n        self__.cnt = 0;\n        return tcoll__$1;\n      } else {\n        if ((self__.cnt - 1 &amp; 31) &gt; 0) {\n          self__.cnt = self__.cnt - 1;\n          return tcoll__$1;\n        } else {\n          var new_tail = cljs.core.unchecked_editable_array_for.call(null, tcoll__$1, self__.cnt - 2);\n          var new_root = function() {\n            var nr = cljs.core.tv_pop_tail.call(null, tcoll__$1, self__.shift, self__.root);\n            if (!(nr == null)) {\n              return nr;\n            } else {\n              return new cljs.core.VectorNode(self__.root.edit, [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null]);\n            }\n          }();\n          if (5 &lt; self__.shift &amp;&amp; cljs.core.pv_aget.call(null, new_root, 1) == null) {\n            var new_root__$1 = cljs.core.tv_ensure_editable.call(null, self__.root.edit, cljs.core.pv_aget.call(null, new_root, 0));\n            self__.root = new_root__$1;\n            self__.shift = self__.shift - 5;\n            self__.cnt = self__.cnt - 1;\n            self__.tail = new_tail;\n            return tcoll__$1;\n          } else {\n            self__.root = new_root;\n            self__.cnt = self__.cnt - 1;\n            self__.tail = new_tail;\n            return tcoll__$1;\n          }\n        }\n      }\n    }\n  } else {\n    throw new Error(&quot;pop! after persistent!&quot;);\n  }\n};\ncljs.core.TransientVector.prototype.cljs$core$ITransientAssociative$_assoc_BANG_$arity$3 = function(tcoll, key, val) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (typeof key === &quot;number&quot;) {\n    return cljs.core._assoc_n_BANG_.call(null, tcoll__$1, key, val);\n  } else {\n    throw new Error(&quot;TransientVector&apos;s key for assoc! must be a number.&quot;);\n  }\n};\ncljs.core.TransientVector.prototype.cljs$core$ITransientCollection$_conj_BANG_$arity$2 = function(tcoll, o) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (self__.root.edit) {\n    if (self__.cnt - cljs.core.tail_off.call(null, tcoll__$1) &lt; 32) {\n      self__.tail[self__.cnt &amp; 31] = o;\n      self__.cnt = self__.cnt + 1;\n      return tcoll__$1;\n    } else {\n      var tail_node = new cljs.core.VectorNode(self__.root.edit, self__.tail);\n      var new_tail = [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null];\n      new_tail[0] = o;\n      self__.tail = new_tail;\n      if (self__.cnt &gt;&gt;&gt; 5 &gt; 1 &lt;&lt; self__.shift) {\n        var new_root_array = [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null];\n        var new_shift = self__.shift + 5;\n        new_root_array[0] = self__.root;\n        new_root_array[1] = cljs.core.new_path.call(null, self__.root.edit, self__.shift, tail_node);\n        self__.root = new cljs.core.VectorNode(self__.root.edit, new_root_array);\n        self__.shift = new_shift;\n        self__.cnt = self__.cnt + 1;\n        return tcoll__$1;\n      } else {\n        var new_root = cljs.core.tv_push_tail.call(null, tcoll__$1, self__.shift, self__.root, tail_node);\n        self__.root = new_root;\n        self__.cnt = self__.cnt + 1;\n        return tcoll__$1;\n      }\n    }\n  } else {\n    throw new Error(&quot;conj! after persistent!&quot;);\n  }\n};\ncljs.core.TransientVector.prototype.cljs$core$ITransientCollection$_persistent_BANG_$arity$1 = function(tcoll) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (self__.root.edit) {\n    self__.root.edit = null;\n    var len = self__.cnt - cljs.core.tail_off.call(null, tcoll__$1);\n    var trimmed_tail = new Array(len);\n    cljs.core.array_copy.call(null, self__.tail, 0, trimmed_tail, 0, len);\n    return new cljs.core.PersistentVector(null, self__.cnt, self__.shift, self__.root, trimmed_tail, null);\n  } else {\n    throw new Error(&quot;persistent! called twice&quot;);\n  }\n};\ncljs.core.__GT_TransientVector = function __GT_TransientVector(cnt, shift, root, tail) {\n  return new cljs.core.TransientVector(cnt, shift, root, tail);\n};\ncljs.core.PersistentQueueSeq = function(meta, front, rear, __hash) {\n  this.meta = meta;\n  this.front = front;\n  this.rear = rear;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 31850572;\n};\ncljs.core.PersistentQueueSeq.cljs$lang$type = true;\ncljs.core.PersistentQueueSeq.cljs$lang$ctorStr = &quot;cljs.core/PersistentQueueSeq&quot;;\ncljs.core.PersistentQueueSeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentQueueSeq&quot;);\n};\ncljs.core.PersistentQueueSeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentQueueSeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__.meta);\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.first.call(null, self__.front);\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var temp__4124__auto__ = cljs.core.next.call(null, self__.front);\n  if (temp__4124__auto__) {\n    var f1 = temp__4124__auto__;\n    return new cljs.core.PersistentQueueSeq(self__.meta, f1, self__.rear, null);\n  } else {\n    if (self__.rear == null) {\n      return cljs.core._empty.call(null, coll__$1);\n    } else {\n      return new cljs.core.PersistentQueueSeq(self__.meta, self__.rear, null, null);\n    }\n  }\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentQueueSeq(meta__$1, self__.front, self__.rear, self__.__hash);\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_PersistentQueueSeq = function __GT_PersistentQueueSeq(meta, front, rear, __hash) {\n  return new cljs.core.PersistentQueueSeq(meta, front, rear, __hash);\n};\ncljs.core.PersistentQueue = function(meta, count, front, rear, __hash) {\n  this.meta = meta;\n  this.count = count;\n  this.front = front;\n  this.rear = rear;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 31858766;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.PersistentQueue.cljs$lang$type = true;\ncljs.core.PersistentQueue.cljs$lang$ctorStr = &quot;cljs.core/PersistentQueue&quot;;\ncljs.core.PersistentQueue.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentQueue&quot;);\n};\ncljs.core.PersistentQueue.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentQueue.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentQueue.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.PersistentQueue.prototype.cljs$core$ICloneable$_clone$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentQueue(self__.meta, self__.count, self__.front, self__.rear, self__.__hash);\n};\ncljs.core.PersistentQueue.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.count;\n};\ncljs.core.PersistentQueue.prototype.cljs$core$IStack$_peek$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.first.call(null, self__.front);\n};\ncljs.core.PersistentQueue.prototype.cljs$core$IStack$_pop$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core.truth_(self__.front)) {\n    var temp__4124__auto__ = cljs.core.next.call(null, self__.front);\n    if (temp__4124__auto__) {\n      var f1 = temp__4124__auto__;\n      return new cljs.core.PersistentQueue(self__.meta, self__.count - 1, f1, self__.rear, null);\n    } else {\n      return new cljs.core.PersistentQueue(self__.meta, self__.count - 1, cljs.core.seq.call(null, self__.rear), cljs.core.PersistentVector.EMPTY, null);\n    }\n  } else {\n    return coll__$1;\n  }\n};\ncljs.core.PersistentQueue.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.PersistentQueue.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.PersistentQueue.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.PersistentQueue.EMPTY;\n};\ncljs.core.PersistentQueue.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.first.call(null, self__.front);\n};\ncljs.core.PersistentQueue.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.rest.call(null, cljs.core.seq.call(null, coll__$1));\n};\ncljs.core.PersistentQueue.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var rear__$1 = cljs.core.seq.call(null, self__.rear);\n  if (cljs.core.truth_(function() {\n    var or__17145__auto__ = self__.front;\n    if (cljs.core.truth_(or__17145__auto__)) {\n      return or__17145__auto__;\n    } else {\n      return rear__$1;\n    }\n  }())) {\n    return new cljs.core.PersistentQueueSeq(null, self__.front, cljs.core.seq.call(null, rear__$1), null);\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentQueue.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentQueue(meta__$1, self__.count, self__.front, self__.rear, self__.__hash);\n};\ncljs.core.PersistentQueue.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core.truth_(self__.front)) {\n    return new cljs.core.PersistentQueue(self__.meta, self__.count + 1, self__.front, cljs.core.conj.call(null, function() {\n      var or__17145__auto__ = self__.rear;\n      if (cljs.core.truth_(or__17145__auto__)) {\n        return or__17145__auto__;\n      } else {\n        return cljs.core.PersistentVector.EMPTY;\n      }\n    }(), o), null);\n  } else {\n    return new cljs.core.PersistentQueue(self__.meta, self__.count + 1, cljs.core.conj.call(null, self__.front, o), cljs.core.PersistentVector.EMPTY, null);\n  }\n};\ncljs.core.__GT_PersistentQueue = function __GT_PersistentQueue(meta, count, front, rear, __hash) {\n  return new cljs.core.PersistentQueue(meta, count, front, rear, __hash);\n};\ncljs.core.PersistentQueue.EMPTY = new cljs.core.PersistentQueue(null, 0, null, cljs.core.PersistentVector.EMPTY, 0);\ncljs.core.NeverEquiv = function() {\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 2097152;\n};\ncljs.core.NeverEquiv.cljs$lang$type = true;\ncljs.core.NeverEquiv.cljs$lang$ctorStr = &quot;cljs.core/NeverEquiv&quot;;\ncljs.core.NeverEquiv.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/NeverEquiv&quot;);\n};\ncljs.core.NeverEquiv.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(o, other) {\n  var self__ = this;\n  var o__$1 = this;\n  return false;\n};\ncljs.core.NeverEquiv.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.__GT_NeverEquiv = function __GT_NeverEquiv() {\n  return new cljs.core.NeverEquiv;\n};\ncljs.core.never_equiv = new cljs.core.NeverEquiv;\ncljs.core.equiv_map = function equiv_map(x, y) {\n  return cljs.core.boolean$.call(null, cljs.core.map_QMARK_.call(null, y) ? cljs.core.count.call(null, x) === cljs.core.count.call(null, y) ? cljs.core.every_QMARK_.call(null, cljs.core.identity, cljs.core.map.call(null, function(xkv) {\n    return cljs.core._EQ_.call(null, cljs.core.get.call(null, y, cljs.core.first.call(null, xkv), cljs.core.never_equiv), cljs.core.second.call(null, xkv));\n  }, x)) : null : null);\n};\ncljs.core.scan_array = function scan_array(incr, k, array) {\n  var len = array.length;\n  var i = 0;\n  while (true) {\n    if (i &lt; len) {\n      if (k === array[i]) {\n        return i;\n      } else {\n        var G__22744 = i + incr;\n        i = G__22744;\n        continue;\n      }\n    } else {\n      return null;\n    }\n    break;\n  }\n};\ncljs.core.obj_map_compare_keys = function obj_map_compare_keys(a, b) {\n  var a__$1 = cljs.core.hash.call(null, a);\n  var b__$1 = cljs.core.hash.call(null, b);\n  if (a__$1 &lt; b__$1) {\n    return-1;\n  } else {\n    if (a__$1 &gt; b__$1) {\n      return 1;\n    } else {\n      return 0;\n    }\n  }\n};\ncljs.core.obj_map__GT_hash_map = function obj_map__GT_hash_map(m, k, v) {\n  var ks = m.keys;\n  var len = ks.length;\n  var so = m.strobj;\n  var mm = cljs.core.meta.call(null, m);\n  var i = 0;\n  var out = cljs.core.transient$.call(null, cljs.core.PersistentHashMap.EMPTY);\n  while (true) {\n    if (i &lt; len) {\n      var k__$1 = ks[i];\n      var G__22745 = i + 1;\n      var G__22746 = cljs.core.assoc_BANG_.call(null, out, k__$1, so[k__$1]);\n      i = G__22745;\n      out = G__22746;\n      continue;\n    } else {\n      return cljs.core.with_meta.call(null, cljs.core.persistent_BANG_.call(null, cljs.core.assoc_BANG_.call(null, out, k, v)), mm);\n    }\n    break;\n  }\n};\ncljs.core.obj_clone = function obj_clone(obj, ks) {\n  var new_obj = function() {\n    var obj22750 = {};\n    return obj22750;\n  }();\n  var l = ks.length;\n  var i_22751 = 0;\n  while (true) {\n    if (i_22751 &lt; l) {\n      var k_22752 = ks[i_22751];\n      new_obj[k_22752] = obj[k_22752];\n      var G__22753 = i_22751 + 1;\n      i_22751 = G__22753;\n      continue;\n    } else {\n    }\n    break;\n  }\n  return new_obj;\n};\ncljs.core.ObjMap = function(meta, keys, strobj, update_count, __hash) {\n  this.meta = meta;\n  this.keys = keys;\n  this.strobj = strobj;\n  this.update_count = update_count;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 16123663;\n  this.cljs$lang$protocol_mask$partition1$ = 4;\n};\ncljs.core.ObjMap.cljs$lang$type = true;\ncljs.core.ObjMap.cljs$lang$ctorStr = &quot;cljs.core/ObjMap&quot;;\ncljs.core.ObjMap.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ObjMap&quot;);\n};\ncljs.core.ObjMap.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.ObjMap.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.ObjMap.prototype.cljs$core$ILookup$_lookup$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._lookup.call(null, coll__$1, k, null);\n};\ncljs.core.ObjMap.prototype.cljs$core$ILookup$_lookup$arity$3 = function(coll, k, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (goog.isString(k) &amp;&amp; !(cljs.core.scan_array.call(null, 1, k, self__.keys) == null)) {\n    return self__.strobj[k];\n  } else {\n    return not_found;\n  }\n};\ncljs.core.ObjMap.prototype.cljs$core$IKVReduce$_kv_reduce$arity$3 = function(coll, f, init) {\n  var self__ = this;\n  var coll__$1 = this;\n  var len = self__.keys.length;\n  var keys__$1 = self__.keys.sort(cljs.core.obj_map_compare_keys);\n  var init__$1 = init;\n  while (true) {\n    if (cljs.core.seq.call(null, keys__$1)) {\n      var k = cljs.core.first.call(null, keys__$1);\n      var init__$2 = f.call(null, init__$1, k, self__.strobj[k]);\n      if (cljs.core.reduced_QMARK_.call(null, init__$2)) {\n        return cljs.core.deref.call(null, init__$2);\n      } else {\n        var G__22756 = cljs.core.rest.call(null, keys__$1);\n        var G__22757 = init__$2;\n        keys__$1 = G__22756;\n        init__$1 = G__22757;\n        continue;\n      }\n    } else {\n      return init__$1;\n    }\n    break;\n  }\n};\ncljs.core.ObjMap.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.ObjMap.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.keys.length;\n};\ncljs.core.ObjMap.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_unordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.ObjMap.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_map.call(null, coll__$1, other);\n};\ncljs.core.ObjMap.prototype.cljs$core$IEditableCollection$_as_transient$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.transient$.call(null, cljs.core.into.call(null, cljs.core.PersistentHashMap.EMPTY, coll__$1));\n};\ncljs.core.ObjMap.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.ObjMap.EMPTY, self__.meta);\n};\ncljs.core.ObjMap.prototype.cljs$core$IMap$_dissoc$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (goog.isString(k) &amp;&amp; !(cljs.core.scan_array.call(null, 1, k, self__.keys) == null)) {\n    var new_keys = cljs.core.aclone.call(null, self__.keys);\n    var new_strobj = cljs.core.obj_clone.call(null, self__.strobj, self__.keys);\n    new_keys.splice(cljs.core.scan_array.call(null, 1, k, new_keys), 1);\n    delete new_strobj[k];\n    return new cljs.core.ObjMap(self__.meta, new_keys, new_strobj, self__.update_count + 1, null);\n  } else {\n    return coll__$1;\n  }\n};\ncljs.core.ObjMap.prototype.cljs$core$IAssociative$_assoc$arity$3 = function(coll, k, v) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (goog.isString(k)) {\n    if (self__.update_count &gt; cljs.core.ObjMap.HASHMAP_THRESHOLD || self__.keys.length &gt;= cljs.core.ObjMap.HASHMAP_THRESHOLD) {\n      return cljs.core.obj_map__GT_hash_map.call(null, coll__$1, k, v);\n    } else {\n      if (!(cljs.core.scan_array.call(null, 1, k, self__.keys) == null)) {\n        var new_strobj = cljs.core.obj_clone.call(null, self__.strobj, self__.keys);\n        new_strobj[k] = v;\n        return new cljs.core.ObjMap(self__.meta, self__.keys, new_strobj, self__.update_count + 1, null);\n      } else {\n        var new_strobj = cljs.core.obj_clone.call(null, self__.strobj, self__.keys);\n        var new_keys = cljs.core.aclone.call(null, self__.keys);\n        new_strobj[k] = v;\n        new_keys.push(k);\n        return new cljs.core.ObjMap(self__.meta, new_keys, new_strobj, self__.update_count + 1, null);\n      }\n    }\n  } else {\n    return cljs.core.obj_map__GT_hash_map.call(null, coll__$1, k, v);\n  }\n};\ncljs.core.ObjMap.prototype.cljs$core$IAssociative$_contains_key_QMARK_$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (goog.isString(k) &amp;&amp; !(cljs.core.scan_array.call(null, 1, k, self__.keys) == null)) {\n    return true;\n  } else {\n    return false;\n  }\n};\ncljs.core.ObjMap.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.keys.length &gt; 0) {\n    return cljs.core.map.call(null, function(coll__$1) {\n      return function(p1__22754_SHARP_) {\n        return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [p1__22754_SHARP_, self__.strobj[p1__22754_SHARP_]], null);\n      };\n    }(coll__$1), self__.keys.sort(cljs.core.obj_map_compare_keys));\n  } else {\n    return null;\n  }\n};\ncljs.core.ObjMap.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.ObjMap(meta__$1, self__.keys, self__.strobj, self__.update_count, self__.__hash);\n};\ncljs.core.ObjMap.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, entry) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core.vector_QMARK_.call(null, entry)) {\n    return cljs.core._assoc.call(null, coll__$1, cljs.core._nth.call(null, entry, 0), cljs.core._nth.call(null, entry, 1));\n  } else {\n    return cljs.core.reduce.call(null, cljs.core._conj, coll__$1, entry);\n  }\n};\ncljs.core.ObjMap.prototype.call = function() {\n  var G__22758 = null;\n  var G__22758__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n  };\n  var G__22758__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n  };\n  G__22758 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22758__2.call(this, self__, k);\n      case 3:\n        return G__22758__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22758.cljs$core$IFn$_invoke$arity$2 = G__22758__2;\n  G__22758.cljs$core$IFn$_invoke$arity$3 = G__22758__3;\n  return G__22758;\n}();\ncljs.core.ObjMap.prototype.apply = function(self__, args22755) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22755)));\n};\ncljs.core.ObjMap.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.ObjMap.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n};\ncljs.core.__GT_ObjMap = function __GT_ObjMap(meta, keys, strobj, update_count, __hash) {\n  return new cljs.core.ObjMap(meta, keys, strobj, update_count, __hash);\n};\ncljs.core.ObjMap.EMPTY = new cljs.core.ObjMap(null, [], function() {\n  var obj22760 = {};\n  return obj22760;\n}(), 0, 0);\ncljs.core.ObjMap.HASHMAP_THRESHOLD = 8;\ncljs.core.ObjMap.fromObject = function(ks, obj) {\n  return new cljs.core.ObjMap(null, ks, obj, 0, null);\n};\ncljs.core.Iterator = function(s) {\n  this.s = s;\n};\ncljs.core.Iterator.cljs$lang$type = true;\ncljs.core.Iterator.cljs$lang$ctorStr = &quot;cljs.core/Iterator&quot;;\ncljs.core.Iterator.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Iterator&quot;);\n};\ncljs.core.Iterator.prototype.next = function() {\n  var self__ = this;\n  var _ = this;\n  if (!(self__.s == null)) {\n    var x = cljs.core.first.call(null, self__.s);\n    self__.s = cljs.core.next.call(null, self__.s);\n    return{&quot;done&quot;:false, &quot;value&quot;:x};\n  } else {\n    return{&quot;done&quot;:true, &quot;value&quot;:null};\n  }\n};\ncljs.core.__GT_Iterator = function __GT_Iterator(s) {\n  return new cljs.core.Iterator(s);\n};\ncljs.core.iterator = function iterator(coll) {\n  return new cljs.core.Iterator(cljs.core.seq.call(null, coll));\n};\ncljs.core.EntriesIterator = function(s) {\n  this.s = s;\n};\ncljs.core.EntriesIterator.cljs$lang$type = true;\ncljs.core.EntriesIterator.cljs$lang$ctorStr = &quot;cljs.core/EntriesIterator&quot;;\ncljs.core.EntriesIterator.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/EntriesIterator&quot;);\n};\ncljs.core.EntriesIterator.prototype.next = function() {\n  var self__ = this;\n  var _ = this;\n  if (!(self__.s == null)) {\n    var vec__22761 = cljs.core.first.call(null, self__.s);\n    var k = cljs.core.nth.call(null, vec__22761, 0, null);\n    var v = cljs.core.nth.call(null, vec__22761, 1, null);\n    self__.s = cljs.core.next.call(null, self__.s);\n    return{&quot;done&quot;:false, &quot;value&quot;:[k, v]};\n  } else {\n    return{&quot;done&quot;:true, &quot;value&quot;:null};\n  }\n};\ncljs.core.__GT_EntriesIterator = function __GT_EntriesIterator(s) {\n  return new cljs.core.EntriesIterator(s);\n};\ncljs.core.entries_iterator = function entries_iterator(coll) {\n  return new cljs.core.EntriesIterator(cljs.core.seq.call(null, coll));\n};\ncljs.core.SetEntriesIterator = function(s) {\n  this.s = s;\n};\ncljs.core.SetEntriesIterator.cljs$lang$type = true;\ncljs.core.SetEntriesIterator.cljs$lang$ctorStr = &quot;cljs.core/SetEntriesIterator&quot;;\ncljs.core.SetEntriesIterator.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/SetEntriesIterator&quot;);\n};\ncljs.core.SetEntriesIterator.prototype.next = function() {\n  var self__ = this;\n  var _ = this;\n  if (!(self__.s == null)) {\n    var x = cljs.core.first.call(null, self__.s);\n    self__.s = cljs.core.next.call(null, self__.s);\n    return{&quot;done&quot;:false, &quot;value&quot;:[x, x]};\n  } else {\n    return{&quot;done&quot;:true, &quot;value&quot;:null};\n  }\n};\ncljs.core.__GT_SetEntriesIterator = function __GT_SetEntriesIterator(s) {\n  return new cljs.core.SetEntriesIterator(s);\n};\ncljs.core.set_entries_iterator = function set_entries_iterator(coll) {\n  return new cljs.core.SetEntriesIterator(cljs.core.seq.call(null, coll));\n};\ncljs.core.array_map_index_of_nil_QMARK_ = function array_map_index_of_nil_QMARK_(arr, m, k) {\n  var len = arr.length;\n  var i = 0;\n  while (true) {\n    if (len &lt;= i) {\n      return-1;\n    } else {\n      if (arr[i] == null) {\n        return i;\n      } else {\n        var G__22762 = i + 2;\n        i = G__22762;\n        continue;\n      }\n    }\n    break;\n  }\n};\ncljs.core.array_map_index_of_keyword_QMARK_ = function array_map_index_of_keyword_QMARK_(arr, m, k) {\n  var len = arr.length;\n  var kstr = k.fqn;\n  var i = 0;\n  while (true) {\n    if (len &lt;= i) {\n      return-1;\n    } else {\n      if (function() {\n        var k_SINGLEQUOTE_ = arr[i];\n        return k_SINGLEQUOTE_ instanceof cljs.core.Keyword &amp;&amp; kstr === k_SINGLEQUOTE_.fqn;\n      }()) {\n        return i;\n      } else {\n        var G__22763 = i + 2;\n        i = G__22763;\n        continue;\n      }\n    }\n    break;\n  }\n};\ncljs.core.array_map_index_of_symbol_QMARK_ = function array_map_index_of_symbol_QMARK_(arr, m, k) {\n  var len = arr.length;\n  var kstr = k.str;\n  var i = 0;\n  while (true) {\n    if (len &lt;= i) {\n      return-1;\n    } else {\n      if (function() {\n        var k_SINGLEQUOTE_ = arr[i];\n        return k_SINGLEQUOTE_ instanceof cljs.core.Symbol &amp;&amp; kstr === k_SINGLEQUOTE_.str;\n      }()) {\n        return i;\n      } else {\n        var G__22764 = i + 2;\n        i = G__22764;\n        continue;\n      }\n    }\n    break;\n  }\n};\ncljs.core.array_map_index_of_identical_QMARK_ = function array_map_index_of_identical_QMARK_(arr, m, k) {\n  var len = arr.length;\n  var i = 0;\n  while (true) {\n    if (len &lt;= i) {\n      return-1;\n    } else {\n      if (k === arr[i]) {\n        return i;\n      } else {\n        var G__22765 = i + 2;\n        i = G__22765;\n        continue;\n      }\n    }\n    break;\n  }\n};\ncljs.core.array_map_index_of_equiv_QMARK_ = function array_map_index_of_equiv_QMARK_(arr, m, k) {\n  var len = arr.length;\n  var i = 0;\n  while (true) {\n    if (len &lt;= i) {\n      return-1;\n    } else {\n      if (cljs.core._EQ_.call(null, k, arr[i])) {\n        return i;\n      } else {\n        var G__22766 = i + 2;\n        i = G__22766;\n        continue;\n      }\n    }\n    break;\n  }\n};\ncljs.core.array_map_index_of = function array_map_index_of(m, k) {\n  var arr = m.arr;\n  if (k instanceof cljs.core.Keyword) {\n    return cljs.core.array_map_index_of_keyword_QMARK_.call(null, arr, m, k);\n  } else {\n    if (goog.isString(k) || typeof k === &quot;number&quot;) {\n      return cljs.core.array_map_index_of_identical_QMARK_.call(null, arr, m, k);\n    } else {\n      if (k instanceof cljs.core.Symbol) {\n        return cljs.core.array_map_index_of_symbol_QMARK_.call(null, arr, m, k);\n      } else {\n        if (k == null) {\n          return cljs.core.array_map_index_of_nil_QMARK_.call(null, arr, m, k);\n        } else {\n          return cljs.core.array_map_index_of_equiv_QMARK_.call(null, arr, m, k);\n        }\n      }\n    }\n  }\n};\ncljs.core.array_map_extend_kv = function array_map_extend_kv(m, k, v) {\n  var arr = m.arr;\n  var l = arr.length;\n  var narr = new Array(l + 2);\n  var i_22767 = 0;\n  while (true) {\n    if (i_22767 &lt; l) {\n      narr[i_22767] = arr[i_22767];\n      var G__22768 = i_22767 + 1;\n      i_22767 = G__22768;\n      continue;\n    } else {\n    }\n    break;\n  }\n  narr[l] = k;\n  narr[l + 1] = v;\n  return narr;\n};\ncljs.core.PersistentArrayMapSeq = function(arr, i, _meta) {\n  this.arr = arr;\n  this.i = i;\n  this._meta = _meta;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32374990;\n};\ncljs.core.PersistentArrayMapSeq.cljs$lang$type = true;\ncljs.core.PersistentArrayMapSeq.cljs$lang$ctorStr = &quot;cljs.core/PersistentArrayMapSeq&quot;;\ncljs.core.PersistentArrayMapSeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentArrayMapSeq&quot;);\n};\ncljs.core.PersistentArrayMapSeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentArrayMapSeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__._meta;\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.i &lt; self__.arr.length - 2) {\n    return new cljs.core.PersistentArrayMapSeq(self__.arr, self__.i + 2, self__._meta);\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return(self__.arr.length - self__.i) / 2;\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.hash_ordered_coll.call(null, coll__$1);\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__._meta);\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.arr[self__.i], self__.arr[self__.i + 1]], null);\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.i &lt; self__.arr.length - 2) {\n    return new cljs.core.PersistentArrayMapSeq(self__.arr, self__.i + 2, self__._meta);\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, new_meta) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentArrayMapSeq(self__.arr, self__.i, new_meta);\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_PersistentArrayMapSeq = function __GT_PersistentArrayMapSeq(arr, i, _meta) {\n  return new cljs.core.PersistentArrayMapSeq(arr, i, _meta);\n};\ncljs.core.persistent_array_map_seq = function persistent_array_map_seq(arr, i, _meta) {\n  if (i &lt;= arr.length - 2) {\n    return new cljs.core.PersistentArrayMapSeq(arr, i, _meta);\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentArrayMap = function(meta, cnt, arr, __hash) {\n  this.meta = meta;\n  this.cnt = cnt;\n  this.arr = arr;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 16647951;\n  this.cljs$lang$protocol_mask$partition1$ = 8196;\n};\ncljs.core.PersistentArrayMap.cljs$lang$type = true;\ncljs.core.PersistentArrayMap.cljs$lang$ctorStr = &quot;cljs.core/PersistentArrayMap&quot;;\ncljs.core.PersistentArrayMap.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentArrayMap&quot;);\n};\ncljs.core.PersistentArrayMap.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentArrayMap.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentArrayMap.prototype.keys = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.keys.call(null, coll));\n};\ncljs.core.PersistentArrayMap.prototype.entries = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.entries_iterator.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.PersistentArrayMap.prototype.values = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.vals.call(null, coll));\n};\ncljs.core.PersistentArrayMap.prototype.has = function(k) {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.contains_QMARK_.call(null, coll, k);\n};\ncljs.core.PersistentArrayMap.prototype.get = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.PersistentArrayMap.prototype.forEach = function(f) {\n  var self__ = this;\n  var coll = this;\n  var seq__22770 = cljs.core.seq.call(null, coll);\n  var chunk__22771 = null;\n  var count__22772 = 0;\n  var i__22773 = 0;\n  while (true) {\n    if (i__22773 &lt; count__22772) {\n      var vec__22774 = cljs.core._nth.call(null, chunk__22771, i__22773);\n      var k = cljs.core.nth.call(null, vec__22774, 0, null);\n      var v = cljs.core.nth.call(null, vec__22774, 1, null);\n      f.call(null, v, k);\n      var G__22779 = seq__22770;\n      var G__22780 = chunk__22771;\n      var G__22781 = count__22772;\n      var G__22782 = i__22773 + 1;\n      seq__22770 = G__22779;\n      chunk__22771 = G__22780;\n      count__22772 = G__22781;\n      i__22773 = G__22782;\n      continue;\n    } else {\n      var temp__4126__auto__ = cljs.core.seq.call(null, seq__22770);\n      if (temp__4126__auto__) {\n        var seq__22770__$1 = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, seq__22770__$1)) {\n          var c__17901__auto__ = cljs.core.chunk_first.call(null, seq__22770__$1);\n          var G__22783 = cljs.core.chunk_rest.call(null, seq__22770__$1);\n          var G__22784 = c__17901__auto__;\n          var G__22785 = cljs.core.count.call(null, c__17901__auto__);\n          var G__22786 = 0;\n          seq__22770 = G__22783;\n          chunk__22771 = G__22784;\n          count__22772 = G__22785;\n          i__22773 = G__22786;\n          continue;\n        } else {\n          var vec__22775 = cljs.core.first.call(null, seq__22770__$1);\n          var k = cljs.core.nth.call(null, vec__22775, 0, null);\n          var v = cljs.core.nth.call(null, vec__22775, 1, null);\n          f.call(null, v, k);\n          var G__22787 = cljs.core.next.call(null, seq__22770__$1);\n          var G__22788 = null;\n          var G__22789 = 0;\n          var G__22790 = 0;\n          seq__22770 = G__22787;\n          chunk__22771 = G__22788;\n          count__22772 = G__22789;\n          i__22773 = G__22790;\n          continue;\n        }\n      } else {\n        return null;\n      }\n    }\n    break;\n  }\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$ILookup$_lookup$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._lookup.call(null, coll__$1, k, null);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$ILookup$_lookup$arity$3 = function(coll, k, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  var idx = cljs.core.array_map_index_of.call(null, coll__$1, k);\n  if (idx === -1) {\n    return not_found;\n  } else {\n    return self__.arr[idx + 1];\n  }\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IKVReduce$_kv_reduce$arity$3 = function(coll, f, init) {\n  var self__ = this;\n  var coll__$1 = this;\n  var len = self__.arr.length;\n  var i = 0;\n  var init__$1 = init;\n  while (true) {\n    if (i &lt; len) {\n      var init__$2 = f.call(null, init__$1, self__.arr[i], self__.arr[i + 1]);\n      if (cljs.core.reduced_QMARK_.call(null, init__$2)) {\n        return cljs.core.deref.call(null, init__$2);\n      } else {\n        var G__22791 = i + 2;\n        var G__22792 = init__$2;\n        i = G__22791;\n        init__$1 = G__22792;\n        continue;\n      }\n    } else {\n      return init__$1;\n    }\n    break;\n  }\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.PersistentArrayMap(self__.meta, self__.cnt, self__.arr, self__.__hash);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.cnt;\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_unordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_map.call(null, coll__$1, other);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IEditableCollection$_as_transient$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.TransientArrayMap(function() {\n    var obj22777 = {};\n    return obj22777;\n  }(), self__.arr.length, cljs.core.aclone.call(null, self__.arr));\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._with_meta.call(null, cljs.core.PersistentArrayMap.EMPTY, self__.meta);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IMap$_dissoc$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  var idx = cljs.core.array_map_index_of.call(null, coll__$1, k);\n  if (idx &gt;= 0) {\n    var len = self__.arr.length;\n    var new_len = len - 2;\n    if (new_len === 0) {\n      return cljs.core._empty.call(null, coll__$1);\n    } else {\n      var new_arr = new Array(new_len);\n      var s = 0;\n      var d = 0;\n      while (true) {\n        if (s &gt;= len) {\n          return new cljs.core.PersistentArrayMap(self__.meta, self__.cnt - 1, new_arr, null);\n        } else {\n          if (cljs.core._EQ_.call(null, k, self__.arr[s])) {\n            var G__22793 = s + 2;\n            var G__22794 = d;\n            s = G__22793;\n            d = G__22794;\n            continue;\n          } else {\n            new_arr[d] = self__.arr[s];\n            new_arr[d + 1] = self__.arr[s + 1];\n            var G__22795 = s + 2;\n            var G__22796 = d + 2;\n            s = G__22795;\n            d = G__22796;\n            continue;\n          }\n        }\n        break;\n      }\n    }\n  } else {\n    return coll__$1;\n  }\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IAssociative$_assoc$arity$3 = function(coll, k, v) {\n  var self__ = this;\n  var coll__$1 = this;\n  var idx = cljs.core.array_map_index_of.call(null, coll__$1, k);\n  if (idx === -1) {\n    if (self__.cnt &lt; cljs.core.PersistentArrayMap.HASHMAP_THRESHOLD) {\n      var arr__$1 = cljs.core.array_map_extend_kv.call(null, coll__$1, k, v);\n      return new cljs.core.PersistentArrayMap(self__.meta, self__.cnt + 1, arr__$1, null);\n    } else {\n      return cljs.core._with_meta.call(null, cljs.core._assoc.call(null, cljs.core.into.call(null, cljs.core.PersistentHashMap.EMPTY, coll__$1), k, v), self__.meta);\n    }\n  } else {\n    if (v === self__.arr[idx + 1]) {\n      return coll__$1;\n    } else {\n      var arr__$1 = function() {\n        var G__22778 = cljs.core.aclone.call(null, self__.arr);\n        G__22778[idx + 1] = v;\n        return G__22778;\n      }();\n      return new cljs.core.PersistentArrayMap(self__.meta, self__.cnt, arr__$1, null);\n    }\n  }\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IAssociative$_contains_key_QMARK_$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  return!(cljs.core.array_map_index_of.call(null, coll__$1, k) === -1);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.persistent_array_map_seq.call(null, self__.arr, 0, null);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentArrayMap(meta__$1, self__.cnt, self__.arr, self__.__hash);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, entry) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core.vector_QMARK_.call(null, entry)) {\n    return cljs.core._assoc.call(null, coll__$1, cljs.core._nth.call(null, entry, 0), cljs.core._nth.call(null, entry, 1));\n  } else {\n    var ret = coll__$1;\n    var es = cljs.core.seq.call(null, entry);\n    while (true) {\n      if (es == null) {\n        return ret;\n      } else {\n        var e = cljs.core.first.call(null, es);\n        if (cljs.core.vector_QMARK_.call(null, e)) {\n          var G__22797 = cljs.core._assoc.call(null, ret, cljs.core._nth.call(null, e, 0), cljs.core._nth.call(null, e, 1));\n          var G__22798 = cljs.core.next.call(null, es);\n          ret = G__22797;\n          es = G__22798;\n          continue;\n        } else {\n          throw new Error(&quot;conj on a map takes map entries or seqables of map entries&quot;);\n        }\n      }\n      break;\n    }\n  }\n};\ncljs.core.PersistentArrayMap.prototype.call = function() {\n  var G__22799 = null;\n  var G__22799__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n  };\n  var G__22799__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n  };\n  G__22799 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22799__2.call(this, self__, k);\n      case 3:\n        return G__22799__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22799.cljs$core$IFn$_invoke$arity$2 = G__22799__2;\n  G__22799.cljs$core$IFn$_invoke$arity$3 = G__22799__3;\n  return G__22799;\n}();\ncljs.core.PersistentArrayMap.prototype.apply = function(self__, args22769) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22769)));\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n};\ncljs.core.__GT_PersistentArrayMap = function __GT_PersistentArrayMap(meta, cnt, arr, __hash) {\n  return new cljs.core.PersistentArrayMap(meta, cnt, arr, __hash);\n};\ncljs.core.PersistentArrayMap.EMPTY = new cljs.core.PersistentArrayMap(null, 0, [], null);\ncljs.core.PersistentArrayMap.HASHMAP_THRESHOLD = 8;\ncljs.core.PersistentArrayMap.fromArray = function(arr, no_clone, no_check) {\n  var arr__$1 = no_clone ? arr : cljs.core.aclone.call(null, arr);\n  if (no_check) {\n    var cnt = arr__$1.length / 2;\n    return new cljs.core.PersistentArrayMap(null, cnt, arr__$1, null);\n  } else {\n    var len = arr__$1.length;\n    var i = 0;\n    var ret = cljs.core.transient$.call(null, cljs.core.PersistentArrayMap.EMPTY);\n    while (true) {\n      if (i &lt; len) {\n        var G__22800 = i + 2;\n        var G__22801 = cljs.core._assoc_BANG_.call(null, ret, arr__$1[i], arr__$1[i + 1]);\n        i = G__22800;\n        ret = G__22801;\n        continue;\n      } else {\n        return cljs.core._persistent_BANG_.call(null, ret);\n      }\n      break;\n    }\n  }\n};\ncljs.core.TransientArrayMap = function(editable_QMARK_, len, arr) {\n  this.editable_QMARK_ = editable_QMARK_;\n  this.len = len;\n  this.arr = arr;\n  this.cljs$lang$protocol_mask$partition1$ = 56;\n  this.cljs$lang$protocol_mask$partition0$ = 258;\n};\ncljs.core.TransientArrayMap.cljs$lang$type = true;\ncljs.core.TransientArrayMap.cljs$lang$ctorStr = &quot;cljs.core/TransientArrayMap&quot;;\ncljs.core.TransientArrayMap.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/TransientArrayMap&quot;);\n};\ncljs.core.TransientArrayMap.prototype.cljs$core$ITransientMap$_dissoc_BANG_$arity$2 = function(tcoll, key) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (cljs.core.truth_(self__.editable_QMARK_)) {\n    var idx = cljs.core.array_map_index_of.call(null, tcoll__$1, key);\n    if (idx &gt;= 0) {\n      self__.arr[idx] = self__.arr[self__.len - 2];\n      self__.arr[idx + 1] = self__.arr[self__.len - 1];\n      var G__22802_22804 = self__.arr;\n      G__22802_22804.pop();\n      G__22802_22804.pop();\n      self__.len = self__.len - 2;\n    } else {\n    }\n    return tcoll__$1;\n  } else {\n    throw new Error(&quot;dissoc! after persistent!&quot;);\n  }\n};\ncljs.core.TransientArrayMap.prototype.cljs$core$ITransientAssociative$_assoc_BANG_$arity$3 = function(tcoll, key, val) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (cljs.core.truth_(self__.editable_QMARK_)) {\n    var idx = cljs.core.array_map_index_of.call(null, tcoll__$1, key);\n    if (idx === -1) {\n      if (self__.len + 2 &lt;= 2 * cljs.core.PersistentArrayMap.HASHMAP_THRESHOLD) {\n        self__.len = self__.len + 2;\n        self__.arr.push(key);\n        self__.arr.push(val);\n        return tcoll__$1;\n      } else {\n        return cljs.core.assoc_BANG_.call(null, cljs.core.array__GT_transient_hash_map.call(null, self__.len, self__.arr), key, val);\n      }\n    } else {\n      if (val === self__.arr[idx + 1]) {\n        return tcoll__$1;\n      } else {\n        self__.arr[idx + 1] = val;\n        return tcoll__$1;\n      }\n    }\n  } else {\n    throw new Error(&quot;assoc! after persistent!&quot;);\n  }\n};\ncljs.core.TransientArrayMap.prototype.cljs$core$ITransientCollection$_conj_BANG_$arity$2 = function(tcoll, o) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (cljs.core.truth_(self__.editable_QMARK_)) {\n    if (function() {\n      var G__22803 = o;\n      if (G__22803) {\n        var bit__17795__auto__ = G__22803.cljs$lang$protocol_mask$partition0$ &amp; 2048;\n        if (bit__17795__auto__ || G__22803.cljs$core$IMapEntry$) {\n          return true;\n        } else {\n          if (!G__22803.cljs$lang$protocol_mask$partition0$) {\n            return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMapEntry, G__22803);\n          } else {\n            return false;\n          }\n        }\n      } else {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMapEntry, G__22803);\n      }\n    }()) {\n      return cljs.core._assoc_BANG_.call(null, tcoll__$1, cljs.core.key.call(null, o), cljs.core.val.call(null, o));\n    } else {\n      var es = cljs.core.seq.call(null, o);\n      var tcoll__$2 = tcoll__$1;\n      while (true) {\n        var temp__4124__auto__ = cljs.core.first.call(null, es);\n        if (cljs.core.truth_(temp__4124__auto__)) {\n          var e = temp__4124__auto__;\n          var G__22805 = cljs.core.next.call(null, es);\n          var G__22806 = cljs.core._assoc_BANG_.call(null, tcoll__$2, cljs.core.key.call(null, e), cljs.core.val.call(null, e));\n          es = G__22805;\n          tcoll__$2 = G__22806;\n          continue;\n        } else {\n          return tcoll__$2;\n        }\n        break;\n      }\n    }\n  } else {\n    throw new Error(&quot;conj! after persistent!&quot;);\n  }\n};\ncljs.core.TransientArrayMap.prototype.cljs$core$ITransientCollection$_persistent_BANG_$arity$1 = function(tcoll) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (cljs.core.truth_(self__.editable_QMARK_)) {\n    self__.editable_QMARK_ = false;\n    return new cljs.core.PersistentArrayMap(null, cljs.core.quot.call(null, self__.len, 2), self__.arr, null);\n  } else {\n    throw new Error(&quot;persistent! called twice&quot;);\n  }\n};\ncljs.core.TransientArrayMap.prototype.cljs$core$ILookup$_lookup$arity$2 = function(tcoll, k) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  return cljs.core._lookup.call(null, tcoll__$1, k, null);\n};\ncljs.core.TransientArrayMap.prototype.cljs$core$ILookup$_lookup$arity$3 = function(tcoll, k, not_found) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (cljs.core.truth_(self__.editable_QMARK_)) {\n    var idx = cljs.core.array_map_index_of.call(null, tcoll__$1, k);\n    if (idx === -1) {\n      return not_found;\n    } else {\n      return self__.arr[idx + 1];\n    }\n  } else {\n    throw new Error(&quot;lookup after persistent!&quot;);\n  }\n};\ncljs.core.TransientArrayMap.prototype.cljs$core$ICounted$_count$arity$1 = function(tcoll) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (cljs.core.truth_(self__.editable_QMARK_)) {\n    return cljs.core.quot.call(null, self__.len, 2);\n  } else {\n    throw new Error(&quot;count after persistent!&quot;);\n  }\n};\ncljs.core.__GT_TransientArrayMap = function __GT_TransientArrayMap(editable_QMARK_, len, arr) {\n  return new cljs.core.TransientArrayMap(editable_QMARK_, len, arr);\n};\ncljs.core.array__GT_transient_hash_map = function array__GT_transient_hash_map(len, arr) {\n  var out = cljs.core.transient$.call(null, cljs.core.PersistentHashMap.EMPTY);\n  var i = 0;\n  while (true) {\n    if (i &lt; len) {\n      var G__22807 = cljs.core.assoc_BANG_.call(null, out, arr[i], arr[i + 1]);\n      var G__22808 = i + 2;\n      out = G__22807;\n      i = G__22808;\n      continue;\n    } else {\n      return out;\n    }\n    break;\n  }\n};\ncljs.core.Box = function(val) {\n  this.val = val;\n};\ncljs.core.Box.cljs$lang$type = true;\ncljs.core.Box.cljs$lang$ctorStr = &quot;cljs.core/Box&quot;;\ncljs.core.Box.cljs$lang$ctorPrWriter = function(this__17715__auto__, writer__17716__auto__, opts__17717__auto__) {\n  return cljs.core._write.call(null, writer__17716__auto__, &quot;cljs.core/Box&quot;);\n};\ncljs.core.__GT_Box = function __GT_Box(val) {\n  return new cljs.core.Box(val);\n};\ncljs.core.key_test = function key_test(key, other) {\n  if (key === other) {\n    return true;\n  } else {\n    if (cljs.core.keyword_identical_QMARK_.call(null, key, other)) {\n      return true;\n    } else {\n      return cljs.core._EQ_.call(null, key, other);\n    }\n  }\n};\ncljs.core.mask = function mask(hash, shift) {\n  return hash &gt;&gt;&gt; shift &amp; 31;\n};\ncljs.core.clone_and_set = function() {\n  var clone_and_set = null;\n  var clone_and_set__3 = function(arr, i, a) {\n    var G__22811 = cljs.core.aclone.call(null, arr);\n    G__22811[i] = a;\n    return G__22811;\n  };\n  var clone_and_set__5 = function(arr, i, a, j, b) {\n    var G__22812 = cljs.core.aclone.call(null, arr);\n    G__22812[i] = a;\n    G__22812[j] = b;\n    return G__22812;\n  };\n  clone_and_set = function(arr, i, a, j, b) {\n    switch(arguments.length) {\n      case 3:\n        return clone_and_set__3.call(this, arr, i, a);\n      case 5:\n        return clone_and_set__5.call(this, arr, i, a, j, b);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  clone_and_set.cljs$core$IFn$_invoke$arity$3 = clone_and_set__3;\n  clone_and_set.cljs$core$IFn$_invoke$arity$5 = clone_and_set__5;\n  return clone_and_set;\n}();\ncljs.core.remove_pair = function remove_pair(arr, i) {\n  var new_arr = new Array(arr.length - 2);\n  cljs.core.array_copy.call(null, arr, 0, new_arr, 0, 2 * i);\n  cljs.core.array_copy.call(null, arr, 2 * (i + 1), new_arr, 2 * i, new_arr.length - 2 * i);\n  return new_arr;\n};\ncljs.core.bitmap_indexed_node_index = function bitmap_indexed_node_index(bitmap, bit) {\n  return cljs.core.bit_count.call(null, bitmap &amp; bit - 1);\n};\ncljs.core.bitpos = function bitpos(hash, shift) {\n  return 1 &lt;&lt; (hash &gt;&gt;&gt; shift &amp; 31);\n};\ncljs.core.edit_and_set = function() {\n  var edit_and_set = null;\n  var edit_and_set__4 = function(inode, edit, i, a) {\n    var editable = inode.ensure_editable(edit);\n    editable.arr[i] = a;\n    return editable;\n  };\n  var edit_and_set__6 = function(inode, edit, i, a, j, b) {\n    var editable = inode.ensure_editable(edit);\n    editable.arr[i] = a;\n    editable.arr[j] = b;\n    return editable;\n  };\n  edit_and_set = function(inode, edit, i, a, j, b) {\n    switch(arguments.length) {\n      case 4:\n        return edit_and_set__4.call(this, inode, edit, i, a);\n      case 6:\n        return edit_and_set__6.call(this, inode, edit, i, a, j, b);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  edit_and_set.cljs$core$IFn$_invoke$arity$4 = edit_and_set__4;\n  edit_and_set.cljs$core$IFn$_invoke$arity$6 = edit_and_set__6;\n  return edit_and_set;\n}();\ncljs.core.inode_kv_reduce = function inode_kv_reduce(arr, f, init) {\n  var len = arr.length;\n  var i = 0;\n  var init__$1 = init;\n  while (true) {\n    if (i &lt; len) {\n      var init__$2 = function() {\n        var k = arr[i];\n        if (!(k == null)) {\n          return f.call(null, init__$1, k, arr[i + 1]);\n        } else {\n          var node = arr[i + 1];\n          if (!(node == null)) {\n            return node.kv_reduce(f, init__$1);\n          } else {\n            return init__$1;\n          }\n        }\n      }();\n      if (cljs.core.reduced_QMARK_.call(null, init__$2)) {\n        return cljs.core.deref.call(null, init__$2);\n      } else {\n        var G__22813 = i + 2;\n        var G__22814 = init__$2;\n        i = G__22813;\n        init__$1 = G__22814;\n        continue;\n      }\n    } else {\n      return init__$1;\n    }\n    break;\n  }\n};\ncljs.core.BitmapIndexedNode = function(edit, bitmap, arr) {\n  this.edit = edit;\n  this.bitmap = bitmap;\n  this.arr = arr;\n};\ncljs.core.BitmapIndexedNode.cljs$lang$type = true;\ncljs.core.BitmapIndexedNode.cljs$lang$ctorStr = &quot;cljs.core/BitmapIndexedNode&quot;;\ncljs.core.BitmapIndexedNode.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/BitmapIndexedNode&quot;);\n};\ncljs.core.BitmapIndexedNode.prototype.ensure_editable = function(e) {\n  var self__ = this;\n  var inode = this;\n  if (e === self__.edit) {\n    return inode;\n  } else {\n    var n = cljs.core.bit_count.call(null, self__.bitmap);\n    var new_arr = new Array(n &lt; 0 ? 4 : 2 * (n + 1));\n    cljs.core.array_copy.call(null, self__.arr, 0, new_arr, 0, 2 * n);\n    return new cljs.core.BitmapIndexedNode(e, self__.bitmap, new_arr);\n  }\n};\ncljs.core.BitmapIndexedNode.prototype.inode_without_BANG_ = function(edit__$1, shift, hash, key, removed_leaf_QMARK_) {\n  var self__ = this;\n  var inode = this;\n  var bit = 1 &lt;&lt; (hash &gt;&gt;&gt; shift &amp; 31);\n  if ((self__.bitmap &amp; bit) === 0) {\n    return inode;\n  } else {\n    var idx = cljs.core.bitmap_indexed_node_index.call(null, self__.bitmap, bit);\n    var key_or_nil = self__.arr[2 * idx];\n    var val_or_node = self__.arr[2 * idx + 1];\n    if (key_or_nil == null) {\n      var n = val_or_node.inode_without_BANG_(edit__$1, shift + 5, hash, key, removed_leaf_QMARK_);\n      if (n === val_or_node) {\n        return inode;\n      } else {\n        if (!(n == null)) {\n          return cljs.core.edit_and_set.call(null, inode, edit__$1, 2 * idx + 1, n);\n        } else {\n          if (self__.bitmap === bit) {\n            return null;\n          } else {\n            return inode.edit_and_remove_pair(edit__$1, bit, idx);\n          }\n        }\n      }\n    } else {\n      if (cljs.core.key_test.call(null, key, key_or_nil)) {\n        removed_leaf_QMARK_[0] = true;\n        return inode.edit_and_remove_pair(edit__$1, bit, idx);\n      } else {\n        return inode;\n      }\n    }\n  }\n};\ncljs.core.BitmapIndexedNode.prototype.edit_and_remove_pair = function(e, bit, i) {\n  var self__ = this;\n  var inode = this;\n  if (self__.bitmap === bit) {\n    return null;\n  } else {\n    var editable = inode.ensure_editable(e);\n    var earr = editable.arr;\n    var len = earr.length;\n    editable.bitmap = bit ^ editable.bitmap;\n    cljs.core.array_copy.call(null, earr, 2 * (i + 1), earr, 2 * i, len - 2 * (i + 1));\n    earr[len - 2] = null;\n    earr[len - 1] = null;\n    return editable;\n  }\n};\ncljs.core.BitmapIndexedNode.prototype.inode_seq = function() {\n  var self__ = this;\n  var inode = this;\n  return cljs.core.create_inode_seq.call(null, self__.arr);\n};\ncljs.core.BitmapIndexedNode.prototype.kv_reduce = function(f, init) {\n  var self__ = this;\n  var inode = this;\n  return cljs.core.inode_kv_reduce.call(null, self__.arr, f, init);\n};\ncljs.core.BitmapIndexedNode.prototype.inode_lookup = function(shift, hash, key, not_found) {\n  var self__ = this;\n  var inode = this;\n  var bit = 1 &lt;&lt; (hash &gt;&gt;&gt; shift &amp; 31);\n  if ((self__.bitmap &amp; bit) === 0) {\n    return not_found;\n  } else {\n    var idx = cljs.core.bitmap_indexed_node_index.call(null, self__.bitmap, bit);\n    var key_or_nil = self__.arr[2 * idx];\n    var val_or_node = self__.arr[2 * idx + 1];\n    if (key_or_nil == null) {\n      return val_or_node.inode_lookup(shift + 5, hash, key, not_found);\n    } else {\n      if (cljs.core.key_test.call(null, key, key_or_nil)) {\n        return val_or_node;\n      } else {\n        return not_found;\n      }\n    }\n  }\n};\ncljs.core.BitmapIndexedNode.prototype.inode_assoc_BANG_ = function(edit__$1, shift, hash, key, val, added_leaf_QMARK_) {\n  var self__ = this;\n  var inode = this;\n  var bit = 1 &lt;&lt; (hash &gt;&gt;&gt; shift &amp; 31);\n  var idx = cljs.core.bitmap_indexed_node_index.call(null, self__.bitmap, bit);\n  if ((self__.bitmap &amp; bit) === 0) {\n    var n = cljs.core.bit_count.call(null, self__.bitmap);\n    if (2 * n &lt; self__.arr.length) {\n      var editable = inode.ensure_editable(edit__$1);\n      var earr = editable.arr;\n      added_leaf_QMARK_.val = true;\n      cljs.core.array_copy_downward.call(null, earr, 2 * idx, earr, 2 * (idx + 1), 2 * (n - idx));\n      earr[2 * idx] = key;\n      earr[2 * idx + 1] = val;\n      editable.bitmap = editable.bitmap | bit;\n      return editable;\n    } else {\n      if (n &gt;= 16) {\n        var nodes = [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null];\n        var jdx = hash &gt;&gt;&gt; shift &amp; 31;\n        nodes[jdx] = cljs.core.BitmapIndexedNode.EMPTY.inode_assoc_BANG_(edit__$1, shift + 5, hash, key, val, added_leaf_QMARK_);\n        var i_22815 = 0;\n        var j_22816 = 0;\n        while (true) {\n          if (i_22815 &lt; 32) {\n            if ((self__.bitmap &gt;&gt;&gt; i_22815 &amp; 1) === 0) {\n              var G__22817 = i_22815 + 1;\n              var G__22818 = j_22816;\n              i_22815 = G__22817;\n              j_22816 = G__22818;\n              continue;\n            } else {\n              nodes[i_22815] = !(self__.arr[j_22816] == null) ? cljs.core.BitmapIndexedNode.EMPTY.inode_assoc_BANG_(edit__$1, shift + 5, cljs.core.hash.call(null, self__.arr[j_22816]), self__.arr[j_22816], self__.arr[j_22816 + 1], added_leaf_QMARK_) : self__.arr[j_22816 + 1];\n              var G__22819 = i_22815 + 1;\n              var G__22820 = j_22816 + 2;\n              i_22815 = G__22819;\n              j_22816 = G__22820;\n              continue;\n            }\n          } else {\n          }\n          break;\n        }\n        return new cljs.core.ArrayNode(edit__$1, n + 1, nodes);\n      } else {\n        var new_arr = new Array(2 * (n + 4));\n        cljs.core.array_copy.call(null, self__.arr, 0, new_arr, 0, 2 * idx);\n        new_arr[2 * idx] = key;\n        new_arr[2 * idx + 1] = val;\n        cljs.core.array_copy.call(null, self__.arr, 2 * idx, new_arr, 2 * (idx + 1), 2 * (n - idx));\n        added_leaf_QMARK_.val = true;\n        var editable = inode.ensure_editable(edit__$1);\n        editable.arr = new_arr;\n        editable.bitmap = editable.bitmap | bit;\n        return editable;\n      }\n    }\n  } else {\n    var key_or_nil = self__.arr[2 * idx];\n    var val_or_node = self__.arr[2 * idx + 1];\n    if (key_or_nil == null) {\n      var n = val_or_node.inode_assoc_BANG_(edit__$1, shift + 5, hash, key, val, added_leaf_QMARK_);\n      if (n === val_or_node) {\n        return inode;\n      } else {\n        return cljs.core.edit_and_set.call(null, inode, edit__$1, 2 * idx + 1, n);\n      }\n    } else {\n      if (cljs.core.key_test.call(null, key, key_or_nil)) {\n        if (val === val_or_node) {\n          return inode;\n        } else {\n          return cljs.core.edit_and_set.call(null, inode, edit__$1, 2 * idx + 1, val);\n        }\n      } else {\n        added_leaf_QMARK_.val = true;\n        return cljs.core.edit_and_set.call(null, inode, edit__$1, 2 * idx, null, 2 * idx + 1, cljs.core.create_node.call(null, edit__$1, shift + 5, key_or_nil, val_or_node, hash, key, val));\n      }\n    }\n  }\n};\ncljs.core.BitmapIndexedNode.prototype.inode_assoc = function(shift, hash, key, val, added_leaf_QMARK_) {\n  var self__ = this;\n  var inode = this;\n  var bit = 1 &lt;&lt; (hash &gt;&gt;&gt; shift &amp; 31);\n  var idx = cljs.core.bitmap_indexed_node_index.call(null, self__.bitmap, bit);\n  if ((self__.bitmap &amp; bit) === 0) {\n    var n = cljs.core.bit_count.call(null, self__.bitmap);\n    if (n &gt;= 16) {\n      var nodes = [null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null, null];\n      var jdx = hash &gt;&gt;&gt; shift &amp; 31;\n      nodes[jdx] = cljs.core.BitmapIndexedNode.EMPTY.inode_assoc(shift + 5, hash, key, val, added_leaf_QMARK_);\n      var i_22821 = 0;\n      var j_22822 = 0;\n      while (true) {\n        if (i_22821 &lt; 32) {\n          if ((self__.bitmap &gt;&gt;&gt; i_22821 &amp; 1) === 0) {\n            var G__22823 = i_22821 + 1;\n            var G__22824 = j_22822;\n            i_22821 = G__22823;\n            j_22822 = G__22824;\n            continue;\n          } else {\n            nodes[i_22821] = !(self__.arr[j_22822] == null) ? cljs.core.BitmapIndexedNode.EMPTY.inode_assoc(shift + 5, cljs.core.hash.call(null, self__.arr[j_22822]), self__.arr[j_22822], self__.arr[j_22822 + 1], added_leaf_QMARK_) : self__.arr[j_22822 + 1];\n            var G__22825 = i_22821 + 1;\n            var G__22826 = j_22822 + 2;\n            i_22821 = G__22825;\n            j_22822 = G__22826;\n            continue;\n          }\n        } else {\n        }\n        break;\n      }\n      return new cljs.core.ArrayNode(null, n + 1, nodes);\n    } else {\n      var new_arr = new Array(2 * (n + 1));\n      cljs.core.array_copy.call(null, self__.arr, 0, new_arr, 0, 2 * idx);\n      new_arr[2 * idx] = key;\n      new_arr[2 * idx + 1] = val;\n      cljs.core.array_copy.call(null, self__.arr, 2 * idx, new_arr, 2 * (idx + 1), 2 * (n - idx));\n      added_leaf_QMARK_.val = true;\n      return new cljs.core.BitmapIndexedNode(null, self__.bitmap | bit, new_arr);\n    }\n  } else {\n    var key_or_nil = self__.arr[2 * idx];\n    var val_or_node = self__.arr[2 * idx + 1];\n    if (key_or_nil == null) {\n      var n = val_or_node.inode_assoc(shift + 5, hash, key, val, added_leaf_QMARK_);\n      if (n === val_or_node) {\n        return inode;\n      } else {\n        return new cljs.core.BitmapIndexedNode(null, self__.bitmap, cljs.core.clone_and_set.call(null, self__.arr, 2 * idx + 1, n));\n      }\n    } else {\n      if (cljs.core.key_test.call(null, key, key_or_nil)) {\n        if (val === val_or_node) {\n          return inode;\n        } else {\n          return new cljs.core.BitmapIndexedNode(null, self__.bitmap, cljs.core.clone_and_set.call(null, self__.arr, 2 * idx + 1, val));\n        }\n      } else {\n        added_leaf_QMARK_.val = true;\n        return new cljs.core.BitmapIndexedNode(null, self__.bitmap, cljs.core.clone_and_set.call(null, self__.arr, 2 * idx, null, 2 * idx + 1, cljs.core.create_node.call(null, shift + 5, key_or_nil, val_or_node, hash, key, val)));\n      }\n    }\n  }\n};\ncljs.core.BitmapIndexedNode.prototype.inode_find = function(shift, hash, key, not_found) {\n  var self__ = this;\n  var inode = this;\n  var bit = 1 &lt;&lt; (hash &gt;&gt;&gt; shift &amp; 31);\n  if ((self__.bitmap &amp; bit) === 0) {\n    return not_found;\n  } else {\n    var idx = cljs.core.bitmap_indexed_node_index.call(null, self__.bitmap, bit);\n    var key_or_nil = self__.arr[2 * idx];\n    var val_or_node = self__.arr[2 * idx + 1];\n    if (key_or_nil == null) {\n      return val_or_node.inode_find(shift + 5, hash, key, not_found);\n    } else {\n      if (cljs.core.key_test.call(null, key, key_or_nil)) {\n        return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [key_or_nil, val_or_node], null);\n      } else {\n        return not_found;\n      }\n    }\n  }\n};\ncljs.core.BitmapIndexedNode.prototype.inode_without = function(shift, hash, key) {\n  var self__ = this;\n  var inode = this;\n  var bit = 1 &lt;&lt; (hash &gt;&gt;&gt; shift &amp; 31);\n  if ((self__.bitmap &amp; bit) === 0) {\n    return inode;\n  } else {\n    var idx = cljs.core.bitmap_indexed_node_index.call(null, self__.bitmap, bit);\n    var key_or_nil = self__.arr[2 * idx];\n    var val_or_node = self__.arr[2 * idx + 1];\n    if (key_or_nil == null) {\n      var n = val_or_node.inode_without(shift + 5, hash, key);\n      if (n === val_or_node) {\n        return inode;\n      } else {\n        if (!(n == null)) {\n          return new cljs.core.BitmapIndexedNode(null, self__.bitmap, cljs.core.clone_and_set.call(null, self__.arr, 2 * idx + 1, n));\n        } else {\n          if (self__.bitmap === bit) {\n            return null;\n          } else {\n            return new cljs.core.BitmapIndexedNode(null, self__.bitmap ^ bit, cljs.core.remove_pair.call(null, self__.arr, idx));\n          }\n        }\n      }\n    } else {\n      if (cljs.core.key_test.call(null, key, key_or_nil)) {\n        return new cljs.core.BitmapIndexedNode(null, self__.bitmap ^ bit, cljs.core.remove_pair.call(null, self__.arr, idx));\n      } else {\n        return inode;\n      }\n    }\n  }\n};\ncljs.core.__GT_BitmapIndexedNode = function __GT_BitmapIndexedNode(edit, bitmap, arr) {\n  return new cljs.core.BitmapIndexedNode(edit, bitmap, arr);\n};\ncljs.core.BitmapIndexedNode.EMPTY = new cljs.core.BitmapIndexedNode(null, 0, []);\ncljs.core.pack_array_node = function pack_array_node(array_node, edit, idx) {\n  var arr = array_node.arr;\n  var len = 2 * (array_node.cnt - 1);\n  var new_arr = new Array(len);\n  var i = 0;\n  var j = 1;\n  var bitmap = 0;\n  while (true) {\n    if (i &lt; len) {\n      if (!(i === idx) &amp;&amp; !(arr[i] == null)) {\n        new_arr[j] = arr[i];\n        var G__22827 = i + 1;\n        var G__22828 = j + 2;\n        var G__22829 = bitmap | 1 &lt;&lt; i;\n        i = G__22827;\n        j = G__22828;\n        bitmap = G__22829;\n        continue;\n      } else {\n        var G__22830 = i + 1;\n        var G__22831 = j;\n        var G__22832 = bitmap;\n        i = G__22830;\n        j = G__22831;\n        bitmap = G__22832;\n        continue;\n      }\n    } else {\n      return new cljs.core.BitmapIndexedNode(edit, bitmap, new_arr);\n    }\n    break;\n  }\n};\ncljs.core.ArrayNode = function(edit, cnt, arr) {\n  this.edit = edit;\n  this.cnt = cnt;\n  this.arr = arr;\n};\ncljs.core.ArrayNode.cljs$lang$type = true;\ncljs.core.ArrayNode.cljs$lang$ctorStr = &quot;cljs.core/ArrayNode&quot;;\ncljs.core.ArrayNode.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ArrayNode&quot;);\n};\ncljs.core.ArrayNode.prototype.ensure_editable = function(e) {\n  var self__ = this;\n  var inode = this;\n  if (e === self__.edit) {\n    return inode;\n  } else {\n    return new cljs.core.ArrayNode(e, self__.cnt, cljs.core.aclone.call(null, self__.arr));\n  }\n};\ncljs.core.ArrayNode.prototype.inode_without_BANG_ = function(edit__$1, shift, hash, key, removed_leaf_QMARK_) {\n  var self__ = this;\n  var inode = this;\n  var idx = hash &gt;&gt;&gt; shift &amp; 31;\n  var node = self__.arr[idx];\n  if (node == null) {\n    return inode;\n  } else {\n    var n = node.inode_without_BANG_(edit__$1, shift + 5, hash, key, removed_leaf_QMARK_);\n    if (n === node) {\n      return inode;\n    } else {\n      if (n == null) {\n        if (self__.cnt &lt;= 8) {\n          return cljs.core.pack_array_node.call(null, inode, edit__$1, idx);\n        } else {\n          var editable = cljs.core.edit_and_set.call(null, inode, edit__$1, idx, n);\n          editable.cnt = editable.cnt - 1;\n          return editable;\n        }\n      } else {\n        return cljs.core.edit_and_set.call(null, inode, edit__$1, idx, n);\n      }\n    }\n  }\n};\ncljs.core.ArrayNode.prototype.inode_seq = function() {\n  var self__ = this;\n  var inode = this;\n  return cljs.core.create_array_node_seq.call(null, self__.arr);\n};\ncljs.core.ArrayNode.prototype.kv_reduce = function(f, init) {\n  var self__ = this;\n  var inode = this;\n  var len = self__.arr.length;\n  var i = 0;\n  var init__$1 = init;\n  while (true) {\n    if (i &lt; len) {\n      var node = self__.arr[i];\n      if (!(node == null)) {\n        var init__$2 = node.kv_reduce(f, init__$1);\n        if (cljs.core.reduced_QMARK_.call(null, init__$2)) {\n          return cljs.core.deref.call(null, init__$2);\n        } else {\n          var G__22833 = i + 1;\n          var G__22834 = init__$2;\n          i = G__22833;\n          init__$1 = G__22834;\n          continue;\n        }\n      } else {\n        var G__22835 = i + 1;\n        var G__22836 = init__$1;\n        i = G__22835;\n        init__$1 = G__22836;\n        continue;\n      }\n    } else {\n      return init__$1;\n    }\n    break;\n  }\n};\ncljs.core.ArrayNode.prototype.inode_lookup = function(shift, hash, key, not_found) {\n  var self__ = this;\n  var inode = this;\n  var idx = hash &gt;&gt;&gt; shift &amp; 31;\n  var node = self__.arr[idx];\n  if (!(node == null)) {\n    return node.inode_lookup(shift + 5, hash, key, not_found);\n  } else {\n    return not_found;\n  }\n};\ncljs.core.ArrayNode.prototype.inode_assoc_BANG_ = function(edit__$1, shift, hash, key, val, added_leaf_QMARK_) {\n  var self__ = this;\n  var inode = this;\n  var idx = hash &gt;&gt;&gt; shift &amp; 31;\n  var node = self__.arr[idx];\n  if (node == null) {\n    var editable = cljs.core.edit_and_set.call(null, inode, edit__$1, idx, cljs.core.BitmapIndexedNode.EMPTY.inode_assoc_BANG_(edit__$1, shift + 5, hash, key, val, added_leaf_QMARK_));\n    editable.cnt = editable.cnt + 1;\n    return editable;\n  } else {\n    var n = node.inode_assoc_BANG_(edit__$1, shift + 5, hash, key, val, added_leaf_QMARK_);\n    if (n === node) {\n      return inode;\n    } else {\n      return cljs.core.edit_and_set.call(null, inode, edit__$1, idx, n);\n    }\n  }\n};\ncljs.core.ArrayNode.prototype.inode_assoc = function(shift, hash, key, val, added_leaf_QMARK_) {\n  var self__ = this;\n  var inode = this;\n  var idx = hash &gt;&gt;&gt; shift &amp; 31;\n  var node = self__.arr[idx];\n  if (node == null) {\n    return new cljs.core.ArrayNode(null, self__.cnt + 1, cljs.core.clone_and_set.call(null, self__.arr, idx, cljs.core.BitmapIndexedNode.EMPTY.inode_assoc(shift + 5, hash, key, val, added_leaf_QMARK_)));\n  } else {\n    var n = node.inode_assoc(shift + 5, hash, key, val, added_leaf_QMARK_);\n    if (n === node) {\n      return inode;\n    } else {\n      return new cljs.core.ArrayNode(null, self__.cnt, cljs.core.clone_and_set.call(null, self__.arr, idx, n));\n    }\n  }\n};\ncljs.core.ArrayNode.prototype.inode_find = function(shift, hash, key, not_found) {\n  var self__ = this;\n  var inode = this;\n  var idx = hash &gt;&gt;&gt; shift &amp; 31;\n  var node = self__.arr[idx];\n  if (!(node == null)) {\n    return node.inode_find(shift + 5, hash, key, not_found);\n  } else {\n    return not_found;\n  }\n};\ncljs.core.ArrayNode.prototype.inode_without = function(shift, hash, key) {\n  var self__ = this;\n  var inode = this;\n  var idx = hash &gt;&gt;&gt; shift &amp; 31;\n  var node = self__.arr[idx];\n  if (!(node == null)) {\n    var n = node.inode_without(shift + 5, hash, key);\n    if (n === node) {\n      return inode;\n    } else {\n      if (n == null) {\n        if (self__.cnt &lt;= 8) {\n          return cljs.core.pack_array_node.call(null, inode, null, idx);\n        } else {\n          return new cljs.core.ArrayNode(null, self__.cnt - 1, cljs.core.clone_and_set.call(null, self__.arr, idx, n));\n        }\n      } else {\n        return new cljs.core.ArrayNode(null, self__.cnt, cljs.core.clone_and_set.call(null, self__.arr, idx, n));\n      }\n    }\n  } else {\n    return inode;\n  }\n};\ncljs.core.__GT_ArrayNode = function __GT_ArrayNode(edit, cnt, arr) {\n  return new cljs.core.ArrayNode(edit, cnt, arr);\n};\ncljs.core.hash_collision_node_find_index = function hash_collision_node_find_index(arr, cnt, key) {\n  var lim = 2 * cnt;\n  var i = 0;\n  while (true) {\n    if (i &lt; lim) {\n      if (cljs.core.key_test.call(null, key, arr[i])) {\n        return i;\n      } else {\n        var G__22837 = i + 2;\n        i = G__22837;\n        continue;\n      }\n    } else {\n      return-1;\n    }\n    break;\n  }\n};\ncljs.core.HashCollisionNode = function(edit, collision_hash, cnt, arr) {\n  this.edit = edit;\n  this.collision_hash = collision_hash;\n  this.cnt = cnt;\n  this.arr = arr;\n};\ncljs.core.HashCollisionNode.cljs$lang$type = true;\ncljs.core.HashCollisionNode.cljs$lang$ctorStr = &quot;cljs.core/HashCollisionNode&quot;;\ncljs.core.HashCollisionNode.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/HashCollisionNode&quot;);\n};\ncljs.core.HashCollisionNode.prototype.ensure_editable = function(e) {\n  var self__ = this;\n  var inode = this;\n  if (e === self__.edit) {\n    return inode;\n  } else {\n    var new_arr = new Array(2 * (self__.cnt + 1));\n    cljs.core.array_copy.call(null, self__.arr, 0, new_arr, 0, 2 * self__.cnt);\n    return new cljs.core.HashCollisionNode(e, self__.collision_hash, self__.cnt, new_arr);\n  }\n};\ncljs.core.HashCollisionNode.prototype.inode_without_BANG_ = function(edit__$1, shift, hash, key, removed_leaf_QMARK_) {\n  var self__ = this;\n  var inode = this;\n  var idx = cljs.core.hash_collision_node_find_index.call(null, self__.arr, self__.cnt, key);\n  if (idx === -1) {\n    return inode;\n  } else {\n    removed_leaf_QMARK_[0] = true;\n    if (self__.cnt === 1) {\n      return null;\n    } else {\n      var editable = inode.ensure_editable(edit__$1);\n      var earr = editable.arr;\n      earr[idx] = earr[2 * self__.cnt - 2];\n      earr[idx + 1] = earr[2 * self__.cnt - 1];\n      earr[2 * self__.cnt - 1] = null;\n      earr[2 * self__.cnt - 2] = null;\n      editable.cnt = editable.cnt - 1;\n      return editable;\n    }\n  }\n};\ncljs.core.HashCollisionNode.prototype.inode_seq = function() {\n  var self__ = this;\n  var inode = this;\n  return cljs.core.create_inode_seq.call(null, self__.arr);\n};\ncljs.core.HashCollisionNode.prototype.kv_reduce = function(f, init) {\n  var self__ = this;\n  var inode = this;\n  return cljs.core.inode_kv_reduce.call(null, self__.arr, f, init);\n};\ncljs.core.HashCollisionNode.prototype.inode_lookup = function(shift, hash, key, not_found) {\n  var self__ = this;\n  var inode = this;\n  var idx = cljs.core.hash_collision_node_find_index.call(null, self__.arr, self__.cnt, key);\n  if (idx &lt; 0) {\n    return not_found;\n  } else {\n    if (cljs.core.key_test.call(null, key, self__.arr[idx])) {\n      return self__.arr[idx + 1];\n    } else {\n      return not_found;\n    }\n  }\n};\ncljs.core.HashCollisionNode.prototype.inode_assoc_BANG_ = function(edit__$1, shift, hash, key, val, added_leaf_QMARK_) {\n  var self__ = this;\n  var inode = this;\n  if (hash === self__.collision_hash) {\n    var idx = cljs.core.hash_collision_node_find_index.call(null, self__.arr, self__.cnt, key);\n    if (idx === -1) {\n      if (self__.arr.length &gt; 2 * self__.cnt) {\n        var editable = cljs.core.edit_and_set.call(null, inode, edit__$1, 2 * self__.cnt, key, 2 * self__.cnt + 1, val);\n        added_leaf_QMARK_.val = true;\n        editable.cnt = editable.cnt + 1;\n        return editable;\n      } else {\n        var len = self__.arr.length;\n        var new_arr = new Array(len + 2);\n        cljs.core.array_copy.call(null, self__.arr, 0, new_arr, 0, len);\n        new_arr[len] = key;\n        new_arr[len + 1] = val;\n        added_leaf_QMARK_.val = true;\n        return inode.ensure_editable_array(edit__$1, self__.cnt + 1, new_arr);\n      }\n    } else {\n      if (self__.arr[idx + 1] === val) {\n        return inode;\n      } else {\n        return cljs.core.edit_and_set.call(null, inode, edit__$1, idx + 1, val);\n      }\n    }\n  } else {\n    return(new cljs.core.BitmapIndexedNode(edit__$1, 1 &lt;&lt; (self__.collision_hash &gt;&gt;&gt; shift &amp; 31), [null, inode, null, null])).inode_assoc_BANG_(edit__$1, shift, hash, key, val, added_leaf_QMARK_);\n  }\n};\ncljs.core.HashCollisionNode.prototype.inode_assoc = function(shift, hash, key, val, added_leaf_QMARK_) {\n  var self__ = this;\n  var inode = this;\n  if (hash === self__.collision_hash) {\n    var idx = cljs.core.hash_collision_node_find_index.call(null, self__.arr, self__.cnt, key);\n    if (idx === -1) {\n      var len = 2 * self__.cnt;\n      var new_arr = new Array(len + 2);\n      cljs.core.array_copy.call(null, self__.arr, 0, new_arr, 0, len);\n      new_arr[len] = key;\n      new_arr[len + 1] = val;\n      added_leaf_QMARK_.val = true;\n      return new cljs.core.HashCollisionNode(null, self__.collision_hash, self__.cnt + 1, new_arr);\n    } else {\n      if (cljs.core._EQ_.call(null, self__.arr[idx], val)) {\n        return inode;\n      } else {\n        return new cljs.core.HashCollisionNode(null, self__.collision_hash, self__.cnt, cljs.core.clone_and_set.call(null, self__.arr, idx + 1, val));\n      }\n    }\n  } else {\n    return(new cljs.core.BitmapIndexedNode(null, 1 &lt;&lt; (self__.collision_hash &gt;&gt;&gt; shift &amp; 31), [null, inode])).inode_assoc(shift, hash, key, val, added_leaf_QMARK_);\n  }\n};\ncljs.core.HashCollisionNode.prototype.ensure_editable_array = function(e, count, array) {\n  var self__ = this;\n  var inode = this;\n  if (e === self__.edit) {\n    self__.arr = array;\n    self__.cnt = count;\n    return inode;\n  } else {\n    return new cljs.core.HashCollisionNode(self__.edit, self__.collision_hash, count, array);\n  }\n};\ncljs.core.HashCollisionNode.prototype.inode_find = function(shift, hash, key, not_found) {\n  var self__ = this;\n  var inode = this;\n  var idx = cljs.core.hash_collision_node_find_index.call(null, self__.arr, self__.cnt, key);\n  if (idx &lt; 0) {\n    return not_found;\n  } else {\n    if (cljs.core.key_test.call(null, key, self__.arr[idx])) {\n      return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.arr[idx], self__.arr[idx + 1]], null);\n    } else {\n      return not_found;\n    }\n  }\n};\ncljs.core.HashCollisionNode.prototype.inode_without = function(shift, hash, key) {\n  var self__ = this;\n  var inode = this;\n  var idx = cljs.core.hash_collision_node_find_index.call(null, self__.arr, self__.cnt, key);\n  if (idx === -1) {\n    return inode;\n  } else {\n    if (self__.cnt === 1) {\n      return null;\n    } else {\n      return new cljs.core.HashCollisionNode(null, self__.collision_hash, self__.cnt - 1, cljs.core.remove_pair.call(null, self__.arr, cljs.core.quot.call(null, idx, 2)));\n    }\n  }\n};\ncljs.core.__GT_HashCollisionNode = function __GT_HashCollisionNode(edit, collision_hash, cnt, arr) {\n  return new cljs.core.HashCollisionNode(edit, collision_hash, cnt, arr);\n};\ncljs.core.create_node = function() {\n  var create_node = null;\n  var create_node__6 = function(shift, key1, val1, key2hash, key2, val2) {\n    var key1hash = cljs.core.hash.call(null, key1);\n    if (key1hash === key2hash) {\n      return new cljs.core.HashCollisionNode(null, key1hash, 2, [key1, val1, key2, val2]);\n    } else {\n      var added_leaf_QMARK_ = new cljs.core.Box(false);\n      return cljs.core.BitmapIndexedNode.EMPTY.inode_assoc(shift, key1hash, key1, val1, added_leaf_QMARK_).inode_assoc(shift, key2hash, key2, val2, added_leaf_QMARK_);\n    }\n  };\n  var create_node__7 = function(edit, shift, key1, val1, key2hash, key2, val2) {\n    var key1hash = cljs.core.hash.call(null, key1);\n    if (key1hash === key2hash) {\n      return new cljs.core.HashCollisionNode(null, key1hash, 2, [key1, val1, key2, val2]);\n    } else {\n      var added_leaf_QMARK_ = new cljs.core.Box(false);\n      return cljs.core.BitmapIndexedNode.EMPTY.inode_assoc_BANG_(edit, shift, key1hash, key1, val1, added_leaf_QMARK_).inode_assoc_BANG_(edit, shift, key2hash, key2, val2, added_leaf_QMARK_);\n    }\n  };\n  create_node = function(edit, shift, key1, val1, key2hash, key2, val2) {\n    switch(arguments.length) {\n      case 6:\n        return create_node__6.call(this, edit, shift, key1, val1, key2hash, key2);\n      case 7:\n        return create_node__7.call(this, edit, shift, key1, val1, key2hash, key2, val2);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  create_node.cljs$core$IFn$_invoke$arity$6 = create_node__6;\n  create_node.cljs$core$IFn$_invoke$arity$7 = create_node__7;\n  return create_node;\n}();\ncljs.core.NodeSeq = function(meta, nodes, i, s, __hash) {\n  this.meta = meta;\n  this.nodes = nodes;\n  this.i = i;\n  this.s = s;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32374860;\n};\ncljs.core.NodeSeq.cljs$lang$type = true;\ncljs.core.NodeSeq.cljs$lang$ctorStr = &quot;cljs.core/NodeSeq&quot;;\ncljs.core.NodeSeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/NodeSeq&quot;);\n};\ncljs.core.NodeSeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.NodeSeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.NodeSeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.NodeSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.NodeSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.NodeSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__.meta);\n};\ncljs.core.NodeSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.NodeSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.NodeSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.s == null) {\n    return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.nodes[self__.i], self__.nodes[self__.i + 1]], null);\n  } else {\n    return cljs.core.first.call(null, self__.s);\n  }\n};\ncljs.core.NodeSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.s == null) {\n    return cljs.core.create_inode_seq.call(null, self__.nodes, self__.i + 2, null);\n  } else {\n    return cljs.core.create_inode_seq.call(null, self__.nodes, self__.i, cljs.core.next.call(null, self__.s));\n  }\n};\ncljs.core.NodeSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  return this$__$1;\n};\ncljs.core.NodeSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.NodeSeq(meta__$1, self__.nodes, self__.i, self__.s, self__.__hash);\n};\ncljs.core.NodeSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_NodeSeq = function __GT_NodeSeq(meta, nodes, i, s, __hash) {\n  return new cljs.core.NodeSeq(meta, nodes, i, s, __hash);\n};\ncljs.core.create_inode_seq = function() {\n  var create_inode_seq = null;\n  var create_inode_seq__1 = function(nodes) {\n    return create_inode_seq.call(null, nodes, 0, null);\n  };\n  var create_inode_seq__3 = function(nodes, i, s) {\n    if (s == null) {\n      var len = nodes.length;\n      var j = i;\n      while (true) {\n        if (j &lt; len) {\n          if (!(nodes[j] == null)) {\n            return new cljs.core.NodeSeq(null, nodes, j, null, null);\n          } else {\n            var temp__4124__auto__ = nodes[j + 1];\n            if (cljs.core.truth_(temp__4124__auto__)) {\n              var node = temp__4124__auto__;\n              var temp__4124__auto____$1 = node.inode_seq();\n              if (cljs.core.truth_(temp__4124__auto____$1)) {\n                var node_seq = temp__4124__auto____$1;\n                return new cljs.core.NodeSeq(null, nodes, j + 2, node_seq, null);\n              } else {\n                var G__22838 = j + 2;\n                j = G__22838;\n                continue;\n              }\n            } else {\n              var G__22839 = j + 2;\n              j = G__22839;\n              continue;\n            }\n          }\n        } else {\n          return null;\n        }\n        break;\n      }\n    } else {\n      return new cljs.core.NodeSeq(null, nodes, i, s, null);\n    }\n  };\n  create_inode_seq = function(nodes, i, s) {\n    switch(arguments.length) {\n      case 1:\n        return create_inode_seq__1.call(this, nodes);\n      case 3:\n        return create_inode_seq__3.call(this, nodes, i, s);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  create_inode_seq.cljs$core$IFn$_invoke$arity$1 = create_inode_seq__1;\n  create_inode_seq.cljs$core$IFn$_invoke$arity$3 = create_inode_seq__3;\n  return create_inode_seq;\n}();\ncljs.core.ArrayNodeSeq = function(meta, nodes, i, s, __hash) {\n  this.meta = meta;\n  this.nodes = nodes;\n  this.i = i;\n  this.s = s;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32374860;\n};\ncljs.core.ArrayNodeSeq.cljs$lang$type = true;\ncljs.core.ArrayNodeSeq.cljs$lang$ctorStr = &quot;cljs.core/ArrayNodeSeq&quot;;\ncljs.core.ArrayNodeSeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ArrayNodeSeq&quot;);\n};\ncljs.core.ArrayNodeSeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.ArrayNodeSeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__.meta);\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.first.call(null, self__.s);\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.create_array_node_seq.call(null, null, self__.nodes, self__.i, cljs.core.next.call(null, self__.s));\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  return this$__$1;\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.ArrayNodeSeq(meta__$1, self__.nodes, self__.i, self__.s, self__.__hash);\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_ArrayNodeSeq = function __GT_ArrayNodeSeq(meta, nodes, i, s, __hash) {\n  return new cljs.core.ArrayNodeSeq(meta, nodes, i, s, __hash);\n};\ncljs.core.create_array_node_seq = function() {\n  var create_array_node_seq = null;\n  var create_array_node_seq__1 = function(nodes) {\n    return create_array_node_seq.call(null, null, nodes, 0, null);\n  };\n  var create_array_node_seq__4 = function(meta, nodes, i, s) {\n    if (s == null) {\n      var len = nodes.length;\n      var j = i;\n      while (true) {\n        if (j &lt; len) {\n          var temp__4124__auto__ = nodes[j];\n          if (cljs.core.truth_(temp__4124__auto__)) {\n            var nj = temp__4124__auto__;\n            var temp__4124__auto____$1 = nj.inode_seq();\n            if (cljs.core.truth_(temp__4124__auto____$1)) {\n              var ns = temp__4124__auto____$1;\n              return new cljs.core.ArrayNodeSeq(meta, nodes, j + 1, ns, null);\n            } else {\n              var G__22840 = j + 1;\n              j = G__22840;\n              continue;\n            }\n          } else {\n            var G__22841 = j + 1;\n            j = G__22841;\n            continue;\n          }\n        } else {\n          return null;\n        }\n        break;\n      }\n    } else {\n      return new cljs.core.ArrayNodeSeq(meta, nodes, i, s, null);\n    }\n  };\n  create_array_node_seq = function(meta, nodes, i, s) {\n    switch(arguments.length) {\n      case 1:\n        return create_array_node_seq__1.call(this, meta);\n      case 4:\n        return create_array_node_seq__4.call(this, meta, nodes, i, s);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  create_array_node_seq.cljs$core$IFn$_invoke$arity$1 = create_array_node_seq__1;\n  create_array_node_seq.cljs$core$IFn$_invoke$arity$4 = create_array_node_seq__4;\n  return create_array_node_seq;\n}();\ncljs.core.PersistentHashMap = function(meta, cnt, root, has_nil_QMARK_, nil_val, __hash) {\n  this.meta = meta;\n  this.cnt = cnt;\n  this.root = root;\n  this.has_nil_QMARK_ = has_nil_QMARK_;\n  this.nil_val = nil_val;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 16123663;\n  this.cljs$lang$protocol_mask$partition1$ = 8196;\n};\ncljs.core.PersistentHashMap.cljs$lang$type = true;\ncljs.core.PersistentHashMap.cljs$lang$ctorStr = &quot;cljs.core/PersistentHashMap&quot;;\ncljs.core.PersistentHashMap.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentHashMap&quot;);\n};\ncljs.core.PersistentHashMap.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentHashMap.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentHashMap.prototype.keys = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.keys.call(null, coll));\n};\ncljs.core.PersistentHashMap.prototype.entries = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.entries_iterator.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.PersistentHashMap.prototype.values = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.vals.call(null, coll));\n};\ncljs.core.PersistentHashMap.prototype.has = function(k) {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.contains_QMARK_.call(null, coll, k);\n};\ncljs.core.PersistentHashMap.prototype.get = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.PersistentHashMap.prototype.forEach = function(f) {\n  var self__ = this;\n  var coll = this;\n  var seq__22843 = cljs.core.seq.call(null, coll);\n  var chunk__22844 = null;\n  var count__22845 = 0;\n  var i__22846 = 0;\n  while (true) {\n    if (i__22846 &lt; count__22845) {\n      var vec__22847 = cljs.core._nth.call(null, chunk__22844, i__22846);\n      var k = cljs.core.nth.call(null, vec__22847, 0, null);\n      var v = cljs.core.nth.call(null, vec__22847, 1, null);\n      f.call(null, v, k);\n      var G__22851 = seq__22843;\n      var G__22852 = chunk__22844;\n      var G__22853 = count__22845;\n      var G__22854 = i__22846 + 1;\n      seq__22843 = G__22851;\n      chunk__22844 = G__22852;\n      count__22845 = G__22853;\n      i__22846 = G__22854;\n      continue;\n    } else {\n      var temp__4126__auto__ = cljs.core.seq.call(null, seq__22843);\n      if (temp__4126__auto__) {\n        var seq__22843__$1 = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, seq__22843__$1)) {\n          var c__17901__auto__ = cljs.core.chunk_first.call(null, seq__22843__$1);\n          var G__22855 = cljs.core.chunk_rest.call(null, seq__22843__$1);\n          var G__22856 = c__17901__auto__;\n          var G__22857 = cljs.core.count.call(null, c__17901__auto__);\n          var G__22858 = 0;\n          seq__22843 = G__22855;\n          chunk__22844 = G__22856;\n          count__22845 = G__22857;\n          i__22846 = G__22858;\n          continue;\n        } else {\n          var vec__22848 = cljs.core.first.call(null, seq__22843__$1);\n          var k = cljs.core.nth.call(null, vec__22848, 0, null);\n          var v = cljs.core.nth.call(null, vec__22848, 1, null);\n          f.call(null, v, k);\n          var G__22859 = cljs.core.next.call(null, seq__22843__$1);\n          var G__22860 = null;\n          var G__22861 = 0;\n          var G__22862 = 0;\n          seq__22843 = G__22859;\n          chunk__22844 = G__22860;\n          count__22845 = G__22861;\n          i__22846 = G__22862;\n          continue;\n        }\n      } else {\n        return null;\n      }\n    }\n    break;\n  }\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$ILookup$_lookup$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._lookup.call(null, coll__$1, k, null);\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$ILookup$_lookup$arity$3 = function(coll, k, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (k == null) {\n    if (self__.has_nil_QMARK_) {\n      return self__.nil_val;\n    } else {\n      return not_found;\n    }\n  } else {\n    if (self__.root == null) {\n      return not_found;\n    } else {\n      return self__.root.inode_lookup(0, cljs.core.hash.call(null, k), k, not_found);\n    }\n  }\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IKVReduce$_kv_reduce$arity$3 = function(coll, f, init) {\n  var self__ = this;\n  var coll__$1 = this;\n  var init__$1 = self__.has_nil_QMARK_ ? f.call(null, init, null, self__.nil_val) : init;\n  if (cljs.core.reduced_QMARK_.call(null, init__$1)) {\n    return cljs.core.deref.call(null, init__$1);\n  } else {\n    if (!(self__.root == null)) {\n      return self__.root.kv_reduce(f, init__$1);\n    } else {\n      return init__$1;\n    }\n  }\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.PersistentHashMap(self__.meta, self__.cnt, self__.root, self__.has_nil_QMARK_, self__.nil_val, self__.__hash);\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.cnt;\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_unordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_map.call(null, coll__$1, other);\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IEditableCollection$_as_transient$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.TransientHashMap(function() {\n    var obj22850 = {};\n    return obj22850;\n  }(), self__.root, self__.cnt, self__.has_nil_QMARK_, self__.nil_val);\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._with_meta.call(null, cljs.core.PersistentHashMap.EMPTY, self__.meta);\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IMap$_dissoc$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (k == null) {\n    if (self__.has_nil_QMARK_) {\n      return new cljs.core.PersistentHashMap(self__.meta, self__.cnt - 1, self__.root, false, null, null);\n    } else {\n      return coll__$1;\n    }\n  } else {\n    if (self__.root == null) {\n      return coll__$1;\n    } else {\n      var new_root = self__.root.inode_without(0, cljs.core.hash.call(null, k), k);\n      if (new_root === self__.root) {\n        return coll__$1;\n      } else {\n        return new cljs.core.PersistentHashMap(self__.meta, self__.cnt - 1, new_root, self__.has_nil_QMARK_, self__.nil_val, null);\n      }\n    }\n  }\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IAssociative$_assoc$arity$3 = function(coll, k, v) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (k == null) {\n    if (self__.has_nil_QMARK_ &amp;&amp; v === self__.nil_val) {\n      return coll__$1;\n    } else {\n      return new cljs.core.PersistentHashMap(self__.meta, self__.has_nil_QMARK_ ? self__.cnt : self__.cnt + 1, self__.root, true, v, null);\n    }\n  } else {\n    var added_leaf_QMARK_ = new cljs.core.Box(false);\n    var new_root = (self__.root == null ? cljs.core.BitmapIndexedNode.EMPTY : self__.root).inode_assoc(0, cljs.core.hash.call(null, k), k, v, added_leaf_QMARK_);\n    if (new_root === self__.root) {\n      return coll__$1;\n    } else {\n      return new cljs.core.PersistentHashMap(self__.meta, added_leaf_QMARK_.val ? self__.cnt + 1 : self__.cnt, new_root, self__.has_nil_QMARK_, self__.nil_val, null);\n    }\n  }\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IAssociative$_contains_key_QMARK_$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (k == null) {\n    return self__.has_nil_QMARK_;\n  } else {\n    if (self__.root == null) {\n      return false;\n    } else {\n      return!(self__.root.inode_lookup(0, cljs.core.hash.call(null, k), k, cljs.core.lookup_sentinel) === cljs.core.lookup_sentinel);\n    }\n  }\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt &gt; 0) {\n    var s = !(self__.root == null) ? self__.root.inode_seq() : null;\n    if (self__.has_nil_QMARK_) {\n      return cljs.core.cons.call(null, new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [null, self__.nil_val], null), s);\n    } else {\n      return s;\n    }\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentHashMap(meta__$1, self__.cnt, self__.root, self__.has_nil_QMARK_, self__.nil_val, self__.__hash);\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, entry) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core.vector_QMARK_.call(null, entry)) {\n    return cljs.core._assoc.call(null, coll__$1, cljs.core._nth.call(null, entry, 0), cljs.core._nth.call(null, entry, 1));\n  } else {\n    var ret = coll__$1;\n    var es = cljs.core.seq.call(null, entry);\n    while (true) {\n      if (es == null) {\n        return ret;\n      } else {\n        var e = cljs.core.first.call(null, es);\n        if (cljs.core.vector_QMARK_.call(null, e)) {\n          var G__22863 = cljs.core._assoc.call(null, ret, cljs.core._nth.call(null, e, 0), cljs.core._nth.call(null, e, 1));\n          var G__22864 = cljs.core.next.call(null, es);\n          ret = G__22863;\n          es = G__22864;\n          continue;\n        } else {\n          throw new Error(&quot;conj on a map takes map entries or seqables of map entries&quot;);\n        }\n      }\n      break;\n    }\n  }\n};\ncljs.core.PersistentHashMap.prototype.call = function() {\n  var G__22865 = null;\n  var G__22865__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n  };\n  var G__22865__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n  };\n  G__22865 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22865__2.call(this, self__, k);\n      case 3:\n        return G__22865__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22865.cljs$core$IFn$_invoke$arity$2 = G__22865__2;\n  G__22865.cljs$core$IFn$_invoke$arity$3 = G__22865__3;\n  return G__22865;\n}();\ncljs.core.PersistentHashMap.prototype.apply = function(self__, args22842) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22842)));\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n};\ncljs.core.__GT_PersistentHashMap = function __GT_PersistentHashMap(meta, cnt, root, has_nil_QMARK_, nil_val, __hash) {\n  return new cljs.core.PersistentHashMap(meta, cnt, root, has_nil_QMARK_, nil_val, __hash);\n};\ncljs.core.PersistentHashMap.EMPTY = new cljs.core.PersistentHashMap(null, 0, null, false, null, 0);\ncljs.core.PersistentHashMap.fromArrays = function(ks, vs) {\n  var len = ks.length;\n  var i = 0;\n  var out = cljs.core.transient$.call(null, cljs.core.PersistentHashMap.EMPTY);\n  while (true) {\n    if (i &lt; len) {\n      var G__22866 = i + 1;\n      var G__22867 = cljs.core._assoc_BANG_.call(null, out, ks[i], vs[i]);\n      i = G__22866;\n      out = G__22867;\n      continue;\n    } else {\n      return cljs.core.persistent_BANG_.call(null, out);\n    }\n    break;\n  }\n};\ncljs.core.TransientHashMap = function(edit, root, count, has_nil_QMARK_, nil_val) {\n  this.edit = edit;\n  this.root = root;\n  this.count = count;\n  this.has_nil_QMARK_ = has_nil_QMARK_;\n  this.nil_val = nil_val;\n  this.cljs$lang$protocol_mask$partition1$ = 56;\n  this.cljs$lang$protocol_mask$partition0$ = 258;\n};\ncljs.core.TransientHashMap.cljs$lang$type = true;\ncljs.core.TransientHashMap.cljs$lang$ctorStr = &quot;cljs.core/TransientHashMap&quot;;\ncljs.core.TransientHashMap.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/TransientHashMap&quot;);\n};\ncljs.core.TransientHashMap.prototype.cljs$core$ITransientMap$_dissoc_BANG_$arity$2 = function(tcoll, key) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  return tcoll__$1.without_BANG_(key);\n};\ncljs.core.TransientHashMap.prototype.cljs$core$ITransientAssociative$_assoc_BANG_$arity$3 = function(tcoll, key, val) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  return tcoll__$1.assoc_BANG_(key, val);\n};\ncljs.core.TransientHashMap.prototype.cljs$core$ITransientCollection$_conj_BANG_$arity$2 = function(tcoll, val) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  return tcoll__$1.conj_BANG_(val);\n};\ncljs.core.TransientHashMap.prototype.cljs$core$ITransientCollection$_persistent_BANG_$arity$1 = function(tcoll) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  return tcoll__$1.persistent_BANG_();\n};\ncljs.core.TransientHashMap.prototype.cljs$core$ILookup$_lookup$arity$2 = function(tcoll, k) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (k == null) {\n    if (self__.has_nil_QMARK_) {\n      return self__.nil_val;\n    } else {\n      return null;\n    }\n  } else {\n    if (self__.root == null) {\n      return null;\n    } else {\n      return self__.root.inode_lookup(0, cljs.core.hash.call(null, k), k);\n    }\n  }\n};\ncljs.core.TransientHashMap.prototype.cljs$core$ILookup$_lookup$arity$3 = function(tcoll, k, not_found) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (k == null) {\n    if (self__.has_nil_QMARK_) {\n      return self__.nil_val;\n    } else {\n      return not_found;\n    }\n  } else {\n    if (self__.root == null) {\n      return not_found;\n    } else {\n      return self__.root.inode_lookup(0, cljs.core.hash.call(null, k), k, not_found);\n    }\n  }\n};\ncljs.core.TransientHashMap.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.edit) {\n    return self__.count;\n  } else {\n    throw new Error(&quot;count after persistent!&quot;);\n  }\n};\ncljs.core.TransientHashMap.prototype.conj_BANG_ = function(o) {\n  var self__ = this;\n  var tcoll = this;\n  if (self__.edit) {\n    if (function() {\n      var G__22868 = o;\n      if (G__22868) {\n        var bit__17795__auto__ = G__22868.cljs$lang$protocol_mask$partition0$ &amp; 2048;\n        if (bit__17795__auto__ || G__22868.cljs$core$IMapEntry$) {\n          return true;\n        } else {\n          if (!G__22868.cljs$lang$protocol_mask$partition0$) {\n            return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMapEntry, G__22868);\n          } else {\n            return false;\n          }\n        }\n      } else {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMapEntry, G__22868);\n      }\n    }()) {\n      return tcoll.assoc_BANG_(cljs.core.key.call(null, o), cljs.core.val.call(null, o));\n    } else {\n      var es = cljs.core.seq.call(null, o);\n      var tcoll__$1 = tcoll;\n      while (true) {\n        var temp__4124__auto__ = cljs.core.first.call(null, es);\n        if (cljs.core.truth_(temp__4124__auto__)) {\n          var e = temp__4124__auto__;\n          var G__22869 = cljs.core.next.call(null, es);\n          var G__22870 = tcoll__$1.assoc_BANG_(cljs.core.key.call(null, e), cljs.core.val.call(null, e));\n          es = G__22869;\n          tcoll__$1 = G__22870;\n          continue;\n        } else {\n          return tcoll__$1;\n        }\n        break;\n      }\n    }\n  } else {\n    throw new Error(&quot;conj! after persistent&quot;);\n  }\n};\ncljs.core.TransientHashMap.prototype.assoc_BANG_ = function(k, v) {\n  var self__ = this;\n  var tcoll = this;\n  if (self__.edit) {\n    if (k == null) {\n      if (self__.nil_val === v) {\n      } else {\n        self__.nil_val = v;\n      }\n      if (self__.has_nil_QMARK_) {\n      } else {\n        self__.count = self__.count + 1;\n        self__.has_nil_QMARK_ = true;\n      }\n      return tcoll;\n    } else {\n      var added_leaf_QMARK_ = new cljs.core.Box(false);\n      var node = (self__.root == null ? cljs.core.BitmapIndexedNode.EMPTY : self__.root).inode_assoc_BANG_(self__.edit, 0, cljs.core.hash.call(null, k), k, v, added_leaf_QMARK_);\n      if (node === self__.root) {\n      } else {\n        self__.root = node;\n      }\n      if (added_leaf_QMARK_.val) {\n        self__.count = self__.count + 1;\n      } else {\n      }\n      return tcoll;\n    }\n  } else {\n    throw new Error(&quot;assoc! after persistent!&quot;);\n  }\n};\ncljs.core.TransientHashMap.prototype.without_BANG_ = function(k) {\n  var self__ = this;\n  var tcoll = this;\n  if (self__.edit) {\n    if (k == null) {\n      if (self__.has_nil_QMARK_) {\n        self__.has_nil_QMARK_ = false;\n        self__.nil_val = null;\n        self__.count = self__.count - 1;\n        return tcoll;\n      } else {\n        return tcoll;\n      }\n    } else {\n      if (self__.root == null) {\n        return tcoll;\n      } else {\n        var removed_leaf_QMARK_ = new cljs.core.Box(false);\n        var node = self__.root.inode_without_BANG_(self__.edit, 0, cljs.core.hash.call(null, k), k, removed_leaf_QMARK_);\n        if (node === self__.root) {\n        } else {\n          self__.root = node;\n        }\n        if (cljs.core.truth_(removed_leaf_QMARK_[0])) {\n          self__.count = self__.count - 1;\n        } else {\n        }\n        return tcoll;\n      }\n    }\n  } else {\n    throw new Error(&quot;dissoc! after persistent!&quot;);\n  }\n};\ncljs.core.TransientHashMap.prototype.persistent_BANG_ = function() {\n  var self__ = this;\n  var tcoll = this;\n  if (self__.edit) {\n    self__.edit = null;\n    return new cljs.core.PersistentHashMap(null, self__.count, self__.root, self__.has_nil_QMARK_, self__.nil_val, null);\n  } else {\n    throw new Error(&quot;persistent! called twice&quot;);\n  }\n};\ncljs.core.__GT_TransientHashMap = function __GT_TransientHashMap(edit, root, count, has_nil_QMARK_, nil_val) {\n  return new cljs.core.TransientHashMap(edit, root, count, has_nil_QMARK_, nil_val);\n};\ncljs.core.tree_map_seq_push = function tree_map_seq_push(node, stack, ascending_QMARK_) {\n  var t = node;\n  var stack__$1 = stack;\n  while (true) {\n    if (!(t == null)) {\n      var G__22871 = ascending_QMARK_ ? t.left : t.right;\n      var G__22872 = cljs.core.conj.call(null, stack__$1, t);\n      t = G__22871;\n      stack__$1 = G__22872;\n      continue;\n    } else {\n      return stack__$1;\n    }\n    break;\n  }\n};\ncljs.core.PersistentTreeMapSeq = function(meta, stack, ascending_QMARK_, cnt, __hash) {\n  this.meta = meta;\n  this.stack = stack;\n  this.ascending_QMARK_ = ascending_QMARK_;\n  this.cnt = cnt;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32374862;\n};\ncljs.core.PersistentTreeMapSeq.cljs$lang$type = true;\ncljs.core.PersistentTreeMapSeq.cljs$lang$ctorStr = &quot;cljs.core/PersistentTreeMapSeq&quot;;\ncljs.core.PersistentTreeMapSeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentTreeMapSeq&quot;);\n};\ncljs.core.PersistentTreeMapSeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentTreeMapSeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt &lt; 0) {\n    return cljs.core.count.call(null, cljs.core.next.call(null, coll__$1)) + 1;\n  } else {\n    return self__.cnt;\n  }\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__.meta);\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  return cljs.core.peek.call(null, self__.stack);\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  var t = cljs.core.first.call(null, self__.stack);\n  var next_stack = cljs.core.tree_map_seq_push.call(null, self__.ascending_QMARK_ ? t.right : t.left, cljs.core.next.call(null, self__.stack), self__.ascending_QMARK_);\n  if (!(next_stack == null)) {\n    return new cljs.core.PersistentTreeMapSeq(null, next_stack, self__.ascending_QMARK_, self__.cnt - 1, null);\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  return this$__$1;\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentTreeMapSeq(meta__$1, self__.stack, self__.ascending_QMARK_, self__.cnt, self__.__hash);\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_PersistentTreeMapSeq = function __GT_PersistentTreeMapSeq(meta, stack, ascending_QMARK_, cnt, __hash) {\n  return new cljs.core.PersistentTreeMapSeq(meta, stack, ascending_QMARK_, cnt, __hash);\n};\ncljs.core.create_tree_map_seq = function create_tree_map_seq(tree, ascending_QMARK_, cnt) {\n  return new cljs.core.PersistentTreeMapSeq(null, cljs.core.tree_map_seq_push.call(null, tree, null, ascending_QMARK_), ascending_QMARK_, cnt, null);\n};\ncljs.core.balance_left = function balance_left(key, val, ins, right) {\n  if (ins instanceof cljs.core.RedNode) {\n    if (ins.left instanceof cljs.core.RedNode) {\n      return new cljs.core.RedNode(ins.key, ins.val, ins.left.blacken(), new cljs.core.BlackNode(key, val, ins.right, right, null), null);\n    } else {\n      if (ins.right instanceof cljs.core.RedNode) {\n        return new cljs.core.RedNode(ins.right.key, ins.right.val, new cljs.core.BlackNode(ins.key, ins.val, ins.left, ins.right.left, null), new cljs.core.BlackNode(key, val, ins.right.right, right, null), null);\n      } else {\n        return new cljs.core.BlackNode(key, val, ins, right, null);\n      }\n    }\n  } else {\n    return new cljs.core.BlackNode(key, val, ins, right, null);\n  }\n};\ncljs.core.balance_right = function balance_right(key, val, left, ins) {\n  if (ins instanceof cljs.core.RedNode) {\n    if (ins.right instanceof cljs.core.RedNode) {\n      return new cljs.core.RedNode(ins.key, ins.val, new cljs.core.BlackNode(key, val, left, ins.left, null), ins.right.blacken(), null);\n    } else {\n      if (ins.left instanceof cljs.core.RedNode) {\n        return new cljs.core.RedNode(ins.left.key, ins.left.val, new cljs.core.BlackNode(key, val, left, ins.left.left, null), new cljs.core.BlackNode(ins.key, ins.val, ins.left.right, ins.right, null), null);\n      } else {\n        return new cljs.core.BlackNode(key, val, left, ins, null);\n      }\n    }\n  } else {\n    return new cljs.core.BlackNode(key, val, left, ins, null);\n  }\n};\ncljs.core.balance_left_del = function balance_left_del(key, val, del, right) {\n  if (del instanceof cljs.core.RedNode) {\n    return new cljs.core.RedNode(key, val, del.blacken(), right, null);\n  } else {\n    if (right instanceof cljs.core.BlackNode) {\n      return cljs.core.balance_right.call(null, key, val, del, right.redden());\n    } else {\n      if (right instanceof cljs.core.RedNode &amp;&amp; right.left instanceof cljs.core.BlackNode) {\n        return new cljs.core.RedNode(right.left.key, right.left.val, new cljs.core.BlackNode(key, val, del, right.left.left, null), cljs.core.balance_right.call(null, right.key, right.val, right.left.right, right.right.redden()), null);\n      } else {\n        throw new Error(&quot;red-black tree invariant violation&quot;);\n      }\n    }\n  }\n};\ncljs.core.balance_right_del = function balance_right_del(key, val, left, del) {\n  if (del instanceof cljs.core.RedNode) {\n    return new cljs.core.RedNode(key, val, left, del.blacken(), null);\n  } else {\n    if (left instanceof cljs.core.BlackNode) {\n      return cljs.core.balance_left.call(null, key, val, left.redden(), del);\n    } else {\n      if (left instanceof cljs.core.RedNode &amp;&amp; left.right instanceof cljs.core.BlackNode) {\n        return new cljs.core.RedNode(left.right.key, left.right.val, cljs.core.balance_left.call(null, left.key, left.val, left.left.redden(), left.right.left), new cljs.core.BlackNode(key, val, left.right.right, del, null), null);\n      } else {\n        throw new Error(&quot;red-black tree invariant violation&quot;);\n      }\n    }\n  }\n};\ncljs.core.tree_map_kv_reduce = function tree_map_kv_reduce(node, f, init) {\n  var init__$1 = !(node.left == null) ? tree_map_kv_reduce.call(null, node.left, f, init) : init;\n  if (cljs.core.reduced_QMARK_.call(null, init__$1)) {\n    return cljs.core.deref.call(null, init__$1);\n  } else {\n    var init__$2 = f.call(null, init__$1, node.key, node.val);\n    if (cljs.core.reduced_QMARK_.call(null, init__$2)) {\n      return cljs.core.deref.call(null, init__$2);\n    } else {\n      var init__$3 = !(node.right == null) ? tree_map_kv_reduce.call(null, node.right, f, init__$2) : init__$2;\n      if (cljs.core.reduced_QMARK_.call(null, init__$3)) {\n        return cljs.core.deref.call(null, init__$3);\n      } else {\n        return init__$3;\n      }\n    }\n  }\n};\ncljs.core.BlackNode = function(key, val, left, right, __hash) {\n  this.key = key;\n  this.val = val;\n  this.left = left;\n  this.right = right;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32402207;\n};\ncljs.core.BlackNode.cljs$lang$type = true;\ncljs.core.BlackNode.cljs$lang$ctorStr = &quot;cljs.core/BlackNode&quot;;\ncljs.core.BlackNode.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/BlackNode&quot;);\n};\ncljs.core.BlackNode.prototype.add_right = function(ins) {\n  var self__ = this;\n  var node = this;\n  return ins.balance_right(node);\n};\ncljs.core.BlackNode.prototype.redden = function() {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.RedNode(self__.key, self__.val, self__.left, self__.right, null);\n};\ncljs.core.BlackNode.prototype.blacken = function() {\n  var self__ = this;\n  var node = this;\n  return node;\n};\ncljs.core.BlackNode.prototype.add_left = function(ins) {\n  var self__ = this;\n  var node = this;\n  return ins.balance_left(node);\n};\ncljs.core.BlackNode.prototype.replace = function(key__$1, val__$1, left__$1, right__$1) {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.BlackNode(key__$1, val__$1, left__$1, right__$1, null);\n};\ncljs.core.BlackNode.prototype.balance_left = function(parent) {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.BlackNode(parent.key, parent.val, node, parent.right, null);\n};\ncljs.core.BlackNode.prototype.balance_right = function(parent) {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.BlackNode(parent.key, parent.val, parent.left, node, null);\n};\ncljs.core.BlackNode.prototype.remove_left = function(del) {\n  var self__ = this;\n  var node = this;\n  return cljs.core.balance_left_del.call(null, self__.key, self__.val, del, self__.right);\n};\ncljs.core.BlackNode.prototype.kv_reduce = function(f, init) {\n  var self__ = this;\n  var node = this;\n  return cljs.core.tree_map_kv_reduce.call(null, node, f, init);\n};\ncljs.core.BlackNode.prototype.remove_right = function(del) {\n  var self__ = this;\n  var node = this;\n  return cljs.core.balance_right_del.call(null, self__.key, self__.val, self__.left, del);\n};\ncljs.core.BlackNode.prototype.cljs$core$ILookup$_lookup$arity$2 = function(node, k) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core._nth.call(null, node__$1, k, null);\n};\ncljs.core.BlackNode.prototype.cljs$core$ILookup$_lookup$arity$3 = function(node, k, not_found) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core._nth.call(null, node__$1, k, not_found);\n};\ncljs.core.BlackNode.prototype.cljs$core$IIndexed$_nth$arity$2 = function(node, n) {\n  var self__ = this;\n  var node__$1 = this;\n  if (n === 0) {\n    return self__.key;\n  } else {\n    if (n === 1) {\n      return self__.val;\n    } else {\n      return null;\n    }\n  }\n};\ncljs.core.BlackNode.prototype.cljs$core$IIndexed$_nth$arity$3 = function(node, n, not_found) {\n  var self__ = this;\n  var node__$1 = this;\n  if (n === 0) {\n    return self__.key;\n  } else {\n    if (n === 1) {\n      return self__.val;\n    } else {\n      return not_found;\n    }\n  }\n};\ncljs.core.BlackNode.prototype.cljs$core$IVector$_assoc_n$arity$3 = function(node, n, v) {\n  var self__ = this;\n  var node__$1 = this;\n  return(new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key, self__.val], null)).cljs$core$IVector$_assoc_n$arity$3(null, n, v);\n};\ncljs.core.BlackNode.prototype.cljs$core$IMeta$_meta$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return null;\n};\ncljs.core.BlackNode.prototype.cljs$core$ICounted$_count$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return 2;\n};\ncljs.core.BlackNode.prototype.cljs$core$IMapEntry$_key$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return self__.key;\n};\ncljs.core.BlackNode.prototype.cljs$core$IMapEntry$_val$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return self__.val;\n};\ncljs.core.BlackNode.prototype.cljs$core$IStack$_peek$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return self__.val;\n};\ncljs.core.BlackNode.prototype.cljs$core$IStack$_pop$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key], null);\n};\ncljs.core.BlackNode.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.BlackNode.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.BlackNode.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.PersistentVector.EMPTY;\n};\ncljs.core.BlackNode.prototype.cljs$core$IReduce$_reduce$arity$2 = function(node, f) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.ci_reduce.call(null, node__$1, f);\n};\ncljs.core.BlackNode.prototype.cljs$core$IReduce$_reduce$arity$3 = function(node, f, start) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.ci_reduce.call(null, node__$1, f, start);\n};\ncljs.core.BlackNode.prototype.cljs$core$IAssociative$_assoc$arity$3 = function(node, k, v) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.assoc.call(null, new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key, self__.val], null), k, v);\n};\ncljs.core.BlackNode.prototype.cljs$core$ISeqable$_seq$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core._conj.call(null, cljs.core._conj.call(null, cljs.core.List.EMPTY, self__.val), self__.key);\n};\ncljs.core.BlackNode.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(node, meta) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.with_meta.call(null, new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key, self__.val], null), meta);\n};\ncljs.core.BlackNode.prototype.cljs$core$ICollection$_conj$arity$2 = function(node, o) {\n  var self__ = this;\n  var node__$1 = this;\n  return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key, self__.val, o], null);\n};\ncljs.core.BlackNode.prototype.call = function() {\n  var G__22874 = null;\n  var G__22874__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var node = self____$1;\n    return node.cljs$core$ILookup$_lookup$arity$2(null, k);\n  };\n  var G__22874__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var node = self____$1;\n    return node.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n  };\n  G__22874 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22874__2.call(this, self__, k);\n      case 3:\n        return G__22874__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22874.cljs$core$IFn$_invoke$arity$2 = G__22874__2;\n  G__22874.cljs$core$IFn$_invoke$arity$3 = G__22874__3;\n  return G__22874;\n}();\ncljs.core.BlackNode.prototype.apply = function(self__, args22873) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22873)));\n};\ncljs.core.BlackNode.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var node = this;\n  return node.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.BlackNode.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var node = this;\n  return node.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n};\ncljs.core.__GT_BlackNode = function __GT_BlackNode(key, val, left, right, __hash) {\n  return new cljs.core.BlackNode(key, val, left, right, __hash);\n};\ncljs.core.RedNode = function(key, val, left, right, __hash) {\n  this.key = key;\n  this.val = val;\n  this.left = left;\n  this.right = right;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32402207;\n};\ncljs.core.RedNode.cljs$lang$type = true;\ncljs.core.RedNode.cljs$lang$ctorStr = &quot;cljs.core/RedNode&quot;;\ncljs.core.RedNode.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/RedNode&quot;);\n};\ncljs.core.RedNode.prototype.add_right = function(ins) {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.RedNode(self__.key, self__.val, self__.left, ins, null);\n};\ncljs.core.RedNode.prototype.redden = function() {\n  var self__ = this;\n  var node = this;\n  throw new Error(&quot;red-black tree invariant violation&quot;);\n};\ncljs.core.RedNode.prototype.blacken = function() {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.BlackNode(self__.key, self__.val, self__.left, self__.right, null);\n};\ncljs.core.RedNode.prototype.add_left = function(ins) {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.RedNode(self__.key, self__.val, ins, self__.right, null);\n};\ncljs.core.RedNode.prototype.replace = function(key__$1, val__$1, left__$1, right__$1) {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.RedNode(key__$1, val__$1, left__$1, right__$1, null);\n};\ncljs.core.RedNode.prototype.balance_left = function(parent) {\n  var self__ = this;\n  var node = this;\n  if (self__.left instanceof cljs.core.RedNode) {\n    return new cljs.core.RedNode(self__.key, self__.val, self__.left.blacken(), new cljs.core.BlackNode(parent.key, parent.val, self__.right, parent.right, null), null);\n  } else {\n    if (self__.right instanceof cljs.core.RedNode) {\n      return new cljs.core.RedNode(self__.right.key, self__.right.val, new cljs.core.BlackNode(self__.key, self__.val, self__.left, self__.right.left, null), new cljs.core.BlackNode(parent.key, parent.val, self__.right.right, parent.right, null), null);\n    } else {\n      return new cljs.core.BlackNode(parent.key, parent.val, node, parent.right, null);\n    }\n  }\n};\ncljs.core.RedNode.prototype.balance_right = function(parent) {\n  var self__ = this;\n  var node = this;\n  if (self__.right instanceof cljs.core.RedNode) {\n    return new cljs.core.RedNode(self__.key, self__.val, new cljs.core.BlackNode(parent.key, parent.val, parent.left, self__.left, null), self__.right.blacken(), null);\n  } else {\n    if (self__.left instanceof cljs.core.RedNode) {\n      return new cljs.core.RedNode(self__.left.key, self__.left.val, new cljs.core.BlackNode(parent.key, parent.val, parent.left, self__.left.left, null), new cljs.core.BlackNode(self__.key, self__.val, self__.left.right, self__.right, null), null);\n    } else {\n      return new cljs.core.BlackNode(parent.key, parent.val, parent.left, node, null);\n    }\n  }\n};\ncljs.core.RedNode.prototype.remove_left = function(del) {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.RedNode(self__.key, self__.val, del, self__.right, null);\n};\ncljs.core.RedNode.prototype.kv_reduce = function(f, init) {\n  var self__ = this;\n  var node = this;\n  return cljs.core.tree_map_kv_reduce.call(null, node, f, init);\n};\ncljs.core.RedNode.prototype.remove_right = function(del) {\n  var self__ = this;\n  var node = this;\n  return new cljs.core.RedNode(self__.key, self__.val, self__.left, del, null);\n};\ncljs.core.RedNode.prototype.cljs$core$ILookup$_lookup$arity$2 = function(node, k) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core._nth.call(null, node__$1, k, null);\n};\ncljs.core.RedNode.prototype.cljs$core$ILookup$_lookup$arity$3 = function(node, k, not_found) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core._nth.call(null, node__$1, k, not_found);\n};\ncljs.core.RedNode.prototype.cljs$core$IIndexed$_nth$arity$2 = function(node, n) {\n  var self__ = this;\n  var node__$1 = this;\n  if (n === 0) {\n    return self__.key;\n  } else {\n    if (n === 1) {\n      return self__.val;\n    } else {\n      return null;\n    }\n  }\n};\ncljs.core.RedNode.prototype.cljs$core$IIndexed$_nth$arity$3 = function(node, n, not_found) {\n  var self__ = this;\n  var node__$1 = this;\n  if (n === 0) {\n    return self__.key;\n  } else {\n    if (n === 1) {\n      return self__.val;\n    } else {\n      return not_found;\n    }\n  }\n};\ncljs.core.RedNode.prototype.cljs$core$IVector$_assoc_n$arity$3 = function(node, n, v) {\n  var self__ = this;\n  var node__$1 = this;\n  return(new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key, self__.val], null)).cljs$core$IVector$_assoc_n$arity$3(null, n, v);\n};\ncljs.core.RedNode.prototype.cljs$core$IMeta$_meta$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return null;\n};\ncljs.core.RedNode.prototype.cljs$core$ICounted$_count$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return 2;\n};\ncljs.core.RedNode.prototype.cljs$core$IMapEntry$_key$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return self__.key;\n};\ncljs.core.RedNode.prototype.cljs$core$IMapEntry$_val$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return self__.val;\n};\ncljs.core.RedNode.prototype.cljs$core$IStack$_peek$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return self__.val;\n};\ncljs.core.RedNode.prototype.cljs$core$IStack$_pop$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key], null);\n};\ncljs.core.RedNode.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.RedNode.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.RedNode.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.PersistentVector.EMPTY;\n};\ncljs.core.RedNode.prototype.cljs$core$IReduce$_reduce$arity$2 = function(node, f) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.ci_reduce.call(null, node__$1, f);\n};\ncljs.core.RedNode.prototype.cljs$core$IReduce$_reduce$arity$3 = function(node, f, start) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.ci_reduce.call(null, node__$1, f, start);\n};\ncljs.core.RedNode.prototype.cljs$core$IAssociative$_assoc$arity$3 = function(node, k, v) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.assoc.call(null, new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key, self__.val], null), k, v);\n};\ncljs.core.RedNode.prototype.cljs$core$ISeqable$_seq$arity$1 = function(node) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core._conj.call(null, cljs.core._conj.call(null, cljs.core.List.EMPTY, self__.val), self__.key);\n};\ncljs.core.RedNode.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(node, meta) {\n  var self__ = this;\n  var node__$1 = this;\n  return cljs.core.with_meta.call(null, new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key, self__.val], null), meta);\n};\ncljs.core.RedNode.prototype.cljs$core$ICollection$_conj$arity$2 = function(node, o) {\n  var self__ = this;\n  var node__$1 = this;\n  return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [self__.key, self__.val, o], null);\n};\ncljs.core.RedNode.prototype.call = function() {\n  var G__22876 = null;\n  var G__22876__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var node = self____$1;\n    return node.cljs$core$ILookup$_lookup$arity$2(null, k);\n  };\n  var G__22876__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var node = self____$1;\n    return node.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n  };\n  G__22876 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22876__2.call(this, self__, k);\n      case 3:\n        return G__22876__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22876.cljs$core$IFn$_invoke$arity$2 = G__22876__2;\n  G__22876.cljs$core$IFn$_invoke$arity$3 = G__22876__3;\n  return G__22876;\n}();\ncljs.core.RedNode.prototype.apply = function(self__, args22875) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22875)));\n};\ncljs.core.RedNode.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var node = this;\n  return node.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.RedNode.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var node = this;\n  return node.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n};\ncljs.core.__GT_RedNode = function __GT_RedNode(key, val, left, right, __hash) {\n  return new cljs.core.RedNode(key, val, left, right, __hash);\n};\ncljs.core.tree_map_add = function tree_map_add(comp, tree, k, v, found) {\n  if (tree == null) {\n    return new cljs.core.RedNode(k, v, null, null, null);\n  } else {\n    var c = comp.call(null, k, tree.key);\n    if (c === 0) {\n      found[0] = tree;\n      return null;\n    } else {\n      if (c &lt; 0) {\n        var ins = tree_map_add.call(null, comp, tree.left, k, v, found);\n        if (!(ins == null)) {\n          return tree.add_left(ins);\n        } else {\n          return null;\n        }\n      } else {\n        var ins = tree_map_add.call(null, comp, tree.right, k, v, found);\n        if (!(ins == null)) {\n          return tree.add_right(ins);\n        } else {\n          return null;\n        }\n      }\n    }\n  }\n};\ncljs.core.tree_map_append = function tree_map_append(left, right) {\n  if (left == null) {\n    return right;\n  } else {\n    if (right == null) {\n      return left;\n    } else {\n      if (left instanceof cljs.core.RedNode) {\n        if (right instanceof cljs.core.RedNode) {\n          var app = tree_map_append.call(null, left.right, right.left);\n          if (app instanceof cljs.core.RedNode) {\n            return new cljs.core.RedNode(app.key, app.val, new cljs.core.RedNode(left.key, left.val, left.left, app.left, null), new cljs.core.RedNode(right.key, right.val, app.right, right.right, null), null);\n          } else {\n            return new cljs.core.RedNode(left.key, left.val, left.left, new cljs.core.RedNode(right.key, right.val, app, right.right, null), null);\n          }\n        } else {\n          return new cljs.core.RedNode(left.key, left.val, left.left, tree_map_append.call(null, left.right, right), null);\n        }\n      } else {\n        if (right instanceof cljs.core.RedNode) {\n          return new cljs.core.RedNode(right.key, right.val, tree_map_append.call(null, left, right.left), right.right, null);\n        } else {\n          var app = tree_map_append.call(null, left.right, right.left);\n          if (app instanceof cljs.core.RedNode) {\n            return new cljs.core.RedNode(app.key, app.val, new cljs.core.BlackNode(left.key, left.val, left.left, app.left, null), new cljs.core.BlackNode(right.key, right.val, app.right, right.right, null), null);\n          } else {\n            return cljs.core.balance_left_del.call(null, left.key, left.val, left.left, new cljs.core.BlackNode(right.key, right.val, app, right.right, null));\n          }\n        }\n      }\n    }\n  }\n};\ncljs.core.tree_map_remove = function tree_map_remove(comp, tree, k, found) {\n  if (!(tree == null)) {\n    var c = comp.call(null, k, tree.key);\n    if (c === 0) {\n      found[0] = tree;\n      return cljs.core.tree_map_append.call(null, tree.left, tree.right);\n    } else {\n      if (c &lt; 0) {\n        var del = tree_map_remove.call(null, comp, tree.left, k, found);\n        if (!(del == null) || !(found[0] == null)) {\n          if (tree.left instanceof cljs.core.BlackNode) {\n            return cljs.core.balance_left_del.call(null, tree.key, tree.val, del, tree.right);\n          } else {\n            return new cljs.core.RedNode(tree.key, tree.val, del, tree.right, null);\n          }\n        } else {\n          return null;\n        }\n      } else {\n        var del = tree_map_remove.call(null, comp, tree.right, k, found);\n        if (!(del == null) || !(found[0] == null)) {\n          if (tree.right instanceof cljs.core.BlackNode) {\n            return cljs.core.balance_right_del.call(null, tree.key, tree.val, tree.left, del);\n          } else {\n            return new cljs.core.RedNode(tree.key, tree.val, tree.left, del, null);\n          }\n        } else {\n          return null;\n        }\n      }\n    }\n  } else {\n    return null;\n  }\n};\ncljs.core.tree_map_replace = function tree_map_replace(comp, tree, k, v) {\n  var tk = tree.key;\n  var c = comp.call(null, k, tk);\n  if (c === 0) {\n    return tree.replace(tk, v, tree.left, tree.right);\n  } else {\n    if (c &lt; 0) {\n      return tree.replace(tk, tree.val, tree_map_replace.call(null, comp, tree.left, k, v), tree.right);\n    } else {\n      return tree.replace(tk, tree.val, tree.left, tree_map_replace.call(null, comp, tree.right, k, v));\n    }\n  }\n};\ncljs.core.PersistentTreeMap = function(comp, tree, cnt, meta, __hash) {\n  this.comp = comp;\n  this.tree = tree;\n  this.cnt = cnt;\n  this.meta = meta;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 418776847;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.PersistentTreeMap.cljs$lang$type = true;\ncljs.core.PersistentTreeMap.cljs$lang$ctorStr = &quot;cljs.core/PersistentTreeMap&quot;;\ncljs.core.PersistentTreeMap.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentTreeMap&quot;);\n};\ncljs.core.PersistentTreeMap.prototype.forEach = function(f) {\n  var self__ = this;\n  var coll = this;\n  var seq__22878 = cljs.core.seq.call(null, coll);\n  var chunk__22879 = null;\n  var count__22880 = 0;\n  var i__22881 = 0;\n  while (true) {\n    if (i__22881 &lt; count__22880) {\n      var vec__22882 = cljs.core._nth.call(null, chunk__22879, i__22881);\n      var k = cljs.core.nth.call(null, vec__22882, 0, null);\n      var v = cljs.core.nth.call(null, vec__22882, 1, null);\n      f.call(null, v, k);\n      var G__22884 = seq__22878;\n      var G__22885 = chunk__22879;\n      var G__22886 = count__22880;\n      var G__22887 = i__22881 + 1;\n      seq__22878 = G__22884;\n      chunk__22879 = G__22885;\n      count__22880 = G__22886;\n      i__22881 = G__22887;\n      continue;\n    } else {\n      var temp__4126__auto__ = cljs.core.seq.call(null, seq__22878);\n      if (temp__4126__auto__) {\n        var seq__22878__$1 = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, seq__22878__$1)) {\n          var c__17901__auto__ = cljs.core.chunk_first.call(null, seq__22878__$1);\n          var G__22888 = cljs.core.chunk_rest.call(null, seq__22878__$1);\n          var G__22889 = c__17901__auto__;\n          var G__22890 = cljs.core.count.call(null, c__17901__auto__);\n          var G__22891 = 0;\n          seq__22878 = G__22888;\n          chunk__22879 = G__22889;\n          count__22880 = G__22890;\n          i__22881 = G__22891;\n          continue;\n        } else {\n          var vec__22883 = cljs.core.first.call(null, seq__22878__$1);\n          var k = cljs.core.nth.call(null, vec__22883, 0, null);\n          var v = cljs.core.nth.call(null, vec__22883, 1, null);\n          f.call(null, v, k);\n          var G__22892 = cljs.core.next.call(null, seq__22878__$1);\n          var G__22893 = null;\n          var G__22894 = 0;\n          var G__22895 = 0;\n          seq__22878 = G__22892;\n          chunk__22879 = G__22893;\n          count__22880 = G__22894;\n          i__22881 = G__22895;\n          continue;\n        }\n      } else {\n        return null;\n      }\n    }\n    break;\n  }\n};\ncljs.core.PersistentTreeMap.prototype.get = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.PersistentTreeMap.prototype.entries = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.entries_iterator.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.PersistentTreeMap.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentTreeMap.prototype.keys = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.keys.call(null, coll));\n};\ncljs.core.PersistentTreeMap.prototype.values = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.vals.call(null, coll));\n};\ncljs.core.PersistentTreeMap.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentTreeMap.prototype.entry_at = function(k) {\n  var self__ = this;\n  var coll = this;\n  var t = self__.tree;\n  while (true) {\n    if (!(t == null)) {\n      var c = self__.comp.call(null, k, t.key);\n      if (c === 0) {\n        return t;\n      } else {\n        if (c &lt; 0) {\n          var G__22896 = t.left;\n          t = G__22896;\n          continue;\n        } else {\n          var G__22897 = t.right;\n          t = G__22897;\n          continue;\n        }\n      }\n    } else {\n      return null;\n    }\n    break;\n  }\n};\ncljs.core.PersistentTreeMap.prototype.has = function(k) {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.contains_QMARK_.call(null, coll, k);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ILookup$_lookup$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._lookup.call(null, coll__$1, k, null);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ILookup$_lookup$arity$3 = function(coll, k, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  var n = coll__$1.entry_at(k);\n  if (!(n == null)) {\n    return n.val;\n  } else {\n    return not_found;\n  }\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IKVReduce$_kv_reduce$arity$3 = function(coll, f, init) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (!(self__.tree == null)) {\n    return cljs.core.tree_map_kv_reduce.call(null, self__.tree, f, init);\n  } else {\n    return init;\n  }\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.PersistentTreeMap(self__.comp, self__.tree, self__.cnt, self__.meta, self__.__hash);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.cnt;\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IReversible$_rseq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt &gt; 0) {\n    return cljs.core.create_tree_map_seq.call(null, self__.tree, false, self__.cnt);\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_unordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_map.call(null, coll__$1, other);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.PersistentTreeMap.EMPTY, self__.meta);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IMap$_dissoc$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  var found = [null];\n  var t = cljs.core.tree_map_remove.call(null, self__.comp, self__.tree, k, found);\n  if (t == null) {\n    if (cljs.core.nth.call(null, found, 0) == null) {\n      return coll__$1;\n    } else {\n      return new cljs.core.PersistentTreeMap(self__.comp, null, 0, self__.meta, null);\n    }\n  } else {\n    return new cljs.core.PersistentTreeMap(self__.comp, t.blacken(), self__.cnt - 1, self__.meta, null);\n  }\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IAssociative$_assoc$arity$3 = function(coll, k, v) {\n  var self__ = this;\n  var coll__$1 = this;\n  var found = [null];\n  var t = cljs.core.tree_map_add.call(null, self__.comp, self__.tree, k, v, found);\n  if (t == null) {\n    var found_node = cljs.core.nth.call(null, found, 0);\n    if (cljs.core._EQ_.call(null, v, found_node.val)) {\n      return coll__$1;\n    } else {\n      return new cljs.core.PersistentTreeMap(self__.comp, cljs.core.tree_map_replace.call(null, self__.comp, self__.tree, k, v), self__.cnt, self__.meta, null);\n    }\n  } else {\n    return new cljs.core.PersistentTreeMap(self__.comp, t.blacken(), self__.cnt + 1, self__.meta, null);\n  }\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IAssociative$_contains_key_QMARK_$arity$2 = function(coll, k) {\n  var self__ = this;\n  var coll__$1 = this;\n  return!(coll__$1.entry_at(k) == null);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt &gt; 0) {\n    return cljs.core.create_tree_map_seq.call(null, self__.tree, true, self__.cnt);\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentTreeMap(self__.comp, self__.tree, self__.cnt, meta__$1, self__.__hash);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, entry) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core.vector_QMARK_.call(null, entry)) {\n    return cljs.core._assoc.call(null, coll__$1, cljs.core._nth.call(null, entry, 0), cljs.core._nth.call(null, entry, 1));\n  } else {\n    var ret = coll__$1;\n    var es = cljs.core.seq.call(null, entry);\n    while (true) {\n      if (es == null) {\n        return ret;\n      } else {\n        var e = cljs.core.first.call(null, es);\n        if (cljs.core.vector_QMARK_.call(null, e)) {\n          var G__22898 = cljs.core._assoc.call(null, ret, cljs.core._nth.call(null, e, 0), cljs.core._nth.call(null, e, 1));\n          var G__22899 = cljs.core.next.call(null, es);\n          ret = G__22898;\n          es = G__22899;\n          continue;\n        } else {\n          throw new Error(&quot;conj on a map takes map entries or seqables of map entries&quot;);\n        }\n      }\n      break;\n    }\n  }\n};\ncljs.core.PersistentTreeMap.prototype.call = function() {\n  var G__22900 = null;\n  var G__22900__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n  };\n  var G__22900__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n  };\n  G__22900 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22900__2.call(this, self__, k);\n      case 3:\n        return G__22900__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22900.cljs$core$IFn$_invoke$arity$2 = G__22900__2;\n  G__22900.cljs$core$IFn$_invoke$arity$3 = G__22900__3;\n  return G__22900;\n}();\ncljs.core.PersistentTreeMap.prototype.apply = function(self__, args22877) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22877)));\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ISorted$_sorted_seq$arity$2 = function(coll, ascending_QMARK_) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt &gt; 0) {\n    return cljs.core.create_tree_map_seq.call(null, self__.tree, ascending_QMARK_, self__.cnt);\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ISorted$_sorted_seq_from$arity$3 = function(coll, k, ascending_QMARK_) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (self__.cnt &gt; 0) {\n    var stack = null;\n    var t = self__.tree;\n    while (true) {\n      if (!(t == null)) {\n        var c = self__.comp.call(null, k, t.key);\n        if (c === 0) {\n          return new cljs.core.PersistentTreeMapSeq(null, cljs.core.conj.call(null, stack, t), ascending_QMARK_, -1, null);\n        } else {\n          if (cljs.core.truth_(ascending_QMARK_)) {\n            if (c &lt; 0) {\n              var G__22901 = cljs.core.conj.call(null, stack, t);\n              var G__22902 = t.left;\n              stack = G__22901;\n              t = G__22902;\n              continue;\n            } else {\n              var G__22903 = stack;\n              var G__22904 = t.right;\n              stack = G__22903;\n              t = G__22904;\n              continue;\n            }\n          } else {\n            if (c &gt; 0) {\n              var G__22905 = cljs.core.conj.call(null, stack, t);\n              var G__22906 = t.right;\n              stack = G__22905;\n              t = G__22906;\n              continue;\n            } else {\n              var G__22907 = stack;\n              var G__22908 = t.left;\n              stack = G__22907;\n              t = G__22908;\n              continue;\n            }\n          }\n        }\n      } else {\n        if (stack == null) {\n          return null;\n        } else {\n          return new cljs.core.PersistentTreeMapSeq(null, stack, ascending_QMARK_, -1, null);\n        }\n      }\n      break;\n    }\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ISorted$_entry_key$arity$2 = function(coll, entry) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.key.call(null, entry);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$ISorted$_comparator$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.comp;\n};\ncljs.core.__GT_PersistentTreeMap = function __GT_PersistentTreeMap(comp, tree, cnt, meta, __hash) {\n  return new cljs.core.PersistentTreeMap(comp, tree, cnt, meta, __hash);\n};\ncljs.core.PersistentTreeMap.EMPTY = new cljs.core.PersistentTreeMap(cljs.core.compare, null, 0, null, 0);\ncljs.core.hash_map = function() {\n  var hash_map__delegate = function(keyvals) {\n    var in$ = cljs.core.seq.call(null, keyvals);\n    var out = cljs.core.transient$.call(null, cljs.core.PersistentHashMap.EMPTY);\n    while (true) {\n      if (in$) {\n        var G__22909 = cljs.core.nnext.call(null, in$);\n        var G__22910 = cljs.core.assoc_BANG_.call(null, out, cljs.core.first.call(null, in$), cljs.core.second.call(null, in$));\n        in$ = G__22909;\n        out = G__22910;\n        continue;\n      } else {\n        return cljs.core.persistent_BANG_.call(null, out);\n      }\n      break;\n    }\n  };\n  var hash_map = function(var_args) {\n    var keyvals = null;\n    if (arguments.length &gt; 0) {\n      keyvals = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return hash_map__delegate.call(this, keyvals);\n  };\n  hash_map.cljs$lang$maxFixedArity = 0;\n  hash_map.cljs$lang$applyTo = function(arglist__22911) {\n    var keyvals = cljs.core.seq(arglist__22911);\n    return hash_map__delegate(keyvals);\n  };\n  hash_map.cljs$core$IFn$_invoke$arity$variadic = hash_map__delegate;\n  return hash_map;\n}();\ncljs.core.array_map = function() {\n  var array_map__delegate = function(keyvals) {\n    return new cljs.core.PersistentArrayMap(null, cljs.core.quot.call(null, cljs.core.count.call(null, keyvals), 2), cljs.core.apply.call(null, cljs.core.array, keyvals), null);\n  };\n  var array_map = function(var_args) {\n    var keyvals = null;\n    if (arguments.length &gt; 0) {\n      keyvals = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return array_map__delegate.call(this, keyvals);\n  };\n  array_map.cljs$lang$maxFixedArity = 0;\n  array_map.cljs$lang$applyTo = function(arglist__22912) {\n    var keyvals = cljs.core.seq(arglist__22912);\n    return array_map__delegate(keyvals);\n  };\n  array_map.cljs$core$IFn$_invoke$arity$variadic = array_map__delegate;\n  return array_map;\n}();\ncljs.core.obj_map = function() {\n  var obj_map__delegate = function(keyvals) {\n    var ks = [];\n    var obj = function() {\n      var obj22916 = {};\n      return obj22916;\n    }();\n    var kvs = cljs.core.seq.call(null, keyvals);\n    while (true) {\n      if (kvs) {\n        ks.push(cljs.core.first.call(null, kvs));\n        obj[cljs.core.first.call(null, kvs)] = cljs.core.second.call(null, kvs);\n        var G__22917 = cljs.core.nnext.call(null, kvs);\n        kvs = G__22917;\n        continue;\n      } else {\n        return cljs.core.ObjMap.fromObject(ks, obj);\n      }\n      break;\n    }\n  };\n  var obj_map = function(var_args) {\n    var keyvals = null;\n    if (arguments.length &gt; 0) {\n      keyvals = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return obj_map__delegate.call(this, keyvals);\n  };\n  obj_map.cljs$lang$maxFixedArity = 0;\n  obj_map.cljs$lang$applyTo = function(arglist__22918) {\n    var keyvals = cljs.core.seq(arglist__22918);\n    return obj_map__delegate(keyvals);\n  };\n  obj_map.cljs$core$IFn$_invoke$arity$variadic = obj_map__delegate;\n  return obj_map;\n}();\ncljs.core.sorted_map = function() {\n  var sorted_map__delegate = function(keyvals) {\n    var in$ = cljs.core.seq.call(null, keyvals);\n    var out = cljs.core.PersistentTreeMap.EMPTY;\n    while (true) {\n      if (in$) {\n        var G__22919 = cljs.core.nnext.call(null, in$);\n        var G__22920 = cljs.core.assoc.call(null, out, cljs.core.first.call(null, in$), cljs.core.second.call(null, in$));\n        in$ = G__22919;\n        out = G__22920;\n        continue;\n      } else {\n        return out;\n      }\n      break;\n    }\n  };\n  var sorted_map = function(var_args) {\n    var keyvals = null;\n    if (arguments.length &gt; 0) {\n      keyvals = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return sorted_map__delegate.call(this, keyvals);\n  };\n  sorted_map.cljs$lang$maxFixedArity = 0;\n  sorted_map.cljs$lang$applyTo = function(arglist__22921) {\n    var keyvals = cljs.core.seq(arglist__22921);\n    return sorted_map__delegate(keyvals);\n  };\n  sorted_map.cljs$core$IFn$_invoke$arity$variadic = sorted_map__delegate;\n  return sorted_map;\n}();\ncljs.core.sorted_map_by = function() {\n  var sorted_map_by__delegate = function(comparator, keyvals) {\n    var in$ = cljs.core.seq.call(null, keyvals);\n    var out = new cljs.core.PersistentTreeMap(cljs.core.fn__GT_comparator.call(null, comparator), null, 0, null, 0);\n    while (true) {\n      if (in$) {\n        var G__22922 = cljs.core.nnext.call(null, in$);\n        var G__22923 = cljs.core.assoc.call(null, out, cljs.core.first.call(null, in$), cljs.core.second.call(null, in$));\n        in$ = G__22922;\n        out = G__22923;\n        continue;\n      } else {\n        return out;\n      }\n      break;\n    }\n  };\n  var sorted_map_by = function(comparator, var_args) {\n    var keyvals = null;\n    if (arguments.length &gt; 1) {\n      keyvals = cljs.core.array_seq(Array.prototype.slice.call(arguments, 1), 0);\n    }\n    return sorted_map_by__delegate.call(this, comparator, keyvals);\n  };\n  sorted_map_by.cljs$lang$maxFixedArity = 1;\n  sorted_map_by.cljs$lang$applyTo = function(arglist__22924) {\n    var comparator = cljs.core.first(arglist__22924);\n    var keyvals = cljs.core.rest(arglist__22924);\n    return sorted_map_by__delegate(comparator, keyvals);\n  };\n  sorted_map_by.cljs$core$IFn$_invoke$arity$variadic = sorted_map_by__delegate;\n  return sorted_map_by;\n}();\ncljs.core.KeySeq = function(mseq, _meta) {\n  this.mseq = mseq;\n  this._meta = _meta;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32374988;\n};\ncljs.core.KeySeq.cljs$lang$type = true;\ncljs.core.KeySeq.cljs$lang$ctorStr = &quot;cljs.core/KeySeq&quot;;\ncljs.core.KeySeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/KeySeq&quot;);\n};\ncljs.core.KeySeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.KeySeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.KeySeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__._meta;\n};\ncljs.core.KeySeq.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var nseq = function() {\n    var G__22925 = self__.mseq;\n    if (G__22925) {\n      var bit__17795__auto__ = G__22925.cljs$lang$protocol_mask$partition0$ &amp; 128;\n      if (bit__17795__auto__ || G__22925.cljs$core$INext$) {\n        return true;\n      } else {\n        if (!G__22925.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.INext, G__22925);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.INext, G__22925);\n    }\n  }() ? cljs.core._next.call(null, self__.mseq) : cljs.core.next.call(null, self__.mseq);\n  if (nseq == null) {\n    return null;\n  } else {\n    return new cljs.core.KeySeq(nseq, self__._meta);\n  }\n};\ncljs.core.KeySeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.hash_ordered_coll.call(null, coll__$1);\n};\ncljs.core.KeySeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.KeySeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__._meta);\n};\ncljs.core.KeySeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.KeySeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.KeySeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var me = cljs.core._first.call(null, self__.mseq);\n  return cljs.core._key.call(null, me);\n};\ncljs.core.KeySeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var nseq = function() {\n    var G__22926 = self__.mseq;\n    if (G__22926) {\n      var bit__17795__auto__ = G__22926.cljs$lang$protocol_mask$partition0$ &amp; 128;\n      if (bit__17795__auto__ || G__22926.cljs$core$INext$) {\n        return true;\n      } else {\n        if (!G__22926.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.INext, G__22926);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.INext, G__22926);\n    }\n  }() ? cljs.core._next.call(null, self__.mseq) : cljs.core.next.call(null, self__.mseq);\n  if (!(nseq == null)) {\n    return new cljs.core.KeySeq(nseq, self__._meta);\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.KeySeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.KeySeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, new_meta) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.KeySeq(self__.mseq, new_meta);\n};\ncljs.core.KeySeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_KeySeq = function __GT_KeySeq(mseq, _meta) {\n  return new cljs.core.KeySeq(mseq, _meta);\n};\ncljs.core.keys = function keys(hash_map) {\n  var temp__4126__auto__ = cljs.core.seq.call(null, hash_map);\n  if (temp__4126__auto__) {\n    var mseq = temp__4126__auto__;\n    return new cljs.core.KeySeq(mseq, null);\n  } else {\n    return null;\n  }\n};\ncljs.core.key = function key(map_entry) {\n  return cljs.core._key.call(null, map_entry);\n};\ncljs.core.ValSeq = function(mseq, _meta) {\n  this.mseq = mseq;\n  this._meta = _meta;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 32374988;\n};\ncljs.core.ValSeq.cljs$lang$type = true;\ncljs.core.ValSeq.cljs$lang$ctorStr = &quot;cljs.core/ValSeq&quot;;\ncljs.core.ValSeq.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ValSeq&quot;);\n};\ncljs.core.ValSeq.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.ValSeq.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.ValSeq.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__._meta;\n};\ncljs.core.ValSeq.prototype.cljs$core$INext$_next$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var nseq = function() {\n    var G__22927 = self__.mseq;\n    if (G__22927) {\n      var bit__17795__auto__ = G__22927.cljs$lang$protocol_mask$partition0$ &amp; 128;\n      if (bit__17795__auto__ || G__22927.cljs$core$INext$) {\n        return true;\n      } else {\n        if (!G__22927.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.INext, G__22927);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.INext, G__22927);\n    }\n  }() ? cljs.core._next.call(null, self__.mseq) : cljs.core.next.call(null, self__.mseq);\n  if (nseq == null) {\n    return null;\n  } else {\n    return new cljs.core.ValSeq(nseq, self__._meta);\n  }\n};\ncljs.core.ValSeq.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.hash_ordered_coll.call(null, coll__$1);\n};\ncljs.core.ValSeq.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.equiv_sequential.call(null, coll__$1, other);\n};\ncljs.core.ValSeq.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__._meta);\n};\ncljs.core.ValSeq.prototype.cljs$core$IReduce$_reduce$arity$2 = function(coll, f) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, coll__$1);\n};\ncljs.core.ValSeq.prototype.cljs$core$IReduce$_reduce$arity$3 = function(coll, f, start) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.seq_reduce.call(null, f, start, coll__$1);\n};\ncljs.core.ValSeq.prototype.cljs$core$ISeq$_first$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var me = cljs.core._first.call(null, self__.mseq);\n  return cljs.core._val.call(null, me);\n};\ncljs.core.ValSeq.prototype.cljs$core$ISeq$_rest$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var nseq = function() {\n    var G__22928 = self__.mseq;\n    if (G__22928) {\n      var bit__17795__auto__ = G__22928.cljs$lang$protocol_mask$partition0$ &amp; 128;\n      if (bit__17795__auto__ || G__22928.cljs$core$INext$) {\n        return true;\n      } else {\n        if (!G__22928.cljs$lang$protocol_mask$partition0$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.INext, G__22928);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.INext, G__22928);\n    }\n  }() ? cljs.core._next.call(null, self__.mseq) : cljs.core.next.call(null, self__.mseq);\n  if (!(nseq == null)) {\n    return new cljs.core.ValSeq(nseq, self__._meta);\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.ValSeq.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return coll__$1;\n};\ncljs.core.ValSeq.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, new_meta) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.ValSeq(self__.mseq, new_meta);\n};\ncljs.core.ValSeq.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.cons.call(null, o, coll__$1);\n};\ncljs.core.__GT_ValSeq = function __GT_ValSeq(mseq, _meta) {\n  return new cljs.core.ValSeq(mseq, _meta);\n};\ncljs.core.vals = function vals(hash_map) {\n  var temp__4126__auto__ = cljs.core.seq.call(null, hash_map);\n  if (temp__4126__auto__) {\n    var mseq = temp__4126__auto__;\n    return new cljs.core.ValSeq(mseq, null);\n  } else {\n    return null;\n  }\n};\ncljs.core.val = function val(map_entry) {\n  return cljs.core._val.call(null, map_entry);\n};\ncljs.core.merge = function() {\n  var merge__delegate = function(maps) {\n    if (cljs.core.truth_(cljs.core.some.call(null, cljs.core.identity, maps))) {\n      return cljs.core.reduce.call(null, function(p1__22929_SHARP_, p2__22930_SHARP_) {\n        return cljs.core.conj.call(null, function() {\n          var or__17145__auto__ = p1__22929_SHARP_;\n          if (cljs.core.truth_(or__17145__auto__)) {\n            return or__17145__auto__;\n          } else {\n            return cljs.core.PersistentArrayMap.EMPTY;\n          }\n        }(), p2__22930_SHARP_);\n      }, maps);\n    } else {\n      return null;\n    }\n  };\n  var merge = function(var_args) {\n    var maps = null;\n    if (arguments.length &gt; 0) {\n      maps = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return merge__delegate.call(this, maps);\n  };\n  merge.cljs$lang$maxFixedArity = 0;\n  merge.cljs$lang$applyTo = function(arglist__22931) {\n    var maps = cljs.core.seq(arglist__22931);\n    return merge__delegate(maps);\n  };\n  merge.cljs$core$IFn$_invoke$arity$variadic = merge__delegate;\n  return merge;\n}();\ncljs.core.merge_with = function() {\n  var merge_with__delegate = function(f, maps) {\n    if (cljs.core.truth_(cljs.core.some.call(null, cljs.core.identity, maps))) {\n      var merge_entry = function(m, e) {\n        var k = cljs.core.first.call(null, e);\n        var v = cljs.core.second.call(null, e);\n        if (cljs.core.contains_QMARK_.call(null, m, k)) {\n          return cljs.core.assoc.call(null, m, k, f.call(null, cljs.core.get.call(null, m, k), v));\n        } else {\n          return cljs.core.assoc.call(null, m, k, v);\n        }\n      };\n      var merge2 = function(merge_entry) {\n        return function(m1, m2) {\n          return cljs.core.reduce.call(null, merge_entry, function() {\n            var or__17145__auto__ = m1;\n            if (cljs.core.truth_(or__17145__auto__)) {\n              return or__17145__auto__;\n            } else {\n              return cljs.core.PersistentArrayMap.EMPTY;\n            }\n          }(), cljs.core.seq.call(null, m2));\n        };\n      }(merge_entry);\n      return cljs.core.reduce.call(null, merge2, maps);\n    } else {\n      return null;\n    }\n  };\n  var merge_with = function(f, var_args) {\n    var maps = null;\n    if (arguments.length &gt; 1) {\n      maps = cljs.core.array_seq(Array.prototype.slice.call(arguments, 1), 0);\n    }\n    return merge_with__delegate.call(this, f, maps);\n  };\n  merge_with.cljs$lang$maxFixedArity = 1;\n  merge_with.cljs$lang$applyTo = function(arglist__22932) {\n    var f = cljs.core.first(arglist__22932);\n    var maps = cljs.core.rest(arglist__22932);\n    return merge_with__delegate(f, maps);\n  };\n  merge_with.cljs$core$IFn$_invoke$arity$variadic = merge_with__delegate;\n  return merge_with;\n}();\ncljs.core.select_keys = function select_keys(map, keyseq) {\n  var ret = cljs.core.PersistentArrayMap.EMPTY;\n  var keys = cljs.core.seq.call(null, keyseq);\n  while (true) {\n    if (keys) {\n      var key = cljs.core.first.call(null, keys);\n      var entry = cljs.core.get.call(null, map, key, new cljs.core.Keyword(&quot;cljs.core&quot;, &quot;not-found&quot;, &quot;cljs.core/not-found&quot;, -1572889185));\n      var G__22933 = cljs.core.not_EQ_.call(null, entry, new cljs.core.Keyword(&quot;cljs.core&quot;, &quot;not-found&quot;, &quot;cljs.core/not-found&quot;, -1572889185)) ? cljs.core.assoc.call(null, ret, key, entry) : ret;\n      var G__22934 = cljs.core.next.call(null, keys);\n      ret = G__22933;\n      keys = G__22934;\n      continue;\n    } else {\n      return ret;\n    }\n    break;\n  }\n};\ncljs.core.PersistentHashSet = function(meta, hash_map, __hash) {\n  this.meta = meta;\n  this.hash_map = hash_map;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 15077647;\n  this.cljs$lang$protocol_mask$partition1$ = 8196;\n};\ncljs.core.PersistentHashSet.cljs$lang$type = true;\ncljs.core.PersistentHashSet.cljs$lang$ctorStr = &quot;cljs.core/PersistentHashSet&quot;;\ncljs.core.PersistentHashSet.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentHashSet&quot;);\n};\ncljs.core.PersistentHashSet.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentHashSet.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentHashSet.prototype.keys = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.PersistentHashSet.prototype.entries = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.set_entries_iterator.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.PersistentHashSet.prototype.values = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.PersistentHashSet.prototype.has = function(k) {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.contains_QMARK_.call(null, coll, k);\n};\ncljs.core.PersistentHashSet.prototype.forEach = function(f) {\n  var self__ = this;\n  var coll = this;\n  var seq__22937 = cljs.core.seq.call(null, coll);\n  var chunk__22938 = null;\n  var count__22939 = 0;\n  var i__22940 = 0;\n  while (true) {\n    if (i__22940 &lt; count__22939) {\n      var vec__22941 = cljs.core._nth.call(null, chunk__22938, i__22940);\n      var k = cljs.core.nth.call(null, vec__22941, 0, null);\n      var v = cljs.core.nth.call(null, vec__22941, 1, null);\n      f.call(null, v, k);\n      var G__22943 = seq__22937;\n      var G__22944 = chunk__22938;\n      var G__22945 = count__22939;\n      var G__22946 = i__22940 + 1;\n      seq__22937 = G__22943;\n      chunk__22938 = G__22944;\n      count__22939 = G__22945;\n      i__22940 = G__22946;\n      continue;\n    } else {\n      var temp__4126__auto__ = cljs.core.seq.call(null, seq__22937);\n      if (temp__4126__auto__) {\n        var seq__22937__$1 = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, seq__22937__$1)) {\n          var c__17901__auto__ = cljs.core.chunk_first.call(null, seq__22937__$1);\n          var G__22947 = cljs.core.chunk_rest.call(null, seq__22937__$1);\n          var G__22948 = c__17901__auto__;\n          var G__22949 = cljs.core.count.call(null, c__17901__auto__);\n          var G__22950 = 0;\n          seq__22937 = G__22947;\n          chunk__22938 = G__22948;\n          count__22939 = G__22949;\n          i__22940 = G__22950;\n          continue;\n        } else {\n          var vec__22942 = cljs.core.first.call(null, seq__22937__$1);\n          var k = cljs.core.nth.call(null, vec__22942, 0, null);\n          var v = cljs.core.nth.call(null, vec__22942, 1, null);\n          f.call(null, v, k);\n          var G__22951 = cljs.core.next.call(null, seq__22937__$1);\n          var G__22952 = null;\n          var G__22953 = 0;\n          var G__22954 = 0;\n          seq__22937 = G__22951;\n          chunk__22938 = G__22952;\n          count__22939 = G__22953;\n          i__22940 = G__22954;\n          continue;\n        }\n      } else {\n        return null;\n      }\n    }\n    break;\n  }\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$ILookup$_lookup$arity$2 = function(coll, v) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._lookup.call(null, coll__$1, v, null);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$ILookup$_lookup$arity$3 = function(coll, v, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core._contains_key_QMARK_.call(null, self__.hash_map, v)) {\n    return v;\n  } else {\n    return not_found;\n  }\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.PersistentHashSet(self__.meta, self__.hash_map, self__.__hash);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._count.call(null, self__.hash_map);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_unordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.set_QMARK_.call(null, other) &amp;&amp; cljs.core.count.call(null, coll__$1) === cljs.core.count.call(null, other) &amp;&amp; cljs.core.every_QMARK_.call(null, function(coll__$1) {\n    return function(p1__22935_SHARP_) {\n      return cljs.core.contains_QMARK_.call(null, coll__$1, p1__22935_SHARP_);\n    };\n  }(coll__$1), other);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$IEditableCollection$_as_transient$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.TransientHashSet(cljs.core._as_transient.call(null, self__.hash_map));\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.PersistentHashSet.EMPTY, self__.meta);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$ISet$_disjoin$arity$2 = function(coll, v) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentHashSet(self__.meta, cljs.core._dissoc.call(null, self__.hash_map, v), null);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.keys.call(null, self__.hash_map);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentHashSet(meta__$1, self__.hash_map, self__.__hash);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentHashSet(self__.meta, cljs.core.assoc.call(null, self__.hash_map, o, null), null);\n};\ncljs.core.PersistentHashSet.prototype.call = function() {\n  var G__22955 = null;\n  var G__22955__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n  };\n  var G__22955__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n  };\n  G__22955 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22955__2.call(this, self__, k);\n      case 3:\n        return G__22955__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22955.cljs$core$IFn$_invoke$arity$2 = G__22955__2;\n  G__22955.cljs$core$IFn$_invoke$arity$3 = G__22955__3;\n  return G__22955;\n}();\ncljs.core.PersistentHashSet.prototype.apply = function(self__, args22936) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22936)));\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n};\ncljs.core.__GT_PersistentHashSet = function __GT_PersistentHashSet(meta, hash_map, __hash) {\n  return new cljs.core.PersistentHashSet(meta, hash_map, __hash);\n};\ncljs.core.PersistentHashSet.EMPTY = new cljs.core.PersistentHashSet(null, cljs.core.PersistentArrayMap.EMPTY, 0);\ncljs.core.PersistentHashSet.fromArray = function(items, no_clone) {\n  var len = items.length;\n  if (len &lt;= cljs.core.PersistentArrayMap.HASHMAP_THRESHOLD) {\n    var arr = no_clone ? items : cljs.core.aclone.call(null, items);\n    var i = 0;\n    var out = cljs.core.transient$.call(null, cljs.core.PersistentArrayMap.EMPTY);\n    while (true) {\n      if (i &lt; len) {\n        var G__22956 = i + 1;\n        var G__22957 = cljs.core._assoc_BANG_.call(null, out, items[i], null);\n        i = G__22956;\n        out = G__22957;\n        continue;\n      } else {\n        return new cljs.core.PersistentHashSet(null, cljs.core._persistent_BANG_.call(null, out), null);\n      }\n      break;\n    }\n  } else {\n    var i = 0;\n    var out = cljs.core.transient$.call(null, cljs.core.PersistentHashSet.EMPTY);\n    while (true) {\n      if (i &lt; len) {\n        var G__22958 = i + 1;\n        var G__22959 = cljs.core._conj_BANG_.call(null, out, items[i]);\n        i = G__22958;\n        out = G__22959;\n        continue;\n      } else {\n        return cljs.core._persistent_BANG_.call(null, out);\n      }\n      break;\n    }\n  }\n};\ncljs.core.TransientHashSet = function(transient_map) {\n  this.transient_map = transient_map;\n  this.cljs$lang$protocol_mask$partition0$ = 259;\n  this.cljs$lang$protocol_mask$partition1$ = 136;\n};\ncljs.core.TransientHashSet.cljs$lang$type = true;\ncljs.core.TransientHashSet.cljs$lang$ctorStr = &quot;cljs.core/TransientHashSet&quot;;\ncljs.core.TransientHashSet.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/TransientHashSet&quot;);\n};\ncljs.core.TransientHashSet.prototype.call = function() {\n  var G__22961 = null;\n  var G__22961__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var tcoll = self____$1;\n    if (cljs.core._lookup.call(null, self__.transient_map, k, cljs.core.lookup_sentinel) === cljs.core.lookup_sentinel) {\n      return null;\n    } else {\n      return k;\n    }\n  };\n  var G__22961__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var tcoll = self____$1;\n    if (cljs.core._lookup.call(null, self__.transient_map, k, cljs.core.lookup_sentinel) === cljs.core.lookup_sentinel) {\n      return not_found;\n    } else {\n      return k;\n    }\n  };\n  G__22961 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22961__2.call(this, self__, k);\n      case 3:\n        return G__22961__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22961.cljs$core$IFn$_invoke$arity$2 = G__22961__2;\n  G__22961.cljs$core$IFn$_invoke$arity$3 = G__22961__3;\n  return G__22961;\n}();\ncljs.core.TransientHashSet.prototype.apply = function(self__, args22960) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22960)));\n};\ncljs.core.TransientHashSet.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var tcoll = this;\n  if (cljs.core._lookup.call(null, self__.transient_map, k, cljs.core.lookup_sentinel) === cljs.core.lookup_sentinel) {\n    return null;\n  } else {\n    return k;\n  }\n};\ncljs.core.TransientHashSet.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var tcoll = this;\n  if (cljs.core._lookup.call(null, self__.transient_map, k, cljs.core.lookup_sentinel) === cljs.core.lookup_sentinel) {\n    return not_found;\n  } else {\n    return k;\n  }\n};\ncljs.core.TransientHashSet.prototype.cljs$core$ILookup$_lookup$arity$2 = function(tcoll, v) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  return cljs.core._lookup.call(null, tcoll__$1, v, null);\n};\ncljs.core.TransientHashSet.prototype.cljs$core$ILookup$_lookup$arity$3 = function(tcoll, v, not_found) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  if (cljs.core._lookup.call(null, self__.transient_map, v, cljs.core.lookup_sentinel) === cljs.core.lookup_sentinel) {\n    return not_found;\n  } else {\n    return v;\n  }\n};\ncljs.core.TransientHashSet.prototype.cljs$core$ICounted$_count$arity$1 = function(tcoll) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  return cljs.core.count.call(null, self__.transient_map);\n};\ncljs.core.TransientHashSet.prototype.cljs$core$ITransientSet$_disjoin_BANG_$arity$2 = function(tcoll, v) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  self__.transient_map = cljs.core.dissoc_BANG_.call(null, self__.transient_map, v);\n  return tcoll__$1;\n};\ncljs.core.TransientHashSet.prototype.cljs$core$ITransientCollection$_conj_BANG_$arity$2 = function(tcoll, o) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  self__.transient_map = cljs.core.assoc_BANG_.call(null, self__.transient_map, o, null);\n  return tcoll__$1;\n};\ncljs.core.TransientHashSet.prototype.cljs$core$ITransientCollection$_persistent_BANG_$arity$1 = function(tcoll) {\n  var self__ = this;\n  var tcoll__$1 = this;\n  return new cljs.core.PersistentHashSet(null, cljs.core.persistent_BANG_.call(null, self__.transient_map), null);\n};\ncljs.core.__GT_TransientHashSet = function __GT_TransientHashSet(transient_map) {\n  return new cljs.core.TransientHashSet(transient_map);\n};\ncljs.core.PersistentTreeSet = function(meta, tree_map, __hash) {\n  this.meta = meta;\n  this.tree_map = tree_map;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 417730831;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.PersistentTreeSet.cljs$lang$type = true;\ncljs.core.PersistentTreeSet.cljs$lang$ctorStr = &quot;cljs.core/PersistentTreeSet&quot;;\ncljs.core.PersistentTreeSet.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/PersistentTreeSet&quot;);\n};\ncljs.core.PersistentTreeSet.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.PersistentTreeSet.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.PersistentTreeSet.prototype.keys = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.PersistentTreeSet.prototype.entries = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.set_entries_iterator.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.PersistentTreeSet.prototype.values = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.iterator.call(null, cljs.core.seq.call(null, coll));\n};\ncljs.core.PersistentTreeSet.prototype.has = function(k) {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.contains_QMARK_.call(null, coll, k);\n};\ncljs.core.PersistentTreeSet.prototype.forEach = function(f) {\n  var self__ = this;\n  var coll = this;\n  var seq__22964 = cljs.core.seq.call(null, coll);\n  var chunk__22965 = null;\n  var count__22966 = 0;\n  var i__22967 = 0;\n  while (true) {\n    if (i__22967 &lt; count__22966) {\n      var vec__22968 = cljs.core._nth.call(null, chunk__22965, i__22967);\n      var k = cljs.core.nth.call(null, vec__22968, 0, null);\n      var v = cljs.core.nth.call(null, vec__22968, 1, null);\n      f.call(null, v, k);\n      var G__22970 = seq__22964;\n      var G__22971 = chunk__22965;\n      var G__22972 = count__22966;\n      var G__22973 = i__22967 + 1;\n      seq__22964 = G__22970;\n      chunk__22965 = G__22971;\n      count__22966 = G__22972;\n      i__22967 = G__22973;\n      continue;\n    } else {\n      var temp__4126__auto__ = cljs.core.seq.call(null, seq__22964);\n      if (temp__4126__auto__) {\n        var seq__22964__$1 = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, seq__22964__$1)) {\n          var c__17901__auto__ = cljs.core.chunk_first.call(null, seq__22964__$1);\n          var G__22974 = cljs.core.chunk_rest.call(null, seq__22964__$1);\n          var G__22975 = c__17901__auto__;\n          var G__22976 = cljs.core.count.call(null, c__17901__auto__);\n          var G__22977 = 0;\n          seq__22964 = G__22974;\n          chunk__22965 = G__22975;\n          count__22966 = G__22976;\n          i__22967 = G__22977;\n          continue;\n        } else {\n          var vec__22969 = cljs.core.first.call(null, seq__22964__$1);\n          var k = cljs.core.nth.call(null, vec__22969, 0, null);\n          var v = cljs.core.nth.call(null, vec__22969, 1, null);\n          f.call(null, v, k);\n          var G__22978 = cljs.core.next.call(null, seq__22964__$1);\n          var G__22979 = null;\n          var G__22980 = 0;\n          var G__22981 = 0;\n          seq__22964 = G__22978;\n          chunk__22965 = G__22979;\n          count__22966 = G__22980;\n          i__22967 = G__22981;\n          continue;\n        }\n      } else {\n        return null;\n      }\n    }\n    break;\n  }\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ILookup$_lookup$arity$2 = function(coll, v) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._lookup.call(null, coll__$1, v, null);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ILookup$_lookup$arity$3 = function(coll, v, not_found) {\n  var self__ = this;\n  var coll__$1 = this;\n  var n = self__.tree_map.entry_at(v);\n  if (!(n == null)) {\n    return n.key;\n  } else {\n    return not_found;\n  }\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$IMeta$_meta$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return self__.meta;\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.PersistentTreeSet(self__.meta, self__.tree_map, self__.__hash);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ICounted$_count$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.count.call(null, self__.tree_map);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$IReversible$_rseq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  if (cljs.core.count.call(null, self__.tree_map) &gt; 0) {\n    return cljs.core.map.call(null, cljs.core.key, cljs.core.rseq.call(null, self__.tree_map));\n  } else {\n    return null;\n  }\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$IHash$_hash$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_unordered_coll.call(null, coll__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(coll, other) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.set_QMARK_.call(null, other) &amp;&amp; cljs.core.count.call(null, coll__$1) === cljs.core.count.call(null, other) &amp;&amp; cljs.core.every_QMARK_.call(null, function(coll__$1) {\n    return function(p1__22962_SHARP_) {\n      return cljs.core.contains_QMARK_.call(null, coll__$1, p1__22962_SHARP_);\n    };\n  }(coll__$1), other);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.PersistentTreeSet.EMPTY, self__.meta);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ISet$_disjoin$arity$2 = function(coll, v) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentTreeSet(self__.meta, cljs.core.dissoc.call(null, self__.tree_map, v), null);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ISeqable$_seq$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.keys.call(null, self__.tree_map);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(coll, meta__$1) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentTreeSet(meta__$1, self__.tree_map, self__.__hash);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ICollection$_conj$arity$2 = function(coll, o) {\n  var self__ = this;\n  var coll__$1 = this;\n  return new cljs.core.PersistentTreeSet(self__.meta, cljs.core.assoc.call(null, self__.tree_map, o, null), null);\n};\ncljs.core.PersistentTreeSet.prototype.call = function() {\n  var G__22982 = null;\n  var G__22982__2 = function(self__, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n  };\n  var G__22982__3 = function(self__, k, not_found) {\n    var self__ = this;\n    var self____$1 = this;\n    var coll = self____$1;\n    return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n  };\n  G__22982 = function(self__, k, not_found) {\n    switch(arguments.length) {\n      case 2:\n        return G__22982__2.call(this, self__, k);\n      case 3:\n        return G__22982__3.call(this, self__, k, not_found);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__22982.cljs$core$IFn$_invoke$arity$2 = G__22982__2;\n  G__22982.cljs$core$IFn$_invoke$arity$3 = G__22982__3;\n  return G__22982;\n}();\ncljs.core.PersistentTreeSet.prototype.apply = function(self__, args22963) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args22963)));\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$IFn$_invoke$arity$1 = function(k) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$2(null, k);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$IFn$_invoke$arity$2 = function(k, not_found) {\n  var self__ = this;\n  var coll = this;\n  return coll.cljs$core$ILookup$_lookup$arity$3(null, k, not_found);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ISorted$_sorted_seq$arity$2 = function(coll, ascending_QMARK_) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.map.call(null, cljs.core.key, cljs.core._sorted_seq.call(null, self__.tree_map, ascending_QMARK_));\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ISorted$_sorted_seq_from$arity$3 = function(coll, k, ascending_QMARK_) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core.map.call(null, cljs.core.key, cljs.core._sorted_seq_from.call(null, self__.tree_map, k, ascending_QMARK_));\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ISorted$_entry_key$arity$2 = function(coll, entry) {\n  var self__ = this;\n  var coll__$1 = this;\n  return entry;\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$ISorted$_comparator$arity$1 = function(coll) {\n  var self__ = this;\n  var coll__$1 = this;\n  return cljs.core._comparator.call(null, self__.tree_map);\n};\ncljs.core.__GT_PersistentTreeSet = function __GT_PersistentTreeSet(meta, tree_map, __hash) {\n  return new cljs.core.PersistentTreeSet(meta, tree_map, __hash);\n};\ncljs.core.PersistentTreeSet.EMPTY = new cljs.core.PersistentTreeSet(null, cljs.core.PersistentTreeMap.EMPTY, 0);\ncljs.core.set_from_indexed_seq = function set_from_indexed_seq(iseq) {\n  var arr = iseq.arr;\n  var ret = function() {\n    var a__17995__auto__ = arr;\n    var i = 0;\n    var res = cljs.core._as_transient.call(null, cljs.core.PersistentHashSet.EMPTY);\n    while (true) {\n      if (i &lt; a__17995__auto__.length) {\n        var G__22983 = i + 1;\n        var G__22984 = cljs.core._conj_BANG_.call(null, res, arr[i]);\n        i = G__22983;\n        res = G__22984;\n        continue;\n      } else {\n        return res;\n      }\n      break;\n    }\n  }();\n  return cljs.core._persistent_BANG_.call(null, ret);\n};\ncljs.core.set = function set(coll) {\n  var in$ = cljs.core.seq.call(null, coll);\n  if (in$ == null) {\n    return cljs.core.PersistentHashSet.EMPTY;\n  } else {\n    if (in$ instanceof cljs.core.IndexedSeq &amp;&amp; in$.i === 0) {\n      return cljs.core.set_from_indexed_seq.call(null, in$);\n    } else {\n      var in$__$1 = in$;\n      var out = cljs.core._as_transient.call(null, cljs.core.PersistentHashSet.EMPTY);\n      while (true) {\n        if (!(in$__$1 == null)) {\n          var G__22985 = cljs.core._next.call(null, in$__$1);\n          var G__22986 = cljs.core._conj_BANG_.call(null, out, cljs.core._first.call(null, in$__$1));\n          in$__$1 = G__22985;\n          out = G__22986;\n          continue;\n        } else {\n          return cljs.core._persistent_BANG_.call(null, out);\n        }\n        break;\n      }\n    }\n  }\n};\ncljs.core.hash_set = function() {\n  var hash_set = null;\n  var hash_set__0 = function() {\n    return cljs.core.PersistentHashSet.EMPTY;\n  };\n  var hash_set__1 = function() {\n    var G__22987__delegate = function(keys) {\n      return cljs.core.set.call(null, keys);\n    };\n    var G__22987 = function(var_args) {\n      var keys = null;\n      if (arguments.length &gt; 0) {\n        keys = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n      }\n      return G__22987__delegate.call(this, keys);\n    };\n    G__22987.cljs$lang$maxFixedArity = 0;\n    G__22987.cljs$lang$applyTo = function(arglist__22988) {\n      var keys = cljs.core.seq(arglist__22988);\n      return G__22987__delegate(keys);\n    };\n    G__22987.cljs$core$IFn$_invoke$arity$variadic = G__22987__delegate;\n    return G__22987;\n  }();\n  hash_set = function(var_args) {\n    var keys = var_args;\n    switch(arguments.length) {\n      case 0:\n        return hash_set__0.call(this);\n      default:\n        return hash_set__1.cljs$core$IFn$_invoke$arity$variadic(cljs.core.array_seq(arguments, 0));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  hash_set.cljs$lang$maxFixedArity = 0;\n  hash_set.cljs$lang$applyTo = hash_set__1.cljs$lang$applyTo;\n  hash_set.cljs$core$IFn$_invoke$arity$0 = hash_set__0;\n  hash_set.cljs$core$IFn$_invoke$arity$variadic = hash_set__1.cljs$core$IFn$_invoke$arity$variadic;\n  return hash_set;\n}();\ncljs.core.sorted_set = function() {\n  var sorted_set__delegate = function(keys) {\n    return cljs.core.reduce.call(null, cljs.core._conj, cljs.core.PersistentTreeSet.EMPTY, keys);\n  };\n  var sorted_set = function(var_args) {\n    var keys = null;\n    if (arguments.length &gt; 0) {\n      keys = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return sorted_set__delegate.call(this, keys);\n  };\n  sorted_set.cljs$lang$maxFixedArity = 0;\n  sorted_set.cljs$lang$applyTo = function(arglist__22989) {\n    var keys = cljs.core.seq(arglist__22989);\n    return sorted_set__delegate(keys);\n  };\n  sorted_set.cljs$core$IFn$_invoke$arity$variadic = sorted_set__delegate;\n  return sorted_set;\n}();\ncljs.core.sorted_set_by = function() {\n  var sorted_set_by__delegate = function(comparator, keys) {\n    return cljs.core.reduce.call(null, cljs.core._conj, new cljs.core.PersistentTreeSet(null, cljs.core.sorted_map_by.call(null, comparator), 0), keys);\n  };\n  var sorted_set_by = function(comparator, var_args) {\n    var keys = null;\n    if (arguments.length &gt; 1) {\n      keys = cljs.core.array_seq(Array.prototype.slice.call(arguments, 1), 0);\n    }\n    return sorted_set_by__delegate.call(this, comparator, keys);\n  };\n  sorted_set_by.cljs$lang$maxFixedArity = 1;\n  sorted_set_by.cljs$lang$applyTo = function(arglist__22990) {\n    var comparator = cljs.core.first(arglist__22990);\n    var keys = cljs.core.rest(arglist__22990);\n    return sorted_set_by__delegate(comparator, keys);\n  };\n  sorted_set_by.cljs$core$IFn$_invoke$arity$variadic = sorted_set_by__delegate;\n  return sorted_set_by;\n}();\ncljs.core.replace = function() {\n  var replace = null;\n  var replace__1 = function(smap) {\n    return cljs.core.map.call(null, function(p1__22991_SHARP_) {\n      var temp__4124__auto__ = cljs.core.find.call(null, smap, p1__22991_SHARP_);\n      if (cljs.core.truth_(temp__4124__auto__)) {\n        var e = temp__4124__auto__;\n        return cljs.core.val.call(null, e);\n      } else {\n        return p1__22991_SHARP_;\n      }\n    });\n  };\n  var replace__2 = function(smap, coll) {\n    if (cljs.core.vector_QMARK_.call(null, coll)) {\n      var n = cljs.core.count.call(null, coll);\n      return cljs.core.reduce.call(null, function(n) {\n        return function(v, i) {\n          var temp__4124__auto__ = cljs.core.find.call(null, smap, cljs.core.nth.call(null, v, i));\n          if (cljs.core.truth_(temp__4124__auto__)) {\n            var e = temp__4124__auto__;\n            return cljs.core.assoc.call(null, v, i, cljs.core.second.call(null, e));\n          } else {\n            return v;\n          }\n        };\n      }(n), coll, cljs.core.take.call(null, n, cljs.core.iterate.call(null, cljs.core.inc, 0)));\n    } else {\n      return cljs.core.map.call(null, function(p1__22992_SHARP_) {\n        var temp__4124__auto__ = cljs.core.find.call(null, smap, p1__22992_SHARP_);\n        if (cljs.core.truth_(temp__4124__auto__)) {\n          var e = temp__4124__auto__;\n          return cljs.core.second.call(null, e);\n        } else {\n          return p1__22992_SHARP_;\n        }\n      }, coll);\n    }\n  };\n  replace = function(smap, coll) {\n    switch(arguments.length) {\n      case 1:\n        return replace__1.call(this, smap);\n      case 2:\n        return replace__2.call(this, smap, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  replace.cljs$core$IFn$_invoke$arity$1 = replace__1;\n  replace.cljs$core$IFn$_invoke$arity$2 = replace__2;\n  return replace;\n}();\ncljs.core.distinct = function distinct(coll) {\n  var step = function step(xs, seen) {\n    return new cljs.core.LazySeq(null, function() {\n      return function(p__22999, seen__$1) {\n        while (true) {\n          var vec__23000 = p__22999;\n          var f = cljs.core.nth.call(null, vec__23000, 0, null);\n          var xs__$1 = vec__23000;\n          var temp__4126__auto__ = cljs.core.seq.call(null, xs__$1);\n          if (temp__4126__auto__) {\n            var s = temp__4126__auto__;\n            if (cljs.core.contains_QMARK_.call(null, seen__$1, f)) {\n              var G__23001 = cljs.core.rest.call(null, s);\n              var G__23002 = seen__$1;\n              p__22999 = G__23001;\n              seen__$1 = G__23002;\n              continue;\n            } else {\n              return cljs.core.cons.call(null, f, step.call(null, cljs.core.rest.call(null, s), cljs.core.conj.call(null, seen__$1, f)));\n            }\n          } else {\n            return null;\n          }\n          break;\n        }\n      }.call(null, xs, seen);\n    }, null, null);\n  };\n  return step.call(null, coll, cljs.core.PersistentHashSet.EMPTY);\n};\ncljs.core.butlast = function butlast(s) {\n  var ret = cljs.core.PersistentVector.EMPTY;\n  var s__$1 = s;\n  while (true) {\n    if (cljs.core.next.call(null, s__$1)) {\n      var G__23003 = cljs.core.conj.call(null, ret, cljs.core.first.call(null, s__$1));\n      var G__23004 = cljs.core.next.call(null, s__$1);\n      ret = G__23003;\n      s__$1 = G__23004;\n      continue;\n    } else {\n      return cljs.core.seq.call(null, ret);\n    }\n    break;\n  }\n};\ncljs.core.name = function name(x) {\n  if (function() {\n    var G__23006 = x;\n    if (G__23006) {\n      var bit__17788__auto__ = G__23006.cljs$lang$protocol_mask$partition1$ &amp; 4096;\n      if (bit__17788__auto__ || G__23006.cljs$core$INamed$) {\n        return true;\n      } else {\n        return false;\n      }\n    } else {\n      return false;\n    }\n  }()) {\n    return cljs.core._name.call(null, x);\n  } else {\n    if (typeof x === &quot;string&quot;) {\n      return x;\n    } else {\n      throw new Error(&quot;Doesn&apos;t support name: &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(x));\n    }\n  }\n};\ncljs.core.zipmap = function zipmap(keys, vals) {\n  var map = cljs.core.transient$.call(null, cljs.core.PersistentArrayMap.EMPTY);\n  var ks = cljs.core.seq.call(null, keys);\n  var vs = cljs.core.seq.call(null, vals);\n  while (true) {\n    if (ks &amp;&amp; vs) {\n      var G__23007 = cljs.core.assoc_BANG_.call(null, map, cljs.core.first.call(null, ks), cljs.core.first.call(null, vs));\n      var G__23008 = cljs.core.next.call(null, ks);\n      var G__23009 = cljs.core.next.call(null, vs);\n      map = G__23007;\n      ks = G__23008;\n      vs = G__23009;\n      continue;\n    } else {\n      return cljs.core.persistent_BANG_.call(null, map);\n    }\n    break;\n  }\n};\ncljs.core.max_key = function() {\n  var max_key = null;\n  var max_key__2 = function(k, x) {\n    return x;\n  };\n  var max_key__3 = function(k, x, y) {\n    if (k.call(null, x) &gt; k.call(null, y)) {\n      return x;\n    } else {\n      return y;\n    }\n  };\n  var max_key__4 = function() {\n    var G__23012__delegate = function(k, x, y, more) {\n      return cljs.core.reduce.call(null, function(p1__23010_SHARP_, p2__23011_SHARP_) {\n        return max_key.call(null, k, p1__23010_SHARP_, p2__23011_SHARP_);\n      }, max_key.call(null, k, x, y), more);\n    };\n    var G__23012 = function(k, x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 3) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n      }\n      return G__23012__delegate.call(this, k, x, y, more);\n    };\n    G__23012.cljs$lang$maxFixedArity = 3;\n    G__23012.cljs$lang$applyTo = function(arglist__23013) {\n      var k = cljs.core.first(arglist__23013);\n      arglist__23013 = cljs.core.next(arglist__23013);\n      var x = cljs.core.first(arglist__23013);\n      arglist__23013 = cljs.core.next(arglist__23013);\n      var y = cljs.core.first(arglist__23013);\n      var more = cljs.core.rest(arglist__23013);\n      return G__23012__delegate(k, x, y, more);\n    };\n    G__23012.cljs$core$IFn$_invoke$arity$variadic = G__23012__delegate;\n    return G__23012;\n  }();\n  max_key = function(k, x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 2:\n        return max_key__2.call(this, k, x);\n      case 3:\n        return max_key__3.call(this, k, x, y);\n      default:\n        return max_key__4.cljs$core$IFn$_invoke$arity$variadic(k, x, y, cljs.core.array_seq(arguments, 3));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  max_key.cljs$lang$maxFixedArity = 3;\n  max_key.cljs$lang$applyTo = max_key__4.cljs$lang$applyTo;\n  max_key.cljs$core$IFn$_invoke$arity$2 = max_key__2;\n  max_key.cljs$core$IFn$_invoke$arity$3 = max_key__3;\n  max_key.cljs$core$IFn$_invoke$arity$variadic = max_key__4.cljs$core$IFn$_invoke$arity$variadic;\n  return max_key;\n}();\ncljs.core.min_key = function() {\n  var min_key = null;\n  var min_key__2 = function(k, x) {\n    return x;\n  };\n  var min_key__3 = function(k, x, y) {\n    if (k.call(null, x) &lt; k.call(null, y)) {\n      return x;\n    } else {\n      return y;\n    }\n  };\n  var min_key__4 = function() {\n    var G__23016__delegate = function(k, x, y, more) {\n      return cljs.core.reduce.call(null, function(p1__23014_SHARP_, p2__23015_SHARP_) {\n        return min_key.call(null, k, p1__23014_SHARP_, p2__23015_SHARP_);\n      }, min_key.call(null, k, x, y), more);\n    };\n    var G__23016 = function(k, x, y, var_args) {\n      var more = null;\n      if (arguments.length &gt; 3) {\n        more = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n      }\n      return G__23016__delegate.call(this, k, x, y, more);\n    };\n    G__23016.cljs$lang$maxFixedArity = 3;\n    G__23016.cljs$lang$applyTo = function(arglist__23017) {\n      var k = cljs.core.first(arglist__23017);\n      arglist__23017 = cljs.core.next(arglist__23017);\n      var x = cljs.core.first(arglist__23017);\n      arglist__23017 = cljs.core.next(arglist__23017);\n      var y = cljs.core.first(arglist__23017);\n      var more = cljs.core.rest(arglist__23017);\n      return G__23016__delegate(k, x, y, more);\n    };\n    G__23016.cljs$core$IFn$_invoke$arity$variadic = G__23016__delegate;\n    return G__23016;\n  }();\n  min_key = function(k, x, y, var_args) {\n    var more = var_args;\n    switch(arguments.length) {\n      case 2:\n        return min_key__2.call(this, k, x);\n      case 3:\n        return min_key__3.call(this, k, x, y);\n      default:\n        return min_key__4.cljs$core$IFn$_invoke$arity$variadic(k, x, y, cljs.core.array_seq(arguments, 3));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  min_key.cljs$lang$maxFixedArity = 3;\n  min_key.cljs$lang$applyTo = min_key__4.cljs$lang$applyTo;\n  min_key.cljs$core$IFn$_invoke$arity$2 = min_key__2;\n  min_key.cljs$core$IFn$_invoke$arity$3 = min_key__3;\n  min_key.cljs$core$IFn$_invoke$arity$variadic = min_key__4.cljs$core$IFn$_invoke$arity$variadic;\n  return min_key;\n}();\ncljs.core.ArrayList = function(arr) {\n  this.arr = arr;\n};\ncljs.core.ArrayList.cljs$lang$type = true;\ncljs.core.ArrayList.cljs$lang$ctorStr = &quot;cljs.core/ArrayList&quot;;\ncljs.core.ArrayList.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/ArrayList&quot;);\n};\ncljs.core.ArrayList.prototype.add = function(x) {\n  var self__ = this;\n  var _ = this;\n  return self__.arr.push(x);\n};\ncljs.core.ArrayList.prototype.size = function() {\n  var self__ = this;\n  var _ = this;\n  return self__.arr.length;\n};\ncljs.core.ArrayList.prototype.clear = function() {\n  var self__ = this;\n  var _ = this;\n  return self__.arr = [];\n};\ncljs.core.ArrayList.prototype.isEmpty = function() {\n  var self__ = this;\n  var _ = this;\n  return self__.arr.length === 0;\n};\ncljs.core.ArrayList.prototype.toArray = function() {\n  var self__ = this;\n  var _ = this;\n  return self__.arr;\n};\ncljs.core.__GT_ArrayList = function __GT_ArrayList(arr) {\n  return new cljs.core.ArrayList(arr);\n};\ncljs.core.array_list = function array_list() {\n  return new cljs.core.ArrayList([]);\n};\ncljs.core.partition_all = function() {\n  var partition_all = null;\n  var partition_all__1 = function(n) {\n    return function(f1) {\n      var a = cljs.core.array_list.call(null);\n      return function(a) {\n        return function() {\n          var G__23018 = null;\n          var G__23018__0 = function() {\n            return f1.call(null);\n          };\n          var G__23018__1 = function(result) {\n            var result__$1 = cljs.core.truth_(a.isEmpty()) ? result : function() {\n              var v = cljs.core.vec.call(null, a.toArray());\n              a.clear();\n              return f1.call(null, result, v);\n            }();\n            return f1.call(null, result__$1);\n          };\n          var G__23018__2 = function(result, input) {\n            a.add(input);\n            if (n === a.size()) {\n              var v = cljs.core.vec.call(null, a.toArray());\n              a.clear();\n              return f1.call(null, result, v);\n            } else {\n              return result;\n            }\n          };\n          G__23018 = function(result, input) {\n            switch(arguments.length) {\n              case 0:\n                return G__23018__0.call(this);\n              case 1:\n                return G__23018__1.call(this, result);\n              case 2:\n                return G__23018__2.call(this, result, input);\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          G__23018.cljs$core$IFn$_invoke$arity$0 = G__23018__0;\n          G__23018.cljs$core$IFn$_invoke$arity$1 = G__23018__1;\n          G__23018.cljs$core$IFn$_invoke$arity$2 = G__23018__2;\n          return G__23018;\n        }();\n      }(a);\n    };\n  };\n  var partition_all__2 = function(n, coll) {\n    return partition_all.call(null, n, n, coll);\n  };\n  var partition_all__3 = function(n, step, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        return cljs.core.cons.call(null, cljs.core.take.call(null, n, s), partition_all.call(null, n, step, cljs.core.drop.call(null, step, s)));\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  partition_all = function(n, step, coll) {\n    switch(arguments.length) {\n      case 1:\n        return partition_all__1.call(this, n);\n      case 2:\n        return partition_all__2.call(this, n, step);\n      case 3:\n        return partition_all__3.call(this, n, step, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  partition_all.cljs$core$IFn$_invoke$arity$1 = partition_all__1;\n  partition_all.cljs$core$IFn$_invoke$arity$2 = partition_all__2;\n  partition_all.cljs$core$IFn$_invoke$arity$3 = partition_all__3;\n  return partition_all;\n}();\ncljs.core.take_while = function() {\n  var take_while = null;\n  var take_while__1 = function(pred) {\n    return function(f1) {\n      return function() {\n        var G__23019 = null;\n        var G__23019__0 = function() {\n          return f1.call(null);\n        };\n        var G__23019__1 = function(result) {\n          return f1.call(null, result);\n        };\n        var G__23019__2 = function(result, input) {\n          if (cljs.core.truth_(pred.call(null, input))) {\n            return f1.call(null, result, input);\n          } else {\n            return cljs.core.reduced.call(null, result);\n          }\n        };\n        G__23019 = function(result, input) {\n          switch(arguments.length) {\n            case 0:\n              return G__23019__0.call(this);\n            case 1:\n              return G__23019__1.call(this, result);\n            case 2:\n              return G__23019__2.call(this, result, input);\n          }\n          throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n        };\n        G__23019.cljs$core$IFn$_invoke$arity$0 = G__23019__0;\n        G__23019.cljs$core$IFn$_invoke$arity$1 = G__23019__1;\n        G__23019.cljs$core$IFn$_invoke$arity$2 = G__23019__2;\n        return G__23019;\n      }();\n    };\n  };\n  var take_while__2 = function(pred, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        if (cljs.core.truth_(pred.call(null, cljs.core.first.call(null, s)))) {\n          return cljs.core.cons.call(null, cljs.core.first.call(null, s), take_while.call(null, pred, cljs.core.rest.call(null, s)));\n        } else {\n          return null;\n        }\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  take_while = function(pred, coll) {\n    switch(arguments.length) {\n      case 1:\n        return take_while__1.call(this, pred);\n      case 2:\n        return take_while__2.call(this, pred, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  take_while.cljs$core$IFn$_invoke$arity$1 = take_while__1;\n  take_while.cljs$core$IFn$_invoke$arity$2 = take_while__2;\n  return take_while;\n}();\ncljs.core.mk_bound_fn = function mk_bound_fn(sc, test, key) {\n  return function(e) {\n    var comp = cljs.core._comparator.call(null, sc);\n    return test.call(null, comp.call(null, cljs.core._entry_key.call(null, sc, e), key), 0);\n  };\n};\ncljs.core.subseq = function() {\n  var subseq = null;\n  var subseq__3 = function(sc, test, key) {\n    var include = cljs.core.mk_bound_fn.call(null, sc, test, key);\n    if (cljs.core.truth_(cljs.core.PersistentHashSet.fromArray([cljs.core._GT_, cljs.core._GT__EQ_], true).call(null, test))) {\n      var temp__4126__auto__ = cljs.core._sorted_seq_from.call(null, sc, key, true);\n      if (cljs.core.truth_(temp__4126__auto__)) {\n        var vec__23022 = temp__4126__auto__;\n        var e = cljs.core.nth.call(null, vec__23022, 0, null);\n        var s = vec__23022;\n        if (cljs.core.truth_(include.call(null, e))) {\n          return s;\n        } else {\n          return cljs.core.next.call(null, s);\n        }\n      } else {\n        return null;\n      }\n    } else {\n      return cljs.core.take_while.call(null, include, cljs.core._sorted_seq.call(null, sc, true));\n    }\n  };\n  var subseq__5 = function(sc, start_test, start_key, end_test, end_key) {\n    var temp__4126__auto__ = cljs.core._sorted_seq_from.call(null, sc, start_key, true);\n    if (cljs.core.truth_(temp__4126__auto__)) {\n      var vec__23023 = temp__4126__auto__;\n      var e = cljs.core.nth.call(null, vec__23023, 0, null);\n      var s = vec__23023;\n      return cljs.core.take_while.call(null, cljs.core.mk_bound_fn.call(null, sc, end_test, end_key), cljs.core.truth_(cljs.core.mk_bound_fn.call(null, sc, start_test, start_key).call(null, e)) ? s : cljs.core.next.call(null, s));\n    } else {\n      return null;\n    }\n  };\n  subseq = function(sc, start_test, start_key, end_test, end_key) {\n    switch(arguments.length) {\n      case 3:\n        return subseq__3.call(this, sc, start_test, start_key);\n      case 5:\n        return subseq__5.call(this, sc, start_test, start_key, end_test, end_key);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  subseq.cljs$core$IFn$_invoke$arity$3 = subseq__3;\n  subseq.cljs$core$IFn$_invoke$arity$5 = subseq__5;\n  return subseq;\n}();\ncljs.core.rsubseq = function() {\n  var rsubseq = null;\n  var rsubseq__3 = function(sc, test, key) {\n    var include = cljs.core.mk_bound_fn.call(null, sc, test, key);\n    if (cljs.core.truth_(cljs.core.PersistentHashSet.fromArray([cljs.core._LT_, cljs.core._LT__EQ_], true).call(null, test))) {\n      var temp__4126__auto__ = cljs.core._sorted_seq_from.call(null, sc, key, false);\n      if (cljs.core.truth_(temp__4126__auto__)) {\n        var vec__23026 = temp__4126__auto__;\n        var e = cljs.core.nth.call(null, vec__23026, 0, null);\n        var s = vec__23026;\n        if (cljs.core.truth_(include.call(null, e))) {\n          return s;\n        } else {\n          return cljs.core.next.call(null, s);\n        }\n      } else {\n        return null;\n      }\n    } else {\n      return cljs.core.take_while.call(null, include, cljs.core._sorted_seq.call(null, sc, false));\n    }\n  };\n  var rsubseq__5 = function(sc, start_test, start_key, end_test, end_key) {\n    var temp__4126__auto__ = cljs.core._sorted_seq_from.call(null, sc, end_key, false);\n    if (cljs.core.truth_(temp__4126__auto__)) {\n      var vec__23027 = temp__4126__auto__;\n      var e = cljs.core.nth.call(null, vec__23027, 0, null);\n      var s = vec__23027;\n      return cljs.core.take_while.call(null, cljs.core.mk_bound_fn.call(null, sc, start_test, start_key), cljs.core.truth_(cljs.core.mk_bound_fn.call(null, sc, end_test, end_key).call(null, e)) ? s : cljs.core.next.call(null, s));\n    } else {\n      return null;\n    }\n  };\n  rsubseq = function(sc, start_test, start_key, end_test, end_key) {\n    switch(arguments.length) {\n      case 3:\n        return rsubseq__3.call(this, sc, start_test, start_key);\n      case 5:\n        return rsubseq__5.call(this, sc, start_test, start_key, end_test, end_key);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  rsubseq.cljs$core$IFn$_invoke$arity$3 = rsubseq__3;\n  rsubseq.cljs$core$IFn$_invoke$arity$5 = rsubseq__5;\n  return rsubseq;\n}();\ncljs.core.Range = function(meta, start, end, step, __hash) {\n  this.meta = meta;\n  this.start = start;\n  this.end = end;\n  this.step = step;\n  this.__hash = __hash;\n  this.cljs$lang$protocol_mask$partition0$ = 32375006;\n  this.cljs$lang$protocol_mask$partition1$ = 8192;\n};\ncljs.core.Range.cljs$lang$type = true;\ncljs.core.Range.cljs$lang$ctorStr = &quot;cljs.core/Range&quot;;\ncljs.core.Range.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Range&quot;);\n};\ncljs.core.Range.prototype.toString = function() {\n  var self__ = this;\n  var coll = this;\n  return cljs.core.pr_str_STAR_.call(null, coll);\n};\ncljs.core.Range.prototype.equiv = function(other) {\n  var self__ = this;\n  var this$ = this;\n  return this$.cljs$core$IEquiv$_equiv$arity$2(null, other);\n};\ncljs.core.Range.prototype.cljs$core$IIndexed$_nth$arity$2 = function(rng, n) {\n  var self__ = this;\n  var rng__$1 = this;\n  if (n &lt; cljs.core._count.call(null, rng__$1)) {\n    return self__.start + n * self__.step;\n  } else {\n    if (self__.start &gt; self__.end &amp;&amp; self__.step === 0) {\n      return self__.start;\n    } else {\n      throw new Error(&quot;Index out of bounds&quot;);\n    }\n  }\n};\ncljs.core.Range.prototype.cljs$core$IIndexed$_nth$arity$3 = function(rng, n, not_found) {\n  var self__ = this;\n  var rng__$1 = this;\n  if (n &lt; cljs.core._count.call(null, rng__$1)) {\n    return self__.start + n * self__.step;\n  } else {\n    if (self__.start &gt; self__.end &amp;&amp; self__.step === 0) {\n      return self__.start;\n    } else {\n      return not_found;\n    }\n  }\n};\ncljs.core.Range.prototype.cljs$core$IMeta$_meta$arity$1 = function(rng) {\n  var self__ = this;\n  var rng__$1 = this;\n  return self__.meta;\n};\ncljs.core.Range.prototype.cljs$core$ICloneable$_clone$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return new cljs.core.Range(self__.meta, self__.start, self__.end, self__.step, self__.__hash);\n};\ncljs.core.Range.prototype.cljs$core$INext$_next$arity$1 = function(rng) {\n  var self__ = this;\n  var rng__$1 = this;\n  if (self__.step &gt; 0) {\n    if (self__.start + self__.step &lt; self__.end) {\n      return new cljs.core.Range(self__.meta, self__.start + self__.step, self__.end, self__.step, null);\n    } else {\n      return null;\n    }\n  } else {\n    if (self__.start + self__.step &gt; self__.end) {\n      return new cljs.core.Range(self__.meta, self__.start + self__.step, self__.end, self__.step, null);\n    } else {\n      return null;\n    }\n  }\n};\ncljs.core.Range.prototype.cljs$core$ICounted$_count$arity$1 = function(rng) {\n  var self__ = this;\n  var rng__$1 = this;\n  if (cljs.core.not.call(null, cljs.core._seq.call(null, rng__$1))) {\n    return 0;\n  } else {\n    return Math.ceil.call(null, (self__.end - self__.start) / self__.step);\n  }\n};\ncljs.core.Range.prototype.cljs$core$IHash$_hash$arity$1 = function(rng) {\n  var self__ = this;\n  var rng__$1 = this;\n  var h__17556__auto__ = self__.__hash;\n  if (!(h__17556__auto__ == null)) {\n    return h__17556__auto__;\n  } else {\n    var h__17556__auto____$1 = cljs.core.hash_ordered_coll.call(null, rng__$1);\n    self__.__hash = h__17556__auto____$1;\n    return h__17556__auto____$1;\n  }\n};\ncljs.core.Range.prototype.cljs$core$IEquiv$_equiv$arity$2 = function(rng, other) {\n  var self__ = this;\n  var rng__$1 = this;\n  return cljs.core.equiv_sequential.call(null, rng__$1, other);\n};\ncljs.core.Range.prototype.cljs$core$IEmptyableCollection$_empty$arity$1 = function(rng) {\n  var self__ = this;\n  var rng__$1 = this;\n  return cljs.core.with_meta.call(null, cljs.core.List.EMPTY, self__.meta);\n};\ncljs.core.Range.prototype.cljs$core$IReduce$_reduce$arity$2 = function(rng, f) {\n  var self__ = this;\n  var rng__$1 = this;\n  return cljs.core.ci_reduce.call(null, rng__$1, f);\n};\ncljs.core.Range.prototype.cljs$core$IReduce$_reduce$arity$3 = function(rng, f, s) {\n  var self__ = this;\n  var rng__$1 = this;\n  return cljs.core.ci_reduce.call(null, rng__$1, f, s);\n};\ncljs.core.Range.prototype.cljs$core$ISeq$_first$arity$1 = function(rng) {\n  var self__ = this;\n  var rng__$1 = this;\n  if (cljs.core._seq.call(null, rng__$1) == null) {\n    return null;\n  } else {\n    return self__.start;\n  }\n};\ncljs.core.Range.prototype.cljs$core$ISeq$_rest$arity$1 = function(rng) {\n  var self__ = this;\n  var rng__$1 = this;\n  if (!(cljs.core._seq.call(null, rng__$1) == null)) {\n    return new cljs.core.Range(self__.meta, self__.start + self__.step, self__.end, self__.step, null);\n  } else {\n    return cljs.core.List.EMPTY;\n  }\n};\ncljs.core.Range.prototype.cljs$core$ISeqable$_seq$arity$1 = function(rng) {\n  var self__ = this;\n  var rng__$1 = this;\n  if (self__.step &gt; 0) {\n    if (self__.start &lt; self__.end) {\n      return rng__$1;\n    } else {\n      return null;\n    }\n  } else {\n    if (self__.start &gt; self__.end) {\n      return rng__$1;\n    } else {\n      return null;\n    }\n  }\n};\ncljs.core.Range.prototype.cljs$core$IWithMeta$_with_meta$arity$2 = function(rng, meta__$1) {\n  var self__ = this;\n  var rng__$1 = this;\n  return new cljs.core.Range(meta__$1, self__.start, self__.end, self__.step, self__.__hash);\n};\ncljs.core.Range.prototype.cljs$core$ICollection$_conj$arity$2 = function(rng, o) {\n  var self__ = this;\n  var rng__$1 = this;\n  return cljs.core.cons.call(null, o, rng__$1);\n};\ncljs.core.__GT_Range = function __GT_Range(meta, start, end, step, __hash) {\n  return new cljs.core.Range(meta, start, end, step, __hash);\n};\ncljs.core.range = function() {\n  var range = null;\n  var range__0 = function() {\n    return range.call(null, 0, Number.MAX_VALUE, 1);\n  };\n  var range__1 = function(end) {\n    return range.call(null, 0, end, 1);\n  };\n  var range__2 = function(start, end) {\n    return range.call(null, start, end, 1);\n  };\n  var range__3 = function(start, end, step) {\n    return new cljs.core.Range(null, start, end, step, null);\n  };\n  range = function(start, end, step) {\n    switch(arguments.length) {\n      case 0:\n        return range__0.call(this);\n      case 1:\n        return range__1.call(this, start);\n      case 2:\n        return range__2.call(this, start, end);\n      case 3:\n        return range__3.call(this, start, end, step);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  range.cljs$core$IFn$_invoke$arity$0 = range__0;\n  range.cljs$core$IFn$_invoke$arity$1 = range__1;\n  range.cljs$core$IFn$_invoke$arity$2 = range__2;\n  range.cljs$core$IFn$_invoke$arity$3 = range__3;\n  return range;\n}();\ncljs.core.take_nth = function() {\n  var take_nth = null;\n  var take_nth__1 = function(n) {\n    return function(f1) {\n      var ia = cljs.core.atom.call(null, -1);\n      return function(ia) {\n        return function() {\n          var G__23028 = null;\n          var G__23028__0 = function() {\n            return f1.call(null);\n          };\n          var G__23028__1 = function(result) {\n            return f1.call(null, result);\n          };\n          var G__23028__2 = function(result, input) {\n            var i = cljs.core.swap_BANG_.call(null, ia, cljs.core.inc);\n            if (cljs.core.rem.call(null, i, n) === 0) {\n              return f1.call(null, result, input);\n            } else {\n              return result;\n            }\n          };\n          G__23028 = function(result, input) {\n            switch(arguments.length) {\n              case 0:\n                return G__23028__0.call(this);\n              case 1:\n                return G__23028__1.call(this, result);\n              case 2:\n                return G__23028__2.call(this, result, input);\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          G__23028.cljs$core$IFn$_invoke$arity$0 = G__23028__0;\n          G__23028.cljs$core$IFn$_invoke$arity$1 = G__23028__1;\n          G__23028.cljs$core$IFn$_invoke$arity$2 = G__23028__2;\n          return G__23028;\n        }();\n      }(ia);\n    };\n  };\n  var take_nth__2 = function(n, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        return cljs.core.cons.call(null, cljs.core.first.call(null, s), take_nth.call(null, n, cljs.core.drop.call(null, n, s)));\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  take_nth = function(n, coll) {\n    switch(arguments.length) {\n      case 1:\n        return take_nth__1.call(this, n);\n      case 2:\n        return take_nth__2.call(this, n, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  take_nth.cljs$core$IFn$_invoke$arity$1 = take_nth__1;\n  take_nth.cljs$core$IFn$_invoke$arity$2 = take_nth__2;\n  return take_nth;\n}();\ncljs.core.split_with = function split_with(pred, coll) {\n  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.take_while.call(null, pred, coll), cljs.core.drop_while.call(null, pred, coll)], null);\n};\ncljs.core.partition_by = function() {\n  var partition_by = null;\n  var partition_by__1 = function(f) {\n    return function(f1) {\n      var a = cljs.core.array_list.call(null);\n      var pa = cljs.core.atom.call(null, new cljs.core.Keyword(&quot;cljs.core&quot;, &quot;none&quot;, &quot;cljs.core/none&quot;, 926646439));\n      return function(a, pa) {\n        return function() {\n          var G__23030 = null;\n          var G__23030__0 = function() {\n            return f1.call(null);\n          };\n          var G__23030__1 = function(result) {\n            var result__$1 = cljs.core.truth_(a.isEmpty()) ? result : function() {\n              var v = cljs.core.vec.call(null, a.toArray());\n              a.clear();\n              return f1.call(null, result, v);\n            }();\n            return f1.call(null, result__$1);\n          };\n          var G__23030__2 = function(result, input) {\n            var pval = cljs.core.deref.call(null, pa);\n            var val = f.call(null, input);\n            cljs.core.reset_BANG_.call(null, pa, val);\n            if (cljs.core.keyword_identical_QMARK_.call(null, pval, new cljs.core.Keyword(&quot;cljs.core&quot;, &quot;none&quot;, &quot;cljs.core/none&quot;, 926646439)) || cljs.core._EQ_.call(null, val, pval)) {\n              a.add(input);\n              return result;\n            } else {\n              var v = cljs.core.vec.call(null, a.toArray());\n              a.clear();\n              a.add(input);\n              return f1.call(null, result, v);\n            }\n          };\n          G__23030 = function(result, input) {\n            switch(arguments.length) {\n              case 0:\n                return G__23030__0.call(this);\n              case 1:\n                return G__23030__1.call(this, result);\n              case 2:\n                return G__23030__2.call(this, result, input);\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          G__23030.cljs$core$IFn$_invoke$arity$0 = G__23030__0;\n          G__23030.cljs$core$IFn$_invoke$arity$1 = G__23030__1;\n          G__23030.cljs$core$IFn$_invoke$arity$2 = G__23030__2;\n          return G__23030;\n        }();\n      }(a, pa);\n    };\n  };\n  var partition_by__2 = function(f, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        var fst = cljs.core.first.call(null, s);\n        var fv = f.call(null, fst);\n        var run = cljs.core.cons.call(null, fst, cljs.core.take_while.call(null, function(fst, fv, s, temp__4126__auto__) {\n          return function(p1__23029_SHARP_) {\n            return cljs.core._EQ_.call(null, fv, f.call(null, p1__23029_SHARP_));\n          };\n        }(fst, fv, s, temp__4126__auto__), cljs.core.next.call(null, s)));\n        return cljs.core.cons.call(null, run, partition_by.call(null, f, cljs.core.seq.call(null, cljs.core.drop.call(null, cljs.core.count.call(null, run), s))));\n      } else {\n        return null;\n      }\n    }, null, null);\n  };\n  partition_by = function(f, coll) {\n    switch(arguments.length) {\n      case 1:\n        return partition_by__1.call(this, f);\n      case 2:\n        return partition_by__2.call(this, f, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  partition_by.cljs$core$IFn$_invoke$arity$1 = partition_by__1;\n  partition_by.cljs$core$IFn$_invoke$arity$2 = partition_by__2;\n  return partition_by;\n}();\ncljs.core.frequencies = function frequencies(coll) {\n  return cljs.core.persistent_BANG_.call(null, cljs.core.reduce.call(null, function(counts, x) {\n    return cljs.core.assoc_BANG_.call(null, counts, x, cljs.core.get.call(null, counts, x, 0) + 1);\n  }, cljs.core.transient$.call(null, cljs.core.PersistentArrayMap.EMPTY), coll));\n};\ncljs.core.reductions = function() {\n  var reductions = null;\n  var reductions__2 = function(f, coll) {\n    return new cljs.core.LazySeq(null, function() {\n      var temp__4124__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4124__auto__) {\n        var s = temp__4124__auto__;\n        return reductions.call(null, f, cljs.core.first.call(null, s), cljs.core.rest.call(null, s));\n      } else {\n        return cljs.core._conj.call(null, cljs.core.List.EMPTY, f.call(null));\n      }\n    }, null, null);\n  };\n  var reductions__3 = function(f, init, coll) {\n    return cljs.core.cons.call(null, init, new cljs.core.LazySeq(null, function() {\n      var temp__4126__auto__ = cljs.core.seq.call(null, coll);\n      if (temp__4126__auto__) {\n        var s = temp__4126__auto__;\n        return reductions.call(null, f, f.call(null, init, cljs.core.first.call(null, s)), cljs.core.rest.call(null, s));\n      } else {\n        return null;\n      }\n    }, null, null));\n  };\n  reductions = function(f, init, coll) {\n    switch(arguments.length) {\n      case 2:\n        return reductions__2.call(this, f, init);\n      case 3:\n        return reductions__3.call(this, f, init, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  reductions.cljs$core$IFn$_invoke$arity$2 = reductions__2;\n  reductions.cljs$core$IFn$_invoke$arity$3 = reductions__3;\n  return reductions;\n}();\ncljs.core.juxt = function() {\n  var juxt = null;\n  var juxt__1 = function(f) {\n    return function() {\n      var G__23041 = null;\n      var G__23041__0 = function() {\n        return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null)], null);\n      };\n      var G__23041__1 = function(x) {\n        return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null, x)], null);\n      };\n      var G__23041__2 = function(x, y) {\n        return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null, x, y)], null);\n      };\n      var G__23041__3 = function(x, y, z) {\n        return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null, x, y, z)], null);\n      };\n      var G__23041__4 = function() {\n        var G__23042__delegate = function(x, y, z, args) {\n          return new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.apply.call(null, f, x, y, z, args)], null);\n        };\n        var G__23042 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__23042__delegate.call(this, x, y, z, args);\n        };\n        G__23042.cljs$lang$maxFixedArity = 3;\n        G__23042.cljs$lang$applyTo = function(arglist__23043) {\n          var x = cljs.core.first(arglist__23043);\n          arglist__23043 = cljs.core.next(arglist__23043);\n          var y = cljs.core.first(arglist__23043);\n          arglist__23043 = cljs.core.next(arglist__23043);\n          var z = cljs.core.first(arglist__23043);\n          var args = cljs.core.rest(arglist__23043);\n          return G__23042__delegate(x, y, z, args);\n        };\n        G__23042.cljs$core$IFn$_invoke$arity$variadic = G__23042__delegate;\n        return G__23042;\n      }();\n      G__23041 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return G__23041__0.call(this);\n          case 1:\n            return G__23041__1.call(this, x);\n          case 2:\n            return G__23041__2.call(this, x, y);\n          case 3:\n            return G__23041__3.call(this, x, y, z);\n          default:\n            return G__23041__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      G__23041.cljs$lang$maxFixedArity = 3;\n      G__23041.cljs$lang$applyTo = G__23041__4.cljs$lang$applyTo;\n      G__23041.cljs$core$IFn$_invoke$arity$0 = G__23041__0;\n      G__23041.cljs$core$IFn$_invoke$arity$1 = G__23041__1;\n      G__23041.cljs$core$IFn$_invoke$arity$2 = G__23041__2;\n      G__23041.cljs$core$IFn$_invoke$arity$3 = G__23041__3;\n      G__23041.cljs$core$IFn$_invoke$arity$variadic = G__23041__4.cljs$core$IFn$_invoke$arity$variadic;\n      return G__23041;\n    }();\n  };\n  var juxt__2 = function(f, g) {\n    return function() {\n      var G__23044 = null;\n      var G__23044__0 = function() {\n        return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null), g.call(null)], null);\n      };\n      var G__23044__1 = function(x) {\n        return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null, x), g.call(null, x)], null);\n      };\n      var G__23044__2 = function(x, y) {\n        return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null, x, y), g.call(null, x, y)], null);\n      };\n      var G__23044__3 = function(x, y, z) {\n        return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null, x, y, z), g.call(null, x, y, z)], null);\n      };\n      var G__23044__4 = function() {\n        var G__23045__delegate = function(x, y, z, args) {\n          return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.apply.call(null, f, x, y, z, args), cljs.core.apply.call(null, g, x, y, z, args)], null);\n        };\n        var G__23045 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__23045__delegate.call(this, x, y, z, args);\n        };\n        G__23045.cljs$lang$maxFixedArity = 3;\n        G__23045.cljs$lang$applyTo = function(arglist__23046) {\n          var x = cljs.core.first(arglist__23046);\n          arglist__23046 = cljs.core.next(arglist__23046);\n          var y = cljs.core.first(arglist__23046);\n          arglist__23046 = cljs.core.next(arglist__23046);\n          var z = cljs.core.first(arglist__23046);\n          var args = cljs.core.rest(arglist__23046);\n          return G__23045__delegate(x, y, z, args);\n        };\n        G__23045.cljs$core$IFn$_invoke$arity$variadic = G__23045__delegate;\n        return G__23045;\n      }();\n      G__23044 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return G__23044__0.call(this);\n          case 1:\n            return G__23044__1.call(this, x);\n          case 2:\n            return G__23044__2.call(this, x, y);\n          case 3:\n            return G__23044__3.call(this, x, y, z);\n          default:\n            return G__23044__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      G__23044.cljs$lang$maxFixedArity = 3;\n      G__23044.cljs$lang$applyTo = G__23044__4.cljs$lang$applyTo;\n      G__23044.cljs$core$IFn$_invoke$arity$0 = G__23044__0;\n      G__23044.cljs$core$IFn$_invoke$arity$1 = G__23044__1;\n      G__23044.cljs$core$IFn$_invoke$arity$2 = G__23044__2;\n      G__23044.cljs$core$IFn$_invoke$arity$3 = G__23044__3;\n      G__23044.cljs$core$IFn$_invoke$arity$variadic = G__23044__4.cljs$core$IFn$_invoke$arity$variadic;\n      return G__23044;\n    }();\n  };\n  var juxt__3 = function(f, g, h) {\n    return function() {\n      var G__23047 = null;\n      var G__23047__0 = function() {\n        return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null), g.call(null), h.call(null)], null);\n      };\n      var G__23047__1 = function(x) {\n        return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null, x), g.call(null, x), h.call(null, x)], null);\n      };\n      var G__23047__2 = function(x, y) {\n        return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null, x, y), g.call(null, x, y), h.call(null, x, y)], null);\n      };\n      var G__23047__3 = function(x, y, z) {\n        return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [f.call(null, x, y, z), g.call(null, x, y, z), h.call(null, x, y, z)], null);\n      };\n      var G__23047__4 = function() {\n        var G__23048__delegate = function(x, y, z, args) {\n          return new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.apply.call(null, f, x, y, z, args), cljs.core.apply.call(null, g, x, y, z, args), cljs.core.apply.call(null, h, x, y, z, args)], null);\n        };\n        var G__23048 = function(x, y, z, var_args) {\n          var args = null;\n          if (arguments.length &gt; 3) {\n            args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n          }\n          return G__23048__delegate.call(this, x, y, z, args);\n        };\n        G__23048.cljs$lang$maxFixedArity = 3;\n        G__23048.cljs$lang$applyTo = function(arglist__23049) {\n          var x = cljs.core.first(arglist__23049);\n          arglist__23049 = cljs.core.next(arglist__23049);\n          var y = cljs.core.first(arglist__23049);\n          arglist__23049 = cljs.core.next(arglist__23049);\n          var z = cljs.core.first(arglist__23049);\n          var args = cljs.core.rest(arglist__23049);\n          return G__23048__delegate(x, y, z, args);\n        };\n        G__23048.cljs$core$IFn$_invoke$arity$variadic = G__23048__delegate;\n        return G__23048;\n      }();\n      G__23047 = function(x, y, z, var_args) {\n        var args = var_args;\n        switch(arguments.length) {\n          case 0:\n            return G__23047__0.call(this);\n          case 1:\n            return G__23047__1.call(this, x);\n          case 2:\n            return G__23047__2.call(this, x, y);\n          case 3:\n            return G__23047__3.call(this, x, y, z);\n          default:\n            return G__23047__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n        }\n        throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n      };\n      G__23047.cljs$lang$maxFixedArity = 3;\n      G__23047.cljs$lang$applyTo = G__23047__4.cljs$lang$applyTo;\n      G__23047.cljs$core$IFn$_invoke$arity$0 = G__23047__0;\n      G__23047.cljs$core$IFn$_invoke$arity$1 = G__23047__1;\n      G__23047.cljs$core$IFn$_invoke$arity$2 = G__23047__2;\n      G__23047.cljs$core$IFn$_invoke$arity$3 = G__23047__3;\n      G__23047.cljs$core$IFn$_invoke$arity$variadic = G__23047__4.cljs$core$IFn$_invoke$arity$variadic;\n      return G__23047;\n    }();\n  };\n  var juxt__4 = function() {\n    var G__23050__delegate = function(f, g, h, fs) {\n      var fs__$1 = cljs.core.list_STAR_.call(null, f, g, h, fs);\n      return function(fs__$1) {\n        return function() {\n          var G__23051 = null;\n          var G__23051__0 = function() {\n            return cljs.core.reduce.call(null, function(fs__$1) {\n              return function(p1__23031_SHARP_, p2__23032_SHARP_) {\n                return cljs.core.conj.call(null, p1__23031_SHARP_, p2__23032_SHARP_.call(null));\n              };\n            }(fs__$1), cljs.core.PersistentVector.EMPTY, fs__$1);\n          };\n          var G__23051__1 = function(x) {\n            return cljs.core.reduce.call(null, function(fs__$1) {\n              return function(p1__23033_SHARP_, p2__23034_SHARP_) {\n                return cljs.core.conj.call(null, p1__23033_SHARP_, p2__23034_SHARP_.call(null, x));\n              };\n            }(fs__$1), cljs.core.PersistentVector.EMPTY, fs__$1);\n          };\n          var G__23051__2 = function(x, y) {\n            return cljs.core.reduce.call(null, function(fs__$1) {\n              return function(p1__23035_SHARP_, p2__23036_SHARP_) {\n                return cljs.core.conj.call(null, p1__23035_SHARP_, p2__23036_SHARP_.call(null, x, y));\n              };\n            }(fs__$1), cljs.core.PersistentVector.EMPTY, fs__$1);\n          };\n          var G__23051__3 = function(x, y, z) {\n            return cljs.core.reduce.call(null, function(fs__$1) {\n              return function(p1__23037_SHARP_, p2__23038_SHARP_) {\n                return cljs.core.conj.call(null, p1__23037_SHARP_, p2__23038_SHARP_.call(null, x, y, z));\n              };\n            }(fs__$1), cljs.core.PersistentVector.EMPTY, fs__$1);\n          };\n          var G__23051__4 = function() {\n            var G__23052__delegate = function(x, y, z, args) {\n              return cljs.core.reduce.call(null, function(fs__$1) {\n                return function(p1__23039_SHARP_, p2__23040_SHARP_) {\n                  return cljs.core.conj.call(null, p1__23039_SHARP_, cljs.core.apply.call(null, p2__23040_SHARP_, x, y, z, args));\n                };\n              }(fs__$1), cljs.core.PersistentVector.EMPTY, fs__$1);\n            };\n            var G__23052 = function(x, y, z, var_args) {\n              var args = null;\n              if (arguments.length &gt; 3) {\n                args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n              }\n              return G__23052__delegate.call(this, x, y, z, args);\n            };\n            G__23052.cljs$lang$maxFixedArity = 3;\n            G__23052.cljs$lang$applyTo = function(arglist__23053) {\n              var x = cljs.core.first(arglist__23053);\n              arglist__23053 = cljs.core.next(arglist__23053);\n              var y = cljs.core.first(arglist__23053);\n              arglist__23053 = cljs.core.next(arglist__23053);\n              var z = cljs.core.first(arglist__23053);\n              var args = cljs.core.rest(arglist__23053);\n              return G__23052__delegate(x, y, z, args);\n            };\n            G__23052.cljs$core$IFn$_invoke$arity$variadic = G__23052__delegate;\n            return G__23052;\n          }();\n          G__23051 = function(x, y, z, var_args) {\n            var args = var_args;\n            switch(arguments.length) {\n              case 0:\n                return G__23051__0.call(this);\n              case 1:\n                return G__23051__1.call(this, x);\n              case 2:\n                return G__23051__2.call(this, x, y);\n              case 3:\n                return G__23051__3.call(this, x, y, z);\n              default:\n                return G__23051__4.cljs$core$IFn$_invoke$arity$variadic(x, y, z, cljs.core.array_seq(arguments, 3));\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          G__23051.cljs$lang$maxFixedArity = 3;\n          G__23051.cljs$lang$applyTo = G__23051__4.cljs$lang$applyTo;\n          G__23051.cljs$core$IFn$_invoke$arity$0 = G__23051__0;\n          G__23051.cljs$core$IFn$_invoke$arity$1 = G__23051__1;\n          G__23051.cljs$core$IFn$_invoke$arity$2 = G__23051__2;\n          G__23051.cljs$core$IFn$_invoke$arity$3 = G__23051__3;\n          G__23051.cljs$core$IFn$_invoke$arity$variadic = G__23051__4.cljs$core$IFn$_invoke$arity$variadic;\n          return G__23051;\n        }();\n      }(fs__$1);\n    };\n    var G__23050 = function(f, g, h, var_args) {\n      var fs = null;\n      if (arguments.length &gt; 3) {\n        fs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 3), 0);\n      }\n      return G__23050__delegate.call(this, f, g, h, fs);\n    };\n    G__23050.cljs$lang$maxFixedArity = 3;\n    G__23050.cljs$lang$applyTo = function(arglist__23054) {\n      var f = cljs.core.first(arglist__23054);\n      arglist__23054 = cljs.core.next(arglist__23054);\n      var g = cljs.core.first(arglist__23054);\n      arglist__23054 = cljs.core.next(arglist__23054);\n      var h = cljs.core.first(arglist__23054);\n      var fs = cljs.core.rest(arglist__23054);\n      return G__23050__delegate(f, g, h, fs);\n    };\n    G__23050.cljs$core$IFn$_invoke$arity$variadic = G__23050__delegate;\n    return G__23050;\n  }();\n  juxt = function(f, g, h, var_args) {\n    var fs = var_args;\n    switch(arguments.length) {\n      case 1:\n        return juxt__1.call(this, f);\n      case 2:\n        return juxt__2.call(this, f, g);\n      case 3:\n        return juxt__3.call(this, f, g, h);\n      default:\n        return juxt__4.cljs$core$IFn$_invoke$arity$variadic(f, g, h, cljs.core.array_seq(arguments, 3));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  juxt.cljs$lang$maxFixedArity = 3;\n  juxt.cljs$lang$applyTo = juxt__4.cljs$lang$applyTo;\n  juxt.cljs$core$IFn$_invoke$arity$1 = juxt__1;\n  juxt.cljs$core$IFn$_invoke$arity$2 = juxt__2;\n  juxt.cljs$core$IFn$_invoke$arity$3 = juxt__3;\n  juxt.cljs$core$IFn$_invoke$arity$variadic = juxt__4.cljs$core$IFn$_invoke$arity$variadic;\n  return juxt;\n}();\ncljs.core.dorun = function() {\n  var dorun = null;\n  var dorun__1 = function(coll) {\n    while (true) {\n      if (cljs.core.seq.call(null, coll)) {\n        var G__23055 = cljs.core.next.call(null, coll);\n        coll = G__23055;\n        continue;\n      } else {\n        return null;\n      }\n      break;\n    }\n  };\n  var dorun__2 = function(n, coll) {\n    while (true) {\n      if (cljs.core.seq.call(null, coll) &amp;&amp; n &gt; 0) {\n        var G__23056 = n - 1;\n        var G__23057 = cljs.core.next.call(null, coll);\n        n = G__23056;\n        coll = G__23057;\n        continue;\n      } else {\n        return null;\n      }\n      break;\n    }\n  };\n  dorun = function(n, coll) {\n    switch(arguments.length) {\n      case 1:\n        return dorun__1.call(this, n);\n      case 2:\n        return dorun__2.call(this, n, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  dorun.cljs$core$IFn$_invoke$arity$1 = dorun__1;\n  dorun.cljs$core$IFn$_invoke$arity$2 = dorun__2;\n  return dorun;\n}();\ncljs.core.doall = function() {\n  var doall = null;\n  var doall__1 = function(coll) {\n    cljs.core.dorun.call(null, coll);\n    return coll;\n  };\n  var doall__2 = function(n, coll) {\n    cljs.core.dorun.call(null, n, coll);\n    return coll;\n  };\n  doall = function(n, coll) {\n    switch(arguments.length) {\n      case 1:\n        return doall__1.call(this, n);\n      case 2:\n        return doall__2.call(this, n, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  doall.cljs$core$IFn$_invoke$arity$1 = doall__1;\n  doall.cljs$core$IFn$_invoke$arity$2 = doall__2;\n  return doall;\n}();\ncljs.core.regexp_QMARK_ = function regexp_QMARK_(o) {\n  return o instanceof RegExp;\n};\ncljs.core.re_matches = function re_matches(re, s) {\n  if (typeof s === &quot;string&quot;) {\n    var matches = re.exec(s);\n    if (cljs.core._EQ_.call(null, cljs.core.first.call(null, matches), s)) {\n      if (cljs.core.count.call(null, matches) === 1) {\n        return cljs.core.first.call(null, matches);\n      } else {\n        return cljs.core.vec.call(null, matches);\n      }\n    } else {\n      return null;\n    }\n  } else {\n    throw new TypeError(&quot;re-matches must match against a string.&quot;);\n  }\n};\ncljs.core.re_find = function re_find(re, s) {\n  if (typeof s === &quot;string&quot;) {\n    var matches = re.exec(s);\n    if (matches == null) {\n      return null;\n    } else {\n      if (cljs.core.count.call(null, matches) === 1) {\n        return cljs.core.first.call(null, matches);\n      } else {\n        return cljs.core.vec.call(null, matches);\n      }\n    }\n  } else {\n    throw new TypeError(&quot;re-find must match against a string.&quot;);\n  }\n};\ncljs.core.re_seq = function re_seq(re, s) {\n  var match_data = cljs.core.re_find.call(null, re, s);\n  var match_idx = s.search(re);\n  var match_str = cljs.core.coll_QMARK_.call(null, match_data) ? cljs.core.first.call(null, match_data) : match_data;\n  var post_match = cljs.core.subs.call(null, s, match_idx + cljs.core.count.call(null, match_str));\n  if (cljs.core.truth_(match_data)) {\n    return new cljs.core.LazySeq(null, function(match_data, match_idx, match_str, post_match) {\n      return function() {\n        return cljs.core.cons.call(null, match_data, cljs.core.seq.call(null, post_match) ? re_seq.call(null, re, post_match) : null);\n      };\n    }(match_data, match_idx, match_str, post_match), null, null);\n  } else {\n    return null;\n  }\n};\ncljs.core.re_pattern = function re_pattern(s) {\n  var vec__23059 = cljs.core.re_find.call(null, /^(?:\\(\\?([idmsux]*)\\))?(.*)/, s);\n  var _ = cljs.core.nth.call(null, vec__23059, 0, null);\n  var flags = cljs.core.nth.call(null, vec__23059, 1, null);\n  var pattern = cljs.core.nth.call(null, vec__23059, 2, null);\n  return new RegExp(pattern, flags);\n};\ncljs.core.pr_sequential_writer = function pr_sequential_writer(writer, print_one, begin, sep, end, opts, coll) {\n  var _STAR_print_level_STAR_23061 = cljs.core._STAR_print_level_STAR_;\n  try {\n    cljs.core._STAR_print_level_STAR_ = cljs.core._STAR_print_level_STAR_ == null ? null : cljs.core._STAR_print_level_STAR_ - 1;\n    if (!(cljs.core._STAR_print_level_STAR_ == null) &amp;&amp; cljs.core._STAR_print_level_STAR_ &lt; 0) {\n      return cljs.core._write.call(null, writer, &quot;#&quot;);\n    } else {\n      cljs.core._write.call(null, writer, begin);\n      if (cljs.core.seq.call(null, coll)) {\n        print_one.call(null, cljs.core.first.call(null, coll), writer, opts);\n      } else {\n      }\n      var coll_23062__$1 = cljs.core.next.call(null, coll);\n      var n_23063 = (new cljs.core.Keyword(null, &quot;print-length&quot;, &quot;print-length&quot;, 1931866356)).cljs$core$IFn$_invoke$arity$1(opts) - 1;\n      while (true) {\n        if (coll_23062__$1 &amp;&amp; (n_23063 == null || !(n_23063 === 0))) {\n          cljs.core._write.call(null, writer, sep);\n          print_one.call(null, cljs.core.first.call(null, coll_23062__$1), writer, opts);\n          var G__23064 = cljs.core.next.call(null, coll_23062__$1);\n          var G__23065 = n_23063 - 1;\n          coll_23062__$1 = G__23064;\n          n_23063 = G__23065;\n          continue;\n        } else {\n          if (cljs.core.seq.call(null, coll_23062__$1) &amp;&amp; n_23063 === 0) {\n            cljs.core._write.call(null, writer, sep);\n            cljs.core._write.call(null, writer, &quot;...&quot;);\n          } else {\n          }\n        }\n        break;\n      }\n      return cljs.core._write.call(null, writer, end);\n    }\n  } finally {\n    cljs.core._STAR_print_level_STAR_ = _STAR_print_level_STAR_23061;\n  }\n};\ncljs.core.write_all = function() {\n  var write_all__delegate = function(writer, ss) {\n    var seq__23070 = cljs.core.seq.call(null, ss);\n    var chunk__23071 = null;\n    var count__23072 = 0;\n    var i__23073 = 0;\n    while (true) {\n      if (i__23073 &lt; count__23072) {\n        var s = cljs.core._nth.call(null, chunk__23071, i__23073);\n        cljs.core._write.call(null, writer, s);\n        var G__23074 = seq__23070;\n        var G__23075 = chunk__23071;\n        var G__23076 = count__23072;\n        var G__23077 = i__23073 + 1;\n        seq__23070 = G__23074;\n        chunk__23071 = G__23075;\n        count__23072 = G__23076;\n        i__23073 = G__23077;\n        continue;\n      } else {\n        var temp__4126__auto__ = cljs.core.seq.call(null, seq__23070);\n        if (temp__4126__auto__) {\n          var seq__23070__$1 = temp__4126__auto__;\n          if (cljs.core.chunked_seq_QMARK_.call(null, seq__23070__$1)) {\n            var c__17901__auto__ = cljs.core.chunk_first.call(null, seq__23070__$1);\n            var G__23078 = cljs.core.chunk_rest.call(null, seq__23070__$1);\n            var G__23079 = c__17901__auto__;\n            var G__23080 = cljs.core.count.call(null, c__17901__auto__);\n            var G__23081 = 0;\n            seq__23070 = G__23078;\n            chunk__23071 = G__23079;\n            count__23072 = G__23080;\n            i__23073 = G__23081;\n            continue;\n          } else {\n            var s = cljs.core.first.call(null, seq__23070__$1);\n            cljs.core._write.call(null, writer, s);\n            var G__23082 = cljs.core.next.call(null, seq__23070__$1);\n            var G__23083 = null;\n            var G__23084 = 0;\n            var G__23085 = 0;\n            seq__23070 = G__23082;\n            chunk__23071 = G__23083;\n            count__23072 = G__23084;\n            i__23073 = G__23085;\n            continue;\n          }\n        } else {\n          return null;\n        }\n      }\n      break;\n    }\n  };\n  var write_all = function(writer, var_args) {\n    var ss = null;\n    if (arguments.length &gt; 1) {\n      ss = cljs.core.array_seq(Array.prototype.slice.call(arguments, 1), 0);\n    }\n    return write_all__delegate.call(this, writer, ss);\n  };\n  write_all.cljs$lang$maxFixedArity = 1;\n  write_all.cljs$lang$applyTo = function(arglist__23086) {\n    var writer = cljs.core.first(arglist__23086);\n    var ss = cljs.core.rest(arglist__23086);\n    return write_all__delegate(writer, ss);\n  };\n  write_all.cljs$core$IFn$_invoke$arity$variadic = write_all__delegate;\n  return write_all;\n}();\ncljs.core.string_print = function string_print(x) {\n  cljs.core._STAR_print_fn_STAR_.call(null, x);\n  return null;\n};\ncljs.core.flush = function flush() {\n  return null;\n};\ncljs.core.char_escapes = function() {\n  var obj23088 = {&apos;&quot;&apos;:&apos;\\\\&quot;&apos;, &quot;\\\\&quot;:&quot;\\\\\\\\&quot;, &quot;\\b&quot;:&quot;\\\\b&quot;, &quot;\\f&quot;:&quot;\\\\f&quot;, &quot;\\n&quot;:&quot;\\\\n&quot;, &quot;\\r&quot;:&quot;\\\\r&quot;, &quot;\\t&quot;:&quot;\\\\t&quot;};\n  return obj23088;\n}();\ncljs.core.quote_string = function quote_string(s) {\n  return&apos;&quot;&apos; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(s.replace(RegExp(&apos;[\\\\\\\\&quot;\\b\\f\\n\\r\\t]&apos;, &quot;g&quot;), function(match) {\n    return cljs.core.char_escapes[match];\n  })) + &apos;&quot;&apos;;\n};\ncljs.core.pr_writer = function pr_writer(obj, writer, opts) {\n  if (obj == null) {\n    return cljs.core._write.call(null, writer, &quot;nil&quot;);\n  } else {\n    if (void 0 === obj) {\n      return cljs.core._write.call(null, writer, &quot;#\\x3cundefined\\x3e&quot;);\n    } else {\n      if (cljs.core.truth_(function() {\n        var and__17133__auto__ = cljs.core.get.call(null, opts, new cljs.core.Keyword(null, &quot;meta&quot;, &quot;meta&quot;, 1499536964));\n        if (cljs.core.truth_(and__17133__auto__)) {\n          var and__17133__auto____$1 = function() {\n            var G__23094 = obj;\n            if (G__23094) {\n              var bit__17795__auto__ = G__23094.cljs$lang$protocol_mask$partition0$ &amp; 131072;\n              if (bit__17795__auto__ || G__23094.cljs$core$IMeta$) {\n                return true;\n              } else {\n                if (!G__23094.cljs$lang$protocol_mask$partition0$) {\n                  return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMeta, G__23094);\n                } else {\n                  return false;\n                }\n              }\n            } else {\n              return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IMeta, G__23094);\n            }\n          }();\n          if (and__17133__auto____$1) {\n            return cljs.core.meta.call(null, obj);\n          } else {\n            return and__17133__auto____$1;\n          }\n        } else {\n          return and__17133__auto__;\n        }\n      }())) {\n        cljs.core._write.call(null, writer, &quot;^&quot;);\n        pr_writer.call(null, cljs.core.meta.call(null, obj), writer, opts);\n        cljs.core._write.call(null, writer, &quot; &quot;);\n      } else {\n      }\n      if (obj == null) {\n        return cljs.core._write.call(null, writer, &quot;nil&quot;);\n      } else {\n        if (obj.cljs$lang$type) {\n          return obj.cljs$lang$ctorPrWriter(obj, writer, opts);\n        } else {\n          if (function() {\n            var G__23095 = obj;\n            if (G__23095) {\n              var bit__17788__auto__ = G__23095.cljs$lang$protocol_mask$partition0$ &amp; 2147483648;\n              if (bit__17788__auto__ || G__23095.cljs$core$IPrintWithWriter$) {\n                return true;\n              } else {\n                return false;\n              }\n            } else {\n              return false;\n            }\n          }()) {\n            return cljs.core._pr_writer.call(null, obj, writer, opts);\n          } else {\n            if (cljs.core.type.call(null, obj) === Boolean || typeof obj === &quot;number&quot;) {\n              return cljs.core._write.call(null, writer, &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(obj));\n            } else {\n              if (cljs.core.object_QMARK_.call(null, obj)) {\n                cljs.core._write.call(null, writer, &quot;#js &quot;);\n                return cljs.core.print_map.call(null, cljs.core.map.call(null, function(k) {\n                  return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [cljs.core.keyword.call(null, k), obj[k]], null);\n                }, cljs.core.js_keys.call(null, obj)), pr_writer, writer, opts);\n              } else {\n                if (obj instanceof Array) {\n                  return cljs.core.pr_sequential_writer.call(null, writer, pr_writer, &quot;#js [&quot;, &quot; &quot;, &quot;]&quot;, opts, obj);\n                } else {\n                  if (goog.isString(obj)) {\n                    if (cljs.core.truth_((new cljs.core.Keyword(null, &quot;readably&quot;, &quot;readably&quot;, 1129599760)).cljs$core$IFn$_invoke$arity$1(opts))) {\n                      return cljs.core._write.call(null, writer, cljs.core.quote_string.call(null, obj));\n                    } else {\n                      return cljs.core._write.call(null, writer, obj);\n                    }\n                  } else {\n                    if (cljs.core.fn_QMARK_.call(null, obj)) {\n                      return cljs.core.write_all.call(null, writer, &quot;#\\x3c&quot;, &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(obj), &quot;\\x3e&quot;);\n                    } else {\n                      if (obj instanceof Date) {\n                        var normalize = function(n, len) {\n                          var ns = &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(n);\n                          while (true) {\n                            if (cljs.core.count.call(null, ns) &lt; len) {\n                              var G__23097 = &quot;0&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(ns);\n                              ns = G__23097;\n                              continue;\n                            } else {\n                              return ns;\n                            }\n                            break;\n                          }\n                        };\n                        return cljs.core.write_all.call(null, writer, &apos;#inst &quot;&apos;, &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(obj.getUTCFullYear()), &quot;-&quot;, normalize.call(null, obj.getUTCMonth() + 1, 2), &quot;-&quot;, normalize.call(null, obj.getUTCDate(), 2), &quot;T&quot;, normalize.call(null, obj.getUTCHours(), 2), &quot;:&quot;, normalize.call(null, obj.getUTCMinutes(), 2), &quot;:&quot;, normalize.call(null, obj.getUTCSeconds(), 2), &quot;.&quot;, normalize.call(null, obj.getUTCMilliseconds(), 3), &quot;-&quot;, &apos;00:00&quot;&apos;);\n                      } else {\n                        if (cljs.core.regexp_QMARK_.call(null, obj)) {\n                          return cljs.core.write_all.call(null, writer, &apos;#&quot;&apos;, obj.source, &apos;&quot;&apos;);\n                        } else {\n                          if (function() {\n                            var G__23096 = obj;\n                            if (G__23096) {\n                              var bit__17795__auto__ = G__23096.cljs$lang$protocol_mask$partition0$ &amp; 2147483648;\n                              if (bit__17795__auto__ || G__23096.cljs$core$IPrintWithWriter$) {\n                                return true;\n                              } else {\n                                if (!G__23096.cljs$lang$protocol_mask$partition0$) {\n                                  return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IPrintWithWriter, G__23096);\n                                } else {\n                                  return false;\n                                }\n                              }\n                            } else {\n                              return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IPrintWithWriter, G__23096);\n                            }\n                          }()) {\n                            return cljs.core._pr_writer.call(null, obj, writer, opts);\n                          } else {\n                            return cljs.core.write_all.call(null, writer, &quot;#\\x3c&quot;, &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(obj), &quot;\\x3e&quot;);\n                          }\n                        }\n                      }\n                    }\n                  }\n                }\n              }\n            }\n          }\n        }\n      }\n    }\n  }\n};\ncljs.core.pr_seq_writer = function pr_seq_writer(objs, writer, opts) {\n  cljs.core.pr_writer.call(null, cljs.core.first.call(null, objs), writer, opts);\n  var seq__23102 = cljs.core.seq.call(null, cljs.core.next.call(null, objs));\n  var chunk__23103 = null;\n  var count__23104 = 0;\n  var i__23105 = 0;\n  while (true) {\n    if (i__23105 &lt; count__23104) {\n      var obj = cljs.core._nth.call(null, chunk__23103, i__23105);\n      cljs.core._write.call(null, writer, &quot; &quot;);\n      cljs.core.pr_writer.call(null, obj, writer, opts);\n      var G__23106 = seq__23102;\n      var G__23107 = chunk__23103;\n      var G__23108 = count__23104;\n      var G__23109 = i__23105 + 1;\n      seq__23102 = G__23106;\n      chunk__23103 = G__23107;\n      count__23104 = G__23108;\n      i__23105 = G__23109;\n      continue;\n    } else {\n      var temp__4126__auto__ = cljs.core.seq.call(null, seq__23102);\n      if (temp__4126__auto__) {\n        var seq__23102__$1 = temp__4126__auto__;\n        if (cljs.core.chunked_seq_QMARK_.call(null, seq__23102__$1)) {\n          var c__17901__auto__ = cljs.core.chunk_first.call(null, seq__23102__$1);\n          var G__23110 = cljs.core.chunk_rest.call(null, seq__23102__$1);\n          var G__23111 = c__17901__auto__;\n          var G__23112 = cljs.core.count.call(null, c__17901__auto__);\n          var G__23113 = 0;\n          seq__23102 = G__23110;\n          chunk__23103 = G__23111;\n          count__23104 = G__23112;\n          i__23105 = G__23113;\n          continue;\n        } else {\n          var obj = cljs.core.first.call(null, seq__23102__$1);\n          cljs.core._write.call(null, writer, &quot; &quot;);\n          cljs.core.pr_writer.call(null, obj, writer, opts);\n          var G__23114 = cljs.core.next.call(null, seq__23102__$1);\n          var G__23115 = null;\n          var G__23116 = 0;\n          var G__23117 = 0;\n          seq__23102 = G__23114;\n          chunk__23103 = G__23115;\n          count__23104 = G__23116;\n          i__23105 = G__23117;\n          continue;\n        }\n      } else {\n        return null;\n      }\n    }\n    break;\n  }\n};\ncljs.core.pr_sb_with_opts = function pr_sb_with_opts(objs, opts) {\n  var sb = new goog.string.StringBuffer;\n  var writer = new cljs.core.StringBufferWriter(sb);\n  cljs.core.pr_seq_writer.call(null, objs, writer, opts);\n  cljs.core._flush.call(null, writer);\n  return sb;\n};\ncljs.core.pr_str_with_opts = function pr_str_with_opts(objs, opts) {\n  if (cljs.core.empty_QMARK_.call(null, objs)) {\n    return &quot;&quot;;\n  } else {\n    return &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.pr_sb_with_opts.call(null, objs, opts));\n  }\n};\ncljs.core.prn_str_with_opts = function prn_str_with_opts(objs, opts) {\n  if (cljs.core.empty_QMARK_.call(null, objs)) {\n    return &quot;\\n&quot;;\n  } else {\n    var sb = cljs.core.pr_sb_with_opts.call(null, objs, opts);\n    sb.append(&quot;\\n&quot;);\n    return &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(sb);\n  }\n};\ncljs.core.pr_with_opts = function pr_with_opts(objs, opts) {\n  return cljs.core.string_print.call(null, cljs.core.pr_str_with_opts.call(null, objs, opts));\n};\ncljs.core.newline = function newline(opts) {\n  cljs.core.string_print.call(null, &quot;\\n&quot;);\n  if (cljs.core.truth_(cljs.core.get.call(null, opts, new cljs.core.Keyword(null, &quot;flush-on-newline&quot;, &quot;flush-on-newline&quot;, -151457939)))) {\n    return cljs.core.flush.call(null);\n  } else {\n    return null;\n  }\n};\ncljs.core.pr_str = function() {\n  var pr_str__delegate = function(objs) {\n    return cljs.core.pr_str_with_opts.call(null, objs, cljs.core.pr_opts.call(null));\n  };\n  var pr_str = function(var_args) {\n    var objs = null;\n    if (arguments.length &gt; 0) {\n      objs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return pr_str__delegate.call(this, objs);\n  };\n  pr_str.cljs$lang$maxFixedArity = 0;\n  pr_str.cljs$lang$applyTo = function(arglist__23118) {\n    var objs = cljs.core.seq(arglist__23118);\n    return pr_str__delegate(objs);\n  };\n  pr_str.cljs$core$IFn$_invoke$arity$variadic = pr_str__delegate;\n  return pr_str;\n}();\ncljs.core.prn_str = function() {\n  var prn_str__delegate = function(objs) {\n    return cljs.core.prn_str_with_opts.call(null, objs, cljs.core.pr_opts.call(null));\n  };\n  var prn_str = function(var_args) {\n    var objs = null;\n    if (arguments.length &gt; 0) {\n      objs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return prn_str__delegate.call(this, objs);\n  };\n  prn_str.cljs$lang$maxFixedArity = 0;\n  prn_str.cljs$lang$applyTo = function(arglist__23119) {\n    var objs = cljs.core.seq(arglist__23119);\n    return prn_str__delegate(objs);\n  };\n  prn_str.cljs$core$IFn$_invoke$arity$variadic = prn_str__delegate;\n  return prn_str;\n}();\ncljs.core.pr = function() {\n  var pr__delegate = function(objs) {\n    return cljs.core.pr_with_opts.call(null, objs, cljs.core.pr_opts.call(null));\n  };\n  var pr = function(var_args) {\n    var objs = null;\n    if (arguments.length &gt; 0) {\n      objs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return pr__delegate.call(this, objs);\n  };\n  pr.cljs$lang$maxFixedArity = 0;\n  pr.cljs$lang$applyTo = function(arglist__23120) {\n    var objs = cljs.core.seq(arglist__23120);\n    return pr__delegate(objs);\n  };\n  pr.cljs$core$IFn$_invoke$arity$variadic = pr__delegate;\n  return pr;\n}();\ncljs.core.print = function() {\n  var cljs_core_print__delegate = function(objs) {\n    return cljs.core.pr_with_opts.call(null, objs, cljs.core.assoc.call(null, cljs.core.pr_opts.call(null), new cljs.core.Keyword(null, &quot;readably&quot;, &quot;readably&quot;, 1129599760), false));\n  };\n  var cljs_core_print = function(var_args) {\n    var objs = null;\n    if (arguments.length &gt; 0) {\n      objs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return cljs_core_print__delegate.call(this, objs);\n  };\n  cljs_core_print.cljs$lang$maxFixedArity = 0;\n  cljs_core_print.cljs$lang$applyTo = function(arglist__23121) {\n    var objs = cljs.core.seq(arglist__23121);\n    return cljs_core_print__delegate(objs);\n  };\n  cljs_core_print.cljs$core$IFn$_invoke$arity$variadic = cljs_core_print__delegate;\n  return cljs_core_print;\n}();\ncljs.core.print_str = function() {\n  var print_str__delegate = function(objs) {\n    return cljs.core.pr_str_with_opts.call(null, objs, cljs.core.assoc.call(null, cljs.core.pr_opts.call(null), new cljs.core.Keyword(null, &quot;readably&quot;, &quot;readably&quot;, 1129599760), false));\n  };\n  var print_str = function(var_args) {\n    var objs = null;\n    if (arguments.length &gt; 0) {\n      objs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return print_str__delegate.call(this, objs);\n  };\n  print_str.cljs$lang$maxFixedArity = 0;\n  print_str.cljs$lang$applyTo = function(arglist__23122) {\n    var objs = cljs.core.seq(arglist__23122);\n    return print_str__delegate(objs);\n  };\n  print_str.cljs$core$IFn$_invoke$arity$variadic = print_str__delegate;\n  return print_str;\n}();\ncljs.core.println = function() {\n  var println__delegate = function(objs) {\n    cljs.core.pr_with_opts.call(null, objs, cljs.core.assoc.call(null, cljs.core.pr_opts.call(null), new cljs.core.Keyword(null, &quot;readably&quot;, &quot;readably&quot;, 1129599760), false));\n    if (cljs.core.truth_(cljs.core._STAR_print_newline_STAR_)) {\n      return cljs.core.newline.call(null, cljs.core.pr_opts.call(null));\n    } else {\n      return null;\n    }\n  };\n  var println = function(var_args) {\n    var objs = null;\n    if (arguments.length &gt; 0) {\n      objs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return println__delegate.call(this, objs);\n  };\n  println.cljs$lang$maxFixedArity = 0;\n  println.cljs$lang$applyTo = function(arglist__23123) {\n    var objs = cljs.core.seq(arglist__23123);\n    return println__delegate(objs);\n  };\n  println.cljs$core$IFn$_invoke$arity$variadic = println__delegate;\n  return println;\n}();\ncljs.core.println_str = function() {\n  var println_str__delegate = function(objs) {\n    return cljs.core.prn_str_with_opts.call(null, objs, cljs.core.assoc.call(null, cljs.core.pr_opts.call(null), new cljs.core.Keyword(null, &quot;readably&quot;, &quot;readably&quot;, 1129599760), false));\n  };\n  var println_str = function(var_args) {\n    var objs = null;\n    if (arguments.length &gt; 0) {\n      objs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return println_str__delegate.call(this, objs);\n  };\n  println_str.cljs$lang$maxFixedArity = 0;\n  println_str.cljs$lang$applyTo = function(arglist__23124) {\n    var objs = cljs.core.seq(arglist__23124);\n    return println_str__delegate(objs);\n  };\n  println_str.cljs$core$IFn$_invoke$arity$variadic = println_str__delegate;\n  return println_str;\n}();\ncljs.core.prn = function() {\n  var prn__delegate = function(objs) {\n    cljs.core.pr_with_opts.call(null, objs, cljs.core.pr_opts.call(null));\n    if (cljs.core.truth_(cljs.core._STAR_print_newline_STAR_)) {\n      return cljs.core.newline.call(null, cljs.core.pr_opts.call(null));\n    } else {\n      return null;\n    }\n  };\n  var prn = function(var_args) {\n    var objs = null;\n    if (arguments.length &gt; 0) {\n      objs = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n    }\n    return prn__delegate.call(this, objs);\n  };\n  prn.cljs$lang$maxFixedArity = 0;\n  prn.cljs$lang$applyTo = function(arglist__23125) {\n    var objs = cljs.core.seq(arglist__23125);\n    return prn__delegate(objs);\n  };\n  prn.cljs$core$IFn$_invoke$arity$variadic = prn__delegate;\n  return prn;\n}();\ncljs.core.print_map = function print_map(m, print_one, writer, opts) {\n  return cljs.core.pr_sequential_writer.call(null, writer, function(e, w, opts__$1) {\n    print_one.call(null, cljs.core.key.call(null, e), w, opts__$1);\n    cljs.core._write.call(null, w, &quot; &quot;);\n    return print_one.call(null, cljs.core.val.call(null, e), w, opts__$1);\n  }, &quot;{&quot;, &quot;, &quot;, &quot;}&quot;, opts, cljs.core.seq.call(null, m));\n};\ncljs.core.IndexedSeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.IndexedSeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.LazySeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.LazySeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentTreeMapSeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.NodeSeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.NodeSeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.BlackNode.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.BlackNode.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;[&quot;, &quot; &quot;, &quot;]&quot;, opts, coll__$1);\n};\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentArrayMapSeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.PersistentTreeSet.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentTreeSet.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;#{&quot;, &quot; &quot;, &quot;}&quot;, opts, coll__$1);\n};\ncljs.core.ChunkedSeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.ChunkedSeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.ObjMap.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.ObjMap.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.print_map.call(null, coll__$1, cljs.core.pr_writer, writer, opts);\n};\ncljs.core.Cons.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.Cons.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.RSeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.RSeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.PersistentHashMap.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentHashMap.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.print_map.call(null, coll__$1, cljs.core.pr_writer, writer, opts);\n};\ncljs.core.ArrayNodeSeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.ArrayNodeSeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.Subvec.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.Subvec.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;[&quot;, &quot; &quot;, &quot;]&quot;, opts, coll__$1);\n};\ncljs.core.PersistentTreeMap.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentTreeMap.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.print_map.call(null, coll__$1, cljs.core.pr_writer, writer, opts);\n};\ncljs.core.PersistentHashSet.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentHashSet.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;#{&quot;, &quot; &quot;, &quot;}&quot;, opts, coll__$1);\n};\ncljs.core.ChunkedCons.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.ChunkedCons.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.Atom.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.Atom.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(a, writer, opts) {\n  var a__$1 = this;\n  cljs.core._write.call(null, writer, &quot;#\\x3cAtom: &quot;);\n  cljs.core.pr_writer.call(null, a__$1.state, writer, opts);\n  return cljs.core._write.call(null, writer, &quot;\\x3e&quot;);\n};\ncljs.core.ValSeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.ValSeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.RedNode.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.RedNode.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;[&quot;, &quot; &quot;, &quot;]&quot;, opts, coll__$1);\n};\ncljs.core.PersistentVector.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentVector.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;[&quot;, &quot; &quot;, &quot;]&quot;, opts, coll__$1);\n};\ncljs.core.PersistentQueueSeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentQueueSeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.EmptyList.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.EmptyList.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core._write.call(null, writer, &quot;()&quot;);\n};\ncljs.core.LazyTransformer.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.LazyTransformer.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.PersistentQueue.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentQueue.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;#queue [&quot;, &quot; &quot;, &quot;]&quot;, opts, cljs.core.seq.call(null, coll__$1));\n};\ncljs.core.PersistentArrayMap.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.PersistentArrayMap.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.print_map.call(null, coll__$1, cljs.core.pr_writer, writer, opts);\n};\ncljs.core.Range.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.Range.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.KeySeq.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.KeySeq.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.List.prototype.cljs$core$IPrintWithWriter$ = true;\ncljs.core.List.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll, writer, opts) {\n  var coll__$1 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$1);\n};\ncljs.core.PersistentVector.prototype.cljs$core$IComparable$ = true;\ncljs.core.PersistentVector.prototype.cljs$core$IComparable$_compare$arity$2 = function(x, y) {\n  var x__$1 = this;\n  return cljs.core.compare_indexed.call(null, x__$1, y);\n};\ncljs.core.Subvec.prototype.cljs$core$IComparable$ = true;\ncljs.core.Subvec.prototype.cljs$core$IComparable$_compare$arity$2 = function(x, y) {\n  var x__$1 = this;\n  return cljs.core.compare_indexed.call(null, x__$1, y);\n};\ncljs.core.Keyword.prototype.cljs$core$IComparable$ = true;\ncljs.core.Keyword.prototype.cljs$core$IComparable$_compare$arity$2 = function(x, y) {\n  var x__$1 = this;\n  return cljs.core.compare_symbols.call(null, x__$1, y);\n};\ncljs.core.Symbol.prototype.cljs$core$IComparable$ = true;\ncljs.core.Symbol.prototype.cljs$core$IComparable$_compare$arity$2 = function(x, y) {\n  var x__$1 = this;\n  return cljs.core.compare_symbols.call(null, x__$1, y);\n};\ncljs.core.alter_meta_BANG_ = function() {\n  var alter_meta_BANG___delegate = function(iref, f, args) {\n    return iref.meta = cljs.core.apply.call(null, f, iref.meta, args);\n  };\n  var alter_meta_BANG_ = function(iref, f, var_args) {\n    var args = null;\n    if (arguments.length &gt; 2) {\n      args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 2), 0);\n    }\n    return alter_meta_BANG___delegate.call(this, iref, f, args);\n  };\n  alter_meta_BANG_.cljs$lang$maxFixedArity = 2;\n  alter_meta_BANG_.cljs$lang$applyTo = function(arglist__23126) {\n    var iref = cljs.core.first(arglist__23126);\n    arglist__23126 = cljs.core.next(arglist__23126);\n    var f = cljs.core.first(arglist__23126);\n    var args = cljs.core.rest(arglist__23126);\n    return alter_meta_BANG___delegate(iref, f, args);\n  };\n  alter_meta_BANG_.cljs$core$IFn$_invoke$arity$variadic = alter_meta_BANG___delegate;\n  return alter_meta_BANG_;\n}();\ncljs.core.reset_meta_BANG_ = function reset_meta_BANG_(iref, m) {\n  return iref.meta = m;\n};\ncljs.core.add_watch = function add_watch(iref, key, f) {\n  return cljs.core._add_watch.call(null, iref, key, f);\n};\ncljs.core.remove_watch = function remove_watch(iref, key) {\n  return cljs.core._remove_watch.call(null, iref, key);\n};\ncljs.core.gensym_counter = null;\ncljs.core.gensym = function() {\n  var gensym = null;\n  var gensym__0 = function() {\n    return gensym.call(null, &quot;G__&quot;);\n  };\n  var gensym__1 = function(prefix_string) {\n    if (cljs.core.gensym_counter == null) {\n      cljs.core.gensym_counter = cljs.core.atom.call(null, 0);\n    } else {\n    }\n    return cljs.core.symbol.call(null, &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(prefix_string) + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.swap_BANG_.call(null, cljs.core.gensym_counter, cljs.core.inc)));\n  };\n  gensym = function(prefix_string) {\n    switch(arguments.length) {\n      case 0:\n        return gensym__0.call(this);\n      case 1:\n        return gensym__1.call(this, prefix_string);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  gensym.cljs$core$IFn$_invoke$arity$0 = gensym__0;\n  gensym.cljs$core$IFn$_invoke$arity$1 = gensym__1;\n  return gensym;\n}();\ncljs.core.fixture1 = 1;\ncljs.core.fixture2 = 2;\ncljs.core.Delay = function(f, value) {\n  this.f = f;\n  this.value = value;\n  this.cljs$lang$protocol_mask$partition1$ = 1;\n  this.cljs$lang$protocol_mask$partition0$ = 32768;\n};\ncljs.core.Delay.cljs$lang$type = true;\ncljs.core.Delay.cljs$lang$ctorStr = &quot;cljs.core/Delay&quot;;\ncljs.core.Delay.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Delay&quot;);\n};\ncljs.core.Delay.prototype.cljs$core$IPending$_realized_QMARK_$arity$1 = function(d) {\n  var self__ = this;\n  var d__$1 = this;\n  return cljs.core.not.call(null, self__.f);\n};\ncljs.core.Delay.prototype.cljs$core$IDeref$_deref$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  if (cljs.core.truth_(self__.f)) {\n    self__.value = self__.f.call(null);\n    self__.f = null;\n  } else {\n  }\n  return self__.value;\n};\ncljs.core.__GT_Delay = function __GT_Delay(f, value) {\n  return new cljs.core.Delay(f, value);\n};\ncljs.core.delay_QMARK_ = function delay_QMARK_(x) {\n  return x instanceof cljs.core.Delay;\n};\ncljs.core.force = function force(x) {\n  if (cljs.core.delay_QMARK_.call(null, x)) {\n    return cljs.core.deref.call(null, x);\n  } else {\n    return x;\n  }\n};\ncljs.core.realized_QMARK_ = function realized_QMARK_(d) {\n  return cljs.core._realized_QMARK_.call(null, d);\n};\ncljs.core.preserving_reduced = function preserving_reduced(f1) {\n  return function(p1__23127_SHARP_, p2__23128_SHARP_) {\n    var ret = f1.call(null, p1__23127_SHARP_, p2__23128_SHARP_);\n    if (cljs.core.reduced_QMARK_.call(null, ret)) {\n      return cljs.core.reduced.call(null, ret);\n    } else {\n      return ret;\n    }\n  };\n};\ncljs.core.flatmap = function() {\n  var flatmap = null;\n  var flatmap__1 = function(f) {\n    return function(f1) {\n      return function() {\n        var G__23129 = null;\n        var G__23129__0 = function() {\n          return f1.call(null);\n        };\n        var G__23129__1 = function(result) {\n          return f1.call(null, result);\n        };\n        var G__23129__2 = function(result, input) {\n          return cljs.core.reduce.call(null, cljs.core.preserving_reduced.call(null, f1), result, f.call(null, input));\n        };\n        G__23129 = function(result, input) {\n          switch(arguments.length) {\n            case 0:\n              return G__23129__0.call(this);\n            case 1:\n              return G__23129__1.call(this, result);\n            case 2:\n              return G__23129__2.call(this, result, input);\n          }\n          throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n        };\n        G__23129.cljs$core$IFn$_invoke$arity$0 = G__23129__0;\n        G__23129.cljs$core$IFn$_invoke$arity$1 = G__23129__1;\n        G__23129.cljs$core$IFn$_invoke$arity$2 = G__23129__2;\n        return G__23129;\n      }();\n    };\n  };\n  var flatmap__2 = function(f, coll) {\n    return cljs.core.sequence.call(null, flatmap.call(null, f), coll);\n  };\n  flatmap = function(f, coll) {\n    switch(arguments.length) {\n      case 1:\n        return flatmap__1.call(this, f);\n      case 2:\n        return flatmap__2.call(this, f, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  flatmap.cljs$core$IFn$_invoke$arity$1 = flatmap__1;\n  flatmap.cljs$core$IFn$_invoke$arity$2 = flatmap__2;\n  return flatmap;\n}();\ncljs.core.dedupe = function() {\n  var dedupe = null;\n  var dedupe__0 = function() {\n    return function(f1) {\n      var pa = cljs.core.atom.call(null, new cljs.core.Keyword(&quot;cljs.core&quot;, &quot;none&quot;, &quot;cljs.core/none&quot;, 926646439));\n      return function(pa) {\n        return function() {\n          var G__23130 = null;\n          var G__23130__0 = function() {\n            return f1.call(null);\n          };\n          var G__23130__1 = function(result) {\n            return f1.call(null, result);\n          };\n          var G__23130__2 = function(result, input) {\n            var prior = cljs.core.deref.call(null, pa);\n            cljs.core.reset_BANG_.call(null, pa, input);\n            if (cljs.core._EQ_.call(null, prior, input)) {\n              return result;\n            } else {\n              return f1.call(null, result, input);\n            }\n          };\n          G__23130 = function(result, input) {\n            switch(arguments.length) {\n              case 0:\n                return G__23130__0.call(this);\n              case 1:\n                return G__23130__1.call(this, result);\n              case 2:\n                return G__23130__2.call(this, result, input);\n            }\n            throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n          };\n          G__23130.cljs$core$IFn$_invoke$arity$0 = G__23130__0;\n          G__23130.cljs$core$IFn$_invoke$arity$1 = G__23130__1;\n          G__23130.cljs$core$IFn$_invoke$arity$2 = G__23130__2;\n          return G__23130;\n        }();\n      }(pa);\n    };\n  };\n  var dedupe__1 = function(coll) {\n    return cljs.core.sequence.call(null, dedupe.call(null), coll);\n  };\n  dedupe = function(coll) {\n    switch(arguments.length) {\n      case 0:\n        return dedupe__0.call(this);\n      case 1:\n        return dedupe__1.call(this, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  dedupe.cljs$core$IFn$_invoke$arity$0 = dedupe__0;\n  dedupe.cljs$core$IFn$_invoke$arity$1 = dedupe__1;\n  return dedupe;\n}();\ncljs.core.random_sample = function() {\n  var random_sample = null;\n  var random_sample__1 = function(prob) {\n    return cljs.core.filter.call(null, function(_) {\n      return cljs.core.rand.call(null) &lt; prob;\n    });\n  };\n  var random_sample__2 = function(prob, coll) {\n    return cljs.core.filter.call(null, function(_) {\n      return cljs.core.rand.call(null) &lt; prob;\n    }, coll);\n  };\n  random_sample = function(prob, coll) {\n    switch(arguments.length) {\n      case 1:\n        return random_sample__1.call(this, prob);\n      case 2:\n        return random_sample__2.call(this, prob, coll);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  random_sample.cljs$core$IFn$_invoke$arity$1 = random_sample__1;\n  random_sample.cljs$core$IFn$_invoke$arity$2 = random_sample__2;\n  return random_sample;\n}();\ncljs.core.Iteration = function(xform, coll) {\n  this.xform = xform;\n  this.coll = coll;\n  this.cljs$lang$protocol_mask$partition1$ = 0;\n  this.cljs$lang$protocol_mask$partition0$ = 2173173760;\n};\ncljs.core.Iteration.cljs$lang$type = true;\ncljs.core.Iteration.cljs$lang$ctorStr = &quot;cljs.core/Iteration&quot;;\ncljs.core.Iteration.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/Iteration&quot;);\n};\ncljs.core.Iteration.prototype.cljs$core$IPrintWithWriter$_pr_writer$arity$3 = function(coll__$1, writer, opts) {\n  var self__ = this;\n  var coll__$2 = this;\n  return cljs.core.pr_sequential_writer.call(null, writer, cljs.core.pr_writer, &quot;(&quot;, &quot; &quot;, &quot;)&quot;, opts, coll__$2);\n};\ncljs.core.Iteration.prototype.cljs$core$IReduce$_reduce$arity$3 = function(_, f, init) {\n  var self__ = this;\n  var ___$1 = this;\n  return cljs.core.transduce.call(null, self__.xform, f, init, self__.coll);\n};\ncljs.core.Iteration.prototype.cljs$core$ISeqable$_seq$arity$1 = function(_) {\n  var self__ = this;\n  var ___$1 = this;\n  return cljs.core.seq.call(null, cljs.core.sequence.call(null, self__.xform, self__.coll));\n};\ncljs.core.__GT_Iteration = function __GT_Iteration(xform, coll) {\n  return new cljs.core.Iteration(xform, coll);\n};\ncljs.core.iteration = function iteration(xform, coll) {\n  return new cljs.core.Iteration(xform, coll);\n};\ncljs.core.run_BANG_ = function run_BANG_(proc, coll) {\n  return cljs.core.reduce.call(null, function(p1__23132_SHARP_, p2__23131_SHARP_) {\n    return proc.call(null, p2__23131_SHARP_);\n  }, null, coll);\n};\ncljs.core.IEncodeJS = function() {\n  var obj23134 = {};\n  return obj23134;\n}();\ncljs.core._clj__GT_js = function _clj__GT_js(x) {\n  if (function() {\n    var and__17133__auto__ = x;\n    if (and__17133__auto__) {\n      return x.cljs$core$IEncodeJS$_clj__GT_js$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return x.cljs$core$IEncodeJS$_clj__GT_js$arity$1(x);\n  } else {\n    var x__17772__auto__ = x == null ? null : x;\n    return function() {\n      var or__17145__auto__ = cljs.core._clj__GT_js[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._clj__GT_js[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IEncodeJS.-clj-\\x3ejs&quot;, x);\n        }\n      }\n    }().call(null, x);\n  }\n};\ncljs.core._key__GT_js = function _key__GT_js(x) {\n  if (function() {\n    var and__17133__auto__ = x;\n    if (and__17133__auto__) {\n      return x.cljs$core$IEncodeJS$_key__GT_js$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return x.cljs$core$IEncodeJS$_key__GT_js$arity$1(x);\n  } else {\n    var x__17772__auto__ = x == null ? null : x;\n    return function() {\n      var or__17145__auto__ = cljs.core._key__GT_js[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._key__GT_js[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IEncodeJS.-key-\\x3ejs&quot;, x);\n        }\n      }\n    }().call(null, x);\n  }\n};\ncljs.core.key__GT_js = function key__GT_js(k) {\n  if (function() {\n    var G__23136 = k;\n    if (G__23136) {\n      var bit__17795__auto__ = null;\n      if (cljs.core.truth_(function() {\n        var or__17145__auto__ = bit__17795__auto__;\n        if (cljs.core.truth_(or__17145__auto__)) {\n          return or__17145__auto__;\n        } else {\n          return G__23136.cljs$core$IEncodeJS$;\n        }\n      }())) {\n        return true;\n      } else {\n        if (!G__23136.cljs$lang$protocol_mask$partition$) {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IEncodeJS, G__23136);\n        } else {\n          return false;\n        }\n      }\n    } else {\n      return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IEncodeJS, G__23136);\n    }\n  }()) {\n    return cljs.core._clj__GT_js.call(null, k);\n  } else {\n    if (typeof k === &quot;string&quot; || typeof k === &quot;number&quot; || k instanceof cljs.core.Keyword || k instanceof cljs.core.Symbol) {\n      return cljs.core.clj__GT_js.call(null, k);\n    } else {\n      return cljs.core.pr_str.call(null, k);\n    }\n  }\n};\ncljs.core.clj__GT_js = function clj__GT_js(x) {\n  if (x == null) {\n    return null;\n  } else {\n    if (function() {\n      var G__23150 = x;\n      if (G__23150) {\n        var bit__17795__auto__ = null;\n        if (cljs.core.truth_(function() {\n          var or__17145__auto__ = bit__17795__auto__;\n          if (cljs.core.truth_(or__17145__auto__)) {\n            return or__17145__auto__;\n          } else {\n            return G__23150.cljs$core$IEncodeJS$;\n          }\n        }())) {\n          return true;\n        } else {\n          if (!G__23150.cljs$lang$protocol_mask$partition$) {\n            return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IEncodeJS, G__23150);\n          } else {\n            return false;\n          }\n        }\n      } else {\n        return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IEncodeJS, G__23150);\n      }\n    }()) {\n      return cljs.core._clj__GT_js.call(null, x);\n    } else {\n      if (x instanceof cljs.core.Keyword) {\n        return cljs.core.name.call(null, x);\n      } else {\n        if (x instanceof cljs.core.Symbol) {\n          return &quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(x);\n        } else {\n          if (cljs.core.map_QMARK_.call(null, x)) {\n            var m = function() {\n              var obj23152 = {};\n              return obj23152;\n            }();\n            var seq__23153_23163 = cljs.core.seq.call(null, x);\n            var chunk__23154_23164 = null;\n            var count__23155_23165 = 0;\n            var i__23156_23166 = 0;\n            while (true) {\n              if (i__23156_23166 &lt; count__23155_23165) {\n                var vec__23157_23167 = cljs.core._nth.call(null, chunk__23154_23164, i__23156_23166);\n                var k_23168 = cljs.core.nth.call(null, vec__23157_23167, 0, null);\n                var v_23169 = cljs.core.nth.call(null, vec__23157_23167, 1, null);\n                m[cljs.core.key__GT_js.call(null, k_23168)] = clj__GT_js.call(null, v_23169);\n                var G__23170 = seq__23153_23163;\n                var G__23171 = chunk__23154_23164;\n                var G__23172 = count__23155_23165;\n                var G__23173 = i__23156_23166 + 1;\n                seq__23153_23163 = G__23170;\n                chunk__23154_23164 = G__23171;\n                count__23155_23165 = G__23172;\n                i__23156_23166 = G__23173;\n                continue;\n              } else {\n                var temp__4126__auto___23174 = cljs.core.seq.call(null, seq__23153_23163);\n                if (temp__4126__auto___23174) {\n                  var seq__23153_23175__$1 = temp__4126__auto___23174;\n                  if (cljs.core.chunked_seq_QMARK_.call(null, seq__23153_23175__$1)) {\n                    var c__17901__auto___23176 = cljs.core.chunk_first.call(null, seq__23153_23175__$1);\n                    var G__23177 = cljs.core.chunk_rest.call(null, seq__23153_23175__$1);\n                    var G__23178 = c__17901__auto___23176;\n                    var G__23179 = cljs.core.count.call(null, c__17901__auto___23176);\n                    var G__23180 = 0;\n                    seq__23153_23163 = G__23177;\n                    chunk__23154_23164 = G__23178;\n                    count__23155_23165 = G__23179;\n                    i__23156_23166 = G__23180;\n                    continue;\n                  } else {\n                    var vec__23158_23181 = cljs.core.first.call(null, seq__23153_23175__$1);\n                    var k_23182 = cljs.core.nth.call(null, vec__23158_23181, 0, null);\n                    var v_23183 = cljs.core.nth.call(null, vec__23158_23181, 1, null);\n                    m[cljs.core.key__GT_js.call(null, k_23182)] = clj__GT_js.call(null, v_23183);\n                    var G__23184 = cljs.core.next.call(null, seq__23153_23175__$1);\n                    var G__23185 = null;\n                    var G__23186 = 0;\n                    var G__23187 = 0;\n                    seq__23153_23163 = G__23184;\n                    chunk__23154_23164 = G__23185;\n                    count__23155_23165 = G__23186;\n                    i__23156_23166 = G__23187;\n                    continue;\n                  }\n                } else {\n                }\n              }\n              break;\n            }\n            return m;\n          } else {\n            if (cljs.core.coll_QMARK_.call(null, x)) {\n              var arr = [];\n              var seq__23159_23188 = cljs.core.seq.call(null, cljs.core.map.call(null, clj__GT_js, x));\n              var chunk__23160_23189 = null;\n              var count__23161_23190 = 0;\n              var i__23162_23191 = 0;\n              while (true) {\n                if (i__23162_23191 &lt; count__23161_23190) {\n                  var x_23192__$1 = cljs.core._nth.call(null, chunk__23160_23189, i__23162_23191);\n                  arr.push(x_23192__$1);\n                  var G__23193 = seq__23159_23188;\n                  var G__23194 = chunk__23160_23189;\n                  var G__23195 = count__23161_23190;\n                  var G__23196 = i__23162_23191 + 1;\n                  seq__23159_23188 = G__23193;\n                  chunk__23160_23189 = G__23194;\n                  count__23161_23190 = G__23195;\n                  i__23162_23191 = G__23196;\n                  continue;\n                } else {\n                  var temp__4126__auto___23197 = cljs.core.seq.call(null, seq__23159_23188);\n                  if (temp__4126__auto___23197) {\n                    var seq__23159_23198__$1 = temp__4126__auto___23197;\n                    if (cljs.core.chunked_seq_QMARK_.call(null, seq__23159_23198__$1)) {\n                      var c__17901__auto___23199 = cljs.core.chunk_first.call(null, seq__23159_23198__$1);\n                      var G__23200 = cljs.core.chunk_rest.call(null, seq__23159_23198__$1);\n                      var G__23201 = c__17901__auto___23199;\n                      var G__23202 = cljs.core.count.call(null, c__17901__auto___23199);\n                      var G__23203 = 0;\n                      seq__23159_23188 = G__23200;\n                      chunk__23160_23189 = G__23201;\n                      count__23161_23190 = G__23202;\n                      i__23162_23191 = G__23203;\n                      continue;\n                    } else {\n                      var x_23204__$1 = cljs.core.first.call(null, seq__23159_23198__$1);\n                      arr.push(x_23204__$1);\n                      var G__23205 = cljs.core.next.call(null, seq__23159_23198__$1);\n                      var G__23206 = null;\n                      var G__23207 = 0;\n                      var G__23208 = 0;\n                      seq__23159_23188 = G__23205;\n                      chunk__23160_23189 = G__23206;\n                      count__23161_23190 = G__23207;\n                      i__23162_23191 = G__23208;\n                      continue;\n                    }\n                  } else {\n                  }\n                }\n                break;\n              }\n              return arr;\n            } else {\n              return x;\n            }\n          }\n        }\n      }\n    }\n  }\n};\ncljs.core.IEncodeClojure = function() {\n  var obj23210 = {};\n  return obj23210;\n}();\ncljs.core._js__GT_clj = function _js__GT_clj(x, options) {\n  if (function() {\n    var and__17133__auto__ = x;\n    if (and__17133__auto__) {\n      return x.cljs$core$IEncodeClojure$_js__GT_clj$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return x.cljs$core$IEncodeClojure$_js__GT_clj$arity$2(x, options);\n  } else {\n    var x__17772__auto__ = x == null ? null : x;\n    return function() {\n      var or__17145__auto__ = cljs.core._js__GT_clj[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._js__GT_clj[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IEncodeClojure.-js-\\x3eclj&quot;, x);\n        }\n      }\n    }().call(null, x, options);\n  }\n};\ncljs.core.js__GT_clj = function() {\n  var js__GT_clj = null;\n  var js__GT_clj__1 = function(x) {\n    return js__GT_clj.call(null, x, new cljs.core.PersistentArrayMap(null, 1, [new cljs.core.Keyword(null, &quot;keywordize-keys&quot;, &quot;keywordize-keys&quot;, 1310784252), false], null));\n  };\n  var js__GT_clj__2 = function() {\n    var G__23231__delegate = function(x, opts) {\n      if (function() {\n        var G__23221 = x;\n        if (G__23221) {\n          var bit__17795__auto__ = null;\n          if (cljs.core.truth_(function() {\n            var or__17145__auto__ = bit__17795__auto__;\n            if (cljs.core.truth_(or__17145__auto__)) {\n              return or__17145__auto__;\n            } else {\n              return G__23221.cljs$core$IEncodeClojure$;\n            }\n          }())) {\n            return true;\n          } else {\n            if (!G__23221.cljs$lang$protocol_mask$partition$) {\n              return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IEncodeClojure, G__23221);\n            } else {\n              return false;\n            }\n          }\n        } else {\n          return cljs.core.native_satisfies_QMARK_.call(null, cljs.core.IEncodeClojure, G__23221);\n        }\n      }()) {\n        return cljs.core._js__GT_clj.call(null, x, cljs.core.apply.call(null, cljs.core.array_map, opts));\n      } else {\n        if (cljs.core.seq.call(null, opts)) {\n          var map__23222 = opts;\n          var map__23222__$1 = cljs.core.seq_QMARK_.call(null, map__23222) ? cljs.core.apply.call(null, cljs.core.hash_map, map__23222) : map__23222;\n          var keywordize_keys = cljs.core.get.call(null, map__23222__$1, new cljs.core.Keyword(null, &quot;keywordize-keys&quot;, &quot;keywordize-keys&quot;, 1310784252));\n          var keyfn = cljs.core.truth_(keywordize_keys) ? cljs.core.keyword : cljs.core.str;\n          var f = function(map__23222, map__23222__$1, keywordize_keys, keyfn) {\n            return function thisfn(x__$1) {\n              if (cljs.core.seq_QMARK_.call(null, x__$1)) {\n                return cljs.core.doall.call(null, cljs.core.map.call(null, thisfn, x__$1));\n              } else {\n                if (cljs.core.coll_QMARK_.call(null, x__$1)) {\n                  return cljs.core.into.call(null, cljs.core.empty.call(null, x__$1), cljs.core.map.call(null, thisfn, x__$1));\n                } else {\n                  if (x__$1 instanceof Array) {\n                    return cljs.core.vec.call(null, cljs.core.map.call(null, thisfn, x__$1));\n                  } else {\n                    if (cljs.core.type.call(null, x__$1) === Object) {\n                      return cljs.core.into.call(null, cljs.core.PersistentArrayMap.EMPTY, function() {\n                        var iter__17870__auto__ = function(map__23222, map__23222__$1, keywordize_keys, keyfn) {\n                          return function iter__23227(s__23228) {\n                            return new cljs.core.LazySeq(null, function(map__23222, map__23222__$1, keywordize_keys, keyfn) {\n                              return function() {\n                                var s__23228__$1 = s__23228;\n                                while (true) {\n                                  var temp__4126__auto__ = cljs.core.seq.call(null, s__23228__$1);\n                                  if (temp__4126__auto__) {\n                                    var s__23228__$2 = temp__4126__auto__;\n                                    if (cljs.core.chunked_seq_QMARK_.call(null, s__23228__$2)) {\n                                      var c__17868__auto__ = cljs.core.chunk_first.call(null, s__23228__$2);\n                                      var size__17869__auto__ = cljs.core.count.call(null, c__17868__auto__);\n                                      var b__23230 = cljs.core.chunk_buffer.call(null, size__17869__auto__);\n                                      if (function() {\n                                        var i__23229 = 0;\n                                        while (true) {\n                                          if (i__23229 &lt; size__17869__auto__) {\n                                            var k = cljs.core._nth.call(null, c__17868__auto__, i__23229);\n                                            cljs.core.chunk_append.call(null, b__23230, new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [keyfn.call(null, k), thisfn.call(null, x__$1[k])], null));\n                                            var G__23232 = i__23229 + 1;\n                                            i__23229 = G__23232;\n                                            continue;\n                                          } else {\n                                            return true;\n                                          }\n                                          break;\n                                        }\n                                      }()) {\n                                        return cljs.core.chunk_cons.call(null, cljs.core.chunk.call(null, b__23230), iter__23227.call(null, cljs.core.chunk_rest.call(null, s__23228__$2)));\n                                      } else {\n                                        return cljs.core.chunk_cons.call(null, cljs.core.chunk.call(null, b__23230), null);\n                                      }\n                                    } else {\n                                      var k = cljs.core.first.call(null, s__23228__$2);\n                                      return cljs.core.cons.call(null, new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [keyfn.call(null, k), thisfn.call(null, x__$1[k])], null), iter__23227.call(null, cljs.core.rest.call(null, s__23228__$2)));\n                                    }\n                                  } else {\n                                    return null;\n                                  }\n                                  break;\n                                }\n                              };\n                            }(map__23222, map__23222__$1, keywordize_keys, keyfn), null, null);\n                          };\n                        }(map__23222, map__23222__$1, keywordize_keys, keyfn);\n                        return iter__17870__auto__.call(null, cljs.core.js_keys.call(null, x__$1));\n                      }());\n                    } else {\n                      return x__$1;\n                    }\n                  }\n                }\n              }\n            };\n          }(map__23222, map__23222__$1, keywordize_keys, keyfn);\n          return f.call(null, x);\n        } else {\n          return null;\n        }\n      }\n    };\n    var G__23231 = function(x, var_args) {\n      var opts = null;\n      if (arguments.length &gt; 1) {\n        opts = cljs.core.array_seq(Array.prototype.slice.call(arguments, 1), 0);\n      }\n      return G__23231__delegate.call(this, x, opts);\n    };\n    G__23231.cljs$lang$maxFixedArity = 1;\n    G__23231.cljs$lang$applyTo = function(arglist__23233) {\n      var x = cljs.core.first(arglist__23233);\n      var opts = cljs.core.rest(arglist__23233);\n      return G__23231__delegate(x, opts);\n    };\n    G__23231.cljs$core$IFn$_invoke$arity$variadic = G__23231__delegate;\n    return G__23231;\n  }();\n  js__GT_clj = function(x, var_args) {\n    var opts = var_args;\n    switch(arguments.length) {\n      case 1:\n        return js__GT_clj__1.call(this, x);\n      default:\n        return js__GT_clj__2.cljs$core$IFn$_invoke$arity$variadic(x, cljs.core.array_seq(arguments, 1));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  js__GT_clj.cljs$lang$maxFixedArity = 1;\n  js__GT_clj.cljs$lang$applyTo = js__GT_clj__2.cljs$lang$applyTo;\n  js__GT_clj.cljs$core$IFn$_invoke$arity$1 = js__GT_clj__1;\n  js__GT_clj.cljs$core$IFn$_invoke$arity$variadic = js__GT_clj__2.cljs$core$IFn$_invoke$arity$variadic;\n  return js__GT_clj;\n}();\ncljs.core.memoize = function memoize(f) {\n  var mem = cljs.core.atom.call(null, cljs.core.PersistentArrayMap.EMPTY);\n  return function(mem) {\n    return function() {\n      var G__23234__delegate = function(args) {\n        var v = cljs.core.get.call(null, cljs.core.deref.call(null, mem), args, cljs.core.lookup_sentinel);\n        if (v === cljs.core.lookup_sentinel) {\n          var ret = cljs.core.apply.call(null, f, args);\n          cljs.core.swap_BANG_.call(null, mem, cljs.core.assoc, args, ret);\n          return ret;\n        } else {\n          return v;\n        }\n      };\n      var G__23234 = function(var_args) {\n        var args = null;\n        if (arguments.length &gt; 0) {\n          args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 0), 0);\n        }\n        return G__23234__delegate.call(this, args);\n      };\n      G__23234.cljs$lang$maxFixedArity = 0;\n      G__23234.cljs$lang$applyTo = function(arglist__23235) {\n        var args = cljs.core.seq(arglist__23235);\n        return G__23234__delegate(args);\n      };\n      G__23234.cljs$core$IFn$_invoke$arity$variadic = G__23234__delegate;\n      return G__23234;\n    }();\n  }(mem);\n};\ncljs.core.trampoline = function() {\n  var trampoline = null;\n  var trampoline__1 = function(f) {\n    while (true) {\n      var ret = f.call(null);\n      if (cljs.core.fn_QMARK_.call(null, ret)) {\n        var G__23236 = ret;\n        f = G__23236;\n        continue;\n      } else {\n        return ret;\n      }\n      break;\n    }\n  };\n  var trampoline__2 = function() {\n    var G__23237__delegate = function(f, args) {\n      return trampoline.call(null, function() {\n        return cljs.core.apply.call(null, f, args);\n      });\n    };\n    var G__23237 = function(f, var_args) {\n      var args = null;\n      if (arguments.length &gt; 1) {\n        args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 1), 0);\n      }\n      return G__23237__delegate.call(this, f, args);\n    };\n    G__23237.cljs$lang$maxFixedArity = 1;\n    G__23237.cljs$lang$applyTo = function(arglist__23238) {\n      var f = cljs.core.first(arglist__23238);\n      var args = cljs.core.rest(arglist__23238);\n      return G__23237__delegate(f, args);\n    };\n    G__23237.cljs$core$IFn$_invoke$arity$variadic = G__23237__delegate;\n    return G__23237;\n  }();\n  trampoline = function(f, var_args) {\n    var args = var_args;\n    switch(arguments.length) {\n      case 1:\n        return trampoline__1.call(this, f);\n      default:\n        return trampoline__2.cljs$core$IFn$_invoke$arity$variadic(f, cljs.core.array_seq(arguments, 1));\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  trampoline.cljs$lang$maxFixedArity = 1;\n  trampoline.cljs$lang$applyTo = trampoline__2.cljs$lang$applyTo;\n  trampoline.cljs$core$IFn$_invoke$arity$1 = trampoline__1;\n  trampoline.cljs$core$IFn$_invoke$arity$variadic = trampoline__2.cljs$core$IFn$_invoke$arity$variadic;\n  return trampoline;\n}();\ncljs.core.rand = function() {\n  var rand = null;\n  var rand__0 = function() {\n    return rand.call(null, 1);\n  };\n  var rand__1 = function(n) {\n    return Math.random.call(null) * n;\n  };\n  rand = function(n) {\n    switch(arguments.length) {\n      case 0:\n        return rand__0.call(this);\n      case 1:\n        return rand__1.call(this, n);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  rand.cljs$core$IFn$_invoke$arity$0 = rand__0;\n  rand.cljs$core$IFn$_invoke$arity$1 = rand__1;\n  return rand;\n}();\ncljs.core.rand_int = function rand_int(n) {\n  return Math.floor.call(null, Math.random.call(null) * n);\n};\ncljs.core.rand_nth = function rand_nth(coll) {\n  return cljs.core.nth.call(null, coll, cljs.core.rand_int.call(null, cljs.core.count.call(null, coll)));\n};\ncljs.core.group_by = function group_by(f, coll) {\n  return cljs.core.reduce.call(null, function(ret, x) {\n    var k = f.call(null, x);\n    return cljs.core.assoc.call(null, ret, k, cljs.core.conj.call(null, cljs.core.get.call(null, ret, k, cljs.core.PersistentVector.EMPTY), x));\n  }, cljs.core.PersistentArrayMap.EMPTY, coll);\n};\ncljs.core.make_hierarchy = function make_hierarchy() {\n  return new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null, &quot;parents&quot;, &quot;parents&quot;, -2027538891), cljs.core.PersistentArrayMap.EMPTY, new cljs.core.Keyword(null, &quot;descendants&quot;, &quot;descendants&quot;, 1824886031), cljs.core.PersistentArrayMap.EMPTY, new cljs.core.Keyword(null, &quot;ancestors&quot;, &quot;ancestors&quot;, -776045424), cljs.core.PersistentArrayMap.EMPTY], null);\n};\ncljs.core._global_hierarchy = null;\ncljs.core.get_global_hierarchy = function get_global_hierarchy() {\n  if (cljs.core._global_hierarchy == null) {\n    cljs.core._global_hierarchy = cljs.core.atom.call(null, cljs.core.make_hierarchy.call(null));\n  } else {\n  }\n  return cljs.core._global_hierarchy;\n};\ncljs.core.swap_global_hierarchy_BANG_ = function() {\n  var swap_global_hierarchy_BANG___delegate = function(f, args) {\n    return cljs.core.apply.call(null, cljs.core.swap_BANG_, cljs.core.get_global_hierarchy.call(null), f, args);\n  };\n  var swap_global_hierarchy_BANG_ = function(f, var_args) {\n    var args = null;\n    if (arguments.length &gt; 1) {\n      args = cljs.core.array_seq(Array.prototype.slice.call(arguments, 1), 0);\n    }\n    return swap_global_hierarchy_BANG___delegate.call(this, f, args);\n  };\n  swap_global_hierarchy_BANG_.cljs$lang$maxFixedArity = 1;\n  swap_global_hierarchy_BANG_.cljs$lang$applyTo = function(arglist__23239) {\n    var f = cljs.core.first(arglist__23239);\n    var args = cljs.core.rest(arglist__23239);\n    return swap_global_hierarchy_BANG___delegate(f, args);\n  };\n  swap_global_hierarchy_BANG_.cljs$core$IFn$_invoke$arity$variadic = swap_global_hierarchy_BANG___delegate;\n  return swap_global_hierarchy_BANG_;\n}();\ncljs.core.isa_QMARK_ = function() {\n  var isa_QMARK_ = null;\n  var isa_QMARK___2 = function(child, parent) {\n    return isa_QMARK_.call(null, cljs.core.deref.call(null, cljs.core.get_global_hierarchy.call(null)), child, parent);\n  };\n  var isa_QMARK___3 = function(h, child, parent) {\n    var or__17145__auto__ = cljs.core._EQ_.call(null, child, parent);\n    if (or__17145__auto__) {\n      return or__17145__auto__;\n    } else {\n      var or__17145__auto____$1 = cljs.core.contains_QMARK_.call(null, (new cljs.core.Keyword(null, &quot;ancestors&quot;, &quot;ancestors&quot;, -776045424)).cljs$core$IFn$_invoke$arity$1(h).call(null, child), parent);\n      if (or__17145__auto____$1) {\n        return or__17145__auto____$1;\n      } else {\n        var and__17133__auto__ = cljs.core.vector_QMARK_.call(null, parent);\n        if (and__17133__auto__) {\n          var and__17133__auto____$1 = cljs.core.vector_QMARK_.call(null, child);\n          if (and__17133__auto____$1) {\n            var and__17133__auto____$2 = cljs.core.count.call(null, parent) === cljs.core.count.call(null, child);\n            if (and__17133__auto____$2) {\n              var ret = true;\n              var i = 0;\n              while (true) {\n                if (!ret || i === cljs.core.count.call(null, parent)) {\n                  return ret;\n                } else {\n                  var G__23240 = isa_QMARK_.call(null, h, child.call(null, i), parent.call(null, i));\n                  var G__23241 = i + 1;\n                  ret = G__23240;\n                  i = G__23241;\n                  continue;\n                }\n                break;\n              }\n            } else {\n              return and__17133__auto____$2;\n            }\n          } else {\n            return and__17133__auto____$1;\n          }\n        } else {\n          return and__17133__auto__;\n        }\n      }\n    }\n  };\n  isa_QMARK_ = function(h, child, parent) {\n    switch(arguments.length) {\n      case 2:\n        return isa_QMARK___2.call(this, h, child);\n      case 3:\n        return isa_QMARK___3.call(this, h, child, parent);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  isa_QMARK_.cljs$core$IFn$_invoke$arity$2 = isa_QMARK___2;\n  isa_QMARK_.cljs$core$IFn$_invoke$arity$3 = isa_QMARK___3;\n  return isa_QMARK_;\n}();\ncljs.core.parents = function() {\n  var parents = null;\n  var parents__1 = function(tag) {\n    return parents.call(null, cljs.core.deref.call(null, cljs.core.get_global_hierarchy.call(null)), tag);\n  };\n  var parents__2 = function(h, tag) {\n    return cljs.core.not_empty.call(null, cljs.core.get.call(null, (new cljs.core.Keyword(null, &quot;parents&quot;, &quot;parents&quot;, -2027538891)).cljs$core$IFn$_invoke$arity$1(h), tag));\n  };\n  parents = function(h, tag) {\n    switch(arguments.length) {\n      case 1:\n        return parents__1.call(this, h);\n      case 2:\n        return parents__2.call(this, h, tag);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  parents.cljs$core$IFn$_invoke$arity$1 = parents__1;\n  parents.cljs$core$IFn$_invoke$arity$2 = parents__2;\n  return parents;\n}();\ncljs.core.ancestors = function() {\n  var ancestors = null;\n  var ancestors__1 = function(tag) {\n    return ancestors.call(null, cljs.core.deref.call(null, cljs.core.get_global_hierarchy.call(null)), tag);\n  };\n  var ancestors__2 = function(h, tag) {\n    return cljs.core.not_empty.call(null, cljs.core.get.call(null, (new cljs.core.Keyword(null, &quot;ancestors&quot;, &quot;ancestors&quot;, -776045424)).cljs$core$IFn$_invoke$arity$1(h), tag));\n  };\n  ancestors = function(h, tag) {\n    switch(arguments.length) {\n      case 1:\n        return ancestors__1.call(this, h);\n      case 2:\n        return ancestors__2.call(this, h, tag);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  ancestors.cljs$core$IFn$_invoke$arity$1 = ancestors__1;\n  ancestors.cljs$core$IFn$_invoke$arity$2 = ancestors__2;\n  return ancestors;\n}();\ncljs.core.descendants = function() {\n  var descendants = null;\n  var descendants__1 = function(tag) {\n    return descendants.call(null, cljs.core.deref.call(null, cljs.core.get_global_hierarchy.call(null)), tag);\n  };\n  var descendants__2 = function(h, tag) {\n    return cljs.core.not_empty.call(null, cljs.core.get.call(null, (new cljs.core.Keyword(null, &quot;descendants&quot;, &quot;descendants&quot;, 1824886031)).cljs$core$IFn$_invoke$arity$1(h), tag));\n  };\n  descendants = function(h, tag) {\n    switch(arguments.length) {\n      case 1:\n        return descendants__1.call(this, h);\n      case 2:\n        return descendants__2.call(this, h, tag);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  descendants.cljs$core$IFn$_invoke$arity$1 = descendants__1;\n  descendants.cljs$core$IFn$_invoke$arity$2 = descendants__2;\n  return descendants;\n}();\ncljs.core.derive = function() {\n  var derive = null;\n  var derive__2 = function(tag, parent) {\n    if (cljs.core.truth_(cljs.core.namespace.call(null, parent))) {\n    } else {\n      throw new Error(&quot;Assert failed: &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.pr_str.call(null, cljs.core.list(new cljs.core.Symbol(null, &quot;namespace&quot;, &quot;namespace&quot;, 1263021155, null), new cljs.core.Symbol(null, &quot;parent&quot;, &quot;parent&quot;, 761652748, null)))));\n    }\n    cljs.core.swap_global_hierarchy_BANG_.call(null, derive, tag, parent);\n    return null;\n  };\n  var derive__3 = function(h, tag, parent) {\n    if (cljs.core.not_EQ_.call(null, tag, parent)) {\n    } else {\n      throw new Error(&quot;Assert failed: &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.pr_str.call(null, cljs.core.list(new cljs.core.Symbol(null, &quot;not\\x3d&quot;, &quot;not\\x3d&quot;, 1466536204, null), new cljs.core.Symbol(null, &quot;tag&quot;, &quot;tag&quot;, 350170304, null), new cljs.core.Symbol(null, &quot;parent&quot;, &quot;parent&quot;, 761652748, null)))));\n    }\n    var tp = (new cljs.core.Keyword(null, &quot;parents&quot;, &quot;parents&quot;, -2027538891)).cljs$core$IFn$_invoke$arity$1(h);\n    var td = (new cljs.core.Keyword(null, &quot;descendants&quot;, &quot;descendants&quot;, 1824886031)).cljs$core$IFn$_invoke$arity$1(h);\n    var ta = (new cljs.core.Keyword(null, &quot;ancestors&quot;, &quot;ancestors&quot;, -776045424)).cljs$core$IFn$_invoke$arity$1(h);\n    var tf = function(tp, td, ta) {\n      return function(m, source, sources, target, targets) {\n        return cljs.core.reduce.call(null, function(tp, td, ta) {\n          return function(ret, k) {\n            return cljs.core.assoc.call(null, ret, k, cljs.core.reduce.call(null, cljs.core.conj, cljs.core.get.call(null, targets, k, cljs.core.PersistentHashSet.EMPTY), cljs.core.cons.call(null, target, targets.call(null, target))));\n          };\n        }(tp, td, ta), m, cljs.core.cons.call(null, source, sources.call(null, source)));\n      };\n    }(tp, td, ta);\n    var or__17145__auto__ = cljs.core.contains_QMARK_.call(null, tp.call(null, tag), parent) ? null : function() {\n      if (cljs.core.contains_QMARK_.call(null, ta.call(null, tag), parent)) {\n        throw new Error(&quot;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(tag) + &quot;already has&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(parent) + &quot;as ancestor&quot;);\n      } else {\n      }\n      if (cljs.core.contains_QMARK_.call(null, ta.call(null, parent), tag)) {\n        throw new Error(&quot;Cyclic derivation:&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(parent) + &quot;has&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(tag) + &quot;as ancestor&quot;);\n      } else {\n      }\n      return new cljs.core.PersistentArrayMap(null, 3, [new cljs.core.Keyword(null, &quot;parents&quot;, &quot;parents&quot;, -2027538891), cljs.core.assoc.call(null, (new cljs.core.Keyword(null, &quot;parents&quot;, &quot;parents&quot;, -2027538891)).cljs$core$IFn$_invoke$arity$1(h), tag, cljs.core.conj.call(null, cljs.core.get.call(null, tp, tag, cljs.core.PersistentHashSet.EMPTY), parent)), new cljs.core.Keyword(null, &quot;ancestors&quot;, &quot;ancestors&quot;, -776045424), tf.call(null, (new cljs.core.Keyword(null, &quot;ancestors&quot;, &quot;ancestors&quot;, -776045424)).cljs$core$IFn$_invoke$arity$1(h), \n      tag, td, parent, ta), new cljs.core.Keyword(null, &quot;descendants&quot;, &quot;descendants&quot;, 1824886031), tf.call(null, (new cljs.core.Keyword(null, &quot;descendants&quot;, &quot;descendants&quot;, 1824886031)).cljs$core$IFn$_invoke$arity$1(h), parent, ta, tag, td)], null);\n    }();\n    if (cljs.core.truth_(or__17145__auto__)) {\n      return or__17145__auto__;\n    } else {\n      return h;\n    }\n  };\n  derive = function(h, tag, parent) {\n    switch(arguments.length) {\n      case 2:\n        return derive__2.call(this, h, tag);\n      case 3:\n        return derive__3.call(this, h, tag, parent);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  derive.cljs$core$IFn$_invoke$arity$2 = derive__2;\n  derive.cljs$core$IFn$_invoke$arity$3 = derive__3;\n  return derive;\n}();\ncljs.core.underive = function() {\n  var underive = null;\n  var underive__2 = function(tag, parent) {\n    cljs.core.swap_global_hierarchy_BANG_.call(null, underive, tag, parent);\n    return null;\n  };\n  var underive__3 = function(h, tag, parent) {\n    var parentMap = (new cljs.core.Keyword(null, &quot;parents&quot;, &quot;parents&quot;, -2027538891)).cljs$core$IFn$_invoke$arity$1(h);\n    var childsParents = cljs.core.truth_(parentMap.call(null, tag)) ? cljs.core.disj.call(null, parentMap.call(null, tag), parent) : cljs.core.PersistentHashSet.EMPTY;\n    var newParents = cljs.core.truth_(cljs.core.not_empty.call(null, childsParents)) ? cljs.core.assoc.call(null, parentMap, tag, childsParents) : cljs.core.dissoc.call(null, parentMap, tag);\n    var deriv_seq = cljs.core.flatten.call(null, cljs.core.map.call(null, function(parentMap, childsParents, newParents) {\n      return function(p1__23242_SHARP_) {\n        return cljs.core.cons.call(null, cljs.core.first.call(null, p1__23242_SHARP_), cljs.core.interpose.call(null, cljs.core.first.call(null, p1__23242_SHARP_), cljs.core.second.call(null, p1__23242_SHARP_)));\n      };\n    }(parentMap, childsParents, newParents), cljs.core.seq.call(null, newParents)));\n    if (cljs.core.contains_QMARK_.call(null, parentMap.call(null, tag), parent)) {\n      return cljs.core.reduce.call(null, function(parentMap, childsParents, newParents, deriv_seq) {\n        return function(p1__23243_SHARP_, p2__23244_SHARP_) {\n          return cljs.core.apply.call(null, cljs.core.derive, p1__23243_SHARP_, p2__23244_SHARP_);\n        };\n      }(parentMap, childsParents, newParents, deriv_seq), cljs.core.make_hierarchy.call(null), cljs.core.partition.call(null, 2, deriv_seq));\n    } else {\n      return h;\n    }\n  };\n  underive = function(h, tag, parent) {\n    switch(arguments.length) {\n      case 2:\n        return underive__2.call(this, h, tag);\n      case 3:\n        return underive__3.call(this, h, tag, parent);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  underive.cljs$core$IFn$_invoke$arity$2 = underive__2;\n  underive.cljs$core$IFn$_invoke$arity$3 = underive__3;\n  return underive;\n}();\ncljs.core.reset_cache = function reset_cache(method_cache, method_table, cached_hierarchy, hierarchy) {\n  cljs.core.swap_BANG_.call(null, method_cache, function(_) {\n    return cljs.core.deref.call(null, method_table);\n  });\n  return cljs.core.swap_BANG_.call(null, cached_hierarchy, function(_) {\n    return cljs.core.deref.call(null, hierarchy);\n  });\n};\ncljs.core.prefers_STAR_ = function prefers_STAR_(x, y, prefer_table) {\n  var xprefs = cljs.core.deref.call(null, prefer_table).call(null, x);\n  var or__17145__auto__ = cljs.core.truth_(function() {\n    var and__17133__auto__ = xprefs;\n    if (cljs.core.truth_(and__17133__auto__)) {\n      return xprefs.call(null, y);\n    } else {\n      return and__17133__auto__;\n    }\n  }()) ? true : null;\n  if (cljs.core.truth_(or__17145__auto__)) {\n    return or__17145__auto__;\n  } else {\n    var or__17145__auto____$1 = function() {\n      var ps = cljs.core.parents.call(null, y);\n      while (true) {\n        if (cljs.core.count.call(null, ps) &gt; 0) {\n          if (cljs.core.truth_(prefers_STAR_.call(null, x, cljs.core.first.call(null, ps), prefer_table))) {\n          } else {\n          }\n          var G__23245 = cljs.core.rest.call(null, ps);\n          ps = G__23245;\n          continue;\n        } else {\n          return null;\n        }\n        break;\n      }\n    }();\n    if (cljs.core.truth_(or__17145__auto____$1)) {\n      return or__17145__auto____$1;\n    } else {\n      var or__17145__auto____$2 = function() {\n        var ps = cljs.core.parents.call(null, x);\n        while (true) {\n          if (cljs.core.count.call(null, ps) &gt; 0) {\n            if (cljs.core.truth_(prefers_STAR_.call(null, cljs.core.first.call(null, ps), y, prefer_table))) {\n            } else {\n            }\n            var G__23246 = cljs.core.rest.call(null, ps);\n            ps = G__23246;\n            continue;\n          } else {\n            return null;\n          }\n          break;\n        }\n      }();\n      if (cljs.core.truth_(or__17145__auto____$2)) {\n        return or__17145__auto____$2;\n      } else {\n        return false;\n      }\n    }\n  }\n};\ncljs.core.dominates = function dominates(x, y, prefer_table) {\n  var or__17145__auto__ = cljs.core.prefers_STAR_.call(null, x, y, prefer_table);\n  if (cljs.core.truth_(or__17145__auto__)) {\n    return or__17145__auto__;\n  } else {\n    return cljs.core.isa_QMARK_.call(null, x, y);\n  }\n};\ncljs.core.find_and_cache_best_method = function find_and_cache_best_method(name, dispatch_val, hierarchy, method_table, prefer_table, method_cache, cached_hierarchy) {\n  var best_entry = cljs.core.reduce.call(null, function(be, p__23249) {\n    var vec__23250 = p__23249;\n    var k = cljs.core.nth.call(null, vec__23250, 0, null);\n    var _ = cljs.core.nth.call(null, vec__23250, 1, null);\n    var e = vec__23250;\n    if (cljs.core.isa_QMARK_.call(null, cljs.core.deref.call(null, hierarchy), dispatch_val, k)) {\n      var be2 = cljs.core.truth_(function() {\n        var or__17145__auto__ = be == null;\n        if (or__17145__auto__) {\n          return or__17145__auto__;\n        } else {\n          return cljs.core.dominates.call(null, k, cljs.core.first.call(null, be), prefer_table);\n        }\n      }()) ? e : be;\n      if (cljs.core.truth_(cljs.core.dominates.call(null, cljs.core.first.call(null, be2), k, prefer_table))) {\n      } else {\n        throw new Error(&quot;Multiple methods in multimethod &apos;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(name) + &quot;&apos; match dispatch value: &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(dispatch_val) + &quot; -\\x3e &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(k) + &quot; and &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(cljs.core.first.call(null, be2)) + &quot;, and neither is preferred&quot;);\n      }\n      return be2;\n    } else {\n      return be;\n    }\n  }, null, cljs.core.deref.call(null, method_table));\n  if (cljs.core.truth_(best_entry)) {\n    if (cljs.core._EQ_.call(null, cljs.core.deref.call(null, cached_hierarchy), cljs.core.deref.call(null, hierarchy))) {\n      cljs.core.swap_BANG_.call(null, method_cache, cljs.core.assoc, dispatch_val, cljs.core.second.call(null, best_entry));\n      return cljs.core.second.call(null, best_entry);\n    } else {\n      cljs.core.reset_cache.call(null, method_cache, method_table, cached_hierarchy, hierarchy);\n      return find_and_cache_best_method.call(null, name, dispatch_val, hierarchy, method_table, prefer_table, method_cache, cached_hierarchy);\n    }\n  } else {\n    return null;\n  }\n};\ncljs.core.IMultiFn = function() {\n  var obj23252 = {};\n  return obj23252;\n}();\ncljs.core._reset = function _reset(mf) {\n  if (function() {\n    var and__17133__auto__ = mf;\n    if (and__17133__auto__) {\n      return mf.cljs$core$IMultiFn$_reset$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return mf.cljs$core$IMultiFn$_reset$arity$1(mf);\n  } else {\n    var x__17772__auto__ = mf == null ? null : mf;\n    return function() {\n      var or__17145__auto__ = cljs.core._reset[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._reset[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMultiFn.-reset&quot;, mf);\n        }\n      }\n    }().call(null, mf);\n  }\n};\ncljs.core._add_method = function _add_method(mf, dispatch_val, method) {\n  if (function() {\n    var and__17133__auto__ = mf;\n    if (and__17133__auto__) {\n      return mf.cljs$core$IMultiFn$_add_method$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return mf.cljs$core$IMultiFn$_add_method$arity$3(mf, dispatch_val, method);\n  } else {\n    var x__17772__auto__ = mf == null ? null : mf;\n    return function() {\n      var or__17145__auto__ = cljs.core._add_method[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._add_method[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMultiFn.-add-method&quot;, mf);\n        }\n      }\n    }().call(null, mf, dispatch_val, method);\n  }\n};\ncljs.core._remove_method = function _remove_method(mf, dispatch_val) {\n  if (function() {\n    var and__17133__auto__ = mf;\n    if (and__17133__auto__) {\n      return mf.cljs$core$IMultiFn$_remove_method$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return mf.cljs$core$IMultiFn$_remove_method$arity$2(mf, dispatch_val);\n  } else {\n    var x__17772__auto__ = mf == null ? null : mf;\n    return function() {\n      var or__17145__auto__ = cljs.core._remove_method[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._remove_method[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMultiFn.-remove-method&quot;, mf);\n        }\n      }\n    }().call(null, mf, dispatch_val);\n  }\n};\ncljs.core._prefer_method = function _prefer_method(mf, dispatch_val, dispatch_val_y) {\n  if (function() {\n    var and__17133__auto__ = mf;\n    if (and__17133__auto__) {\n      return mf.cljs$core$IMultiFn$_prefer_method$arity$3;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return mf.cljs$core$IMultiFn$_prefer_method$arity$3(mf, dispatch_val, dispatch_val_y);\n  } else {\n    var x__17772__auto__ = mf == null ? null : mf;\n    return function() {\n      var or__17145__auto__ = cljs.core._prefer_method[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._prefer_method[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMultiFn.-prefer-method&quot;, mf);\n        }\n      }\n    }().call(null, mf, dispatch_val, dispatch_val_y);\n  }\n};\ncljs.core._get_method = function _get_method(mf, dispatch_val) {\n  if (function() {\n    var and__17133__auto__ = mf;\n    if (and__17133__auto__) {\n      return mf.cljs$core$IMultiFn$_get_method$arity$2;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return mf.cljs$core$IMultiFn$_get_method$arity$2(mf, dispatch_val);\n  } else {\n    var x__17772__auto__ = mf == null ? null : mf;\n    return function() {\n      var or__17145__auto__ = cljs.core._get_method[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._get_method[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMultiFn.-get-method&quot;, mf);\n        }\n      }\n    }().call(null, mf, dispatch_val);\n  }\n};\ncljs.core._methods = function _methods(mf) {\n  if (function() {\n    var and__17133__auto__ = mf;\n    if (and__17133__auto__) {\n      return mf.cljs$core$IMultiFn$_methods$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return mf.cljs$core$IMultiFn$_methods$arity$1(mf);\n  } else {\n    var x__17772__auto__ = mf == null ? null : mf;\n    return function() {\n      var or__17145__auto__ = cljs.core._methods[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._methods[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMultiFn.-methods&quot;, mf);\n        }\n      }\n    }().call(null, mf);\n  }\n};\ncljs.core._prefers = function _prefers(mf) {\n  if (function() {\n    var and__17133__auto__ = mf;\n    if (and__17133__auto__) {\n      return mf.cljs$core$IMultiFn$_prefers$arity$1;\n    } else {\n      return and__17133__auto__;\n    }\n  }()) {\n    return mf.cljs$core$IMultiFn$_prefers$arity$1(mf);\n  } else {\n    var x__17772__auto__ = mf == null ? null : mf;\n    return function() {\n      var or__17145__auto__ = cljs.core._prefers[goog.typeOf(x__17772__auto__)];\n      if (or__17145__auto__) {\n        return or__17145__auto__;\n      } else {\n        var or__17145__auto____$1 = cljs.core._prefers[&quot;_&quot;];\n        if (or__17145__auto____$1) {\n          return or__17145__auto____$1;\n        } else {\n          throw cljs.core.missing_protocol.call(null, &quot;IMultiFn.-prefers&quot;, mf);\n        }\n      }\n    }().call(null, mf);\n  }\n};\ncljs.core.throw_no_method_error = function throw_no_method_error(name, dispatch_val) {\n  throw new Error(&quot;No method in multimethod &apos;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(name) + &quot;&apos; for dispatch value: &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(dispatch_val));\n};\ncljs.core.MultiFn = function(name, dispatch_fn, default_dispatch_val, hierarchy, method_table, prefer_table, method_cache, cached_hierarchy) {\n  this.name = name;\n  this.dispatch_fn = dispatch_fn;\n  this.default_dispatch_val = default_dispatch_val;\n  this.hierarchy = hierarchy;\n  this.method_table = method_table;\n  this.prefer_table = prefer_table;\n  this.method_cache = method_cache;\n  this.cached_hierarchy = cached_hierarchy;\n  this.cljs$lang$protocol_mask$partition0$ = 4194305;\n  this.cljs$lang$protocol_mask$partition1$ = 256;\n};\ncljs.core.MultiFn.cljs$lang$type = true;\ncljs.core.MultiFn.cljs$lang$ctorStr = &quot;cljs.core/MultiFn&quot;;\ncljs.core.MultiFn.cljs$lang$ctorPrWriter = function(this__17712__auto__, writer__17713__auto__, opt__17714__auto__) {\n  return cljs.core._write.call(null, writer__17713__auto__, &quot;cljs.core/MultiFn&quot;);\n};\ncljs.core.MultiFn.prototype.cljs$core$IHash$_hash$arity$1 = function(this$) {\n  var self__ = this;\n  var this$__$1 = this;\n  return goog.getUid(this$__$1);\n};\ncljs.core.MultiFn.prototype.cljs$core$IMultiFn$_reset$arity$1 = function(mf) {\n  var self__ = this;\n  var mf__$1 = this;\n  cljs.core.swap_BANG_.call(null, self__.method_table, function(mf__$1) {\n    return function(mf__$2) {\n      return cljs.core.PersistentArrayMap.EMPTY;\n    };\n  }(mf__$1));\n  cljs.core.swap_BANG_.call(null, self__.method_cache, function(mf__$1) {\n    return function(mf__$2) {\n      return cljs.core.PersistentArrayMap.EMPTY;\n    };\n  }(mf__$1));\n  cljs.core.swap_BANG_.call(null, self__.prefer_table, function(mf__$1) {\n    return function(mf__$2) {\n      return cljs.core.PersistentArrayMap.EMPTY;\n    };\n  }(mf__$1));\n  cljs.core.swap_BANG_.call(null, self__.cached_hierarchy, function(mf__$1) {\n    return function(mf__$2) {\n      return null;\n    };\n  }(mf__$1));\n  return mf__$1;\n};\ncljs.core.MultiFn.prototype.cljs$core$IMultiFn$_add_method$arity$3 = function(mf, dispatch_val, method) {\n  var self__ = this;\n  var mf__$1 = this;\n  cljs.core.swap_BANG_.call(null, self__.method_table, cljs.core.assoc, dispatch_val, method);\n  cljs.core.reset_cache.call(null, self__.method_cache, self__.method_table, self__.cached_hierarchy, self__.hierarchy);\n  return mf__$1;\n};\ncljs.core.MultiFn.prototype.cljs$core$IMultiFn$_remove_method$arity$2 = function(mf, dispatch_val) {\n  var self__ = this;\n  var mf__$1 = this;\n  cljs.core.swap_BANG_.call(null, self__.method_table, cljs.core.dissoc, dispatch_val);\n  cljs.core.reset_cache.call(null, self__.method_cache, self__.method_table, self__.cached_hierarchy, self__.hierarchy);\n  return mf__$1;\n};\ncljs.core.MultiFn.prototype.cljs$core$IMultiFn$_get_method$arity$2 = function(mf, dispatch_val) {\n  var self__ = this;\n  var mf__$1 = this;\n  if (cljs.core._EQ_.call(null, cljs.core.deref.call(null, self__.cached_hierarchy), cljs.core.deref.call(null, self__.hierarchy))) {\n  } else {\n    cljs.core.reset_cache.call(null, self__.method_cache, self__.method_table, self__.cached_hierarchy, self__.hierarchy);\n  }\n  var temp__4124__auto__ = cljs.core.deref.call(null, self__.method_cache).call(null, dispatch_val);\n  if (cljs.core.truth_(temp__4124__auto__)) {\n    var target_fn = temp__4124__auto__;\n    return target_fn;\n  } else {\n    var temp__4124__auto____$1 = cljs.core.find_and_cache_best_method.call(null, self__.name, dispatch_val, self__.hierarchy, self__.method_table, self__.prefer_table, self__.method_cache, self__.cached_hierarchy);\n    if (cljs.core.truth_(temp__4124__auto____$1)) {\n      var target_fn = temp__4124__auto____$1;\n      return target_fn;\n    } else {\n      return cljs.core.deref.call(null, self__.method_table).call(null, self__.default_dispatch_val);\n    }\n  }\n};\ncljs.core.MultiFn.prototype.cljs$core$IMultiFn$_prefer_method$arity$3 = function(mf, dispatch_val_x, dispatch_val_y) {\n  var self__ = this;\n  var mf__$1 = this;\n  if (cljs.core.truth_(cljs.core.prefers_STAR_.call(null, dispatch_val_x, dispatch_val_y, self__.prefer_table))) {\n    throw new Error(&quot;Preference conflict in multimethod &apos;&quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(self__.name) + &quot;&apos;: &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(dispatch_val_y) + &quot; is already preferred to &quot; + cljs.core.str.cljs$core$IFn$_invoke$arity$1(dispatch_val_x));\n  } else {\n  }\n  cljs.core.swap_BANG_.call(null, self__.prefer_table, function(mf__$1) {\n    return function(old) {\n      return cljs.core.assoc.call(null, old, dispatch_val_x, cljs.core.conj.call(null, cljs.core.get.call(null, old, dispatch_val_x, cljs.core.PersistentHashSet.EMPTY), dispatch_val_y));\n    };\n  }(mf__$1));\n  return cljs.core.reset_cache.call(null, self__.method_cache, self__.method_table, self__.cached_hierarchy, self__.hierarchy);\n};\ncljs.core.MultiFn.prototype.cljs$core$IMultiFn$_methods$arity$1 = function(mf) {\n  var self__ = this;\n  var mf__$1 = this;\n  return cljs.core.deref.call(null, self__.method_table);\n};\ncljs.core.MultiFn.prototype.cljs$core$IMultiFn$_prefers$arity$1 = function(mf) {\n  var self__ = this;\n  var mf__$1 = this;\n  return cljs.core.deref.call(null, self__.prefer_table);\n};\ncljs.core.MultiFn.prototype.call = function() {\n  var G__23254 = null;\n  var G__23254__2 = function(self__, a) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a);\n  };\n  var G__23254__3 = function(self__, a, b) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b);\n  };\n  var G__23254__4 = function(self__, a, b, c) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c);\n  };\n  var G__23254__5 = function(self__, a, b, c, d) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d);\n  };\n  var G__23254__6 = function(self__, a, b, c, d, e) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e);\n  };\n  var G__23254__7 = function(self__, a, b, c, d, e, f) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f);\n  };\n  var G__23254__8 = function(self__, a, b, c, d, e, f, g) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g);\n  };\n  var G__23254__9 = function(self__, a, b, c, d, e, f, g, h) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h);\n  };\n  var G__23254__10 = function(self__, a, b, c, d, e, f, g, h, i) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i);\n  };\n  var G__23254__11 = function(self__, a, b, c, d, e, f, g, h, i, j) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j);\n  };\n  var G__23254__12 = function(self__, a, b, c, d, e, f, g, h, i, j, k) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k);\n  };\n  var G__23254__13 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l);\n  };\n  var G__23254__14 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);\n  };\n  var G__23254__15 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n  };\n  var G__23254__16 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);\n  };\n  var G__23254__17 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);\n  };\n  var G__23254__18 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);\n  };\n  var G__23254__19 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);\n  };\n  var G__23254__20 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);\n  };\n  var G__23254__21 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);\n  };\n  var G__23254__22 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {\n    var self__ = this;\n    var self____$1 = this;\n    var mf = self____$1;\n    var dispatch_val = cljs.core.apply.call(null, self__.dispatch_fn, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);\n    var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n    if (cljs.core.truth_(target_fn)) {\n    } else {\n      cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n    }\n    return cljs.core.apply.call(null, target_fn, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);\n  };\n  G__23254 = function(self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest) {\n    switch(arguments.length) {\n      case 2:\n        return G__23254__2.call(this, self__, a);\n      case 3:\n        return G__23254__3.call(this, self__, a, b);\n      case 4:\n        return G__23254__4.call(this, self__, a, b, c);\n      case 5:\n        return G__23254__5.call(this, self__, a, b, c, d);\n      case 6:\n        return G__23254__6.call(this, self__, a, b, c, d, e);\n      case 7:\n        return G__23254__7.call(this, self__, a, b, c, d, e, f);\n      case 8:\n        return G__23254__8.call(this, self__, a, b, c, d, e, f, g);\n      case 9:\n        return G__23254__9.call(this, self__, a, b, c, d, e, f, g, h);\n      case 10:\n        return G__23254__10.call(this, self__, a, b, c, d, e, f, g, h, i);\n      case 11:\n        return G__23254__11.call(this, self__, a, b, c, d, e, f, g, h, i, j);\n      case 12:\n        return G__23254__12.call(this, self__, a, b, c, d, e, f, g, h, i, j, k);\n      case 13:\n        return G__23254__13.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l);\n      case 14:\n        return G__23254__14.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m);\n      case 15:\n        return G__23254__15.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n      case 16:\n        return G__23254__16.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o);\n      case 17:\n        return G__23254__17.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p);\n      case 18:\n        return G__23254__18.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q);\n      case 19:\n        return G__23254__19.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r);\n      case 20:\n        return G__23254__20.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s);\n      case 21:\n        return G__23254__21.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t);\n      case 22:\n        return G__23254__22.call(this, self__, a, b, c, d, e, f, g, h, i, j, k, l, m, n, o, p, q, r, s, t, rest);\n    }\n    throw new Error(&quot;Invalid arity: &quot; + arguments.length);\n  };\n  G__23254.cljs$core$IFn$_invoke$arity$2 = G__23254__2;\n  G__23254.cljs$core$IFn$_invoke$arity$3 = G__23254__3;\n  G__23254.cljs$core$IFn$_invoke$arity$4 = G__23254__4;\n  G__23254.cljs$core$IFn$_invoke$arity$5 = G__23254__5;\n  G__23254.cljs$core$IFn$_invoke$arity$6 = G__23254__6;\n  G__23254.cljs$core$IFn$_invoke$arity$7 = G__23254__7;\n  G__23254.cljs$core$IFn$_invoke$arity$8 = G__23254__8;\n  G__23254.cljs$core$IFn$_invoke$arity$9 = G__23254__9;\n  G__23254.cljs$core$IFn$_invoke$arity$10 = G__23254__10;\n  G__23254.cljs$core$IFn$_invoke$arity$11 = G__23254__11;\n  G__23254.cljs$core$IFn$_invoke$arity$12 = G__23254__12;\n  G__23254.cljs$core$IFn$_invoke$arity$13 = G__23254__13;\n  G__23254.cljs$core$IFn$_invoke$arity$14 = G__23254__14;\n  G__23254.cljs$core$IFn$_invoke$arity$15 = G__23254__15;\n  G__23254.cljs$core$IFn$_invoke$arity$16 = G__23254__16;\n  G__23254.cljs$core$IFn$_invoke$arity$17 = G__23254__17;\n  G__23254.cljs$core$IFn$_invoke$arity$18 = G__23254__18;\n  G__23254.cljs$core$IFn$_invoke$arity$19 = G__23254__19;\n  G__23254.cljs$core$IFn$_invoke$arity$20 = G__23254__20;\n  G__23254.cljs$core$IFn$_invoke$arity$21 = G__23254__21;\n  G__23254.cljs$core$IFn$_invoke$arity$22 = G__23254__22;\n  return G__23254;\n}();\ncljs.core.MultiFn.prototype.apply = function(self__, args23253) {\n  var self__ = this;\n  var self____$1 = this;\n  return self____$1.call.apply(self____$1, [self____$1].concat(cljs.core.aclone.call(null, args23253)));\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$1 = function(a) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$2 = function(a, b) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$3 = function(a, b, c) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$4 = function(a, b, c, d) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$5 = function(a, b, c, d, e) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$6 = function(a, b, c, d, e, f) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e, f);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$7 = function(a, b, c, d, e, f, g) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e, f, g);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$8 = function(a, b, c, d, e, f, g, h) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e, f, g, h);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$9 = function(a, b, c, d, e, f, g, h, i) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e, f, g, h, i);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$10 = function(a, b, c, d, e, f, g, h, i, j) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e, f, g, h, i, j);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$11 = function(a, b, c, d, e, f, g, h, i, j, k) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$12 = function(a, b, c, d, e, f, g, h, i, j, k, l) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$13 = function(a, b, c, d, e, f, g, h, i, j, k, l, m) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m);\n};\ncljs.core.MultiFn.prototype.cljs$core$IFn$_invoke$arity$14 = function(a, b, c, d, e, f, g, h, i, j, k, l, m, n) {\n  var self__ = this;\n  var mf = this;\n  var dispatch_val = self__.dispatch_fn.call(null, a, b, c, d, e, f, g, h, i, j, k, l, m, n);\n  var target_fn = mf.cljs$core$IMultiFn$_get_method$arity$2(null, dispatch_val);\n  if (cljs.core.truth_(target_fn)) {\n  } else {\n    cljs.core.throw_no_method_error.call(null, self__.name, dispatch_val);\n  }\n  return target_fn.call(null, a, b, c, d, e, f, g, h, i