(ns de.sushi3003.batteries
  (:require [clojure.string :as str]))

(def not-nil? (complement nil?))

(def not-zero? (complement zero?))

(def not-empty? (complement empty?))

(def return-nil 
  "This function takes any number of arguments and always returns `nil`.

    (return-nil) => nil
    (return-nil :a) => nil"
  (constantly nil))

  "Diese Funktion liefert immer `nil` zurueck."

;; ## Functions operating on maps

(defn map-kv
  "Applies the `key-fn` to all keys- and the `val-fn` to all values of the `map`."
  [key-fn val-fn map]
  (reduce-kv (fn [memo k v] (conj memo [(key-fn k) (val-fn v)])) 
             {} 
             map))

(defn map-keys
  "Apply a function to all keys of a map.

    (map-keys name {:a 1 :b 2}) => {\"b\" 2, \"a\" 1}"
  [f map]
  (reduce-kv (fn [memo k v] (conj memo [(f k) v])) {} map))

(defn map-values
  "Apply a function to all values of a map, whose `key` is in `keyset`. 
   If the `keyset` is omitted `f` gets applied to _every_ value.

    (map-values inc #{:a :b} {:a 1 :b 2 :c 3}) => {:a 2 :b 3 :c 3}

    (map-values inc {:a 1 :b 2}) => {:a 2 :b 3}"
  ([f map]
   (reduce-kv (fn [memo k v] (conj memo [k (f v)])) {} map))
  ([f keyset map]
   (reduce-kv (fn [memo k v] (conj memo [k (if (keyset k) (f v) v)])) {} map)))

(defn defaults 
  "Returns a new map where all nil values are replaced with their appropriate
   default values.

    (defaults {:a 0 :b 0} {:a nil :b 1}) => {:a 0 :b 1}"
  [default-values map]
  (reduce-kv (fn [memo k v] 
               (conj memo [k (or v (default-values k))])) 
             {} 
             map))

(defn select-values
  "Returns a vector containing only those values whose key is in `keyseq`.  The
   values are in the same order as the keys in `keyseq`. The optional function
   `f` is applied to each selected value.

    (select-values [:b :a :c :a] {:a 1 :b 2}) => [2 1 nil 1]"
  ([keyseq map]
   (select-values keyseq identity map))
  ([keyseq f map]
   (persistent! (reduce #(conj! %1 (f (get map %2))) (transient []) keyseq))))

(defn index-by 
  "Returns a map of the elements of `coll` keyed by the result of `f` on each
   element.  The value at each key will be a single element (in contrast to
   `clojure.core/group-by`).  Therefore `f` should generally return an unique
   key for every element - otherwise elements get discarded."
  [f coll]
  (persistent! (reduce #(assoc! %1 (f %2) %2) (transient {}) coll)))

(def group-by-id 
  "*Deprecated* - use `index-by` instead." 
  index-by)

(defn submap?
  "Checks whether `map` contains all entries in `sub`.

    (submap? { :a 1 :b 2 } { :a 1 :b 2 :c 3 }) => true"
  [sub map]
  (reduce (fn [memo item] 
            (let[[k v] item]
              (and (contains? map k) (= (map k) v) memo))) 
          true 
          sub))

(defn has-keys?
  "Checks whether the map `m` contains all `keys`.

    (has-keys? [:a :b] {:a 1 :b 2 :c 3}) => true"
  [keys m]
  (every? (partial contains? m) keys))

(defn keyset 
  "Just a convenience function to retrieve all keys of the map `m` as a set."
  [m]
  (set (keys m)))

;; - - -
;; ## String functions

(defn str-trim 
  "Removes whitespace from both ends of string (like `clojure.string/trim`),
   but if nil is passed in, nil is returned (unlike `clojure.string/trim`, wich
   throws a NullPointerException)."
  [s]
  (when s (.trim s)))

(defn str-limit 
  "Cuts off strings longer than `limit` chars.

    (str-limit 3 \"Jan Hermanns\") => \"Jan\""
  [limit s]
  (when s
    (if (> (count s) limit)
      (subs s 0 limit)
      s)))

(defn str->int 
  "Parses the string argument as a signed decimal integer.
   If nil is passed in, nil is returned"
  [s]
  (when s (Integer/parseInt s)))

(defn intstr?
  "Returns true, if the string can be parsed as an decimal integer."
  [s]
  (try
    (Integer/parseInt s)  true
    (catch NumberFormatException _ false)))

(defn str-repeat 
  "Repeats the string `n` times.  Returns nil, if `n` is zero or negative.

    (str-repeat 2 \"hi\") => \"hihi\""
  [n s]
  (when (and (> n 0) s)
    (letfn [(str-repeat* [n s buf]
              (if (= n 0) buf
                (recur (dec n) s (.append buf s))))]
      (str (str-repeat* n s (StringBuilder.))))))

;; TODO **Known Issue** `(str-stuff-left "xxx" 6 "hello") => "xxxhello"`
(defn str-stuff-left 
  "If the string `s` is too short (i.e. less than `len` chars in length), it get
   stuffed by prepending `x` (repeatedly, if neccessary).

    (str-stuff-left \"x\" 8 \"hello\") => \"xxxhello\""
  [x len s]
  (when s 
    (let [n (- len (count s))]
      (str (str-repeat n x) s))))

(defn time-str? 
  "Checks wether `s` is a time string.

    (time-str? \"5:45\") => true
    (time-str? \"23:59\") => true"
  [s]
  (when s
    (let [[time hh mm] (re-find #"^(\d\d?):(\d\d)$" s)]
      (when time 
        (let [h (str->int hh)
              m (str->int mm)]
          (and (>= h 0) (<= h 23) (>= m 0) (<= m 59)))))))

(defn csv->seq
  "Returns a seq containing each comma-separated value.

   `(csv->seq nil) => nil`
   `(csv->seq \"\") => ()`
   `(csv->seq \"A,B,C\") => (\"A\",\"B\",\"C\")`
   `(csv->seq \"A,,C\") => (\"A\",\"C\")`
   `(csv->seq \"Hello   ,    World\") => (\"Hello\",\"World\")`
   `(csv->seq \"Hello   ;    World\" :separator \";\" :trim? false)  
      => (\"Hello   \",\"    World\")`"
  [str & {:keys [separator trim?] :or {separator "," trim? true}}]
  (when str
    (let [res (remove str/blank? (str/split str (re-pattern separator)))]
      (if trim? (map str/trim res) res))))

(defn char-range 
  "Returns a lazy seq of chars from start (inclusive).

    (take 3 (char-range \\a)) => (\\a \\b \\c)"
  [^java.lang.Character start]
  (iterate #(char (inc (int %))) start))

;; - - -
;; ## Date functions

(defn format-date
  "Formats the Date into a date/time string, according to the format-string."
  [fmt d]
  (when d
    (.format (java.text.SimpleDateFormat. fmt) d)))

(defn parse-date
  "Parses the String with the format `fmt` to produce a Date."
  [fmt s]
  (.parse (java.text.SimpleDateFormat. fmt) s))

(defn contains-date?
  "Checks, whether `date` lies between `start` and `end`. Both `start` and
   `end` are inclusive."
  [^java.util.Date start ^java.util.Date end ^java.util.Date date]
  {:pre [(not-nil? start) (not-nil? end) (not-nil? date)]}
  (not (or (.after start date) (.before end date))))

;; - - -
;; ## Misc 

(defn through 
  "Apply all functions in a seq to the same argument.

    (through [dec identity inc] 1) => [0 1 2]"
  [fns x]
  ((apply juxt fns) x))

(defn apply-to-each 
  "Applies `f` to each item in the collection. Therefore each item should be a
   vector, whose length matches the number of `f`s' input params.

    (apply-to-each + [[1 2] [3]]) => [3 3]"
  [f coll]
  (map #(apply f %) coll))

(def sum 
  "Returns the sum of all nums in a seq.

    (sum [1 2 3]) => 6" 
  (partial reduce +))

(defn between 
  "Returns a new seq where `x` is inserted at the specified `indices`.

    (between [1 2 3 4] :x [0])       => (:x 1 2 3 4)
    (between [1 2 3 4] :x [2])       => (1 2 :x 3 4)
    (between [1 2 3 4 5 6] :x [2 4]) => (1 2 :x 3 4 :x 5 6)
    (between [1 2 3 4 5 6] :x [2 3]) => (1 2 :x 3 :x 4 5 6)"
  [coll x indices]
  (let [splitter (partial subvec (vec coll))
        subvec-indices (partition 2 1 (into [0] (-> (sort indices) 
                                                    (vec) 
                                                    (conj (count coll)))))]
    (flatten (interpose x (apply-to-each splitter subvec-indices)))))

(defn reduce-indexed 
  "`f` should be a function of 3 arguments. Returns the result of applying `f` to
   `val`, 0 and the first item in `coll`, then applying `f` to that result, 1 and
   the second item in `coll`, then applying `f` to that result, 2 and the third
   item in `coll`, etc.  If `coll` contains no items, returns `val` and `f` is not
   called.

    (reduce-indexed (fn [memo i item] (assoc memo item i)) 
                    {} 
                    [:a :b :c]) => {:c 2, :b 1, :a 0}"
  [f val coll]
  (first 
    (reduce #(let[[memo i] %1] [(f memo i %2) (inc i)]) [val 0] coll)))

(defn indices 
  "Seeks the index of every `needle` in `haystack`. Returns a seq with those
   indices in the same order.

    (indices [:a :b] [:b :c :a :b]) => (1 nil 0 1)"
  [haystack needles]
  (let [index-map (reduce-indexed (fn[memo i item] (assoc memo item i)) {} haystack)]
    (map index-map needles)))

(defn find-first 
  "Returns the first item from `coll`, which satisfies the predicate `pred`.

    (find-first even? [1 1 2 3 5]) => 2"
  [pred coll]
  (first (filter pred coll)))

(defn where 
  "If you want to filter a sequence of maps this function is very handy,
   because it returns a function, which can be used as a predicate for
   `clojure.core/filter`.

    (filter (where :b 2 3) 
            [{:a 1 :b 2} {:a 2 :b 3} {:a 1 :b 4}]) 
    => ({:a 1 :b 2} {:a 2 :b 3})"
  [k & vals]
  (fn [e] (let [vs (into #{} vals)
                v (k e)] 
            (vs v))))

(defn append 
  "Appends any non-nil value at the end of the `seq`.

    (append 4 '(1 2 3)) => (1 2 3 4)"
  [x seq]
  (if x (concat seq [x]) seq))

(defn transpose 
  "Transposes the first two levels in `seq-of-seqs`.
   
    (transpose [[:a :b][:x :y]]) => ((:a :x) (:b :y))" 
  [seq-of-seqs] 
  (apply map list seq-of-seqs))

(defn transposev
  "Like `transpose`, but returns vectors instead of lists.
   
    (transposev [[:a :b][:x :y]]) => [[:a :x] [:b :y]]" 
  [seq-of-seqs] 
  (apply mapv vector seq-of-seqs))

(defprotocol Compactable
  "Enables the removal of nil values."
  (compact [this]))

;; TODO What about records?
(extend-protocol Compactable
  nil
  (compact [this] nil)
  clojure.lang.IPersistentSet
  (compact [this] (into #{} (filter (complement nil?) this)))
  clojure.lang.IPersistentVector
  (compact [this] (into [] (filter (complement nil?) this)))
  clojure.lang.IPersistentMap
  (compact [this] (apply dissoc this (for [[k v] this :when (nil? v)] k)))
  Object
  (compact [this] (filter (complement nil?) this)))

(defn lt 
  "Returns true, if `x` is less than `y`.
   This function is based on `clojure.core/compare`."
  [x y]
  (neg? (compare x y)))
  
(defn le 
  "Returns true, if `x` is less or equal to `y`.
   This function is based on `clojure.core/compare`."
  [x y]
  (let [res (compare x y)]
    (or (neg? res) (zero? res))))

(defn gt 
  "Returns true, if `x` is greater than `y`.
   This function is based on `clojure.core/compare`."
  [x y]
  (pos? (compare x y)))

(defn ge 
  "Returns true, if `x` is greater or equal to `y`.
   This function is based on `clojure.core/compare`."
  [x y]
  (let [res (compare x y)]
    (or (pos? res) (zero? res))))

