(ns de.sushi3003.batteries)

;(defn compact
;  "Remove all nil values from a map.
;   e.g.
;   (compact {:a 1 :b nil :c 3}) => {:c 3, :a 1}"
;  [map]
;  (apply dissoc map (for [[k v] map :when (nil? v)] k)))

(defn map-kv
  "Applies the key-fn to all keys- and the val-fn to all values of the map."
  [key-fn val-fn map]
  (reduce-kv (fn [memo k v] (conj memo [(key-fn k) (val-fn v)])) 
             {} 
             map))

(defn map-keys
  "Apply a fn to all keys of a map.
   e.g.
   (map-keys name {:a 1 :b 2}) => {\"b\" 2, \"a\" 1}"
  [f map]
  (reduce-kv (fn [memo k v] (conj memo [(f k) v])) {} map))

(defn map-values
  "Apply a fn to all values of a map.
   e.g.
   (map-values inc {:a 1 :b 2}) => {:a 2 :b 3}"
  [f map]
  (reduce-kv (fn [memo k v] (conj memo [k (f v)])) {} map))

(defn defaults 
  "Returns a new map where all nil values are replaced with their appropriate
   default values.
   e.g.
   (defaults{:a 0 :b 0} {:a nil :b 1}) => {:a 0 :b 1}"
  [default-values map]
  (reduce-kv (fn [memo k v] 
               (conj memo [k (or v (default-values k))])) 
             {} 
             map))

(defn select-values
  "Returns a vector containing only those values whose key is in keyseq.  The
   values are in the same order as the keys in keyseq. The optional function f
   is applied to each selected value.
   e.g.
   (select-values {:a 1 :b 2} [:b :a :c :a]) => [2 1 nil 1]"
  ([map keyseq]
   (select-values map keyseq identity))
  ([map keyseq f]
   (persistent! (reduce #(conj! %1 (f (map %2))) (transient []) keyseq))))

(defn group-by-id 
  "Returns a map of the elements of coll keyed by the result of f on each
   element.  The value at each key will be a single element (in contrast to
   clojure.core/group-by).  Therefore f should generally return an unique key
   for every element - otherwise elements get discarded."
  [f coll]
  (persistent! (reduce #(assoc! %1 (f %2) %2) (transient {}) coll)))

(defn str-trim 
  "Removes whitespace from both ends of string (like clojure.string/trim), but
   if nil is passed in, nil is returned (unlike clojure.string/trim, wich
   throws a NullPointerException)."
  [s]
  (when s (.trim s)))

(defn format-date
  "Formats the Date into a date/time string, according to the format-string."
  [fmt d]
  (.format (java.text.SimpleDateFormat. fmt) d))

(defn through 
  "Apply all functions in a seq to the same argument.
   e.g.
   (through [dec identity inc] 1) => [0 1 2]"
  [fns x]
  ((apply juxt fns) x))

(defn apply-to-each 
  "Applies f to each item in the collection. Therefore each item should be a
   vector, whose length matches the number of fs' input params.
   e.g.
  (apply-to-each + [[1 2] [3]]) => [3 3]"
  [f coll]
  (map #(apply f %) coll))

(def sum 
  "Returns the sum of all nums in a seq.
   e.g.
   (sum [1 2 3]) => 6" 
  (partial reduce +))

(defn str->int 
  "Parses the string argument as a signed decimal integer."
  [s]
  (Integer/parseInt s))

(defn not-zero?
  "Returns true if n is not zero."
  [n]
  (not (zero? n)))

(defn submap?
  "Checks whether map contains all entries in sub.
   e.g.
   (submap? { :a 1 :b 2 } { :a 1 :b 2 :c 3 }) => true"
  [sub map]
  (reduce (fn [memo item] 
            (let[[k v] item]
              (and (contains? map k) (= (map k) v) memo))) 
          true 
          sub))

(defn between 
  "Returns a new seq where x is inserted at the specified indices.
   e.g.
   (between [1 2 3 4] :x [0])       => (:x 1 2 3 4)
   (between [1 2 3 4] :x [2])       => (1 2 :x 3 4)
   (between [1 2 3 4 5 6] :x [2 4]) => (1 2 :x 3 4 :x 5 6)
   (between [1 2 3 4 5 6] :x [2 3]) => (1 2 :x 3 :x 4 5 6)"
  [coll x indices]
  (let [splitter (partial subvec (vec coll))
        subvec-indices (partition 2 1 (into [0] (-> (sort indices) 
                                                    (vec) 
                                                    (conj (count coll)))))]
    (flatten (interpose x (apply-to-each splitter subvec-indices)))))

(defn reduce-indexed 
  "f should be a function of 3 arguments. Returns the result of applying f to
   val, 0 and the first item in coll, then applying f to that result, 1 and
   the second item in coll, then applying f to that result, 2 and the third
   item in coll, etc.  If coll contains no items, returns val and f is not
   called.
   e.g. 
   (reduce-indexed (fn [memo i item] (assoc memo item i)) {} [:a :b :c]) 
     => {:c 2, :b 1, :a 0}"
  [f val coll]
  (first 
    (reduce #(let[[memo i] %1] [(f memo i %2) (inc i)]) [val 0] coll)))

(defn indices 
  "Seeks the index of every needle in haystack. Returns a seq with those
   indices in the same order.
   e.g.
   (indices [:a :b] [:b :c :a :b]) => (1 nil 0 1)"
  [haystack needles]
  (let [index-map (reduce-indexed (fn[memo i item] (assoc memo item i)) {} haystack)]
    (map index-map needles)))

(defprotocol Compactable
  "Enables the removal of nil values."
  (compact [this]))

(extend-protocol Compactable
  nil
  (compact [this] nil)
  clojure.lang.IPersistentList 
  (compact [this] (filter (complement nil?) this))
  clojure.lang.IPersistentSet
  (compact [this] (into #{} (filter (complement nil?) this)))
  clojure.lang.IPersistentVector
  (compact [this] (into [] (filter (complement nil?) this)))
  clojure.lang.IPersistentMap
  (compact [this] (apply dissoc this (for [[k v] this :when (nil? v)] k))))

