(ns de.phenomdevel.formalicious.helper
  "Provides generic helper for manipulating hiccup or generate event-handler."
  (:require
   [plumbing.core :refer-macros [letk]]
   [re-frame.core :refer [subscribe dispatch-sync dispatch]]

   [de.phenomdevel.formalicious.util :as u]
   [de.phenomdevel.formalicious.util.dom.element :as element]))


;; =============================================================================
;; Event Handler Helper

(defn- on-click-handler
  [handler]
  (fn []
    (dispatch (u/collify handler))))

(defn on-change-handler
  "Returns a default on-change-handler which will call `dispatch-sync`
  with event `:state/update-at`

  NOTE: We use `dispatch-sync` here because otherwise you can lose inputs if you
  type fast enough."
  [data-path value]
  (dispatch-sync [:state/update-at data-path value]))

(defn value-on-change-handler
  "Returns a on-change-handler which will call `on-change-handler` with the
  given elements value for `:value` attribute."
  [value-transform data-path]
  (fn [e]
    (let [value
          (-> e
              (element/value)
              (value-transform))]

      (on-change-handler data-path value))))

(defn checked-on-change-handler
  "Returns a on-change-handler which will call `on-change-handler` with the
  given elements value for `:checked` attribute."
  [data-path]
  (fn [e]
    (let [checked?
          (element/checked? e)]

      (on-change-handler data-path checked?))))


;; =============================================================================
;; Form-Field Attribute Helper

(defn with-value
  "Assocs key `:value` with the given value into the given `attrs` map."
  [attrs value]
  (assoc attrs :value (or value "")))

(defn with-on-change
  "Assocs key `:on-change` with the given on-change-handler
  or default on-change-handler if none is provided on `attrs`."
  [attrs on-change value-transform update-path]
  (cond-> attrs
    (fn? on-change)
    (assoc :on-change on-change)

    (nil? on-change)
    (assoc :on-change (value-on-change-handler value-transform update-path))))

(defn with-disabled
  "Assocs key `:disabled` with the given value on `attrs`."
  [attrs disabled]
  (assoc attrs :disabled disabled))

(defn with-on-click
  "Assocs key `:on-click` with the given on-click-handler."
  [attrs on-click]
  (let [on-click*
        (cond
          (fn? on-click)
          on-click

          (keyword? on-click)
          (on-click-handler on-click))]

    (assoc attrs :on-click on-click*)))

(defn with-class
  [attrs spec]
  (cond-> attrs
    (:class spec)
    (assoc :class (:class spec))))


;; =============================================================================
;; Form-Field Helper

(defn with-input-field-wrapper
  "Wraps given `attrs` map within a [:div.input-field] element."
  [attrs]
  [:div.pd__field
   attrs])

(defn with-datalist-input
  [attrs]
  [:input attrs])

(defn with-option-wrapper
  "Wraps given `attrs` map within a [:option] element with
  `:key` assoced in `attrs` with value of given id."
  [attrs id]
  [:option.pd__select__option (assoc attrs :key id)])

(defn with-select-field-wrapper
  "Wraps given `attrs` map within a [:select.browser-default] element."
  [attrs]
  [:select.pd__select attrs])

(defn with-datalist-wrapper
  "Wraps given `attrs` map within a [:datalist] element."
  [attrs]
  [:datalist.pd__datalist attrs])

(defn default-option
  "Returns a basic option hiccup element generated of given `field-spec`"
  [field-spec]
  (letk [[{label ""}]
         field-spec]

    [:option.pd__select__option
     {:disabled true
      :value nil}
     label]))

(defn with-label
  "Conj's label element into the given `field-wrapper`."
  [field-wrapper field-spec]
  (letk [[{id ""}
          {label nil}]
         field-spec

         label
         [:label.pd__label
          {:for id}
          label]]

    (if label
      (conj field-wrapper label)
      field-wrapper)))

(defn with-contents
  "Puts given contents into the given field-wrapper."
  [field-wrapper & contents]
  (into field-wrapper contents))

(defn with-hint
  [field-wrapper hint]
  (->> [:span.pd__field.pd__hint
        {:title hint}
        "!"]
       (conj field-wrapper)))

(defn ensure-id
  [option]
  (if-not (map? option)
    {:id option :label option}
    (letk [[{id nil} {label nil}]
           option]
      (if (nil? id)
        (merge option {:id label :label label})
        option))))

(defn wrap-validation-message
  [message]
  [:span.pd__validation-message
   message])

(defn wrap-validation-messages
  [messages]
  (->> (mapv wrap-validation-message messages)
       (into [:span.pd__validation-messages])))

(defn with-validation
  [field-wrapper messages]
  (->> (wrap-validation-messages messages)
       (conj field-wrapper)))

(defn validate
  [validations value]
  (letk [[{checks []}
          {multiple? false}]
         validations]

    (reduce
     (fn [messages validation]
       (let [[pred msg]
             validation]

         (if-not (pred value)
           messages
           (if multiple?
             (conj messages msg)
             (reduced (conj messages msg))))))
     []
     checks)))
