(ns de.phenomdevel.formalicious.form.table
  (:require
   [plumbing.core :refer [letk]]
   [re-frame.core :refer [dispatch subscribe reg-event-db]]

   [de.phenomdevel.formalicious.util.log :as log]
   [de.phenomdevel.formalicious.form.shared :as shared]
   [de.phenomdevel.formalicious.form.paragraph :as paragraph]))


;; =============================================================================
;; Helper

(defn- render-cell
  [value]
  ^{:key value}
  [:td.pd__table__body__cell
   [paragraph/render
    {:type :paragraph}
    value]])

(defn- render-selection-cell
  [table-state-path]
  [:td.pd__table__body__cell
   [:input
    {:type :checkbox}]])

(defn- render-sorting
  [direction]
  (let [icon
        (case direction
          :ASC "sort-asc"
          :DESC "sort-desc"
          "sort")]

    [:div.pd__sorting
     [:span.fa {:class (str "fa-" icon)}]]))

(defn- trigger-sorting
  [table-state-path path direction]
  (let [update-path
        (conj table-state-path :sorting)

        direction*
        (case direction
          :ASC :DESC
          :DESC :NONE
          :NONE :ASC
          :ASC)]

    (dispatch [:state/update-at update-path [path direction*]])))

;; NOTE: Dadurch, dass wir `direction` an zwei verschiedenen Stellen brauchen
;; Wird hier jede Header-Zelle neu gezeichnet
(defn- render-header-cell
  [table-spec cell-spec]
  (let [table-state-path
        (:table-state-path table-spec)

        ;; NOTE: This maybe a map in the future with table-id
        ;; as key and vector of [path direction] as value
        !sorting
        (subscribe [:state/get-at (conj table-state-path :sorting)])]

    (fn [table-spec cell-spec]
      (let [[sorting-path sorting-direction]
            @!sorting

            path
            (:path cell-spec)

            direction
            (if (= path sorting-path)
              sorting-direction
              :NONE)]

        [:td.pd__table__header__cell
         {:on-click (partial trigger-sorting table-state-path path direction)}
         [paragraph/render {:type :paragraph} (get cell-spec :label "NO LABEL")]
         [render-sorting direction]]))))

(defn- render-header
  [{:keys [column-specs] :as table-spec}]
  [:thead.pd__table__header
   [:tr.pd__table__header__row
    (for [column-spec column-specs]
      ^{:key (:path column-spec)}
      [render-header-cell table-spec column-spec])]])

(defn- render-row
  [table-spec row]
  (letk [[{cell-default ""}
          {column-specs []}
          {empty-value-default ""}
          {with-selection? true}
          {table-state-path []}]
         table-spec]

    [:tr.pd__table__body__row
     (when with-selection?
       [render-selection-cell table-state-path])
     (for [{:keys [path] :as column-spec} column-specs]
       ^{:key path}
       [render-cell (get-in row path cell-default)])]))

(defn- render-body
  [table-spec rows]
  (let [rendered-rows
        (for [row rows]
          [render-row table-spec row])]

    (into [:tbody.pd__table__body] rendered-rows)))


;; =============================================================================
;; Public API

(defn render
  [table-spec _]
  (letk [[{class nil}]
         table-spec

         base-attrs
         (shared/field-spec->attrs table-spec)]

    (fn [table-spec rows]
      [:table.pd__table
       base-attrs
       [render-header table-spec]
       [render-body table-spec rows]])))



;; TODO: Reset sorting
;; TODO: Visibility options
;; TODO: Column order
;; TODO: Click-Handler per row
;; TODO: Select?
;; TODO: Paging
