(ns de.phenomdevel.formalicious.core
  "Provides public api functions for rendering a form or a single input-field."
  (:require
   [re-frame.core :refer [subscribe]]
   [plumbing.core :refer-macros [letk]]

   [de.phenomdevel.formalicious.handler]
   [de.phenomdevel.formalicious.subscription]

   [de.phenomdevel.formalicious.util :as u]
   [de.phenomdevel.formalicious.form :as form]
   [de.phenomdevel.formalicious.dynamic :as dyn]))


;; =============================================================================
;; Helper

(defn- interpolate-data-root
  [data-root field]
  (letk [[{id ""}
          {data-path nil}]
         field

         value
         (if data-path
           (u/to-data-path data-path)
           (u/to-data-path data-root id))]

    (assoc field :data-path value)))

(defn- transform-form-spec
  [form-spec data]
  (letk [[{fields []}
          {data-root []}]
         form-spec]

    (mapv (partial interpolate-data-root data-root) fields)))

(defn- field-spec->subscription
  [field-spec data]
  (letk [[{data-path nil}]
         field-spec

         value-in-data?
         (not-empty (get-in data data-path))]

    (when (and data-path (not value-in-data?))
      (subscribe [:state/get-at data-path]))))

(defn- form-spec->subscriptions
  [form-spec data]
  (reduce
   (fn [acc {:keys [id] :as field-spec}]
     (if-let [sub
              (field-spec->subscription field-spec data)]
       (assoc acc id sub)
       acc))
   {}
   (:fields form-spec)))


;; =============================================================================
;; Public API

(defn set-debug-logging!
  "Takes `true` or `false` as parameters and enables or disables debug logging."
  [x]
  {:pre (boolean? x)}
  (reset! dyn/*print-debug?* x))

(defn render-form
  "Takes a `form-spec` and `data` which will be rendered as react-components using re-frame.

   Optionally you can provide `opts` which can contain additional options:
   - `:wrapper` overrides the default hiccup-element wrapper [:div ...].

  A basic `form-spec` may look like this:
  {:data-root [:form-data]
   :fields {:field-1
            {:label \"Field-1\"
             :type :text}}}

  This would render a form with one input field of type `text` which will
  get it's value of the provided data-map on path [:field-1].

  For more details on how to write a `form-spec` please see the README.md."
  ([form-spec]
   (let [!form-data
         (subscribe [:state/get-at (:data-root form-spec)])]

     [render-form form-spec @!form-data]))

  ([form-spec data]
   [render-form form-spec data {}])

  ([form-spec data opts]
   (letk [[{wrapper [:div.pd__form]}]
          opts

          subscription-data
          (form-spec->subscriptions form-spec data)]

     (fn [form-spec data opts]
       (let [fields
             (transform-form-spec form-spec data)

             rendered-fields
             (for [field fields]
               (letk [[{id ""}
                       {data-path nil}]
                      field

                      value
                      (or (get data id)
                          (get-in data data-path)
                          (get subscription-data id))

                      value*
                      (if (instance? reagent.ratom/Reaction value)
                        @value
                        value)]

                 [form/render-field field value*]))]

         (into wrapper rendered-fields))))))

(defn render-field
  "Takes a `field-spec` and `data` which will be rendered as a react-component using re-frame.

  A basic `field-spec` may look like this:
  {:field-1
   {:label \"Field-1\"
    :type :text}}}

  This would render a single input field of type `text` which will
  get it's value of the provided data-map on path [:field-1].

  For more details on how to write a `field-spec` please see the README.md."
  [field-spec data]
  (let [subscription-data
        (field-spec->subscription field-spec data)]

    (fn [field-spec data]
      (letk [[id]
             field-spec

             value
             (or (get data id)
                 (get subscription-data id))

             value*
             (if (instance? reagent.ratom/Reaction value)
               @value
               value)]

        [form/render-field field-spec value*]))))
