# formalicious [![Clojars Project](https://img.shields.io/clojars/v/de.phenomdevel/formalicious.svg)](https://clojars.org/de.phenomdevel/formalicious)

With this library you can build generic html forms using [clojurescript](https://clojurescript.org) and [re-frame](https://github.com/Day8/re-frame).

## Get it
Add following dependency to your `project.clj`

This library is currently in development. If you're looking for a stable form generator `formalicious` won't help you.
I am trying to publish a `stable` version any time soon.
```clj
[de.phenomdevel/formalicious "0.2.0-SNAPSHOT"]
```

## Motivation
In the last past weeks i started writing [SPA](https://en.wikipedia.org/wiki/Single-page_application)'s with [re-frame](https://github.com/Day8/re-frame) and in each of them i needed some kind of forms. So after the second or third time writing the same code over and over again i started thinking.

Well i wanted some kind of reusable form elements and started writing this library.

## Supported Form-Fields
The library currently supports following form-field types:
- text
- textarea
- button
- password
- select
- multiselect
- checkbox
- radiobutton
- radiogroup (collection of radiobuttons where only one radiobutton can be active at a time)
- date (this is the standard html5 datepicker which is currently not very well supported by browsers)

## Currently Unsupported Form-Fields
- HTML5 Slider
- Fileupload


If you have a request for any other `form-field` which should be supported please feel free to contact me or make a pull-request.

## Supported Attributes
There are a lot of attributes you can set within the `form-spec`. They are described in detail below. Please remember that not all of these attributes are "standard" html attributes. A lot of them are used by the library to provide functionality with a simple spec behind it.
Theoretically there is only one attribute required => `type`. But you should concider using the attributes provided to give the `input-fields` a good look and feel.
### Common Attributes
Follwing are the common `form-field` attributes. These are attributes each of the `form-field` types you can specify supports. These will be described in detail below.
- `:type` (keyword) indicates which input-field should be rendered => Required
- `:label` (string) Holds the label for the rendered input-field
- `:class` (string) Holds any additonal classes which should be applied to the parent html-element of the input-field
- `:placeholder` (string) Holds a placeholder for text-input-fields such as text-inputs or textareas
- `:pos` (integer) This will determine in which order the input-fields will be rendered. If you leave out this key you won't have a guaranteed order
- `:data-path` (either keyword, or vector of keywords) if you don't want the input-field to receive the data of the provided data-map with the id you can specify a data-path. Formalicious will then look into the given data-map and search for a value on the given path. If there is no value it will subscribe to the path provided.
- `:disabled` (either boolean, or function which has to return a boolean value) Determines if the input-field should be disabled or not
### Form-Field Specific Attributes
There are a few form-field types which have some special attributes. These will be described in detail below.
#### Select Form-Field
- `:options` (any collection) This collection will used to generate the options for the select input-field
- `:options-path` (keyword or vector of keywords) If no options are provided this will be used to subscribe to a specific data-path
- `:option-label-key` (keyword) Each option will be interpolate so it is represented as map containing `:id` and `:label` at minium. `:options-label-key` will be used to determine which key in the option-map should be used as label.
- `:option-transform` (function) Function which will be applied to each option. E.g `(clojure.string/capitalize)`.
- `:order-by` (either keyword or function) Keyword/Function will be used as predicate for sort-by.

#### Attribute Transformation
Most of the provided attributes will be used as they appear. However some of them will be transformed.
##### `on-click`
You can either provide a `keyword`, or a `function`
There is no default on-click handler for any `form-field`.
- `keyword` or vector of `keywords`: Will be used as event-handler to dispatch to with re-frame.
- `function`: Will be called on click with `event` as parameter. You have to manually handle the event.

##### `on-change`
You can either provide a `keyword`, or a `function`.
There is a default behavious each `form-field` has. It will dispatch on the generic `:state/update-at` handler.
This handler will simply update the `app-state` at the `data-path` of the `form-field` which is usually `data-root` + `id`.
You can override the `data-path` if you want the on-change handler to update another path in the `app-state`.

- `keyword` or vector of `keywords`: Will be used as event-handler to dispatch to with re-frame.
- `function`: Will be called on click with `event` as parameter. You have to manually handle the event.

##### `disabled`
You can either provide a `keyword`, a `function`, or a `boolean`.
The default is `false`.

- `keyword` or vector of `keywords`: This will be used as subscription path on the app-state which has to return a `boolean` value.
- `function`: A `function` which will be called to determine the `disabled` state of the gieven `form-field`.
- `boolean`: Will be just passed through.

## Usage
In the following section i'll try to give you a little introduction in how to use this library.
### Form-Spec
The most important thing we have to think about is the `form-spec`. The `form-spec` is used by the library to generate the `form-fields`. First of all what does it look like?
```clj
(def form-spec
  {:data-root
   [:example :form-data]

   ;; Here you would specify all form-fields you want to render
   ;; They are described below
   :fields
   {:form-field-key
    {:type :text
     :label "Textinput"
     :class "some css classes"
     :pos 1
     :data-path [:not :the :default :path]}}})
 ```
 We have a `form-spec` which does not have many much information except for the `data-root`. The `data-root` is the path which will be used to populate our state to. This will be done through a generic `dispatch`. It also has a `fields` key whose value is a map. This map is structured as shown above. Each key indicates a new `form-field` and the value is the description of this `form-field`.
 ### Render Forms
 As shown above we need a form-spec to render a form.

```clj

(def form-spec
  {:data-root
   [:keywords]

   :fields
   {:keyword
    field-specs}})
```

That's pretty much it. All you have to do now is to call the render function and you're good to go.
```clj
(:require
  [de.phenomdevel.formalicious.core :as formalicious])

;; You can call `render-fields` if you want map which is a direct reflection of the `form-spec` fields map.
;; The only diffrence is that each field within the `fields` map is a rendered component which you can
;; use to put it anywhere in your view code.
(formalicious/render-fields form-spec some-data)

;; If you want to have a complete form wrapped into an html-element
;; you can call `render` which will return the form as hiccup.
(formalicious/render form-spec some-data)

;; By default all form-fields will be wrapped into [:div ...]
;; If you want to put the form-fields into another wrapper you can provide it as option
(formalicious/render form-spec some-data {:wrapper [:some-other-html-element]})
```

### Rendered Form-Fields
Most of the `form-fields` will be rendered as shown below.
```clj
[:div.input-field
  {:class "class-string"}
   [:label "label-string"]
    [input
     {:attributes-map ...}]]
```
The only exceptions are radiogroups and buttons.
##### Radiogroup
```clj
[:div.radiogroup
 {:class "class-string"}
 [:input
  {:type :radio ,,,}]]
```
##### Button
```clj
[:button.input-field
 {:class "class-string"
  :additional-key :additonal-value}
 "Label"]
```
## Some Example Form-Field Specs

### Input: Text

Lets asume we have following form-spec:
```clj
(def form-spec
  {:data-root
   [:example :form-data]

   :fields
   {:simple-text
    {:type
     :text

     :label
     "Simple Textinput"}}})
```

And we want to render it like this:
```cljs
(defn view
  []
  (let [!form-data
        @(subscribe [:form-data])]

    [:div.form
     [formalicious/render form-spec !form-data]]))
```
And this is what it would look like after formalicious did it's job:
```clj
[:div.form
 [:div.input-field
  [:label "Simple Textinput"]
   [:input
    {:type
     :text

     :value
     ""

     :on-change
     #(dispatch-sync [:state/update-at [:example :form-data :simple-text] (-> % (.-target) (.-value))]]]
```

This will render a single text-input field. The value will be provided through the `!form-data` which we subscribed on.
The input-field will get it's value of this map through a simple lookup with the input-field-id.
If no id is provided the key which has the field-spec as value will be used as id.


Well obviously it is not the most complex example but you can try it out yourself with the example application provided with this repo.


### Input: Select
Now a more complex example just to show you what formalicious will do for you.

Lets asume we have following form-spec:
```cljs
(def form-spec
  {:data-root
   [:example :form-data]

   :fields
   {:normal-select
    {:type :select
     :label "Select"
     :options [{:name "option 1" :label "Option One"}
               {:name "option 2" :label "Option Two"}
               {:name "option 3" :label "Option Three"}]
     :option-label-key :label
     :option-transform #(update % :label str/reverse)
     :order-by :name}}})
```
And we want to render it like this:
```cljs
(defn view
  []
  (let [!form-data
        @(subscribe [:form-data])]

    [:div.form
     [formalicious/render form-spec !form-data]]))

;; You could also just update the options-key with some other value.
;; For example you subsribe to Entity-A
;; Then (assoc-in form-spec [:fields :your-select :options] @Entity-A)
;; Now you have a select-box which shows options from some other data in your app-state
```
And this is what it would look like after formalicious did it's job:
```clj
[:div.form
 [:div.input-field
  [:select
   {:value ""
    :on-change
    #(dispatch-sync [:state/update-at [:example :form-data :normal-select] (-> % (.-target) (.-value))])}
   [:option
    {:disabled true :value ""}
    "Select"]
   [:option
    ;; ID = Label because formalicious will use label if there is a map as option
    ;; or the string itself if the option-values are strings
    {:value "Option One" :key "Option One"}
    "enO noitpO"]
   [:option
    {:value "Option Two" :key "Option Two"}
    "owT noitpO"]
   [:option
    {:value "Option One" :key "Option Three"}
    "eerhT noitpO"]]]]
```
Did you notice the additional `option` with an empty value? This will be produced if you provide a `:label` to the `field-spec` of an select input field.
## Pros
- You can even use the same spec for different forms by simply changing the `data-root`
- Descriptive datastructure for form rendering
- You don't have to handle with the right `subscriptions` and `dispatches`. The library will do it for you.
- Easy ways to manipulate the default behaviour

## Cons
- Generic form generators won't solve all your problems you run into
- You have to handle my understanding of `input-fields` in terms of html structure
- You might have a spcial case where this library can't help due to diffrent facts
  - Either you need a different html-structure
  - You want to edit attributes you cannot manipulate
  - etc.

## Try it out yourself
You want to try it yourself and see how well it can perform?

1. Clone this repo
2. Startup a repl within the repo directory
3. type `(user/system-restart!)` => Enter
4. type `(user/fig-init)` => Enter
5. You should now be ready to go

Now you should be ready to go. Just navigate your browser to `localhost:1338/app` try the forms.

If you want to play around and check the possibilities just go to the cljs/example namespace and have fun.

## TODOS
- Tests using phantomjs
- More complex examples (using `data-path` and showing how the attributes work)
- Cleanup code base
- More `form-field` types
- Better support for rendering single `form-fields`
- Use clojure.spec for form-spec/field-spec validation

## License

Copyright © 2017 Kevin Kaiser [PhenomDevel](https://github.com/PhenomDevel/formalicious)

Distributed under the Eclipse Public License version 1.0.
