(ns de.phenomdevel.formalicious.form
  (:require
   [plumbing.core :refer-macros [letk]]
   [re-frame.core :refer [subscribe]]

   [de.phenomdevel.formalicious.util :as u]
   [de.phenomdevel.formalicious.helper :as h]
   [de.phenomdevel.formalicious.util.dom.element :as element]))


;; =============================================================================
;; Common Helper

(def default-attribute-keys
  "These are the attribute keys which will be used without any transformation.
  If they're provided they will be rendered within the input-field."
  [:id :type :multiple :checked :placeholder :min :max :step :size])

(defn- default-attrs
  [field-spec]
  (reduce
   (fn [acc [k v]]
     (if v
       (assoc acc k v)
       acc))
   {}
   (select-keys field-spec default-attribute-keys)))

(defn- field-spec->attrs
  [field-spec]
  (letk [[{disabled nil}
          {on-click nil}
          {on-change nil}
          {data-path nil}]
         field-spec]

    (-> field-spec
        (default-attrs)
        (h/with-disabled disabled)
        (h/with-on-click on-click)
        (h/with-on-change on-change data-path))))


;; =============================================================================
;; Field Specs

(defmulti field
  (fn [field-spec]
    (:type field-spec)))


;; =============================================================================
;; Default

(defmethod field :default
  [_]
  [:div "Input-Type not implemented."])


;; =============================================================================
;; Text

(defmethod field :text
  [field-spec]
  (fn [field-spec]
    (letk [[{class nil}]
           field-spec

           attrs
           (field-spec->attrs field-spec)

           input-field
           [:input attrs]]

      (-> {:class class}
          (h/with-input-field-wrapper)
          (h/with-label field-spec)
          (h/with-contents input-field)))))


;; =============================================================================
;; Password

(defmethod field :password
  [field-spec]
  (fn [field-spec]
    (letk [[{id ""}
            {class ""}]
           field-spec

           attrs
           (field-spec->attrs field-spec)

           input-field
           [:input attrs]]

      (-> {:class class}
          (h/with-input-field-wrapper)
          (h/with-label field-spec)
          (h/with-contents input-field)))))



;; =============================================================================
;; Select

(defn field-spec->options
  [field-spec]
  (letk [[{options nil}
          {options-path nil}
          {option-transform nil}]
         field-spec

         options
         (or options
             @(subscribe [:state/get-at (u/collify options-path)]))]

    (if-not option-transform
      options
      (mapv option-transform options))))

(defn ensure-id
  [option]
  (if-not (map? option)
    {:id option :label option}
    (letk [[{id nil} {label nil}]
           option]
      (if (nil? id)
        {:id label :label label}
        option))))

(defmethod field :select
  [field-spec]
  (fn [field-spec]
    (letk [[{id ""}
            {class nil}
            {order-by :id}
            {multiple false}
            {option-label-key :label}
            {option-value-key :id}]
           field-spec

           attrs
           (field-spec->attrs field-spec)

           options
           (->> field-spec
                (field-spec->options)
                (map ensure-id)
                (sort-by order-by))

           option-fields
           (for [option options]
             (letk [[{id ""}]
                    option]

               (-> {:value (get option option-value-key (get option :id))}
                   (h/with-option-wrapper id)
                   (conj (str (get option option-label-key "NO VALUE"))))))

           select-field
           (-> attrs
               (h/with-select-field-wrapper)
               (h/with-contents (h/default-option field-spec) option-fields))]

      (-> {:class class}
          (h/with-input-field-wrapper)
          (h/with-contents select-field)))))



;; =============================================================================
;; Multiselect

(defmethod field :multiselect
  [field-spec]
  (fn [field-spec]
    (letk [[data-path value]
           field-spec

           field-spec*
           (-> field-spec
               (assoc :type :select
                      :multiple true
                      :on-change (fn [e]
                                   (let [options
                                         (element/options e)]

                                     ;; NOTE: This is done due to a bug in chrome where
                                     ;; the on-change is fired two times
                                     (if (not= value options)
                                       (h/on-change-handler data-path options))))))]

      [field field-spec*])))



;; =============================================================================
;; Textarea

(defmethod field :textarea
  [field-spec]
  (fn [field-spec]
    (letk [[{class nil}]
           field-spec

           attrs
           (field-spec->attrs field-spec)

           input-field
           [:textarea attrs]]

      (-> {:class class}
          (h/with-input-field-wrapper)
          (h/with-label field-spec)
          (h/with-contents input-field)))))


;; =============================================================================
;; Button

(defmethod field :button
  [field-spec]
  (fn [field-spec]
    (letk [[{label ""}
            {class ""}]
           field-spec

           attrs
           (field-spec->attrs field-spec)]

      [:button.input-field
       (assoc attrs :class class)
       label])))


;; =============================================================================
;; Checkbox

(defmethod field :checkbox
  [field-spec]
  (fn [field-spec]
    (letk [[{class ""}
            data-path]
           field-spec

           attrs
           (-> (field-spec->attrs field-spec)
               (assoc :on-change (h/checked-on-change-handler data-path)))

           checkbox-field
           [:input attrs]]

      (-> {:class class}
          (h/with-input-field-wrapper)
          (h/with-label field-spec)
          (h/with-contents checkbox-field)))))


;; =============================================================================
;; Radiobutton

(defmethod field :radiobutton
  [field-spec]
  (fn [field-spec]
    (letk [[{id ""}
            {class ""}
            value]
           field-spec

           attrs
           (-> (field-spec->attrs field-spec)
               (assoc :type :radio)
               (assoc :checked (when-not (nil? id)
                                 (= value (name id)))))

           input-field
           [:input (assoc attrs :value id)]]

      (-> {:class class :key id}
          (h/with-input-field-wrapper)
          (h/with-label field-spec)
          (h/with-contents input-field)))))


;; =============================================================================
;; Radiogroup

(defmethod field :radiogroup
  [field-spec]
  (fn [field-spec]
    (letk [[{id ""}
            {class ""}
            {value ""}
            ;; TODO configurable
            {options [{:id "1" :label "Eins"}
                      {:id "2" :label "Zwei"}]}
            {data-path id}]
           field-spec

           on-change
           (h/value-on-change-handler data-path)

           attrs
           (field-spec->attrs field-spec)

           radiogroup-field
           [:div.radiogroup
            (for [option options]
              (let [f-spec
                    (assoc option
                           :type :radiobutton
                           :value value
                           :data-path data-path)]
                ^{:key (:id f-spec)}
                [field f-spec]))]]

      (-> {:class class}
          (h/with-input-field-wrapper)
          (h/with-label field-spec)
          (h/with-contents radiogroup-field)))))


;; =============================================================================
;; Datepicker

(defmethod field :date
  [field-spec]
  (fn [field-spec]
    (letk [[{class nil}]
           field-spec

           attrs
           (field-spec->attrs field-spec)

           input-field
           [:input attrs]]

      (-> {:class class}
          (h/with-input-field-wrapper)
          (h/with-label field-spec)
          (h/with-contents input-field)))))


;; =============================================================================
;; HTML 5 Range

(defmethod field :range
  [field-spec]
  (fn [field-spec]
    (letk [[{class nil}]
           field-spec

           attrs
           (field-spec->attrs field-spec)

           input-field
           [:input attrs]]

      (-> {:class class}
          (h/with-input-field-wrapper)
          (h/with-label field-spec)
          (h/with-contents input-field)))))
