(ns de.phenomdevel.components.discord-bot-sender
  (:require
   [clojure.pprint :as pprint]

   [taoensso.timbre :as log]
   [discljord.messaging :as dism]
   [com.stuartsierra.component :as c]

   [de.phenomdevel.protocols.idiscord-bot-sender :as idiscord-bot-sender]
   [de.phenomdevel.protocols.idiscord-bot-guild-state :as idiscord-bot-guild-state]))


;; =============================================================================
;; Private Helper

(defn- pprint-str
  [x]
  (-> x
      (pprint/pprint)
      (with-out-str)))


;; =============================================================================
;; Component

(defrecord DiscordBotSender [token default-channel-id discord-bot-connection discord-bot-guild-state]
  c/Lifecycle
  (start
    [this]
    (log/info "[DiscordBotSender] Started.")
    this)

  (stop
    [this]
    (log/info "[DiscordBotSender] Stopped.")
    this)

  idiscord-bot-sender/IDiscordBotSender
  (create-message!
    [this channel-id message]
    (idiscord-bot-sender/create-message! this channel-id message {}))

  (create-message!
    [this channel-id message embed]
    (let [channel-id
          (or channel-id default-channel-id)]

      (log/info (format "[DiscordBotSender] Sending new message with embed to channel `%s`" channel-id))
      (log/trace (format "[DiscordBotSender] Message\n%s" (pprint-str {:channel-id channel-id :message message :embed embed})))
      (dism/create-message! (:sender-ch discord-bot-connection) channel-id :content (or message "") :embed embed)))

  (edit-message!
    [this channel-id message-id message]
    (idiscord-bot-sender/edit-message! this channel-id message-id message {}))

  (edit-message!
    [this channel-id message-id message embed]
    (let [channel-id
          (or channel-id default-channel-id)]

      (log/info (format "[DiscordBotSender] Editing message `%s`" message-id))
      (log/trace (format "[DiscordBotSender] New message\n%s" (pprint-str {:channel-id channel-id :message message})))
      (dism/edit-message! (:sender-ch discord-bot-connection) channel-id message-id :content (or message "") :embed embed)))

  (create-dm-message!
    [this user-id message]
    (idiscord-bot-sender/create-dm-message! this user-id message {}))

  (create-dm-message!
    [this user-id message embed]
    (let [dm-channel-id
          (-> @(dism/create-dm! (:sender-ch discord-bot-connection) user-id)
              :id)]

      (log/info (format "[DiscordBotSender] Send new direct message to `%s`" user-id))
      (log/trace (format "[DiscordBotSender] Message\n%s" (pprint-str {:channel-id dm-channel-id :message message})))

      (dism/create-message! (:sender-ch discord-bot-connection) dm-channel-id :content message :embed embed)))

  (delete-message!
    [this channel-id message-id]
    (log/info (format "[DiscordBotSender] Delete message `%s`" message-id))
    (dism/delete-message! (:sender-ch discord-bot-connection) (or channel-id default-channel-id) message-id))

  (create-reaction!
    [this channel-id message-id emoji-id]
    (let [emoji-string
          (idiscord-bot-guild-state/get-emoji-string discord-bot-guild-state (:guild-id discord-bot-connection) emoji-id)]

      (log/info (format "[DiscordBotSender] Create reaction `%s` on message `%s`" emoji-string message-id))

      (dism/create-reaction! (:sender-ch discord-bot-connection) (or channel-id default-channel-id) message-id emoji-string)))

  (remove-reaction!
    [this channel-id message-id emoji-id user-id]
    (let [emoji-string
          (idiscord-bot-guild-state/get-emoji-string discord-bot-guild-state (:guild-id discord-bot-connection) emoji-id)]

      (log/info (format "[DiscordBotSender] Remove reaction `%s` for user `%s` on message `%s`" emoji-string user-id message-id))

      (dism/delete-user-reaction! (:sender-ch discord-bot-connection) (or channel-id default-channel-id) message-id emoji-string user-id)))

  (remove-all-reactions!
    [this channel-id message-id]
    (log/info (format "[DiscordBotSender] Removing all reactions on message `%s`" message-id))
    (dism/delete-all-reactions! (:sender-ch discord-bot-connection) (or channel-id default-channel-id) message-id))

  ,,,)


;; =============================================================================
;; Public API

(defn new-discord-bot-sender
  [config]
  (map->DiscordBotSender config))
