(ns de.otto.tesla.kafkareader.kafkareader
  (:require [clojure.core.async :refer [>!!]]
            [com.stuartsierra.component :as component]
            [metrics.meters :as meters]
            [clj-kafka.consumer.zk :as zk-consumer]
            [clj-kafka.core :as kafka]
            [clojure.tools.logging :as log])
  (:import (java.nio.channels ClosedByInterruptException)
           (org.I0Itec.zkclient.exception ZkInterruptedException)))

(defn config-consumer [kafka-config]
  {"zookeeper.connect" (:zookeeper-connect kafka-config)
   "group.id"          (:group-id kafka-config)
   "auto.offset.reset" (or (:auto-offset-reset kafka-config) "largest")
   ;"auto.commit.enable"    "false"
   ;"rebalance.max.retries" "20"
   ;"rebalance.backoff.ms" "4000"
   ;"fetch.message.max.bytes" (:consumer-fetch-message-max-bytes config)
   })

(defn extract-messages [messages]
  (map #(when-let [value (:value %)] (String. value)) messages))

(defn a-new-consumer [kafka-config]
  (let [conf (config-consumer kafka-config)]
    (log/info "kafka config: " conf)
    (zk-consumer/consumer conf)))

(defn stop-forwarding-if [stopping? messages]
  (if (not @stopping?)
    messages))

(defn handle-messages [kafka-config-key out-channel msgs]
  (doseq [msg msgs]
    (meters/mark! (meters/meter ["kafka-reader" (name kafka-config-key) "message-read"]))
    (>!! out-channel msg)))

(defn handle-kafka-messages [{:keys [out-channel stopping? transform-fn kafka-config-key]} messages]
  (some->> messages
           (stop-forwarding-if stopping?)
           (extract-messages)
           (transform-fn)
           (handle-messages kafka-config-key out-channel)))

(defn kafka-messages [c kafka-config]
  (zk-consumer/messages c (:topic kafka-config)))

(defn start-consumer [self kafka-config]
  (let [stopping? (:stopping? self)]
    (loop []
      (when-not @stopping?
        (try
          (kafka/with-resource [c (a-new-consumer kafka-config)]
                               zk-consumer/shutdown
                               (handle-kafka-messages self (kafka-messages c kafka-config)))
          (catch InterruptedException _
            (log/info "Handling InterruptedException"))
          (catch ClosedByInterruptException _
            (log/info "Handling java.nio.channels.ClosedByInterruptException"))
          (catch ZkInterruptedException _
            (log/info "Handling org.I0Itec.zkclient.exception.ZkInterruptedException"))
          (catch Exception e
            (log/error e "Error while consuming messages: " (.getMessage e))
            (log/error "Next try in 3 seconds")
            (Thread/sleep 3000)))
        (recur)))))

(defrecord KafkaReader [config out-channel kafka-config-key]
  component/Lifecycle
  (start [self]
    (log/info "-> starting KafkaReader for Queue " kafka-config-key)
    (let [new-self (assoc self
                     :stopping? (atom false)
                     :last-event-written-at (atom 0))]
      (assoc new-self :thread (doto
                                (Thread. #(start-consumer new-self (get-in config [:config :kafka kafka-config-key])) "kafka-reader")
                                (.setDaemon true)
                                (.start)))))
  (stop [self]
    (log/info "<- stopping KafkaReader")
    (reset! (:stopping? self) true)
    (.interrupt (:thread self))
    (log/info "<- waiting for KafkaReader-Thread to stop")
    (.join (:thread self))
    (log/info "<- stopped KafkaReader")))

(defn new-kafka-reader
  ([out-channel]
   (map->KafkaReader {:out-channel      out-channel
                      :kafka-config-key :default
                      :transform-fn     identity}))
  ([out-channel kafka-config-key]
   (map->KafkaReader {:out-channel      out-channel
                      :kafka-config-key kafka-config-key
                      :transform-fn     identity}))
  ([out-channel kafka-config-key filter-fun]
   (map->KafkaReader {:out-channel      out-channel
                      :kafka-config-key kafka-config-key
                      :transform-fn     filter-fun})))
