(ns de.blubberquark.microdata.export
  (:require [de.robertpfeiffervz.microdata.extract :as ex])
  (:import (org.joda.time.format
	    DateTimeFormatter
	    DateTimeFormatterBuilder
	    ISODateTimeFormat)
	   org.joda.time.ReadableInstant))

(def org-time-format
     (-> (new DateTimeFormatterBuilder)
	 (.appendYear 4 4)
	 (.appendLiteral "-")
	 (.appendMonthOfYear 2)
	 (.appendLiteral "-")
	 (.appendDayOfMonth 2)
	 (.appendLiteral " ")
	 (.appendDayOfWeekText)
	 (.appendLiteral " ")
	 (.appendHourOfDay 2) 
	 (.appendLiteral ":")
	 (.appendMinuteOfHour 2)
	 (.toFormatter)))

(defmulti render (fn [a & rest] (type a)))

(defmethod render java.net.URL [url & options]
	   (.toExternalForm url))

(defmethod render org.joda.time.ReadableInstant [time & {active :active}]
	   (let [[open close] (if active ["<" ">"] ["[" "]"])]
	     (str open (.print org-time-format-inactive time) close)))

(defmethod render :default [x & _]
	   (str x))

(def *org-known-attributes*
     #{:startDate :url :description :photo :summary :geo})

(defn org-location [loc]
  (if (:address loc)
    (recur (:address loc))
    (str (:street-address loc) ", " (:postal-code loc) " " (:locality loc))))

(defn org-time [time]
  (.print org-time-format time))

(defmacro property-drawer [& body]
  `(do
     (println ":PROPERTIES:")
     ~@body
     (println ":END:")))

(defn org-link [url text]
  (if url
    (str "[[" (render url) "][" text "]]")
    ""))

(defn print-org-event [item]
  (println '* (:summary item))
  (property-drawer
   (when (:location item)
     (println ":LOCATION:" (org-location (:location item))))
   (doall
    (for [[key value] item]
      (when-not (or (*org-known-attributes* key)
		    (coll? value))
       (println (.toUpperCase (str key ":")) (render value))))))
  (println "SCHEDULED:" (org-time (:startDate item)))
  (println (org-link (:url item) "link")
	   (org-link (:photo item) "photo"))
  (println (:description item)))
