(ns sqlosure.lang
  (:require [active.clojure.condition :as c]
            [active.clojure.lens :as lens]
            [active.clojure.monad :as monad]
            [active.clojure.record :refer [define-record-type]]

            [sqlosure.core :as s]
            [sqlosure.db-connection :as db]
            [sqlosure.relational-algebra :as rel]
            [sqlosure.query-comprehension :as q]))

;;   The higher order language should include
;; 1 ways to interact at a CRUD level
;; 2 higher order functions (map, fold, filter)
;; 3 an environment to refer to to get the connection and backend-implementation ndetails

;; 1 CRUD
(define-record-type Create
  (^:private make-create table record) create?
  [table create-table
   record create-record])

(defn create!
  [table record]
  (when-not (rel/base-relation? table)
    (c/assertion-violation `create! "table must be a base-relation" table))
  (when-not (sequential? record)
    (c/assertion-violation `create! "record must be sequential" record))
  (make-create table record))

(defn create-many!
  [table & records]
  (monad/sequ_ (map (partial create! table) records)))

(define-record-type Read
  (^:private make-read table condition projection) read?
  [table read-table
   condition read-condition
   projection read-projection])

(defn read!
  ([table]
   (when-not (rel/base-relation? table)
     (c/assertion-violation `read! "table must be a base-relation" table))
   (make-read table nil nil)))

(defn read-restrict!
  [read condition]
  (lens/shove read read-condition condition))

(defn read-project!
  [read projection]
  (lens/shove read read-projection projection))

(define-record-type Update
  (^:private make-update table condition replacement) update?
  [table update-table
   condition update-condition
   replacement update-replacement])

(defn update!
  [table condition replacement]
  (when-not (rel/base-relation? table)
    (c/assertion-violation `update! "table must be a base-relation" table))
  (make-update table condition replacement))

(define-record-type Delete
  (^:private make-delete table condition) delete?
  [table delete-table
   condition delete-condition])

(defn delete!
  [table condition]
  (when-not (rel/base-relation? table)
    (c/assertion-violation `delete! "table must be a base-relation" table))
  (make-delete table condition))


;; 2 higher order functions
(defn filter!
  [condition table]
  (s/query [rel (s/<- table)]
           (s/restrict! (condition rel))
           (s/project rel)))

;; 3 env
(define-record-type Env
  (make-env conn implementation) env?
  [conn env-conn
   implementation env-implementation])


;; runner
(defn run
  [run-any state env m]
  (let [e    (::env env)
        conn (env-conn e)
        impl (env-implementation e)]
    (cond
      (create? m)
      (let [t (create-table m)
            r (create-record m)]
        (apply db/insert! conn t m))

      (read? m)
      (let [t          (read-table m)
            condition  (read-condition m)
            projection (read-projection m)]
        (db/run-query conn (s/query [r (s/<- t)]
                                    (if condition
                                      (s/restrict! condition)
                                      (monad/return))
                                    (if projection
                                      (s/project (projection r))
                                      (s/project r))))))))



;; Examples

(def person-table (s/table "person" [["id" s/$integer-t]
                                     ["name" s/$string-t]]))

(defn person-id=
  [id]
  (fn [p]
    (s/$= (s/! p "id") (s/$integer id))))

(defn set-person-name
  [n]
  (fn [id _]
    {"id"   id
     "name" n}))

(monad/monadic (create! person-table [0 "Marco Schneider"])
               (create! person-table [1 "Markus Schlegel"])
               [marco (-> (read! person-table)
                          (read-restrict! (person-id= 0)))]
               (update! person-table (person-id= 0) (lens/shove marco (lens/member "name") "Mike Sperber"))
               (delete! person-table (person-id= 1))
               (-> (read! (person-table))
                   (read-project! (fn [res]
                                    ["name" (s/! res "name")]))))
