(ns reacl-c-basics.forms.core
  (:require [reacl-c.core :as c :include-macros true]
            [reacl-c.dom :as dom :include-macros true]
            [active.clojure.functions :as f]
            [active.clojure.lens :as lens]
            goog.object))

(let [on-change (fn [_ ev]
                  (.-value (.-target ev)))]
  (c/defn-item ^:private input-base [attrs]
    (c/with-state-as value
      (dom/input (dom/merge-attributes {:value value
                                        :onchange on-change}
                                       attrs)))))

(dom/defn-dom ^:private input-parsed-base
  "Special attributes:
   ::unparse must return a string
   if ::parse throws, :onerror is emitted."
  [attrs]
  (c/with-state-as [value local :local {:text nil :previous nil :error nil}]
    (let [parse (::parse attrs)
          unparse (::unparse attrs)]
      (c/fragment
       ;; init initially, or reinit when new value from outside.
       (when (or (not= (:previous local) value)
                 (nil? (:text local)))
         (c/init [value {:text (unparse value) :previous value :error nil}]))
       
       (-> (c/focus (lens/>> lens/second :text (lens/default ""))
                    (input-base (dom/merge-attributes {:onblur (f/constantly (c/return :action ::reset))}
                                                      (dissoc attrs :onerror ::parse ::unparse))))
           (c/handle-state-change
            (fn [_ [value local]]
              (try (let [text (or (:text local) "")
                         v (parse text)]
                     ;; (c/call (:onerror attrs) nil) ??? always, or when in error state previously?
                     (c/return :state [v {:text text :previous v :error nil}]))
                   (catch :default e
                     (c/merge-returned (c/return :state [value (assoc local :error e)])
                                       (c/call (:onerror attrs) e))))))
           (c/handle-action
            (fn [[value local] a]
              (if (= ::reset a)
                ;; if in error state, then reset text; otherwise keep as user entered it.
                (if (some? (:error local))
                  [value (assoc local :text (unparse value))]
                  [value local])
                (c/return :action a)))
            ))))))


(defn native-type [type]
  (assert (or (nil? type) (string? type)) type)
  {::base input-base
   ::attributes {:type type}})

(defn parsed-type [native-type parse unparse]
  (assert (string? native-type))
  {::base input-parsed-base
   ::attributes {:type native-type
                 ::parse parse
                 ::unparse unparse}})

(defn restrict-type [type f]
  (if (some? (::parse (::attributes type)))
    (update-in type [::attributes ::parse]
               (fn [g]
                 (f/comp f g)))
    (parsed-type type f identity)))

(def type:number
  (parsed-type "number"
               (fn parse [s]
                 ;; TODO: exception
                 ;; Note: "" parses as NaN although it's not isNaN; parseFloat ignores trailing extra chars; but isNaN does not:
                 (let [x (when (not (js/isNaN s))
                           (js/parseFloat s))]
                   (if (js/isNaN x)
                     nil
                     x)))
               str))

(def type:int
  (-> type:number
      (restrict-type (fn [v]
                       (if (integer? v)
                         v
                         ;; TODO: which exception?
                         (throw (js/Error. "Not an integer.")))))))

;; type:optional-int ... type:non-empty-string ?

(dom/defn-dom input [attrs]
  (let [t (:type attrs)]
    (if (or (nil? t) (string? t))
      (input-base attrs)
      ((::base t) (dom/merge-attributes (::attributes t) (dissoc attrs :type))))))

(defn- option-value-placeholder [value]
  (cond
    (nil? value) ""
    :else (pr-str value)))

(defrecord ^:private OptionValue [placeholder value])

(dom/defn-dom option [attrs & contents]
  (let [v (:value attrs)
        placeholder (option-value-placeholder v)]
    ;; replace :value with a (generated) string, and report actual-value as an action to surrounding select/datalist.
    (c/fragment
     (apply dom/option (assoc attrs
                              :value placeholder)
            contents)
     (when (not= v placeholder) ;; = not string?
       (c/init (c/return :action (OptionValue. placeholder v)))))))

(dom/defn-dom optgroup [attrs & options]
  (apply dom/optgroup attrs options))

(dom/defn-dom select "A typed select element." [attrs & options]
  (c/with-state-as [value ph-map :local {}]
    (-> (apply dom/select
               ;; TODO: allow user to set custom placeholder in option/or as a function? Then wait for the map to build up here?
               {:value (option-value-placeholder value)
                :onchange (fn [[value ph-map] ev]
                            (let [options (.-options (.-target ev))
                                  placeholder (.-value (aget options (.-selectedIndex options)))]
                              (c/return :state [(get ph-map placeholder placeholder) ph-map])))}
               options)
        (c/handle-action (fn [[st ph-map] a]
                           (if (instance? OptionValue a)
                             (c/return :state [st (assoc ph-map (:placeholder a) (:value a))])
                             (c/return :action a)))))))

#_(dom/defn-dom datalist [attrs & options]
  ;; using a :type ?
  )

(dom/defn-dom textarea [attrs]
  (apply dom/textarea attrs))

(dom/defn-dom form [attrs & content]
  (apply dom/form attrs content))

