(ns leiningen.cofx
  (:require
    [clojure.walk :as walk]
    [cuddlefish.core :as git])
  (:import
    (java.time LocalDateTime)
    (java.time.format DateTimeFormatter)))

(def default-config
  "The default configuration values."
  {:git              "git"
   :describe-pattern git/git-describe-pattern})

(defn git-status-to-version
  [config]
  (let [{:keys [tag ahead ahead? dirty?]} (git/status config)]
    (if-not (string? tag)
      ;; If git status is nil (e.g. IntelliJ reading project.clj) then return...
      "git-tag-unavailable"
      (if (and (not ahead?) (not dirty?))
        tag
        (let [[_ major minor patch suffix] (re-find #"v?(\d+)\.(\d+)\.(\d+)(-.+)?" tag)]
          (if (nil? major)
            ;; If tag is poorly formatted then return...
            "git-tag-invalid"
            (let [patch' (try (Long/parseLong patch) (catch Throwable _ 0))
                  patch+ (inc patch')]
              (str major "." minor "." patch+ suffix "-" ahead "-SNAPSHOT"))))))))

(def x->f
  {:lein-cofx/iso-date-time         (fn [_] (.format (LocalDateTime/now) DateTimeFormatter/ISO_DATE_TIME))
   :lein-cofx/iso-week-date         (fn [_] (.format (LocalDateTime/now) DateTimeFormatter/ISO_WEEK_DATE))
   :lein-cofx/git-status-to-version git-status-to-version
   :lein-cofx/username              (fn [_] (System/getProperty "user.name"))})

(defn middleware
  [{:keys [cofx]
    :as   project}]
  (let [config   (merge default-config cofx)
        project' (walk/prewalk
                   (fn [x]
                     (if-let [f (get x->f x)]
                       (f config)
                       x))
                   project)]
    project'))