# anomalies [![Build Status](https://travis-ci.org/dawcs/anomalies.svg?branch=master)](https://travis-ci.org/dawcs/anomalies)

"Errors as data" approach and tooling inspired by https://github.com/cognitect-labs/anomalies

Library is currenty in alpha, so anything can be changed in future versions.

## Usage

Leiningen dependency information:
```
[dawcs/anomalies "0.1.0"]
```

Maven dependency information:
```
<dependency>
  <groupId>dawcs</groupId>
  <artifactId>anomalies</artifactId>
  <version>0.1.0</version>
</dependency>
```

```clojure
(require '[anomalies.core :as a :refer [!!]])

;; anomaly is just a map
(defn get-value-with-fake-connection
  [value connection-timeout deref-timeout]
  (-> (Thread/sleep connection-timeout)
      (future value)
      (deref deref-timeout {:anomaly/category :busy
                            :anomaly/message "Connection timeout"})))

(get-value-with-fake-connection "hello" 1 100) ;; => "hello"
(get-value-with-fake-connection "hello" 100 1) ;; => #:anomaly{:category :busy, :message "Connection timeout"}

;; is map syntax looking too verbose? there's nice helper
(a/anomaly :busy "Connection timeout")  ;; => #:anomaly{:category :busy, :message "Connection timeout"}

;; still too much typing? there's short form
(!! :busy "Connection timeout") ;; => #:anomaly{:category :busy, :message "Connection timeout"}

;; want even more succinct form?
(a/busy "Connection timeout") ;; => #:anomaly{:category :busy, :message "Connection timeout"}

;; no need for message?
(a/busy) ;; => #:anomaly{:category :busy}

;; happy with one category to cover them all? :fault is default one
(!!) ;; => #:anomaly{:category :fault}

;; need to store some additional data along with message?
(!! :forbidden "Cannot perform operation" {:user-id 2128506}) ;; => #:anomaly{:category :forbidden, :message "Cannot perform operation", :data {:user-id 2128506}}
(!! "Cannot perform operation" {:user-id 2128506}) ;; => #:anomaly{:category :fault, :message "Cannot perform operation", :data {:user-id 2128506}}

;; prefer just data instead of message?
(!! :forbidden {:user-id 2128506}) ;; => #:anomaly{:forbidden :fault, :data {:user-id 2128506}}
(!! {:user-id 2128506});; => #:anomaly{:category :fault, :data {:user-id 2128506}}

;; so what we can do with anomaly?
;; first, we can check if value is anomaly
(a/anomaly? (!!)) ;; => true
(a/anomaly? {:user 1}) ;; => false
(a/anomaly? (Exception. "Bad stuff")) ;; => false

;; do you prefer imperative error checking?
(let [result (do-stuff)]
  (if (a/anomaly? result)
    (say-oooops result)
    (say-hooray result)))

;; that's fine but also there are much better functional tools
;; let's start from functions
(inc 1) ;; => 2
(inc (!!)) ;; BOOOM!!! Unhandled java.lang.ClassCastException clojure.lang.PersistentArrayMap cannot be cast to java.lang.Number

;; let's make it aware of anomalies
(def ainc (a/aware inc))
(ainc 1) ;; => 2
(ainc (!!)) ;; => #:anomaly{:category :fault}

;; aware accepts function as first argument which makes it perfect for ->> macro
(->> 1 (a/aware inc) (a/aware str)) ;; "2"
(->> (!!) (a/aware inc) (a/aware str)) ;; => #:anomaly{:category :fault}

;; anomalies aware chain can be also done using macros similar to some-> and some->>
(a/aware-> 1 inc) ;; => 2
(a/aware-> (!!) inc) ;; => #:anomaly{:category :fault}
(a/aware-> 1 (!!) inc) ;; => #:anomaly{:category :fault, :data 1}

(a/aware->> [1 2 3] (map inc)) ;; => (2 3 4)
(a/aware->> (!!) (map inc))  ;; => #:anomaly{:category :fault}
(a/aware->> [1 2 3] (!! :conflict "Ooops") (map inc)) ;; => #:anomaly{:category :conflict, :message "Ooops", :data [1 2 3]}

;; there's also functional version of chain if macros magic is not desired
(a/aware-chain [1 2 3] (partial map inc)) ;; => (2 3 4)
(a/aware-chain [1 2 3] (partial a/unsupported) (partial map inc));; => #:anomaly{:category :unsupported, :data [1 2 3]}

;; with-anomaly can help in the case if we want to handle anomaly somehow and then return it
;; the following code prints anomaly message and category and returns given anomaly
(a/with-anomaly
  (a/forbidden "Bad password" {:user-id 2128506})
  (comp prn :anomaly/message)
  (comp prn :anomaly/category))
;; => #:anomaly{:category :forbidden, :message "Bad password", :data {:user-id 2128506}}

;; if given value is not anomaly, then chain is skipped and this value is returned
(a/with-anomaly 1 (comp prn :anomaly/message)) ;; => 1

;; if some chain function returns another anomaly, it's passed to next function in chain
(a/with-anomaly
  (a/conflict "Uh-oh")
  (fn [x] (a/busy (a/message x) x))
  (comp prn a/category)) ;; prints :busy
;; => #:anomaly{:category :busy, :message "Uh-oh", :data #:anomaly{:category :conflict, :message "Uh-oh"}}

;; with-anomaly and achain accepts value as first argument so can be used together in -> macro
(-> "hello"
     a/anomaly
     (a/achain clojure.string/upper-case)
     (a/with-anomaly (comp prn a/message))) ;; prints anomaly message to console
;; => #:anomaly{:category : fault, :message "hello"}

;; returning to imperative error handling example, we can rewrite it using functional chain
(-> (do-stuff)
    (a/achain say-hooray)
    (a/with-anomaly say-oooops))

;; what about fallback to default?
;; either allows to choose among anomaly and some default non-anomaly value
(a/either (!!) 1) ;; => 1
(apply a/either [(a/busy) (a/fault) (a/conflict) (a/not-found) 1]) ;; => 1

;; if only anomaly values given, either returns last given value
(a/either (a/busy) (a/unsupported)) ;; => {:anomaly/category :unsupported}

;; and it's also very useful in -> marco chain
(-> "hello"
    a/anomaly
    (a/achain clojure.string/upper-case)
    (a/with-anomaly prn) ;; prints anomaly to console
    (a/either "goodbye"))
;; => "goodbye"

;; alet is anomalies aware version of let macro
(a/alet [a 1 b 2] (+ a b)) ;; => 3
(a/alet [a 1 b (!!)] (+ a b)) ;; => #:anomaly{:category : fault}

;; and sometimes you need to deal with Java exceptions
(a/catch-all (/ 1 0)) ;; => #:anomaly{:category :fault, :message "Divide by zero", :data #error { ... }

;; need to throw some exceptions and catch other ones?
(a/catch-except #{NullPointerException} (/ 1 0)) ;; => #:anomaly{:category :fault, :message "Divide by zero", :data #error { ... }
(a/catch-except #{NullPointerException} (/ 1 nil)) ;; throws java.lang.NullPointerException

;; catching only certain exceptions required?
(a/catch-only #{NullPointerException} (/ 1 0)) ;; throws java.lang.ArithmeticException
(a/catch-only #{NullPointerException} (/ 1 nil))  ;; => #:anomaly{:category :fault, :data #error { ... }

;; and full control for setting category, message and data of produced anomaly is also possible
(a/catch-anomaly
 {:category :conflict
  :message "Uh-oh"
  :data (atom 1)}
 (/ 1 0)) ;; => #:anomaly{:category :conflict, :message "Uh-oh", :data #atom[1 0x306e335]}

(a/catch-anomaly
 {:message "Uh-oh"
  :only #{ArithmeticException}}
 (/ 1 0)) ;; => #:anomaly{:category :fault, :message "Uh-oh", :data #error { ... }

(a/catch-anomaly
 {:except #{ArithmeticException}}
 (/ 1 0)) ;; throws java.lang.ArithmeticException

;; WARNING! at the moment, exceptions hierarchy doesn't affect processing, e.g. specifying Throwable doesn't will not catch any of it's descendants

(a/with-default-category
  :conflict
  (a/catch-all (/ 1 0))) ;; => #:anomaly{:category :conflict, :message "Divide by zero", :data #error { ... }
(!!) ;; => #:anomaly{:category :fault}

;; need to assign some category for certain class of exceptions? that's also possible
(a/with-exception-categories
  {NullPointerException :unsupported}
  (a/catch-all (+ 1 nil))) ;; => #:anomaly{:category :unsupported, :data #error { ... }
```

## TODO

- Make documentation more readable
- ClojureScript support

## License

Copyright © 2017 Cognitect, Inc. All rights reserved.
Copyright © 2018 DAWCS

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
