(ns saldo.core
  "Core transaction functions and support for dispatch by map."
  (:require [datahike.api :as d]
            [clojure.java.io :as io]
            [saldo.transact :as transact]
            [saldo.query :as query]))


(defn transfer [conn from to amount unit block-height]
  (let [txid (d/tempid -1)]
    (d/transact conn [[:db.fn/call transact/transfer
                       [:account/name from]
                       [:account/name to]
                       amount
                       unit]
                      (merge
                       (when block-height
                         {:block/height block-height})
                       {:db/id txid,
                        :transfer/from [:account/name from] 
                        :transfer/to [:account/name to]
                        :transfer/amount amount
                        :transfer/unit unit
                        :transfer/recorded (java.util.Date.)})])))


(defn can-transfer? [conn from to amount unit block-height]
  (let [txid (d/tempid -1)]
    (d/with conn [[:db.fn/call transact/transfer
                    [:account/name from]
                    [:account/name to]
                    amount
                    unit]
                   (merge
                    (when block-height
                      {:block/height block-height})
                    {:db/id txid,
                     :transfer/from [:account/name from] 
                     :transfer/to [:account/name to]
                     :transfer/amount amount
                     :transfer/unit unit
                     :transfer/recorded (java.util.Date.)})])))

(comment
  @(transfer conn 1 2 100 42)
  )


(defn delegate [conn from to block-height]
  ;; TODO support retraction (nil)
  (let [txid (d/tempid -1)]
    (d/transact conn [{:db/id [:account/name from]
                       :voting/delegate [:account/name to]}
                      (merge
                       (when block-height
                         {:block/height block-height})
                       {:db/id txid
                        :voting/from [:account/name from]
                        :voting/to [:account/name to]
                        :voting/recorded (java.util.Date.)})])))


(defmulti dispatch
  "Dispatch on database operations."
  :type)


(defmethod dispatch :transfer
  [{:keys [from to amount unit block/height]} conn]
  @(transfer conn from to amount unit height))


(defmethod dispatch :delegate
  [{:keys [from to block/height]} conn]
  @(delegate conn from to height))


(defmulti valid
  "Check wheter a desired operation is valid."
  :type)


(defmethod valid :transfer
  [{:keys [from to amount unit block/height]} conn]
  (try
    (can-transfer? conn from to amount unit height)
    true
    (catch Exception e
      false)))


(defmethod valid :delegate
  [{:keys [from to block/height]} conn]
  true)





(comment
  (d/q query/transfers-query
       @conn))


;; phase 1
;; hardcoded txs types

(comment
  (dispatch {:type :transfer
             :amount 5
             :from 1
             :to 2
             :block/height 42}
            conn)


  (dispatch {:type :delegate
             :from 1
             :to 2
             :block/height 45}
            conn))

