(ns datomic.schema
  (:refer-clojure :exclude [ref partition])
  (:require [clojure.string :as str]
            [medley.core :refer [map-keys map-vals]]))

(defmacro compile-if
  "Evaluate `exp` and if it returns logical true and doesn't error, expand to
  `then`.  Else expand to `else`. "
  ([exp then]
   `(compile-if ~exp ~then nil))
  ([exp then else]
   (if (try (eval exp) true
            (catch Throwable _ false))
     then
     else)))

(defmacro compile-when [exp & body]
  `(compile-if ~exp (do ~@body)))

(defn prismatic-schema [ent])

(defrecord Entity [partition ns attrs schemas enums])

(compile-when
 (require 'schema.core)

 (extend-protocol schema.core/Schema
   Entity
   (spec [this]
     (schema.core/spec  (:prismatic/schema this)))
   (explain [this]
     (schema.core/explain (:prismatic/schema this)))

   clojure.lang.Var
   (spec [this]
     (schema.core/spec    @this))
   (explain [this]
     (schema.core/explain @this)))

 (compile-if
  (require 'datomic.api)
  (def DbId datomic.db.DbId)
  (def DbId String))

 (defn prismatic-schema [ent]
   (if-let [es (seq (:enums ent))]
     (apply schema.core/enum es)
     (->> (assoc (:attrs ent) :db/id :ident)
          (map-keys schema.core/optional-key)
          (map-vals (fn [v]
                      (if (var? v)
                        (schema.core/either Long DbId String v [v])
                        ({:string  String
                          :boolean Boolean
                          :long    Long
                          :bigint  BigInteger
                          :float   Float
                          :double  Double
                          :bigdec  java.math.BigDecimal
                          :instant java.util.Date
                          :uuid    java.util.UUID
                          :uri     java.net.URI
                          :bytes   (Class/forName "[B")
                          :ident   schema.core/Any}
                         (keyword (name v))))))))))

(compile-if
 (require 'datomic.api)
 (defn tempid
   ([part]
    (datomic.api/tempid part))
   ([part val]
    (datomic.api/tempid part val)))
 (defn tempid
   ([_]
    (str (java.util.UUID/randomUUID)))
   ([part _]
    (tempid part))))

(defn partition [ent partition]
  (assoc ent :partition partition))

(defn attr-ns [ent ns]
  (assoc ent :ns ns))

(defn camel->ns [clazz]
  (-> (name clazz)
      (str/replace-first #"\w" str/lower-case)
      (str/replace #"[A-Z]" #(str "." (str/lower-case %)))
      symbol))

(defn qualify-ident [ns ident]
  (cond
    (nil? ns) (keyword ident)
    (string? ident) (qualify-ident ns (keyword ident))
    (namespace ident) ident
    :else (keyword (name ns) (name ident))))

(defn attrs
  ([ent decl & decls]
   (reduce attrs ent (cons decl decls)))
  ([{:as ent ns :ns} [ident type opts]]
   (let [qualified-ident (qualify-ident ns ident)
         type-keyword    (if (var? type)
                           :ref
                           type)
         {:as   schema
          :keys [db/unique db/cardinality]}
         (merge {:db/id                 (tempid :db.part/db)
                 :db/ident              qualified-ident
                 :db/valueType          (qualify-ident 'db.type type-keyword)
                 :db/cardinality        :db.cardinality/one
                 :db.install/_attribute :db.part/db}
                (map-keys #(qualify-ident 'db %) opts))

         schema (cond-> schema
                  true   (assoc :db/cardinality
                                (qualify-ident 'db.cardinality cardinality))
                  unique (assoc :db/unique
                                (qualify-ident 'db.unique unique)))]
     (-> ent
         (vary-meta assoc ::entity? true)
         (update :schemas conj schema)
         (assoc-in [:attrs qualified-ident] type)))))

(defn enums
  ([ent e & es]
   (reduce enums ent (cons e es)))
  ([{:as ent :keys [ns partition]} e]
   (let [e (if (map? e)
             (assoc e :db/id (or (:db/id e) (tempid partition)))
             {:db/id    (tempid partition)
              :db/ident (qualify-ident ns e)})]
     (-> ent
         (vary-meta assoc ::enum? true)
         (update :schemas conj
                 (vary-meta e assoc ::enum? true))
         (update :enums conj (:db/ident e))))))

(defn as [{:as ent :keys [ns attrs partition]} x]
  (cond
    (sequential? x)
      (map #(as ent %) x)
    (-> ent meta ::enum?)
      (qualify-ident ns x)
    :else
      (let [x (map-keys #(qualify-ident ns %) x)
            x (reduce-kv (fn [x k v]
                           (if (and (x k) (var? v))
                             (update x k #(as @v %))
                             x))
                         x
                         attrs)]
        (assoc x :db/id
               (or (:db/id x) (tempid partition))))))

(defmacro defentity [name & declares]
  `(do
     (def ~name
       (-> {:partition        :db.part/user
            :ns               '~(camel->ns name)
            :attrs            {}
            :schemas          []
            :enums            #{}
            :prismatic/schema nil
            :spec             nil}
           (map->Entity)
           ~@declares
           (#(assoc % :prismatic/schema (prismatic-schema %)))))
     (defn ~(symbol (str "->" name)) [m#]
       (as ~name m#))))

(defn schema-txes [entities]
  (let [partitions (map :partition entities)
        schemas    (mapcat :schemas entities)
        enum?      #(::enum? (meta %))]
    [(for [p partitions]
       {:db/id                 (tempid :db.part/db)
        :db/ident              p
        :db.install/_partition :db.part/db})
     (remove enum? schemas)
     (filter enum? schemas)]))
