
(in-ns 'datomic.schema)
(require '[clojure.spec.alpha :as s])


(letfn [(?type [& clazzs]
          (clojure.core/fn [x]
            (some #(instance? % x) clazzs)))]
  (s/def :db.type/keyword (?type clojure.lang.Keyword))
  (s/def :db.type/string  (?type String))
  (s/def :db.type/boolean (?type Boolean))
  (s/def :db.type/long    (?type Integer Long))
  (s/def :db.type/bigint  (?type BigInteger clojure.lang.BigInt))
  (s/def :db.type/float   (?type Float))
  (s/def :db.type/double  (?type Double))
  (s/def :db.type/bigdec  (?type BigDecimal))
  (s/def :db.type/uuid    (?type java.util.UUID))
  (s/def :db.type/lookup  (s/cat :key keyword?
                                 :val :db/valueType))
  (s/def :db.type/ident (s/or :ident  keyword?
                              :id     int?
                              :lookup :db.type/lookup))
  (s/def :db/valueType  (s/or :keyword :db.type/keyword
                              :string  :db.type/string
                              :boolean :db.type/boolean
                              :long    :db.type/boolean
                              :bigint  :db.type/bigint
                              :float   :db.type/float
                              :double  :db.type/double
                              :bigdec  :db.type/bigdec)))

(defn- schema-spec [{:as ent :keys [coercions tx-data]}]
  (if (enum? ent)
    (eval `(s/spec ~(into #{}
                          (comp (filter enum?)
                             (map :db/ident))
                          (vals tx-data))))
    (do
      (doseq [[k spec] coercions
              :let     [spec  (s/specize* spec)
                        many? (-> tx-data
                                  (get k)
                                  (:db/cardinality)
                                  #{:db.cardinality/many})]]
        (eval `(s/def ~k ~(if many?
                            (s/or :one  spec
                                  :many (s/+ spec))
                            spec))))
      (eval `(s/keys :opt ~(keys coercions))))))


(extend-protocol s/Specize
  datomic.schema.Schema
  (specize*
    ([this]
     (let [spec (:spec this)]
       (or @spec
           (swap! spec #(or % (schema-spec this))))))
    ([this _]
     (s/specize* this)))

  clojure.lang.Var
  (specize*
    ([v]   (s/specize* (var-get v)))
    ([v _] (s/specize* (var-get v)))))
