(ns ksqldb.client
  (:require [cheshire.core :as json]
            [clj-http.client :as http]
            [clojure.core.async :as async]
            [ksql.gen.core-error-msg :as emsg]
            [ksql.gen.protocol :as p]
            [ksqldb.client.broker-api]
            [ksqldb.client.connector-api]
            [ksqldb.client.schema-api]
            [clojure.tools.logging :as log])
  (:import [io.confluent.ksql.api.client Row StreamedQueryResult BatchedQueryResult KsqlObject]
           [org.reactivestreams Subscriber Subscription]
           [io.confluent.ksql.api.client Client ClientOptions]
           [java.util.concurrent CompletableFuture]
           [java.io BufferedReader DataOutputStream InputStreamReader]
           [java.net HttpURLConnection URL]
           (java.util HashMap)
           (java.time Duration)))


(def ^:dynamic query-context {"ksql.streams.auto.offset.reset", "earliest"
                              "total-empty"                     10})


(defn create-client [^String host-name ^Integer port]
  (let [^ClientOptions client-options (doto (ClientOptions/create)
                                        (.setHost host-name)
                                        (.setPort port))]
    (Client/create client-options)))


(defn close-client [^Client client]
  (.close client))


(defn init-client [context]
  (let [[_ url port] (clojure.string/split (get context :ksql-url) #":")
        host (clojure.string/replace url "//" "")
        client (create-client host (Integer/parseInt port))]
    (assoc context :ksql-client client)))


(defn reset-client [context]
  (let [[_ url port] (clojure.string/split (get context :ksql-url) #":")
        host (clojure.string/replace url "//" "")
        c (get context :ksql-client)
        _ (close-client c)
        client (create-client host (Integer/parseInt port))]
    (assoc context :ksql-client client)))


(defn invoke
  ([op-name]
   (p/invoke {:op op-name}))
  ([op-name request]
   (p/invoke {:op op-name :request request})))



;; latest
;; earliest

(defn print-topic-async [client topic-name offset callback]
  (let [offset (get #{"earliest" "latest"} offset "earliest")
        ;_ (println "--offset" offset)
        ;total-request 1000 ;(or total 1)
        stoped? (promise)
        v (if (= offset "earliest")
            (str "print  '" topic-name "' FROM BEGINNING  ;")
            (str "print  '" topic-name "'  ;")
            )
        ;v (str "print  '" topic-name "' FROM BEGINNING limit " total-request ";")
        url (str (get client :ksql-url) "/query")
        req {:url   url
             :query v}]
    (future
      (try
        (let [url-str (get req :url)
              ^URL obj (URL. url-str)
              ^HttpURLConnection con (doto
                                       (cast HttpURLConnection (.openConnection obj))
                                       (.setRequestMethod "POST")
                                       (.setRequestProperty "Content-Type", "application/vnd.ksql.v1+json; charset=utf-8")
                                       (.setDoOutput true))
              m {:ksql              (get req :query)
                 :streamsProperties {:ksql.streams.auto.offset.reset offset}}
              body (json/generate-string m)
              _ (doto
                  (DataOutputStream. (.getOutputStream con))
                  (.writeBytes body)
                  (.flush)
                  (.close))
              responseCode (.getResponseCode con)
              ^BufferedReader iny (BufferedReader. (InputStreamReader. (.getInputStream con)))]
          (loop [output (.readLine iny)]
            ; (println "--" output)
            (when-not (realized? stoped?)
              (when-not (clojure.string/blank? output)
                ;(callback output)

                (callback output)
                )

              (recur (.readLine iny))
              )
            )
          (log/info "Done for " req)
          (.close iny))
        (catch Exception e
          (do
            #_(clojure.pprint/pprint (Throwable->map e))
            (log/error e)
            (callback (:cause (Throwable->map e)))))))

    stoped?
    )

  )

(defn print-topic-by-time
  [context ^String topic-name offset duration callback]
  (let [consumer-promise (promise)
        producer-promise (print-topic-async context
                                            topic-name
                                            offset
                                            callback)]
    ;(println "--" duration)
    (when producer-promise
      (async/go
        (async/<! (async/timeout duration))
        (deliver producer-promise "done")
        (deliver consumer-promise "done"))
      consumer-promise
      )
    ))



(defn get-client [context] (get context :ksql-client))


(comment

  )


(defn try! [f & a]
  (try
    (apply f a)
    (catch Exception e
      (-> (ex-data e)
          (:body)
          (json/parse-string)))))




(defn ksql
  #_([context ksql-str] (ksql context ksql-str "earliest"))
  [context ksql-str]

  (let [offset (get #{"earliest" "latest"} (get context :offset) "latest")
        req (clojure.string/trim ksql-str)
        req (if (clojure.string/ends-with? req ";")
              req
              (str req ";"))
        ;'cache.max.bytes.buffering' = '0'
        m {:ksql              req
           :streamsProperties {:ksql.streams.auto.offset.reset         offset
                               :ksql.streams.cache.max.bytes.buffering "0"}}]

    (try
       (log/debug "ksql execute " m)
      (let [body (json/generate-string m)]
        (-> (http/post (str (get context :ksql-url) "/ksql")
                       {:body    body
                        :headers {"Content-Type" "application/vnd.ksql.v1+json; charset=utf-8"}
                        :accept  :json})
            (:body)
            (json/parse-string true)
            (first)))
      (catch Exception e
        (throw (emsg/ex-info-for-remote-exec (get context :ksql-url) m (.getMessage e  )))
        )
      )
    )



  )



(defn validate-topic-name [client topic-name]
  (let [topic-name-list (p/invoke  {:op "ksql-show-topics-name"})
        topic-name-set (into #{} topic-name-list)
        ]
    ;(println "--" topic-name-list)
    (when-not (contains? topic-name-set topic-name)
      (throw (ex-info (str "Topic name not found available topic name \n" (clojure.string/join "\n" topic-name-list)) {}))
      )
    )

  )








(defn terminate-push-query [^Client client ^String queryId]
  (try
    (log/info "terminate ksqldb query " queryId)
    (when queryId
      (.get (.terminatePushQuery client queryId)))
    (catch Exception e
      (log/error "ksqldb query termination failed " e))))


(defn convrt-row->map [^Row row]
  (let [column-list (mapv str (seq (.columnNames row)))
        v (mapv (fn [v]
                  (let [w (.getValue row v)
                        ;w (if (instance? ))
                        ]
                    w
                    )
                  ) column-list)
        column-list (into [] (comp (map clojure.string/lower-case) (map keyword)) column-list)]
    (->> (zipmap column-list v)
         (clojure.walk/postwalk (fn [v]
                                  (cond
                                    (instance? io.vertx.core.json.JsonArray v)
                                    (json/parse-string (.toString v) true)

                                    (instance? io.vertx.core.json.JsonObject v)
                                    (json/parse-string (.toString v) true)

                                    :else
                                    v
                                    )
                                  ))

         )

    ))


(defn convert-to-hashMap [m]
  (let [stream-property (HashMap.)
        _ (doseq [[k v] m]
            (.put stream-property k v))]
    stream-property))


;(def )

(defn stream-query
  ([^Client client ^String query-str ^Integer total] (stream-query client query-str total "latest"))
  ([^Client client ^String query-str ^Integer total offset]
   (let [offset (get #{"earliest" "latest"} offset "earliest")
         qc {"ksql.streams.auto.offset.reset", offset}
         ^HashMap m (convert-to-hashMap qc)
         ^StreamedQueryResult streamedQueryResult (.get (.streamQuery client query-str m))
         queryId (.queryID streamedQueryResult)
         total-empty-value (get query-context "total-empty" 1)
         ]
     (loop [out []
            total-empty 0]
       (if (or (> (count out) total)
               (< total-empty-value total-empty))
         (do
           (log/debug "Terminate query " {:total-empty total-empty :total-load (count out) :qeury-id queryId})
           ;           (terminate-push-query client queryId)
           out)
         (let [^Row row (.poll streamedQueryResult (Duration/ofMillis 1000))
               _ (log/debug "stream result " row)
               total-empty (if (= nil row) (inc total-empty) total-empty)
               out (if (= nil row)
                     out
                     (conj out (convrt-row->map row)))]
           (recur out total-empty)))))))



(defn execute-query
  ([^Client client ^String query-str] (execute-query client query-str query-context))
  ([^Client client ^String query-str m]
   (let [^BatchedQueryResult batchedQueryResult (.executeQuery client query-str (convert-to-hashMap m))
         row-coll (.get batchedQueryResult)
         out (mapv convrt-row->map row-coll)
         ;   _ (println "-------------------" out)
         ^String queryId (.get (.queryID batchedQueryResult))]
     (terminate-push-query client queryId)
     out)))



(defn stream-query-async
  [^Client client ^String query-str offset out-channel]
  (log/info "Executing query " query-str)
  ;(println "--" client)
  (let [offset (get #{"earliest" "latest"} offset "earliest")
        qc {"ksql.streams.auto.offset.reset", offset}
        ^HashMap m (convert-to-hashMap qc)
        query-id-p (promise)
        subscrition (atom nil)
        row-handler (reify
                      Subscriber
                      (^void onSubscribe [this ^Subscription s]
                        (reset! subscrition s)
                        (.request @subscrition 1))
                      (^void onNext [this r]
                        (log/debug "ksqldb msg " r)
                        ;(async/>!! out-channel r)
                        (async/put! out-channel r)
                        (.request @subscrition 1))
                      (^void onError [this ^Throwable v]
                        (log/error "Query error " v)
                        (deliver query-id-p (Throwable->map v))
                        (async/put! out-channel (:cause (Throwable->map v))))
                      (^void onComplete [this]
                        (log/info "Query complete " (get this :query-str))))]
    (doto
      (.streamQuery client query-str m)
      (.thenAccept (reify java.util.function.Consumer
                     (accept [this t]
                       (let [^StreamedQueryResult t1 t
                             query-id (.queryID t1)]
                         (deliver query-id-p query-id)
                         (.subscribe t row-handler)))))
      (.exceptionally (reify java.util.function.Function
                        (apply [this t]
                          (log/error "Error " t)
                          (deliver query-id-p nil)
                          (async/put! out-channel (:cause (Throwable->map t)))))))
    @query-id-p))


#_(defn query
    [context ^String query-str offset callback]
    (stream-query-async (get-client context)
                        query-str
                        offset
                        callback))


(defn stream-query-async-with-duration
  [context ^String query-str callback offset duration]
  (let [consumer-promise (promise)
        out-channel (async/chan (async/sliding-buffer 1) (comp (map (fn [r]
                                                                      (if (instance? Row r)
                                                                        (convrt-row->map r)
                                                                        r)))))
        query-id (stream-query-async (get-client context)
                                     query-str
                                     offset
                                     out-channel)]
    (log/info "start consuming data with query id" {:query-id query-id :ksql query-str :offset offset})
    (async/thread

      (async/go
        (async/<! (async/timeout duration))
        (deliver consumer-promise "done!"))

      (loop []
        (let [[v] (async/alts!! [(async/timeout duration) out-channel])]
          (when v
            (callback v)
            (when-not (realized? consumer-promise)
              (recur)))))

      (terminate-push-query (get-client context) query-id)
      (deliver consumer-promise "done!")

      )

    #_(when query-id

        )
    consumer-promise

    ))




(defn- recur-async-channel-to-lazy-seq
  [to-chan]
  (when-let [item (async/<!! to-chan)]
    (cons item (lazy-seq (recur-async-channel-to-lazy-seq to-chan)))))


(defn async-channel-to-lazy-seq
  "Convert a core-async channel into a lazy sequence where each item
is read via async/<!!.  Sequence ends when channel returns nil."
  [to-chan]
  ;;Avoid reading from to-chan immediately as this could force an immediate
  ;;block where one wasn't expected
  (lazy-seq (recur-async-channel-to-lazy-seq to-chan)))



(defn stream-query2
  [context ^String query-str offset]
  (let [
        ;consumer-promise (promise)
        out-channel (async/chan (async/sliding-buffer 1) (comp (map (fn [r]
                                                                      (if (instance? Row r)
                                                                        (convrt-row->map r)
                                                                        r)))))
        query-id (stream-query-async (get-client context)
                                     query-str
                                     offset
                                     out-channel)]
    (log/info "start consuming data with query id" {:query-id query-id :ksql query-str})
    #_(future
        (loop []
          (if (realized? consumer-promise)
            (do
              (terminate-push-query (get-client context) query-id))
            (do
              (callback (async/<!! out-channel))
              (recur)))))
    (future
      (Thread/sleep 10000)
      (async/close! out-channel)
      (terminate-push-query (get-client context) query-id)
      #_(async/go
          (async/<! (async/timeout 10000))

          #_(deliver consumer-promise "done!"))
      )

    (async-channel-to-lazy-seq out-channel)))





(comment


  (let [v (promise)
        _ (println (realized? v))
        _ (deliver v "vv")
        _ (println (realized? v))
        _ (println (realized? v))
        ]

    )

  (println (fn? (async/chan)))

  (println (fn? println))

  (let [v (promise)]
    (deliver v 10)
    (println (realized? v))
    (println (realized? v))

    )

  )


(defn total-row [^Client client m ^String query-str]
  (let [^StreamedQueryResult streamedQueryResult (.get (.streamQuery client query-str (convert-to-hashMap m)))
        queryId (.queryID streamedQueryResult)]
    (loop [total 0
           done? false]
      (if (or done?
              #_(.isComplete streamedQueryResult)
              #_(.isFailed streamedQueryResult))
        (do
          ;  (println "--terminal query ")
          (terminate-push-query client queryId)
          total)
        (let [^Row row (.poll streamedQueryResult (Duration/ofMillis 1000))
              done? (if (= nil row) true done?)]
          ;(println "--total " total)
          (recur (+ total 1) done?))))))


(defn as-ksqlobject [key-seq m]
  (let [w (into {} (map (fn [v] {v nil})) key-seq)
        m (select-keys m key-seq)
        m (merge w m)
        ksql-obj (KsqlObject.)]
    (doseq [kv key-seq]
      (.put ksql-obj (clojure.string/upper-case (name kv)) (get m kv)))
    ksql-obj))


(defn insert-into-batch [client stream-name stream-name-order row-coll]
  (let [xf (comp (map (partial as-ksqlobject stream-name-order))
                 (map (fn [v] (.insertInto client stream-name v))))
        w-coll (into [] xf row-coll)
        ^CompletableFuture result (->> (into-array ^CompletableFuture w-coll)
                                       (CompletableFuture/allOf))]
    (.thenRun result (reify Runnable
                       (run [this]
                         ; (println "--Done ")
                         (log/info "Processing"))))))


#_(defn pull-data-async-by-time
    ([context stream-name] (pull-data-async-by-time context stream-name 3000 println println))
    ([context ^String stream-name total-time data-handler] (pull-data-async-by-time context stream-name total-time data-handler println))
    ([context ^String stream-name total-time data-handler error-handler]
     (let [query-str (str "select * from " stream-name " emit changes;")
           query-id (query-stream-async (get-client context) query-str data-handler error-handler query-context)]
       (Thread/sleep total-time)
       (terminate-push-query (get-client context) query-id))))


#_(defn pull-data-by-time
    ([context stream-name] (pull-data-by-time context stream-name 3000))
    ([context ^String stream-name total-time]
     (let [out (atom [])
           query-str (str "select * from " stream-name " emit changes;")
           query-id (query-stream-async (get-client context) query-str
                                        (fn [v]
                                          (swap! out conj v))
                                        (fn [v] (reset! out v)) query-context)]
       (Thread/sleep total-time)
       (terminate-push-query (get-client context) query-id)
       @out)))



#_(defn pull-data-by-total
    ([context stream-name] (pull-data-by-total context stream-name 1))
    ([context stream-name total]
     (let [query-str (str "select * from " stream-name " emit changes limit " total " ;")]
       (execute-query (get-client context) query-str))))


(defn push-data-batch [context stream-name rows]
  (let [request (clojure.string/upper-case stream-name)
        stream-m (-> (ksql context (str "DESCRIBE " request ";"))
                     :sourceDescription)
        fields (into [] (comp (map :name)
                              (map clojure.string/lower-case)
                              (map keyword)) (get stream-m :fields))]
    (insert-into-batch (get-client context) stream-name fields rows)))












(defmethod p/invoke "ksql-show-functions"
  [ {:keys [request]}]
  (-> (ksql p/context " LIST FUNCTIONS;")
      (:functions)
      (->> (sort-by :name))))


(defmethod p/invoke "ksql-show-queries"
  [{:keys [request]}]
  (-> (ksql p/context "SHOW QUERIES;")
      :queries
     ; (:functions)
      #_(->> (sort-by :name))))


(defmethod p/invoke "ksql-show-functions"
  [{:keys [request]}]
  (-> (ksql p/context " LIST FUNCTIONS;")
      (:functions)
      (->> (sort-by :name))))

(defmethod p/invoke "ksql-describe-function"
  [{:keys [request]}]
  (when (nil? request)
    (throw (ex-info "Invalid request, please provide function name to get details" {:function-name request}))
    )
  (-> (ksql p/context (str "DESCRIBE FUNCTION " (clojure.string/upper-case request) ";"))))


(defmethod p/invoke "ksql-show-streams"
  [{:keys [request]}]
  (-> (ksql p/context " SHOW STREAMS;")
      :streams
      )
  #_(ksql/show-streams))

(defmethod p/invoke "ksql-show-tables"
  [{:keys [request]}]
  (-> (ksql p/context " SHOW TABLES;")
      :tables)
  #_(ksql/show-streams))


(defmethod p/invoke "ksql-show-streams-name"
  [{:keys [request]}]
  (-> (ksql p/context " SHOW STREAMS;")
      :streams
      (->> (mapv :name))
      )
  #_(ksql/show-streams))

(defmethod p/invoke "ksql-show-tables-name"
  [{:keys [request]}]
  (-> (ksql p/context " SHOW TABLES;")
      :tables
      (->> (mapv :name))
      )
  #_(ksql/show-streams))


(defmethod p/invoke "ksql-drop-stream"
  [{:keys [request]}]
  (when (nil? request)
    (throw (ex-info "Invalid request, stream name is null please provide stream name to drop stream" {:stream-name request}))
    )
  (let [w (p/invoke  {:op "ksql-describe-stream" :request request})
        t-queries (->> (mapv :id (into (get w :writeQueries) (get w :readQueries)))
                       (mapv #(str "TERMINATE " % "; "))
                       (distinct)
                       (into [])
                       (apply str))
        topic-name (get w :topic)]
    (when-not (clojure.string/blank? t-queries)
      (ksql p/context t-queries)
      )

    (ksql p/context (str "DROP STREAM IF EXISTS " request " DELETE TOPIC  ; "))))


(defmethod p/invoke "ksql-show-topics"
  [{:keys [request]}]
  (-> (ksql p/context " SHOW TOPICS;")
      (:topics)))

(defmethod p/invoke "ksql-show-topics-name"
  [{:keys [request]}]
  (-> (ksql p/context " SHOW TOPICS;")
      (:topics)
      (->> (map :name)
           (remove (fn [v] (or (clojure.string/starts-with? v "docker")
                               (clojure.string/starts-with? v "_schemas")
                               (clojure.string/starts-with? v "_confluent")
                               )))
           (into #{}))))


(defmethod p/invoke "ksql-describe-stream"
  [{:keys [request]}]
  (when (nil? request)
    (throw (ex-info "Invalid request, stream name is null please provide stream name " {:stream-name request}))
    )
  (let [request (clojure.string/upper-case request)]
    (-> (ksql p/context (str "DESCRIBE " request ";"))
        :sourceDescription)))

(defmethod p/invoke "ksql-describe-table"
  [{:keys [request]}]
  (when (nil? request)
    (throw (ex-info "Invalid request, table name is null please provide table name " {:stream-name request}))
    )
  (let [request (clojure.string/upper-case request)]
    (-> (ksql p/context (str "DESCRIBE " request ";"))
        :sourceDescription)))


#_(defmethod c/invoke "print-topic"
    [client {:keys [request]}]
    (let [{:keys [topic-name total]
           :or   {total 1}} request]
      (if topic-name
        (c/print-topic-async client topic-name total)
        "topic-name is empty ")))



