(ns ksql.gen.protocol
  (:require [clojure.pprint]
            [ksql.gen.core-error-msg :as emsg]
            [ingestion.api.client.connector-template :as ctemp]))



(defprotocol NS-Store
  (-get-ns-names [this])
  (-get-ns-by-name [this ns-name])
  (-persist-ns [this ns-name ns-v closed?])
  (-ns-exits? [this ns-name]))


(defn get-ns-names [ns-store]
  (-get-ns-names ns-store))


(defn get-ns-by-name [ns-store ns-name]
  (-get-ns-by-name ns-store ns-name))


(defn persist-ns
  ([ns-store ns-name ns-v] (persist-ns ns-store ns-name ns-v false))
  ([ns-store ns-name ns-v closed?]
   (-persist-ns ns-store ns-name ns-v closed?)))


(defn ns-exit? [ns-store ns-name]
  (-ns-exits? ns-store ns-name))



(def ^:dynamic context nil)


(defmulti invoke (fn [m] (get m :op)))

(defmethod invoke
  :default
  [{:keys [op]}]
  (let [api-list (-> (into #{} (keys (methods invoke)))
                     (disj :default)
                     (sort)
                     )]
    (throw (emsg/ex-info-for-op-not-found op api-list))))


(defn get-default-context [] {:kafka-rest-url       "http://localhost:8082"
                              :schema-url           "http://localhost:8081"
                              :connector-url        "http://localhost:8083"
                              :ksql-url             "http://localhost:8088"
                              :kafka-broker         "localhost:29092"

                              :monitor-stream        true
                              :namespace            "default"
                              :offset               "latest" ;;earliest
                              :partitions           1
                              :replicas             1
                              :join-window          "WITHIN 5 MINUTES"
                              :distinct-duration    "10 MINUTES"
                              :dv-attr              {:gdp_src_sys_id 2 :gdp_oe_id 20 :gdp_pipeline_id 200 :test 3}
                              :join-split           false
                              :offline              false
                              :reader               :triple ;; default reader
                              :error-stream-name    "error"

                              :connector-template-m (atom (ctemp/get-default-connector-template-m))
                              :query-id-m           (atom {})
                              :mapping-v            (atom "")
                              :api-playground-v     (atom "")
                              :metadata-repo        (atom [])
                              :connector-mapping-m  (atom "")})


(defn context->edn-data [context-v]
  (let [connector-template-m @(get context-v :connector-template-m)
        query-id-m @(get context-v :query-id-m)
        mapping-v @(get context-v :mapping-v)
        api-playground-v @(get context-v :api-playground-v)
        connector-mapping-m @(get context-v :connector-mapping-m)]
    (-> context-v
        (dissoc :connector-template-m :query-id-m :mapping-v :api-playground-v :metadata-repo :connector-mapping-m)
        (assoc :connector-template-m connector-template-m)
        (assoc :query-id-m query-id-m)
        (assoc :mapping-v mapping-v)
        (assoc :api-playground-v api-playground-v)
        (assoc :connector-mapping-m connector-mapping-m))))



(defn edn-data->context [context-edn]
  (let [connector-template-m (get context-edn :connector-template-m)
        query-id-m (get context-edn :query-id-m)
        mapping-v (get context-edn :mapping-v)
        api-playground-v (get context-edn :api-playground-v)
        connector-mapping-m (get context-edn :connector-mapping-m)
        out-context (-> context-edn
                        (dissoc :connector-template-m :query-id-m :mapping-v :api-playground-v :metadata-repo :connector-mapping-m)
                        (assoc :metadata-repo (atom []))
                        (assoc :connector-template-m (atom connector-template-m))
                        (assoc :query-id-m (atom query-id-m))
                        (assoc :mapping-v (atom mapping-v))
                        (assoc :api-playground-v (atom api-playground-v))
                        (assoc :connector-mapping-m (atom connector-mapping-m)))]
    (binding [context out-context]
      (invoke {:op "df-do-compile" :request mapping-v})
      context
      )
    ))


(defn reset-mapping! [mapping-v]
  (update-in context [:mapping-v] (fn [v] (reset! v mapping-v))))


(defn get-md-schema-coll []
  @(get context :metadata-repo)
  #_(into [] (comp (map second) cat) @(get context :metadata-repo)))


(defn reset-md-repo! []
  (update-in context [:metadata-repo] (fn [v]
                                        (reset! v []))))


(defn update-md-repo [md-schema-coll]
  (update-in context [:metadata-repo] (fn [v]
                                        (reset! v md-schema-coll))))




(def context-user-key-list [:kafka-rest-url
                            :schema-url
                            :connector-url
                            :kafka-broker

                            :offset
                            :partitions
                            :replicas

                            :dv-attr

                            :join-window
                            :distinct-duration
                            :join-split
                            :monitor-stream
                            :error-stream-name
                            ])






(defn alter-context [m]
  (alter-var-root #'context (constantly m)))


(defn copy-context [con]
  (let [c-temp @(get con :connector-template-m)
        m @(get con :mapping-v)
        c-mapping @(get con :connector-mapping-m)
        api-p-v @(get con :api-playground-v)
        ]

    (-> con
        (assoc :mapping-v (atom m))
        (assoc :connector-mapping-m (atom c-mapping))
        (assoc :connector-template-m (atom c-temp))
        (assoc :metadata-repo (atom []) #_(new-metadata-repo))
        (assoc :api-playground-v (atom api-p-v))
        (assoc :query-id-m (atom {})))))








;(def arvo-p-type #{"string" "bytes" "double" "float" "long" "int" "boolean" "null"})


(defmulti read-req (fn [t v] t))

(defmulti gen-ksql :ksql-gen-type)

(def gen-ksql-rekey-stream-from-mapping "rekey")
(def gen-ksql-stream-insert-from-mapping "insert-into")
(def gen-ksql-stream-from-mapping "stream-to-stream")
(def gen-ksql-create-stream "create-stream")
(def gen-ksql-table-from-stream-after-rekey "table-from-stream-after-rekey")
(def gen-ksql-create-table-as "create-table-as")
(def gen-ksql-create-table "create-table")

(defn build-req
  ([op] {:op op})
  ([app-ns-name op] {:app-ns-name app-ns-name :op op})
  ([app-ns-name op req] {:app-ns-name app-ns-name :op op :request req}))




(defmulti expand-step (fn [_ _ mapping-fields]
                        (get-in mapping-fields [2 0])))

(defmethod expand-step :default
  [_ _ w]
  ;(vector m)
  [w]
  )


(defrecord ExpandTerminate [v])

(defn new-expand-terminate [v]
  (ExpandTerminate. v))

(defn get-expand-terminate-value [t]
  (get t :v))

(defn is-terminate-step [v]
  (instance? ExpandTerminate v)
  )



(defmulti expand-flow (fn [_ mapping-fields] (first (get (first mapping-fields) :transfer_fn))))



(defmethod expand-flow :default
  [_ mapping]
  ;(println "--" (count mapping) "--" mapping)
  [(new-expand-terminate mapping)]
  ;(new-expand-terminate [mapping] )
  )



(defmulti transfer-fn (fn [f]
                        (-> (first f)
                            (clojure.string/trim)
                            (clojure.string/lower-case))))


(defmulti gen-connector (fn [m]
                          ;(get m t)
                          (get-in m [:connector-config :config :connector.class])

                          ))

(defmethod gen-connector
  :default
  [connector-schema]
  ; (  connector-schema)
  (get connector-schema :connector-config))



(defn log-v
  ([v] (log-v "" v))
  ([msg v]
   (println "--start--" msg)
   (clojure.pprint/pprint v)
   (println "--end--")
   v)

  )

(def xf-log-v
  (map (fn [v]
         (log-v v)
         v
         ))
  )


(defprotocol GetValue
  (-getValue [_]))

(defrecord ConstantValue [v]
  GetValue
  (-getValue [_] v))


(defn new-constant [v]
  (->ConstantValue v))


(defn constant-string? [v]
  (instance? ConstantValue v))


(defrecord RefValue [v]
  GetValue
  (-getValue [_] v))


(defrecord QuoteStringValue [v]
  GetValue
  (-getValue [_] v))

(defn new-quote-string [v]
  (->QuoteStringValue v))


(defn quote-string? [v]
  (instance? QuoteStringValue v))

(defn get-quote-value [v]
  (get v :v)
  )


(defn as-referance [v]
  (->RefValue v))



(defn get-value [v-type]
  (-getValue v-type))



(def msg {:100 "Could not find this reference on current context "
          :500 "Key is missing in mapping or source stream/table "

          })









