(ns ksql.gen.macro.metadata-crawler-impl
  (:require [clojure.tools.reader.edn :as edn]
            [clojure.tools.logging :as log]
            [cheshire.core :as json]
            [ksql.gen.protocol :as p]
            [ksql.gen.reader.file-reader :as r]))


(defn map-other-fields [e-name r-mapping]
  (into [] (comp (map (fn [m] ((juxt :field_name :transfer_fn) m)))
                 (map (fn [[f m]]
                        (let [f (if (= "_" f) "" f)
                              m (if (sequential? m)
                                  (str "( " (clojure.string/join " " m) " ) ")
                                  m
                                  )
                              ]
                          [f m]
                          )
                        ))
                 (map (fn [v]
                        (into [e-name] v)
                        ))
                 cat

                 ) r-mapping)
  )


(defn load-metadata-from-csv-file [mapping]
  (let [t-mapping (first mapping)
        [_ file_name] (get t-mapping :transfer_fn)
        n (get t-mapping :name)
        n (if (= "_" n) nil n)
        n (or n (r/get-file-name file_name))

        r-mapping (map-other-fields n (rest mapping))
        ; _ (p/log-v r-mapping)

        header (r/read-csv-header (clojure.string/replace file_name "'" ""))
        header (r/format-csv-header header)
        fields (reduce (fn [acc k]
                         (into acc [n k " "])
                         ) [] header)

        fields (into fields r-mapping)
        fields (into fields [n "" "(value_format 'AVRO')"])
        ;_ (p/log-v fields)
        ]
    [fields]))


(defn load-metadata-from-csv-folder [mapping]
  (let [t-mapping (first mapping)
        [_ csv-folder-name] (get t-mapping :transfer_fn)
        out (r/get-all-file-name csv-folder-name "glob:*.{csv}")]
    (into [] (map (fn [[file-name file-path]]
                    (let [header (r/read-csv-header file-path)
                          header (r/format-csv-header header)
                          out (reduce (fn [acc k]
                                        (into acc [file-name k " "])
                                        ) [] header)

                          r-mapping (map-other-fields file-name (rest mapping))
                          out (into out r-mapping)
                          out (into out [file-name "" "(value_format 'AVRO')"])]
                      out)
                    )) out)))






(comment

  (get-all-file-name "app/itmp/data/source" "glob:*.{json}")


  #_(edn/read-string (pr-str {:f_name "adsf" :l_name "asdf" :address {:road_no 334}}))

  )


(defn get-type [v]
  ;(println "--" v)
  (cond
    (double? v) "double"
    (number? v) "bigint"
    :else "string"
    )
  )




(defn from-edn-to-mapping [stream-name stream-value]
  ;  (p/log-v stream-value)

  (let []

    (letfn [(xf [e-type e-name m]
              (let [e-name (clojure.string/lower-case e-name)
                    w (into [] (comp (filter (fn [v] (or (map? (second v))
                                                         (and (sequential? (second v))
                                                              (map? (first (second v)))
                                                              )
                                                         ) #_(coll? (second v))))
                                     (map (fn [v]
                                            ;     (println v)
                                            (xf "struct" (str e-name "_" (name (first v))) (if (sequential? (second v))
                                                                                             (first (second v))
                                                                                             (second v)
                                                                                             ))))
                                     cat
                                     ) m)]
                (into w (comp (map (fn [[index k]]
                                     (let [
                                           ;e-name (if (= "struct" e-type)     (str e-name "_struct")              e-name                                               )
                                           t (cond
                                               (and (sequential? (get m k))
                                                    (map? (first (get m k))))
                                               (str e-name "_" (clojure.string/lower-case (name k)) "_struct" " array")

                                               (and (sequential? (get m k))
                                                    (not (sequential? (first (get m k))))
                                                    )
                                               (str (get-type (first (get m k))) " array")

                                               (map? (get m k))
                                               (str e-name "_" (clojure.string/lower-case (name k)) "_struct")

                                               :else (get-type (get m k)))

                                           e-name (if (= e-type "struct")
                                                    (str e-name "_struct")
                                                    e-name
                                                    )

                                           n (if (= 0 index)
                                               (str e-name " " e-type)
                                               e-name)

                                           ]
                                       [n (str (name k) " " t) nil])))
                              ) (map-indexed vector (keys m)))))]
      (->> (into (xf "stream" stream-name stream-value) [[stream-name nil "(value_format 'json')"]])
           ;(p/log-v)
           (into [] cat))
      )
    )

  )





(comment



  (map-from-json-data-file "test/resources/person.json")

  )


(defn load-metadata-from-edn-data [mapping]
  (let [mapping (first mapping)
        [_ edn-data] (get mapping :transfer_fn)
        n (get mapping :name)
        n (if (= "_" n) nil n)
   ;     _ (clojure.pprint/pprint edn-data)
        edn-data (if (string? edn-data)
                   (edn/read-string edn-data)
                   edn-data
                   )
      ;  _ (println "--------------")
       ; _ (clojure.pprint/pprint edn-data)
        w (from-edn-to-mapping n edn-data)]
    [w])
  )


(defn load-metadata-from-json-event [mapping]
  (let [mapping (first mapping)
        [_ json-str] (get mapping :transfer_fn)
        n (get mapping :name)
        n (if (= "_" n) nil n)
        ;     _ (clojure.pprint/pprint edn-data)
        edn-data  (json/parse-string json-str true) #_(if (string? edn-data)
                   (edn/read-string edn-data)
                   edn-data
                   )
   ;     _ (clojure.pprint/pprint edn-data)
        w (from-edn-to-mapping n edn-data)
        ]
    [w]

    )

  )




(defn from-xml-to-mapping-impl
 ; ([n file-path] (from-xml-to-mapping-impl n file-path false))
  [n file-path v]
  ;(println "---from xml-to-mapping" v)
   (let [f-name (or n (r/get-file-name file-path))
         edn-data (r/convert-xml-file-to-edn file-path v)
       ;  _ (p/log-v edn-data)
         w (from-edn-to-mapping f-name edn-data)]
     w))


(comment


  (r/convert-xml-file-to-edn "/home/mamun/books.xml" false)

  (from-xml-to-mapping-impl "books" "/home/mamun/books.xml" false)

  )


(defn load-metadata-from-xml-file [mapping]
  (let [mapping (first mapping)
        n (get mapping :name)
        n (if (= "_" n) nil n)
        [_ file-path] (get mapping :transfer_fn)
        w (from-xml-to-mapping-impl n file-path false)]
    [w]))



(defn load-metadata-from-xml-folder [mapping]
  (let [mapping (first mapping)
        ; n (get mapping :name)
        [_ json-folder-name] (get mapping :transfer_fn)

        out (r/get-all-file-name json-folder-name "glob:*.{xml}")]
    (into [] (comp
               (map (fn [[file-name file-path]]
                      (from-xml-to-mapping-impl file-name file-path false)))
               ) out))
  )




(defn map-from-json-schema-file [file-name json-file-path]
  (let [json-str (slurp json-file-path)
        edn-str (json/parse-string json-str true)
        f-name (or
                 file-name
                 (r/get-file-name json-file-path))
        w (from-edn-to-mapping f-name edn-str)]
    w
    )

  )


(defn map-from-json-data-file [n json-file-name]
  (let [json-str (with-open [rdr (clojure.java.io/reader json-file-name)]
                   (first (line-seq rdr)))
        edn-str (json/parse-string json-str true)
        ;_ (p/log-v edn-str)
        f-name (or n
                   (-> json-file-name
                       (clojure.string/split #"/")
                       (last)
                       (clojure.string/split #"\.")
                       (first)))
        w (from-edn-to-mapping f-name edn-str)
        ;     _ (clojure.pprint/pprint w)

        ]
    ;edn-str
    w
    )

  )


(defn load-metadata-from-json-file [mapping]
  (let [mapping (first mapping)
        ; n (get mapping :name)
        [_ json_file_name] (get mapping :transfer_fn)
        n (get mapping :name)
        n (if (= "_" n) nil n)
        w (map-from-json-data-file n json_file_name)]
    [w]
    )
  )


(defn load-metadata-from-json-folder [mapping]
  (let [mapping (first mapping)
        ; n (get mapping :name)
        [_ json-folder-name] (get mapping :transfer_fn)
        out (r/get-all-file-name json-folder-name "glob:*.{json}")]
    (into [] (comp
               (map (fn [[file-name file-path]]
                      (map-from-json-data-file file-name file-path)))
               ) out)))

;;;;;;;;;;;;;;;;


(defmethod p/expand-flow "load_md_from_csv_file"
  [_ mapping]
  (load-metadata-from-csv-file mapping))



(defmethod p/expand-flow "load_md_from_csv_folder"
  [_ mapping]
  (load-metadata-from-csv-folder mapping))


(defmethod p/expand-flow "load_md_from_json_event"
  [_ mapping]
  (load-metadata-from-json-event mapping)
  )



(defmethod p/expand-flow "load_md_from_edn_event"
  [_ mapping]
  (load-metadata-from-edn-data mapping)
  )


(defmethod p/expand-flow "load_md_from_xml_file"
  [_ mapping]
  (load-metadata-from-xml-file mapping))


(defmethod p/expand-flow "load_md_from_xml_folder"
  [_ mapping]
  (load-metadata-from-xml-folder mapping)
  )


(defmethod p/expand-flow "load_md_from_json_file"
  [_ mapping]
  (load-metadata-from-json-file mapping)
  )


(defmethod p/expand-flow "load_md_from_json_folder"
  [_ mapping]
  (load-metadata-from-json-folder mapping))


(defmethod p/expand-flow "load_md_from_folder"
  [_ mapping]
  (->
      (into [] (load-metadata-from-csv-folder mapping))
      (into  (load-metadata-from-json-folder mapping))
      (into  (load-metadata-from-xml-folder mapping))
      )


  )


;;;;;;;;;;;;;;;;;

(defmethod p/expand-flow "from_csv_file"
  [_ mapping]
  (log/info "deprecated api, use load_md_from_csv_file ")
  (load-metadata-from-csv-file mapping))



(defmethod p/expand-flow "from_csv_folder"
  [_ mapping]
  (log/info "deprecated api, use load_md_from_csv_folder ")
  (load-metadata-from-csv-folder mapping))


(defmethod p/expand-flow "from_edn_data"
  [_ mapping]
  (log/info "deprecated api, use load_md_from_edn_event ")
  (load-metadata-from-edn-data mapping)
  )


(defmethod p/expand-flow "from_xml_file"
  [_ mapping]
  (log/info "deprecated api, use load_md_from_xml_file")
  (load-metadata-from-xml-file mapping))


(defmethod p/expand-flow "from_xml_folder"
  [_ mapping]
  (log/info "deprecated api, use load_md_from_xml_folder")
  (load-metadata-from-xml-folder mapping)
  )


(defmethod p/expand-flow "from_json_file"
  [_ mapping]
  (log/info "deprecated api, use load_md_from_json_file")
  (load-metadata-from-json-file mapping)
  )


(defmethod p/expand-flow "from_json_folder"
  [_ mapping]
  (log/info "deprecated api, use load_md_from_json_folder")
  (load-metadata-from-json-folder mapping))





#_(defmethod p/expand-flow "from_json_schema_file"
  [_ mapping]

  ;(println "--" mapping)
  (let [mapping (first mapping)
        [_ json_file_name] (get mapping :transfer_fn)
        n (get mapping :name)
        n (if (= "_" n) nil n)
        w (map-from-json-schema-file n json_file_name)]
    [w]))


#_(defmethod p/expand-flow "from_json_schema_folder"
  [_ mapping]
  (let [mapping (first mapping)
        ; n (get mapping :name)
        [_ json-folder-name] (get mapping :transfer_fn)
        out (r/get-all-file-name json-folder-name "glob:*.{json}")]
    (into [] (comp
               (map (fn [[file-name file-path]]
                      (map-from-json-schema-file file-name file-path)))
               ) out)))
