(ns io.datafy.http.server.util
  (:require [clojure.pprint :as pprint]
            [clojure.tools.logging :as log]
            [clojure.string :as stri]
            [clojure.walk :as w]
            [clojure.zip :as z]))


(defn is-include? [filter-v w]
  (stri/includes? (stri/lower-case (if (string? w) w (str w)))
                  (stri/lower-case filter-v)))


(comment

  (is-include? "v" :0)

  )

(defn postwalk-filter
  "remove pairs of key-value that has nil value from a (possibly nested) map. also transform map to nil if all of its value are nil"
  [filter-v nm]
  (if (or (nil? filter-v)
          (empty? filter-v))
    nm
    (w/postwalk
      (fn [el]
        (if (map? el)
          (reduce-kv (fn [acc k v]
                       (if (or
                             (is-include? filter-v k)
                             (is-include? filter-v v)
                             )
                         (assoc acc k v)
                         acc

                         )
                       ) {} el)
          ;  (into {} (filter (partial is-include? filter-v) el))
          el))
      nm)))

(defn postwalk-remove-with
  [f nm]
  (w/postwalk
    (fn [el]
      (if (map? el)
        (let [m (into {} (remove (comp f second) el))]
          (when (seq m)
            m))
        el))
    nm))

(defn postwalk-remove-nils
  "remove pairs of key-value that has nil value from a (possibly nested) map. also transform map to nil if all of its value are nil"
  [nm]
  (postwalk-remove-with nil? nm))

(defn keyword->str
  [v]
  (if (keyword? v)
    (name v)
    v))

(defn replace-mk
  [f1 m]
  (let [f (fn [[k v]] [(f1 k) v])]
    (into {} (map f m))))

(defn postwalk-replace-key-with
  "Recursively transforms all map and first  vector keys from keywords to strings."
  {:added "1.1"}
  [f m]
  (w/postwalk (fn [x]
                (cond (map? x)
                      (replace-mk f x)
                      (vector? x)
                      (mapv f x)
                      :else x)) m))

(defn- replace-mv
  [f1 m]
  (let [f (fn [[k v]] [k (f1 v)])]
    (into {} (map f m))))

(defn postwalk-replace-value-with
  "Recursively transforms all map keys from strings to keywords."
  {:added "1.1"}
  [f m]
  (w/postwalk (fn [x] (cond
                        (map? x)
                        (replace-mv f x)
                        (vector? x)
                        (mapv f x)
                        :else x)) m))

(defn- get-key [prefix key]
  (if (nil? prefix)
    (keyword key)
    (keyword (str (name prefix) "." key))))

(defn- flatten-map-kvs
  ([m] (flatten-map-kvs m nil))
  ([m prefix]
   (if (map? m)
     (reduce-kv (fn [acc k v]
                  (cond
                    (map? v)
                    (concat acc (flatten-map-kvs v (get-key prefix (name k))))
                    (sequential? v)
                    (->> (map-indexed vector v)
                         (reduce (fn [acc [i w]]
                                   (->> (get-key (get-key prefix (name k)) (str (+ 1 i)))
                                        (flatten-map-kvs w)
                                        (concat acc))
                                   ) acc))
                    :else
                    (conj acc [(get-key prefix (name k)) v]))
                  ) [] m)
     m)))


(defn to-flatten-map
  [m]
  (into {} (flatten-map-kvs m)))


(defn to-simple-flatten-map [m]
  (let [out (group-by (fn [[k v]]
                        (if (coll? v) true false)
                        ) m)
        simple (into {} (get out false))
        comp (into {} (get out true))
        comp (into {} (map (fn [[k v]]
                             (if (sequential? v)
                               (to-simple-flatten-map (first v))
                               (to-simple-flatten-map v)))
                           ) comp)]
    (merge simple comp)))


(defn to-simple-flat [m]
  (reduce-kv (fn [acc k v]
               (if (sequential? v)
                 (into acc (mapv to-simple-flatten-map v))
                 (conj acc (to-simple-flatten-map v)))
               ) [] m))


(defn map-vec-zipper [m]
  (z/zipper
    (fn [x] (or (and (map? x)
                     (some coll? (vals x)))
                (and (sequential? x)
                     (map? (first x)))))
    seq
    (fn [p xs]
      (if (isa? (type p) clojure.lang.MapEntry)
        (into [] xs)
        (into (empty p) xs)))
    m))


(defn death-to-big-numbers
  [loc]
  (loop [loc loc]
    (if (z/end? loc)
      loc
      (do
        (println (z/node loc))
        (recur (z/next loc))))))


(comment



  (-> [{:department [{:dept_name "Bus"
                      :employee  [{:firstname "Abba", :lastname "Zoma"}
                                  {:firstname "Jal2", :lastname "Kona"}]}
                     {:dept_name "Market",
                      :employee  [{:firstname "Bdsf", :lastname "jkk"}]}
                     {:dept_name "Hr", :employee [{:firstname "asdf", :lastname "asdf"}]}]}]
      (map-vec-zipper)
      (death-to-big-numbers))


  (-> (map-vec-zipper [{1 [21 22] 3 [4]}])
      z/down
      (z/edit assoc :e 99)
      z/down
      ;; Note that the map does not guarantee particular entries ordering.
      z/down                                                ;; Getting into map entry.
      z/next
      (z/edit conj 77)
      z/root)


  (to-simple-flat
    {:department [{:dept_name "Bus"
                   :employee  [{:firstname "Abba", :lastname "Zoma"}
                               {:firstname "Jal2", :lastname "Kona"}]}
                  {:dept_name "Market",
                   :employee  [{:firstname "Bdsf", :lastname "jkk"}]}
                  {:dept_name "Hr", :employee [{:firstname "asdf", :lastname "asdf"}]}]}
    )


  (group-by (fn [[k v]]
              (if (coll? v) true false)
              ) {:a 3 :address [{:road 3} {:road 4}] :hello [{:a 3}]})

  ;(coll? {:a 3})

  (to-simple-flat {:hello [{:a 3 :addre {:b 3 :check {:check 3}}}
                           {:a 4}]})

  )


(defn format-k [w]
  (if-let [r (re-find (re-matcher #"(?<=\s|^)\d+(?=\s|$)" (name w)))]
    (Integer/parseInt r)
    (keyword w)))

(defn- do-split [[k v]]
  (let [k (->> (clojure.string/split (name k) #"\.")
               (mapv format-k))
        v {(last k) v}
        k1 (reverse (butlast k))]
    (reduce (fn [acc k]
              {k acc}
              ) v k1)))

(defn- deep-merge-with
  [f & maps]
  (apply
    (fn m [& maps]
      (if (every? map? maps)
        (apply merge-with m maps)
        (apply f maps)))
    maps))

(defn- as-vecotr [m]
  (let [w (fn [m]
            (if (and (map? m)
                     (not (empty? m))
                     (every? int? (keys m)))
              (do
                (->> (sort-by key m)
                     (mapv second))
                )

              m))]
    (if (map? m)
      (reduce-kv (fn [acc k v]
                   (->> (as-vecotr v)
                        (w)
                        (assoc acc k))
                   ) {} m)
      m)))

(defn from-flatten-map [m]
  (if (and (map? m)
           (not (empty? m)))
    (->> m
         (map do-split)
         (apply deep-merge-with merge)
         (as-vecotr))
    m))


(comment

  (from-flatten-map {:a {}})

  )

(defn apply-meta-key [meta w]
  (log/debug "Apply meta key " meta)
  (let [_remove (:io.datafy.core.spec2/remove meta)
        _filter (:io.datafy.core.spec2/filter meta)
        _flat (:io.datafy.core.spec2/flat meta)

        w (if (or (contains? meta :io.datafy.core.spec2/remove)
                  (= _remove "null"))
            (postwalk-remove-nils w)
            w)
        w (if _filter
            (postwalk-filter _filter w)
            w)
        w (if (contains? meta :io.datafy.core.spec2/sim)
            (to-simple-flat w)
            w)
        ; w
        #_(if (contains? meta :io.datafy.core.spec2/flat)
            (to-flatten-map w)
            w)
        ]
    w))

;;;;;;;;;;;;;; Extend Clojure core ;;;;;;;;;;;

(defn pformat [& v]
  (with-out-str
    (apply pprint/pprint v)))

(defn debug
  ([args] (debug "default:" args))
  ([msg args]
   (let [v (with-out-str
             (pprint/pprint args))]
     (log/info (format "<<<<<<<<<<< %s >>>>>>>>>>>\n" msg) v)
     args)))

(def xf-debug (map debug))

(defn xf-map-value [f]
  (comp (map (fn [m] (map (fn [[k v]] [k (f v)]) m)))
        cat))

(comment                                                    ;  (into {} (xf-map-value inc ) {:a 3 :b 4}   )
  (into [] (xf-map-value inc) [{:a 3} {:a 6}]))

(defn try-one [ops & r]
  (reduce (fn [acc v]
            (when-let [w (apply v r)]
              (reduced w))) nil ops))

