(ns io.datafy.http.server.error
  (:require [clojure.tools.logging :as log]
            [io.datafy.http.server.util :as u])
  (:import [org.apache.commons.text.similarity LevenshteinDistance]))

(def algo (LevenshteinDistance.))

(defn find-similar-word [coll-text search-text]
  (try
    (let [search-text (name search-text)]
      (->> coll-text
           (map (fn [v]
                  {v (.apply algo search-text (name v))}))
           (into {})
           (sort-by val >)
           (reverse)
           (map first)
           (take 10)))
    (catch Exception e
      (log/debug "Exception in " e)
      coll-text
      )
    )
  )


(comment

  (let [b (LevenshteinDistance.)]
    (.apply b "abba" "abba1")
    )


  (->> (find-similar-word ["check " "idd" "name" "tes" "idd2" "idddd"] "iddd")
       (clojure.pprint/pprint)
       )

  )



;;;;;;;;;;;;;;;;;;;;;;Error message

(def bad-request 400)
(def not-found 404)

(defn thorw-for-invalid-api-name [api-name available-api-set]
  (throw (ex-info "Invalid op name "
                  {:for        :api-name
                   :provided   api-name
                   :expected   (clojure.string/join "," (mapv name available-api-set))
                   :msg        "Invalid op name "
                   :error-code bad-request})))

(defn throw-for-invalid-content [content-type supported-type]
  (throw (ex-info "Invalid content "
                  {:for        :content-type
                   :provided   content-type
                   :expected   (clojure.string/join "," (mapv name supported-type))
                   :msg        "Invalid content "
                   :error-code bad-request})))

(defn throw-for-missing-entity-id [entity-name]
  (throw (ex-info  (format "Entity id is missing for : %s" (name entity-name))
                  {:for        :entity-id
                   :msg        (format "Entity id is missing for : %s" (name entity-name))
                   :error-code bad-request}))
  )


(defn throw-for-missing-attribute-name [attribute-name]
  (if (keyword? attribute-name)
    (throw (ex-info (format "Missing attribute %s" (str attribute-name))
                    {:for        :attr-name
                     :expected   attribute-name
                     :msg        (format "Missing attribute: %s" (str attribute-name))
                     :error-code bad-request}))
    (let [w (clojure.string/join "," (mapv (comp str name) attribute-name))]
      (throw (ex-info (format "Missing attribute: %s" w)
                      {:expected   attribute-name
                       :for        :attr-name
                       :msg        (format "Missing attribute: %s" (str w))
                       :error-code bad-request})))))

(defn throw-for-invalid-attribute-name [attr-list invalid-attr-name]
  (let [v (format "expected attribute name:  %s" (clojure.string/join "," (mapv name attr-list)))]
    (throw (ex-info v {:for        :attr-name
                       :expected   attr-list
                       :provided   invalid-attr-name
                       :msg        v
                       :error-code not-found}))))

(defn throw-for-invalid-attribute-value [attribute-name attribute-value expected-type]
  (let [et (if (keyword? expected-type) (name expected-type) (pr-str expected-type))
        v-type (str (type attribute-value))
        w (format "Invalid attribute value: %s for attribute name: %s, expected attribute type is: %s " (str attribute-value) (str (name attribute-name)) et)]
    (log/debug "Invalid attribute value ...... " {attribute-name {attribute-value v-type}})
    (throw (ex-info w {:for           :attr-spec
                       :provided      {attribute-name {attribute-value v-type}}
                       :provided-type v-type
                       :expected      et
                       :msg           w
                       :error-code    bad-request}))))

(defn throw-for-external-connection-exception [msg e]
  (log/debug "Error in external connection " e)
  (throw (ex-info (or msg "External Error ") {:for        :connection
                                              :provided   (get e :attr-filter)

                                              :msg        (or msg "External Error ")
                                              :error-code bad-request})))

(defn throw-for-invalid-join-ident [ename expected-e-list]
  (let [w (if ename
            (format "Invalid join iden name %s" (name ename))
            "Invalid join iden name")]
    (throw (ex-info w {:for        :ident-name
                       :provided   ename
                       :expected   expected-e-list
                       :msg        w
                       :error-code not-found}))))

(defn throw-for-invalid-ident [ename expected-e-list]
  (let [w (if ename
            (format "Invalid iden name %s" (name ename))
            "Invalid iden name")]
    (throw (ex-info w {:for        :ident-name
                       :provided   ename
                       :expected   expected-e-list
                       :msg        w
                       :error-code not-found}))))

(defn throw-for-invalid-ident-value [ename value]
  (throw (ex-info "Invalid iden value " {:for        :attr
                                         :provided   {ename value}
                                         :msg        "Invalid iden value expected as key value "
                                         :error-code not-found})))

(defn throw-for-empty-attribute [p]
  (throw (ex-info "Attribute should not be empty "
                  {:msg        "Param should not be empty "
                   :error-code bad-request
                   :for        :request
                   :provided   p})))

(defn throw-for-invalid-request [request]
  (throw (ex-info "Invalid request or don't found any request conformer "
                  {:msg        "Invalid request or don't found any request conformer "
                   :error-code bad-request
                   :provided   request
                   :for        :request})))

(defn thorw-for-invalid-repository-name [repository-name repository-name-list]
  (let [
        v (format "Invalid repository name %s" (name (or repository-name "nil")))]
    (throw (ex-info v {:for        (name :ds-name)
                       :provided   repository-name
                       :expected   (or repository-name-list {})
                       :msg        v
                       :error-code not-found}))))

(defn throw-for-invalid-datasource [ds-name expected]
  (throw (ex-info (format "Invalid data source %s " (name ds-name))
                  {:provided   ds-name
                   :for        (name :ds-m)
                   :expected   expected
                   :error-code not-found})))

(defn throw-for-invalid-api-request [r & [details]]
  (throw (ex-info "Invalid request  "
                  {:msg        "Invalid request, API request will be map "
                   :error-code bad-request
                   :for        r
                   :details details
                   })))

(defn throw-for-invalid-api-name [provided expected]
  (throw (ex-info "Invalid api name   "
                  {:msg        "Invalid API name "
                   :error-code bad-request
                   :provided   provided
                   :expected   expected})))

(defn throw-for-invalid-meta-request [r]
  (throw (ex-info "Invalid request  "
                  {:msg        "Invalid request  "
                   :error-code bad-request
                   :for        r
                   :expected   {:_remove    "null or any value "
                                :_filter    "any value"
                                :_flat      true
                                :_sim       true
                                :page       1
                                :page-limit 10}})))


;;;;;;;;;;;;;;;;; Compile error message

(comment

  (let [a 4
        b nil]
    (cond-> {}
            a (assoc :a a)
            b (assoc :b b)
            )
    )

  )


(defn compiler-log [msg for error-details expected-v]
  (let [o (cond-> {:msg msg}
                  for (assoc :for for)
                  expected-v (assoc :expected expected-v)
                  error-details (assoc :details error-details)
                  true u/pformat)]
    (log/info (format " \n---------------------- Compiler Exception ------------------ \n\n %s \n---------------------- Compiler Exception ------------------\n" o))))

(defn throw-for-compiler-exception
  ([msg] (throw-for-compiler-exception msg nil nil nil))
  ([msg for] (throw-for-compiler-exception msg for nil nil))
  ([msg for error-details] (throw-for-compiler-exception msg for error-details nil))
  ([msg for error-details expected-v]
   (compiler-log msg for error-details expected-v)
   (throw (ex-info "Compiling fail " (or error-details {:msg msg :for for :expected expected-v})))))

